/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * Copyright 2008 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: configset.cxx,v $
 * $Revision: 1.32 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_configmgr.hxx"
#include <stdio.h> // needed for Solaris 8
#include "configset.hxx"
#include "nodechange.hxx"
#include "nodechangeimpl.hxx"
#include "treeimpl.hxx"
#include "template.hxx"
#include "templateimpl.hxx"
#include "configgroup.hxx"
#include "valuenode.hxx"
#include "valuenodeimpl.hxx"
#include "setnodeimpl.hxx"
#include <vos/refernce.hxx>

namespace configmgr
{
	namespace configuration
	{
//-----------------------------------------------------------------------------
typedef SetInsertImpl  SetInsertTreeImpl;
typedef SetRemoveImpl  SetRemoveTreeImpl;
typedef SetReplaceImpl SetReplaceTreeImpl;
typedef SetInsertImpl  SetInsertValueImpl;
typedef SetRemoveImpl  SetRemoveValueImpl;
typedef SetReplaceImpl SetReplaceValueImpl;
//-----------------------------------------------------------------------------
// class ElementRef
//-----------------------------------------------------------------------------

ElementRef::ElementRef(ElementTreeImpl* pTree)
: m_aTreeHolder(pTree)
{
}
//-----------------------------------------------------------------------------

ElementRef::ElementRef(ElementRef const& aOther)
: m_aTreeHolder(aOther.m_aTreeHolder)
{
}
//-----------------------------------------------------------------------------

ElementRef& ElementRef::operator=(ElementRef const& aOther)
{
	m_aTreeHolder = aOther.m_aTreeHolder;
	return *this;
}
//-----------------------------------------------------------------------------

ElementRef::~ElementRef()
{
}
//-----------------------------------------------------------------------------

bool ElementRef::isValid() const
{
	return !!m_aTreeHolder.is();
}

//-----------------------------------------------------------------------------

ElementTree ElementRef::getElementTree() const
{
   return ElementTree(m_aTreeHolder);
}
//-----------------------------------------------------------------------------

Path::Component ElementRef::getFullName() const
{
    if (!isValid()) return Path::makeEmptyComponent();

    return m_aTreeHolder->getExtendedRootName();
}
//-----------------------------------------------------------------------------

TemplateHolder ElementRef::getTemplate() const
{
	if (!isValid()) return TemplateHolder();

    return m_aTreeHolder->getTemplate();
}
//-----------------------------------------------------------------------------

TreeRef ElementRef::getTreeRef() const
{
	return TreeRef(m_aTreeHolder.get());
}
//-----------------------------------------------------------------------------

ElementRef ElementRef::extract(TreeRef const& aTree)
{
	TreeImpl* pTree = TreeImplHelper::impl(aTree);
	ElementTreeImpl* pImpl = pTree ? pTree->asElementTree() : 0;
	return ElementRef(pImpl);
}

//-----------------------------------------------------------------------------
// class ElementTree
//-----------------------------------------------------------------------------

ElementTree::ElementTree()
: m_aTreeHolder()
{
}
//-----------------------------------------------------------------------------

ElementTree::ElementTree(ElementTreeImpl* pTree)
: m_aTreeHolder(pTree)
{
}
//-----------------------------------------------------------------------------

ElementTree::ElementTree(ElementTreeHolder const& pTree)
: m_aTreeHolder(pTree)
{
}
//-----------------------------------------------------------------------------

ElementTree::ElementTree(ElementTree const& aOther)
: m_aTreeHolder(aOther.m_aTreeHolder)
{
}
//-----------------------------------------------------------------------------

ElementTree& ElementTree::operator=(ElementTree const& aOther)
{
	m_aTreeHolder = aOther.m_aTreeHolder;
	return *this;
}
//-----------------------------------------------------------------------------

ElementTree::~ElementTree()
{
}
//-----------------------------------------------------------------------------

bool ElementTree::isValid() const
{
	return !!m_aTreeHolder.is();
}
//-----------------------------------------------------------------------------

ElementTreeHolder ElementTree::get() const
{
	return m_aTreeHolder;
}
//-----------------------------------------------------------------------------

ElementTreeImpl* ElementTree::getImpl() const
{
	return m_aTreeHolder.get();
}
//-----------------------------------------------------------------------------

ElementTreeImpl* ElementTree::operator->() const
{
	return m_aTreeHolder.operator->();
}
//-----------------------------------------------------------------------------

ElementTreeImpl& ElementTree::operator*() const
{
	return m_aTreeHolder.operator*();
}
//-----------------------------------------------------------------------------

Tree ElementTree::getTree() const
{
	return Tree(m_aTreeHolder.get());
}
//-----------------------------------------------------------------------------

ElementTree ElementTree::extract(Tree const& aTree)
{
    return ElementRef::extract(aTree.getRef()).getElementTree();
}

//-----------------------------------------------------------------------------
// class TemplateInfo
//-----------------------------------------------------------------------------

TemplateInfo::TemplateInfo(TemplateHolder const& aTemplate)
: m_aTemplate(aTemplate)
{
	OSL_ENSURE(m_aTemplate.is(), "ERROR: Configuration: Creating TemplateInfo without template information");
	if (!m_aTemplate.is())
		throw configuration::Exception("Missing template information");
}
//-----------------------------------------------------------------------------

TemplateHolder TemplateInfo::getTemplate() const
{
	return m_aTemplate;
}
//-----------------------------------------------------------------------------

UnoType TemplateInfo::getType() const
{
	return m_aTemplate->getInstanceType();
}

//-----------------------------------------------------------------------------

OUString TemplateInfo::getTemplatePathString() const
{
	return m_aTemplate->getPathString();
}
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// class SetElementInfo
//-----------------------------------------------------------------------------

SetElementInfo::SetElementInfo(TemplateHolder const& aTemplate)
: m_aTemplateInfo(aTemplate)
{
}
//-----------------------------------------------------------------------------

TemplateHolder SetElementInfo::getTemplate() const
{
	return m_aTemplateInfo.getTemplate();
}
//-----------------------------------------------------------------------------

TemplateInfo SetElementInfo::getTemplateInfo() const
{
	return m_aTemplateInfo;
}
//-----------------------------------------------------------------------------

TemplateHolder SetElementInfo::extractElementInfo(Tree const& aTree, NodeRef const& aNode)
{
	OSL_ENSURE(!aTree.isEmpty(), "ERROR: Getting Element Info requires a valid tree");
	OSL_ENSURE(aNode.isValid(), "ERROR: Getting Element Info requires a valid node");
	OSL_ENSURE(aTree.isValidNode(aNode), "ERROR: Tree/Node mismatch");
	if (aNode.isValid() )
	{
        view::ViewTreeAccess aView = aTree.getView();

		OSL_ENSURE (aView.isSetNode(aNode), "WARNING: Getting Element Info requires a SET node");
		if (aView.isSetNode(aNode))
			return aView.getElementTemplate(aView.toSetNode(aNode));
	}
	return TemplateHolder();
}

//-----------------------------------------------------------------------------
// class SetElementFactory
//-----------------------------------------------------------------------------

SetElementFactory::SetElementFactory(TemplateProvider const& aProvider)
: m_aProvider(aProvider)
{
	OSL_ENSURE(aProvider.m_aImpl.is(), "WARNING: Template Instance Factory created without template provider - cannot instantiate elements");
}
//-----------------------------------------------------------------------------

SetElementFactory::SetElementFactory(SetElementFactory const& aOther)
: m_aProvider(aOther.m_aProvider)
{
}
//-----------------------------------------------------------------------------

SetElementFactory& SetElementFactory::operator=(SetElementFactory const& aOther)
{
	m_aProvider = aOther.m_aProvider;
	return *this;
}
//-----------------------------------------------------------------------------

SetElementFactory::~SetElementFactory()
{
}

//-----------------------------------------------------------------------------
ElementTree SetElementFactory::instantiateTemplate(TemplateHolder const& aTemplate)
{
	OSL_ENSURE(m_aProvider.m_aImpl.is(), "ERROR: Template Instance Factory has no template provider - cannot instantiate element");
	OSL_ENSURE(aTemplate.is(), "ERROR: Template is NULL - cannot instantiate element");

    if (!m_aProvider.m_aImpl.is()) return ElementTree::emptyElement();

	if (!aTemplate.is()) return ElementTree::emptyElement();

    data::TreeSegment aInstanceTree( m_aProvider.m_aImpl->instantiate(aTemplate) );
	OSL_ENSURE(aInstanceTree.is(), "ERROR: Cannot create Element Instance: Provider could not instantiate template");

    
	if (!aInstanceTree.is()) return ElementTree::emptyElement();
    //set removable state 
	aInstanceTree.markRemovable();
	
	ElementTree aRet( new ElementTreeImpl( aInstanceTree, aTemplate, m_aProvider ) );

	return aRet;
}
//-----------------------------------------------------------------------------
ElementTree SetElementFactory::instantiateOnDefault(data::TreeSegment const& _aElementData, TemplateHolder const& aDummyTemplate)
{
//	OSL_ENSURE(m_aProvider.m_aImpl(), "ERROR: Template Instance Factory has no template provider - cannot instantiate element");
	OSL_ENSURE(_aElementData.is(), "ERROR: Tree is NULL - cannot instantiate element");
	OSL_ENSURE(aDummyTemplate.is(), "ERROR: Template is NULL - cannot instantiate element");

    if (!_aElementData.is()) return ElementTree::emptyElement();

	ElementTree aRet( new ElementTreeImpl( _aElementData, aDummyTemplate, m_aProvider ) );
	// ElementTreeImpl* pNewTree = new ElementTreeImpl( NodeType::getDeferredChangeFactory(),*aTree, c_TreeDepthAll, aDummyTemplate, m_aProvider );
	// pNewTree->takeNodeFrom(aTree);

	// return ElementTree( pNewTree );

	return aRet;
}
//-----------------------------------------------------------------------------

TemplateProvider SetElementFactory::findTemplateProvider(Tree const& aTree, NodeRef const& aNode)
{
	OSL_ENSURE(!aTree.isEmpty(), "ERROR: Getting Element Factory requires a valid tree");
	OSL_ENSURE(aNode.isValid(), "ERROR: Getting Element Factory requires a valid node");
	OSL_ENSURE(aTree.isValidNode(aNode), "ERROR: Tree/Node mismatch");
	if (aNode.isValid() )
	{
        view::ViewTreeAccess aView = aTree.getView();

		OSL_ENSURE (aView.isSetNode(aNode), "WARNING: Getting Element Factory requires a SET node");
		if (aView.isSetNode(aNode))
			return aView.getTemplateProvider(aView.toSetNode(aNode));
	}
	return TemplateProvider();
}

//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// class TreeSetUpdater and ValueSetUpdater
//-----------------------------------------------------------------------------

static node::Attributes getNewElementAttributes(bool bInserting)
{
    node::Attributes aResult;
    aResult.setState( node::isReplaced );
	//Check if you are inserting new dynamic property
	if(bInserting)
	{
        aResult.markRemovable();
	}
    return aResult;
}

// Value Element Factory methods
//-----------------------------------------------------------------------------

ElementTreeHolder ValueSetUpdater::makeValueElement(Name const& aName, UnoAny const& aValue, bool bInserting)
{
   
	const node::Attributes aNewValueAttributes = getNewElementAttributes(bInserting); // TODO: get real value
	
	UnoType aType = m_aTemplate->getInstanceType();

    OUString aTypeName = m_aTemplate->getName().toString();

	std::auto_ptr<INode> pNode;
	if (aValue.hasValue())
		pNode.reset( new ValueNode(aTypeName, aValue, aNewValueAttributes) );
	else
		pNode.reset( new ValueNode(aTypeName, aType, aNewValueAttributes) );

    data::TreeSegment aValueTree = data::TreeSegment::createNew(aName.toString(), pNode);

	return new ElementTreeImpl(aValueTree, m_aTemplate, TemplateProvider() );
}
//-----------------------------------------------------------------------------

ElementTreeHolder ValueSetUpdater::makeValueElement(Name const& aName, ElementNodeRef const& , UnoAny const& aValue, bool bInserting)
{
	// for now ignoring the node.
	// TODO: merge attributes etc. from that node's value
	return makeValueElement(aName, aValue,bInserting);
}
//-----------------------------------------------------------------------------


TreeSetUpdater::TreeSetUpdater(Tree const& aParentTree, NodeRef const& aSetNode, SetElementInfo const& aInfo)
: m_aParentTree(aParentTree)
, m_aSetNode(aSetNode)
, m_aTemplate(aInfo.getTemplate())
{
	implValidateSet();
}
//-----------------------------------------------------------------------------

ValueSetUpdater::ValueSetUpdater(Tree const& aParentTree, NodeRef const& aSetNode,
								 SetElementInfo const& aInfo, UnoTypeConverter const& xConverter)
: m_aParentTree(aParentTree)
, m_aSetNode(aSetNode)
, m_aTemplate(aInfo.getTemplate())
, m_xTypeConverter(xConverter)
{
	implValidateSet();
}
//-----------------------------------------------------------------------------

SetDefaulter::SetDefaulter(Tree const& aParentTree, NodeRef const& aSetNode,
						    DefaultProvider const& aDefaultProvider)
: m_aParentTree(aParentTree)
, m_aSetNode(aSetNode)
, m_aDefaultProvider(aDefaultProvider)
{
	implValidateSet();
}
//-----------------------------------------------------------------------------

/// validates that a actual set and an updater's construction parameters match
static void doValidateSet(Tree const& aParentTree, NodeRef const& aSetNode)
{
	if (aParentTree.isEmpty())
		throw Exception("INTERNAL ERROR: Set Update: Unexpected NULL tree");

	if (!aSetNode.isValid())
		throw Exception("INTERNAL ERROR: Set Update: Unexpected NULL node");

	if (!aParentTree.isValidNode(aSetNode))
		throw Exception("INTERNAL ERROR: Set Update: node does not match tree");

	if (! aParentTree.getView().isSetNode(aSetNode))
		throw Exception("INTERNAL ERROR: Set Update: node is not a set");

	if (aParentTree.getAttributes(aSetNode).isReadonly()) 
		throw ConstraintViolation( "Set Update: Set is read-only !" );
}
//-----------------------------------------------------------------------------

/// validates that the actual set and the construction parameters match
void TreeSetUpdater::implValidateSet()
{
	doValidateSet(m_aParentTree,m_aSetNode);

	if (!m_aTemplate.is())
		throw Exception("INTERNAL ERROR: No template available for tree set update");

	if (m_aTemplate->isInstanceValue())
		throw Exception("INTERNAL ERROR: Tree set update invoked on a value-set");

    view::ViewTreeAccess aParentView = m_aParentTree.getView();

    if ( aParentView.getElementTemplate(aParentView.toSetNode(m_aSetNode)) != m_aTemplate)
		throw Exception("INTERNAL ERROR: Set Update: template mismatch");
}
//-----------------------------------------------------------------------------

/// validates that the actual set and the construction parameters match
void ValueSetUpdater::implValidateSet()
{
	doValidateSet(m_aParentTree,m_aSetNode);

	UnoType aThisType = m_aTemplate->getInstanceType();

	switch ( aThisType.getTypeClass())
	{
	case uno::TypeClass_VOID:		throw Exception("INTERNAL ERROR: Value set element type is void");
	case uno::TypeClass_INTERFACE:	throw Exception("INTERNAL ERROR: Value update invoked on a complex set");

	case uno::TypeClass_STRUCT:
	case uno::TypeClass_EXCEPTION:	throw Exception("INTERNAL ERROR: Unexpected/Invalid type for set elements");

	default: break;
	}

    view::ViewTreeAccess aParentView = m_aParentTree.getView();

	if ( aParentView.getElementTemplate(aParentView.toSetNode(m_aSetNode))->getInstanceType() != aThisType)
		throw Exception("INTERNAL ERROR: Set Update: element type mismatch");
}
//-----------------------------------------------------------------------------

/// validates that the actual set and the construction parameters match
void SetDefaulter::implValidateSet()
{
	doValidateSet(m_aParentTree,m_aSetNode);

	if (!m_aDefaultProvider.isValid())
		throw Exception("INTERNAL ERROR: No default provider available for restoring set default state");
}
//-----------------------------------------------------------------------------

static void doValidateElement(ElementRef const& aElement, bool bReqRemovable)
{
	if (!aElement.isValid())
		throw Exception("INTERNAL ERROR: Set Update: Unexpected NULL element");

// DISABLED: replaceable/removable != writable
//	if (!aElement.getAttributes().writable)
//		throw ConstraintViolation( "Set Update: Existing element is read-only !" );

	if ( bReqRemovable)
	{
		Tree aElementTree = aElement.getElementTree().getTree();
	
		if(!aElementTree.getAttributes(aElementTree.getRootNode()).isRemovable())
			throw ConstraintViolation( "New Set Update: Existing element cannot be removed (or replaced) !" );
	}
}
//-----------------------------------------------------------------------------

/// validates that the given element is valid in this context and returns its name
Path::Component TreeSetUpdater::implValidateElement(ElementRef const& aElement, bool bReqRemovable)
{
	doValidateElement(aElement,bReqRemovable);

#if 0 // maybe reeanable for OSL_DEBUG_LEVEL>1 ?
	ElementTreeImpl* pElement = TreeImplHelper::elementImpl(aTree)->isTemplateInstance();
	OSL_ENSURE( pElement, "INTERNAL ERROR: Set Element has wrong type of tree");
	OSL_ENSURE( !pElement || pElement->isTemplateInstance(), "INTERNAL ERROR: Set Element without associated template found");
	OSL_ENSURE( !pElement || pElement->isInstanceOf(m_aTemplate), "INTERNAL ERROR: Set Update: existing element does not match template");
#endif

	return aElement.getFullName();
}
//-----------------------------------------------------------------------------

/// validates that the given element is valid and can be replaced in this context and returns its name
Path::Component ValueSetUpdater::implValidateElement(ElementRef const& aElement, bool mReqRemovable)
{
	doValidateElement(aElement,mReqRemovable);

#if OSL_DEBUG_LEVEL > 0
    UnoType aNodeType = ElementHelper::getUnoType(aElement.getElementTree());

	OSL_ENSURE(aNodeType.getTypeClass() != uno::TypeClass_VOID, "INTERNAL ERROR: Set Element without associated type found");
	OSL_ENSURE(aNodeType.getTypeClass() != uno::TypeClass_INTERFACE,"INTERNAL ERROR: Set Element with complex type found");

	OSL_ENSURE(aNodeType == m_aTemplate->getInstanceType() ||
			   uno::TypeClass_ANY == m_aTemplate->getInstanceType().getTypeClass(),
			   "INTERNAL ERROR: Set Update: existing element does not match template type");
#endif

	return aElement.getFullName();
}
//-----------------------------------------------------------------------------
static void checkEligibleChild(ElementTree const& aElementTree, Tree const& aParentTree)
{
	ElementTreeImpl const * const pElement = aElementTree.getImpl();	OSL_ASSERT(pElement);

	if (pElement->getContextTree() != NULL)
		throw ConstraintViolation( "Set Update: cannot insert an element that already has a parent." );

	TreeImpl const* pAncestor = TreeImplHelper::impl(aParentTree);
	while (pAncestor != NULL)
	{
		if (pElement == pAncestor)
			throw ConstraintViolation( "Set Update: Circular insertion - trying to insert an element into self or descendant" );

		pAncestor = pAncestor->getContextTree();
		OSL_ENSURE(pAncestor != TreeImplHelper::impl(aParentTree), "ERROR: Circular tree found");
	}
}

//-----------------------------------------------------------------------------

void TreeSetUpdater::implValidateTree(ElementTree const& aElementTree)
{
	if (!aElementTree.isValid())
		throw ConstraintViolation( "Set Update: cannot replace element of complex set with NULL node. Remove the element instead !" );

	checkEligibleChild(aElementTree,m_aParentTree);

	if (! aElementTree->isTemplateInstance())
	{
		throw TypeMismatch(OUString(RTL_CONSTASCII_USTRINGPARAM("<Unnamed> (Template missing)")),
									m_aTemplate->getName().toString(), " - new element without template in Set Update");
	}

	if (!aElementTree->isInstanceOf(m_aTemplate))
	{
		throw TypeMismatch( aElementTree->getTemplate()->getPathString(),
							m_aTemplate->getPathString(), " - new element without template in Set Update");
	}
}
//-----------------------------------------------------------------------------

UnoAny ValueSetUpdater::implValidateValue(UnoAny const& aValue)
{
	UnoType const aThisType = m_aTemplate->getInstanceType();

    OSL_ENSURE( aThisType.getTypeClass() == uno::TypeClass_ANY || isPossibleValueType(aThisType),
                "Invalid element type for value set" );

	UnoAny aRet;
	if (aValue.hasValue())
	{
	    UnoType const aValType = aValue.getValueType();

	    if (aValType.getTypeClass() == uno::TypeClass_INTERFACE)
		    throw TypeMismatch(aValType.getTypeName(), aThisType.getTypeName(), " - cannot replace value by complex tree in Set update");

        if (aValType == aThisType)
        {
            aRet = aValue;
        }

        else if ( uno::TypeClass_ANY == aThisType.getTypeClass() )
        {
            if ( ! isPossibleValueType(aValType) )
				throw TypeMismatch(aValType.getTypeName(), aThisType.getTypeName(), " - new element has no legal configuration data type");

            aRet = aValue;
        }

        else
		{
			if (!convertCompatibleValue(m_xTypeConverter, aRet, aValue, aThisType))
				throw TypeMismatch(aValType.getTypeName(), aThisType.getTypeName(), " - new element does not match template type in SetUpdate");
		}

	    OSL_ASSERT( isPossibleValueType(aRet.getValueType()) );
	}
	else
	{
		//  cannot do anything about null values here
		OSL_ASSERT(aValue.getValueTypeClass() == uno::TypeClass_VOID);

	}
	return aRet;
}
//-----------------------------------------------------------------------------

UnoAny ValueSetUpdater::implValidateValue(ElementNodeRef const& aElementTree, UnoAny const& aValue)
{
    node::Attributes aAttributes = aElementTree.getAttributes(aElementTree.getRootNode());
	// Here we assume writable == removable/replaceable
	if (aAttributes.isReadonly()) 
		throw ConstraintViolation( "Set Update: Existing element is read-only !" );

	// Here we assume nullable != removable
	if (!aValue.hasValue())
	{
		if (!aAttributes.isNullable())
			throw ConstraintViolation( "Set Update: Value is not nullable !" );
	}
	return implValidateValue( aValue);
}
//-----------------------------------------------------------------------------

NodeChange TreeSetUpdater::validateInsertElement (Name const& aName, ElementTree const& aNewElement)
{
    view::ViewTreeAccess aParentView = m_aParentTree.getView();

    SetEntry anEntry = aParentView.findElement(aParentView.toSetNode(m_aSetNode),aName);
	if (anEntry.isValid())
		throw Exception("INTERNAL ERROR: Set Update: Element to be inserted already exists");

	implValidateTree(aNewElement);

    std::auto_ptr<SetElementChangeImpl> pChange( new SetInsertTreeImpl(aNewElement->makeExtendedName(aName), aNewElement.get()) );

	pChange->setTarget(aParentView.makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}
//-----------------------------------------------------------------------------

NodeChange ValueSetUpdater::validateInsertElement (Name const& aName, UnoAny const& aNewValue)
{
    view::ViewTreeAccess aParentView = m_aParentTree.getView();

    SetEntry anEntry = aParentView.findElement(aParentView.toSetNode(m_aSetNode),aName);
	if (anEntry.isValid())
		throw Exception("INTERNAL ERROR: Set Update: Element to be inserted already exists");

	UnoAny aValidValue = implValidateValue(aNewValue);

    ElementTreeHolder aNewElement = makeValueElement(aName, aValidValue,true);

    std::auto_ptr<SetElementChangeImpl> pChange( new SetInsertValueImpl(aNewElement->makeExtendedName(aName), aNewElement) );

	pChange->setTarget(aParentView.makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}
//-----------------------------------------------------------------------------

NodeChange TreeSetUpdater::validateReplaceElement(ElementRef const& aElement, ElementTree const& aNewElement)
{
	Path::Component aName = implValidateElement(aElement,true);

	implValidateTree(aNewElement);

	std::auto_ptr<SetElementChangeImpl> pChange( new SetReplaceTreeImpl(aName, aNewElement.get()) );

	pChange->setTarget(m_aParentTree.getView().makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}
//-----------------------------------------------------------------------------

NodeChange ValueSetUpdater::validateReplaceElement(ElementRef const& aElement, UnoAny const& aNewValue)
{
	Path::Component aName = implValidateElement(aElement,false);

    ElementNodeRef aElementNode = extractElementNode(aElement);

	UnoAny aValidValue = implValidateValue(aElementNode, aNewValue);
   
    ElementNodeRef aElementTree = aElement.getElementTree().getTree();
	
	ElementTreeHolder aNewElement;
	if(aElementTree.getAttributes(aElementTree.getRootNode()).isRemovable())
	{
		aNewElement = makeValueElement(aName.getName(), aElementNode, aValidValue,true);
	}
	else
	{
		aNewElement = makeValueElement(aName.getName(), aElementNode, aValidValue,false);
	}

    std::auto_ptr<SetElementChangeImpl> pChange( new SetReplaceValueImpl(aName, aNewElement) );

	pChange->setTarget(m_aParentTree.getView().makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}
//-----------------------------------------------------------------------------

NodeChange TreeSetUpdater::validateRemoveElement (ElementRef const& aElement)
{
	Path::Component aName = implValidateElement(aElement,true);

	std::auto_ptr<SetElementChangeImpl> pChange( new SetRemoveTreeImpl(aName) );

	pChange->setTarget(m_aParentTree.getView().makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}

//-----------------------------------------------------------------------------

NodeChange ValueSetUpdater::validateRemoveElement (ElementRef const& aElement)
{
	Path::Component aName = implValidateElement(aElement,true);

	std::auto_ptr<SetElementChangeImpl> pChange( new SetRemoveValueImpl(aName) );

	pChange->setTarget(m_aParentTree.getView().makeNode(m_aSetNode));

	return NodeChange(pChange.release());
}

//-----------------------------------------------------------------------------

NodeChange SetDefaulter::validateSetToDefaultState()
{
    std::auto_ptr< ISubtree > aDefault = m_aDefaultProvider.getDefaultTree(m_aParentTree,m_aSetNode);

	// now build the specific change
	std::auto_ptr<SetChangeImpl> pChange;

    if (aDefault.get())
    {
        TemplateProvider aProvider = SetElementFactory::findTemplateProvider(m_aParentTree,m_aSetNode);

        configmgr::configuration::SetElementFactory aTmp(aProvider);
        pChange.reset( new SetResetImpl(aTmp, aDefault) );
	    pChange->setTarget(m_aParentTree.getView().makeNode(m_aSetNode));
    }
	return NodeChange(pChange.release());
}
//-----------------------------------------------------------------------------

ValueSetUpdater::ElementNodeRef ValueSetUpdater::extractElementNode (ElementRef const& aElement)
{
    return aElement.getElementTree().getTree();
}
//-----------------------------------------------------------------------------

#if OSL_DEBUG_LEVEL > 0
UnoType	ElementHelper::getUnoType(ElementTree const& aElement)
{
	OSL_PRECOND( aElement.isValid(), "ERROR: Configuration: ElementRef operation requires valid node" );
	if (!aElement.isValid()) return getVoidCppuType();

    Tree aElementTree = aElement.getTree();

    NodeRef aNode = aElementTree.getRootNode();
	OSL_ASSERT( aNode.isValid() );

    view::ViewTreeAccess aElementView = aElementTree.getView();

    if ( aElementView.isValueNode(aNode) )
	{
		return aElementView.getValueType(aElementView.toValueNode(aNode));
	}
	else
	{
		uno::Reference< uno::XInterface > const * const selectInterface=0;
		return ::getCppuType(selectInterface);
	}
}
#endif
//-----------------------------------------------------------------------------
	}
}

