/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dialogcontrolling.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 15:51:54 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef DBACCESS_SOURCE_UI_INC_DIALOGCONTROLLING_HXX
#define DBACCESS_SOURCE_UI_INC_DIALOGCONTROLLING_HXX

#include <vector>
#include <boost/shared_ptr.hpp>

#ifndef _LINK_HXX
#include <tools/link.hxx>
#endif

#ifndef _SV_BUTTON_HXX
#include <vcl/button.hxx>
#endif

//........................................................................
namespace dbaui
{
//........................................................................

    //=====================================================================
    //= IWindowAction
	//=====================================================================
    /** an abstract interface for operating on a ->Window
    */
    class IWindowOperator
    {
    public:
        virtual void operateOn( Window& _rOperateOn ) const = 0;

        virtual ~IWindowOperator() { }
    };
    typedef ::boost::shared_ptr< IWindowOperator >  PWindowOperator;

    //=====================================================================
    //= IWindowEventFilter
	//=====================================================================
    /** an abstract interface for deciding whether a ->VclSimpleEvent
        is worth paying attention to
    */
    class IWindowEventFilter
    {
    public:
        virtual bool payAttentionTo( const VclSimpleEvent& _rEvent ) const = 0;

        virtual ~IWindowEventFilter() { }
    };
    typedef ::boost::shared_ptr< IWindowEventFilter >   PWindowEventFilter;

    //=====================================================================
    //= DialogController
	//=====================================================================
    /** a class controlling interactions between dialog controls

        An instance of this class listens to all events fired by a certain
        ->Control (more precise, a ->Window), the so-called instigator.

        Additionally, the ->DialogController maintains a list of windows which
        are affected by changes in the instigator window. Let's call those the
        dependent windows.

        Now, by help of an owner-provided ->IWindowEventFilter, the ->DialogController
        decides which events are worth attention. By help of an owner-provided
        ->IWindowOperator, it handles those events for all dependent windows.
    */
    class DialogController
    {
    private:
        Window&                     m_rInstigator;
        ::std::vector< Window* >    m_aConcernedWindows;
        const PWindowEventFilter    m_pEventFilter;
        const PWindowOperator       m_pOperator;

    public:
        DialogController( Window& _rInstigator, const PWindowEventFilter _pEventFilter, const PWindowOperator _pOperator );
        virtual ~DialogController();

        /** adds a window to the list of dependent windows

            @param _rWindow
                The window to add to the list of dependent windows.

                The caller is responsible for life-time control: The given window
                must live at least as long as the ->DialogController instance does.
        */
        void    addDependentWindow( Window& _rWindow );

    private:
        void    impl_Init();
        void    impl_updateAll();
        void    impl_update( Window& _rWindow );

        DECL_LINK( OnWindowEvent, const VclSimpleEvent* );
    };

    //=====================================================================
    //= EnableOnCheck
	//=====================================================================
    /** a helper class implementing the ->IWindowOperator interface,
        which enables a dependent window depending on the check state of
        an instigator window.

        @see DialogController
    */
    template< class CHECKABLE >
    class EnableOnCheck : public IWindowOperator
    {
    public:
        typedef CHECKABLE   SourceType;

    private:
        SourceType& m_rCheckable;
        BOOL        m_bAlwaysDisabled;

    public:
        /** constructs the instance

            @param _rCheckable
                a ->Window instance which supports a boolean method IsCheckable. Usually
                a ->RadioButton or ->CheckBox
            @param _bAlwaysDisabled
                a enable flag ANDed with check state of _rCheckable. This might be useful
                when you know that all your controls shall be disabled, regardless of
                any check states of your instigating controls (for instance, because
                your whole dialog is disabled). In this case, just pass <TRUE/> here.
        */
        EnableOnCheck( SourceType& _rCheckable, BOOL _bAlwaysDisabled = FALSE )
            :m_rCheckable( _rCheckable )
            ,m_bAlwaysDisabled( _bAlwaysDisabled )
        {
        }

        virtual void operateOn( Window& _rOperateOn ) const
        {
            _rOperateOn.Enable( m_bAlwaysDisabled ? FALSE : m_rCheckable.IsChecked() );
        }
    };

    //=====================================================================
    //= FilterForRadioOrCheckToggle
	//=====================================================================
    /** a helper class implementing the ->IWindowEventFilter interface,
        which filters for radio buttons or check boxes being toggled.

        Technically, the class simply filters for the ->VCLEVENT_RADIOBUTTON_TOGGLE
        and the ->VCLEVENT_CHECKBOX_TOGGLE event.
    */
    class FilterForRadioOrCheckToggle : public IWindowEventFilter
    {
        const Window&   m_rWindow;
    public:
        FilterForRadioOrCheckToggle( const Window& _rWindow )
            :m_rWindow( _rWindow )
        {
        }

        bool payAttentionTo( const VclSimpleEvent& _rEvent ) const
        {
            const VclWindowEvent& rWindowEvent = dynamic_cast< const VclWindowEvent& >( _rEvent );
            if  (   ( rWindowEvent.GetWindow() == &m_rWindow )
                &&  (   ( rWindowEvent.GetId() == VCLEVENT_RADIOBUTTON_TOGGLE )
                    ||  ( rWindowEvent.GetId() == VCLEVENT_CHECKBOX_TOGGLE )
                    )
                )
                return true;
            return false;
        }
    };

    //=====================================================================
    //= RadioDependentEnabler
	//=====================================================================
    /** a ->DialogController derivee which enables or disables its dependent windows,
        depending on the check state of a radio button.

        The usage of this class is as simple as
        <code>
        pController = new RadioDependentEnabler( m_aOptionSelectSomething );
        pController->addDependentWindow( m_aLabelSelection );
        pController->addDependentWindow( m_aListSelection );
        </code>

        With this, both <code>m_aLabelSelection</code> and <code>m_aListSelection</code> will
        be disabled if and only <code>m_aOptionSelectSomething</code> is checked.
    */
    class RadioDependentEnabler : public DialogController
    {
    public:
        RadioDependentEnabler( RadioButton& _rButton, BOOL _bAlwaysDisabled = FALSE )
            :DialogController( _rButton,
                PWindowEventFilter( new FilterForRadioOrCheckToggle( _rButton ) ),
                PWindowOperator( new EnableOnCheck< RadioButton >( _rButton, _bAlwaysDisabled ) ) )
        {
        }
    };

//........................................................................
} // namespace dbaui
//........................................................................

#endif // DBACCESS_SOURCE_UI_INC_DIALOGCONTROLLING_HXX

