/*************************************************************************
 *
 *  $RCSfile: speedctl.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:51:06 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifdef WNT
#include <tools/svwin.h>
#endif // WNT

#include <math.h>
#include <tools/time.hxx>
#include <vcl/poly.hxx>
#include <vcl/salbtype.hxx>
#include <vcl/svapp.hxx>
#include <vcl/window.hxx>
#include "speedctl.hxx"

// -----------
// - Defines -
// -----------

#define SPEED_TIMEDIST_MS			40
#define SPEEDCTL_MIN_INC			0.001
#define SPEED_RESCHEDULE			5
#define TIME_TO_MSEC( _aTime )		((_aTime).Seconds*1000UL+(_aTime).Nanosec/1000000)

// --------------
// - DistStruct -
// --------------

struct DistInfo
{
	Point	maPt;
	double	mfDist;
};

// ----------------
// - SpeedControl -
// ----------------

SpeedControl::SpeedControl( Window* pWin ) :
	mpWin			( pWin ),
	mpDist			( NULL ),
	mfUnitsPerSec	( 0. ),
	mfCurStep		( 0. ),
	mfCurSteps		( 0. ),
	mnMinDistInc	( 0UL ),
	mnLastTime		( 0UL ),
	mnLastStep		( 0UL ),
	mnActStep		( 0UL )
{
}

// ------------------------------------------------------------------------

SpeedControl::SpeedControl( const Polygon& rPath, Window* pWin ) :
	mpWin			( pWin ),
	mfUnitsPerSec	( 0. ),
	mfCurStep		( 0. ),
	mfCurSteps		( 0. ),
	mnMinDistInc	( 0UL ),
	mnMaxDistInc	( 0UL ),
	mnLastTime		( 0UL ),
	mnLastStep		( 0UL ),
	mnActStep		( 0UL ),
	mnActDist		( 0UL )
{
	mpDist = new DistInfo[ mnDistCount = rPath.GetSize() ];
	ImplCalcDistances( rPath );
}

// ------------------------------------------------------------------------

void SpeedControl::ImplCalcDistances( const Polygon& rPath )
{
	const USHORT nCount = rPath.GetSize();

	if( nCount )
	{
		Point		aLast( rPath[ 0 ] );
		DistInfo*	pTmp = mpDist;

		pTmp->maPt = aLast;
		pTmp->mfDist = 0.0;
		pTmp++;

		for( USHORT i = 1; i < nCount; i++, pTmp++ )
		{
			const Point&	rAct = rPath[ i ];
			const double	fDistX = rAct.X() - aLast.X();
			const double	fDistY = rAct.Y() - aLast.Y();

			pTmp->maPt = rAct;
			pTmp->mfDist = ( pTmp - 1 )->mfDist + sqrt( fDistX * fDistX + fDistY * fDistY );
			aLast = rAct;
		}
	}
	else
	{
		mnDistCount = 0UL;
		delete mpDist;
		mpDist = NULL;
	}
}

// ------------------------------------------------------------------------

SpeedControl::~SpeedControl()
{
	if( mpDist )
		delete[] mpDist;
}

// ------------------------------------------------------------------------

void SpeedControl::Reset( ULONG nUnitsPerSec, ULONG nMinDistInc, ULONG nMaxDistInc )
{
	mfUnitsPerSec = nUnitsPerSec;
	mfCurStep =	mfLastStep = 0.0;
	mfCurSteps = mfLastSteps = 0.;
	mnLastTime = 0UL;
	mnActStep = mnLastStep = 0UL;
	mnActDist = 0UL;
	mnMinDistInc = nMinDistInc;
	mnMaxDistInc = nMaxDistInc;
	mnStepCount = 0;
}

// ------------------------------------------------------------------------

ULONG SpeedControl::GetNextStep()
{
	static ULONG nTimeCount = 0UL;

	if( !mnLastTime )
		mnLastTime = Time::GetSystemTicks();

	const ULONG nActTime = Time::GetSystemTicks();
	const ULONG	nDiff = nActTime - mnLastTime;

	if( nDiff >= SPEED_TIMEDIST_MS )
	{
		const double mfCurSpeed = ( mfCurSteps - mfLastSteps ) * 1000. / nDiff;

		mfLastStep = mfCurStep;

		if( mfCurSpeed > 0. )
			mfCurStep *= ( mfUnitsPerSec / mfCurSpeed );
		else if( mnMinDistInc )
			mfCurStep = mnMinDistInc;
		else
			mfCurStep = SPEEDCTL_MIN_INC;

		if( mnMaxDistInc && ( mfCurStep > (double) mnMaxDistInc ) )
			mfCurStep = mnMaxDistInc;

		mfCurStep = ( mfCurStep + mfLastStep ) * 0.5;
		mfLastSteps = mfCurSteps;

		// reschedule only a few times
/*		if( !( nTimeCount++ % SPEED_RESCHEDULE ) )
			Application::Reschedule();
*/
		mnLastTime = Time::GetSystemTicks();
	}

	mfCurSteps += mfCurStep;
	mnLastStep = mnActStep;
	mnActStep = (ULONG) ( mfCurSteps + .5 );
	mnStepCount++;

	if( mpWin )
		mpWin->Sync();

#ifdef WNT
		Sleep( 1 );
#endif // WNT

	return( mnActStep - mnLastStep );
}

// ------------------------------------------------------------------------

BOOL SpeedControl::GetNextPathPoint( Point& rPt )
{
	BOOL bRet = FALSE;

	if( mpDist )
	{
		BOOL bFound = FALSE;

		if( !mnLastTime )
		{
			rPt = mpDist[ 0 ].maPt;
			bRet = TRUE;
		}
		else
		{
			for( ULONG i = mnActDist + 1; i < mnDistCount; i++, mnActDist++ )
			{
				const double	fActStep = mnActStep;
				const DistInfo& rAct = mpDist[ i ];

				if( rAct.mfDist >= fActStep )
				{
					const DistInfo&	rLast = mpDist[ i - 1UL ];
					const Point&	rStart = rLast.maPt;
					const Point&	rEnd = rAct.maPt;
					const double	fCurDist = rAct.mfDist - rLast.mfDist;

					if( fCurDist != 0.0 )
					{
						const double fFact = ( fActStep - rLast.mfDist ) / fCurDist;

						rPt.X() = rStart.X() + FRound( fFact * ( rEnd.X() - rStart.X() ) );
						rPt.Y() = rStart.Y() + FRound( fFact * ( rEnd.Y() - rStart.Y() ) );
					}
					else
						rPt = rStart;

					bRet = TRUE;
					break;
				}
			}
		}

		if( bRet )
			GetNextStep();
		else
			rPt = mpDist[ mnDistCount - 1UL ].maPt;
	}
	else
		rPt = Point();

	return bRet;
}
