/*************************************************************************
 *
 *  $RCSfile: profile.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mfe $ $Date: 2001/03/12 17:27:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef WNT
#include <io.h>
#elif UNX
#include <unistd.h>
#endif

#include "jsnative.hxx"
#include "profile.hxx"

#define _OSL_VERIFY(c, f, l)		((void)(c))
#define OSL_VERIFY(c)				_OSL_VERIFY(c, OSL_THIS_FILE, __LINE__)

#define _PATH_MAX		16*1024

#define LINES_INI       32
#define LINES_ADD       10
#define SECTIONS_INI    5
#define SECTIONS_ADD    3
#define ENTRIES_INI     5
#define ENTRIES_ADD     3

#define STR_INI_EXTENSION	"rc"
#define STR_INI_METAHOME	"?~"
#define STR_INI_METASYS		"?$"
#define STR_INI_METACFG		"?^"
#define STR_INI_METAINS		"?#"

#define STR_INI_BOOLYES     "yes"
#define STR_INI_BOOLON      "on"
#define STR_INI_BOOLONE     "1"
#define STR_INI_BOOLNO      "no"
#define STR_INI_BOOLOFF     "off"
#define STR_INI_BOOLZERO    "0"

#define FLG_USER			0x00FF
#define FLG_AUTOOPEN		0x0100
#define FLG_MODIFIED		0x0200

#define SVERSION_LOCATION   STR_INI_METACFG
#define SVERSION_FALLBACK   STR_INI_METASYS
#define SVERSION_NAME   	"sversion"
#define SVERSION_SECTION    "Versions"
#define SVERSION_SOFFICE    "StarOffice"
#define SVERSION_PROFILE    "sofficerc"
#define SVERSION_OPTION     "userid:"
#define SVERSION_DIRS		{ "bin", "program" }
#define SVERSION_USER       "user"

#ifdef WNT
#define DEFAULT_PMODE   (_S_IREAD | _S_IWRITE)
#endif

#ifdef UNX
#define DEFAULT_PMODE 	(S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#endif

#define _BUILD_STR_(n)	# n
#define BUILD_STR(n)	_BUILD_STR_(n)


/*****************************************************************************/
/* Data Type Definition */
/*****************************************************************************/

typedef time_t  osl_TStamp;

typedef enum _osl_TLockMode
{
	un_lock, read_lock, write_lock
} osl_TLockMode;

typedef struct _osl_TFile
{
	int				m_Handle;
	char*			m_pReadPtr;
	char			m_ReadBuf[100*1024];
    char*  			m_pWriteBuf;
    unsigned long 	m_nWriteBufLen;
    unsigned long 	m_nWriteBufFree;
} osl_TFile;

typedef struct _osl_TProfileEntry
{
    unsigned long	m_Line;
    unsigned long 	m_Offset;
    unsigned long 	m_Len;
} osl_TProfileEntry;

typedef struct _osl_TProfileSection
{
    unsigned long	m_Line;
    unsigned long	m_Offset;
    unsigned long	m_Len;
    unsigned long	m_NoEntries;
    unsigned long	m_MaxEntries;
    osl_TProfileEntry*	m_Entries;
} osl_TProfileSection;


typedef struct _osl_TProfileImpl
{
	unsigned long	m_Flags;
	osl_TFile*	m_pFile;
	osl_TStamp	m_Stamp;
	char    m_Filename[_PATH_MAX + 1];
	unsigned long	m_NoLines;
	unsigned long  m_MaxLines;
	unsigned long  m_NoSections;
	unsigned long  m_MaxSections;
	char**	m_Lines;
	osl_TProfileSection* m_Sections;
    BOOL    m_bIsValid;
} osl_TProfileImpl;


/*****************************************************************************/
/* Static Module Function Declarations */
/*****************************************************************************/

static osl_TFile*	OslProfile_openFile(const char* pszFilename, oslProfileOption ProfileFlags);
static osl_TStamp	OslProfile_closeFile(osl_TFile* pFile, oslProfileOption Flags);
static BOOL  		OslProfile_rewindFile(osl_TFile* pFile, BOOL bTruncate, char* pFilename);
static osl_TStamp	OslProfile_getFileStamp(osl_TFile* pFile);

static char*   		OslProfile_getLine(osl_TFile* pFile);
static BOOL   		OslProfile_putLine(osl_TFile* pFile, const char *pszLine);
static char* 		stripBlanks(char* String, unsigned long* pLen);
static char* 		addLine(osl_TProfileImpl* pProfile, const char* Line);
static char* 		insertLine(osl_TProfileImpl* pProfile, const char* Line, unsigned long LineNo);
static void 		removeLine(osl_TProfileImpl* pProfile, unsigned long LineNo);
static void			setEntry(osl_TProfileImpl* pProfile, osl_TProfileSection* pSection,
                     	unsigned long NoEntry, unsigned long Line,
                     	char* Entry, unsigned long Len);
static BOOL addEntry(osl_TProfileImpl* pProfile, osl_TProfileSection *pSection,
                         int Line, char* Entry, unsigned long Len);
static void removeEntry(osl_TProfileSection *pSection, unsigned long NoEntry);
static BOOL addSection(osl_TProfileImpl* pProfile, int Line, const char* Section, unsigned long Len);
static void removeSection(osl_TProfileImpl* pProfile, osl_TProfileSection *pSection);
static osl_TProfileSection* findEntry(osl_TProfileImpl* pProfile, const char* Section,
                                      const char* Entry, unsigned long *pNoEntry);
static BOOL loadProfile(osl_TFile* pFile, osl_TProfileImpl* pProfile);
static BOOL storeProfile(osl_TFile* pFile, osl_TProfileImpl* pProfile, BOOL bCleanup);
static osl_TProfileImpl* acquireProfile(oslProfile Profile, BOOL bWriteable);
static BOOL releaseProfile(osl_TProfileImpl* pProfile);
static BOOL lookupProfile(const char *pszPath, const char *pszFile, char *pPath);

static BOOL writeProfileImpl (osl_TFile* pFile);

oslProfile osl_psz_openProfile(const char *pszProfileName, oslProfileOption Flags);
BOOL osl_psz_getProfileName(const char* pszPath, const char* pszName,
                                         char* pszBuffer, unsigned long MaxLen);


/*****************************************************************************/
/* Exported Module Functions */
/*****************************************************************************/

oslProfile osl_openProfile(char *pszProfileName, oslProfileOption Options)
{
    oslProfile pProfile=0;
    pProfile = osl_psz_openProfile(pszProfileName,Options);
    return pProfile;
}


oslProfile osl_psz_openProfile(const char *pszProfileName, oslProfileOption Flags)
{
	osl_TFile*		  pFile;
	osl_TProfileImpl* pProfile;
	char          Filename[_PATH_MAX] = "";
    BOOL bRet = FALSE;

    if (pszProfileName == NULL)
	{
		OSL_VERIFY(osl_psz_getProfileName(NULL, NULL, Filename, sizeof(Filename)));
		pszProfileName = Filename;
    }



	if ( ( pFile = OslProfile_openFile(pszProfileName, Flags ) ) == NULL )
    {
		return (NULL);
    }


	pProfile = (osl_TProfileImpl*)calloc(1, sizeof(osl_TProfileImpl));

    if ( pProfile == 0 )
    {
        return 0;
    }

	pProfile->m_Flags = Flags & FLG_USER;

	if ( Flags & ( osl_Profile_READLOCK | osl_Profile_WRITELOCK | osl_Profile_FLUSHWRITE ) )
    {
		pProfile->m_pFile = pFile;
    }

    pProfile->m_bIsValid=TRUE;

	pProfile->m_Stamp = OslProfile_getFileStamp(pFile);
	bRet=loadProfile(pFile, pProfile);

	strcpy( pProfile->m_Filename, pszProfileName );

	if (pProfile->m_pFile == NULL)
		OslProfile_closeFile(pFile,pProfile->m_Flags);

	return (pProfile);
}

BOOL osl_closeProfile(oslProfile Profile)
{
	osl_TProfileImpl* pProfile = (osl_TProfileImpl*)Profile;
    BOOL bRet = FALSE;

    if ( Profile == 0 )
    {
        return FALSE;
    }


    if ( pProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

    pProfile->m_bIsValid=FALSE;

	if (pProfile->m_Flags & FLG_MODIFIED)
	{
        pProfile = acquireProfile(Profile,TRUE);

        if ( pProfile != 0 )
        {
			bRet=storeProfile(pProfile->m_pFile, pProfile, TRUE);
		}
	}
	else
	{
		pProfile = acquireProfile(Profile,FALSE);
	}


    if ( pProfile == 0 )
    {
        return FALSE;
    }

    if (pProfile->m_pFile != NULL)
        OslProfile_closeFile(pProfile->m_pFile,pProfile->m_Flags);

    pProfile->m_pFile = NULL;
    pProfile->m_Filename[0] = '\0';

	if ( pProfile->m_NoLines > 0)
	{
		unsigned int index=0;
		if ( pProfile->m_Lines != 0 )
		{
			for ( index = 0 ; index < pProfile->m_NoLines ; ++index)
			{
				if ( pProfile->m_Lines[index] != 0 )
				{
					free(pProfile->m_Lines[index]);
                    pProfile->m_Lines[index]=0;
				}
			}
			free(pProfile->m_Lines);
            pProfile->m_Lines=0;
		}
		if ( pProfile->m_Sections != 0 )
		{
			for ( index = 0 ; index < pProfile->m_NoSections ; ++index )
			{
				if ( pProfile->m_Sections[index].m_Entries != 0 )
				{
					free(pProfile->m_Sections[index].m_Entries);
                    pProfile->m_Sections[index].m_Entries=0;
				}
			}
			free(pProfile->m_Sections);
            pProfile->m_Sections=0;
		}
	}

	free(pProfile);

	return (TRUE);
}


BOOL osl_flushProfile(oslProfile Profile)
{
	osl_TProfileImpl* pProfile = (osl_TProfileImpl*) Profile;
	osl_TFile* pFile;
    BOOL bRet = FALSE;

    if ( pProfile == 0 )
    {
        return FALSE;
    }


    if ( pProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

	pFile = pProfile->m_pFile;
    if ( !( pFile != 0 && pFile->m_Handle >= 0 ) )
    {
        return FALSE;
    }

	if ( pProfile->m_Flags & FLG_MODIFIED )
	{
		bRet = storeProfile(pFile,pProfile,FALSE);
	}

    return bRet;
}

static BOOL writeProfileImpl(osl_TFile* pFile)
{
	int BytesWritten=0;

    if ( !( pFile != 0 && pFile->m_Handle >= 0 ) || ( pFile->m_pWriteBuf == 0 ) )
    {
        return FALSE;
    }

    BytesWritten = write(pFile->m_Handle, pFile->m_pWriteBuf, pFile->m_nWriteBufLen - pFile->m_nWriteBufFree);

    if ( BytesWritten <= 0 )
    {
        return (FALSE);
    }


    free(pFile->m_pWriteBuf);
    pFile->m_pWriteBuf=0;
	pFile->m_nWriteBufLen=0;
	pFile->m_nWriteBufFree=0;
    return TRUE;
}


BOOL osl_readProfileString(oslProfile Profile,
                              const char* pszSection, const char* pszEntry,
                              char* pszString, unsigned long MaxLen,
                              const char* pszDefault)
{
    unsigned long    NoEntry;
    char* pStr=0;
    osl_TProfileSection* pSec;
	osl_TProfileImpl*    pProfile=0;
	osl_TProfileImpl*    pTmpProfile=0;
    BOOL bRet = FALSE;

    pTmpProfile = (osl_TProfileImpl*) Profile;

    if ( pTmpProfile == 0 )
    {
        return FALSE;
    }


    if ( pTmpProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

    pProfile = acquireProfile(Profile, FALSE);

    if ( pProfile == NULL )
    {
        return (FALSE);
    }

	if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
	{
        if (((pSec = findEntry(pProfile, pszSection, pszEntry, &NoEntry)) != NULL) &&
            (NoEntry < pSec->m_NoEntries) &&
            ((pStr = strchr(pProfile->m_Lines[pSec->m_Entries[NoEntry].m_Line],
                            '=')) != NULL))
        {
            pStr++;
        }
        else
        {
            pStr = (char*)pszDefault;
        }

        if ( pStr != 0 )
        {
            pStr = stripBlanks(pStr, NULL);
            MaxLen = (MaxLen - 1 < strlen(pStr)) ? (MaxLen - 1) : strlen(pStr);
            pStr = stripBlanks(pStr, &MaxLen);
            strncpy(pszString, pStr, MaxLen);
            pszString[MaxLen] = '\0';
        }
    }
    else
    { /* not implemented */ }


    bRet=releaseProfile(pProfile);

	if ( pStr == 0 )
    {
		return FALSE;
    }

    return (TRUE);
}


BOOL osl_readProfileBool(oslProfile Profile,
                            const char* pszSection, const char* pszEntry,
							BOOL Default)
{
	char Line[32] = "";

    if (osl_readProfileString(Profile, pszSection, pszEntry, Line, sizeof(Line), ""))
    {
        if ((strcmp(Line, STR_INI_BOOLYES) == 0) ||
            (strcmp(Line, STR_INI_BOOLON)  == 0) ||
            (strcmp(Line, STR_INI_BOOLONE) == 0))
            Default = TRUE;
        else
            if ((strcmp(Line, STR_INI_BOOLNO)   == 0) ||
                (strcmp(Line, STR_INI_BOOLOFF)  == 0) ||
                (strcmp(Line, STR_INI_BOOLZERO) == 0))
                Default = FALSE;
    }

	return (Default);
}


unsigned long osl_readProfileIdent(oslProfile Profile,
                              const char* pszSection, const char* pszEntry,
							  unsigned long FirstId, const char* Strings[],
							  unsigned long Default)
{
    unsigned long	i;
	char		Line[256] = "";

    if (osl_readProfileString(Profile, pszSection, pszEntry, Line, sizeof(Line), ""))
    {
        i = 0;
        while (Strings[i] != NULL)
        {
            if (strcmp(Line, Strings[i]) == 0)
            {
                Default = i + FirstId;
                break;
            }
            i++;
        }
    }

    return (Default);
}

BOOL osl_writeProfileString(oslProfile Profile,
                               const char* pszSection, const char* pszEntry,
							   const char* pszString)
{
    unsigned long	i;
    BOOL bRet = FALSE;
    unsigned long    NoEntry;
    char* pStr;
	char		Line[1024] = "";
    osl_TProfileSection* pSec;
	osl_TProfileImpl*    pProfile = 0;
	osl_TProfileImpl*    pTmpProfile = 0;

    pTmpProfile = (osl_TProfileImpl*) Profile;

    if ( pTmpProfile == 0 )
    {
        return FALSE;
    }


    if ( pTmpProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

    pProfile=acquireProfile(Profile, TRUE);

    if (pProfile == NULL)
    {
        return (FALSE);
    }


	if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
	{
        if ((pSec = findEntry(pProfile, pszSection, pszEntry, &NoEntry)) == NULL)
        {
            Line[0] = '\0';
            addLine(pProfile, Line);

            Line[0] = '[';
            strcpy(&Line[1], pszSection);
            Line[1 + strlen(pszSection)] = ']';
            Line[2 + strlen(pszSection)] = '\0';

            if (((pStr = addLine(pProfile, Line)) == NULL) ||
                (! addSection(pProfile, pProfile->m_NoLines - 1, &pStr[1], strlen(pszSection))))
            {
                bRet=releaseProfile(pProfile);
                return (FALSE);
            }

            pSec = &pProfile->m_Sections[pProfile->m_NoSections - 1];
            NoEntry = pSec->m_NoEntries;
        }

        Line[0] = '\0';
        strcpy(&Line[0], pszEntry);
        Line[0 + strlen(pszEntry)] = '=';
        strcpy(&Line[1 + strlen(pszEntry)], pszString);

        if (NoEntry >= pSec->m_NoEntries)
        {
            if (pSec->m_NoEntries > 0)
                i = pSec->m_Entries[pSec->m_NoEntries - 1].m_Line + 1;
            else
                i = pSec->m_Line + 1;

            if (((pStr = insertLine(pProfile, Line, i)) == NULL) ||
                (! addEntry(pProfile, pSec, i, pStr, strlen(pszEntry))))
            {
                bRet=releaseProfile(pProfile);

                return (FALSE);
            }

            pProfile->m_Flags |= FLG_MODIFIED;
        }
        else
        {
            i = pSec->m_Entries[NoEntry].m_Line;
            free(pProfile->m_Lines[i]);
            pProfile->m_Lines[i] = strdup(Line);
            setEntry(pProfile, pSec, NoEntry, i, pProfile->m_Lines[i], strlen(pszEntry));

            pProfile->m_Flags |= FLG_MODIFIED;
        }
    }
    else {
        /* not implemented */
    }

    bRet = releaseProfile(pProfile);
	return bRet;
}


BOOL osl_writeProfileBool(oslProfile Profile,
                             const char* pszSection, const char* pszEntry,
							 BOOL Value)
{
    BOOL bRet=FALSE;

    if (Value)
        bRet=osl_writeProfileString(Profile, pszSection, pszEntry, STR_INI_BOOLONE);
    else
        bRet=osl_writeProfileString(Profile, pszSection, pszEntry, STR_INI_BOOLZERO);

    return bRet;
}


BOOL osl_writeProfileIdent(oslProfile Profile,
                              const char* pszSection, const char* pszEntry,
							  unsigned long FirstId, const char* Strings[],
							  unsigned long Value)
{
    int i, n;
    BOOL bRet=FALSE;

    for (n = 0; Strings[n] != NULL; n++);

    if ((i = Value - FirstId) >= n)
        bRet=FALSE;
    else
        bRet = osl_writeProfileString(Profile, pszSection, pszEntry, Strings[i]);
    return bRet;
}


BOOL osl_removeProfileEntry(oslProfile Profile,
                               const char *pszSection, const char *pszEntry)
{
    unsigned long    NoEntry;
    osl_TProfileSection* pSec;
	osl_TProfileImpl*    pProfile = 0;
	osl_TProfileImpl*    pTmpProfile = 0;
    BOOL bRet = FALSE;

    pTmpProfile = (osl_TProfileImpl*) Profile;

    if ( pTmpProfile == 0 )
    {
        return FALSE;
    }



    if ( pTmpProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }


    pProfile = acquireProfile(Profile, TRUE);

    if (pProfile == NULL)
    {
    	return (FALSE);
    }


	if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
	{
        if (((pSec = findEntry(pProfile, pszSection, pszEntry, &NoEntry)) != NULL) &&
            (NoEntry < pSec->m_NoEntries))
        {
            removeLine(pProfile, pSec->m_Entries[NoEntry].m_Line);
            removeEntry(pSec, NoEntry);
            if (pSec->m_NoEntries == 0)
            {
                removeLine(pProfile, pSec->m_Line);
                if ((pSec->m_Line > 0) && (pProfile->m_Lines[pSec->m_Line - 1][0] == '\0'))
                    removeLine(pProfile, pSec->m_Line - 1);

                removeSection(pProfile, pSec);
            }

            pProfile->m_Flags |= FLG_MODIFIED;
        }
    }
    else
    { /* not implemented */ }


    bRet = releaseProfile(pProfile);
	return bRet;
}


unsigned long osl_getProfileSectionEntries(oslProfile Profile, const char *pszSection,
									  char* pszBuffer, unsigned long MaxLen)
{
    unsigned long    i, n = 0;
    unsigned long    NoEntry;
    osl_TProfileSection* pSec;
	osl_TProfileImpl*    pProfile = 0;
	osl_TProfileImpl*    pTmpProfile = 0;
    BOOL bRet = FALSE;

    pTmpProfile = (osl_TProfileImpl*) Profile;

    if ( pTmpProfile == 0 )
    {
        return FALSE;
    }



    if ( pTmpProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

    pProfile = acquireProfile(Profile, FALSE);

    if (pProfile == NULL)
    {
        return (0);
    }


	if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
	{
        if ((pSec = findEntry(pProfile, pszSection, "", &NoEntry)) != NULL)
        {
            if (MaxLen != 0)
            {
                for (i = 0; i < pSec->m_NoEntries; i++)
                {
                    if ((n + pSec->m_Entries[i].m_Len + 1) < MaxLen)
                    {
                        strncpy(&pszBuffer[n], &pProfile->m_Lines[pSec->m_Entries[i].m_Line]
                                [pSec->m_Entries[i].m_Offset], pSec->m_Entries[i].m_Len);
                        n += pSec->m_Entries[i].m_Len;
                        pszBuffer[n++] = '\0';
                    }
                    else
                        break;

                }

                pszBuffer[n++] = '\0';
            }
            else
            {
                for (i = 0; i < pSec->m_NoEntries; i++)
                    n += pSec->m_Entries[i].m_Len + 1;

                n += 1;
            }
        }
        else
            n = 0;
    }
    else {
        /* not implemented */
    }

	bRet=releaseProfile(pProfile);
    return (n);
}

unsigned long getSectionCount(oslProfile Profile)
{
	osl_TProfileImpl* pProfile = (osl_TProfileImpl*)Profile;
	return pProfile->m_NoSections;
}

unsigned long osl_getProfileSections(oslProfile Profile, char* pszBuffer, unsigned long MaxLen)
{
	unsigned long    i, n = 0;
	osl_TProfileSection* pSec;
	osl_TProfileImpl*    pProfile = 0;
	osl_TProfileImpl*    pTmpProfile = 0;
    BOOL bRet = FALSE;

    pTmpProfile = (osl_TProfileImpl*) Profile;

    if ( pTmpProfile == 0 )
    {
        return FALSE;
    }



    if ( pTmpProfile->m_bIsValid == FALSE )
    {
        return FALSE;
    }

    pProfile = acquireProfile(Profile, FALSE);

	if (pProfile == NULL)
    {
        return (0);
    }

	if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
	{
		if (MaxLen != 0)
		{
			for (i = 0; i < pProfile->m_NoSections; i++)
			{
				pSec = &pProfile->m_Sections[i];

				if ((n + pSec->m_Len + 1) < MaxLen)
				{
					strncpy(&pszBuffer[n], &pProfile->m_Lines[pSec->m_Line][pSec->m_Offset],
						    pSec->m_Len);
					n += pSec->m_Len;
					pszBuffer[n++] = '\0';
				}
				else
					break;
			}

			pszBuffer[n++] = '\0';
		}
		else
		{
			for (i = 0; i < pProfile->m_NoSections; i++)
				n += pProfile->m_Sections[i].m_Len + 1;

			n += 1;
		}
	}
	else
	{ /* not implemented */ }


    bRet=releaseProfile(pProfile);
	return (n);
}


BOOL osl_psz_getProfileName(const char* pszPath, const char* pszName, char* pszBuffer, unsigned long MaxLen)
{
	char *pChr=0;
    const char *pStr=0;
	char  Home[_PATH_MAX] = "";
	char  Config[_PATH_MAX] = "";
	char  Path[_PATH_MAX] = "";
	char  File[_PATH_MAX] = "";
	BOOL  bHidden = FALSE;

	if (Home[strlen(Home) - 1] != '/')
    {
        strcat(Home, "/");
    }

	if (Config[strlen(Config) - 1] != '/')
    {
        strcat(Config, "/");
    }


	if (pszName == NULL)
	{
	}
	else
	{
		pStr = pszName;
	}

	strcpy(File, pStr);

	if (strchr(File + 1, '.') == NULL)
		strcat(File, STR_INI_EXTENSION);

	if (pszPath)
	{
		char  Dir[_PATH_MAX] = "";
		char  Loc[_PATH_MAX] = "";

		if ((strncmp(pszPath, STR_INI_METAHOME, sizeof(STR_INI_METAHOME) - 1) == 0) &&
		    ((pszPath[sizeof(STR_INI_METAHOME) - 1] == '\0') ||
		     (pszPath[sizeof(STR_INI_METAHOME) - 1] == '/')))
		{
			strcpy(Dir, Home);
			if (pszPath[sizeof(STR_INI_METAHOME) - 1] == '/')
				strcat(Dir, pszPath + sizeof(STR_INI_METAHOME));
			pszPath = Dir;
		}
		else if ((strncmp(pszPath, STR_INI_METACFG, sizeof(STR_INI_METACFG) - 1) == 0) &&
		         ((pszPath[sizeof(STR_INI_METACFG) - 1] == '\0') ||
		         (pszPath[sizeof(STR_INI_METACFG) - 1] == '/')))
		{
			strcpy(Dir, Config);
			if (pszPath[sizeof(STR_INI_METACFG) - 1] == '/')
				strcat(Dir, pszPath + sizeof(STR_INI_METACFG));
			pszPath = Dir;
		}
		else if ((strncmp(pszPath, STR_INI_METASYS, sizeof(STR_INI_METASYS) - 1) == 0) &&
		         ((pszPath[sizeof(STR_INI_METASYS) - 1] == '\0') ||
		         (pszPath[sizeof(STR_INI_METASYS) - 1] == '/')))
		{
			strcpy(Dir, "/etc/");
			if (pszPath[sizeof(STR_INI_METASYS) - 1] == '/')
				strcat(Dir, pszPath + sizeof(STR_INI_METASYS));
			pszPath = Dir;
		}
		else if ((strncmp(pszPath, STR_INI_METAINS, sizeof(STR_INI_METAINS) - 1) == 0) &&
		         ((pszPath[sizeof(STR_INI_METAINS) - 1] == '\0') ||
		          (pszPath[sizeof(STR_INI_METAINS) - 1] == '/') ||
		          (pszPath[sizeof(STR_INI_METAINS) - 1] == '"')))
		{
			if (! lookupProfile(pszPath + sizeof(STR_INI_METAINS) - 1, File, Dir))
            {
				return (FALSE);
            }

			pszPath = Dir;
		}

		strcpy( Loc, Home );
		strcpy( Path, pszPath );
		if ( TRUE )
		{
			if (Path[strlen(Path) - 1] != '/') strcat(Path, "/");

			if (strcmp(Loc, Path) == 0)
				bHidden = TRUE;
		}

		strcpy(Path, pszPath);
		if (Path[strlen(Path) - 1] != '/') strcat(Path, "/");
	}
	else
	{
		if (strcmp(Home, Config) == 0)
			bHidden = TRUE;

		strcpy(Path, Config);
	}

	if (bHidden && (Path[0] != '.'))
		strcat(Path, ".");

	strcat(Path, File);

	strncpy(pszBuffer, Path, MaxLen);
	pszBuffer[MaxLen - 1] = '\0';

	return (strlen(File) < MaxLen);
}

/*****************************************************************************/
/* Static Module Functions */
/*****************************************************************************/

static osl_TStamp OslProfile_getFileStamp(osl_TFile* pFile)
{
	struct stat status;

	if ( (pFile->m_Handle < 0) || (fstat(pFile->m_Handle, &status) < 0) )
    {
		return (0);
    }


	return (status.st_mtime);
}

static osl_TFile* OslProfile_openFile(const char* pszFilename, oslProfileOption ProfileFlags )
{
#ifdef UNX
	int        Flags;
#endif
	osl_TFile* pFile = (osl_TFile*) calloc(1, sizeof(osl_TFile));
    BOOL bWriteable = FALSE;

    if ( ProfileFlags & ( osl_Profile_WRITELOCK | osl_Profile_FLUSHWRITE ) )
    {
        bWriteable=TRUE;
    }

	if (! bWriteable)
    {
        pFile->m_Handle = open(pszFilename, O_RDONLY);
    }
	else
    {
		if (((pFile->m_Handle = open(pszFilename, O_RDWR | O_CREAT | O_EXCL, DEFAULT_PMODE)) < 0) &&
			((pFile->m_Handle = open(pszFilename, O_RDWR)) < 0))
		{
			free(pFile);
			return (NULL);
		}
    }

#ifdef UNX
	if ((Flags = fcntl(pFile->m_Handle, F_GETFD, 0)) != -1)
	{
		Flags |= FD_CLOEXEC;
		fcntl(pFile->m_Handle, F_SETFD, Flags);
	}
#endif

    pFile->m_pWriteBuf=0;
    pFile->m_nWriteBufFree=0;
    pFile->m_nWriteBufLen=0;

    if ( ProfileFlags & (osl_Profile_WRITELOCK | osl_Profile_READLOCK ) )
    {
    }

	return (pFile);
}

static osl_TStamp OslProfile_closeFile(osl_TFile* pFile, oslProfileOption Flags)
{
	osl_TStamp stamp = 0;

    if (pFile == 0 )
    {
        return stamp;
    }

	if (pFile->m_Handle >= 0)
	{
		stamp = OslProfile_getFileStamp(pFile);

		close(pFile->m_Handle);
        pFile->m_Handle = -1;
	}


    if ( pFile->m_pWriteBuf )
    {
        free(pFile->m_pWriteBuf);
    }

	free(pFile);

	return(stamp);
}

static BOOL OslProfile_rewindFile(osl_TFile* pFile, BOOL bTruncate, char* pFilename)
{
    if (pFile->m_Handle >= 0)
	{
		pFile->m_pReadPtr  = pFile->m_ReadBuf + sizeof(pFile->m_ReadBuf);
		lseek(pFile->m_Handle, SEEK_SET, 0L);
		if (bTruncate)
		{
#ifdef UNX
			ftruncate(pFile->m_Handle, 0L);
#else
			close(pFile->m_Handle);
			pFile->m_Handle = open(pFilename, O_RDWR | O_TRUNC, DEFAULT_PMODE);
#endif
		}
	}
	return (TRUE);
}


static char* OslProfile_getLine(osl_TFile* pFile)
{
	int   Max, Free, Bytes, nLineBytes = 0;
	char* pChr;
	char* pLine = NULL;
	char* pNewLine;

	if ( pFile == 0 )
	{
		return 0;
	}

	if (pFile->m_Handle < 0)
		return NULL;

	do
	{
		Bytes = sizeof(pFile->m_ReadBuf) - (pFile->m_pReadPtr - pFile->m_ReadBuf);

		if (Bytes <= 1)
		{
			memcpy(pFile->m_ReadBuf, pFile->m_pReadPtr, Bytes);
			pFile->m_pReadPtr = pFile->m_ReadBuf;

			Free = sizeof(pFile->m_ReadBuf) - Bytes;

			if ((Max = read(pFile->m_Handle, &pFile->m_ReadBuf[Bytes], Free)) < 0)
			{
				if( pLine )
					delete pLine;
				pLine = NULL;
				break;
			}

			if (Max < Free)
			{
 				if ((Max == 0) && ! pLine)
 					break;

			 	pFile->m_ReadBuf[Bytes + Max] = '\0';
			}
		}

		for (pChr = pFile->m_pReadPtr;
		     (*pChr != '\n') && (*pChr != '\r') && (*pChr != '\0') &&
		     (pChr < (pFile->m_ReadBuf + sizeof(pFile->m_ReadBuf) - 1));
			 pChr++);

		Max = pChr - pFile->m_pReadPtr;
		pNewLine = (char*) new char [ nLineBytes + Max + 1 ];
		if( pLine )
		{
			memcpy( pNewLine, pLine, nLineBytes );
			delete pLine;
		}
		memcpy(pNewLine+nLineBytes, pFile->m_pReadPtr, Max);
		nLineBytes += Max;
		pNewLine[ nLineBytes ] = 0;
		pLine = pNewLine;

		if (pChr < (pFile->m_ReadBuf + sizeof(pFile->m_ReadBuf) - 1))
		{
			if (*pChr != '\0')
			{
				if ((pChr[0] == '\r') && (pChr[1] == '\n'))
					pChr += 2;
				else
					pChr += 1;
			}

			if ((pChr < (pFile->m_ReadBuf + sizeof(pFile->m_ReadBuf))) &&
			    (*pChr == '\0'))
				pChr = pFile->m_ReadBuf + sizeof(pFile->m_ReadBuf);

			Max = -1;
		}

		pFile->m_pReadPtr = pChr;
	}
	while (Max > 0);

	return pLine;
}

static BOOL OslProfile_putLine(osl_TFile* pFile, const char *pszLine)
{
	unsigned int Len = strlen(pszLine);

	if ( pFile == 0 || pFile->m_Handle < 0 )
    {
		return (FALSE);
    }

    if ( pFile->m_pWriteBuf == 0 )
    {
        pFile->m_pWriteBuf = (char*) malloc(Len+3);
        pFile->m_nWriteBufLen = Len+3;
		pFile->m_nWriteBufFree = Len+3;
    }
    else
    {
        if ( pFile->m_nWriteBufFree <= Len + 3 )
        {
            char* pTmp;

            pTmp=(char*) realloc(pFile->m_pWriteBuf,( ( pFile->m_nWriteBufLen + Len ) * 2) );
            if ( pTmp == 0 )
            {
                return FALSE;
            }
            pFile->m_pWriteBuf = pTmp;
            pFile->m_nWriteBufFree = pFile->m_nWriteBufFree + pFile->m_nWriteBufLen + ( 2 * Len );
            pFile->m_nWriteBufLen = ( pFile->m_nWriteBufLen + Len ) * 2;
            memset( (pFile->m_pWriteBuf) + ( pFile->m_nWriteBufLen - pFile->m_nWriteBufFree ), 0, pFile->m_nWriteBufFree);
        }
    }

	memcpy(pFile->m_pWriteBuf + ( pFile->m_nWriteBufLen - pFile->m_nWriteBufFree ),pszLine,Len+1);

	pFile->m_pWriteBuf[pFile->m_nWriteBufLen - pFile->m_nWriteBufFree + Len]='\n';
    pFile->m_pWriteBuf[pFile->m_nWriteBufLen - pFile->m_nWriteBufFree + Len + 1]='\0';

    pFile->m_nWriteBufFree-=Len+1;

	return TRUE;
}

static char* stripBlanks(char* String, unsigned long* pLen)
{
    if ( ( pLen != NULL ) && ( *pLen != 0 ) )
    {
        while ((String[*pLen - 1] == ' ') || (String[*pLen - 1] == '\t'))
            (*pLen)--;

        while ( (*String == ' ') || (*String == '\t') )
        {
            String++;
            (*pLen)--;
        }
    }
    else
        while ( (*String == ' ') || (*String == '\t') )
            String++;

    return (String);
}

static char* addLine(osl_TProfileImpl* pProfile, const char* Line)
{
    if (pProfile->m_NoLines >= pProfile->m_MaxLines)
    {
        if (pProfile->m_Lines == NULL)
        {
            pProfile->m_MaxLines = LINES_INI;
            pProfile->m_Lines = (char **)malloc(pProfile->m_MaxLines * sizeof(char *));
			memset(pProfile->m_Lines,0,pProfile->m_MaxLines * sizeof(char *));
        }
        else
        {
			unsigned int index=0;
			unsigned int oldmax=pProfile->m_MaxLines;

            pProfile->m_MaxLines += LINES_ADD;
            pProfile->m_Lines = (char **)realloc(pProfile->m_Lines,
				                                 pProfile->m_MaxLines * sizeof(char *));
			for ( index = oldmax ; index < pProfile->m_MaxLines ; ++index )
			{
				pProfile->m_Lines[index]=0;
			}
        }

        if (pProfile->m_Lines == NULL)
        {
            pProfile->m_NoLines  = 0;
            pProfile->m_MaxLines = 0;
            return (NULL);
        }

    }

	if ( pProfile->m_Lines != 0 && pProfile->m_Lines[pProfile->m_NoLines] != 0 )
	{
        free(pProfile->m_Lines[pProfile->m_NoLines]);
	}
    pProfile->m_Lines[pProfile->m_NoLines++] = strdup(Line);

    return (pProfile->m_Lines[pProfile->m_NoLines - 1]);
}

static char* insertLine(osl_TProfileImpl* pProfile, const char* Line, unsigned long LineNo)
{
    if (pProfile->m_NoLines >= pProfile->m_MaxLines)
    {
        if (pProfile->m_Lines == NULL)
        {
            pProfile->m_MaxLines = LINES_INI;
            pProfile->m_Lines = (char **)malloc(pProfile->m_MaxLines * sizeof(char *));
			memset(pProfile->m_Lines,0,pProfile->m_MaxLines * sizeof(char *));
        }
        else
        {
            pProfile->m_MaxLines += LINES_ADD;
            pProfile->m_Lines = (char **)realloc(pProfile->m_Lines,
				                                 pProfile->m_MaxLines * sizeof(char *));

            memset(&pProfile->m_Lines[pProfile->m_NoLines],
				0,
				(pProfile->m_MaxLines - pProfile->m_NoLines - 1) * sizeof(char*));
        }

        if (pProfile->m_Lines == NULL)
        {
            pProfile->m_NoLines  = 0;
            pProfile->m_MaxLines = 0;
            return (NULL);
        }
    }

	LineNo = LineNo > pProfile->m_NoLines ? pProfile->m_NoLines : LineNo;

	if (LineNo < pProfile->m_NoLines)
	{
		unsigned long i, n;
        osl_TProfileSection* pSec;

		memmove(&pProfile->m_Lines[LineNo + 1], &pProfile->m_Lines[LineNo],
				(pProfile->m_NoLines - LineNo) * sizeof(char *));


		for (i = 0; i < pProfile->m_NoSections; i++)
		{
			pSec = &pProfile->m_Sections[i];

			if (pSec->m_Line >= LineNo)
				pSec->m_Line++;

			for (n = 0; n < pSec->m_NoEntries; n++)
				if (pSec->m_Entries[n].m_Line >= LineNo)
					pSec->m_Entries[n].m_Line++;
		}
	}

	pProfile->m_NoLines++;

	pProfile->m_Lines[LineNo] = strdup(Line);

    return (pProfile->m_Lines[LineNo]);
}

static void removeLine(osl_TProfileImpl* pProfile, unsigned long LineNo)
{
    if (LineNo < pProfile->m_NoLines)
    {
        free(pProfile->m_Lines[LineNo]);
        pProfile->m_Lines[LineNo]=0;
        if (pProfile->m_NoLines - LineNo > 1)
		{
			unsigned long i, n;
			osl_TProfileSection* pSec;

            memmove(&pProfile->m_Lines[LineNo], &pProfile->m_Lines[LineNo + 1],
                    (pProfile->m_NoLines - LineNo - 1) * sizeof(char *));

			memset(&pProfile->m_Lines[pProfile->m_NoLines - 1],
				0,
				(pProfile->m_MaxLines - pProfile->m_NoLines) * sizeof(char*));

			for (i = 0; i < pProfile->m_NoSections; i++)
			{
				pSec = &pProfile->m_Sections[i];

				if (pSec->m_Line > LineNo)
					pSec->m_Line--;

				for (n = 0; n < pSec->m_NoEntries; n++)
					if (pSec->m_Entries[n].m_Line > LineNo)
						pSec->m_Entries[n].m_Line--;
			}
		}
		else
		{
			pProfile->m_Lines[LineNo] = 0;
		}

        pProfile->m_NoLines--;
    }

    return;
}

static void setEntry(osl_TProfileImpl* pProfile, osl_TProfileSection* pSection,
                     unsigned long NoEntry, unsigned long Line,
                     char* Entry, unsigned long Len)
{
    Entry = stripBlanks(Entry, &Len);
    pSection->m_Entries[NoEntry].m_Line   = Line;
    pSection->m_Entries[NoEntry].m_Offset = Entry - pProfile->m_Lines[Line];
    pSection->m_Entries[NoEntry].m_Len    = Len;

    return;
}

static BOOL addEntry(osl_TProfileImpl* pProfile, osl_TProfileSection *pSection,
                         int Line, char* Entry, unsigned long Len)
{
    if (pSection != NULL)
    {
        if (pSection->m_NoEntries >= pSection->m_MaxEntries)
        {
            if (pSection->m_Entries == NULL)
            {
                pSection->m_MaxEntries = ENTRIES_INI;
                pSection->m_Entries = (osl_TProfileEntry *)malloc(
                                pSection->m_MaxEntries * sizeof(osl_TProfileEntry));
            }
            else
            {
                pSection->m_MaxEntries += ENTRIES_ADD;
                pSection->m_Entries = (osl_TProfileEntry *)realloc(pSection->m_Entries,
                                pSection->m_MaxEntries * sizeof(osl_TProfileEntry));
            }

            if (pSection->m_Entries == NULL)
            {
                pSection->m_NoEntries  = 0;
                pSection->m_MaxEntries = 0;
                return (FALSE);
            }
        }

        pSection->m_NoEntries++;

        Entry = stripBlanks(Entry, &Len);
        setEntry(pProfile, pSection, pSection->m_NoEntries - 1, Line,
                 Entry, Len);

        return (TRUE);
    }

    return (FALSE);
}

static void removeEntry(osl_TProfileSection *pSection, unsigned long NoEntry)
{
    if (NoEntry < pSection->m_NoEntries)
    {
        if (pSection->m_NoEntries - NoEntry > 1)
        {
            memmove(&pSection->m_Entries[NoEntry],
                    &pSection->m_Entries[NoEntry + 1],
                    (pSection->m_NoEntries - NoEntry - 1) * sizeof(osl_TProfileEntry));
			pSection->m_Entries[pSection->m_NoEntries - 1].m_Line=0;
			pSection->m_Entries[pSection->m_NoEntries - 1].m_Offset=0;
			pSection->m_Entries[pSection->m_NoEntries - 1].m_Len=0;
        }

        pSection->m_NoEntries--;
    }

    return;
}

static BOOL addSection(osl_TProfileImpl* pProfile, int Line, const char* Section, unsigned long Len)
{
    if (pProfile->m_NoSections >= pProfile->m_MaxSections)
    {
        if (pProfile->m_Sections == NULL)
        {
            pProfile->m_MaxSections = SECTIONS_INI;
            pProfile->m_Sections = (osl_TProfileSection *)malloc(pProfile->m_MaxSections * sizeof(osl_TProfileSection));
			memset(pProfile->m_Sections,0,pProfile->m_MaxSections * sizeof(osl_TProfileSection));
        }
        else
        {
			unsigned int index=0;
			unsigned int oldmax=pProfile->m_MaxSections;

            pProfile->m_MaxSections += SECTIONS_ADD;
            pProfile->m_Sections = (osl_TProfileSection *)realloc(pProfile->m_Sections,
                                          pProfile->m_MaxSections * sizeof(osl_TProfileSection));
			for ( index = oldmax ; index < pProfile->m_MaxSections ; ++index )
			{
				pProfile->m_Sections[index].m_Entries=0;
			}
        }

        if (pProfile->m_Sections == NULL)
        {
            pProfile->m_NoSections = 0;
            pProfile->m_MaxSections = 0;
            return (FALSE);
        }
    }

    pProfile->m_NoSections++;

	if ( pProfile->m_Sections[(pProfile->m_NoSections) - 1].m_Entries != 0 )
	{
 		free(pProfile->m_Sections[(pProfile->m_NoSections) - 1].m_Entries);
    }
    pProfile->m_Sections[pProfile->m_NoSections - 1].m_Entries    = NULL;
    pProfile->m_Sections[pProfile->m_NoSections - 1].m_NoEntries  = 0;
    pProfile->m_Sections[pProfile->m_NoSections - 1].m_MaxEntries = 0;

    pProfile->m_Sections[pProfile->m_NoSections - 1].m_Line = Line;
    pProfile->m_Sections[pProfile->m_NoSections - 1].m_Offset = Section - pProfile->m_Lines[Line];
    pProfile->m_Sections[pProfile->m_NoSections - 1].m_Len = Len;

    return (TRUE);
}

static void removeSection(osl_TProfileImpl* pProfile, osl_TProfileSection *pSection)
{
    unsigned long Section;

    if ((Section = pSection - pProfile->m_Sections) < pProfile->m_NoSections)
    {
        free (pSection->m_Entries);
		pSection->m_Entries=0;
        if (pProfile->m_NoSections - Section > 1)
        {
            memmove(&pProfile->m_Sections[Section], &pProfile->m_Sections[Section + 1],
                    (pProfile->m_NoSections - Section - 1) * sizeof(osl_TProfileSection));

            memset(&pProfile->m_Sections[pProfile->m_NoSections - 1],
                   0,
                   (pProfile->m_MaxSections - pProfile->m_NoSections) * sizeof(osl_TProfileSection));
			pProfile->m_Sections[pProfile->m_NoSections - 1].m_Entries = 0;
        }
		else
		{
			pSection->m_Entries = 0;
		}

        pProfile->m_NoSections--;
    }

    return;
}

static osl_TProfileSection* findEntry(osl_TProfileImpl* pProfile, const char* Section,
                                      const char* Entry, unsigned long *pNoEntry)
{
static  unsigned long    Sect = 0;
        unsigned long    i, n;
        unsigned long	Len;
        const char*	pStr;
        osl_TProfileSection* pSec=0;

    Len = strlen(Section);
	n = Sect;

    for (i = 0; i < pProfile->m_NoSections; i++)
    {
		n %= pProfile->m_NoSections;
        pSec = &pProfile->m_Sections[n];
        if ((Len == pSec->m_Len) &&
            (strncmp(Section, &pProfile->m_Lines[pSec->m_Line][pSec->m_Offset], pSec->m_Len)
             == 0))
            break;
        n++;
    }

	Sect = n;

    if (i < pProfile->m_NoSections)
    {
        Len = strlen(Entry);

        *pNoEntry = pSec->m_NoEntries;

        for (i = 0; i < pSec->m_NoEntries; i++)
        {
            pStr = &pProfile->m_Lines[pSec->m_Entries[i].m_Line]
                                     [pSec->m_Entries[i].m_Offset];
            if ((Len == pSec->m_Entries[i].m_Len) &&
                (strncmp(Entry, pStr, pSec->m_Entries[i].m_Len)
                 == 0))
            {
                *pNoEntry = i;
                break;
            }
        }
    }
    else
        pSec = NULL;

    return (pSec);
}

static BOOL loadProfile(osl_TFile* pFile, osl_TProfileImpl* pProfile)
{
    unsigned long	i;
    char*		pStr;
	char*		pChar;

	char* pLine;
	char* bWasAdded = NULL;

    pProfile->m_NoLines    = 0;
    pProfile->m_NoSections = 0;

	if ( pFile == 0 )
	{
		return FALSE;
	}

	if ( pProfile == 0 )
	{
		return FALSE;
	}

	OSL_VERIFY(OslProfile_rewindFile(pFile, FALSE, pProfile->m_Filename));

    while ( ( pLine=OslProfile_getLine(pFile) ) != 0 )
    {
		bWasAdded = addLine( pProfile, pLine );
		delete pLine;
        if ( ! bWasAdded )
			return (FALSE);
    }

    for (i = 0; i < pProfile->m_NoLines; i++)
    {
        pStr = (char *)stripBlanks(pProfile->m_Lines[i], NULL);

        if ((*pStr == '\0') || (*pStr == ';'))
            continue;

        if ((*pStr != '[') || ((pChar = strrchr(pStr, ']')) == NULL) ||
            ((pChar - pStr) <= 2))
        {
            if (pProfile->m_NoSections < 1)
                continue;

            if ((pChar = strchr(pStr, '=')) == NULL)
                pChar = pStr + strlen(pStr);

            if (! addEntry(pProfile, &pProfile->m_Sections[pProfile->m_NoSections - 1],
				           i, pStr, pChar - pStr))
            {
                continue;
            }

        }
        else
        {
            if (! addSection(pProfile, i, pStr + 1, pChar - pStr - 1))
            {
                continue;
            }

        }
    }

    return (TRUE);
}

static BOOL storeProfile(osl_TFile* pFile, osl_TProfileImpl* pProfile, BOOL bCleanup)
{
	if (pProfile->m_Lines != NULL)
    {
        if (pProfile->m_Flags & FLG_MODIFIED)
        {
		    unsigned long i;

			OSL_VERIFY(OslProfile_rewindFile(pFile, TRUE, pProfile->m_Filename));

			for (i = 0; i < pProfile->m_NoLines; i++)
            {
				OSL_VERIFY(OslProfile_putLine(pFile, pProfile->m_Lines[i]));
            }

			if ( ! writeProfileImpl(pProfile->m_pFile) )
            {
                return FALSE;
            }

	        pProfile->m_Flags &= ~FLG_MODIFIED;
        }

		if (bCleanup)
		{
	        while (pProfile->m_NoLines > 0)
	            removeLine(pProfile, pProfile->m_NoLines - 1);

	        free(pProfile->m_Lines);
	        pProfile->m_Lines = NULL;
            pProfile->m_NoLines = 0;
	        pProfile->m_MaxLines = 0;

	        while (pProfile->m_NoSections > 0)
	            removeSection(pProfile, &pProfile->m_Sections[pProfile->m_NoSections - 1]);

	        free(pProfile->m_Sections);
	        pProfile->m_Sections = NULL;
            pProfile->m_NoSections = 0;
	        pProfile->m_MaxSections = 0;
		}
    }

	return (TRUE);
}

static osl_TProfileImpl* acquireProfile(oslProfile Profile, BOOL bWriteable)
{
	osl_TProfileImpl* pProfile = (osl_TProfileImpl*)Profile;
    oslProfileOption PFlags=0;
    BOOL bRet=FALSE;

    if ( bWriteable )
    {
        PFlags = osl_Profile_DEFAULT | osl_Profile_WRITELOCK;
    }
    else
    {
        PFlags = osl_Profile_DEFAULT;
    }


	if (pProfile == NULL)
	{
		if ( ( pProfile = (osl_TProfileImpl*) osl_openProfile(0, PFlags ) ) != NULL )
        {
			pProfile->m_Flags |= FLG_AUTOOPEN;
        }
	}
	else
	{
		if (! (pProfile->m_Flags & osl_Profile_SYSTEM))
		{
            if (! (pProfile->m_Flags & (osl_Profile_READLOCK | osl_Profile_WRITELOCK | osl_Profile_FLUSHWRITE )))
            {
                osl_TStamp Stamp;
                if (! (pProfile->m_pFile = OslProfile_openFile(pProfile->m_Filename, pProfile->m_Flags | PFlags )))
                    return NULL;

                Stamp = OslProfile_getFileStamp(pProfile->m_pFile);

                if (memcmp(&Stamp, &(pProfile->m_Stamp), sizeof(osl_TStamp)))
                {
                    pProfile->m_Stamp = Stamp;

                    bRet=loadProfile(pProfile->m_pFile, pProfile);
                }
            }
            else
            {
                if ((pProfile->m_Flags & osl_Profile_READLOCK) && bWriteable)
                {
                    return (NULL);
                }
            }
        }
	}

	return (pProfile);
}

static BOOL releaseProfile(osl_TProfileImpl* pProfile)
{
    BOOL bRet=FALSE;

    if ( pProfile == 0 )
    {
        return FALSE;
    }

	if (pProfile->m_Flags & FLG_AUTOOPEN)
    {
		return (osl_closeProfile((oslProfile)pProfile));
    }
	else
	{
		if (! (pProfile->m_Flags & (osl_Profile_READLOCK | osl_Profile_WRITELOCK | osl_Profile_FLUSHWRITE )))
		{
			if (pProfile->m_Flags & FLG_MODIFIED)
            {
				bRet=storeProfile(pProfile->m_pFile, pProfile, FALSE);
            }


			OslProfile_closeFile(pProfile->m_pFile,pProfile->m_Flags);
			pProfile->m_pFile = NULL;
		}
	}

	return (TRUE);
}

static BOOL lookupProfile(const char *pszPath, const char *pszFile, char *pPath)
{
	char *pChr, *pStr;
	char Path[_PATH_MAX] = "";
	char Product[132] = "";
	char Buffer[1024] = "";
	char szSUPD[20] = "";

	if (*pszPath == '"')
	{
		int i = 0;

		pszPath++;

		while ((*pszPath != '"') && (*pszPath != '\0'))
			Product[i++] = *pszPath++;

		Product[i] = '\0';

		if (*pszPath == '"')
			pszPath++;

		if ( *pszPath == '/' )
		{
			pszPath++;
		}
	}
	else
	{
		if (strcmp(SVERSION_PROFILE, pszFile) == 0)
		{
			char   Profile[_PATH_MAX] = "";
			char   Dir[_PATH_MAX] = "";
			oslProfile hProfile;
			if ((osl_psz_getProfileName(SVERSION_FALLBACK, SVERSION_NAME, Profile, sizeof(Profile))) &&
				(hProfile = osl_psz_openProfile(Profile, osl_Profile_READLOCK)))
			{
			  	osl_getProfileSectionEntries(hProfile, SVERSION_SECTION,
	   							             Buffer, sizeof(Buffer));

				for (pChr = Buffer; *pChr != '\0'; pChr += strlen(pChr) + 1)
				{
					if ((strncmp(pChr, SVERSION_SOFFICE, sizeof(SVERSION_SOFFICE) - 1) == 0) &&
						(strcmp(Product, pChr) < 0))
					{
						osl_readProfileString(hProfile, SVERSION_SECTION, pChr,
						                      Dir, sizeof(Dir), "");

						if (access(Dir, 0) >= 0)
							strcpy(Product, pChr);
					}
				}

				osl_closeProfile(hProfile);
			}

			if ((strcmp(SVERSION_LOCATION, SVERSION_FALLBACK) != 0) &&
			    (osl_psz_getProfileName(SVERSION_LOCATION, SVERSION_NAME, Profile, sizeof(Profile))) &&
				(hProfile = osl_psz_openProfile(Profile, osl_Profile_READLOCK)))
			{
			  	osl_getProfileSectionEntries(hProfile, SVERSION_SECTION,
	   							             Buffer, sizeof(Buffer));

				for (pChr = Buffer; *pChr != '\0'; pChr += strlen(pChr) + 1)
				{
					if ((strncmp(pChr, SVERSION_SOFFICE, sizeof(SVERSION_SOFFICE) - 1) == 0) &&
						(strcmp(Product, pChr) < 0))
					{
						osl_readProfileString(hProfile, SVERSION_SECTION, pChr,
						                      Dir, sizeof(Dir), "");

						if (access(Dir, 0) >= 0)
							strcpy(Product, pChr);
					}
				}

				osl_closeProfile(hProfile);
			}

			if ((pChr = strrchr(Product, '/')) != NULL)
				*pChr = '\0';
		}
	}


	if (strlen(Path) <= 0)
	{
		if ((pChr = strrchr(Path, '/')) == NULL)
			if ((pChr = strrchr(Path, ':')) == NULL)
				return (FALSE);
			else
				*pChr = '\0';
		else
			*pChr = '\0';

		if (strlen(Product) <= 0)
		{
			strcpy(Product, pChr + 1);

			if ((pChr = strrchr(Product, '.')) != NULL)
				*pChr = '\0';
		}

		pStr = strrchr(Path, '/');

		strcat(Path, "/");

		if (strlen(pszPath) <= 0 )
		{
			strcat(Path,SVERSION_USER);
		}
		else
		{
			strcat(Path, pszPath);
		}

		if (((access(Path, 0) < 0) && (pStr != NULL)) || (strlen(pszPath) <= 0))
		{
			static char *SubDirs[] = SVERSION_DIRS;

			unsigned int i = 0;

			for (i = 0; i < (sizeof(SubDirs) / sizeof(SubDirs[0])); i++)
				if (strncmp(pStr + 1, SubDirs[i], strlen(SubDirs[i])) == 0)
				{
					if ( strlen(pszPath) <= 0 )
					{
						strcpy(pStr + 1, SVERSION_USER);
						if ( access(Path, 0) < 0 )
						{
							*(pStr+1)='\0';
						}
					}
					else
					{
						strcpy(pStr + 1, pszPath);
					}

					break;
				}
		}

		pChr = &Path[strlen(Path)];
		if (Path[strlen(Path) - 1] != '/')
			strcat(Path, "/");
		strcat(Path, pszFile);

		if ((access(Path, 0) < 0) && (strlen(Product) > 0))
		{
			char   Profile[_PATH_MAX] = "";
			*pChr = '\0';

		}
		else
			*pChr = '\0';
	}

	strcpy(pPath, Path);

	return (TRUE);
}

