/*************************************************************************
 *
 *  $RCSfile: XclExpStream.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: dr $ $Date: 2001/10/26 16:45:21 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifdef PCH
#include "filt_pch.hxx"
#endif

#pragma hdrstop

//___________________________________________________________________

#ifndef _SC_XCLEXPSTREAM_HXX
#include "XclExpStream.hxx"
#endif

#ifndef _EXCDEFS_HXX
#include "excdefs.hxx"
#endif

//___________________________________________________________________

XclExpStream::XclExpStream( SvStream& rOutStrm, sal_uInt32 _nMaxRecLen ) :
    rStrm( rOutStrm ),
    nMaxRecLen( _nMaxRecLen ),
    nMaxContLen( _nMaxRecLen ),
    nMaxLen( 0 ),
    nMaxSliceLen( 0 ),
    nCalcLen( 0 ),
    nHeaderLen( 0 ),
    nCurrLen( 0 ),
    nSliceLen( 0 ),
    nLastLenPos( 0 ),
    bInRec( FALSE )
{
}

XclExpStream::~XclExpStream()
{
}

void XclExpStream::InitRecord( sal_uInt16 nRecNum )
{
    rStrm.Seek( STREAM_SEEK_TO_END );
    rStrm << nRecNum;

    nLastLenPos = rStrm.Tell();
    nHeaderLen = Min( nCalcLen, nMaxLen );
    rStrm << static_cast< sal_uInt16 >( nHeaderLen );
    nCurrLen = nSliceLen = 0;
}

void XclExpStream::UpdateRecLen()
{
    if( nCurrLen != nHeaderLen )
    {
        rStrm.Seek( nLastLenPos );
        rStrm << static_cast< sal_uInt16 >( nCurrLen );
    }
}

void XclExpStream::UpdateLenVars( sal_uInt32 nLen )
{
    nCurrLen += nLen;

    if( nMaxSliceLen )
    {
        nSliceLen += nLen;
        DBG_ASSERT( nSliceLen <= nMaxSliceLen, "XclExpStream::UpdateLenVars - slice overwritten" );
        if( nSliceLen >= nMaxSliceLen )
            nSliceLen = 0;
    }
}

void XclExpStream::StartContinue()
{
    UpdateRecLen();
    nMaxLen = nMaxContLen;
    nCalcLen -= nCurrLen;
    InitRecord( EXC_CONT );
}

void XclExpStream::PrepareWrite( sal_uInt32 nLen )
{
    if( !bInRec ) return;

    if( (nCurrLen + nLen > nMaxLen) ||
        (nMaxSliceLen && !nSliceLen && (nCurrLen + nMaxSliceLen > nMaxLen)) )
        StartContinue();
    UpdateLenVars( nLen );
}

sal_uInt32 XclExpStream::PrepareWrite()
{
    if( !bInRec )
        return 0;

    if( (nCurrLen >= nMaxLen) ||
        (nMaxSliceLen && !nSliceLen && (nCurrLen + nMaxSliceLen > nMaxLen)) )
        StartContinue();
    UpdateLenVars( 0 );

    return nMaxSliceLen ? nMaxSliceLen - nSliceLen : nMaxLen - nCurrLen;
}

void XclExpStream::StartRecord( sal_uInt16 nRecNum, sal_uInt32 nRecLen )
{
    DBG_ASSERT( !bInRec, "XclExpStream::StartRecord - another record still open" );
    nMaxContLen = nMaxLen = nMaxRecLen;
    nCalcLen = nRecLen;
    bInRec = TRUE;
    InitRecord( nRecNum );
    SetSliceLen( 0 );
}

void XclExpStream::EndRecord()
{
    DBG_ASSERT( bInRec, "XclExpStream::EndRecord - no record open" );
    UpdateRecLen();
    rStrm.Seek( STREAM_SEEK_TO_END );
    bInRec = FALSE;
}

void XclExpStream::SetSliceLen( sal_uInt32 nLen )
{
    nMaxSliceLen = nLen;
    nSliceLen = 0;
}

sal_uInt32 XclExpStream::Write( const void* pData, sal_uInt32 nBytes )
{
    if( !pData || !nBytes )
        return 0;

    sal_uInt32 nRet = 0;
    if( bInRec )
    {
        const sal_uInt8* pBuffer = (const sal_uInt8*) pData;
        sal_uInt32 nBytesLeft = nBytes;
        sal_Bool bValid = TRUE;

        while( bValid && nBytesLeft )
        {
            sal_uInt32 nWriteLen = Min( PrepareWrite(), nBytesLeft );
            sal_uInt32 nWriteRet = rStrm.Write( pBuffer, nWriteLen );
            bValid = (nWriteLen == nWriteRet);
            DBG_ASSERT( bValid, "XclExpStream::Write - stream write error" );
            pBuffer += nWriteRet;
            nRet += nWriteRet;
            nBytesLeft -= nWriteRet;
            UpdateLenVars( nWriteRet );
        }
    }
    else
        nRet = rStrm.Write( pData, nBytes );

    return nRet;
}

void XclExpStream::WriteRawZeroBytes( sal_uInt32 nBytes )
{
    const sal_uInt32 nData = 0;
    sal_uInt32 nBytesLeft = nBytes;
    while( nBytesLeft >= sizeof( sal_uInt32 ) )
    {
        rStrm << nData;
        nBytesLeft -= sizeof( sal_uInt32 );
    }
    if( nBytesLeft )
        rStrm.Write( &nData, nBytesLeft );
}

void XclExpStream::WriteZeroBytes( sal_uInt32 nBytes )
{
    if( !nBytes ) return;

    if( bInRec )
    {
        sal_uInt32 nBytesLeft = nBytes;
        while( nBytesLeft )
        {
            sal_uInt32 nWriteLen = Min( PrepareWrite(), nBytesLeft );
            WriteRawZeroBytes( nWriteLen );
            nBytesLeft -= nWriteLen;
            UpdateLenVars( nWriteLen );
        }
    }
    else
        WriteRawZeroBytes( nBytes );
}

sal_uInt32 XclExpStream::CopyFromStream( SvStream& rInStrm, sal_uInt32 nBytes )
{
    sal_uInt32 nStreamPos = rInStrm.Tell();
    sal_uInt32 nStreamLen = rInStrm.Seek( STREAM_SEEK_TO_END );
    rInStrm.Seek( nStreamPos );

    sal_uInt32 nBytesLeft = Min( nBytes, nStreamLen - nStreamPos );
    if( !nBytesLeft )
        return 0;

    const sal_uInt32 nMaxBuffer = 0x00001000;
    sal_uInt8* pBuffer = new sal_uInt8[ Min( nBytesLeft, nMaxBuffer ) ];
    sal_uInt32 nRet = 0;
    sal_Bool bValid = TRUE;

    while( bValid && nBytesLeft )
    {
        sal_uInt32 nWriteLen = Min( nBytesLeft, nMaxBuffer );
        rInStrm.Read( pBuffer, nWriteLen );
        sal_uInt32 nWriteRet = Write( pBuffer, nWriteLen );
        bValid = (nWriteLen == nWriteRet);
        nRet += nWriteRet;
        nBytesLeft -= nWriteRet;
    }

    delete[] pBuffer;
    return nRet;
}

void XclExpStream::WriteUnicodeBuffer( const sal_uInt16* pBuffer, sal_uInt32 nChars, sal_uInt8 nFlags )
{
    SetSliceLen( 0 );
    if( !pBuffer || !nChars ) return;

    sal_uInt32 nCharLen = (nFlags & EXC_STR_16BIT) ? 2 : 1;
    for( sal_uInt32 nIndex = 0; nIndex < nChars; nIndex++ )
    {
        if( bInRec && (nCurrLen + nCharLen > nMaxLen) )
        {
            StartContinue();
            // repeat only 16bit flag
            operator<<( static_cast< sal_uInt8 >( nFlags & EXC_STR_16BIT ) );
        }
        if( nCharLen == 2 )
            operator<<( pBuffer[ nIndex ] );
        else
            operator<<( static_cast< sal_uInt8 >( pBuffer[ nIndex ] ) );
    }
}

void XclExpStream::WriteByteStringBuffer( const ByteString& rString, sal_uInt16 nMaxLen )
{
    SetSliceLen( 0 );
    Write( rString.GetBuffer(), Min( rString.Len(), static_cast< xub_StrLen >( nMaxLen ) ) );
}

// ER: #71367# Xcl has an obscure sense of whether starting a new record or not,
// and crashes if it encounters the string header at the very end of a record.
// Thus we add 1 to give some room, seems like they do it that way but with another count (10?)
void XclExpStream::WriteByteString( const ByteString& rString, sal_uInt16 nMaxLen, sal_Bool b16BitCount )
{
    SetSliceLen( 0 );
    sal_uInt16 nLen = static_cast< sal_uInt16 >( Min( rString.Len(), static_cast< xub_StrLen >( nMaxLen ) ) );
    if( !b16BitCount )
        nLen = Min( nLen, static_cast< sal_uInt16 >( 0x00FF ) );

    sal_uInt32 nLeft = PrepareWrite();
    if( bInRec && (nLeft <= (b16BitCount ? 2UL : 1UL)) )
        StartContinue();

    if( b16BitCount )
        operator<<( nLen );
    else
        operator<<( static_cast< sal_uInt8 >( nLen ) );
    Write( rString.GetBuffer(), nLen );
}

sal_uInt32 XclExpStream::SetStreamPos( sal_uInt32 nPos )
{
    DBG_ASSERT( !bInRec, "XclExpStream::SetStreamPos - not allowed inside of a record" );
    return bInRec ? 0 : rStrm.Seek( nPos );
}

