/*************************************************************************
 *
 *  $RCSfile: uri.c,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: sb $ $Date: 2001/05/31 11:25:04 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif
#ifndef _RTL_TEXTENC_H
#include <rtl/textenc.h>
#endif
#ifndef _RTL_URI_H_
#include <rtl/uri.h>
#endif
#ifndef _RTL_USTRBUF_H_
#include <rtl/ustrbuf.h>
#endif
#ifndef _RTL_USTRING_H_
#include <rtl/ustring.h>
#endif
#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#define rtl_Uri_CharClassSize 128

#define rtl_Uri_EscapePrefix 0x25 /* '%' */

#define rtl_Uri_isHighSurrogate(cUnicode) /* sal_Unicode -> sal_Bool */ \
    ((cUnicode >= 0xD800) && (cUnicode <= 0xDBFF))

#define rtl_Uri_isLowSurrogate(cUnicode) /* sal_Unicode -> sal_Bool */ \
    ((cUnicode >= 0xDC00) && (cUnicode <= 0xDFFF))

#define rtl_Uri_getHexWeight(nUcs4) \
        /* sal_uInt32 -> int  (0..15; -1 if not hex digit) */ \
    ((nUcs4) >= 0x30 && (nUcs4) <= 0x39 ? /* '0'--'9' */ \
         (int) ((nUcs4) - 0x30) : \
     (nUcs4) >= 0x41 && (nUcs4) <= 0x46 ? /* 'A'--'F' */ \
         (int) ((nUcs4) - 0x41 + 10) : \
     (nUcs4) >= 0x61 && (nUcs4) <= 0x66 ? /* 'a'--'f' */ \
         (int) ((nUcs4) - 0x61 + 10) : \
         -1)

#define rtl_Uri_isValid(pCharClass, nUcs4) /* sal_uInt32 -> sal_Bool */ \
    ((nUcs4) < rtl_Uri_CharClassSize && pCharClass[nUcs4])

#define rtl_Uri_writeUnicode(pBuffer, pCapacity, cChar) \
        /* rtl_uString **, sal_Int32 *, sal_Unicode -> void */ \
    rtl_uStringbuffer_insert((pBuffer), (pCapacity), (*(pBuffer))->length, \
                             &(cChar), 1)

typedef enum
{
    rtl_UriEscapeNo,
    rtl_UriEscapeChar,
    rtl_UriEscapeOctet
}
rtl_UriEscapeType;

/* Read any of the following:

   - sequence of escape sequences representing character from eCharset,
     translated to single UCS4 character; or

   - pair of UTF-16 surrogates, translated to single UCS4 character; or

   _ single UTF-16 character, extended to UCS4 character.
 */
static sal_uInt32 rtl_Uri_readUcs4(sal_Unicode const ** pBegin,
                                   sal_Unicode const * pEnd,
                                   sal_Bool bEncoded,
                                   rtl_TextEncoding eCharset,
                                   rtl_UriEscapeType * pType)
{
    sal_uInt32 nChar = *(*pBegin)++;
    int nWeight1;
    int nWeight2;
    if (nChar == rtl_Uri_EscapePrefix
        && bEncoded
        && pEnd - *pBegin >= 2
        && (nWeight1 = rtl_Uri_getHexWeight((*pBegin)[0])) >= 0
        && (nWeight2 = rtl_Uri_getHexWeight((*pBegin)[1])) >= 0)
    {
        *pBegin += 2;
        nChar = (sal_uInt32) (nWeight1 << 4 | nWeight2);
        switch (eCharset)
        {
        case RTL_TEXTENCODING_ASCII_US:
            *pType = nChar <= 0x7F ? rtl_UriEscapeChar : rtl_UriEscapeOctet;
            return nChar;

        case RTL_TEXTENCODING_ISO_8859_1:
            *pType = rtl_UriEscapeChar;
            return nChar;

        default:
            OSL_ENSURE(sal_False, "unsupported eCharset"); /*TODO!*/
        case RTL_TEXTENCODING_UTF8:
            if (nChar <= 0x7F)
                *pType = rtl_UriEscapeChar;
            else
            {
                if (nChar >= 0xC0 && nChar <= 0xFC)
                {
                    sal_uInt32 nEncoded;
                    int nShift;
                    sal_uInt32 nMin;
                    sal_Unicode const * p = *pBegin;
                    sal_Bool bUTF8 = sal_True;
                    if (nChar <= 0xDF)
                    {
                        nEncoded = (nChar & 0x1F) << 6;
                        nShift = 0;
                        nMin = 0x80;
                    }
                    else if (nChar <= 0xEF)
                    {
                        nEncoded = (nChar & 0x0F) << 12;
                        nShift = 6;
                        nMin = 0x800;
                    }
                    else if (nChar <= 0xF7)
                    {
                        nEncoded = (nChar & 0x07) << 18;
                        nShift = 12;
                        nMin = 0x10000;
                    }
                    else if (nChar <= 0xFB)
                    {
                        nEncoded = (nChar & 0x03) << 24;
                        nShift = 18;
                        nMin = 0x200000;
                    }
                    else
                    {
                        nEncoded = 0;
                        nShift = 24;
                        nMin = 0x4000000;
                    }
                    for (; nShift >= 0; nShift -= 6)
                    {
                        if (pEnd - p < 3
                            || p[0] != rtl_Uri_EscapePrefix
                            || (nWeight1 = rtl_Uri_getHexWeight(p[1])) < 8
                            || nWeight1 > 11
                            || (nWeight2 = rtl_Uri_getHexWeight(p[2])) < 0)
                        {
                            bUTF8 = sal_False;
                            break;
                        }
                        p += 3;
                        nEncoded
                            |= ((nWeight1 & 3) << 4 | nWeight2) << nShift;
                    }
                    if (bUTF8
                        && nEncoded >= nMin
                        && !rtl_Uri_isHighSurrogate(nEncoded)
                        && !rtl_Uri_isLowSurrogate(nEncoded)
                        && nEncoded <= 0x10FFFF)
                    {
                        *pBegin = p;
                        *pType = rtl_UriEscapeChar;
                        return nEncoded;
                    }
                }
                *pType = rtl_UriEscapeOctet;
            }
            return nChar;
        }
    }
    else
    {
        *pType = rtl_UriEscapeNo;
        return rtl_Uri_isHighSurrogate(nChar)
               && *pBegin < pEnd
               && rtl_Uri_isLowSurrogate(**pBegin) ?
                   ((nChar & 0x3FF) << 10 | *(*pBegin)++ & 0x3FF) + 0x10000 :
                   nChar;
    }
}

static void rtl_Uri_writeSurrogates(rtl_uString ** pBuffer,
                                    sal_Int32 * pCapacity,
                                    sal_uInt32 nUcs4)
{
    sal_Unicode cUnicode;
    OSL_ENSURE(nUcs4 > 0xFFFF && nUcs4 <= 0x10FFFF, "bad UTF-32 char");
    nUcs4 -= 0x10000;
    cUnicode = (sal_Unicode) (nUcs4 >> 10 | 0xD800);
    rtl_Uri_writeUnicode(pBuffer, pCapacity, cUnicode);
    cUnicode = (sal_Unicode) (nUcs4 & 0x3FF | 0xDC00);
    rtl_Uri_writeUnicode(pBuffer, pCapacity, cUnicode);
}

static void rtl_Uri_writeEscapeOctet(rtl_uString ** pBuffer,
                                     sal_Int32 * pCapacity,
                                     sal_uInt32 nOctet)
{
    static sal_Unicode const aHex[16]
        = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
            0x41, 0x42, 0x43, 0x44, 0x45, 0x46 }; /* '0'--'9', 'A'--'F' */

    sal_Unicode cEscapePrefix = rtl_Uri_EscapePrefix;
    rtl_Uri_writeUnicode(pBuffer, pCapacity, cEscapePrefix);
    OSL_ENSURE(nOctet <= 0xFF, "bad octet");
    rtl_Uri_writeUnicode(pBuffer, pCapacity, aHex[nOctet >> 4]);
    rtl_Uri_writeUnicode(pBuffer, pCapacity, aHex[nOctet & 15]);
}

static void rtl_Uri_writeEscapeChar(rtl_uString ** pBuffer,
                                    sal_Int32 * pCapacity,
                                    sal_uInt32 nUcs4,
                                    rtl_TextEncoding eCharset)
{
    switch (eCharset)
    {
    case RTL_TEXTENCODING_ASCII_US:
    case RTL_TEXTENCODING_ISO_8859_1:
        OSL_ENSURE(nUcs4 <= (eCharset == RTL_TEXTENCODING_ASCII_US ? 0x7FU :
                                                                     0xFFU),
                   "bad ASCII or ISO 8859-1 char");
        rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4);
        break;

    default:
        OSL_ENSURE(sal_False, "unsupported eCharset"); /*TODO!*/
    case RTL_TEXTENCODING_UTF8:
        OSL_ENSURE(nUcs4 <= 0x7FFFFFFF, "bad UCS-4 char");
        if (nUcs4 < 0x80)
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4);
        else if (nUcs4 < 0x800)
        {
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 >> 6 | 0xC0);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 & 0x3F | 0x80);
        }
        else if (nUcs4 < 0x10000)
        {
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 >> 12 | 0xE0);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 6 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 & 0x3F | 0x80);
        }
        else if (nUcs4 < 0x200000)
        {
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 >> 18 | 0xF0);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 12 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 6 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 & 0x3F | 0x80);
        }
        else if (nUcs4 < 0x4000000)
        {
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 >> 24 | 0xF8);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 18 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 12 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 6 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 & 0x3F | 0x80);
        }
        else
        {
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 >> 30 | 0xFC);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 24 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 18 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 12 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity,
                                     nUcs4 >> 6 & 0x3F | 0x80);
            rtl_Uri_writeEscapeOctet(pBuffer, pCapacity, nUcs4 & 0x3F | 0x80);
        }
        break;
    }
}

sal_Bool const * rtl_getUriCharClass(rtl_UriCharClass eCharClass)
{
    static sal_Bool const aCharClass[][rtl_Uri_CharClassSize]
    = {{ 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* None */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* !"#$%&'()*+,-./*/
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /*0123456789:;<=>?*/
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /*@ABCDEFGHIJKLMNO*/
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /*PQRSTUVWXYZ[\]^_*/
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /*`abcdefghijklmno*/
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* Uric */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* UricNoSlash */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* RelSegment */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 0, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* RegName */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* Userinfo */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0, /*0123456789:;<=>?*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* Pchar */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       },
       { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* UnoParamValue */
         0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
         0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, /* !"#$%&'()*+,-./*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*0123456789:;<=>?*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*@ABCDEFGHIJKLMNO*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 0, 1, /*PQRSTUVWXYZ[\]^_*/
         0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, /*`abcdefghijklmno*/
         1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 0  /*pqrstuvwxyz{|}~ */
       }};
    OSL_ENSURE(eCharClass < sizeof aCharClass / sizeof aCharClass[0],
               "bad eCharClass");
    return aCharClass[eCharClass];
}

void rtl_uriEncode(rtl_uString * pText,
                   sal_Bool const * pCharClass,
                   rtl_UriEncodeMechanism eMechanism,
                   rtl_TextEncoding eCharset,
                   rtl_uString ** pResult)
{
    sal_Unicode const * p = pText->buffer;
    sal_Unicode const * pEnd = p + pText->length;
    sal_Int32 nCapacity = 0;
    OSL_ENSURE(!pCharClass[0x25], "bad pCharClass");
        /* make sure the percent sign is encoded... */
    rtl_uString_new(pResult);
    while (p < pEnd)
    {
        rtl_UriEscapeType eType;
        sal_uInt32 nUcs4
            = rtl_Uri_readUcs4(&p,
                               pEnd,
                               (sal_Bool)
                                   (eMechanism != rtl_UriEncodeIgnoreEscapes),
                               eCharset,
                               &eType);
        switch (eType)
        {
        case rtl_UriEscapeNo:
            if (rtl_Uri_isValid(pCharClass, nUcs4))
                    /* implies nUcs4 <= 0x7F */
            {
                sal_Unicode cUnicode = (sal_Unicode) nUcs4;
                rtl_Uri_writeUnicode(pResult, &nCapacity, cUnicode);
            }
            else
                rtl_Uri_writeEscapeChar(pResult, &nCapacity, nUcs4, eCharset);
            break;

        case rtl_UriEscapeChar:
            if (eMechanism == rtl_UriEncodeCheckEscapes
                && rtl_Uri_isValid(pCharClass, nUcs4))
                    /* implies nUcs4 <= 0x7F */
            {
                sal_Unicode cUnicode = (sal_Unicode) nUcs4;
                rtl_Uri_writeUnicode(pResult, &nCapacity, cUnicode);
            }
            else
                rtl_Uri_writeEscapeChar(pResult, &nCapacity, nUcs4, eCharset);
            break;

        case rtl_UriEscapeOctet:
            rtl_Uri_writeEscapeOctet(pResult, &nCapacity, nUcs4);
            break;
        }
    }
}

void rtl_uriDecode(rtl_uString * pText,
                   rtl_UriDecodeMechanism eMechanism,
                   rtl_TextEncoding eCharset,
                   rtl_uString ** pResult)
{
    switch (eMechanism)
    {
    case rtl_UriDecodeNone:
        rtl_uString_assign(pResult, pText);
        break;

    case rtl_UriDecodeToIuri:
        eCharset = RTL_TEXTENCODING_UTF8;
    default: /* rtl_UriDecodeWithCharset */
        {
            sal_Unicode const * p = pText->buffer;
            sal_Unicode const * pEnd = p + pText->length;
            sal_Int32 nCapacity = 0;
            rtl_uString_new(pResult);
            while (p < pEnd)
            {
                rtl_UriEscapeType eType;
                sal_uInt32 nUcs4
                    = rtl_Uri_readUcs4(&p, pEnd, sal_True, eCharset, &eType);
                switch (eType)
                {
                case rtl_UriEscapeChar:
                    if (nUcs4 <= 0x7F && eMechanism == rtl_UriDecodeToIuri)
                    {
                        rtl_Uri_writeEscapeOctet(pResult, &nCapacity, nUcs4);
                        break;
                    }
                case rtl_UriEscapeNo:
                    if (nUcs4 <= 0xFFFF)
                    {
                        sal_Unicode cUnicode = (sal_Unicode) nUcs4;
                        rtl_Uri_writeUnicode(pResult, &nCapacity, cUnicode);
                    }
                    else if (nUcs4 <= 0x10FFFF)
                        rtl_Uri_writeSurrogates(pResult, &nCapacity, nUcs4);
                    else
                        rtl_Uri_writeEscapeChar(pResult, &nCapacity,
                                                nUcs4, eCharset);
                    break;

                case rtl_UriEscapeOctet:
                    rtl_Uri_writeEscapeOctet(pResult, &nCapacity, nUcs4);
                    break;
                }
            }
        }
        break;
    }
}
