/*************************************************************************
 *
 *  $RCSfile: thread.c,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/14 09:48:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <premac.h>
#include <Timer.h>
#include <Threads.h>
#include <postmac.h>

#include <string.h>
#include <assert.h>

#include <osl/diagnose.h>
#include <osl/thread.h>

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif


OSErr _ThreadError = 0;


/* some memory helper macros */
#define NEW(type)	malloc(sizeof(type))
#define DELETE(ptr)	free(ptr)
#define FILL(ptr,val,type) memset(ptr,val,sizeof(type))

/* constants for thread status flags */
#define TF_SHOULD_TERMINATE	0x0001
#define TF_SHOULD_SLEEP		0x0002
#define TF_TERMINATED		0x0100
#define TF_SLEEPING			0x0200

/* Task and Thread info for time Manager */
typedef struct TMInfo {
	TMTask			tmTask; /* Place this first */
	ThreadTaskRef	ttRef;
	ThreadID		tID;
	sal_uInt32	delay;
} TMInfo;

/* The implementation of our internal thread structure */
typedef struct {
	ThreadEntryProcPtr	entry;
	void				*param;
} ThreadEntryWrapInfo;

typedef struct _ThreadImpl *ThreadPtr;

typedef struct _ThreadImpl {
	ThreadID			threadID;		/* the thread ID handled by Mac OS			*/
	sal_uInt32			flags;			/* status flags of thread					*/
	const TMInfo		*ptmInfo;		/* task info if we are sleeping (somewhere on stack!) */
	TimeValue			delay;
	ThreadEntryWrapInfo	wrapInfo;
} ThreadImpl;

/* local static prototypes */
static pascal void		_InitThreadManager(void);
static pascal ThreadID	_ThreadScheduler(SchedulerInfoRecPtr schedulerInfo);
static pascal void		_ThreadTerminator(ThreadID threadID, void *pData);
static pascal void		_ThreadEnterContext(ThreadID id, void *param);
static pascal void		_ThreadLeaveContext(ThreadID id, void *param);
static pascal void *_ThreadEntryWrapper(void *param);
static pascal void		_WakeTMTask(void);
static ThreadPtr		_CreateThread(oslWorkerFunction pFunc, void *pData, ThreadOptions options);
static sal_uInt32 	_osl_delayThread(const TimeValue* pDelay);
static ThreadImpl	_mainThread;
static ThreadImpl	*_context = NULL;
static int _ThreadsInitialized = 0;

#define CHECK_INIT() {if (!_ThreadsInitialized) _InitThreadManager();}

/*********************************************************/
static pascal void _InitThreadManager()
{
	MaxApplZone();

	GetCurrentThread(&_mainThread.threadID);
	_mainThread.flags = 0;
	_mainThread.ptmInfo = NULL;

	_context = &_mainThread;

	SetThreadSwitcher(_mainThread.threadID, _ThreadEnterContext, &_mainThread, sal_True);
	SetThreadSwitcher(_mainThread.threadID, _ThreadLeaveContext, &_mainThread, sal_False);
	SetThreadScheduler(_ThreadScheduler);

	_ThreadsInitialized = 1;
}

static pascal void *_ThreadEntryWrapper(void *param)
{
	ThreadEntryWrapInfo	*info = (ThreadEntryWrapInfo *)param;

	return info->entry(info->param);
}

static pascal void	_ThreadTerminator(ThreadID threadID, void *pData)
{
#pragma unused(threadID)
	ThreadPtr	pThread = (ThreadPtr)pData;

	OSL_ASSERT(threadID == pThread->threadID);

	pThread->flags |= TF_TERMINATED;
	pThread->flags &= ~(TF_SHOULD_SLEEP | TF_SLEEPING);
	if (pThread->ptmInfo)
	{
		RmvTime((QElemPtr)pThread->ptmInfo);
		pThread->ptmInfo = NULL;
	}
}

static pascal void	_ThreadEnterContext(ThreadID id, void *param)
{
	_context = (ThreadImpl *)param;
	if ( id != _context->threadID )
		id = _context->threadID;
}

static pascal void	_ThreadLeaveContext(ThreadID id, void *param)
{
#pragma unused(id, param)
	/* NULL to indicate a thread created outside of OSL */
	_context = (ThreadImpl *)NULL;
}

static pascal ThreadID	_ThreadScheduler(SchedulerInfoRecPtr schedulerInfo)
{
	return schedulerInfo->SuggestedThreadID;
}

/************************************************************/
/* Own functions											*/
/************************************************************/


static ThreadPtr _CreateThread(oslWorkerFunction pWorker, void *pData, ThreadOptions options)
{
	ThreadPtr	pThread;
	OSErr		err;

	pThread = NEW(ThreadImpl);
	pThread->flags = 0;
	pThread->ptmInfo = NULL;

/* Must not be a structur on stuck */
	pThread->wrapInfo.entry = (ThreadEntryProcPtr)pWorker;
	pThread->wrapInfo.param = pData;

	err = NewThread(
			kCooperativeThread,			/* only cooperative threads					*/
			_ThreadEntryWrapper,		/* thread entry function					*/
			&pThread->wrapInfo,			/* params passed to entry function			*/
			0,							/* default stack size 						*/
			options,					/* kNewSuspend or 0							*/
			NULL,						/* address where to place result of thread	*/
			&pThread->threadID);		/* store new thread ID						*/

	SetThreadSwitcher(pThread->threadID, _ThreadEnterContext, pThread, sal_True);
	SetThreadSwitcher(pThread->threadID, _ThreadLeaveContext, pThread, sal_False);
	SetThreadTerminator(pThread->threadID, _ThreadTerminator, pThread);

	return pThread;
}

/************************************************************/
/* osl wrapper functions									*/
/************************************************************/

/*****************************************************************************/
/* osl_createThread */
/*****************************************************************************/
oslThread SAL_CALL osl_createThread(oslWorkerFunction pWorker, void *pThreadData)
{
	CHECK_INIT();
	return (oslThread)_CreateThread(pWorker, pThreadData, 0);
}

/*****************************************************************************/
/* osl_createSuspendedThread */
/*****************************************************************************/
oslThread SAL_CALL osl_createSuspendedThread(oslWorkerFunction pWorker, void *pThreadData)
{
	CHECK_INIT();
	return (oslThread)_CreateThread(pWorker, pThreadData, kNewSuspend);
}

/*****************************************************************************/
/* osl_getThreadIdentifier */
/*****************************************************************************/
oslThreadIdentifier SAL_CALL osl_getThreadIdentifier(oslThread Thread)
{	return (oslThreadIdentifier)((ThreadPtr)Thread)->threadID;}


/*****************************************************************************/
/* osl_destroyThread */
/*****************************************************************************/


void SAL_CALL osl_destroyThread(oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();
	if (hThread != NULL)
	{
		DisposeThread(pThread->threadID, NULL, sal_False);
		osl_freeThreadHandle(hThread);
	}
}

/*****************************************************************************/
/* osl_freeThreadHandle */
/*****************************************************************************/
void SAL_CALL osl_freeThreadHandle(oslThread hThread)
{
	CHECK_INIT();
	if (hThread != NULL)
		DELETE((ThreadPtr)hThread);
}

/*****************************************************************************/
/* osl_resumeThread */
/*****************************************************************************/
void SAL_CALL osl_resumeThread(oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();
	/* No check for NULL cause hThread must be valid */
	SetThreadState(pThread->threadID, kReadyThreadState, kNoThreadID);
	YieldToThread(pThread->threadID);
}

/*****************************************************************************/
/* osl_suspendThread */
/*****************************************************************************/
void SAL_CALL osl_suspendThread(oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();
	/* No check for NULL cause hThread must be valid */
	SetThreadState(pThread->threadID, kStoppedThreadState, kNoThreadID);
}

/*****************************************************************************/
/* osl_setThreadPriority */
/*****************************************************************************/
void SAL_CALL osl_setThreadPriority(oslThread hThread, oslThreadPriority Priority)
{
#pragma unused(hThread, Priority)
	CHECK_INIT();
}

/*****************************************************************************/
/* osl_getThreadPriority */
/*****************************************************************************/
oslThreadPriority SAL_CALL osl_getThreadPriority(oslThread hThread)
{
#pragma unused(hThread)
	CHECK_INIT();
	return osl_Thread_PriorityUnknown;
}

/*****************************************************************************/
/* osl_isThreadRunning */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isThreadRunning(const oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();
	if (pThread == NULL)
		return sal_False;
	else
		return (sal_Bool)((pThread->flags & TF_TERMINATED) == 0);
}

/*****************************************************************************/
/* osl_isEqualThread */
/*****************************************************************************/
sal_Bool osl_isEqualThread(const oslThread hThread1, const oslThread hThread2)
{
	CHECK_INIT();
	if (hThread1 == NULL || hThread2 == NULL)
		return sal_False;
	else
		return (sal_Bool)( ((ThreadPtr)hThread1)->threadID == ((ThreadPtr)hThread2)->threadID );
}

/*****************************************************************************/
/* osl_joinWithThread */
/*****************************************************************************/
void SAL_CALL osl_joinWithThread(oslThread hThread)
{
	CHECK_INIT();
	/* Simple polling */
	while (osl_isThreadRunning(hThread))
		osl_yieldThread();
}

/*****************************************************************************/
/* osl_sleepThread */
/*****************************************************************************/
oslThreadSleep SAL_CALL osl_sleepThread(oslThread hThread, const TimeValue* pDelay)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;
	ThreadID	curThreadID;
	CHECK_INIT();

	if (pThread == NULL)
		return osl_Thread_SleepError;

	GetCurrentThread(&curThreadID);
	if (pThread->threadID == curThreadID)
		return (_osl_delayThread(pDelay) ? osl_Thread_SleepCancel : osl_Thread_SleepNormal);
	else if ((pThread->flags & (TF_SHOULD_SLEEP | TF_SLEEPING)) || pThread->ptmInfo)
		return osl_Thread_SleepActive;
	else
	{
		pThread->flags |= TF_SHOULD_SLEEP;
		pThread->delay = *pDelay;
		return osl_Thread_SleepPending;
	}
}

/*****************************************************************************/
/* osl_awakeThread */
/*****************************************************************************/
sal_Bool SAL_CALL osl_awakeThread(oslThread hThread)
{
#pragma unused(hThread)
	ThreadPtr	pThread = (ThreadPtr)hThread;
	CHECK_INIT();

	if (pThread == NULL)
		return sal_False;

	if (pThread->ptmInfo)
	{
		ThreadBeginCritical();

		RmvTime((QElemPtr)pThread->ptmInfo);
		pThread->ptmInfo = 0;
		SetThreadReadyGivenTaskRef(pThread->ptmInfo->ttRef, pThread->ptmInfo->tID);

		ThreadEndCritical();
	}

	pThread->flags &= ~(TF_SHOULD_SLEEP | TF_SLEEPING);

	return sal_True;
}

/*****************************************************************************/
/* osl_waitThread */
/*****************************************************************************/
static pascal void WakeTMTask(TMInfo *ptmInfo)
{
	ThreadState		state;

	GetThreadStateGivenTaskRef(ptmInfo->ttRef, ptmInfo->tID, &state);

	if (state != kStoppedThreadState)
		PrimeTime((QElemPtr)ptmInfo, ptmInfo->delay);
	else
		SetThreadReadyGivenTaskRef(ptmInfo->ttRef, ptmInfo->tID);
}

static sal_uInt32 _osl_delayThread(const TimeValue* pDelay)
{
	TMInfo			tmInfo;
	sal_uInt32	milliSecs;

	milliSecs = pDelay->Seconds * 1000L + pDelay->Nanosec / 1000000L;

	tmInfo.tmTask.tmAddr = NewTimerProc(WakeTMTask);
	tmInfo.tmTask.tmWakeUp = 0;
	tmInfo.tmTask.tmReserved = 0;
	tmInfo.delay = milliSecs;

	GetThreadCurrentTaskRef(&tmInfo.ttRef);
	GetCurrentThread(&tmInfo.tID);

	ThreadBeginCritical();

	if (_context) /* if we are not called from outside */
	{
		_context->ptmInfo = &tmInfo;
		_context->flags &= ~TF_SHOULD_SLEEP;
		_context->flags |= TF_SLEEPING;
	}

	InsTime((QElemPtr)&tmInfo);
	PrimeTime((QElemPtr)&tmInfo, milliSecs);

	SetThreadStateEndCritical(kCurrentThreadID, kStoppedThreadState, kNoThreadID);

	ThreadBeginCritical();

	if (_context) /* if we are not called from outside */
	{
		_context->ptmInfo = NULL;
		_context->flags &= ~TF_SLEEPING;
	}

	RmvTime((QElemPtr)&tmInfo);

	ThreadEndCritical();

	return tmInfo.tmTask.tmCount;
}

void SAL_CALL osl_waitThread(const TimeValue* pDelay)
{
	_osl_delayThread(pDelay);
}

/*****************************************************************************/
/* osl_terminateThread */
/*****************************************************************************/
void SAL_CALL osl_terminateThread(oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();
	pThread->flags |= TF_SHOULD_TERMINATE;
}

/*****************************************************************************/
/* osl_scheduleThread */
/*****************************************************************************/
sal_Bool SAL_CALL osl_scheduleThread(oslThread hThread)
{
	ThreadPtr	pThread = (ThreadPtr)hThread;

	CHECK_INIT();

	if (pThread->flags & TF_SHOULD_SLEEP)
	{
		pThread->flags &= ~TF_SHOULD_SLEEP;
		osl_waitThread(&pThread->delay);
	}
	else
		osl_yieldThread();

	return (sal_Bool)((pThread->flags & TF_SHOULD_TERMINATE) == 0);
}

/*****************************************************************************/
/* osl_yieldThread */
/*****************************************************************************/
void SAL_CALL osl_yieldThread()
{
	ThreadID	id;
	OSErr	error;

	CHECK_INIT();
	GetCurrentThread(&id);
	assert(_context->threadID == id);
	error = YieldToAnyThread();
	if (error != noErr)
	{
		_ThreadError = error;
	}
}

oslThreadKey SAL_CALL osl_createThreadKey(void)
{
	return NULL;
}

void SAL_CALL osl_destroyThreadKey(oslThreadKey Key)
{
}

void* SAL_CALL osl_getThreadKeyData(oslThreadKey Key)
{
	return NULL;
}

sal_Bool SAL_CALL osl_setThreadKeyData(oslThreadKey Key, void *pData)
{
	return sal_False;
}

/*****************************************************************************/
/* osl_getThreadTextEncoding */
/*****************************************************************************/
rtl_TextEncoding SAL_CALL osl_getThreadTextEncoding()
{
	return RTL_TEXTENCODING_ISO_8859_1;
}


/*****************************************************************************/
/* osl_setThreadTextEncoding */
/*****************************************************************************/
rtl_TextEncoding SAL_CALL osl_setThreadTextEncoding(rtl_TextEncoding Encoding)
{
	return RTL_TEXTENCODING_ISO_8859_1;
}


