/*************************************************************************
 *
 *  $RCSfile: ftpimpl.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: mhu $ $Date: 2001/07/20 17:23:46 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INET_FTPIMPL_CXX "$Revision: 1.3 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRBUF_HXX_
#include <rtl/strbuf.hxx>
#endif

#ifndef _VOS_MACROS_HXX_
#include <vos/macros.hxx>
#endif
#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_OBJECT_HXX_
#include <vos/object.hxx>
#endif
#ifndef _VOS_REF_HXX
#include <vos/ref.hxx>
#endif

#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif

#ifndef _INET_FTPIMPL_HXX
#include <ftpimpl.hxx>
#endif
#ifndef _INET_FTPCNTL_HXX
#include <ftpcntl.hxx>
#endif
#ifndef _INET_FTPDATA_HXX
#include <ftpdata.hxx>
#endif

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OString;
using rtl::OStringBuffer;

using rtl::OUString;

/*========================================================================
 *
 * Debug trace environment.
 *
 *======================================================================*/
#ifdef DBG_UTIL_PRIVATE
#include <stdio.h>

#if (defined(WIN) || defined(WNT) || defined(OS2))
#define _TRACELOGNAME "c:\\inetftp.log"
#elif (defined(UNX))
#define _TRACELOGNAME "/tmp/inetftp.log"
#else
#define _TRACELOGNAME "inetftp.log"
#endif

#ifdef WIN
#define _TRACELOGFORMAT "%ls\n"
#else
#define _TRACELOGFORMAT "%s\n"
#endif

#undef  _TRACE
#define _TRACE(a) \
{ \
	FILE *fp = fopen (_TRACELOGNAME, "a+"); \
	if (fp) \
	{ \
		fprintf (fp, _TRACELOGFORMAT, (const char *)(a)); \
		fclose (fp); \
	} \
}

#else
#define _TRACE(a)
#endif /* DBG_UTIL_PRIVATE */

/*========================================================================
 *
 * INetFTPConnection_Impl internals.
 *
 *======================================================================*/
#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetFTPCommandContext_Impl
{
	/** Representation.
	 */
	INetFTPCommandStream *m_pCommand;
	INetFTPInputStream   *m_pSource;
	INetFTPOutputStream  *m_pTarget;
	INetFTPCallback      *m_pfnCB;
	void                 *m_pDataCB;

	/** Construction, destruction.
	 */
	INetFTPCommandContext_Impl (
		INetFTPCommandStream *pCommand,
		INetFTPInputStream   *pSource,
		INetFTPOutputStream  *pTarget,
		INetFTPCallback      *pfnCB,
		void                 *pDataCB)
		: m_pCommand (pCommand),
		  m_pSource  (pSource),
		  m_pTarget  (pTarget),
		  m_pfnCB    (pfnCB),
		  m_pDataCB  (pDataCB)
	{}

	~INetFTPCommandContext_Impl (void)
	{
		delete m_pCommand;
		delete m_pSource;
		delete m_pTarget;
	}

	/** Command callback.
	 */
	void setCallback (INetFTPCallback *pfnCB, void *pDataCB)
	{
		m_pfnCB   = pfnCB;
		m_pDataCB = pDataCB;
	}
};

#ifdef _USE_NAMESPACE
}
#endif

typedef INetFTPCntlContext cntl_type;
typedef INetFTPDataContext data_type;

typedef NAMESPACE_INET(INetSocket)       socket_type;
typedef NAMESPACE_VOS(ORef)<socket_type> OSocketRef;

#define INETFTP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
	(socket)->deregisterEventHandler(INetFTPConnection::onSocketEvent); \
	(socket)->close(); \
	(socket).unbind(); \
}

/*========================================================================
 *
 * INetFTPConnection_Impl implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetFTPConnection_Impl, inet),
	VOS_NAMESPACE (INetFTPConnection_Impl, inet),
	VOS_NAMESPACE (INetFTPConnection, inet),
	0);

/*
 * INetFTPConnection_Impl.
 */
INetFTPConnection_Impl::INetFTPConnection_Impl (void)
	: m_pCtx    (NULL),
	  m_nGwPort (0)
{
}

/*
 * ~INetFTPConnection_Impl.
 */
INetFTPConnection_Impl::~INetFTPConnection_Impl (void)
{
	INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xDataSocket);
	INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xPasvSocket);
	INETFTP_SOCKET_DISPOSE (m_aCntlCtx.m_xSocket);
}

/*
 * open.
 */
sal_Bool INetFTPConnection_Impl::open (
	const OUString  &rHost,
	sal_uInt16       nPort,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetFTPConnection_Impl> xThis (this);

	// Check arguments.
	if (!rHost.getLength())
		return sal_False;
	if (!nPort)
		nPort = INETCOREFTP_CMD_PORT;
	if (!pfnCB)
		return sal_False;

	// Switch context.
	INetFTPCommandContext_Impl *pCtx = new INetFTPCommandContext_Impl (
		new INetFTPOpenCommandStream (OString()),
		NULL, NULL, pfnCB, pDataCB);
	switchContext (pCtx);

	m_aCntlCtx.setState (cntl_type::STATE_RESOLVE, REPLY_RESOLVER_WAIT);

	// Start resolution.
	return m_aResolver.GetHostByName (
		new INetCoreDNSHostEntry (rHost, nPort), onResolverEvent, this);
}

/*
 * isOpen.
 */
sal_Bool INetFTPConnection_Impl::isOpen (void) const
{
	// Obtain Open state.
	return m_aCntlCtx.isOpen();
}

/*
 * close.
 */
sal_Bool INetFTPConnection_Impl::close (
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand (
		new INetFTPOpenCommandStream ("QUIT\015\012"),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * abort.
 */
void INetFTPConnection_Impl::abort (void)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetFTPConnection_Impl> xThis (this);

	// Switch context.
	INetFTPCommandContext_Impl *pCtx = switchContext (NULL);
	if (pCtx)
	{
		m_aCntlCtx.setState (cntl_type::STATE_ABORT);
		m_aDataCtx.setState (data_type::STATE_ABORT);

		delete pCtx;
	}
}

/*
 * loginUsername.
 */
sal_Bool INetFTPConnection_Impl::loginUsername (
	const OUString  &rUsername,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(rUsername.getLength() && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("USER ");
	aCommand.append (OString (
		rUsername.pData->buffer,
		rUsername.pData->length,
		RTL_TEXTENCODING_ASCII_US));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPLoginCommandStream (aCommand.makeStringAndClear()),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * loginPassword.
 */
sal_Bool INetFTPConnection_Impl::loginPassword (
	const OUString  &rPassword,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(rPassword.getLength() && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("PASS ");
	aCommand.append (OString (
		rPassword.pData->buffer,
		rPassword.pData->length,
		RTL_TEXTENCODING_ASCII_US));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPLoginCommandStream (aCommand.makeStringAndClear()),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * loginAccount.
 */
sal_Bool INetFTPConnection_Impl::loginAccount (
	const OUString  &rAccount,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(rAccount.getLength() && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("ACCT ");
	aCommand.append (OString (
		rAccount.pData->buffer,
		rAccount.pData->length,
		RTL_TEXTENCODING_ASCII_US));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPLoginCommandStream (aCommand.makeStringAndClear()),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * isLoggedIn.
 */
sal_Bool INetFTPConnection_Impl::isLoggedIn (void) const
{
	// Obtain LoggedIn state.
	return m_aCntlCtx.isLoggedIn();
}

/*
 * noop.
 */
sal_Bool INetFTPConnection_Impl::noop (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	// Check arguments.
	if (!pfnCB)
		return sal_False;

	// Start command.
	return startCommand (
		new INetFTPCommandStream ("NOOP\015\012"),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * setCurDir.
 */
sal_Bool INetFTPConnection_Impl::setCurDir (
	const OUString  &rPathname,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Reset working directory.
	m_aCntlCtx.setPath (OUString());

	// Start command.
	return startCommand ("CWD ", rPathname, pfnCB, pDataCB);
}

/*
 * getCurDir.
 */
sal_Bool INetFTPConnection_Impl::getCurDir (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	// Check arguments.
	if (!pfnCB)
		return sal_False;

	// Start command.
	return startCommand (
		new INetFTPPwdCommandStream ("PWD\015\012"),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * getCurDir.
 */
const OUString& INetFTPConnection_Impl::getCurDir (void) const
{
	// Obtain working directory.
	return m_aCntlCtx.getPath();
}

/*
 * makeDir.
 */
sal_Bool INetFTPConnection_Impl::makeDir (
	const OUString  &rPathname,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand ("MKD ", rPathname, pfnCB, pDataCB);
}

/*
 * removeDir.
 */
sal_Bool INetFTPConnection_Impl::removeDir (
	const OUString  &rPathname,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand ("RMD ", rPathname, pfnCB, pDataCB);
}

/*
 * remove.
 */
sal_Bool INetFTPConnection_Impl::remove (
	const OUString  &rPathname,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand ("DELE ", rPathname, pfnCB, pDataCB);
}

/*
 * renameFrom.
 */
sal_Bool INetFTPConnection_Impl::renameFrom (
	const OUString  &rFromPath,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand ("RNFR ", rFromPath, pfnCB, pDataCB);
}

/*
 * renameTo.
 */
sal_Bool INetFTPConnection_Impl::renameTo (
	const OUString  &rToPath,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Start command.
	return startCommand ("RNTO ", rToPath, pfnCB, pDataCB);
}

/*
 * setTypeAscii.
 */
sal_Bool INetFTPConnection_Impl::setTypeAscii (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	typedef INetFTPCommandStream command_type;

	// Check arguments.
	if (!pfnCB)
		return sal_False;

	// Start command.
	return startCommand (
		new command_type ("TYPE A\015\012", command_type::CMD_TYPE_ASCII),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * setTypeImage.
 */
sal_Bool INetFTPConnection_Impl::setTypeImage (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	typedef INetFTPCommandStream command_type;

	// Check arguments.
	if (!pfnCB)
		return sal_False;

	// Start command.
	return startCommand (
		new command_type ("TYPE I\015\012", command_type::CMD_TYPE_IMAGE),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * getDataType.
 */
INetFTPConnection::DataType
INetFTPConnection_Impl::getDataType (void) const
{
	// Obtain data type.
	return m_aDataCtx.getDataType();
}

/*
 * getListType.
 */
INetFTPConnection::ListType
INetFTPConnection_Impl::getListType (void) const
{
	// Obtain list type.
	return m_aDataCtx.getListType();
}

/*
 * getNameList.
 */
sal_Bool INetFTPConnection_Impl::getNameList (
	const OUString  &rPathname,
	List            &rNameList,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!pfnCB)
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("LIST");

	ListType eListType = m_aDataCtx.getListType();
	if ((eListType == LIST_TYPE_UNDETERMINED) ||
		(eListType == LIST_TYPE_UNKNOWN     )    )
	{
		// Request plain namelist, only.
		aCommand.insert (0, "NLST");
	}

	if (rPathname.getLength())
	{
		aCommand.append (sal_Char (' '));
		aCommand.append (OString (
			rPathname.pData->buffer,
			rPathname.pData->length,
			RTL_TEXTENCODING_UTF8));
	}
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPPasvCommandStream (aCommand.makeStringAndClear()),
		NULL,
		new INetFTPDirectoryStream (rNameList, eListType),
		pfnCB, pDataCB);
}

/*
 * retrieve.
 */
sal_Bool INetFTPConnection_Impl::retrieve (
	const OUString  &rFromPath,
	SvOpenLockBytes *pToPath,
	sal_uInt32       nRestartOffset,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(rFromPath.getLength() && pToPath && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("RETR ");
	aCommand.append (OString (
		rFromPath.pData->buffer,
		rFromPath.pData->length,
		RTL_TEXTENCODING_UTF8));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPPasvCommandStream (aCommand.makeStringAndClear()),
		NULL,
		new INetFTPRetrieveStream (pToPath, nRestartOffset),
		pfnCB, pDataCB);
}

/*
 * getRetrieveCount.
 */
sal_uInt32 INetFTPConnection_Impl::getRetrieveCount (void) const
{
	// Obtain retrieve count.
	return m_aDataCtx.getRecvCount();
}

/*
 * store.
 */
sal_Bool INetFTPConnection_Impl::store (
	SvLockBytes     *pFromPath,
	const OUString  &rToPath,
	sal_uInt32       nRestartOffset,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(pFromPath && rToPath.getLength() && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand ("STOR ");
	aCommand.append (OString (
		rToPath.pData->buffer,
		rToPath.pData->length,
		RTL_TEXTENCODING_UTF8));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPPasvCommandStream (aCommand.makeStringAndClear()),
		new INetFTPStoreStream (pFromPath, nRestartOffset),
		NULL,
		pfnCB, pDataCB);
}

/*
 * getStoreCount.
 */
sal_uInt32 INetFTPConnection_Impl::getStoreCount (void) const
{
	// Obtain store count.
	return m_aDataCtx.getSendCount();
}

/*
 * abortTransfer.
 */
sal_Bool INetFTPConnection_Impl::abortTransfer (sal_Bool bSilent)
{
	// Acquire exclusive control connection access.
	NAMESPACE_VOS(OGuard) aGuard (m_aCntlCtx);

	// Check control connection state.
	cntl_type::State cstate = m_aCntlCtx.getState();
	if (cstate == cntl_type::STATE_NONE)
	{
		// No command in progress.
		return sal_False;
	}

	// Check data connection state.
	data_type::State dstate = m_aDataCtx.getState();
	if (dstate == data_type::STATE_NONE)
	{
		// No transfer in progress.
		return sal_False;
	}

	// Abort data transfer.
	if (cstate == cntl_type::STATE_RECV)
	{
		// Send ABOR command.
		m_aCntlCtx.m_xSocket->send ("ABOR\015\012", 6);
	}
	m_aDataCtx.setState (data_type::STATE_ABORT);

	// Close data connection...
	if (m_aDataCtx.m_xDataSocket.isValid())
		m_aDataCtx.m_xDataSocket->close();

	// Done.
	return sal_True;
}

/*
 * setTransferCallback.
 */
sal_Bool INetFTPConnection_Impl::setTransferCallback (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	// Check context.
	sal_Bool result = (!(m_aDataCtx.getState() == data_type::STATE_ABORT));
	if (result)
	{
		// Set data transfer callback.
		m_aDataCtx.setCallback (pfnCB, pDataCB);
	}
	return (result);
}

/*
 * setTerminateCallback.
 */
sal_Bool INetFTPConnection_Impl::setTerminateCallback (
	INetFTPCallback *pfnCB, void *pDataCB)
{
	// Check context.
	sal_Bool result = (!(m_aCntlCtx.getState() == cntl_type::STATE_ABORT));
	if (result)
	{
		// Set connection termination callback.
		m_aCntlCtx.setCallback (pfnCB, pDataCB);
	}
	return (result);
}

/*
 * getSocksGatewayName.
 */
const OUString& INetFTPConnection_Impl::getSocksGatewayName (void) const
{
	return m_aGwName;
}

/*
 * getSocksGatewayPort.
 */
sal_uInt16 INetFTPConnection_Impl::getSocksGatewayPort (void) const
{
	return m_nGwPort;
}

/*
 * hasSocksGateway.
 */
sal_Bool INetFTPConnection_Impl::hasSocksGateway (void) const
{
	return ((m_aGwName.getLength() > 0) && (m_nGwPort > 0));
}

/*
 * setSocksGateway.
 */
void INetFTPConnection_Impl::setSocksGateway (
	const OUString& rName, sal_uInt16 nPort)
{
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
	m_aGwName = rName;
	m_nGwPort = nPort;
}

/*
 * switchContext.
 */
INetFTPCommandContext_Impl*
INetFTPConnection_Impl::switchContext (INetFTPCommandContext_Impl *pCtx)
{
	// Acquire exclusive access.
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

	// Switch context.
	INetFTPCommandContext_Impl *pCurCtx = m_pCtx;
	m_pCtx = pCtx;
	return pCurCtx;
}

/*
 * startCommand.
 */
sal_Bool INetFTPConnection_Impl::startCommand (
	const OString   &rCommand,
	const OUString  &rArgument,
	INetFTPCallback *pfnCB,
	void            *pDataCB)
{
	// Check arguments.
	if (!(rCommand.getLength() && rArgument.getLength() && pfnCB))
		return sal_False;

	// Build command line.
	OStringBuffer aCommand (rCommand);
	aCommand.append (OString (
		rArgument.pData->buffer,
		rArgument.pData->length,
		RTL_TEXTENCODING_UTF8));
	aCommand.append ("\015\012");

	// Start command.
	return startCommand (
		new INetFTPCommandStream (aCommand.makeStringAndClear()),
		NULL, NULL, pfnCB, pDataCB);
}

/*
 * startCommand.
 */
sal_Bool INetFTPConnection_Impl::startCommand (
	INetFTPCommandStream *pCommand,
	INetFTPInputStream   *pSource,
	INetFTPOutputStream  *pTarget,
	INetFTPCallback      *pfnCB,
	void                 *pDataCB)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetFTPConnection_Impl> xThis (this);

	// Create context.
	INetFTPCommandContext_Impl *pCtx = new INetFTPCommandContext_Impl (
		pCommand, pSource, pTarget, pfnCB, pDataCB);

	// Check context.
	if (m_pCtx)
	{
		delete pCtx;
		return sal_False;
	}

	// Check control connection.
	OSocketRef xSocket (&*(m_aCntlCtx.m_xSocket));
	if (!xSocket.isValid())
	{
		delete pCtx;
		return sal_False;
	}

	// Check state.
	cntl_type::State eState = m_aCntlCtx.getState();
	if (eState == cntl_type::STATE_ABORT)
	{
		delete pCtx;
		return sal_False;
	}

	// Check data connection state.
	if (!(m_aDataCtx.getState() == data_type::STATE_NONE))
	{
		// Acquire exclusive access.
		NAMESPACE_VOS(OGuard) aGuard (m_aDataCtx);
		if (!(m_aDataCtx.getState() == data_type::STATE_NONE))
		{
			// Dispose data connection.
			INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xDataSocket);
			INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xPasvSocket);
			m_aDataCtx.setState (data_type::STATE_NONE);
		}
	}

	eState = cntl_type::STATE_SEND;
	if (pSource || pTarget)
	{
		// Data transfer command.
		eState = cntl_type::STATE_PORT;
		if (pSource)
		{
			m_aDataCtx.setOffset (pSource->getOffset());
			m_aDataCtx.setSendCount(0);
		}
		if (pTarget)
		{
			m_aDataCtx.setOffset (pTarget->getOffset());
			m_aDataCtx.setRecvCount(0);
		}

		// Initialize passive socket.
		m_aDataCtx.m_xPasvSocket = new INetPassiveTCPSocket();
		m_aDataCtx.m_xPasvSocket->registerEventHandler (
			onSocketEvent, this);

		// Check for SocksGateway.
		if (m_aGwName.getLength() && m_nGwPort)
		{
			// Listen via SocksGateway.
			m_aDataCtx.m_xPasvSocket->setSocksGateway (
				NAMESPACE_VOS(OInetSocketAddr)(m_aGwName, m_nGwPort));
		}
		m_aDataCtx.setState (data_type::STATE_PORT);
	}
	m_aCntlCtx.setState (eState, REPLY_REQUEST_WAIT);

	// Switch context.
	switchContext (pCtx);
	if (eState == cntl_type::STATE_PORT)
	{
		// Obtain RelatedToAddr.
		NAMESPACE_VOS(OInetSocketAddr) aToAddr;
		m_aCntlCtx.m_xSocket->getToAddr (aToAddr);

		// Start listening.
		if (!m_aDataCtx.m_xPasvSocket->listen (
			NAMESPACE_VOS(OInetSocketAddr)(), &aToAddr))
		{
			delete (switchContext (NULL));
			return sal_False;
		}
	}
	else
	{
		// Start control command.
		if (!m_aCntlCtx.m_xSocket->postEvent (socket_type::EVENT_WRITE))
		{
			delete (switchContext (NULL));
			return sal_False;
		}
	}
	return sal_True;
}

/*
 * completeCommand.
 */
void INetFTPConnection_Impl::completeCommand (INetFTPCommandStream *pStrm)
{
	typedef INetFTPCommandStream stream_type;

	stream_type::Command eCommand = pStrm->getCommand();
	if (eCommand == stream_type::CMD_OTHER)
	{
		// Nothing to do.
		return;
	}
	if (eCommand == stream_type::CMD_OPEN)
	{
		sal_Int32 nCode = pStrm->getReplyCode();
		m_aCntlCtx.isOpen (nCode == REPLY_SERVICE_READY);
		return;
	}
	if (eCommand == stream_type::CMD_LOGIN)
	{
		sal_Int32 nCode = pStrm->getReplyCode();
		m_aCntlCtx.isLoggedIn (nCode/100 == 2);
		return;
	}
	if (eCommand == stream_type::CMD_PASV)
	{
		typedef INetFTPPasvCommandStream command_type;

		if (pStrm->isKindOf (VOS_CLASSINFO (command_type)))
		{
			command_type *cmd = (command_type*)pStrm;
			// m_aDataCtx.setPeerAddr (Addr, Port);
		}
		return;
	}
	if (eCommand == stream_type::CMD_PWD)
	{
		typedef INetFTPPwdCommandStream command_type;

		if (pStrm->isKindOf (VOS_CLASSINFO (command_type)))
		{
			command_type *cmd = (command_type*)pStrm;

			OUString aPath (cmd->getcwd());
			if (m_aDataCtx.getListType() == LIST_TYPE_UNDETERMINED)
			{
				// Determine ListType from working directory.
				m_aDataCtx.setListType (aPath);
			}
			m_aCntlCtx.setPath (aPath);
		}
		return;
	}
	if (eCommand == stream_type::CMD_TYPE_ASCII)
	{
		m_aDataCtx.setDataType (DATA_TYPE_ASCII);
		return;
	}
	if (eCommand == stream_type::CMD_TYPE_IMAGE)
	{
		m_aDataCtx.setDataType (DATA_TYPE_IMAGE);
		return;
	}
}

/*
 * handleResolverEvent.
 */
sal_Bool INetFTPConnection_Impl::handleResolverEvent (
	sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
	// Jump into state machine.
	while (m_pCtx)
	{
		// Check state.
		cntl_type::State eState = m_aCntlCtx.getState();
		if (eState == cntl_type::STATE_RESOLVE)
		{
			// Check reason.
			if (nStatus == INETCOREDNS_RESOLVER_START)
			{
				// Resolution started.
				m_aCntlCtx.setStateCode (REPLY_RESOLVER_WAIT);

				// Notify caller.
				if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
					this,
					REPLY_RESOLVER_WAIT,
					NULL,
					m_pCtx->m_pDataCB);

				// Wait for next callback.
				return sal_True;
			}
			else if ((nStatus == INETCOREDNS_RESOLVER_SUCCESS) ||
					 (nStatus == INETCOREDNS_RESOLVER_EXPIRED)    )
			{
				// Resolution finished.
				m_aCntlCtx.setState (
					cntl_type::STATE_CONNECT, REPLY_RESOLVER_DONE);

				// Notify caller.
				if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
					this,
					REPLY_RESOLVER_DONE,
					NULL,
					m_pCtx->m_pDataCB);
			}
			else
			{
				// Resolution failed.
				m_aCntlCtx.setState (
					cntl_type::STATE_ERROR, REPLY_RESOLVER_ERROR);
			}
		}
		else if (eState == cntl_type::STATE_CONNECT)
		{
			// Initialize active socket.
			m_aCntlCtx.m_xSocket = new INetActiveTCPSocket();
			m_aCntlCtx.m_xSocket->registerEventHandler (onSocketEvent, this);

			// Check for gateway.
			if (m_aGwName.getLength() && m_nGwPort)
			{
				// Connect via SocksGateway.
				m_aCntlCtx.m_xSocket->setSocksGateway (
					NAMESPACE_VOS(OInetSocketAddr)(m_aGwName, m_nGwPort));
			}

			// Initiate connect.
			if (m_aCntlCtx.m_xSocket->connect (
				NAMESPACE_VOS(OInetSocketAddr)(
					pHostEntry->GetDottedDecimalName(),
					pHostEntry->GetPort())))
			{
				// Connect in progress.
				m_aCntlCtx.setStateCode (REPLY_CONNECT_WAIT);

				// Cleanup.
				delete pHostEntry;

				// Notify caller.
				if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
					this,
					REPLY_CONNECT_WAIT,
					NULL,
					m_pCtx->m_pDataCB);

				// Wait for next callback.
				return sal_True;
			}

			// Connect failure.
			m_aCntlCtx.setState (
				cntl_type::STATE_ERROR, REPLY_CONNECT_ERROR);
		}
		else
		{
			// Abort or failure.
			INetFTPCommandContext_Impl *pCtx = switchContext (NULL);
			if (pCtx)
			{
				// Obtain reply code.
				sal_Int32 nReplyCode = m_aCntlCtx.getStateCode();

				// Notify caller.
				if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
					this,
					nReplyCode,
					NULL,
					pCtx->m_pDataCB);

				// Cleanup.
				delete pCtx;
			}

			// Cleanup.
			delete pHostEntry;
		}
	}

	// Leave.
	return sal_False;
}

/*
 * handleSocketEvent.
 */
sal_Bool INetFTPConnection_Impl::handleSocketEvent (
	const OSocketRef &rxSocket, sal_Int32 nEvent)
{
	sal_Int32 nCode = 0;
	if (rxSocket.isEqualBody (m_aCntlCtx.m_xSocket.getBodyPtr()))
	{
		// Control connection event.
		nCode = handleCommand (nEvent);
		if (nCode == REPLY_CONNECT_DONE)
		{
			// Notify caller (CommandCallback).
			if (m_pCtx && m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
				this, REPLY_CONNECT_DONE, NULL, m_pCtx->m_pDataCB);
		}
		if (nCode == REPLY_SERVICE_UNAVAIL)
		{
			// Notify connection termination callback.
			if (m_aCntlCtx.m_pfnCB) (m_aCntlCtx.m_pfnCB) (
				this, REPLY_SERVICE_UNAVAIL, NULL, m_aCntlCtx.m_pDataCB);
		}
	}
	else
	{
		// Data connection event.
		nCode = handleTransfer (nEvent);
		if (nCode == REPLY_TRANSFER_WAIT)
		{
			// Notify data transfer callback.
			if (m_aDataCtx.m_pfnCB) (m_aDataCtx.m_pfnCB) (
				this, REPLY_TRANSFER_WAIT, NULL, m_aDataCtx.m_pDataCB);
		}
	}

	if (m_aCntlCtx.getState() == cntl_type::STATE_NONE)
	{
		// Aborted, failed or finished.
		INetFTPCommandContext_Impl *pCtx = switchContext (NULL);
		if (pCtx)
		{
			// Obtain reply text.
			const sal_Char *pReplyText = NULL;
			if (pCtx->m_pCommand)
				pReplyText = pCtx->m_pCommand->getReplyText();

			// Release data streams.
			DELETEZ (pCtx->m_pSource);
			DELETEZ (pCtx->m_pTarget);

			// Notify caller.
			if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
				this,
				m_aCntlCtx.getStateCode(),
				pReplyText,
				pCtx->m_pDataCB);

			// Cleanup.
			delete pCtx;
		}
	}
	return sal_True;
}

/*
 * handleCommand (USER-PI).
 */
sal_Int32 INetFTPConnection_Impl::handleCommand (sal_Int32 nEvent)
{
	if (m_aCntlCtx.getState() == cntl_type::STATE_NONE)
	{
		m_aCntlCtx.setStateCode(0);
		if (nEvent & socket_type::EVENT_READ)
		{
			// Eat up any unsolicited response.
			sal_Int32 nCode = m_aCntlCtx.recv (NULL);
			if (nCode == REPLY_NETWORK_ERROR)
			{
				// Shutdown control connection.
				m_aCntlCtx.m_xSocket->close();
			}
		}
		if (nEvent & socket_type::EVENT_CLOSE)
		{
			// Control connection closed.
			m_aCntlCtx.m_xSocket.unbind();

			// Service unavailable (421).
			m_aCntlCtx.setStateCode (REPLY_SERVICE_UNAVAIL);
		}
	}

	while (!(m_aCntlCtx.getState() == cntl_type::STATE_NONE))
	{
		// Acquire exclusive access.
		NAMESPACE_VOS(OGuard) aGuard (m_aCntlCtx);

		// Check control connection state.
		sal_Int32 nCode = m_aCntlCtx.getStateCode();
		switch (m_aCntlCtx.getState())
		{
			case cntl_type::STATE_CONNECT:
				if (!(nEvent & socket_type::EVENT_OOB))
				{
					// Connect finished.
					m_aCntlCtx.setState (
						cntl_type::STATE_RECV, REPLY_CONNECT_DONE);

					// Read greeting reply.
					nEvent = socket_type::EVENT_READ;
					if (m_aCntlCtx.m_xSocket->postEvent (nEvent))
					{
						// Wait for next event.
						return REPLY_CONNECT_DONE;
					}
					nEvent = socket_type::EVENT_OOB;
				}
				if (nEvent & socket_type::EVENT_OOB)
				{
					// Connect failure.
					m_aCntlCtx.setState (
						cntl_type::STATE_ERROR, REPLY_CONNECT_ERROR);

					// Cleanup.
					m_aCntlCtx.m_xSocket.unbind();
				}
				break;

			case cntl_type::STATE_PORT:
				if (nEvent & socket_type::EVENT_LISTEN)
				{
					// Check USER-DTP state.
					if (m_aDataCtx.getState() == data_type::STATE_OPEN)
					{
						// Obtain connection endpoint.
						NAMESPACE_VOS(OInetSocketAddr) aAddr;
						m_aDataCtx.m_xPasvSocket->getMyAddr (aAddr);

						OUString  aHost;
						sal_Int32 nPort = aAddr.getPort();
						aAddr.getDottedAddr (aHost);
						if (aHost.compareToAscii ("0.0.0.0") == 0)
						{
							m_aCntlCtx.m_xSocket->getMyAddr (aAddr);
							aAddr.getDottedAddr (aHost);
						}

						ByteString aPort (
							aHost.pData->buffer,
							aHost.pData->length,
							RTL_TEXTENCODING_ASCII_US);
						aPort.SearchAndReplaceAll ('.', ',');

						// Setup PORT command.
						OStringBuffer aCommand ("PORT ");
						aCommand.append (aPort);
						aCommand.append (sal_Char  (','));
						aCommand.append (sal_Int32 ((nPort >> 8) & 0xff));
						aCommand.append (sal_Char  (','));
						aCommand.append (sal_Int32 ((nPort >> 0) & 0xff));
						aCommand.append ("\015\012");

						m_aCntlCtx.copy (aCommand, aCommand.getLength());

						// Send PORT command.
						nEvent = socket_type::EVENT_WRITE;
					}
					else
					{
						// Failure.
						m_aCntlCtx.setStateCode (REPLY_NETWORK_ERROR);
						m_aCntlCtx.setState (cntl_type::STATE_ERROR);
					}
				}
				if (nEvent & socket_type::EVENT_WRITE)
				{
					// Send command.
					nCode = m_aCntlCtx.send (NULL);
					if (nCode == REPLY_REQUEST_WAIT)
					{
						// Wait for next event.
						return REPLY_REQUEST_WAIT;
					}
					if (nCode == REPLY_REQUEST_DONE)
					{
						// Read reply.
						nEvent = socket_type::EVENT_READ;
					}
				}
				if (nEvent & socket_type::EVENT_READ)
				{
					// Receive reply.
					nCode = m_aCntlCtx.recv (m_pCtx->m_pCommand);
					if (nCode == REPLY_RESPONSE_WAIT)
					{
						// Wait for next event.
						return REPLY_RESPONSE_WAIT;
					}
					if (nCode/100 == 2)
					{
						// Check restart offset.
						sal_uInt32 nOffset = m_aDataCtx.getOffset();
						if (nOffset)
						{
							// Build REST command.
							OStringBuffer aCommand ("REST ");
							aCommand.append (sal_Int32 (nOffset));
							aCommand.append ("\015\012");

							m_aCntlCtx.copy (aCommand, aCommand.getLength());
						}

						// Send command.
						if (nOffset)
							m_aCntlCtx.setState (cntl_type::STATE_REST);
						else
							m_aCntlCtx.setState (cntl_type::STATE_SEND);
						nEvent = socket_type::EVENT_WRITE;
					}
				}
				break;

			case cntl_type::STATE_REST:
				if (nEvent & socket_type::EVENT_WRITE)
				{
					// Send command.
					nCode = m_aCntlCtx.send (NULL);
					if (nCode == REPLY_REQUEST_WAIT)
					{
						// Wait for next event.
						return REPLY_REQUEST_WAIT;
					}
					if (nCode == REPLY_REQUEST_DONE)
					{
						// Read reply.
						nEvent = socket_type::EVENT_READ;
					}
				}
				if (nEvent & socket_type::EVENT_READ)
				{
					// Receive reply.
					nCode = m_aCntlCtx.recv (m_pCtx->m_pCommand);
					if (nCode == REPLY_RESPONSE_WAIT)
					{
						// Wait for next event.
						return REPLY_RESPONSE_WAIT;
					}
					if (nCode == REPLY_NEED_XFERCMD)
					{
						// Send transfer command.
						m_aCntlCtx.setState (cntl_type::STATE_SEND);
					}
				}
				break;

			case cntl_type::STATE_SEND:
				// Send command.
				nCode = m_aCntlCtx.send (m_pCtx->m_pCommand);
				if (nCode == REPLY_REQUEST_WAIT)
				{
					// Wait for next event.
					return REPLY_REQUEST_WAIT;
				}
				if (nCode == REPLY_REQUEST_DONE)
				{
					// Read reply.
					m_aCntlCtx.setState (cntl_type::STATE_RECV);
				}
				break;

			case cntl_type::STATE_RECV:
				// Receive reply.
				nCode = m_aCntlCtx.recv (m_pCtx->m_pCommand);
				if (nCode == REPLY_RESPONSE_WAIT)
				{
					// Wait for next event.
					return sal_True;
				}
				else if ((nCode/100 == 1) || (nCode == REPLY_TRANSFER_ABORTED))
				{
					// Preliminary success (1xx) or transient failure (426).
					continue;
				}
				else if (nCode/100 == 2)
				{
					// Positive completion (2xx).
					m_aCntlCtx.setState (cntl_type::STATE_DONE);
				}
				else
				{
					// Intermediate success (3xx) or failure (4xx, 5xx).
					m_aCntlCtx.setState (cntl_type::STATE_ERROR);
				}
				break;

			case cntl_type::STATE_DONE:
				if (m_aDataCtx.getState() == data_type::STATE_NONE)
				{
					// Complete command.
					completeCommand (m_pCtx->m_pCommand);
				}
				else
				{
					// Transfer still in progress.
					return REPLY_TRANSFER_WAIT;
				}
			case cntl_type::STATE_ABORT:
			case cntl_type::STATE_ERROR:
				m_aCntlCtx.setState (cntl_type::STATE_NONE);
				break;

			default:
				break;
		}
	}
	return m_aCntlCtx.getStateCode();
}

/*
 * handleTransfer (USER-DTP).
 */
sal_Int32 INetFTPConnection_Impl::handleTransfer (sal_Int32 nEvent)
{
	while (!(m_aDataCtx.getState() == data_type::STATE_NONE))
	{
		// Acquire exclusive access.
		NAMESPACE_VOS(OGuard) aGuard (m_aDataCtx);

		// Check data connection state.
		switch (m_aDataCtx.getState())
		{
			case data_type::STATE_PORT:
				if (nEvent & socket_type::EVENT_LISTEN)
				{
					if (!(nEvent & socket_type::EVENT_OOB))
					{
						// Listening.
						m_aDataCtx.setStateCode (REPLY_CONNECT_WAIT);
						m_aDataCtx.setState (data_type::STATE_OPEN);
					}
					else
					{
						// Failure.
						m_aDataCtx.setStateCode (REPLY_NETWORK_ERROR);
						m_aDataCtx.setState (data_type::STATE_ERROR);
					}
					if (m_aCntlCtx.m_xSocket.isValid())
					{
						// Synchronize with USER-PI.
						m_aCntlCtx.m_xSocket->postEvent (nEvent);
					}
				}

				// Wait for next event.
				return REPLY_TRANSFER_WAIT;

			case data_type::STATE_OPEN:
				if (nEvent & socket_type::EVENT_ACCEPT)
				{
					// Accept data connection.
					NAMESPACE_VOS(OInetSocketAddr) aFromAddr;
					if (m_aDataCtx.m_xPasvSocket->accept (
						m_aDataCtx.m_xDataSocket, aFromAddr))
					{
						// Register event handler.
						m_aDataCtx.m_xDataSocket->registerEventHandler (
							onSocketEvent, this);

						// Enter transfer state.
						m_aDataCtx.setStateCode (REPLY_TRANSFER_WAIT);
						m_aDataCtx.setState (data_type::STATE_XFER);
					}
					else
					{
						// Failure.
						m_aDataCtx.setStateCode (REPLY_NETWORK_ERROR);
						m_aDataCtx.setState (data_type::STATE_ERROR);
					}
				}
				else
				{
					// Wait for next event.
					return REPLY_TRANSFER_WAIT;
				}
				break;

			case data_type::STATE_XFER:
				if (m_pCtx && m_pCtx->m_pSource)
				{
					// Send data stream.
					sal_Int32 nCode = m_aDataCtx.send (m_pCtx->m_pSource);
					if (nCode == REPLY_REQUEST_WAIT)
					{
						// Wait for next event.
						return REPLY_TRANSFER_WAIT;
					}
				}
				if (m_pCtx && m_pCtx->m_pTarget)
				{
					// Receive data stream.
					sal_Int32 nCode = m_aDataCtx.recv (m_pCtx->m_pTarget);
					if (nCode == REPLY_RESPONSE_WAIT)
					{
						// Wait for next event.
						return REPLY_TRANSFER_WAIT;
					}
				}

				// Transfer finished.
				m_aDataCtx.setState (data_type::STATE_DONE);
				break;

			case data_type::STATE_ABORT:
			case data_type::STATE_ERROR:
			case data_type::STATE_DONE:
				INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xDataSocket);
				INETFTP_SOCKET_DISPOSE (m_aDataCtx.m_xPasvSocket);
				m_aDataCtx.setState (data_type::STATE_NONE);

				if (m_aCntlCtx.m_xSocket.isValid())
				{
					// Synchronize with USER-PI.
					m_aCntlCtx.m_xSocket->postEvent (socket_type::EVENT_READ);
				}
				break;

			default:
				break;
		}
	}
	return m_aDataCtx.getStateCode();
}

/*========================================================================
 *
 * INetFTPConnection implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetFTPConnection, inet),
	VOS_NAMESPACE (INetFTPConnection, inet),
	VOS_NAMESPACE (INetClientConnection_Impl, inet),
	0);

/*
 * INetFTPConnection.
 */
INetFTPConnection::INetFTPConnection (void)
{
}

/*
 * ~INetFTPConnection.
 */
INetFTPConnection::~INetFTPConnection (void)
{
}

/*
 * createInstance.
 */
sal_Bool INetFTPConnection::createInstance (
	NAMESPACE_VOS(ORef)<INetFTPConnection> &rxConnection)
{
	rxConnection = new INetFTPConnection_Impl();
	return (rxConnection.isValid());
}

