/*************************************************************************
 *
 *  $RCSfile: inetsmtp.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: th $ $Date: 2001/05/11 12:06:24 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETSMTP_CXX "$Revision: 1.6 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRBUF_HXX_
#include <rtl/strbuf.hxx>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _VOS_MACROS_HXX_
#include <vos/macros.hxx>
#endif
#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif
#ifndef _INETSMTP_HXX
#include <inetsmtp.hxx>
#endif

#ifndef _INETCOREMSG_HXX
#include <inetmsg.hxx>
#endif
#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif

#include <stdlib.h>

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OString;
using rtl::OStringBuffer;

using rtl::OUString;
using rtl::OUStringBuffer;

/*=======================================================================
 *
 * INetCoreSMTPConnection Implementation.
 *
 * Reference: RFC  821 - Simple Mail Transfer Protocol (STD 10).
 *            RFC 1123 - Requirements for Internet Hosts (STD 3).
 *            RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
typedef NAMESPACE_INET(INetSocket) socket_type;

enum INetCoreSMTPStreamState
{
    INETCORESMTP_EOL_BEGIN =  0,
    INETCORESMTP_EOL_SCR   =  1,
    INETCORESMTP_EOL_FCR   =  2,
    INETCORESMTP_EOL_FLF   =  3
};

#define INETCORESMTP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
    (socket)->deregisterEventHandler(onSocketEvent); \
    (socket)->close(); \
    (socket).unbind(); \
}

#define INETCORESMTP_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

//=======================================================================

inline sal_Bool ascii_isDigit( sal_Unicode ch )
{
    return ((ch >= 0x0030) && (ch <= 0x0039));
}

/*========================================================================
 *
 * INetCoreSMTPReplyStream Interface.
 *
 *======================================================================*/
class INetCoreSMTPReplyStream
{
protected:
    sal_Char               *m_pBuffer;
    sal_uInt32              m_nBufSiz;
    sal_uInt32              m_nBufLen;

    INetCoreSMTPStreamState m_eState;
    sal_Bool                m_bFirstLine;

    sal_Int32               m_nReplyCode;
    OString                 m_aReplyText;
    OStringBuffer           m_aReplyBuffer;

    virtual void            AddReplyTextLine (void);
    virtual int             ParseLine (void *pCtx);

public:
    INetCoreSMTPReplyStream (sal_uInt32 nBufSiz = 512);
    virtual ~INetCoreSMTPReplyStream (void);

    int             Write (const sal_Char *b, int l, void *pCtx);

    sal_Int32       GetReplyCode (void) const;
    const sal_Char* GetReplyText (void);
};

inline sal_Int32 INetCoreSMTPReplyStream::GetReplyCode (void) const
{
    return m_nReplyCode;
}

inline const sal_Char* INetCoreSMTPReplyStream::GetReplyText (void)
{
    if (m_aReplyBuffer.getLength())
        m_aReplyText = m_aReplyBuffer.makeStringAndClear();
    if (m_aReplyText.getLength())
        return m_aReplyText.getStr();
    else
        return NULL;
}

/*========================================================================
 *
 * INetCoreSMTPConnectReplyStream Interface.
 *
 *======================================================================*/
class INetCoreSMTPConnectReplyStream : public INetCoreSMTPReplyStream
{
    virtual int ParseLine (void *pCtx);

public:
    INetCoreSMTPConnectReplyStream (sal_uInt32 nBufSiz = 512);
    virtual ~INetCoreSMTPConnectReplyStream (void);
};

/*========================================================================
 *
 * INetCoreSMTPInputStream Interface.
 *
 *======================================================================*/
class INetCoreSMTPInputStream : public INetCoreIStream
{
    sal_uInt32               m_nBufSiz;
    sal_Char                *m_pBuffer;
    sal_Char                *m_pRead;
    sal_Char                *m_pWrite;

    INetCoreSMTPStreamState  m_eState;
    sal_Bool                 m_bEndOfMessage;

    virtual int GetData (sal_Char *pData, ULONG nSize, void *pCtx);

protected:
    virtual int GetLine (sal_Char *pData, ULONG nSize, void *pCtx) = 0;

public:
    INetCoreSMTPInputStream (sal_uInt32 nBufferSize = 1024);
    virtual ~INetCoreSMTPInputStream (void);
};

/*=======================================================================
 *
 * INetCoreSMTPMailInputStream Interface.
 *
 *=====================================================================*/
class INetCoreSMTPMailInputStream : public INetCoreSMTPInputStream
{
    INetCoreMessageIStream *m_pMsgStream;

    virtual int GetLine (sal_Char *pData, ULONG nSize, void *pCtx);

public:
    INetCoreSMTPMailInputStream (INetCoreMessageIStream& rMessageStream);
    virtual ~INetCoreSMTPMailInputStream (void);
};

/*=========================================================================
 *
 * INetCoreSMTPCommandContext Interface.
 *
 *=======================================================================*/
enum INetCoreSMTPCommandState
{
    INETCORESMTP_CMD_SUCCESS = -2,
    INETCORESMTP_CMD_ERROR   = -1,
    INETCORESMTP_CMD_CONNECT =  0,
    INETCORESMTP_CMD_SEND    =  1,
    INETCORESMTP_CMD_RECV    =  2,
    INETCORESMTP_CMD_XFER    =  3
};

#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCoreSMTPCommandContext
{
    INetCoreSMTPCommandContext (
        const OString           &rCommandLine,
        INetCoreSMTPReplyStream *pReplyStream,
        INetCoreSMTPInputStream *pSource,
        INetCoreSMTPCallback    *pfnCallback,
        void                    *pData);
    ~INetCoreSMTPCommandContext (void);

    INetCoreSMTPCommandState  m_eState;
    int                       m_nCode;
    INetCoreSMTPCommandState  m_eOkState;
    int                       m_nOkCode;

    OString                   m_aCmdLine;

    INetCoreSMTPReplyStream  *m_pReplyStream;
    INetCoreSMTPInputStream  *m_pSource;

    INetCoreSMTPCallback     *m_pfnCB;
    void                     *m_pDataCB;
};

#ifdef _USE_NAMESPACE
}
#endif

/*======================================================================
 *
 * INetCoreSMTPConnectionContext Interface.
 *
 *====================================================================*/
#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetCoreSMTPConnectionContext
{
    INetCoreSMTPConnectionContext  (sal_uInt32 nBufferSize = 4096);
    ~INetCoreSMTPConnectionContext (void);

    INetCoreSMTPCommandContext *m_pCmdCtx;
    sal_Bool                    m_bIsOpen;
    sal_Bool                    m_bAborting;


    INetCoreDNSResolver        *m_pResolver;
    INetCoreDNSHostEntry        m_aDestAddr;
    NAMESPACE_VOS(ORef)<INetActiveTCPSocket> m_xSocket;
    sal_uInt32                  m_nXferCount;

    sal_uInt32                  m_nBufSiz;
    sal_Char                   *m_pBuffer;
    sal_Char                   *m_pRead;
    sal_Char                   *m_pWrite;

    INetCoreSMTPCallback       *m_pfnXferCB;
    void                       *m_pXferData;
    INetCoreSMTPCallback       *m_pfnTermCB;
    void                       *m_pTermData;

    /** create.
     */
    void create (INetCoreDNSHostEntry &rDstAddr);
};

#ifdef _USE_NAMESPACE
}
#endif

/*========================================================================
 *
 * INetCoreSMTPConnection Implementation.
 *
 *======================================================================*/
/*
 * INetCoreSMTPConnection.
 */
INetCoreSMTPConnection::INetCoreSMTPConnection (void)
    : m_pConCtx (new INetCoreSMTPConnectionContext())
{
    VOS_POSTCOND(m_pConCtx, "INetCoreSMTPConnection::ctor(): no context");
}

/*
 * ~INetCoreSMTPConnection.
 */
INetCoreSMTPConnection::~INetCoreSMTPConnection (void)
{
    VOS_PRECOND(m_pConCtx, "INetCoreSMTPConnection::dtor(): no context");
    if (m_pConCtx)
    {
        INETCORESMTP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
        delete m_pConCtx;
    }
}

/*
 * handleResolverEvent.
 */
sal_Bool INetCoreSMTPConnection::handleResolverEvent (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    // Check connection context.
    if (m_pConCtx == NULL)
        return 0;

    // Check for idle state (No command context).
    INetCoreSMTPCommandContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (pCtx == NULL)
        return 1;

    // Check for abort during name resolution.
    if (m_pConCtx->m_bAborting)
        nStatus = INETCOREDNS_RESOLVER_ERROR;

    switch (nStatus)
    {
        case INETCOREDNS_RESOLVER_START:
            // Notify caller.
            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCORESMTP_REPLY_RESOLVER_WAIT,
                NULL, pCtx->m_pDataCB);

            // Wait for next event.
            return 1;

        case INETCOREDNS_RESOLVER_SUCCESS:
        case INETCOREDNS_RESOLVER_EXPIRED:
            // Initialize active socket to destination.
            m_pConCtx->create (*pHostEntry);
            m_pConCtx->m_xSocket->registerEventHandler (onSocketEvent, this);

            // Initiate connect.
            if (m_pConCtx->m_xSocket->connect (
                NAMESPACE_VOS(OInetSocketAddr)(
                    pHostEntry->GetDottedDecimalName(),
                    pHostEntry->GetPort())))
            {
                // Notify caller.
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCORESMTP_REPLY_CONNECT_WAIT,
                    NULL, pCtx->m_pDataCB);

                // Wait for connect event.
                return 1;
            }
            else
            {
                // Failure. Cleanup and notify caller.
                m_pConCtx->m_xSocket.unbind();

                m_pConCtx->m_pCmdCtx = NULL;
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    this, INETCORESMTP_REPLY_CONNECT_ERROR,
                    NULL, pCtx->m_pDataCB);
                delete pCtx;
            }
            break;

        default:
            // Failure. Cleanup and notify caller.
            m_pConCtx->m_pCmdCtx = NULL;

            if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                this, INETCORESMTP_REPLY_RESOLVER_ERROR,
                NULL, pCtx->m_pDataCB);
            delete pCtx;
            break;
    }

    // Done.
    return 0;
}

/*
 * handleSocketEvent.
 * Implements SMTP as Finite State Machine.
 */
sal_Bool INetCoreSMTPConnection::handleSocketEvent (
    const NAMESPACE_VOS(ORef)<INetSocket> &rxSocket, sal_Int32 nEvent)
{
    // Check context.
    if (m_pConCtx == NULL)
        return 0;

    // Check command context.
    INetCoreSMTPCommandContext *pCtx = m_pConCtx->m_pCmdCtx;
    if (!pCtx)
    {
        // No command context (STATE_IDLE).
        if (nEvent & socket_type::EVENT_READ)
        {
            // Jump into idle handler.
            while (1)
            {
                sal_Int32 nRead = rxSocket->recv (
                    m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                if (nRead > 0)
                {
                    // Absorbing whatever comes in.
                    continue;
                }
                else if (nRead == INETCORESMTP_SOCKET_WOULDBLOCK)
                {
                    // Wait for next event.
                    return 1;
                }
                else
                {
                    // Connection closed or Network failure.
                    rxSocket->close();
                    return 1;
                }
            }
        }

        if (nEvent & socket_type::EVENT_CLOSE)
        {
            // Connection closed.
            m_pConCtx->m_bIsOpen = sal_False;
            m_pConCtx->m_xSocket.unbind();

            // Notify caller (terminate callback).
            if (m_pConCtx->m_pfnTermCB) (m_pConCtx->m_pfnTermCB) (
                this, INETCORESMTP_REPLY_NETWORK_ERROR, NULL,
                m_pConCtx->m_pTermData);
        }

        // Leave.
        return 1;
    }

    // Check event.
    if (nEvent & socket_type::EVENT_CLOSE)
    {
        // Premature closure.
        m_pConCtx->m_bIsOpen = sal_False;
        m_pConCtx->m_xSocket.unbind();

        // Finish with network error.
        m_pConCtx->m_pCmdCtx->m_eState = INETCORESMTP_CMD_ERROR;
        m_pConCtx->m_pCmdCtx->m_nCode  = INETCORESMTP_REPLY_NETWORK_ERROR;
    }

    // Jump into state machine.
    while (1)
    {
        switch (pCtx->m_eState)
        {
            case INETCORESMTP_CMD_CONNECT:
                if (nEvent & socket_type::EVENT_CONNECT)
                {
                    if (nEvent & socket_type::EVENT_OOB)
                    {
                        // Finish with connect error.
                        pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                        pCtx->m_nCode  = INETCORESMTP_REPLY_CONNECT_ERROR;

                        // Cleanup.
                        m_pConCtx->m_xSocket.unbind();
                    }
                    else
                    {
                        // Read greeting reply.
                        pCtx->m_eState = INETCORESMTP_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORESMTP_CMD_SEND:
                if (nEvent & socket_type::EVENT_WRITE)
                {
                    // Write out command buffer.
                    OString aCommand (pCtx->m_aCmdLine);

                    sal_Int32 nWrite = rxSocket->send (
                        aCommand.pData->buffer, aCommand.pData->length);
                    if (nWrite > 0)
                    {
                        // Read reply.
                        pCtx->m_eState = INETCORESMTP_CMD_RECV;
                        nEvent = socket_type::EVENT_READ;
                    }
                    else if (nWrite == INETCORESMTP_SOCKET_WOULDBLOCK)
                    {
                        // Wait for next event.
                        return 1;
                    }
                    else
                    {
                        // Socket write error.
                        pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                        pCtx->m_nCode = INETCORESMTP_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORESMTP_CMD_RECV:
                if (nEvent & socket_type::EVENT_READ)
                {
                    sal_Int32 nRead = rxSocket->recv (
                        m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz);
                    if (nRead > 0)
                    {
                        // Ok, analyze reply.
                        int status = pCtx->m_pReplyStream->Write (
                            m_pConCtx->m_pBuffer, nRead, m_pConCtx);
                        if (status == INETCORESTREAM_STATUS_LOADED)
                        {
                            pCtx->m_nCode =
                                pCtx->m_pReplyStream->GetReplyCode();
                            if (pCtx->m_nCode == pCtx->m_nOkCode)
                            {
                                pCtx->m_eState =
                                    pCtx->m_eOkState;
                                pCtx->m_eOkState =
                                    INETCORESMTP_CMD_SUCCESS;
                                pCtx->m_nOkCode =
                                    INETCORESMTP_REPLY_ACTION_OK;
                                nEvent = socket_type::EVENT_WRITE;
                            }
                            else
                            {
                                pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                            }
                        }
                        else
                        {
                            if (status != INETCORESTREAM_STATUS_OK)
                            {
                                // Syntax error, i.e. no reply code.
                                pCtx->m_eState =
                                    INETCORESMTP_CMD_ERROR;
                                pCtx->m_nCode =
                                    INETCORESMTP_REPLY_COMMAND_SYNTAX;
                            }
                        }
                    }
                    else if (nRead == INETCORESMTP_SOCKET_WOULDBLOCK)
                    {
                        // Wait for next event.
                        return 1;
                    }
                    else
                    {
                        // Socket read error.
                        pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                        pCtx->m_nCode  = INETCORESMTP_REPLY_NETWORK_ERROR;
                        rxSocket->close();
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            case INETCORESMTP_CMD_XFER:
                if (nEvent & socket_type::EVENT_WRITE)
                {
                    // Send mail body.
                    if ((m_pConCtx->m_pRead - m_pConCtx->m_pWrite) > 0)
                    {
                        // Bytes not yet written. Write out to Socket.
                        sal_Int32 nWrite = rxSocket->send (
                            m_pConCtx->m_pWrite,
                            (m_pConCtx->m_pRead - m_pConCtx->m_pWrite));
                        if (nWrite > 0)
                        {
                            // Have written some bytes.
                            m_pConCtx->m_nXferCount += nWrite;
                            m_pConCtx->m_pWrite     += nWrite;
                        }
                        else if (nWrite == INETCORESMTP_SOCKET_WOULDBLOCK)
                        {
                            // Socket would block. Notify caller.
                            INetCoreSMTPCallback *pfnCB =
                                m_pConCtx->m_pfnXferCB;
                            if (pfnCB) (pfnCB) (
                                this, INETCORESMTP_REPLY_TRANSFER_WAIT,
                                NULL, m_pConCtx->m_pXferData);

                            // Wait for next write event.
                            return 1;
                        }
                        else
                        {
                            // Socket write error.
                            pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                            pCtx->m_nCode = INETCORESMTP_REPLY_NETWORK_ERROR;
                            rxSocket->close();
                        }
                    }
                    else
                    {
                        // Buffer empty. Reset to begin.
                        m_pConCtx->m_pRead = m_pConCtx->m_pWrite =
                            m_pConCtx->m_pBuffer;

                        // Read source stream.
                        int nRead = pCtx->m_pSource->Read (
                            m_pConCtx->m_pBuffer, m_pConCtx->m_nBufSiz,
                            m_pConCtx);
                        if (nRead > 0)
                        {
                            // Have read some bytes.
                            m_pConCtx->m_pRead += nRead;
                        }
                        else if (nRead == 0)
                        {
                            // Source loaded. Read reply.
                            pCtx->m_eState = INETCORESMTP_CMD_RECV;
                            nEvent = socket_type::EVENT_READ;
                        }
                        else
                        {
                            // Source read error.
                            pCtx->m_eState = INETCORESMTP_CMD_ERROR;
                            pCtx->m_nCode  = 0;
                        }
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            default: // (_SUCCESS || _ERROR)
                if (pCtx)
                {
                    // Restore idle state.
                    m_pConCtx->m_pCmdCtx = NULL;

                    // Notify caller.
                    if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                        this, pCtx->m_nCode,
                        pCtx->m_pReplyStream->GetReplyText(),
                        pCtx->m_pDataCB);

                    // Cleanup and leave.
                    delete pCtx;
                    return 1;
                }
                break;
        } // switch (pCtx->m_eState)
    } // while (1)
}

/*
 * StartCommand.
 */
sal_Bool INetCoreSMTPConnection::StartCommand (
    INetCoreSMTPCommandContext *pCtx)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if (m_pConCtx && m_pConCtx->m_bIsOpen && !m_pConCtx->m_bAborting)
    {
        // Check command context.
        if (!m_pConCtx->m_pCmdCtx && pCtx)
        {
            // Setup connection context.
            m_pConCtx->m_pCmdCtx = pCtx;
            m_pConCtx->m_nXferCount = 0;

            // Start command.
            if (m_pConCtx->m_xSocket->postEvent (socket_type::EVENT_WRITE))
            {
                // Command started.
                return sal_True;
            }
            m_pConCtx->m_pCmdCtx = NULL;
        }
    }

    // Command failed.
    delete pCtx;
    return sal_False;
}

/*
 * Open.
 * Initiate connection to SMTPD.
 */
sal_Bool INetCoreSMTPConnection::Open (
    const OUString &rHost, sal_uInt16 nPort,
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

    // Check connection context.
    if ((m_pConCtx == NULL) || (m_pConCtx->m_bIsOpen))
        return sal_False;

    // Check arguments.
    if ((rHost.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;
    if (nPort == 0)
        nPort = INETCORESMTP_DEF_PORT;

    // Get this machines domain name.
    INetCoreDNSHostEntry aHostent (OUString::createFromAscii ("localhost"));
    m_pConCtx->m_pResolver->GetHostName (&aHostent);

    // Construct HELO command line with own domain name.
    OStringBuffer aBuffer ("HELO ");
    if (aHostent.GetCanonicalName().getLength())
    {
        OUString aDomain (aHostent.GetCanonicalName());
        aBuffer.append (OString (
            aDomain.pData->buffer, aDomain.pData->length,
            RTL_TEXTENCODING_ASCII_US));
    }
    else if (aHostent.GetDomainName().getLength())
    {
        OUString aDomain (aHostent.GetDomainName());
        aBuffer.append (OString (
            aDomain.pData->buffer, aDomain.pData->length,
            RTL_TEXTENCODING_ASCII_US));
    }
    else
    {
        aBuffer.append ("localhost");
    }
    aBuffer.append ("\015\012");

    // Initialize command context.
    INetCoreSMTPCommandContext *pCtx = new INetCoreSMTPCommandContext (
        aBuffer.makeStringAndClear(),
        new INetCoreSMTPConnectReplyStream(), NULL,
        pfnCallback, pData);

    pCtx->m_eState   = INETCORESMTP_CMD_CONNECT;
    pCtx->m_eOkState = INETCORESMTP_CMD_SEND;
    pCtx->m_nOkCode  = INETCORESMTP_REPLY_SERVICE_READY;

    // Set connection context.
    m_pConCtx->m_pCmdCtx = pCtx;

    // Start domain name resolution.
    m_pConCtx->m_aDestAddr = INetCoreDNSHostEntry (rHost, nPort);

    if (!m_pConCtx->m_pResolver->GetHostByName (
        &(m_pConCtx->m_aDestAddr), onResolverEvent, this))
    {
        // Cleanup and fail.
        DELETEZ (m_pConCtx->m_pCmdCtx);
    }
    return (m_pConCtx->m_pCmdCtx != NULL);
}

/*
 * IsOpen.
 */
sal_Bool INetCoreSMTPConnection::IsOpen (void)
{
    return (m_pConCtx ? m_pConCtx->m_bIsOpen : sal_False);
}

/*
 * Close.
 */
sal_Bool INetCoreSMTPConnection::Close (
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    // Initialize command context.
    INetCoreSMTPCommandContext *pCtx = new INetCoreSMTPCommandContext (
        "QUIT\015\012", new INetCoreSMTPConnectReplyStream, NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORESMTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORESMTP_REPLY_SERVICE_CLOSING;

    return StartCommand (pCtx);
}

/*
 * Destroy.
 */
void INetCoreSMTPConnection::Destroy (void)
{
    if (m_pConCtx)
    {
        m_pConCtx->m_bAborting = sal_True;

        if (m_pConCtx->m_pCmdCtx)
        {
            m_pConCtx->m_pCmdCtx->m_pSource = NULL; /* @@@ */
            m_pConCtx->m_pCmdCtx->m_pfnCB   = NULL;
            m_pConCtx->m_pCmdCtx->m_pDataCB = NULL;
        }

        m_pConCtx->m_pfnXferCB = NULL;
        m_pConCtx->m_pXferData = NULL;

        m_pConCtx->m_pfnTermCB = NULL;
        m_pConCtx->m_pTermData = NULL;

        INETCORESMTP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
    }
}

/*
 * _GetAddressFromMailbox.
 */
static OString _GetAddressFromMailbox (const OUString &rMailbox)
{
    /*
     * addr-spec := local-part "@" domain.
     */
    UniString aAddress (rMailbox);
    xub_StrLen nBra = aAddress.Search ('<');
    xub_StrLen nKet = aAddress.Search ('>');
    if ((nBra == STRING_NOTFOUND) && (nKet == STRING_NOTFOUND))
    {
        /*
         * mailbox := [comment] addr-spec [comment].
         * Remove (optional) comment(s).
         */
        const sal_Unicode *pLParen = NULL, *pRParen = NULL;
        sal_Int32 nLParen = 0;

        const sal_Unicode *pStart = aAddress.GetBuffer();
        const sal_Unicode *p = pStart;

        while (*p)
        {
            switch (*p)
            {
                case '(':
                    if (nLParen++ == 0) pLParen = p;
                    break;

                case ')':
                    if (--nLParen == 0) pRParen = p;
                    break;

                default:
                    break;
            }
            p++;

            if (nLParen == 0)
            {
                if (pLParen && pRParen)
                {
                    // Cut out comment.
                    aAddress.Erase ((xub_StrLen)(pLParen - pStart),
                                    (xub_StrLen)(pRParen - pLParen + 1));

                    // Reset (if not yet finished).
                    if (*p)
                    {
                        pLParen = pRParen = NULL;

                        pStart = aAddress.GetBuffer();
                        p = pStart;
                    }
                }
            }
        }

        // Remove leading and trailing spaces.
        aAddress.EraseLeadingAndTrailingChars (' ');
    }
    else
    {
        /*
         * mailbox := phrase "<" addr-spec ">".
         * Copy addr-spec.
         */
        if (nBra == STRING_NOTFOUND)
            aAddress = aAddress.Copy (0, nKet - 1);
        else if (nKet == STRING_NOTFOUND)
            aAddress = aAddress.Copy (nBra + 1, aAddress.Len() - nBra);
        else
            aAddress = aAddress.Copy (nBra + 1, nKet - nBra - 1);
    }

    // Done.
    return OString (
        aAddress.GetBuffer(), aAddress.Len(), RTL_TEXTENCODING_ASCII_US);
}

/*
 * MailFrom.
 */
sal_Bool INetCoreSMTPConnection::MailFrom (
    const OUString &rSender,
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rSender.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Extract <reverse-path>.
    OString aPath (_GetAddressFromMailbox (rSender));

    // Build command line.
    OStringBuffer aBuffer ("MAIL FROM:<");
    aBuffer.append (aPath);
    aBuffer.append (">\015\012");

    // Initialize command context.
    INetCoreSMTPCommandContext *pCtx = new INetCoreSMTPCommandContext (
        aBuffer.makeStringAndClear(),
        new INetCoreSMTPReplyStream(), NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORESMTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORESMTP_REPLY_ACTION_OK;

    // Start MAIL command.
    return StartCommand (pCtx);
}

/*
 * MailTo.
 */
sal_Bool INetCoreSMTPConnection::MailTo (
    const OUString &rRecipient,
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if ((rRecipient.getLength() == 0) || (pfnCallback == NULL))
        return sal_False;

    // Extract <forward-path>.
    OString aPath (_GetAddressFromMailbox (rRecipient));

    // Build command line.
    OStringBuffer aBuffer ("RCPT TO:<");
    aBuffer.append (aPath);
    aBuffer.append (">\015\012");

    // Initialize command context.
    INetCoreSMTPCommandContext *pCtx = new INetCoreSMTPCommandContext (
        aBuffer.makeStringAndClear(),
        new INetCoreSMTPReplyStream(), NULL,
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORESMTP_CMD_SUCCESS;
    pCtx->m_nOkCode  = INETCORESMTP_REPLY_ACTION_OK;

    // Start RCPT command.
    return StartCommand (pCtx);
}

/*
 * TransferData.
 */
sal_Bool INetCoreSMTPConnection::TransferData (
    INetCoreMessageIStream& rMessageStream,
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    // Check arguments.
    if (pfnCallback == NULL)
        return sal_False;

    // Initialize command context.
    INetCoreSMTPCommandContext *pCtx = new INetCoreSMTPCommandContext (
        "DATA\015\012", new INetCoreSMTPReplyStream,
        new INetCoreSMTPMailInputStream (rMessageStream),
        pfnCallback, pData);

    pCtx->m_eOkState = INETCORESMTP_CMD_XFER;
    pCtx->m_nOkCode  = INETCORESMTP_REPLY_START_MAILING;

    // Start DATA command.
    return StartCommand (pCtx);
}

/*
 * GetTransferCount.
 */
sal_uInt32 INetCoreSMTPConnection::GetTransferCount (void)
{
    return (m_pConCtx ? m_pConCtx->m_nXferCount : 0);
}

/*
 * SetTransferCallback.
 */
sal_Bool INetCoreSMTPConnection::SetTransferCallback (
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    if (m_pConCtx && !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnXferCB = pfnCallback;
        m_pConCtx->m_pXferData = pData;
        return sal_True;
    }
    return sal_False;
}

/*
 * SetTerminateCallback.
 */
sal_Bool INetCoreSMTPConnection::SetTerminateCallback (
    INetCoreSMTPCallback *pfnCallback, void *pData)
{
    if (m_pConCtx && !m_pConCtx->m_bAborting)
    {
        m_pConCtx->m_pfnTermCB = pfnCallback;
        m_pConCtx->m_pTermData = pData;
        return sal_True;
    }
    return sal_False;
}

/*======================================================================
 *
 * INetCoreSMTPConnectionContext Implementation.
 *
 *====================================================================*/
/*
 * INetCoreSMTPConnectionContext.
 */
INetCoreSMTPConnectionContext::INetCoreSMTPConnectionContext (
    sal_uInt32 nBufferSize)
    : m_pCmdCtx (NULL),
      m_bIsOpen   (sal_False),
      m_bAborting (sal_False),
      m_pResolver (new INetCoreDNSResolver()),
      m_aDestAddr (OUString(), 0),
      m_xSocket   (NULL),
      m_nXferCount (0),
      m_nBufSiz    (nBufferSize),
      m_pBuffer    ((sal_Char*)(rtl_allocateMemory (m_nBufSiz))),
      m_pRead      (m_pBuffer),
      m_pWrite     (m_pBuffer),
      m_pfnXferCB  (NULL),
      m_pXferData  (NULL),
      m_pfnTermCB  (NULL),
      m_pTermData  (NULL)
{
}

/*
 * ~INetCoreSMTPConnectionContext.
 */
INetCoreSMTPConnectionContext::~INetCoreSMTPConnectionContext (void)
{
    rtl_freeMemory (m_pBuffer);
    if (m_xSocket.isValid())
    {
        m_xSocket->close();
        m_xSocket.unbind();
    }
    delete m_pResolver;
    delete m_pCmdCtx;
}

/*
 * create.
 */
void INetCoreSMTPConnectionContext::create (INetCoreDNSHostEntry &rDstAddr)
{
    // Initialize active socket.
    m_xSocket = new INetActiveTCPSocket();

    // Check for SocksGateway.
    NAMESPACE_VOS(ORef)<INetConfig> xConfig;
    if (INetConfig::getOrCreate (xConfig))
    {
        NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
            xConfig->getProxyPolicy());
        if (xProxyPolicy.isValid())
        {
            OUStringBuffer aBuffer (OUString::createFromAscii ("out://"));
            if (rDstAddr.GetCanonicalName().getLength())
                aBuffer.append (rDstAddr.GetCanonicalName());
            else
                aBuffer.append (rDstAddr.GetDomainName());
            aBuffer.append (sal_Unicode (':'));
            aBuffer.append (rDstAddr.GetPort());
            aBuffer.append (sal_Unicode ('/'));

            OUString        aUrl (aBuffer.makeStringAndClear());
            INetProxyConfig aProxyConfig;

            if (xProxyPolicy->shouldUseProxy (aUrl, aProxyConfig))
            {
                if (aProxyConfig.hasSocksProxy())
                {
                    // Use SocksGateway.
                    m_xSocket->setSocksGateway (
                        NAMESPACE_VOS(OInetSocketAddr)(
                            aProxyConfig.getSocksProxyName(),
                            aProxyConfig.getSocksProxyPort()));
                }
            }
        }
    }
}

/*========================================================================
 *
 * INetCoreSMTPCommandContext Implementation.
 *
 *======================================================================*/
INetCoreSMTPCommandContext::INetCoreSMTPCommandContext (
    const OString           &rCommandLine,
    INetCoreSMTPReplyStream *pReplyStream,
    INetCoreSMTPInputStream *pSource,
    INetCoreSMTPCallback    *pfnCallback,
    void                    *pData)
    : m_eState       (INETCORESMTP_CMD_SEND),
      m_nCode        (0),
      m_eOkState     (INETCORESMTP_CMD_SUCCESS),
      m_nOkCode      (INETCORESMTP_REPLY_ACTION_OK),
      m_aCmdLine     (rCommandLine),
      m_pReplyStream (pReplyStream),
      m_pSource      (pSource),
      m_pfnCB        (pfnCallback),
      m_pDataCB      (pData)
{
}

INetCoreSMTPCommandContext::~INetCoreSMTPCommandContext (void)
{
    delete m_pSource;
    delete m_pReplyStream;
}

/*========================================================================
 *
 * INetCoreSMTPReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreSMTPReplyStream.
 */
INetCoreSMTPReplyStream::INetCoreSMTPReplyStream (sal_uInt32 nBufSiz)
{
    m_nBufSiz = VOS_MAX(nBufSiz, 512);
    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_nBufLen = 0;

    m_eState     = INETCORESMTP_EOL_BEGIN;
    m_bFirstLine = sal_True;
    m_nReplyCode = 0;
}

/*
 * ~INetCoreSMTPReplyStream.
 */
INetCoreSMTPReplyStream::~INetCoreSMTPReplyStream (void)
{
    rtl_freeMemory (m_pBuffer);
}

/*
 * Write.
 */
int INetCoreSMTPReplyStream::Write (const char *b, int l, void *pCtx)
{
    int status;

    while (l--)
    {
        if (m_eState == INETCORESMTP_EOL_FCR)
        {
            if (*b == '\012')
            {
                // Found CRLF.
                *(m_pBuffer + m_nBufLen) = '\0';
                AddReplyTextLine();
                status = ParseLine (pCtx);
                if (status == INETCORESTREAM_STATUS_ERROR)
                    return status;
                if ((status == INETCORESTREAM_STATUS_LOADED) && (l == 0))
                    return status;
            }
            else
            {
                // Found CR only.
                *(m_pBuffer + m_nBufLen) = '\0';
                AddReplyTextLine();
                status = ParseLine (pCtx);
                if (status == INETCORESTREAM_STATUS_ERROR)
                    return status;
                if ((status == INETCORESTREAM_STATUS_LOADED) && (l == 0))
                    return status;
            }
        }
        else if (*b == '\015')
        {
            // Found CR.
            m_eState = INETCORESMTP_EOL_FCR;
        }
        else if (*b == '\012')
        {
            // Found LF only.
            *(m_pBuffer + m_nBufLen) = '\0';
            AddReplyTextLine();
            status = ParseLine (pCtx);
            if (status == INETCORESTREAM_STATUS_ERROR)
                return status;
            if ((status == INETCORESTREAM_STATUS_LOADED) && (l == 0))
                return status;
        }
        else
        {
            *(m_pBuffer + m_nBufLen++) = *b;
            if (m_nBufLen == m_nBufSiz)
            {
                *(m_pBuffer + m_nBufLen) = '\0';
                AddReplyTextLine();
                status = ParseLine (pCtx);
                if (status == INETCORESTREAM_STATUS_ERROR)
                    return status;
                if ((status == INETCORESTREAM_STATUS_LOADED) && (l == 0))
                    return status;
            }
        }
        b++;
    }
    return INETCORESTREAM_STATUS_OK;
}

/*
 * AddReplyTextLine.
 */
void INetCoreSMTPReplyStream::AddReplyTextLine (void)
{
    const sal_Char *p = m_pBuffer;
    if (ascii_isDigit (*p))
    {
        // Skip over reply code.
        p += 4;
    }
    m_aReplyBuffer.append (p);
    m_aReplyBuffer.append (sal_Char ('\012'));
}

/*
 * ParseLine.
 */
int INetCoreSMTPReplyStream::ParseLine (void *pCtx)
{
    int  reply = 0;
    char cont  = '\0';

    if (ascii_isDigit (*m_pBuffer))
    {
        if (m_nBufLen > 3)
        {
            cont = *(m_pBuffer + 3);
            *(m_pBuffer + 3) = '\0';
            reply = atoi (m_pBuffer);
            *(m_pBuffer + 3) = cont;
        }
        else
        {
            reply = atoi (m_pBuffer);
        }
    }

    if (m_bFirstLine)
    {
        if (!reply) return INETCORESTREAM_STATUS_ERROR;
        m_nReplyCode = reply;
        m_bFirstLine = sal_False;
    }

    m_nBufLen = 0;
    m_eState = INETCORESMTP_EOL_BEGIN;

    if (cont != '-')
    {
        m_bFirstLine = sal_True;
        return INETCORESTREAM_STATUS_LOADED;
    }

    return INETCORESTREAM_STATUS_OK;
}

/*========================================================================
 *
 * INetCoreSMTPConnectReplyStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreSMTPConnectReplyStream.
 */
INetCoreSMTPConnectReplyStream::INetCoreSMTPConnectReplyStream (
    sal_uInt32 nBufSiz)
    : INetCoreSMTPReplyStream (nBufSiz)
{
}

/*
 * ~INetCoreSMTPConnectReplyStream.
 */
INetCoreSMTPConnectReplyStream::~INetCoreSMTPConnectReplyStream (void)
{
}

/*
 * ParseLine.
 */
int INetCoreSMTPConnectReplyStream::ParseLine (void *pCtx)
{
    int status = INetCoreSMTPReplyStream::ParseLine (pCtx);
    if (status == INETCORESTREAM_STATUS_LOADED)
    {
        INetCoreSMTPConnectionContext *ctx =
            (INetCoreSMTPConnectionContext *)pCtx;
        int nCode = GetReplyCode();
        ctx->m_bIsOpen = ((nCode == INETCORESMTP_REPLY_SERVICE_READY) ||
                          (nCode == INETCORESMTP_REPLY_ACTION_OK    )    );
    }
    return status;
}

/*========================================================================
 *
 * INetCoreSMTPInputStream Implementation.
 *
 *======================================================================*/
/*
 * INetCoreSMTPInputStream.
 */
INetCoreSMTPInputStream::INetCoreSMTPInputStream (sal_uInt32 nBufferSize)
    : INetCoreIStream ()
{
    m_nBufSiz = nBufferSize;
    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_pRead = m_pWrite = m_pBuffer;

    m_eState = INETCORESMTP_EOL_SCR;
    m_bEndOfMessage = sal_False;
}

/*
 * ~INetCoreSMTPInputStream.
 */
INetCoreSMTPInputStream::~INetCoreSMTPInputStream (void)
{
    rtl_freeMemory (m_pBuffer);
}

/*
 * GetData.
 */
int INetCoreSMTPInputStream::GetData (sal_Char *pData, ULONG nSize, void *pCtx)
{
    sal_Char *pWrite = pData;
    while (pWrite < (pData + nSize))
    {
        // Caller's buffer not yet filled.
        if ((m_pRead - m_pWrite) > 0)
        {
            // Bytes still in buffer.
            if (m_eState == INETCORESMTP_EOL_BEGIN)
            {
                m_eState = INETCORESMTP_EOL_SCR;
                if ((*m_pWrite == '.') && !m_bEndOfMessage)
                {
                    /*
                     * A period as the first character must be doubled,
                     * except at end of message, of course.
                     */
                    *pWrite++ = '.';
                }
            }
            else if (m_eState == INETCORESMTP_EOL_FCR)
            {
                m_eState = INETCORESMTP_EOL_BEGIN;
                if (*m_pWrite != '\012')
                {
                    // Found <CR> only. Insert missing <LF>.
                    *pWrite++ = '\012';
                }
                else
                {
                    // Found standard <CR><LF> sequence.
                    *pWrite++ = *m_pWrite++;
                }
            }
            else if (*m_pWrite == '\015')
            {
                // Found <CR>.
                m_eState = INETCORESMTP_EOL_FCR;
                *pWrite++ = *m_pWrite++;
            }
            else if (*m_pWrite == '\012')
            {
                // Found <LF> only. Insert missing <CR>.
                m_eState = INETCORESMTP_EOL_FCR;
                *pWrite++ = '\015';
            }
            else
            {
                *pWrite++ = *m_pWrite++;
            }
        }
        else
        {
            // Buffer empty. Reset to <Begin-of-Buffer>.
            m_pRead = m_pWrite = m_pBuffer;

            // Read next message line.
            int nRead = GetLine (m_pBuffer, m_nBufSiz, pCtx);
            if (nRead > 0)
            {
                // Set read pointer.
                m_pRead = m_pBuffer + nRead;
            }
            else
            {
                if (!m_bEndOfMessage)
                {
                    // Generate <End-of-Message> sequence.
                    if (!(m_eState == INETCORESMTP_EOL_BEGIN))
                    {
                        *m_pRead++ = '\015';
                        *m_pRead++ = '\012';
                    }
                    *m_pRead++ = '.';
                    *m_pRead++ = '\015';
                    *m_pRead++ = '\012';

                    // Mark we're done.
                    m_bEndOfMessage = sal_True;
                }
                else
                {
                    // Done.
                    return (pWrite - pData);
                }
            }
        }
    }
    return (pWrite - pData);
}

/*=======================================================================
 *
 * INetCoreSMTPMailInputStream Implementation.
 *
 *=====================================================================*/
/*
 * INetCoreSMTPMailInputStream.
 */
INetCoreSMTPMailInputStream::INetCoreSMTPMailInputStream (
    INetCoreMessageIStream& rMessageStream)
    : INetCoreSMTPInputStream ()
{
    m_pMsgStream = &rMessageStream;
}

/*
 * ~INetCoreSMTPMailInputStream.
 */
INetCoreSMTPMailInputStream::~INetCoreSMTPMailInputStream (void)
{
}

/*
 * GetLine.
 */
int INetCoreSMTPMailInputStream::GetLine (
    sal_Char *pData, ULONG nSize, void *pCtx)
{
    INetCoreSMTPConnectionContext *ctx =
        (INetCoreSMTPConnectionContext *)pCtx;

    if ((ctx == NULL) || (ctx->m_bAborting))
        return INETCORESTREAM_STATUS_ERROR;
    else
        return m_pMsgStream->Read (pData, nSize, NULL);
}

