/*************************************************************************
 *
 *  $RCSfile: inethttp.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: pl $ $Date: 2001/05/11 12:23:22 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETHTTP_CXX "$Revision: 1.4 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif

#ifndef _VOS_MACROS_HXX_
#include <vos/macros.hxx>
#endif
#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_OBJECT_HXX_
#include <vos/object.hxx>
#endif
#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif
#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif
#ifndef _TOOLS_INETMSG_HXX
#include <tools/inetmsg.hxx>
#endif
#ifndef _TOOLS_INETSTRM_HXX
#include <tools/inetstrm.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_CLIENT_HXX
#include <inet/client.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif

#ifndef _INETHTTP_HXX
#include <inethttp.hxx>
#endif

#ifdef SOLAR_SSL
#ifndef _INETSSL_HXX
#include <inetssl.hxx>
#endif
#endif /* SOLAR_SSL */

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OString;
using rtl::OUString;

#ifndef INETSTREAM_STATUS_HEADER
#define INETSTREAM_STATUS_HEADER (INETSTREAM_STATUS_LOADED - 2)
#endif

/*========================================================================
 *
 * INetHTTP implementation.
 *
 * References:
 *   RFC 1945 - Hypertext Transfer Protocol -- HTTP/1.0 (Informational)
 *   RFC 2068 - Hypertext Transfer Protocol -- HTTP/1.1 (Proposed)
 *   RFC 2500 - Internet Official Protocol Standards    (STD 1)
 *
 *======================================================================*/
#define INETHTTP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
    (socket)->deregisterEventHandler(onSocketEvent); \
    (socket)->close(); \
    (socket).unbind(); \
}

#define INETHTTP_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

#ifdef DBG_UTIL_PRIVATE
#include <stdio.h>

#if (defined(WIN) || defined(WNT) || defined(OS2))
#define _TRACELOGNAME "c:\\inethttp.log"
#elif (defined(UNX))
#define _TRACELOGNAME "/tmp/inethttp.log"
#else
#define _TRACELOGNAME "inethttp.log"
#endif

#ifdef WIN
#define _TRACELOGFORMAT "%ls\n"
#else
#define _TRACELOGFORMAT "%s\n"
#endif

#define _TRACE(s) \
{ \
    FILE *fp = fopen (_TRACELOGNAME, "a+"); \
    if (fp) \
    { \
        fprintf (fp, _TRACELOGFORMAT, (const char *)(s)); \
        fclose (fp); \
    } \
}

#endif /* DBG_UTIL_PRIVATE */

/*========================================================================
 *
 * INetHTTPInputStream interface.
 *
 *======================================================================*/
class INetHTTPInputStream : public INetIStream
{
public:
    INetHTTPInputStream (
        INetHTTPMessage &rMessage, sal_uInt32 nBufSiz = 1024);
    virtual ~INetHTTPInputStream (void);

private:
    /** State.
     */
    enum State
    {
        STATE_BEGIN,
        STATE_METHOD,
        STATE_URI,
        STATE_VERSION
    };

    /** Representation.
     */
    INetHTTPMessageStream m_aStrm;
    State                 m_eState;
    sal_Bool              m_bRequestGenerated;

    sal_uInt32            m_nBufSiz;
    sal_Char             *m_pBuffer;
    sal_Char             *m_pRead;
    sal_Char             *m_pWrite;

    /** Implementation.
     */
    virtual int GetData (sal_Char *pData, ULONG nSize, void *pCtx);
    virtual int GetLine (sal_Char *pData, ULONG nSize, void *pCtx);
};

/*========================================================================
 *
 * INetHTTPOutputStream interface.
 *
 *======================================================================*/
class INetHTTPOutputStream : public INetOStream
{
public:
    INetHTTPOutputStream (
        INetHTTPMessage &rMessage, sal_uInt32 nBufSiz = 256);
    virtual ~INetHTTPOutputStream (void);

private:
    /** Representation.
     */
    INetHTTPMessageStream  m_aStrm;
    INetMessageStreamState m_eState;
    sal_Bool               m_bStatusParsed;

    sal_uInt32             m_nBufSiz;
    sal_uInt32             m_nBufLen;
    sal_Char              *m_pBuffer;

    /** Implementation.
     */
    virtual int PutData (const sal_Char *pData, ULONG nSize, void *pCtx);
    virtual int PutLine (const sal_Char *pData, ULONG nSize, void *pCtx);
};

/*========================================================================
 *
 * INetHTTPRequestContext interface.
 *
 *======================================================================*/
typedef INetHTTPRequest        request_type;
typedef INetHTTPRequestContext context_type;

#ifdef _USE_NAMESPACE
namespace inet {
#endif

struct INetHTTPRequestContext : public NAMESPACE_VOS(OMutex)
{
    /** Destination.
     */
    enum Destination
    {
        DEST_ORIGIN,
        DEST_PROXY,
        DEST_NONE
    };

    /** State.
     */
    enum State
    {
        STATE_ABORT = -2,
        STATE_ERROR = -1,
        STATE_BEGIN =  0,
        STATE_RESOLVE,
        STATE_CONNECT,
        STATE_SEND,
        STATE_RECV,
        STATE_DONE
    };

    /** StateBlock.
     */
    struct StateBlock
    {
        State      m_ePrev;
        State      m_eState;

        sal_Int32  m_nState;
        sal_Int32  m_nReply;
        sal_Char  *m_pReply;

        StateBlock (void)
            : m_ePrev  (STATE_BEGIN),
              m_eState (STATE_BEGIN),
              m_nState (-1),
              m_nReply (-1),
              m_pReply (NULL)
        {}
        ~StateBlock (void)
        {
            rtl_freeMemory (m_pReply);
        }

        State getState (void) const { return m_eState; }
        void  setState (State eState)
        {
            m_ePrev  = m_eState;
            m_eState = eState;
        }

        sal_Int32 getStateCode (void) const { return m_nState; }
        void setStateCode (sal_Int32 nState)
        {
            m_nState = nState;
        }

        sal_Int32 getReplyCode (void) const
        {
            return ((m_nReply >= 0) ? m_nReply : m_nState);
        }
        void setReplyCode (sal_Int32 nReply)
        {
            m_nReply = nReply;
        }

        const sal_Char* getReplyText (void) const { return m_pReply; }
        void setReplyText (const sal_Char *pReply)
        {
            sal_Int32 n = rtl_str_getLength (pReply) + 1;
            m_pReply = (sal_Char*)(rtl_reallocateMemory (m_pReply, n));
            rtl_copyMemory (m_pReply, pReply, n);
        }
    };

    /** SendBlock.
     */
    struct SendBlock
    {
        INetHTTPInputStream *m_pStrm;
        sal_uInt32           m_nDocSiz;
        sal_uInt32           m_nDocLen;

        SendBlock (void)
            : m_pStrm(NULL), m_nDocSiz(0), m_nDocLen(0)
        {}
        ~SendBlock (void)
        {
            delete m_pStrm;
        }

        void create (INetHTTPMessage &rMsg)
        {
            m_pStrm   = new INetHTTPInputStream (rMsg);
            m_nDocSiz = (sal_uInt32)(rMsg.GetContentLength().ToInt32());
            m_nDocLen = 0;
        }
        void destroy (void)
        {
            delete m_pStrm, m_pStrm = NULL;
        }

        int read (sal_Char *pBuffer, sal_uInt32 nBufSiz, void *pCtx)
        {
            if (m_pStrm)
                return m_pStrm->Read (pBuffer, nBufSiz, pCtx);
            else
                return INETSTREAM_STATUS_ERROR;
        }
    };

    /** RecvBlock.
     */
    struct RecvBlock
    {
        INetHTTPOutputStream *m_pStrm;
        sal_uInt32            m_nDocSiz;
        sal_uInt32            m_nDocLen;
        sal_Bool              m_bHeaderParsed;

        RecvBlock (void)
            : m_pStrm(NULL), m_nDocSiz(0), m_nDocLen(0), m_bHeaderParsed(0)
        {}
        ~RecvBlock (void)
        {
            delete m_pStrm;
        }

        void create (INetHTTPMessage &rMsg)
        {
            m_pStrm   = new INetHTTPOutputStream (rMsg);
            m_nDocSiz = (sal_uInt32)(-1);
            m_nDocLen = 0;
        }
        void destroy (void)
        {
            delete m_pStrm, m_pStrm = NULL;
        }

        int write (const sal_Char *pBuffer, sal_uInt32 nBufSiz, void *pCtx)
        {
            if (m_pStrm)
                return m_pStrm->Write (pBuffer, nBufSiz, pCtx);
            else
                return INETSTREAM_STATUS_ERROR;
        }
    };

    /** Representation.
     */
    request_type         *m_pRequest;
    request_type::Method  m_eMethod;
    request_type::Scheme  m_eScheme;

    INetURLObject         m_aReqURI;
    INetProxyConfig       m_aProxyConfig;
    INetCoreDNSHostEntry  m_aDestAddr;
    Destination           m_eDestType;

    SendBlock  m_aSendBlk;
    RecvBlock  m_aRecvBlk;

    StateBlock m_aStateBlk;

    INetHTTPRequestCallback *m_pfnCB;
    void                    *m_pDataCB;

    /** Implementation.
     */
    INetHTTPRequestContext (
        request_type         *pRequest,
        request_type::Method  eMethod);
    ~INetHTTPRequestContext (void);

    sal_Bool create (
        const OUString          &rReqURI,
        INetHTTPMessage         &rReqMsg,
        INetHTTPMessage         &rResMsg,
        INetHTTPRequestCallback *pfnCB,
        void                    *pDataCB);

    OString getURI (void) const;

    void destination (
        const INetProxyConfig &rProxyConfig);

    /** Destination type and address.
     */
    void destination (
        Destination     eDest,
        const OUString &rHost = OUString(),
        sal_uInt16      nPort = 0)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_eDestType = eDest;
        m_aDestAddr = INetCoreDNSHostEntry (rHost, nPort);
    }

    /** Cleanup.
     */
    void destroy (void)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_aStateBlk.setState (STATE_ABORT);

        m_aSendBlk.destroy();
        m_aRecvBlk.destroy();
    }

    /** State.
     */
    State getState (void) const
    {
        return m_aStateBlk.getState();
    }
    void setState (State eState)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_aStateBlk.setState (eState);
    }

    sal_Int32 getStateCode (void) const
    {
        return m_aStateBlk.getStateCode();
    }
    void setStateCode (sal_Int32 nState)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_aStateBlk.setStateCode (nState);
    }

    /** Reply.
     */
    sal_Int32 getReplyCode (void) const
    {
        return m_aStateBlk.getReplyCode();
    }
    void setReplyCode (sal_Int32 nReply)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_aStateBlk.setReplyCode (nReply);
    }

    const sal_Char* getReplyText (void) const
    {
        return m_aStateBlk.getReplyText();
    }
    void setReplyText (const sal_Char *pReply)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        m_aStateBlk.setReplyText (pReply);
    }

    /** Send/Recv.
     */
    int read (sal_Char *pBuffer, sal_uInt32 nBufSiz)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        return m_aSendBlk.read (pBuffer, nBufSiz, this);
    }
    int write (const sal_Char *pBuffer, sal_uInt32 nBufSiz)
    {
        NAMESPACE_VOS(OGuard) aGuard (*this);
        return m_aRecvBlk.write (pBuffer, nBufSiz, this);
    }
};

#ifdef _USE_NAMESPACE
}
#endif

/*========================================================================
 *
 * INetHTTPRequestContext implementation.
 *
 *======================================================================*/
/*
 * INetHTTPRequestContext.
 */
INetHTTPRequestContext::INetHTTPRequestContext (
    request_type         *pRequest,
    request_type::Method  eMethod)
    : m_pRequest   (pRequest),
      m_eMethod    (eMethod),
      m_eScheme    (request_type::SCHEME_HTTP),
      m_aDestAddr  (OUString(), 0),
      m_eDestType  (DEST_NONE),
      m_pfnCB      (NULL),
      m_pDataCB    (NULL)
{
}

/*
 * ~INetHTTPRequestContext.
 */
INetHTTPRequestContext::~INetHTTPRequestContext (void)
{
}

/*
 * create.
 */
sal_Bool INetHTTPRequestContext::create (
    const OUString          &rReqURI,
    INetHTTPMessage         &rReqMsg,
    INetHTTPMessage         &rResMsg,
    INetHTTPRequestCallback *pfnCB,
    void                    *pDataCB)
{
    // Acquire exclusive access.
    NAMESPACE_VOS(OGuard) aGuard (*this);

    // Parse request URI.
    m_aReqURI.SetURL (rReqURI);
    if (m_aReqURI.HasError())
        return sal_False;

    // Determine request scheme.
    switch (m_aReqURI.GetProtocol())
    {
        case INET_PROT_FTP:
            m_eScheme = request_type::SCHEME_FTP;
            break;

        case INET_PROT_HTTP:
            m_eScheme = request_type::SCHEME_HTTP;
            break;

        case INET_PROT_HTTPS:
            m_eScheme = request_type::SCHEME_HTTPS;
            break;

        default:
            m_eScheme = request_type::SCHEME_OTHER;
            break;
    }

    // Check for missing port.
    if (!m_aReqURI.HasPort())
    {
        // Provide default port.
        switch (m_eScheme)
        {
            case request_type::SCHEME_FTP:
                m_aReqURI.SetPort (eINET_FTP_PORT);
                break;

            case request_type::SCHEME_HTTP:
                m_aReqURI.SetPort (eINET_HTTP_PORT);
                break;

            case request_type::SCHEME_HTTPS:
                m_aReqURI.SetPort (eINET_HTTPS_PORT);
                break;

            default: // SCHEME_OTHER.
                break;
        }
    }

    // Initialize destination.
    m_aProxyConfig = INetProxyConfig();
    destination (DEST_NONE);

    // Check for proxy destination.
    NAMESPACE_VOS(ORef)<INetConfig> xConfig;
    if (INetConfig::getOrCreate (xConfig))
    {
        NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
            xConfig->getProxyPolicy());
        if (xProxyPolicy.isValid())
        {
            INetProxyConfig aProxyConfig;
            if (xProxyPolicy->shouldUseProxy (
                m_aReqURI.GetMainURL(), aProxyConfig))
            {
                // Determine proxy destination.
                destination (aProxyConfig);
                if (m_aProxyConfig.hasHttpProxy())
                {
                    // Request via Proxy.
                    destination (
                        DEST_PROXY,
                        m_aProxyConfig.getHttpProxyName(),
                        m_aProxyConfig.getHttpProxyPort());
                }
            }
        }
    }

    // Check destination.
    if (m_eDestType == DEST_NONE)
    {
        switch (m_eScheme)
        {
            case request_type::SCHEME_FTP:
                // Mission impossible.
                return sal_False;

            case request_type::SCHEME_HTTP:
            case request_type::SCHEME_HTTPS:
                // Request from Origin.
                destination (
                    DEST_ORIGIN,
                    m_aReqURI.GetHost(),
                    m_aReqURI.GetPort());
                break;

            default: // SCHEME_OTHER.
                // Mission impossible.
                return sal_False;
        }
    }

    // ...
    m_aSendBlk.create (rReqMsg);
    m_aRecvBlk.create (rResMsg);
    m_pfnCB   = pfnCB;
    m_pDataCB = pDataCB;

    // Done.
    return sal_True;
}

/*
 * getURI.
 */
OString INetHTTPRequestContext::getURI (void) const
{
    if (m_eDestType == DEST_ORIGIN)
    {
        rtl::OUStringBuffer aBuffer (
            OUString (m_aReqURI.GetURLPath (INetURLObject::NO_DECODE)));
        if (m_aReqURI.HasParam())
        {
            aBuffer.append (
                sal_Unicode (INET_PARAM_TOKEN));
            aBuffer.append (
                OUString (m_aReqURI.GetParam (INetURLObject::NO_DECODE)));
        }

        OUString aURI (aBuffer.makeStringAndClear());
        return OString (
            aURI.pData->buffer,
            aURI.pData->length,
            RTL_TEXTENCODING_ASCII_US);
    }
    else
    {
        OUString aURI (m_aReqURI.GetMainURL (INetURLObject::NO_DECODE));
        return OString (
            aURI.pData->buffer,
            aURI.pData->length,
            RTL_TEXTENCODING_ASCII_US);
    }
}

/*
 * destination.
 */
void INetHTTPRequestContext::destination (const INetProxyConfig &rProxyConfig)
{
    switch (m_eScheme)
    {
        case request_type::SCHEME_FTP:
            if (rProxyConfig.hasFtpProxy())
            {
                // Use FtpProxy.
                m_aProxyConfig.setHttpProxy (
                    rProxyConfig.getFtpProxyName(),
                    rProxyConfig.getFtpProxyPort());
            }
            else if (rProxyConfig.hasHttpProxy())
            {
                // Use HttpProxy.
                m_aProxyConfig.setHttpProxy (
                    rProxyConfig.getHttpProxyName(),
                    rProxyConfig.getHttpProxyPort());
            }
            break;

        case request_type::SCHEME_HTTP:
            if (rProxyConfig.hasHttpProxy())
            {
                // Use HttpProxy.
                m_aProxyConfig.setHttpProxy (
                    rProxyConfig.getHttpProxyName(),
                    rProxyConfig.getHttpProxyPort());
            }
            else if (rProxyConfig.hasSocksProxy())
            {
                // Use SocksProxy.
                m_aProxyConfig.setSocksProxy (
                    rProxyConfig.getSocksProxyName(),
                    rProxyConfig.getSocksProxyPort());
            }
            break;

        case request_type::SCHEME_HTTPS:
            if (rProxyConfig.hasSecureSocketProxy())
            {
                // Use SecureSocketProxy.
                m_aProxyConfig.setSecureSocketProxy (
                    rProxyConfig.getSecureSocketProxyName(),
                    rProxyConfig.getSecureSocketProxyPort());
            }
            else if (rProxyConfig.hasSocksProxy())
            {
                // Use SocksProxy.
                m_aProxyConfig.setSocksProxy (
                    rProxyConfig.getSocksProxyName(),
                    rProxyConfig.getSocksProxyPort());
            }
            break;

        default: // SCHEME_OTHER.
            if (rProxyConfig.hasHttpProxy())
            {
                // Use HttpProxy.
                m_aProxyConfig.setHttpProxy (
                    rProxyConfig.getHttpProxyName(),
                    rProxyConfig.getHttpProxyPort());
            }
            break;
    }
}

/*========================================================================
 *
 * INetHTTPConnection_Impl interface.
 *
 *======================================================================*/
typedef NAMESPACE_INET(INetSocket) socket_type;

#ifdef _USE_NAMESPACE
namespace inet {
#endif

class INetHTTPConnection_Impl : public NAMESPACE_INET(INetHTTPConnection)
{
    VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (INetHTTPConnection_Impl, inet));

public:
    INetHTTPConnection_Impl (void);

    /** startRequest.
     */
    virtual sal_Bool startRequest (INetHTTPRequestContext *pCtx);

    /** abortRequest.
     */
    virtual sal_Bool abortRequest (INetHTTPRequestContext *pCtx);

protected:
    virtual ~INetHTTPConnection_Impl (void);

private:
    /** Representation.
     */
    INetCoreDNSResolver   m_aResolver;
    INetCoreDNSHostEntry *m_pHostEntry;

#ifdef SOLAR_SSL
    NAMESPACE_VOS(ORef)<INetActiveTCPSecureSocket> m_xSocket;
#else
    NAMESPACE_VOS(ORef)<INetActiveTCPSocket>       m_xSocket;
#endif /* SOLAR_SSL */

    sal_uInt32    m_nBufSiz;
    sal_Char     *m_pBuffer;
    sal_Char     *m_pRead;
    sal_Char     *m_pWrite;

    context_type *m_pCtx;

    inline context_type* switchContext (context_type *pCtx);

    /** handleResolverEvent.
     */
    virtual sal_Bool handleResolverEvent (
        sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry);

    /** handleSocketEvent.
     */
    virtual sal_Bool handleSocketEvent (
        const NAMESPACE_VOS(ORef)<socket_type> &rxSocket, sal_Int32 nEvent);

    /** Not implemented.
     */
    INetHTTPConnection_Impl (const INetHTTPConnection_Impl&);
    INetHTTPConnection_Impl& operator= (const INetHTTPConnection_Impl&);
};

/*
 * switchContext.
 */
inline context_type*
INetHTTPConnection_Impl::switchContext (context_type *pCtx)
{
    context_type *pCurCtx = m_pCtx;
    m_pCtx = pCtx;
    return pCurCtx;
}

#ifdef _USE_NAMESPACE
}
#endif

/*========================================================================
 *
 * INetHTTPConnection_Impl implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPConnection_Impl, inet),
    VOS_NAMESPACE (INetHTTPConnection_Impl, inet),
    VOS_NAMESPACE (INetHTTPConnection, inet),
    0);

/*
 * INetHTTPConnection_Impl.
 */
INetHTTPConnection_Impl::INetHTTPConnection_Impl (void)
    : m_pHostEntry (NULL),
      m_xSocket    (NULL),
      m_nBufSiz    (16384),
      m_pCtx       (NULL)
{
    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_pRead = m_pWrite = m_pBuffer;
}

/*
 * ~INetHTTPConnection_Impl.
 */
INetHTTPConnection_Impl::~INetHTTPConnection_Impl (void)
{
    delete m_pHostEntry;
    INETHTTP_SOCKET_DISPOSE (m_xSocket);
    rtl_freeMemory (m_pBuffer);
}

/*
 * startRequest.
 */
sal_Bool INetHTTPConnection_Impl::startRequest (INetHTTPRequestContext *pCtx)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetHTTPConnection_Impl> xThis (this);

    // Check argument and context.
    if (!pCtx)
        return sal_False;
    if (!!m_pCtx)
        return sal_False;

    // Switch context.
    switchContext (pCtx);

    m_pCtx->setState (context_type::STATE_RESOLVE);
    m_pCtx->setStateCode (request_type::REPLY_RESOLVER_WAIT);

    // Start resolution.
    m_pHostEntry = new INetCoreDNSHostEntry (m_pCtx->m_aDestAddr);
    if (m_aResolver.GetHostByName (m_pHostEntry, onResolverEvent, this))
    {
        // Ok. Wait for next callback.
        return sal_True;
    }

    // Failure.
    pCtx = switchContext (NULL);
    if (pCtx)
        pCtx->destroy();
    return sal_False;
}

/*
 * handleResolverEvent.
 */
sal_Bool INetHTTPConnection_Impl::handleResolverEvent (
    sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    // Jump into state machine.
    while (m_pCtx)
    {
        // Check state.
        context_type::State eState = m_pCtx->getState();
        if (eState == context_type::STATE_RESOLVE)
        {
            // Check reason.
            if (nStatus == INETCOREDNS_RESOLVER_START)
            {
                // Resolution started.
                m_pCtx->setStateCode (request_type::REPLY_RESOLVER_WAIT);

                // Notify caller.
                if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                    m_pCtx->m_pRequest,
                    request_type::REPLY_RESOLVER_WAIT,
                    NULL,
                    m_pCtx->m_pDataCB);

                // Wait for next callback.
                return sal_True;
            }
            else if ((nStatus == INETCOREDNS_RESOLVER_SUCCESS) ||
                     (nStatus == INETCOREDNS_RESOLVER_EXPIRED)    )
            {
                // Resolution finished.
                m_pCtx->setState (context_type::STATE_CONNECT);
                m_pCtx->setStateCode (request_type::REPLY_RESOLVER_DONE);

                // Notify caller.
                if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                    m_pCtx->m_pRequest,
                    request_type::REPLY_RESOLVER_DONE,
                    NULL,
                    m_pCtx->m_pDataCB);
            }
            else
            {
                // Resolution failed.
                m_pCtx->setState (context_type::STATE_ERROR);
                m_pCtx->setStateCode (request_type::REPLY_RESOLVER_ERROR);
            }
        }
        else if (eState == context_type::STATE_CONNECT)
        {
            // Initialize active (secure) socket.
#ifdef SOLAR_SSL
            m_xSocket = new INetActiveTCPSecureSocket();
#else
            m_xSocket = new INetActiveTCPSocket();
#endif /* SOLAR_SSL */
            m_xSocket->registerEventHandler (onSocketEvent, this);

            // Check proxy configuration.
            const INetProxyConfig &rProxyConfig = m_pCtx->m_aProxyConfig;
            if (rProxyConfig.hasSocksProxy())
            {
                // Connect via SocksGateway.
                m_xSocket->setSocksGateway (
                    NAMESPACE_VOS(OInetSocketAddr)(
                        rProxyConfig.getSocksProxyName(),
                        rProxyConfig.getSocksProxyPort()));
            }

            // Initiate connect.
            if (m_xSocket->connect (
                NAMESPACE_VOS(OInetSocketAddr)(
                    m_pHostEntry->GetDottedDecimalName(),
                    m_pHostEntry->GetPort())))
            {
                // Connect in progress.
                m_pCtx->setStateCode (request_type::REPLY_CONNECT_WAIT);

                // Notify caller.
                if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                    m_pCtx->m_pRequest,
                    request_type::REPLY_CONNECT_WAIT,
                    NULL,
                    m_pCtx->m_pDataCB);

                // Wait for next callback.
                return sal_True;
            }

            // Connect failure.
            m_pCtx->setState (context_type::STATE_ERROR);
            m_pCtx->setStateCode (request_type::REPLY_CONNECT_ERROR);
        }
        else
        {
            // Abort or failure.
            context_type *pCtx = switchContext (NULL);
            if (pCtx)
            {
                // Cleanup.
                pCtx->destroy();

                // Notify caller.
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    pCtx->m_pRequest,
                    pCtx->getReplyCode(),
                    NULL,
                    pCtx->m_pDataCB);
            }
        }
    }

    // Leave.
    return sal_False;
}

/*
 * handleSocketEvent.
 */
sal_Bool INetHTTPConnection_Impl::handleSocketEvent (
    const NAMESPACE_VOS(ORef)<socket_type> &rxSocket, sal_Int32 nEvent)
{
    // Check context.
    if (!m_pCtx)
    {
        // No context (STATE_IDLE).
        if (nEvent & socket_type::EVENT_READ)
        {
            while (m_xSocket.isValid())
            {
                sal_Int32 nRead = m_xSocket->recv (m_pBuffer, m_nBufSiz);
                if (nRead > 0)
                {
                    // Absorbing whatever comes in.
                    continue;
                }
                else if (nRead == INETHTTP_SOCKET_WOULDBLOCK)
                {
                    // Wait for next callback.
                    return sal_True;
                }
                else
                {
                    // Connection closed or read error.
                    INETHTTP_SOCKET_DISPOSE (m_xSocket);
                }
            }
        }

        if (nEvent & socket_type::EVENT_CLOSE)
        {
            // Connection closed.
            m_xSocket.unbind();
        }

        // Leave.
        return sal_True;
    }

    // Check event.
    if (nEvent & socket_type::EVENT_CLOSE)
    {
        // Premature closure.
        m_xSocket.unbind();

        // Force into final state.
        context_type::State eState = m_pCtx->getState();
        if (eState == context_type::STATE_RECV)
            eState = context_type::STATE_DONE;
        else
            eState = context_type::STATE_ERROR;

        m_pCtx->setState (eState);
        m_pCtx->setStateCode (request_type::REPLY_CONNECTION_TERMINATED);
    }

    // Jump into state machine.
    while (m_pCtx)
    {
        // Check state.
        context_type::State eState = m_pCtx->getState();
        if (eState == context_type::STATE_CONNECT)
        {
            if (nEvent & socket_type::EVENT_CONNECT)
            {
                if (nEvent & socket_type::EVENT_OOB)
                {
                    // Connect failure.
                    m_pCtx->setState (context_type::STATE_ERROR);
                    m_pCtx->setStateCode (request_type::REPLY_CONNECT_ERROR);

                    // Cleanup.
                    m_xSocket.unbind();
                }
                nEvent = socket_type::EVENT_WRITE;

                if (m_xSocket.isValid())
                {
                    // Connect finished.
                    m_pCtx->setState (context_type::STATE_SEND);
                    m_pCtx->setStateCode (request_type::REPLY_CONNECT_DONE);

                    // Notify caller.
                    if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                        m_pCtx->m_pRequest,
                        request_type::REPLY_CONNECT_DONE,
                        NULL,
                        m_pCtx->m_pDataCB);
                }
            }
            else
            {
                // Ignored.
                return sal_True;
            }
        }
        else if (eState == context_type::STATE_SEND)
        {
            if (nEvent & socket_type::EVENT_WRITE)
            {
                // Send request.
                if ((m_pRead - m_pWrite) > 0)
                {
                    // Buffer not yet written.
                    sal_Int32 nWrite = m_xSocket->send (
                        m_pWrite, (m_pRead - m_pWrite));
                    if (nWrite > 0)
                    {
                        // Adjust write pointer.
                        m_pWrite += nWrite;
                    }
                    else if (nWrite == INETHTTP_SOCKET_WOULDBLOCK)
                    {
                        m_pCtx->setStateCode (
                            request_type::REPLY_REQUEST_WAIT);

                        // Notify caller.
                        if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                            m_pCtx->m_pRequest,
                            request_type::REPLY_REQUEST_WAIT,
                            NULL,
                            m_pCtx->m_pDataCB);

                        // Wait for next callback.
                        return sal_True;
                    }
                    else
                    {
                        // Socket write error.
                        m_pCtx->setState (context_type::STATE_ERROR);
                        m_pCtx->setStateCode (
                            request_type::REPLY_NETWORK_ERROR);
                        m_xSocket->close();
                    }
                }
                else
                {
                    // Buffer written. Reset to <Begin-of-Buffer>.
                    m_pRead = m_pWrite = m_pBuffer;

                    // Read input stream.
                    int nRead = m_pCtx->read (m_pBuffer, m_nBufSiz);
                    if (nRead > 0)
                    {
                        // Adjust read pointer.
                        m_pRead += nRead;
                    }
                    else if (nRead == 0)
                    {
                        // Request generated.
                        m_pCtx->setState (context_type::STATE_RECV);
                        m_pCtx->setStateCode (
                            request_type::REPLY_REQUEST_DONE);

                        // Notify caller.
                        if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                            m_pCtx->m_pRequest,
                            request_type::REPLY_REQUEST_DONE,
                            NULL,
                            m_pCtx->m_pDataCB);
                    }
                    else
                    {
                        // Stream read error.
                        m_pCtx->setState (context_type::STATE_ERROR);
                        m_pCtx->setStateCode (
                            request_type::REPLY_REQUEST_ERROR);
                    }
                }
            }
            else
            {
                m_pCtx->setState (context_type::STATE_RECV);
            }
        }
        else if (eState == context_type::STATE_RECV)
        {
            // Read response.
            sal_Int32 nRead = m_xSocket->recv (m_pBuffer, m_nBufSiz);
            if (nRead > 0)
            {
                // Write output stream.
                int status = m_pCtx->write (m_pBuffer, nRead);
                if (status != INETSTREAM_STATUS_OK)
                {
                    if (status == INETSTREAM_STATUS_HEADER)
                    {
                        // Notify caller.
                        if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                            m_pCtx->m_pRequest,
                            request_type::REPLY_RESPONSE_WAIT,
                            NULL,
                            m_pCtx->m_pDataCB);
                    }
                    else if (status == INETSTREAM_STATUS_LOADED)
                    {
                        m_pCtx->setState (context_type::STATE_DONE);
                        m_pCtx->setStateCode (
                            request_type::REPLY_RESPONSE_DONE);
                    }
                    else
                    {
                        m_pCtx->setState (context_type::STATE_ERROR);
                        m_pCtx->setStateCode (
                            request_type::REPLY_RESPONSE_ERROR);
                    }
                }
            }
            else if (nRead == INETHTTP_SOCKET_WOULDBLOCK)
            {
                m_pCtx->setStateCode (request_type::REPLY_RESPONSE_WAIT);

                // Notify caller.
                if (m_pCtx->m_pfnCB) (m_pCtx->m_pfnCB) (
                    m_pCtx->m_pRequest,
                    request_type::REPLY_RESPONSE_WAIT,
                    NULL,
                    m_pCtx->m_pDataCB);

                // Wait for next callback.
                return sal_True;
            }
            else if (nRead == 0)
            {
                // Connection closed.
                if (m_pCtx->getReplyCode() < 0)
                    m_pCtx->setState (context_type::STATE_ERROR);
                else
                    m_pCtx->setState (context_type::STATE_DONE);
                m_pCtx->setStateCode (
                    request_type::REPLY_CONNECTION_TERMINATED);
                m_xSocket->close();
            }
            else
            {
                // Socket read error.
                m_pCtx->setState (context_type::STATE_ERROR);
                m_pCtx->setStateCode (
                    request_type::REPLY_NETWORK_ERROR);
                m_xSocket->close();
            }
        }
        else
        {
            // Aborted, finished or failure.
            context_type *pCtx = switchContext (NULL);
            if (pCtx)
            {
                // Cleanup.
                pCtx->destroy();

                // Notify caller.
                if (pCtx->m_pfnCB) (pCtx->m_pfnCB) (
                    pCtx->m_pRequest,
                    pCtx->getReplyCode(),
                    pCtx->getReplyText(),
                    pCtx->m_pDataCB);
            }
        }
    }

    // Leave.
    return sal_True;
}

/*
 * abortRequest.
 */
sal_Bool INetHTTPConnection_Impl::abortRequest (INetHTTPRequestContext *pCtx)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetHTTPConnection_Impl> xThis (this);

    // Check argument and context.
    if (!pCtx)
        return sal_False;
    if (!(pCtx == m_pCtx))
        return sal_False;

    // Switch context.
    switchContext (NULL);

    // Obtain state and cleanup.
    context_type::State eState = pCtx->getState();
    pCtx->destroy();

    // Check state.
    if (!((eState == context_type::STATE_ABORT) ||
          (eState == context_type::STATE_ERROR) ||
          (eState == context_type::STATE_BEGIN) ||
          (eState == context_type::STATE_DONE )    ))
    {
        // Abort connection.
        INETHTTP_SOCKET_DISPOSE (m_xSocket);
    }

    // Done.
    return sal_True;
}

/*========================================================================
 *
 * INetHTTPConnection implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPConnection, inet),
    VOS_NAMESPACE (INetHTTPConnection, inet),
    VOS_NAMESPACE (INetClientConnection_Impl, inet),
    0);

/*
 * INetHTTPConnection.
 */
INetHTTPConnection::INetHTTPConnection (void)
{
}

/*
 * ~INetHTTPConnection.
 */
INetHTTPConnection::~INetHTTPConnection (void)
{
}

/*
 * createInstance.
 */
sal_Bool INetHTTPConnection::createInstance (
    NAMESPACE_VOS(ORef)<INetHTTPConnection> &rxConnection)
{
    rxConnection = new INetHTTPConnection_Impl();
    return (rxConnection.isValid());
}

/*========================================================================
 *
 * INetHTTPRequest implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    VOS_NAMESPACE (OObject, vos),
    0);

/*
 * INetHTTPRequest.
 */
INetHTTPRequest::INetHTTPRequest (Method eMethod)
{
    m_pContext = new INetHTTPRequestContext (this, eMethod);
    VOS_ASSERT(m_pContext);
}

/*
 * ~INetHTTPRequest.
 */
INetHTTPRequest::~INetHTTPRequest (void)
{
    VOS_ASSERT(m_pContext);
    if (m_xConnection.isValid())
    {
        m_xConnection->abortRequest (m_pContext);
        m_xConnection.unbind();
    }
    delete m_pContext;
}

/*
 * start.
 */
sal_Bool INetHTTPRequest::start (
    const OUString          &rReqURI,
    INetHTTPMessage         &rReqMsg,
    INetHTTPMessage         &rResMsg,
    INetHTTPRequestCallback *pfnCB,
    void                    *pDataCB)
{
    // Ensure clean destruction.
    NAMESPACE_VOS(ORef)<INetHTTPRequest> xThis (this);

    // Check context.
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return sal_False;

    // Check arguments.
    switch (m_pContext->m_eMethod)
    {
        case METHOD_POST:
        case METHOD_PUT:
            if (!rReqMsg.GetDocumentLB())
                return sal_False;

        case METHOD_DELETE:
        case METHOD_GET:
            if (!rResMsg.GetDocumentLB())
                return sal_False;

        default:
            if (!rReqURI.getLength())
                return sal_False;
            if (!pfnCB)
                return sal_False;
            break;
    }

    // Initialize context.
    if (!m_pContext->create (rReqURI, rReqMsg, rResMsg, pfnCB, pDataCB))
        return sal_False;

    // Obtain connection. (Manager::getOrCreate (m_pCtx->m_aDestAddr));
    if (INetHTTPConnection::createInstance (m_xConnection))
        return m_xConnection->startRequest (m_pContext);
    else
        return sal_False;
}

/*
 * abort.
 */
void INetHTTPRequest::abort (void)
{
    // Check context.
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return;

    // Check connection.
    if (m_xConnection.isValid())
    {
        m_xConnection->abortRequest (m_pContext);
        m_xConnection.unbind();
    }
}

/*
 * getEndpoint.
 */
sal_Bool INetHTTPRequest::getEndpoint (
    OUString &rHost, sal_uInt16 &rPort) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return sal_False;

    rHost = m_pContext->m_aDestAddr.GetDomainName();
    rPort = m_pContext->m_aDestAddr.GetPort();
    return ((rHost.getLength() > 0) && (rPort > 0));
}

/*
 * getStatusCode.
 */
sal_Int32 INetHTTPRequest::getStatusCode (void) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return (-1);

    return m_pContext->getReplyCode();
}

/*
 * isRequestHeaderGenerated.
 */
sal_Bool INetHTTPRequest::isRequestHeaderGenerated (void) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return sal_False;

    // NYI.
    return sal_False;
}

/*
 * isResponseHeaderParsed.
 */
sal_Bool INetHTTPRequest::isResponseHeaderParsed (void) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return sal_False;

    return m_pContext->m_aRecvBlk.m_bHeaderParsed;
}

/*
 * getRequestDocumentLength.
 */
sal_uInt32 INetHTTPRequest::getRequestDocumentLength (void) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return 0;

    return m_pContext->m_aSendBlk.m_nDocLen;
}

/*
 * getResponseDocumentLength.
 */
sal_uInt32 INetHTTPRequest::getResponseDocumentLength (void) const
{
    VOS_ASSERT(m_pContext);
    if (!m_pContext)
        return 0;

    return m_pContext->m_aRecvBlk.m_nDocLen;
}

/*========================================================================
 *
 * INetHTTPGetRequest implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPGetRequest, inet),
    VOS_NAMESPACE (INetHTTPGetRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    0);

/*========================================================================
 *
 * INetHTTPHeadRequest implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPHeadRequest, inet),
    VOS_NAMESPACE (INetHTTPHeadRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    0);

/*========================================================================
 *
 * INetHTTPPostRequest implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPPostRequest, inet),
    VOS_NAMESPACE (INetHTTPPostRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    0);

/*========================================================================
 *
 * INetHTTPPutRequest implementation (HTTP/1.1).
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPPutRequest, inet),
    VOS_NAMESPACE (INetHTTPPutRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    0);

/*========================================================================
 *
 * INetHTTPDeleteRequest implementation (HTTP/1.1).
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
    VOS_CLASSNAME (INetHTTPDeleteRequest, inet),
    VOS_NAMESPACE (INetHTTPDeleteRequest, inet),
    VOS_NAMESPACE (INetHTTPRequest, inet),
    0);

/*========================================================================
 *
 * INetHTTPInputStream implementation.
 *
 *======================================================================*/
/*
 * INetHTTPInputStream.
 */
INetHTTPInputStream::INetHTTPInputStream (
    INetHTTPMessage &rMessage, sal_uInt32 nBufSiz)
{
    m_aStrm.SetSourceMessage (&rMessage);
    m_nBufSiz = nBufSiz;

    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_pRead = m_pWrite = m_pBuffer;

    m_eState = STATE_BEGIN;
    m_bRequestGenerated = sal_False;
}

/*
 * ~INetHTTPInputStream.
 */
INetHTTPInputStream::~INetHTTPInputStream (void)
{
    rtl_freeMemory (m_pBuffer);
}

/*
 * GetData.
 */
int INetHTTPInputStream::GetData (sal_Char *pData, ULONG nSize, void *pCtx)
{
    // Check context.
    context_type *ctx = (context_type*)pCtx;
    if (ctx == NULL)
        return INETSTREAM_STATUS_ERROR;
    if (ctx->getState() == context_type::STATE_ABORT)
        return INETSTREAM_STATUS_ERROR;

    // Fill callers buffer.
    sal_Char *p = pData, *q = pData + nSize;
    while (p < q)
    {
        // Callers buffer not yet filled.
        sal_Int32 n = m_pRead - m_pWrite;
        if (n > 0)
        {
            // Bytes still in buffer.
            sal_Int32 k = VOS_MIN((q - p), n);
            rtl_copyMemory (p, m_pWrite, k);
            m_pWrite += k;
            p += k;
        }
        else
        {
            // Buffer empty. Reset to <Begin-of-Buffer>.
            m_pRead = m_pWrite = m_pBuffer;
            if (m_bRequestGenerated)
            {
                // Read message.
                int nRead = m_aStrm.Read (m_pBuffer, m_nBufSiz);
                if (nRead > 0)
                {
                    // Set read pointer.
                    m_pRead = m_pBuffer + nRead;
                }
                else
                {
                    // Message generated.
                    return (p - pData);
                }
            }
            else
            {
                // Read request line.
                int nRead = GetLine (m_pBuffer, m_nBufSiz, pCtx);
                if (nRead > 0)
                {
                    // Set read pointer.
                    m_pRead = m_pBuffer + nRead;
                }
                else
                {
                    // Request line generated.
                    m_bRequestGenerated = TRUE;
                }
            }
        }
    }
    return (p - pData);
}

/*
 * GetLine.
 */
int INetHTTPInputStream::GetLine (sal_Char *pData, ULONG nSize, void *pCtx)
{
    // Check context.
    context_type *ctx = (context_type*)pCtx;
    if (ctx == NULL)
        return INETSTREAM_STATUS_ERROR;
    if (ctx->getState() == context_type::STATE_ABORT)
        return INETSTREAM_STATUS_ERROR;

    // Generate request line.
    static const sal_Char pTrailer[] = " HTTP/1.0\015\012";
    sal_Int32 nTrailer = rtl_str_getLength (pTrailer);

    sal_Char *p = pData;
    while (!m_bRequestGenerated)
    {
        // Generate request line.
        switch (m_eState)
        {
            case STATE_BEGIN:
                m_eState = STATE_METHOD;
                break;

            case STATE_METHOD:
                m_eState = STATE_URI;
                switch (ctx->m_eMethod)
                {
                    case request_type::METHOD_GET:
                        rtl_copyMemory (p, "GET ", (3 + 2));
                        break;

                    case request_type::METHOD_HEAD:
                        rtl_copyMemory (p, "HEAD ", (4 + 2));
                        break;

                    case request_type::METHOD_POST:
                        rtl_copyMemory (p, "POST ", (4 + 2));
                        break;

                    case request_type::METHOD_PUT:
                        rtl_copyMemory (p, "PUT ", (3 + 2));
                        break;

                    case request_type::METHOD_DELETE:
                        rtl_copyMemory (p, "DELETE ", (6 + 2));
                        break;

#ifdef __HTTP_METHOD_CONNECT
                    case request_type::METHOD_CONNECT:
                        rtl_copyMemory (p, "CONNECT ", (7 + 2));
                        break;
#endif

                    default:
                        return INETSTREAM_STATUS_ERROR;
                        break;
                }
                p += rtl_str_getLength (p);
                break;

            case STATE_URI:
                if (p < (pData + nSize))
                {
#if 0  /* OLD */
                    // Callers buffer not yet filled.
                    if ((ctx->pRead - ctx->pWrite) > 0)
                    {
                        // Bytes still in buffer.
                        *p++ = *(ctx->pWrite++);
                    }
                    else
                    {
                        // Buffer empty.
                        m_eState = STATE_VERSION;
                    }
#endif /* OLD */
#if 1  /* NEW */
                    OString aURI (ctx->getURI());
                    rtl_copyMemory (p, aURI.pData->buffer, aURI.pData->length);
                    p += aURI.getLength();
                    m_eState = STATE_VERSION;
#endif /* NEW */
                }
                else
                {
                    // Callers buffer filled up.
                    return (p - pData);
                }
                break;

            case STATE_VERSION:
                if ((p + nTrailer) < (pData + nSize))
                {
                    // Copy trailer into callers buffer.
                    rtl_copyMemory (p, pTrailer, nTrailer);
                    p += nTrailer;

                    // Request line generated.
                    m_bRequestGenerated = TRUE;
                }
                else
                {
                    // Callers buffer filled up.
                    return (p - pData);
                }
                break;

            default: // Never reached.
                return INETSTREAM_STATUS_ERROR;
        }
    }
    return (p - pData);
}

/*========================================================================
 *
 * INetHTTPOutputStream implementation.
 *
 *======================================================================*/
/*
 * INetHTTPOutputStream.
 */
INetHTTPOutputStream::INetHTTPOutputStream (
    INetHTTPMessage &rMessage, sal_uInt32 nBufSiz)
{
    rMessage.SetContentLength (String());
    m_aStrm.SetTargetMessage (&rMessage);
    m_nBufSiz = nBufSiz;

    m_pBuffer = (sal_Char*)(rtl_allocateMemory (m_nBufSiz));
    m_nBufLen = 0;

    m_eState = INETMSG_EOL_SCR;
    m_bStatusParsed = sal_False;
}

/*
 * ~INetHTTPOutputStream.
 */
INetHTTPOutputStream::~INetHTTPOutputStream (void)
{
    rtl_freeMemory (m_pBuffer);
}

/*
 * PutData.
 */
int INetHTTPOutputStream::PutData (
    const sal_Char *pData, ULONG nSize, void *pCtx)
{
    // Check context.
    context_type *ctx = (context_type*)pCtx;
    if (ctx == NULL)
        return INETSTREAM_STATUS_ERROR;
    if (ctx->getState() == context_type::STATE_ABORT)
        return INETSTREAM_STATUS_ERROR;

    // Parse HTTP status line.
    const sal_Char *pStop = (pData + nSize);
    while (!m_bStatusParsed && (pData < pStop))
    {
        if (m_eState == INETMSG_EOL_BEGIN)
        {
            // Skip any 2nd line break character.
            if ((*pData == '\r') || (*pData == '\n')) pData++;

            // Zero terminate buffer.
            *(m_pBuffer + m_nBufLen) = '\0';

            // Parse buffered status line.
            int nStatus = PutLine (m_pBuffer, m_nBufLen, pCtx);
            if (nStatus != INETSTREAM_STATUS_OK)
                return nStatus;
        }
        else if ((*pData == '\r') || (*pData == '\n'))
        {
            // Skip any line break character.
            pData++;

            // Indicate <Begin-of-Line>.
            m_eState = INETMSG_EOL_BEGIN;
        }
        else
        {
            // Insert any other character.
            *(m_pBuffer + m_nBufLen++) = *pData++;
            if (m_nBufLen == 5)
            {
                // Check for HTTP/0.9 message.
                if (rtl_str_compareIgnoreAsciiCase_WithLength (
                    m_pBuffer, 5, "http/", 5))
                {
                    // Assume <Begin-of-Line>.
                    m_eState = INETMSG_EOL_BEGIN;
                }
            }

            // Beware of buffer overflow.
            if ((m_nBufLen + 1) == m_nBufSiz)
                m_eState = INETMSG_EOL_BEGIN;
        }
    }

    if (m_bStatusParsed && (pData < pStop))
    {
        // Put HTTP message down-stream.
        int status = m_aStrm.Write (pData, (pStop - pData));
        if (status != INETSTREAM_STATUS_ERROR)
        {
            if (m_aStrm.IsHeaderParsed())
            {
                INetHTTPMessage *msg = m_aStrm.GetTargetMessage();
                if (!ctx->m_aRecvBlk.m_bHeaderParsed)
                {
                    sal_uInt32 nDocSiz = sal_uInt32(-1);
                    if (ctx->m_eMethod == request_type::METHOD_HEAD)
                        nDocSiz = 0;
                    else if (msg->GetContentEncoding().Len())
                        nDocSiz = sal_uInt32(-1);
                    else if (msg->GetContentLength().Len())
                        nDocSiz = sal_uInt32(msg->GetContentLength().ToInt32());

                    ctx->m_aRecvBlk.m_nDocSiz = nDocSiz;
                    ctx->m_aRecvBlk.m_bHeaderParsed = sal_True;
                    status = INETSTREAM_STATUS_HEADER;
                }

                ctx->m_aRecvBlk.m_nDocLen = msg->GetDocumentSize();
                if (ctx->m_aRecvBlk.m_nDocLen >= ctx->m_aRecvBlk.m_nDocSiz)
                    status = INETSTREAM_STATUS_LOADED;
            }
        }
        return status;
    }
    return INETSTREAM_STATUS_OK;
}

/*
 * PutLine.
 */
int INetHTTPOutputStream::PutLine (
    const sal_Char *pData, ULONG nSize, void *pCtx)
{
    // Check context.
    context_type *ctx = (context_type*)pCtx;
    if (ctx == NULL)
        return INETSTREAM_STATUS_ERROR;
    if (ctx->getState() == context_type::STATE_ABORT)
        return INETSTREAM_STATUS_ERROR;

    // Parse HTTP status line.
    m_bStatusParsed = sal_True;
    if (rtl_str_compareIgnoreAsciiCase_WithLength (pData, 5, "http/", 5) == 0)
    {
        // HTTP/1.x -> Extract status code.
        const sal_Char *p = (const sal_Char*)(pData + 5);
        sal_Int32 n = rtl_str_getLength (p);
        sal_Int32 k = rtl_str_indexOfChar_WithLength (p, n, ' ') + 1;
        if (k > 0)
        {
            p = &p[k];
            ctx->setReplyCode (atoi(p));

            // Extract rest of line as status text (if any).
            k = rtl_str_indexOfChar_WithLength (p, n - k, ' ') + 1;
            if (k > 0)
            {
                p = &p[k];
                ctx->setReplyText(p);
            }
        }
    }
    else
    {
        // HTTP/0.9 -> Set status code.
        ctx->setReplyCode (request_type::REPLY_09);

        // Indicate unknown content type.
        INetHTTPMessage *msg = m_aStrm.GetTargetMessage();
        msg->SetContentType (
            OUString::createFromAscii ("application/octet-stream"));

        // Set header to parsed.
        m_aStrm.ParseHeader (FALSE);

        // Put HTTP/0.9 message down-stream.
        int status = m_aStrm.Write (pData, nSize);
        if (status != INETSTREAM_STATUS_OK)
            return status;
    }
    return INETSTREAM_STATUS_OK;
}

/*========================================================================
 *
 * @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
 *
 *======================================================================*/
#if 0  /* OBSOLETE */

/*
 * ResolverHandler.
 */
int INetCoreHTTPRequest_Impl::ResolverHandler (
    int nStatus, INetCoreDNSHostEntry *pHostEntry)
{
    switch (nStatus)
    {
        case INETCOREDNS_RESOLVER_SUCCESS:
        case INETCOREDNS_RESOLVER_EXPIRED:
            // Initiate connect.
#ifdef SOLAR_SSL
            if ((m_pCtx->eScheme == request_type::SCHEME_HTTPS  ) &&
                (m_pCtx->eMethod != request_type::METHOD_CONNECT)    )
            {
                if (!m_pSocket->Connect (
                    pHostEntry->GetDottedDecimalName(),
                    pHostEntry->GetPort(),
                    (INetCoreSSLConnectCallback *)RequestCallback, m_pOwner))
                    m_pCtx->nReplyCode = INETCOREHTTP_REPLY_CONNECT_ERROR;
            }
            else
#endif /* SOLAR_SSL */
            {
                INetCoreActiveTCPSocket *pBase = m_pSocket;
                if (!pBase->Connect (
                    pHostEntry->GetDottedDecimalName(),
                    pHostEntry->GetPort(),
                    (INetCoreIPSocketCallback *)RequestCallback, m_pOwner))
                    m_pCtx->nReplyCode = INETCOREHTTP_REPLY_CONNECT_ERROR;
            }

            if (m_pCtx->nReplyCode == INETCOREHTTP_REPLY_CONNECT_ERROR)
            {
                // Cleanup and fail.
                nStatus = INETCOREDNS_RESOLVER_ERROR;

                DELETEZ (m_pSocket);

                m_pCtx->ePrev = m_pCtx->eState;
                m_pCtx->eState = INETCOREHTTP_REQUEST_ERROR;
            }

            // Notify caller.
            if (m_pCtx->pfnCB) (m_pCtx->pfnCB) (
                m_pOwner, m_pCtx->nReplyCode, NULL, m_pCtx->pDataCB);
            break;

        default:
            break;
    }
    return (nStatus != INETCOREDNS_RESOLVER_ERROR);
}

/*
 * RequestHandler.
 * (Implements HTTP as a Finite State Machine).
 */
sal_Bool INetCoreHTTPRequest_Impl::RequestHandler (
#ifdef SOLAR_SSL
    INetCoreActiveTCPSecureSocket *pSocket,
#else
    INetCoreActiveTCPSocket       *pSocket,
#endif /* SOLAR_SSL */
    sal_Int32                      nEvent,
    sal_Bool                       bError)
{
    /*
     * Synchronize state INETCOREHTTP_REQUEST_CONNECT with
     * event INETCORESOCK_EVENT_CONNECT.
     */
    if ((m_pCtx->eState == INETCOREHTTP_REQUEST_CONNECT) &&
        (!(nEvent & INETCORESOCK_EVENT_CONNECT)))
    {
        /*
         * This can only happen by coincidence.
         * Consider the following scenario:
         *   A small document is read and the server closes the connection.
         *   This will generate an INETCORESOCK_EVENT_CLOSE event.
         *   If approximatly at the same time our caller cancels the
         *   request, a second INETCORESOCK_EVENT_CLOSE event is posted
         *   while the first one is still in the message queue.
         *   If now a new socket with same descriptor value as the previous
         *   socket is created while the second INETCORESOCK_EVENT_CLOSE
         *   message is still waiting, that INETCORESOCK_EVENT_CLOSE will be
         *   the first event that arrives here and erroneously kills
         *   this request.
         * But as we know that the first event can only be
         * INETCORESOCK_EVENT_CONNECT, we can safely ignore any other event
         * here and return back to the event loop and wait for that
         * INETCORESOCK_EVENT_CONNECT.
         */
        return 1;
    }

    // Enter the state machine.
    while (1)
    {
        switch (m_pCtx->eState)
        {
            case INETCOREHTTP_REQUEST_CONNECT:
                break;

            case INETCOREHTTP_REQUEST_SEND:
            case INETCOREHTTP_REQUEST_RECV:
                if ((m_pCtx->eState == INETCOREHTTP_REQUEST_SEND) &&
                    (nEvent & INETCORESOCK_EVENT_WRITE))
                {
                    // Send request body.
                }
                else if (nEvent & INETCORESOCK_EVENT_READ)
                {
                    // Read response.
                    int nRead = pSocket->Read (m_pBuffer, m_nBufSiz);
                    if (nRead > 0)
                    {
                    }
                    else if (nRead == INETCORESOCK_STATUS_WOULDBLOCK)
                    {
                    }
                    else if (nRead == INETCORESOCK_STATUS_CLOSED)
                    {
                        /*
                         * Connection closed.
                         * Either the document is loaded or a
                         * persistent connection re-use has failed.
                         */
                    }
                    else
                    {
                        // Socket read error.
                    }
                }
                else
                {
                    // Ignore event.
                    return 1;
                }
                break;

            default: // (eState == INETCOREHTTP_REQUEST_ERROR)
                int nReplyCode = m_pCtx->nReplyCode;
                if ((nReplyCode < 0) || (nReplyCode > 399))
                {
                    /*
                     * Technical or protocol error. Discard this connection.
                     *
                     * Note: Next state upon 4xx and 5xx status codes.
                     *
                     * State has always been ERROR prior to introduction of
                     * persistent connections.
                     * Has then been changed into SUCCES.
                     * More experience with Netscape-Enterprise/2.0a server
                     * shows the following behaviour:
                     *   Server always replies with a "Connection: keep-alive"
                     *   header field, including 4xx and 5xx status code
                     *   replies, but upon  those codes just closes the
                     *   connection.
                     * For preventing any wrong conclusions about the
                     * connection state, based upon "Connection: keep-alive"
                     * response header field only, next state has been changed
                     * back into ERROR upon 4xx and 5xx status codes.
                     */
                    m_pSocket = NULL;
                }

                // Check connection persistence.
                if (m_pSocket && m_pCtx->pSourceMsg && m_pCtx->pTargetMsg)
                {
                    if (m_pCtx->pSourceMsg->GetConnection().Len())
                    {
                        String aCon = m_pCtx->pTargetMsg->GetConnection ();
                        if (aCon.Len ())
                        {
                            aCon.EraseLeadingChars (' ');
                            if (aCon.ICompare ("keep-alive") == COMPARE_EQUAL)
                                nReplyCode |= INETCOREHTTP_REPLY_M_PERSISTENT;
                            else
                                m_pSocket = NULL;
                        }
                        else
                        {
                            m_pSocket = NULL;
                        }
                    }
                    else
                    {
                        if (m_pCtx->eMethod ==
                            INetCoreHTTPRequest::METHOD_CONNECT)
                            nReplyCode |= INETCOREHTTP_REPLY_M_PERSISTENT;
                        else
                            m_pSocket = NULL;
                    }
                }

                if (m_pSocket == NULL)
                {
                    // Close connection.
                    pSocket->Shutdown();
                }

                if (m_pCtx->pfnCB)
                {
#ifdef DBG_UTIL_PRIVATE
                    String aTrace (
                        "INetCoreHTTPRequest_Impl::RequestHandler: ");
                    aTrace += (ULONG)this; aTrace += " (";
                    aTrace += nReplyCode;  aTrace += ')';
                    _TRACE(aTrace);
#endif /* DBG_UTIL_PRIVATE */
                }
        } // switch (m_pCtx->eState)
    } // while (1)
}

/*
 * ConnectCallback.
 * (Handles CONNECT method).
 */
int INetCoreHTTPRequest_Impl::ConnectCallback (
    INetCoreHTTPRequest *pRequest,
    int nReplyCode, String *pReplyText,
    INetCoreHTTPRequestContext *pNextCtx)
{
    switch (nReplyCode)
    {
        case INETCOREHTTP_REPLY_RESOLVER_WAIT:
        case INETCOREHTTP_REPLY_RESOLVER_DONE:
        case INETCOREHTTP_REPLY_CONNECT_WAIT:
            // Forward to caller.
            if (pNextCtx->pfnCB) (pNextCtx->pfnCB) (
                pRequest, nReplyCode, pReplyText, pNextCtx->pDataCB);
            break;

        case INETCOREHTTP_REPLY_CONNECT_DONE:
        case INETCOREHTTP_REPLY_REQUEST_WAIT:
        case INETCOREHTTP_REPLY_REQUEST_DONE:
        case INETCOREHTTP_REPLY_RESPONSE_WAIT:
            // Hide from caller.
            break;

        case (INETCOREHTTP_REPLY_OK | INETCOREHTTP_REPLY_M_PERSISTENT):
            /*
             * Physical connection established (tunnel is open).
             * Establish logical connection.
             */
            pRequest->m_pImpl->m_pSocket->RemoveCallback (
                (INetCoreIPSocketCallback *)RequestCallback);

            pRequest->m_pImpl->m_pCtx->pDataCB = NULL;
            delete (pRequest->m_pImpl->m_pCtx);
            pRequest->m_pImpl->m_pCtx = pNextCtx;

            pNextCtx->ePrev = pNextCtx->eState;
            pNextCtx->eState = INETCOREHTTP_REQUEST_CONNECT;

            pRequest->m_pImpl->m_pSocket->Connect (
                pNextCtx->pHostEntry->GetDomainName(),
                pNextCtx->pHostEntry->GetPort(),
#ifdef SOLAR_SSL
                (INetCoreSSLConnectCallback *)RequestCallback,
#else
                (INetCoreIPSocketCallback *)RequestCallback,
#endif /* SOLAR_SSL */
                pRequest);

            // Done.
            break;

        default: // Any other means failure.
            break;
    } // switch (nReplyCode)
    return 1;
}

/*
 * InitializeRequest.
 */
INetCoreHTTPRequestContext*
INetCoreHTTPRequest_Impl::InitializeRequest (
    const String& rReqURI, INetCoreHTTPRequestContext *pNewCtx)
{
#ifdef DBG_UTIL_PRIVATE
    String aTrace ("INetCoreHTTPRequest_Impl::InitializeRequest: ");
    aTrace += (ULONG)this;
    aTrace += " ("; aTrace += rReqURI; aTrace += ')';
    _TRACE(aTrace);
#endif /* DBG_UTIL_PRIVATE */

    // Check request URI.
    // Parse request URI.
    // Obtain Scheme.
    // Obtain Host and Port.
    // Scheme dependent initialization.
    switch (pNewCtx->eScheme)
    {
#ifdef SOLAR_SSL
        case INetCoreHTTPRequest::SCHEME_HTTPS:
#ifdef KEEP_ALIVE
            pNewCtx->pSourceMsg->SetConnection ("keep-alive");
            pNewCtx->pSourceMsg->SetKeepAlive ("timeout=30");
#else
            pNewCtx->pSourceMsg->SetConnection ("");
            pNewCtx->pSourceMsg->SetKeepAlive ("");
#endif /* KEEP_ALIVE */

            pNewCtx->pHostEntry = new INetCoreDNSHostEntry (aHost, nPort);
            pNewCtx->SetRequestURI (aPath);

            if (m_pOwner->aSecurityProxyName.Len())
            {
                // Request via tunneling Proxy (using CONNECT method).
                INetHTTPMessage *pReqMsg = new INetHTTPMessage;

                pReqMsg->SetUserAgent (
                    pNewCtx->pSourceMsg->GetUserAgent());
                pReqMsg->SetProxyAuthorization (
                    pNewCtx->pSourceMsg->GetProxyAuthorization());

                INetCoreHTTPRequestContext *pNextCtx =
                    new INetCoreHTTPRequestContext (
                        INETCOREHTTP_METHOD_CONNECT,
                        *pReqMsg, *(pNewCtx->pTargetMsg),
                        (INetCoreHTTPRequestCallback *)ConnectCallback,
                        pNewCtx);
                pNextCtx->eScheme = pNewCtx->eScheme;

                String aDest (aHost); aDest += ':'; aDest += nPort;
                pNextCtx->SetRequestURI (aDest);

                pNextCtx->pHostEntry = new INetCoreDNSHostEntry (
                    m_pOwner->aSecurityProxyName,
                    (m_pOwner->nSecurityProxyPort ?
                     m_pOwner->nSecurityProxyPort : INETCOREHTTP_DEF_PORT));

                return pNextCtx;
            }
            break;
#endif /* SOLAR_SSL */

        case INetCoreHTTPRequest::SCHEME_OTHER:
            // Check for Gateway and fall through.
            if (m_pOwner->aProxyName.Len() == 0) return NULL;

        default: // SCHEME_HTTP
            if (m_pOwner->aProxyName.Len() == 0)
            {
                // Request from Origin.
#ifdef KEEP_ALIVE
                pNewCtx->pSourceMsg->SetConnection ("keep-alive");
                pNewCtx->pSourceMsg->SetKeepAlive ("timeout=30");
#else
                pNewCtx->pSourceMsg->SetConnection ("");
                pNewCtx->pSourceMsg->SetKeepAlive ("");
#endif /* KEEP_ALIVE */

                pNewCtx->pHostEntry = new INetCoreDNSHostEntry (aHost, nPort);
                pNewCtx->SetRequestURI (aPath);
            }
            else
            {
                // Request via Proxy or Gateway.
                pNewCtx->pSourceMsg->SetConnection ("");
                pNewCtx->pSourceMsg->SetKeepAlive ("");

                pNewCtx->pHostEntry = new INetCoreDNSHostEntry (
                    m_pOwner->aProxyName,
                    (m_pOwner->nProxyPort ?
                     m_pOwner->nProxyPort : INETCOREHTTP_DEF_PORT));
                pNewCtx->SetRequestURI (rReqURI);
            }
            break;
    }
    return pNewCtx;
}

/*
 * StartRequest.
 * (Initiate a HTTP request).
 */
BOOL INetCoreHTTPRequest_Impl::StartRequest (
    const String& rAbsURI, INetCoreHTTPRequestContext *pNewCtx)
{
    // Check current request context.
    // Initialize and store new request context.
    m_pCtx = InitializeRequest (rAbsURI, pNewCtx);

    // Start request.
    if (m_pSocket)
    {
        // Try using warm (persistent) connection.
        INetCoreHTTPRequestCallback *pfnCB = m_pCtx->pfnCB;
        m_pCtx->pfnCB = NULL;

        m_pCtx->ePrev = m_pCtx->eState;
        m_pCtx->eState = INETCOREHTTP_REQUEST_SEND;
        nRBytes = -10; /* _EXPERIMENTAL_ */

        int nRet = RequestHandler (m_pSocket, INETCORESOCK_EVENT_WRITE, FALSE);
        if (m_pCtx->eState == INETCOREHTTP_REQUEST_BEGIN)
        {
            // Oops, already finished! Notify caller.
            if (pfnCB)
            {
                int nReplyCode = m_pCtx->nReplyCode;
                if (nRet) nReplyCode |= INETCOREHTTP_REPLY_M_PERSISTENT;

                (pfnCB) (m_pOwner, nReplyCode, NULL, m_pCtx->pDataCB);
            }

            // Done.
            return TRUE;
        }
        else
        {
            // Check return value.
            if (nRet)
            {
                // Ok. Request in progress using warm connection.
                m_pCtx->pfnCB = pfnCB;
                return TRUE;
            }
            else
            {
                // Connection has gotten cold. Fall through.
                m_pSocket = NULL;
                nRBytes = 0; /* _EXPERIMENTAL_ */
                m_pCtx->pfnCB = pfnCB;
            }
        }
    }

    if (m_pSocket == NULL)
    {
        // Initiate a new connection.
        // Start with domain name resolution.
    }

    // Never reached.
    return FALSE;
}

#endif /* OBSOLETE */

