/*************************************************************************
 *
 *  $RCSfile: syncbase.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:16:53 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <errno.h>
#include <vos/timer.hxx>
#include "synccont.hxx"
#include "syncbase.hxx"

// -----------
// - Defines -
// -----------

#define INITIAL_KEY_LENGTH	1024
#define INITIAL_DATA_LENGTH	1024

// -------------
// - BaseTimer -
// -------------

using namespace vos;

class BaseTimer : public OTimer
{
protected:

	SyncBase&				mrBase;

	virtual void SAL_CALL	onShot();

public:

							BaseTimer( SyncBase& rBase, const TTimeValue& rTimeout ) : OTimer( rTimeout, rTimeout ), mrBase( rBase ) { start(); }
};

// -----------------------------------------------------------------------------

void SAL_CALL BaseTimer::onShot()
{
	if( mrBase.IsModified() )
		mrBase.Flush();
}

// ------------------
// - Error handling -
// ------------------

#ifdef DEBUG

extern "C" void __cdecl BaseErrorFnc( const char* pErrPrefix, char* pMsg )
{
	DBG_ERROR( pMsg );
}

#endif

// -------------------
// - SyncBaseElement -
// -------------------

SyncBaseElement::SyncBaseElement()
{
	ImplInit( OUString(), NULL );
}

// -----------------------------------------------------------------------------

SyncBaseElement::SyncBaseElement( const OUString& rElementIdentifier, const Dbt* pData )
{
	ImplInit( rElementIdentifier, pData );
}

// -----------------------------------------------------------------------------

SyncBaseElement::SyncBaseElement( const Dbt* pKey, const Dbt* pData )
{
	if( pKey )
	{
		sal_uChar* pTmpKey = (sal_uChar*) pKey->get_data();
		ImplInit( Get_OWString( pTmpKey, pKey->get_size() >> 1 ), pData );
	}
	else
		ImplInit( OUString(), NULL );
}

// -----------------------------------------------------------------------------

SyncBaseElement::SyncBaseElement( const OUString& rContentIdentifier, sal_uInt32 nType,
								  sal_uInt32 nUID, sal_uInt32 nGeneration,
								  sal_uInt32 nEvent, sal_uInt32 nAction )
{
	ContentIdentifier = rContentIdentifier;
	Type = nType;
	UID = nUID;
	Generation = nGeneration;
	Event = nEvent;
	Action = nAction;
}

// -----------------------------------------------------------------------------

void SyncBaseElement::ImplInit( const OUString& rElementIdentifier, const Dbt* pData )
{
	if( pData )
	{
		sal_uChar*			pTmpData = (sal_uChar*) pData->get_data();
		const sal_uInt32	nVersion = Get_uInt32( pTmpData );

		// assign content identifier
		ContentIdentifier = rElementIdentifier;

		// read out data
		Type = Get_uInt32( pTmpData ); 
		UID = Get_uInt32( pTmpData ); 
		Generation = Get_uInt32( pTmpData ); 
		Event = Get_uInt32( pTmpData ); 
		Action = Get_uInt32( pTmpData );
	}
	else
	{
		Type = SyncType::NONE;
		UID = Generation = Event = Action = 0;
	}
}

// -----------------------------------------------------------------------------

sal_uInt32 SyncBaseElement::GenerateUID()
{
	return Time::GetSystemTicks();
}

// -----------------------------------------------------------------------------

sal_uInt32 SyncBaseElement::Add_uInt32( sal_uChar*& rpBuf, sal_uInt32 nVal )
{
	*rpBuf++ = (sal_uChar) ( ( nVal & 0xff000000 ) >> 24 );
	*rpBuf++ = (sal_uChar) ( ( nVal & 0x00ff0000 ) >> 16 );
	*rpBuf++ = (sal_uChar) ( ( nVal & 0x0000ff00 ) >> 8 );
	*rpBuf++ = (sal_uChar) ( nVal & 0x000000ff );

	return 4;
}

// -----------------------------------------------------------------------------

sal_uInt32 SyncBaseElement::Add_OWString( sal_uChar*& rpBuf, const OUString& rStr, BOOL bLenInfo )
{
	const sal_uInt32	nStrLen = rStr.getLength();
	sal_uInt32			nDataLen = ( nStrLen << 1 );

	if( bLenInfo )
	{
		Add_uInt32( rpBuf, nStrLen );
		nDataLen += 4;
	}

	for( sal_uInt32 i = 0; i < nStrLen; i++ )
	{
		const sal_uInt16 aCode = (sal_uInt16) rStr[ i ];

		*rpBuf++ = (sal_uChar) ( aCode & 0x00ff );
		*rpBuf++ = (sal_uChar) ( ( aCode & 0xff00 ) >> 8 );
	}

	return nDataLen;
}

// -----------------------------------------------------------------------------

OUString SyncBaseElement::Get_OWString( sal_uChar*& rpBuf, sal_uInt32 nStrLen )
{
	OUString aStr;

	if( ( nStrLen = nStrLen ? nStrLen : Get_uInt32( rpBuf ) ) > 0 )
	{
		sal_Unicode* pCode = new sal_Unicode[ nStrLen ];

		for( sal_uInt32 i = 0; i < nStrLen; i++ )
		{
			sal_Unicode& rCode = pCode[ i ];
			rCode = *rpBuf++; rCode |= ( (sal_uInt16) (*rpBuf++) ) << 8;
		}

		aStr = OUString( pCode, nStrLen );
		delete[] pCode;
	}

	return aStr;
}

// -----------------------------------------------------------------------------

sal_uInt32 SyncBaseElement::Get_uInt32( sal_uChar*& rpBuf )
{
	sal_uInt32 nRet = ( (sal_uInt32) *rpBuf++ ) << 24;

	nRet |= ( (sal_uInt32) *rpBuf++ ) << 16;
	nRet |= ( (sal_uInt32) *rpBuf++ ) << 8;

	return( nRet |= (sal_uInt32) *rpBuf++ );
}

// -----------------------------------------------------------------------------

Dbt* SyncBaseElement::GetDbt() const
{
	static const sal_uInt32 nVersion = 1;

	Dbt* pRet;

	if( !IsEmpty() )
	{
		const sal_uInt32	nDataLen =	sizeof( nVersion ) + 
										sizeof( Type ) +
										sizeof( UID ) + 
										sizeof( Generation ) + 
										sizeof( Event ) + 
										sizeof( Action );
		sal_uChar*			pDataBuf = new sal_uChar[ nDataLen ];
		sal_uChar*			pTmp = pDataBuf;

		// fill buffer
		Add_uInt32( pTmp, nVersion );
		Add_uInt32( pTmp, GetType() );
		Add_uInt32( pTmp, GetUID() );
		Add_uInt32( pTmp, GetGeneration() );
		Add_uInt32( pTmp, GetEvent() );
		Add_uInt32( pTmp, GetAction() );

		pRet = new Dbt( pDataBuf, nDataLen );
	}
	else
		pRet = NULL;

	return pRet;
}

// ------------
// - SyncBase -
// ------------

SyncBase::SyncBase( const OUString& rBaseFile, const OUString& rName, sal_uInt32 nBaseFlags ) :
	maBaseFile	( rBaseFile ),
	maName		( rName ),
	mpBase		( new Db( NULL, DB_CXX_NO_EXCEPTIONS ) ),
	mbModified	( sal_False )
{
#ifdef DEBUG
	mpBase->set_errpfx( "SyncBase" );
	// mpBase->set_errcall( (void (__stdcall *)(const char *,char *)) &BaseErrorFnc ); 
#endif

	mpBase->set_pagesize( 1024 );
	mpBase->set_cachesize(0, 32 * 1024, 0);
	mpBase->open( ByteString( UniString( rBaseFile ), RTL_TEXTENCODING_UTF8 ).GetBuffer(), 
				  ByteString( UniString( rName ), RTL_TEXTENCODING_UTF8 ).GetBuffer(), 
				  DB_BTREE, DB_CREATE, 0664 );

	if( nBaseFlags & BASE_FLAGS_AUTOFLUSH )
		mpTimer = new BaseTimer( *this, 5000 );
	else
		mpTimer = NULL;
}

// -----------------------------------------------------------------------------

SyncBase::~SyncBase()
{
	delete mpTimer;

	if( mpBase )
	{
		mpBase->close( 0 );
		delete mpBase;
	}
}

// -----------------------------------------------------------------------------

Dbt* SyncBase::GetKey( const OUString& rElementIdentifier )
{
	Dbt*				pRet;
	const sal_uInt32	nKeyLen = ( rElementIdentifier.getLength() << 1 );

	if( nKeyLen )
	{
		sal_uChar* pData = new sal_uChar[ nKeyLen ];
		sal_uChar* pTmp = pData;

		SyncBaseElement::Add_OWString( pTmp, rElementIdentifier, sal_False );
		pRet = new Dbt( pData, nKeyLen );
	}
	else
		pRet = NULL;

	return pRet;
}

// -----------------------------------------------------------------------------

void SyncBase::ReleaseDbt( Dbt* pDbt )
{
	if( pDbt )
	{
		delete[] (sal_uChar*) pDbt->get_data();
		delete pDbt;
	}
}

// -----------------------------------------------------------------------------

void SyncBase::RemovePermanent()
{
	if( mpBase )
	{
		mpBase->close( 0 );
		// mpBase->remove( U2B( GetBaseFile() ), U2B( GetName() ), 0 );
		delete mpBase;
		mpBase = NULL;
	}
}

// -----------------------------------------------------------------------------

sal_Bool SyncBase::PutElement( const SyncBaseElement& rElement, sal_Bool bFlush )
{
	sal_Bool bRet = sal_False;

	if( mpBase )
	{
		Dbt* pKey = GetKey( rElement.GetContentIdentifier() );

		if( pKey )
		{
			Dbt* pData = rElement.GetDbt();

			if( pData )
			{
				bRet = ( 0 == mpBase->put( NULL, pKey, pData, 0 ) );
				ReleaseDbt( pData );

				if( bRet )
				{
					mbModified = sal_True;

					if( bFlush )
						Flush();
				}
			}

			ReleaseDbt( pKey );
		}
	}

	return bRet;
}

// -----------------------------------------------------------------------------

SyncBaseElement* SyncBase::GetElement( const OUString& rElementIdentifier )
{
	SyncBaseElement* pRet = NULL;

	if( mpBase )
	{
		Dbt* pKey = GetKey( rElementIdentifier );

		if( pKey )
		{
			sal_uChar*	pDataBuf = new sal_uChar[ INITIAL_DATA_LENGTH ];
			Dbt*		pData = new Dbt( pDataBuf, INITIAL_DATA_LENGTH );

			pData->set_flags( pData->get_flags() | DB_DBT_USERMEM );
			pData->set_ulen( INITIAL_DATA_LENGTH );

			sal_uInt32 nRet = mpBase->get( NULL, pKey, pData, 0 );

			if( ENOMEM == nRet )
			{
				delete[] pDataBuf;
				pData->set_data( pDataBuf = new sal_uChar[ pData->get_size() ] ); 
				pData->set_ulen( pData->get_size() );

				nRet = mpBase->get( NULL, pKey, pData, 0 );
			}

			pRet = ( 0 == nRet ) ? new SyncBaseElement( rElementIdentifier, pData ) : NULL;
			ReleaseDbt( pData );
			ReleaseDbt( pKey );
		}
	}

	return pRet;
}
// -----------------------------------------------------------------------------

sal_Bool SyncBase::RemoveElement( const OUString& rElementIdentifier, sal_Bool bFlush )
{
	sal_Bool bRet = sal_False;

	if( mpBase )
	{
		Dbt* pKey = GetKey( rElementIdentifier );

		if( pKey )
		{
			bRet = ( 0 == mpBase->del( NULL, pKey, 0 ) );

			if( bRet )
			{
				mbModified = sal_True;

				if( bFlush )
					Flush();
			}

			ReleaseDbt( pKey );
		}
	}

	return bRet;
}

// -----------------------------------------------------------------------------

SyncCursor* SyncBase::GetCursor()
{
	SyncCursor* pCursor = new SyncCursor( this );

	if( !mpBase || ( 0 != mpBase->cursor( NULL, &pCursor->Cursor(), 0 ) ) )
		delete pCursor, pCursor = NULL;

	return pCursor;
}

// -----------------------------------------------------------------------------

sal_uInt32 SyncBase::Count()
{
	sal_uInt32 nRet = 0;

	if( mpBase )
	{
		SyncCursor*	pCursor = GetCursor();

		if( pCursor )
		{
			for( const SyncBaseElement* pElem = pCursor->First(); pElem; pElem = pCursor->Next() )
				nRet++;

			delete pCursor;
		}
	}

	return nRet;
}

// -----------------------------------------------------------------------------

void SyncBase::Flush()
{
	if( mpBase )
		mpBase->sync( 0 );

	mbModified = sal_False;
}

// --------------
// - SyncCursor -
// --------------

SyncCursor::~SyncCursor()
{
	mpCursor->close();
	delete mpAct;
}

// -----------------------------------------------------------------------------

const SyncBaseElement* SyncCursor::GetElement( sal_uInt32 nCursorPos )
{
	sal_uChar*		pKeyBuf = new sal_uChar[ INITIAL_KEY_LENGTH ];
	sal_uChar*		pDataBuf = new sal_uChar[ INITIAL_DATA_LENGTH ];
	Dbt*			pKey = new Dbt( pKeyBuf, INITIAL_KEY_LENGTH );
	Dbt*			pData = new Dbt( pDataBuf, INITIAL_DATA_LENGTH );

	pKey->set_flags( pKey->get_flags() | DB_DBT_USERMEM );
	pKey->set_ulen( pKey->get_size() );

	pData->set_flags( pData->get_flags() | DB_DBT_USERMEM );
	pData->set_ulen( pData->get_size() );

	sal_uInt32 nRet = mpCursor->get( pKey, pData, nCursorPos );

	if( ENOMEM == nRet )
	{
		if( pKey->get_ulen() < pKey->get_size() )
		{
			delete[] pKeyBuf;
			pKey->set_data( pKeyBuf = new sal_uChar[ pKey->get_size() ] ); 
			pKey->set_ulen( pKey->get_size() );
		}

		if( pData->get_ulen() < pData->get_size() )
		{
			delete[] pDataBuf;
			pData->set_data( pDataBuf = new sal_uChar[ pData->get_size() ] ); 
			pData->set_ulen( pData->get_size() );
		}

		nRet = mpCursor->get( pKey, pData, nCursorPos );
		
		if( ENOMEM == nRet )
		{
			if( pKey->get_ulen() < pKey->get_size() )
			{
				delete[] pKeyBuf;
				pKey->set_data( pKeyBuf = new sal_uChar[ pKey->get_size() ] ); 
				pKey->set_ulen( pKey->get_size() );
			}

			if( pData->get_ulen() < pData->get_size() )
			{
				delete[] pDataBuf;
				pData->set_data( pDataBuf = new sal_uChar[ pData->get_size() ] ); 
				pData->set_ulen( pData->get_size() );
			}

			nRet = mpCursor->get( pKey, pData, nCursorPos );
		}
	}

	delete mpAct, mpAct = ( ( 0 == nRet ) ? new SyncBaseElement( pKey, pData ) : NULL );
	
	mpBase->ReleaseDbt( pData );
	mpBase->ReleaseDbt( pKey );

	return mpAct;
}
