/*************************************************************************
 *
 *  $RCSfile: redirector.cxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: jb $ $Date: 2001/07/05 17:05:48 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#ifndef _CONFIGMGR_SESSION_REDIRECTOR_HXX_
#include "redirector.hxx"
#endif
#ifndef _CONFIGMGR_SESSION_RS_TYPES_HXX_
#include "rs_types.hxx"
#endif
#ifndef CONFIGMGR_CONFIGPATH_HXX_
#include "configpath.hxx"
#endif
#ifndef _CONFIGMGR_TRACER_HXX_
#include "tracer.hxx"
#endif

#ifndef _TOOLS_DEBUG_HXX
#include <osl/diagnose.h>
#endif

using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::xml;
using namespace ::com::sun::star::xml::sax;
using namespace ::osl;
using namespace ::comphelper;

//..........................................................................
namespace configmgr
{
//..........................................................................

/*	the transition table for startElement :

					ENVELOPE	HEADER	TRANSACTION	BODY	PARAMS	DATA	VALUE

	(0)	EXPECTING_ENVELOPE	1		-	-		-	-	-	-
	(1)	ENVELOPE_STARTED	-		2	-		-	-	-	-
	(2)	HEADER_STARTED		-		-	3		-	-	-	-
	(3)	TRANS_STARTED		-		-	-		-	-	-	-
	(4)	TRANS_DONE		-		-	-		-	-	-	-
	(5)	HEADER_DONE		-		-	-		6	-	-	-
	(6)	BODY_STARTED		-		-	-		-	7	-	-
	(7)	PARAMS_STARTED		-		-	-		-	-	8	-
	(8)	DATA_STARTED		-		-	-		-	-	-	9
	(9)	VALUE_STARTED		-		-	-		-	-	-	-
	(10)	VALUE_DONE		-		-	-		-	-	-	-
	(11)	DATA_DONE		-		-	-		-	-	8	-
	(12)	PARAMS_DONE		-		-	-		-	-	-	-
	(13)	BODY_DONE		-		-	-		-	-	-	-

	note (1) :<BR>
		all "-" means that for every startElement (no matter what element it is) we increment a counter for
		"unknown element depth" (<property scope="OResponseRedirector">m_nElementDepth</property>),
		which is decremented by endElement and, as long as it is greater 0, locks the transition tables
		and the handling of all other tokens (i.e. characters). For clarity all these transitions are
		ommitted here.
		<BR>
		Nearly the same holds for the FORWARDING_DATA state, which also locks the transition table
		and works on a counter, as long as this counter is not 0.

	the transition table for endElement :

					ENVELOPE	HEADER	TRANSACTION	BODY	PARAMS	DATA	VALUE

	(0)	EXPECTING_ENVELOPE	-		-	-		-	-	-	-
	(1)	ENVELOPE_STARTED	0		-	-		-	-	-	-
	(2)	HEADER_STARTED		-		5	-		-	-	-	-
	(3)	TRANS_STARTED		-		-	4		-	-	-	-
	(4)	TRANS_DONE		-		5	-		-	-	-	-
	(5)	HEADER_DONE		0		-	-		-	-	-	-
	(6)	BODY_STARTED		-		-	-		13	-	-	-
	(7)	PARAMS_STARTED		-		-	-		-	12	-	-
	(8)	DATA_STARTED		-		-	-		-	-	11	-
	(9)	VALUE_STARTED		-		-	-		-	-	-	10
	(10)	VALUE_DONE		-		-	-		-	-	11	-
	(11)	DATA_DONE		-		-	-		-	12	-	-
	(12)	PARAMS_DONE		-		-	-		13	-	-	-
	(13)	BODY_DONE		0		-	-		-	-	-	-

	note (1) :<BR>
		in some cases some transitions are defined mainly for robustness, if they're used at runtime this would mean
		a violation of the dtd of the communication protocol
		An example for this would be an "endElement(PARAMS)" if we're in state PARAMS_STARTED : this would lead
		to state 14 = PARAMS_DONE, so we can continue from there, though this is a violation of the dtd
		constraints and will result in nonsense forwarded to the callbacks.<BR>
	note (2) :<BR>
		the states UNKNOWN_ELEMENT and FORWARDING_DATA are not in the table above, they require some special
		handling which includes a counter for the elemnt nesting depth ....<BR>
	note (3) :<BR>
		all "-" in the above table are a heavy and unrecoverable violation of the dtd constraints, so we have no
		transition for them.<BR>
		At the moment I think we should handle this through moving to the EXPECTING_ENVELOPE state, so only the
		current envelope is lost and we can start over again when reading the next one.
*/

// lucky as we are :), the transition tables are sparse matrixes, with (best of all) only one element per row.
// So we use a vector to store them : the access index is the current state, the indexed element is a pair
// of the allowed element name and the new state.

//--------------------------------------------------------------------------
// Macros
//--------------------------------------------------------------------------

#define STATE_COUNT (BODY_DONE + 1) // number of regular states

//--------------------------------------------------------------------------
// OResponseRedirector - inlines
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// types for "class configmgr::ElementTable configmgr:: aTranslationElement"
//--------------------------------------------------------------------------

DECLARE_STL_MAP ( ::rtl::OUString, ELEMENT, UStringMixLess, ElementTable );

static ElementTable aTranslationElement ( UStringMixLess ( sal_False ) );

//--------------------------------------------------------------------------

inline ELEMENT OResponseRedirector::translateElement ( const ::rtl::OUString& _rElementName )
	{
		if ( !aTranslationElement.size ( ) )
			{
				// initialize the table

				aTranslationElement [ ::rtl::OUString::createFromAscii("envelope") ]    = ENVELOPE;
				aTranslationElement [ ::rtl::OUString::createFromAscii("header") ]      = HEADER;
				aTranslationElement [ ::rtl::OUString::createFromAscii("request") ]     = REQUEST;
				aTranslationElement [ ::rtl::OUString::createFromAscii("transaction") ] = TRANSACTION;
				aTranslationElement [ ::rtl::OUString::createFromAscii("body") ]        = BODY;
				aTranslationElement [ ::rtl::OUString::createFromAscii("params") ]      = PARAMS;
				aTranslationElement [ ::rtl::OUString::createFromAscii("data") ]        = DATA;
				aTranslationElement [ ::rtl::OUString::createFromAscii("value") ]       = VALUE;
			} // if

		ConstElementTableIterator aPos = aTranslationElement.find ( _rElementName );
		
		if ( aTranslationElement.end ( ) == aPos )
			{
				return UNKNOWN;
			} // if

		return aPos->second;
	} // inline method OResponseRedirector::translateElement

//------------------------------------------------------------------------------------
// types for "class configmgr::TransTypeTable configmgr:: aTranslationTransactionType"
//------------------------------------------------------------------------------------

DECLARE_STL_MAP ( ::rtl::OUString, TransactionType, UStringMixLess, TransTypeTable );
		
static TransTypeTable aTranslationTransactionType ( UStringMixLess ( sal_False ) );

//--------------------------------------------------------------------------

inline TransactionType OResponseRedirector::translateTransactionType ( const ::rtl::OUString& _rTransType )
	{
		if ( !aTranslationTransactionType.size ( ) )
			{
				// initialize the table

				aTranslationTransactionType [ ::rtl::OUString::createFromAscii("request") ]      = ttRequest;
				aTranslationTransactionType [ ::rtl::OUString::createFromAscii("response") ]     = ttResponse;
				aTranslationTransactionType [ ::rtl::OUString::createFromAscii("error") ]        = ttError;
				aTranslationTransactionType [ ::rtl::OUString::createFromAscii("ACK") ]          = ttAcknowledgment;
				aTranslationTransactionType [ ::rtl::OUString::createFromAscii("notification") ] = ttNotification;
			}

		ConstTransTypeTableIterator aPos = aTranslationTransactionType.find(_rTransType);
		
		if ( aTranslationTransactionType.end ( ) == aPos )
			{
				return ttUnknown;
			} // if

		return aPos->second;
	} // inline method OResponseRedirector::translateTransactionType

//------------------------------------------------------------------------------------
// types for "class configmgr::TransNameTable configmgr:: aTranslationTransactionName"
//------------------------------------------------------------------------------------

DECLARE_STL_MAP ( ::rtl::OUString, TransactionName, UStringMixLess, TransNameTable );

static TransNameTable aTranslationTransactionName ( UStringMixLess ( sal_False ) );

//--------------------------------------------------------------------------

inline TransactionName OResponseRedirector::translateTransactionName ( const ::rtl::OUString& _rTransName )
	{
		if ( !aTranslationTransactionName.size ( ) )
			{
				// initialize the table

				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("openSession") ]       = tnOpenSession;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("closeSession") ]      = tnCloseSession;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("openNode") ]          = tnOpenNode;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("closeNode") ]         = tnCloseNode;				
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("updateNode") ]        = tnUpdateNode;												
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("getStatus") ]         = tnGetStatus;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("cancelTransaction") ] = tnCancelTransaction;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("addUser") ]           = tnAddUser;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("deleteUser") ]        = tnDeleteUser;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("addGroup") ]          = tnAddGroup;
				aTranslationTransactionName [ ::rtl::OUString::createFromAscii("deleteGroup") ]       = tnDeleteGroup;
			} // if

		ConstTransNameTableIterator aPos = aTranslationTransactionName.find ( _rTransName );

		if ( aTranslationTransactionName.end ( ) == aPos )
			{
				return tnUnknown;
			} // if

		return aPos->second;
	} // inline method OResponseRedirector::translateTransactionName

//-------------------------------------------------------------------------------
// types for "class configmgr::PDTypeTable configmgr:: aTranslationParamDataType"
//-------------------------------------------------------------------------------

DECLARE_STL_MAP(::rtl::OUString, ParamDataType, UStringMixLess, PDTypeTable);
		
static PDTypeTable aTranslationParamDataType ( UStringMixLess ( sal_False ) );

//--------------------------------------------------------------------------

inline ParamDataType OResponseRedirector::translateParamDataType ( const ::rtl::OUString& _rParamType )
	{
		if ( !aTranslationParamDataType.size ( ) )
			{
				// initialize the table

				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("string") ]   = pdtString;
				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("int") ]	   = pdtInteger;
				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("boolean") ]  = pdtBoolean;
				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("nodepath") ] = pdtNodepath;
				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("node") ]     = pdtNode;
				aTranslationParamDataType [ ::rtl::OUString::createFromAscii("binary") ]   = pdtBinary;

				#ifdef _WRONG_ERROR_RESPONSE_PARAM_DATA_TYPE_
					aTranslationParamDataType [ ::rtl::OUString::createFromAscii("0") ] = pdtString;
				#endif

				#ifdef _SERVER_KNOWS_DATA_TYPE_LONG_
					aTranslationParamDataType [ ::rtl::OUString::createFromAscii("long") ] = pdtInteger;
				#endif
			} // if

		ConstPDTypeTableIterator aPos = aTranslationParamDataType.find ( _rParamType );

		if ( aTranslationParamDataType.end ( ) == aPos )
			{
				return pdtUnknown;
			} // if

		return aPos->second;
	} // inline method OResponseRedirector::translateParamDataType


//==========================================================================
//= OResponseRedirector
//==========================================================================

//--------------------------------------------------------------------------
OResponseRedirector::OResponseRedirector()
	:m_eState(EXPECTING_ENVELOPE)
	,m_ePushedState(EXPECTING_ENVELOPE)
	,m_nElementDepth(0)
	,m_nExpectedParams(0)
	,m_nCurrentParamNum(-1)
	,m_eCurrentParamType(pdtUnknown)	
	,m_bIgnoreEnvelopeTail(sal_False)
{
}

//--------------------------------------------------------------------------
OResponseRedirector::~OResponseRedirector()
{
}

//--------------------------------------------------------------------------
void OResponseRedirector::registerCallback(const ::rtl::OUString& _rRequestId, const::vos::ORef< IRequestCallback >& _rCB)
{
	MutexGuard aGuard(m_aMutex);

	OSL_ENSURE(m_aCallbacks.find(_rRequestId) == m_aCallbacks.end(),
		"OResponseRedirector::registerCallback : request id already used !");
	OSL_ENSURE(!_rCB.isEmpty(), "OResponseRedirector::registerCallback : invalid callback interface !");
	if (!_rCB.isEmpty())
		// only insert if it's valid, so we can save some checks later (where we would have to do it more often)
		m_aCallbacks[_rRequestId] = CallbackDescription(_rCB);
}

//--------------------------------------------------------------------------
void OResponseRedirector::registerCallback(const ::rtl::OUString& _rRequestId, const::vos::ORef< IDataRequestCallback >& _rCB)
{
	MutexGuard aGuard(m_aMutex);

	OSL_ENSURE(m_aCallbacks.find(_rRequestId) == m_aCallbacks.end(),
		"OResponseRedirector::registerCallback : request id already used !");
	OSL_ENSURE(!_rCB.isEmpty(), "OResponseRedirector::registerCallback : invalid callback interface !");
	if (!_rCB.isEmpty())
		// only insert if it's valid, so we can save some checks later (where we would have to do it more often)
		m_aCallbacks[_rRequestId] = CallbackDescription(_rCB);
}

//--------------------------------------------------------------------------
void OResponseRedirector::registerCallback(const ::rtl::OUString& _rRequestId, 
										   const::vos::ORef< IOpenObjectCallback >& _rCB,
										   const ::vos::ORef< INotifyCallback >& _rNotifyCB)
{
	MutexGuard aGuard(m_aMutex);

	OSL_ENSURE(m_aCallbacks.find(_rRequestId) == m_aCallbacks.end(),
		"OResponseRedirector::registerCallback : request id already used !");
	OSL_ENSURE(!_rCB.isEmpty(), "OResponseRedirector::registerCallback : invalid callback interface !");
	if (!_rCB.isEmpty())
		// only insert if it's valid, so we can save some checks later (where we would have to do it more often)
		m_aCallbacks[_rRequestId] = CallbackDescription(_rCB, _rNotifyCB);
}

//--------------------------------------------------------------------------
void OResponseRedirector::removeCallback(const ::rtl::OUString& _rNodeId)
{
	MutexGuard aGuard(m_aMutex);
	OSL_ENSURE(_rNodeId.getLength(),  "OResponseRedirector::removeCallback: nodeId can not be NULL");

	if (_rNodeId.getLength())
	{
		for (CallbacksIterator i = m_aCallbacks.begin(); i != m_aCallbacks.end(); i++)
		{
			if ((*i).second.sNodeId == _rNodeId)
			{
				m_aCallbacks.erase(i);
				break;
			}
		}
	}
}
//--------------------------------------------------------------------------

void SAL_CALL OResponseRedirector::startDocument(  ) throw(sax::SAXException, RuntimeException)
{
	MutexGuard aGuard(m_aMutex);

//	OSL_ENSURE(sal_False, "OResponseRedirector::startDocument : suspocious call : we should not have complete documents !");
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::endDocument(  ) throw(sax::SAXException, RuntimeException)
{
	MutexGuard aGuard(m_aMutex);

	// this means that en envelope has ended, so reset the state machine
	OSL_ENSURE(EXPECTING_ENVELOPE == m_eState, "OResponseRedirector::endDocument : called within an envelope !");

	if (EXPECTING_ENVELOPE != m_eState)
	{
		m_nElementDepth = 0;
		m_nExpectedParams = 0;
		m_nCurrentParamNum = -1;
		m_eCurrentParamType = pdtUnknown;
		m_aCurrentBodyParams.resize(0);
		m_sCurrentParamValue = ::rtl::OUString();
		m_xCurrentNodeReader = NULL;
		m_aCurrentHeader.eTransType = ttUnknown;
		m_aCurrentHeader.eTransName = tnUnknown;

		m_bIgnoreEnvelopeTail = sal_True;
	}
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::startElement( const ::rtl::OUString& _rName, const Reference< sax::XAttributeList >& _rxAttribs ) throw(sax::SAXException, RuntimeException)
{
	MutexGuard aGuard(m_aMutex);

	// special current states
	if (UNKNOWN_ELEMENT == m_eState)
	{
		OSL_ENSURE(m_nElementDepth, "OResponseRedirector::startElement : m_eState & m_nElementDepth are inconsistent !");
		++m_nElementDepth;
	}
	else if (FORWARDING_DATA == m_eState)
	{
		++m_nElementDepth;
		if (m_xCurrentNodeReader.is())
			m_xCurrentNodeReader->startElement(_rName, _rxAttribs);
			// TODO : think about handling the exceptions ....
	}
	else
	{
		// a (static) transition table (see at the beginning of the file)
		static Transitions aTransitions;
		if (!aTransitions.size())
		{	// first call, initialize the table
			aTransitions.resize(STATE_COUNT);
			aTransitions[EXPECTING_ENVELOPE]	= ValidTransition("envelope",	ENVELOPE_STARTED);
			aTransitions[ENVELOPE_STARTED]		= ValidTransition("header",		HEADER_STARTED);
			aTransitions[HEADER_STARTED]		= ValidTransition("transaction",TRANS_STARTED);
			aTransitions[TRANS_STARTED]			= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start within the transaction scope
			aTransitions[TRANS_DONE]			= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start after the transaction scope
			aTransitions[HEADER_DONE]			= ValidTransition("body",		BODY_STARTED);
			aTransitions[BODY_STARTED]			= ValidTransition("params",		PARAMS_STARTED);
			aTransitions[PARAMS_STARTED]		= ValidTransition("data",		DATA_STARTED);
			aTransitions[DATA_STARTED]			= ValidTransition("value",		VALUE_STARTED);
			aTransitions[VALUE_STARTED]			= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start within the value scope
			aTransitions[VALUE_DONE]			= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start after the value scope
			aTransitions[DATA_DONE]				= ValidTransition("data",		DATA_STARTED);
			aTransitions[PARAMS_DONE]			= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start after the params scope
			aTransitions[BODY_DONE]				= ValidTransition(NULL,			UNKNOWN_ELEMENT);	// no element is allowed to start after the body scope
		}

		// the only allowed next element
		sal_Char* pAllowedElement = aTransitions[m_eState].first;

		if (pAllowedElement && (0 == _rName.compareToAscii(pAllowedElement)))
		{	// jupp ... it's the only allowed element which starts here ... we're lucky today
			m_eState = aTransitions[m_eState].second;

			if (m_bIgnoreEnvelopeTail)
				// nothing to do here anymore, we were resetted while reading the current envelope
				return;

			switch (m_eState)
			{
				case TRANS_STARTED:
				{
					// verify the attribute structure (nonpro version only)
					OSL_ENSURE(_rxAttribs.is() && (3 <= _rxAttribs->getLength()),
						"OResponseRedirector::startElement(transaction) : invalid attribute list !");
#ifdef DBG_UTIL
					try
					{
						OSL_ENSURE(	_rxAttribs.is()
								&&	(0 == _rxAttribs->getNameByIndex(0).compareToAscii("type"))
								&&	(0 == _rxAttribs->getNameByIndex(1).compareToAscii("name"))
								&&	(0 == _rxAttribs->getNameByIndex(2).compareToAscii("clientId"))
								&&	(0 == _rxAttribs->getNameByIndex(3).compareToAscii("serverId")),
							"OResponseRedirector::startElement(transaction) : wrong attribute structure !");
					}
					catch(...) { OSL_ENSURE(sal_False, "OResponseRedirector::startElement(transaction) : could not verify the attributes !"); }
#endif
					::rtl::OUString sName, sType, sClientID, sDesc, sServerId;
						// don't assign it directly to the m_aCurrentHeader members, we want to have empty strings in case of failure
					if (_rxAttribs.is())
					{
						try
						{
							sType		= _rxAttribs->getValueByIndex(0);
							sName		= _rxAttribs->getValueByIndex(1);
							sClientID	= _rxAttribs->getValueByIndex(2);
							sServerId	= _rxAttribs->getValueByIndex(3);
							if (_rxAttribs->getLength() > 4)
								sDesc	= _rxAttribs->getValueByIndex(4);
						}
						catch (RuntimeException&)
						{
							OSL_ENSURE(sal_False, "OResponseRedirector::startElement(transaction) : could not collect the elements attributes !");
						}
					}
					m_aCurrentHeader.eTransType = translateTransactionType(sType);
					m_aCurrentHeader.eTransName = translateTransactionName(sName);
					m_aCurrentHeader.sRequestId = sClientID;
					m_aCurrentHeader.nTransId = sServerId.toInt32();
					m_aCurrentHeader.sDescription = sDesc;

					OSL_ENSURE((ttRequest != m_aCurrentHeader.eTransType) && (ttUnknown != m_aCurrentHeader.eTransType),
						"OResponseRedirector::startElement(transaction) : unknown or invalid transaction name !");
				}
				break;
				case DATA_STARTED:
				{
					// check if we have the room to store the upcoming parameter (we should, if the "total" parameter
					// of the "params" element was correct
					if (++m_nCurrentParamNum >= m_aCurrentBodyParams.size())
					{
						// currently notification send 2 parameter instead of 1
						// so no assertion in this case						
						OSL_ENSURE(ttNotification == m_aCurrentHeader.eTransType, "OResponseRedirector::startElement(data) : invalid parameter count !");
						m_aCurrentBodyParams.resize(m_nCurrentParamNum + 1);
					}

					// extract the value type from the attributes list
					OSL_ENSURE(_rxAttribs.is() && (1 == _rxAttribs->getLength()),
						"OResponseRedirector::startElement(data) : invalid attribute list !");
#ifdef DBG_UTIL
					try
					{
						OSL_ENSURE(_rxAttribs.is() && (0 == _rxAttribs->getNameByIndex(0).compareToAscii("type")),
							"OResponseRedirector::startElement(data) : wrong attribute structure !");
					}
					catch(...) { OSL_ENSURE(sal_False, "OResponseRedirector::startElement(data) : could not verify the attributes !"); }
#endif

					::rtl::OUString sType;
					if (_rxAttribs.is())
					{
						try
						{
							sType	= _rxAttribs->getValueByIndex(0);
						}
						catch (RuntimeException&)
						{
							OSL_ENSURE(sal_False, "OResponseRedirector::startElement(data) : could not collect the elements attributes !");
						}
					}
					m_eCurrentParamType = translateParamDataType(sType);
					OSL_ENSURE((pdtBinary != m_eCurrentParamType) && (pdtUnknown != m_eCurrentParamType),
						"OResponseRedirector::startElement(data) : unknown or invalid parameter type !");
						// we're not recognizing and handling m_eCurrentParamType currently
				}
				break;
				case VALUE_STARTED:
				{
					m_sCurrentParamValue = ::rtl::OUString();

					// if the type of the current value is "Node" and the callback for the current response is
					// of type ctDataRequest, immediately transit to FORWARDING_DATA state
					if (pdtNode == m_eCurrentParamType)
					{
						m_ePushedState = m_eState;
						m_eState = FORWARDING_DATA;
						m_nElementDepth = 1;	// 1, not 0 !
						m_xCurrentNodeReader = NULL;

						// get the reader
						if (m_aCallbacks.end() != m_aCurrentCallback)
						{	// we have a callback for the current request
							if	((ctDataRequest == m_aCurrentCallback->second.eType) ||	
								 (ctObjectOpener == m_aCurrentCallback->second.eType) ||
								 (ctNotifyRequest == m_aCurrentCallback->second.eType)
								 
								)
							{	// and this callback can handle node data
								// -> ask it for the reader
								m_xCurrentNodeReader = 
									static_cast< IDataRequestCallback* >(m_aCurrentCallback->second.aCallback.getBodyPtr())->getDataReader();
								if (m_xCurrentNodeReader.is())
									m_xCurrentNodeReader->startDocument();
							}
						}
					}
				}
				break;
				case PARAMS_STARTED:
				{
					OSL_ENSURE(_rxAttribs.is() && (1 == _rxAttribs->getLength()),
						"OResponseRedirector::startElement(params) : invalid attribute list !");

					sal_Int32 nExpectedParams(0);
					if (_rxAttribs.is())
					{
						try
						{
							nExpectedParams = _rxAttribs->getValueByIndex(0).toInt32();
							OSL_ENSURE(nExpectedParams > 0, "OResponseRedirector::startElement : why a params section without params ?");
						}
						catch (RuntimeException&)
						{
							OSL_ENSURE(sal_False, "OResponseRedirector::startElement(params) : could not collect the elements attributes !");
						}
					}
					m_aCurrentBodyParams.resize(nExpectedParams);
				}
				break;
			}

			// if we're starting a new envelope, initialize some members related to the current envelope
			if (ENVELOPE_STARTED == m_eState)
			{
				m_aCurrentHeader.sRequestId = m_aCurrentHeader.sDescription = ::rtl::OUString();
				m_aCurrentHeader.nTransId = -1;
				m_aCurrentHeader.eTransType = ttUnknown;
				m_aCurrentHeader.eTransName = tnUnknown;
				m_aCurrentCallback = m_aCallbacks.end();
				m_nElementDepth = 0;
			}
			// analogous for the body related members
			if (BODY_STARTED == m_eState)
			{
				m_aCurrentBodyParams.resize(0);
				m_nCurrentParamNum = -1;
				m_eCurrentParamType = pdtUnknown;
				m_xCurrentNodeReader = NULL;
			}
		}
		else
		{
			// an unknown (or disallowed) element ... 
			OSL_ENSURE(sal_False, "OResponseRedirector::startElement : an unknown element started ! ignoring all data 'til the corresponding endElement call !");

			// remember the current state
			m_ePushedState = m_eState;

			// and move into that unknown territory
			OSL_ENSURE(m_nElementDepth == 0, "OResponseRedirector::startElement : inconsistent : valid state, but \"unknown element depth\" not 0 !");
			m_eState = UNKNOWN_ELEMENT;
			++m_nElementDepth;
		}
	}
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::endElement( const ::rtl::OUString& _rName ) throw(sax::SAXException, RuntimeException)
{
	ClearableMutexGuard aGuard(m_aMutex);

	// special current states
	if (UNKNOWN_ELEMENT == m_eState)
	{
		OSL_ENSURE(m_nElementDepth, "OResponseRedirector::endElement : m_eState & m_nElementDepth are inconsistent !");
		if (0 == --m_nElementDepth)
		{	// we left the unknown territory ....
			m_eState = m_ePushedState;
		}
	}
	else if (FORWARDING_DATA == m_eState)
	{
		uno::Reference< sax::XDocumentHandler > xReader(m_xCurrentNodeReader);

		if (0 == --m_nElementDepth)
		{	// we left the node description
			m_eState = m_ePushedState;
			// call this method again, this will bring us into the else-branch below
			endElement(_rName);

			aGuard.clear();
			// release the guard to avoid deadlocks
			// this can happen for notification send by the server
			if (xReader.is())
				xReader->endDocument();
		}
		else
		{
			aGuard.clear();
			// release the guard to avoid deadlocks
			// this can happen for notification send by the server
			if (xReader.is())
				xReader->endElement(_rName);
			// TODO : think about handling the exceptions ....
		}
	}
	else
	{
		// a (static) transition table (see at the beginning of the file)
		static Transitions aTransitions;
		if (!aTransitions.size())
		{	// first call, initialize the table
			aTransitions.resize(STATE_COUNT);
			aTransitions[EXPECTING_ENVELOPE]	= ValidTransition(NULL,			UNKNOWN_ELEMENT);
			aTransitions[ENVELOPE_STARTED]		= ValidTransition("envelope",	EXPECTING_ENVELOPE);// this is an empty envelope ... which is suspicious
			aTransitions[HEADER_STARTED]		= ValidTransition("header",		HEADER_DONE);		// empty header ... suspicious, too
			aTransitions[TRANS_STARTED]			= ValidTransition("transaction",TRANS_DONE);
			aTransitions[TRANS_DONE]			= ValidTransition("header",		HEADER_DONE);
			aTransitions[HEADER_DONE]			= ValidTransition("envelope",	EXPECTING_ENVELOPE);// header without body - allowed
			aTransitions[BODY_STARTED]			= ValidTransition("body",		BODY_DONE);			// empty body ... suspicious
			aTransitions[PARAMS_STARTED]		= ValidTransition("params",		PARAMS_DONE);		// empty params section .. suspicious
			aTransitions[DATA_STARTED]			= ValidTransition("data",		DATA_DONE);			// empty data section ... suspicious
			aTransitions[VALUE_STARTED]			= ValidTransition("value",		VALUE_DONE);
			aTransitions[VALUE_DONE]			= ValidTransition("data",		DATA_DONE);
			aTransitions[DATA_DONE]				= ValidTransition("params",		PARAMS_DONE);
			aTransitions[PARAMS_DONE]			= ValidTransition("body",		BODY_DONE);
			aTransitions[BODY_DONE]				= ValidTransition("envelope",	EXPECTING_ENVELOPE);
		}
		sal_Char* pAllowedElement = aTransitions[m_eState].first;
		if (pAllowedElement && (0 == _rName.compareToAscii(pAllowedElement)))
		{	// the only allowed element which ends here ...
			m_eState = aTransitions[m_eState].second;

			if (m_bIgnoreEnvelopeTail)
			{
				if (EXPECTING_ENVELOPE == m_eState)
					m_bIgnoreEnvelopeTail = sal_False;	// now the reset is really done
				return;
			}

			switch (m_eState)
			{
				case HEADER_DONE:
					// now that we have all header informations, check if we have a callback					
					m_aCurrentCallback = m_aCallbacks.find(m_aCurrentHeader.sRequestId);					

					// and do some notifications, if we can
					if (m_aCallbacks.end() != m_aCurrentCallback)
					{	// only if we have a callback for the request the header described					
						if (ttAcknowledgment == m_aCurrentHeader.eTransType)
						{
							CFG_TRACE_INFO("redirector: acknowledging the transaction (id: %i) to the callback (client id: %s)", m_aCurrentHeader.nTransId, OUSTRING2ASCII(m_aCurrentHeader.sRequestId));
							// this was a simple acknowledgment that the request was queued
							// m_aCurrentCallback->second.aCallback->acknowledged(m_aCurrentHeader.nTransId);
						}
					}
					break;
				case VALUE_DONE:
					// interpret the collected characters and remember the value
					OSL_ENSURE(m_nCurrentParamNum < m_aCurrentBodyParams.size(),
						"OResponseRedirector::endElement(value) : inconsistent !");
						// this size shoud have been adjusted in startElement(value) !
					switch (m_eCurrentParamType)
					{
						case pdtString:
						case pdtNodepath:
							m_aCurrentBodyParams[m_nCurrentParamNum] <<= m_sCurrentParamValue;
							break;
						case pdtBoolean:
						{
							sal_Bool bInterpreted = (0 != m_sCurrentParamValue.toInt32());
							m_aCurrentBodyParams[m_nCurrentParamNum].setValue(&bInterpreted, getBooleanCppuType());
						}
						break;
						case pdtInteger:
						{
							sal_Int32 nInterpreted = m_sCurrentParamValue.toInt32();
							m_aCurrentBodyParams[m_nCurrentParamNum] <<= nInterpreted;
						}
						break;
					}
					break;
				case DATA_DONE:
					// a single parameter is finished ... check if we have to do some callbacks
					if (m_aCallbacks.end() != m_aCurrentCallback)
					{						
						switch (m_aCurrentHeader.eTransType)
						{
							case ttNotification:
								if (0 == m_nCurrentParamNum)
								{
									OSL_ENSURE(ctNotifyRequest == m_aCurrentCallback->second.eType, "OResponseRedirector::startElement : invalid callback type for notification");
									if (ctNotifyRequest == m_aCurrentCallback->second.eType)
									{
										vos::ORef<INotifyCallback> xNotifyCallBack = static_cast< INotifyCallback* >(m_aCurrentCallback->second.aCallback.getBodyPtr());
										m_xCurrentNodeReader = xNotifyCallBack->getDataReader();
										OSL_ENSURE(m_aCurrentBodyParams.size(), "OResponseRedirector::startElement : had a notification without params !");
										if (m_xCurrentNodeReader.is() && m_aCurrentBodyParams.size())
										{
											::rtl::OUString sNodePath;
											m_aCurrentBodyParams[m_nCurrentParamNum] >>= sNodePath;
                                            xNotifyCallBack->notifyOccured(remote::mapServerPath(sNodePath));										
										}
									}
								}
								break;						
							case ttResponse:
								if (1 == m_nCurrentParamNum)
								{
									if (tnOpenSession == m_aCurrentHeader.eTransName)
									{	// extract the node id
										OSL_ENSURE(pdtString == m_eCurrentParamType,
											"OResponseRedirector::endElement(data) : for this transaction type the second parameter should be an string ('til the server is fixed) !");
										::rtl::OUString sSessionId;
										m_aCurrentBodyParams[m_nCurrentParamNum] >>= sSessionId;
										if (ctObjectOpener == m_aCurrentCallback->second.eType)
										{
											// and we have a callback which is interested in the id
											CFG_TRACE_INFO("redirector: forwarding the session id (%s) to the callback (client id: %s)", OUSTRING2ASCII(sSessionId), OUSTRING2ASCII(m_aCurrentHeader.sRequestId));
											static_cast<IOpenObjectCallback*>(m_aCurrentCallback->second.aCallback.getBodyPtr())->gotObjectId(sSessionId);
										}
									}
								}
								else if (0 == m_nCurrentParamNum)
								{	// it was the first param, which contains - by dtd - the object id
									if	(	(tnOpenNode == m_aCurrentHeader.eTransName)
										||	(tnGetNode == m_aCurrentHeader.eTransName)
										)
									{	// it was one of the operations expecting an object id
										OSL_ENSURE(pdtString == m_eCurrentParamType,
											"OResponseRedirector::endElement(data) : for this transaction type the first parameter should be an integer !");
										::rtl::OUString sObjectId;
										m_aCurrentBodyParams[m_nCurrentParamNum] >>= sObjectId;
										if (ctObjectOpener == m_aCurrentCallback->second.eType)
										{
											// and we have a callback which is interested in the id
											CFG_TRACE_INFO("redirector: forwarding the node id (%s) to the callback (client id: %s)", OUSTRING2ASCII(sObjectId), OUSTRING2ASCII(m_aCurrentHeader.sRequestId));
											static_cast<IOpenObjectCallback*>(m_aCurrentCallback->second.aCallback.getBodyPtr())->gotObjectId(sObjectId);
											
											// store the object id for the callback, this is needed for notifications
											m_aCurrentCallback->second.setNodeId(sObjectId);
										}
									}
								}
								break;
						}
					}
					break;
				case PARAMS_DONE:
					// do all callbacks which could wait 'til the last param
					if (m_aCallbacks.end() != m_aCurrentCallback)
					{
						sal_Bool bFreeCallback = sal_False;
						switch (m_aCurrentHeader.eTransType)
						{
							case ttError:
								// now that we have collected the error infos, forward them to the callback
								if (1 <= m_aCurrentBodyParams.size())
								{	// TODO : need a final specification for the error message format
									StatusInfo aStatus;
									aStatus.nCode = -1;	// generic error, TODO
									m_aCurrentBodyParams[0] >>= aStatus.sMessage;
									CallbackDescription aTest = m_aCurrentCallback->second;

									CFG_TRACE_INFO("redirector: sending the error (%s) to the callback (client id: %s)", OUSTRING2ASCII(aStatus.sMessage), OUSTRING2ASCII(m_aCurrentHeader.sRequestId));
									m_aCurrentCallback->second.aCallback->done(aStatus);
								}
								break;
							case ttResponse:
							{
								StatusInfo aSuccessStatus;
								aSuccessStatus.nCode = 0;
								CFG_TRACE_INFO("redirector: sending the success message to the callback (client id: %s)", OUSTRING2ASCII(m_aCurrentHeader.sRequestId));
								m_aCurrentCallback->second.aCallback->done(aSuccessStatus);
								// TODO : the current spec says that in case of success, no status will be returned.
								// So perhaps our callback interface needs to differ between "done successfully"
								// and "done, but had an error"
							}
							break;
						}
					}
					break;
				case BODY_DONE:
					break;
				case EXPECTING_ENVELOPE:
				{
					// if this was the last request affecting for a specific callback, release this callback					
					if (ttResponse == m_aCurrentHeader.eTransType  || 
						ttError == m_aCurrentHeader.eTransType)
					{
						if (m_aCallbacks.end() != m_aCurrentCallback)
						{							
							// opennode request may contain notification callbacks. These callbacks are used for
							// receiving the notification envelopes
							// if a opennode was successful and we have a callback for notifications
							// we replace the callback for opennode with the callback
							// for following notifications							
							if (ttResponse == m_aCurrentHeader.eTransType &&
								m_aCurrentCallback->second.sNodeId.getLength() != 0 &&
								m_aCurrentCallback->second.aNotifyCallback.getBodyPtr() != NULL)
							{
								m_aCallbacks[m_aCurrentCallback->first] = 
									CallbackDescription(m_aCurrentCallback->second.aNotifyCallback, m_aCurrentCallback->second.sNodeId);
							}
							else							
								m_aCallbacks.erase(m_aCurrentCallback);
							m_aCurrentCallback = m_aCallbacks.end();
						}
					}
				}
				break;
			}
		}
		else
		{
			// TODO : a handling for this case, which is a heavy violation if the dtd.
		}
	}
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::characters( const ::rtl::OUString& _rChars ) throw(sax::SAXException, RuntimeException)
{
	MutexGuard aGuard(m_aMutex);

	switch (m_eState)
	{
		case FORWARDING_DATA:
			if (m_xCurrentNodeReader.is())
				m_xCurrentNodeReader->characters(_rChars);
			// TODO : think about handling the exceptions ....
			break;
		case UNKNOWN_ELEMENT:
			// nothing to do
			break;
		case VALUE_STARTED:
			// just collect the characters, they will be interpreted as soon as the value section ends
			m_sCurrentParamValue += _rChars;
			break;
		default:
#ifdef DEBUG
			{
				::rtl::OUString sChars = _rChars.trim();
				if (0 == sChars.getLength())
					return;
				OSL_ENSURE(sal_False, "OResponseRedirector::characters : we're in a state where no characters should appear !");
			}
#endif
			break;
	}
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::ignorableWhitespace( const ::rtl::OUString& _rWhitespaces ) throw(sax::SAXException, RuntimeException)
{
	MutexGuard aGuard(m_aMutex);
	// as the name suggests : ignore it ...

	// (... not copletely)
	// if we're in state FORWARDING_DATA, forward this
	if ((FORWARDING_DATA == m_eState) && m_xCurrentNodeReader.is())
		m_xCurrentNodeReader->ignorableWhitespace(_rWhitespaces);
	// TODO : think about handling the exceptions ....
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::processingInstruction( const ::rtl::OUString& aTarget, const ::rtl::OUString& aData ) throw(sax::SAXException, RuntimeException)
{
	OSL_ENSURE(sal_False, "OResponseRedirector::processingInstruction : ooops ... should never be called, me thinks ...");
}

//--------------------------------------------------------------------------
void SAL_CALL OResponseRedirector::setDocumentLocator( const Reference< sax::XLocator >& xLocator ) throw(sax::SAXException, RuntimeException)
{
}

//--------------------------------------------------------------------------
void OResponseRedirector::reset()
{
	MutexGuard aGuard(m_aMutex);

	// reset all not state-related members
	m_nElementDepth = 0;
	m_nExpectedParams = 0;
	m_nCurrentParamNum = -1;
	m_eCurrentParamType = pdtUnknown;
	m_aCurrentBodyParams.resize(0);
	m_sCurrentParamValue = ::rtl::OUString();
	m_xCurrentNodeReader = NULL;
	m_aCurrentHeader.eTransType = ttUnknown;
	m_aCurrentHeader.eTransName = tnUnknown;

	m_aCallbacks.clear();
	m_aCurrentCallback = m_aCallbacks.end();

	// if we're within an envelope
	if (EXPECTING_ENVELOPE == m_eState)
	{	// a simple reset ...
		m_eState = m_ePushedState = EXPECTING_ENVELOPE;
	}
	else
	{
		// don't modify the state here, we want to leave the envelope regulary
		m_bIgnoreEnvelopeTail = sal_True;
	}

}

//..........................................................................
}	// namespace configmgr
//..........................................................................


