/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server;

import com.yammer.metrics.core.Gauge;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.message.AssignReplicasToDirsRequestData;
import org.apache.kafka.common.message.AssignReplicasToDirsResponseData;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.AssignReplicasToDirsRequest;
import org.apache.kafka.common.requests.AssignReplicasToDirsResponse;
import org.apache.kafka.common.utils.ExponentialBackoff;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.queue.EventQueue;
import org.apache.kafka.queue.KafkaEventQueue;
import org.apache.kafka.server.ControllerRequestCompletionHandler;
import org.apache.kafka.server.NodeToControllerChannelManager;
import org.apache.kafka.server.common.TopicIdPartition;
import org.apache.kafka.server.metrics.KafkaMetricsGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AssignmentsManager {
    private static final Logger log = LoggerFactory.getLogger(AssignmentsManager.class);
    private static final long DISPATCH_INTERVAL_NS = TimeUnit.MILLISECONDS.toNanos(500L);
    private static final long MAX_BACKOFF_INTERVAL_MS = TimeUnit.SECONDS.toNanos(10L);
    static final String QUEUE_REPLICA_TO_DIR_ASSIGNMENTS_METRIC_NAME = "QueuedReplicaToDirAssignments";
    private final Time time;
    private final NodeToControllerChannelManager channelManager;
    private final int brokerId;
    private final Supplier<Long> brokerEpochSupplier;
    private final KafkaEventQueue eventQueue;
    private final KafkaMetricsGroup metricsGroup = new KafkaMetricsGroup(this.getClass());
    private volatile Map<TopicIdPartition, AssignmentEvent> inflight = null;
    private volatile Map<TopicIdPartition, AssignmentEvent> pending = new HashMap<TopicIdPartition, AssignmentEvent>();
    private final ExponentialBackoff resendExponentialBackoff = new ExponentialBackoff(100L, 2, MAX_BACKOFF_INTERVAL_MS, 0.02);
    private int failedAttempts = 0;

    public AssignmentsManager(Time time, NodeToControllerChannelManager channelManager, int brokerId, Supplier<Long> brokerEpochSupplier) {
        this.time = time;
        this.channelManager = channelManager;
        this.brokerId = brokerId;
        this.brokerEpochSupplier = brokerEpochSupplier;
        this.eventQueue = new KafkaEventQueue(time, new LogContext("[AssignmentsManager id=" + brokerId + "]"), "broker-" + brokerId + "-directory-assignments-manager-", (EventQueue.Event)new ShutdownEvent());
        channelManager.start();
        this.metricsGroup.newGauge(QUEUE_REPLICA_TO_DIR_ASSIGNMENTS_METRIC_NAME, (Gauge)new Gauge<Integer>(){

            public Integer value() {
                return this.getMapSize(AssignmentsManager.this.inflight) + this.getMapSize(AssignmentsManager.this.pending);
            }

            private int getMapSize(Map<TopicIdPartition, AssignmentEvent> map) {
                return map == null ? 0 : map.size();
            }
        });
    }

    public void close() throws InterruptedException {
        try {
            this.eventQueue.close();
        }
        finally {
            this.metricsGroup.removeMetric(QUEUE_REPLICA_TO_DIR_ASSIGNMENTS_METRIC_NAME);
        }
    }

    public void onAssignment(TopicIdPartition topicPartition, Uuid dirId) {
        this.onAssignment(topicPartition, dirId, null);
    }

    public void onAssignment(TopicIdPartition topicPartition, Uuid dirId, Runnable callback) {
        if (callback == null) {
            callback = () -> {};
        }
        this.eventQueue.append((EventQueue.Event)new AssignmentEvent(this.time.nanoseconds(), topicPartition, dirId, callback));
    }

    void wakeup() {
        this.eventQueue.wakeup();
    }

    private void scheduleDispatch() {
        if (this.pending.size() < 2250) {
            this.scheduleDispatch(DISPATCH_INTERVAL_NS);
        } else {
            log.debug("Too many pending assignments, dispatching immediately");
            this.eventQueue.enqueue(EventQueue.EventInsertionType.APPEND, "dispatch-immediate", (Function)new EventQueue.NoDeadlineFunction(), (EventQueue.Event)new DispatchEvent());
        }
    }

    private void scheduleDispatch(long delayNs) {
        log.debug("Scheduling dispatch in {}ns", (Object)delayNs);
        this.eventQueue.enqueue(EventQueue.EventInsertionType.DEFERRED, "dispatch", (Function)new EventQueue.LatestDeadlineFunction(this.time.nanoseconds() + delayNs), (EventQueue.Event)new DispatchEvent());
    }

    private void requeueAllAfterFailure() {
        if (this.inflight != null) {
            log.debug("Re-queueing all in-flight assignments after failure");
            for (AssignmentEvent event : this.inflight.values()) {
                this.pending.put(event.partition, event);
            }
            this.inflight = null;
            ++this.failedAttempts;
            long backoffNs = TimeUnit.MILLISECONDS.toNanos(this.resendExponentialBackoff.backoff((long)this.failedAttempts));
            this.scheduleDispatch(DISPATCH_INTERVAL_NS + backoffNs);
        }
    }

    private static boolean responseIsError(ClientResponse response) {
        if (response == null) {
            log.debug("Response is null");
            return true;
        }
        if (response.authenticationException() != null) {
            log.error("Failed to propagate directory assignments because authentication failed", (Throwable)response.authenticationException());
            return true;
        }
        if (response.versionMismatch() != null) {
            log.error("Failed to propagate directory assignments because the request version is unsupported", (Throwable)response.versionMismatch());
            return true;
        }
        if (response.wasDisconnected()) {
            log.error("Failed to propagate directory assignments because the connection to the controller was disconnected");
            return true;
        }
        if (response.wasTimedOut()) {
            log.error("Failed to propagate directory assignments because the request timed out");
            return true;
        }
        if (response.responseBody() == null) {
            log.error("Failed to propagate directory assignments because the Controller returned an empty response");
            return true;
        }
        if (!(response.responseBody() instanceof AssignReplicasToDirsResponse)) {
            log.error("Failed to propagate directory assignments because the Controller returned an invalid response type");
            return true;
        }
        AssignReplicasToDirsResponseData data = ((AssignReplicasToDirsResponse)response.responseBody()).data();
        Errors error = Errors.forCode((short)data.errorCode());
        if (error != Errors.NONE) {
            log.error("Failed to propagate directory assignments because the Controller returned error {}", (Object)error.name());
            return true;
        }
        return false;
    }

    private static Set<AssignmentEvent> filterFailures(AssignReplicasToDirsResponseData data, Map<TopicIdPartition, AssignmentEvent> sent) {
        HashSet<AssignmentEvent> failures = new HashSet<AssignmentEvent>();
        HashSet<TopicIdPartition> acknowledged = new HashSet<TopicIdPartition>();
        for (AssignReplicasToDirsResponseData.DirectoryData directory : data.directories()) {
            for (AssignReplicasToDirsResponseData.TopicData topic : directory.topics()) {
                for (AssignReplicasToDirsResponseData.PartitionData partition : topic.partitions()) {
                    TopicIdPartition topicPartition = new TopicIdPartition(topic.topicId(), partition.partitionIndex());
                    AssignmentEvent event = sent.get(topicPartition);
                    if (event == null) {
                        log.error("AssignReplicasToDirsResponse contains unexpected partition {} into directory {}", (Object)partition, (Object)directory.id());
                        continue;
                    }
                    acknowledged.add(topicPartition);
                    Errors error = Errors.forCode((short)partition.errorCode());
                    if (error == Errors.NOT_LEADER_OR_FOLLOWER) {
                        log.info("Dropping late directory assignment for partition {} into directory {} because this broker is no longer a replica", (Object)partition, (Object)event.dirId);
                        continue;
                    }
                    if (error == Errors.NONE) continue;
                    log.error("Controller returned error {} for assignment of partition {} into directory {}", new Object[]{error.name(), partition, event.dirId});
                    failures.add(event);
                }
            }
        }
        for (AssignmentEvent event : sent.values()) {
            if (acknowledged.contains(event.partition)) continue;
            log.error("AssignReplicasToDirsResponse is missing assignment of partition {} into directory {}", (Object)event.partition, (Object)event.dirId);
            failures.add(event);
        }
        return failures;
    }

    static AssignReplicasToDirsRequestData buildRequestData(int brokerId, long brokerEpoch, Map<TopicIdPartition, Uuid> assignment) {
        HashMap<Uuid, AssignReplicasToDirsRequestData.DirectoryData> directoryMap = new HashMap<Uuid, AssignReplicasToDirsRequestData.DirectoryData>();
        HashMap<Uuid, Map> topicMap = new HashMap<Uuid, Map>();
        for (Map.Entry<TopicIdPartition, Uuid> entry : assignment.entrySet()) {
            TopicIdPartition topicPartition = entry.getKey();
            Uuid directoryId = entry.getValue();
            AssignReplicasToDirsRequestData.DirectoryData directory = directoryMap.computeIfAbsent(directoryId, d -> new AssignReplicasToDirsRequestData.DirectoryData().setId(directoryId));
            AssignReplicasToDirsRequestData.TopicData topic = topicMap.computeIfAbsent(directoryId, d -> new HashMap()).computeIfAbsent(topicPartition.topicId(), topicId -> {
                AssignReplicasToDirsRequestData.TopicData data = new AssignReplicasToDirsRequestData.TopicData().setTopicId(topicId);
                directory.topics().add(data);
                return data;
            });
            AssignReplicasToDirsRequestData.PartitionData partition = new AssignReplicasToDirsRequestData.PartitionData().setPartitionIndex(topicPartition.partitionId());
            topic.partitions().add(partition);
        }
        return new AssignReplicasToDirsRequestData().setBrokerId(brokerId).setBrokerEpoch(brokerEpoch).setDirectories(new ArrayList(directoryMap.values()));
    }

    private class AssignReplicasToDirsRequestCompletionHandler
    implements ControllerRequestCompletionHandler {
        private AssignReplicasToDirsRequestCompletionHandler() {
        }

        @Override
        public void onTimeout() {
            log.warn("Request to controller timed out");
            this.appendResponseEvent(null);
        }

        public void onComplete(ClientResponse response) {
            log.debug("Received controller response: {}", (Object)response);
            this.appendResponseEvent(response);
        }

        void appendResponseEvent(ClientResponse response) {
            AssignmentsManager.this.eventQueue.prepend((EventQueue.Event)new AssignmentResponseEvent(response));
        }
    }

    private class AssignmentResponseEvent
    extends Event {
        private final ClientResponse response;

        public AssignmentResponseEvent(ClientResponse response) {
            this.response = response;
        }

        public void run() throws Exception {
            if (AssignmentsManager.this.inflight == null) {
                throw new IllegalStateException("Bug. Cannot not be handling a client response if there is are no assignments in flight");
            }
            if (AssignmentsManager.responseIsError(this.response)) {
                AssignmentsManager.this.requeueAllAfterFailure();
            } else {
                AssignmentsManager.this.failedAttempts = 0;
                AssignReplicasToDirsResponseData data = ((AssignReplicasToDirsResponse)this.response.responseBody()).data();
                Set failed = AssignmentsManager.filterFailures(data, AssignmentsManager.this.inflight);
                Set completed = Utils.diff(HashSet::new, AssignmentsManager.this.inflight.values().stream().collect(Collectors.toSet()), (Set)failed);
                for (AssignmentEvent assignmentEvent : completed) {
                    assignmentEvent.onComplete();
                }
                if (!failed.isEmpty()) {
                    log.warn("Re-queueing assignments: {}", (Object)failed);
                    for (AssignmentEvent event : failed) {
                        AssignmentsManager.this.pending.put(event.partition, event);
                    }
                }
                AssignmentsManager.this.inflight = null;
                if (!AssignmentsManager.this.pending.isEmpty()) {
                    AssignmentsManager.this.scheduleDispatch();
                }
            }
        }
    }

    private class DispatchEvent
    extends Event {
        static final String TAG = "dispatch";

        private DispatchEvent() {
        }

        public void run() throws Exception {
            if (AssignmentsManager.this.inflight != null) {
                throw new IllegalStateException("Bug. Should not be dispatching while there are assignments in flight");
            }
            if (AssignmentsManager.this.pending.isEmpty()) {
                log.trace("No pending assignments, no-op dispatch");
                return;
            }
            Collection events = AssignmentsManager.this.pending.values();
            AssignmentsManager.this.pending = new HashMap();
            AssignmentsManager.this.inflight = new HashMap();
            for (AssignmentEvent event : events) {
                if (AssignmentsManager.this.inflight.size() < 2250) {
                    AssignmentsManager.this.inflight.put(event.partition, event);
                    continue;
                }
                AssignmentsManager.this.pending.put(event.partition, event);
            }
            if (!AssignmentsManager.this.pending.isEmpty()) {
                log.warn("Too many assignments ({}) to fit in one call, sending only {} and queueing the rest", (Object)(2250 + AssignmentsManager.this.pending.size()), (Object)2250);
            }
            Map<TopicIdPartition, Uuid> assignment = AssignmentsManager.this.inflight.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> ((AssignmentEvent)e.getValue()).dirId));
            log.debug("Dispatching {} assignments:  {}", (Object)assignment.size(), assignment);
            AssignmentsManager.this.channelManager.sendRequest((AbstractRequest.Builder<? extends AbstractRequest>)new AssignReplicasToDirsRequest.Builder(AssignmentsManager.buildRequestData(AssignmentsManager.this.brokerId, (Long)AssignmentsManager.this.brokerEpochSupplier.get(), assignment)), new AssignReplicasToDirsRequestCompletionHandler());
        }
    }

    private class AssignmentEvent
    extends Event {
        final long timestampNs;
        final TopicIdPartition partition;
        final Uuid dirId;
        final List<Runnable> completionHandlers;

        AssignmentEvent(long timestampNs, TopicIdPartition partition, Uuid dirId, Runnable onComplete) {
            this.timestampNs = timestampNs;
            this.partition = partition;
            this.dirId = dirId;
            this.completionHandlers = new ArrayList<Runnable>();
            if (onComplete != null) {
                this.completionHandlers.add(onComplete);
            }
        }

        void merge(AssignmentEvent other) {
            if (!this.partition.equals((Object)other.partition)) {
                throw new IllegalArgumentException("Cannot merge events for different partitions");
            }
            this.completionHandlers.addAll(other.completionHandlers);
        }

        void onComplete() {
            for (Runnable onComplete : this.completionHandlers) {
                onComplete.run();
            }
        }

        public void run() throws Exception {
            log.trace("Received assignment {}", (Object)this);
            AssignmentEvent existing = AssignmentsManager.this.pending.getOrDefault(this.partition, null);
            boolean existingIsInFlight = false;
            if (existing == null && AssignmentsManager.this.inflight != null) {
                existing = AssignmentsManager.this.inflight.getOrDefault(this.partition, null);
                existingIsInFlight = true;
            }
            if (existing != null) {
                if (existing.dirId.equals((Object)this.dirId)) {
                    existing.merge(this);
                    log.debug("Ignoring duplicate assignment {}", (Object)this);
                    return;
                }
                if (existing.timestampNs > this.timestampNs) {
                    existing.merge(this);
                    log.debug("Dropping assignment {} because it's older than existing {}", (Object)this, (Object)existing);
                    return;
                }
                if (!existingIsInFlight) {
                    this.merge(existing);
                    log.debug("Dropping existing assignment {} because it's older than {}", (Object)existing, (Object)this);
                }
            }
            log.debug("Queueing new assignment {}", (Object)this);
            AssignmentsManager.this.pending.put(this.partition, this);
            if (AssignmentsManager.this.inflight == null || AssignmentsManager.this.inflight.isEmpty()) {
                AssignmentsManager.this.scheduleDispatch();
            }
        }

        public String toString() {
            return "AssignmentEvent{timestampNs=" + this.timestampNs + ", partition=" + this.partition + ", dirId=" + this.dirId + '}';
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AssignmentEvent that = (AssignmentEvent)o;
            return this.timestampNs == that.timestampNs && Objects.equals(this.partition, that.partition) && Objects.equals(this.dirId, that.dirId);
        }

        public int hashCode() {
            return Objects.hash(this.timestampNs, this.partition, this.dirId);
        }
    }

    private class ShutdownEvent
    extends Event {
        private ShutdownEvent() {
        }

        public void run() throws Exception {
            AssignmentsManager.this.channelManager.shutdown();
        }
    }

    private static abstract class Event
    implements EventQueue.Event {
        private Event() {
        }

        public void handleException(Throwable e) {
            log.error("Unexpected error handling {}", (Object)this, (Object)e);
        }
    }
}

