/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.plugin.insights.core.service;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.PriorityQueue;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporter;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporterFactory;
import org.opensearch.plugin.insights.core.exporter.SinkType;
import org.opensearch.plugin.insights.rules.model.MetricType;
import org.opensearch.plugin.insights.rules.model.SearchQueryRecord;
import org.opensearch.plugin.insights.settings.QueryInsightsSettings;
import org.opensearch.threadpool.ThreadPool;

public class TopQueriesService {
    private final Logger logger = LogManager.getLogger();
    private boolean enabled = false;
    private final MetricType metricType;
    private int topNSize;
    private TimeValue windowSize;
    private long windowStart;
    private final PriorityQueue<SearchQueryRecord> topQueriesStore;
    private final AtomicReference<List<SearchQueryRecord>> topQueriesCurrentSnapshot;
    private final AtomicReference<List<SearchQueryRecord>> topQueriesHistorySnapshot;
    private final QueryInsightsExporterFactory queryInsightsExporterFactory;
    private final ThreadPool threadPool;
    private QueryInsightsExporter exporter;

    TopQueriesService(MetricType metricType, ThreadPool threadPool, QueryInsightsExporterFactory queryInsightsExporterFactory) {
        this.metricType = metricType;
        this.threadPool = threadPool;
        this.queryInsightsExporterFactory = queryInsightsExporterFactory;
        this.topNSize = 3;
        this.windowSize = QueryInsightsSettings.DEFAULT_WINDOW_SIZE;
        this.windowStart = -1L;
        this.exporter = null;
        this.topQueriesStore = new PriorityQueue(this.topNSize, (a, b) -> SearchQueryRecord.compare(a, b, metricType));
        this.topQueriesCurrentSnapshot = new AtomicReference(new ArrayList());
        this.topQueriesHistorySnapshot = new AtomicReference(new ArrayList());
    }

    public void setTopNSize(int topNSize) {
        this.topNSize = topNSize;
    }

    public int getTopNSize() {
        return this.topNSize;
    }

    public void validateTopNSize(int size) {
        if (size < 1 || size > 100) {
            throw new IllegalArgumentException("Top N size setting for [" + String.valueOf(this.metricType) + "] should be between 1 and 100, was (" + size + ")");
        }
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public void setWindowSize(TimeValue windowSize) {
        this.windowSize = windowSize;
        this.windowStart = -1L;
    }

    public void validateWindowSize(TimeValue windowSize) {
        if (windowSize.compareTo(QueryInsightsSettings.MAX_WINDOW_SIZE) > 0 || windowSize.compareTo(QueryInsightsSettings.MIN_WINDOW_SIZE) < 0) {
            throw new IllegalArgumentException("Window size setting for [" + String.valueOf(this.metricType) + "] should be between [" + String.valueOf(QueryInsightsSettings.MIN_WINDOW_SIZE) + "," + String.valueOf(QueryInsightsSettings.MAX_WINDOW_SIZE) + "]was (" + String.valueOf(windowSize) + ")");
        }
        if (!QueryInsightsSettings.VALID_WINDOW_SIZES_IN_MINUTES.contains(windowSize) && windowSize.getMinutes() % 60L != 0L) {
            throw new IllegalArgumentException("Window size setting for [" + String.valueOf(this.metricType) + "] should be multiple of 1 hour, or one of " + String.valueOf(QueryInsightsSettings.VALID_WINDOW_SIZES_IN_MINUTES) + ", was (" + String.valueOf(windowSize) + ")");
        }
    }

    public void setExporter(Settings settings) {
        if (settings.get("type") != null) {
            SinkType expectedType = SinkType.parse(settings.get("type", QueryInsightsSettings.DEFAULT_TOP_QUERIES_EXPORTER_TYPE));
            if (this.exporter != null && expectedType == SinkType.getSinkTypeFromExporter(this.exporter)) {
                this.queryInsightsExporterFactory.updateExporter(this.exporter, settings.get("config.index", "'top_queries-'YYYY.MM.dd"));
            } else {
                try {
                    this.queryInsightsExporterFactory.closeExporter(this.exporter);
                }
                catch (IOException e) {
                    this.logger.error("Fail to close the current exporter when updating exporter, error: ", (Throwable)e);
                }
                this.exporter = this.queryInsightsExporterFactory.createExporter(SinkType.parse(settings.get("type", QueryInsightsSettings.DEFAULT_TOP_QUERIES_EXPORTER_TYPE)), settings.get("config.index", "'top_queries-'YYYY.MM.dd"));
            }
        } else {
            try {
                this.queryInsightsExporterFactory.closeExporter(this.exporter);
                this.exporter = null;
            }
            catch (IOException e) {
                this.logger.error("Fail to close the current exporter when disabling exporter, error: ", (Throwable)e);
            }
        }
    }

    public void validateExporterConfig(Settings settings) {
        this.queryInsightsExporterFactory.validateExporterConfig(settings);
    }

    public List<SearchQueryRecord> getTopQueriesRecords(boolean includeLastWindow) throws IllegalArgumentException {
        if (!this.enabled) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Cannot get top n queries for [%s] when it is not enabled.", this.metricType.toString()));
        }
        ArrayList queries = new ArrayList(this.topQueriesCurrentSnapshot.get());
        if (includeLastWindow) {
            queries.addAll(this.topQueriesHistorySnapshot.get());
        }
        return Stream.of(queries).flatMap(Collection::stream).sorted((a, b) -> SearchQueryRecord.compare(a, b, this.metricType) * -1).collect(Collectors.toList());
    }

    void consumeRecords(List<SearchQueryRecord> records) {
        long currentWindowStart = this.calculateWindowStart(System.currentTimeMillis());
        ArrayList<SearchQueryRecord> recordsInLastWindow = new ArrayList<SearchQueryRecord>();
        ArrayList<SearchQueryRecord> recordsInThisWindow = new ArrayList<SearchQueryRecord>();
        for (SearchQueryRecord record : records) {
            if (!record.getMeasurements().containsKey(this.metricType)) continue;
            if (record.getTimestamp() < currentWindowStart) {
                recordsInLastWindow.add(record);
                continue;
            }
            recordsInThisWindow.add(record);
        }
        this.addToTopNStore(recordsInLastWindow);
        this.rotateWindowIfNecessary(currentWindowStart);
        this.addToTopNStore(recordsInThisWindow);
        ArrayList<SearchQueryRecord> newSnapShot = new ArrayList<SearchQueryRecord>(this.topQueriesStore);
        newSnapShot.sort((a, b) -> SearchQueryRecord.compare(a, b, this.metricType));
        this.topQueriesCurrentSnapshot.set(newSnapShot);
    }

    private void addToTopNStore(List<SearchQueryRecord> records) {
        this.topQueriesStore.addAll(records);
        while (this.topQueriesStore.size() > this.topNSize) {
            this.topQueriesStore.poll();
        }
    }

    private void rotateWindowIfNecessary(long newWindowStart) {
        if (this.windowStart < newWindowStart) {
            ArrayList<SearchQueryRecord> history = new ArrayList<SearchQueryRecord>();
            if (this.windowStart == newWindowStart - this.windowSize.getMillis()) {
                history.addAll(this.topQueriesStore);
            }
            this.topQueriesHistorySnapshot.set(history);
            this.topQueriesStore.clear();
            this.topQueriesCurrentSnapshot.set(new ArrayList());
            this.windowStart = newWindowStart;
            if (this.exporter != null) {
                this.threadPool.executor("query_insights_executor").execute(() -> this.exporter.export(history));
            }
        }
    }

    private long calculateWindowStart(long timestamp) {
        LocalDateTime currentTime = LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp), ZoneId.of("UTC"));
        LocalDateTime windowStartTime = currentTime.truncatedTo(ChronoUnit.HOURS);
        while (!windowStartTime.plusMinutes(this.windowSize.getMinutes()).isAfter(currentTime)) {
            windowStartTime = windowStartTime.plusMinutes(this.windowSize.getMinutes());
        }
        return windowStartTime.toInstant(ZoneOffset.UTC).getEpochSecond() * 1000L;
    }

    public List<SearchQueryRecord> getTopQueriesCurrentSnapshot() {
        return this.topQueriesCurrentSnapshot.get();
    }

    public void close() throws IOException {
        this.queryInsightsExporterFactory.closeExporter(this.exporter);
    }
}

