/*
 * Decompiled with CFR 0.152.
 */
package jadx.gui.device.debugger;

import io.github.skylot.jdwp.JDWP;
import io.reactivex.annotations.NonNull;
import jadx.api.plugins.input.data.AccessFlags;
import jadx.gui.device.debugger.DbgUtils;
import jadx.gui.device.debugger.EventListenerAdapter;
import jadx.gui.device.debugger.RuntimeType;
import jadx.gui.device.debugger.SmaliDebuggerException;
import jadx.gui.device.debugger.SuspendInfo;
import jadx.gui.device.debugger.smali.RegisterInfo;
import jadx.gui.utils.IOUtils;
import jadx.gui.utils.ObjectPool;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.atomic.AtomicInteger;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SmaliDebugger {
    private static final Logger LOG = LoggerFactory.getLogger(SmaliDebugger.class);
    private final JDWP jdwp;
    private final int localTcpPort;
    private final InputStream inputStream;
    private final OutputStream outputStream;
    private static final Executor EVENT_LISTENER_QUEUE = Executors.newSingleThreadExecutor();
    private static final Executor SUSPEND_LISTENER_QUEUE = Executors.newSingleThreadExecutor();
    private final Map<Integer, ICommandResult> callbackMap = new ConcurrentHashMap<Integer, ICommandResult>();
    private final Map<Integer, EventListenerAdapter> eventListenerMap = new ConcurrentHashMap<Integer, EventListenerAdapter>();
    private final Map<String, JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData> classMap = new ConcurrentHashMap<String, JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData>();
    private final Map<Long, JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData> classIDMap = new ConcurrentHashMap<Long, JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData>();
    private final Map<Long, List<JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData>> clsMethodMap = new ConcurrentHashMap<Long, List<JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData>>();
    private final Map<Long, List<JDWP.ReferenceType.FieldsWithGeneric.FieldsWithGenericData>> clsFieldMap = new ConcurrentHashMap<Long, List<JDWP.ReferenceType.FieldsWithGeneric.FieldsWithGenericData>>();
    private Map<Long, Map<Long, RuntimeDebugInfo>> varMap = Collections.emptyMap();
    private final JDWP.EventRequest.Set.CountRequest oneOffEventReq;
    private final AtomicInteger idGenerator = new AtomicInteger(1);
    private final SuspendInfo suspendInfo = new SuspendInfo();
    private final SuspendListener suspendListener;
    private ObjectPool<List<JDWP.StackFrame.GetValues.GetValuesSlots>> slotsPool;
    private ObjectPool<List<JDWP.EventRequestEncoder>> stepReqPool;
    private ObjectPool<SynchronousQueue<JDWP.Packet>> syncQueuePool;
    private ObjectPool<List<Long>> fieldIdPool;
    private final Map<Integer, Thread> syncQueueMap = new ConcurrentHashMap<Integer, Thread>();
    private final AtomicInteger syncQueueID = new AtomicInteger(0);
    private static final ICommandResult SKIP_RESULT = res -> {};
    private ClassListenerInfo clsListener;
    private final EventListenerAdapter stepListener = new EventListenerAdapter(){

        @Override
        void onSingleStep(JDWP.Event.Composite.SingleStepEvent event) {
            SmaliDebugger.this.onSuspended(event.thread, event.location.classID, event.location.methodID, event.location.index);
        }
    };

    private SmaliDebugger(SuspendListener suspendListener, int localTcpPort, JDWP jdwp, InputStream inputStream, OutputStream outputStream) {
        this.jdwp = jdwp;
        this.localTcpPort = localTcpPort;
        this.suspendListener = suspendListener;
        this.inputStream = inputStream;
        this.outputStream = outputStream;
        this.oneOffEventReq = jdwp.eventRequest().cmdSet().newCountRequest();
        this.oneOffEventReq.count = 1;
    }

    public static SmaliDebugger attach(String host, int port, SuspendListener suspendListener) throws SmaliDebuggerException {
        try {
            byte[] bytes = JDWP.IDSizes.encode().getBytes();
            JDWP.setPacketID((byte[])bytes, (int)1);
            LOG.debug("Connecting to ADB {}:{}", (Object)host, (Object)port);
            Socket socket = new Socket(host, port);
            InputStream inputStream = socket.getInputStream();
            OutputStream outputStream = socket.getOutputStream();
            socket.setSoTimeout(5000);
            JDWP jdwp = SmaliDebugger.initJDWP(outputStream, inputStream);
            socket.setSoTimeout(0);
            SmaliDebugger debugger = new SmaliDebugger(suspendListener, port, jdwp, inputStream, outputStream);
            debugger.decodingLoop();
            debugger.listenClassUnloadEvent();
            debugger.initPools();
            return debugger;
        }
        catch (IOException e) {
            throw new SmaliDebuggerException("Attach failed", e);
        }
    }

    private void onSuspended(long thread, long clazz, long mth, long offset) {
        this.suspendInfo.update().updateThread(thread).updateClass(clazz).updateMethod(mth).updateOffset(offset);
        if (this.suspendInfo.isAnythingChanged()) {
            SUSPEND_LISTENER_QUEUE.execute(() -> this.suspendListener.onSuspendEvent(this.suspendInfo));
        }
    }

    public void stepInto() throws SmaliDebuggerException {
        this.sendStepRequest(this.suspendInfo.getThreadID(), 0);
    }

    public void stepOver() throws SmaliDebuggerException {
        this.sendStepRequest(this.suspendInfo.getThreadID(), 1);
    }

    public void stepOut() throws SmaliDebuggerException {
        this.sendStepRequest(this.suspendInfo.getThreadID(), 2);
    }

    public void exit() throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.virtualMachine().cmdExit().encode(-1));
        SmaliDebugger.tryThrowError(res);
    }

    public void detach() throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.virtualMachine().cmdDispose().encode());
        SmaliDebugger.tryThrowError(res);
    }

    private void initPools() {
        this.slotsPool = new ObjectPool<List>(() -> {
            ArrayList<JDWP.StackFrame.GetValues.GetValuesSlots> slots = new ArrayList<JDWP.StackFrame.GetValues.GetValuesSlots>(1);
            JDWP.StackFrame.GetValues.GetValuesSlots slot = this.jdwp.stackFrame().cmdGetValues().newValuesSlots();
            slot.slot = 0;
            slot.sigbyte = (byte)76;
            slots.add(slot);
            return slots;
        });
        this.stepReqPool = new ObjectPool<List>(() -> {
            ArrayList<Object> eventEncoders = new ArrayList<Object>(2);
            eventEncoders.add(this.jdwp.eventRequest().cmdSet().newStepRequest());
            eventEncoders.add(this.oneOffEventReq);
            return eventEncoders;
        });
        this.syncQueuePool = new ObjectPool<SynchronousQueue>(SynchronousQueue::new);
        this.fieldIdPool = new ObjectPool<List>(() -> {
            ArrayList<Long> ids = new ArrayList<Long>(1);
            ids.add(-1L);
            return ids;
        });
    }

    public RuntimeRegister getRegisterSync(long threadID, long frameID, int regNum, RuntimeType type) throws SmaliDebuggerException {
        List<JDWP.StackFrame.GetValues.GetValuesSlots> slots = this.slotsPool.get();
        JDWP.StackFrame.GetValues.GetValuesSlots slot = slots.get(0);
        slot.slot = regNum;
        slot.sigbyte = (byte)type.getTag();
        JDWP.Packet res = this.sendCommandSync(this.jdwp.stackFrame().cmdGetValues().encode(threadID, frameID, slots));
        SmaliDebugger.tryThrowError(res);
        this.slotsPool.put(slots);
        JDWP.StackFrame.GetValues.GetValuesReplyData val = this.jdwp.stackFrame().cmdGetValues().decode(res.getBuf(), 11);
        return this.buildRegister(regNum, ((JDWP.StackFrame.GetValues.GetValuesReplyDataValues)val.values.get((int)0)).slotValue.tag, ((JDWP.StackFrame.GetValues.GetValuesReplyDataValues)val.values.get((int)0)).slotValue.idOrValue);
    }

    public long getThisID(long threadID, long frameID) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.stackFrame().cmdThisObject().encode(threadID, frameID));
        SmaliDebugger.tryThrowError(res);
        JDWP.StackFrame.ThisObject.ThisObjectReplyData data = this.jdwp.stackFrame().cmdThisObject().decode(res.getBuf(), 11);
        return data.objectThis.objectID;
    }

    public List<RuntimeField> getAllFieldsSync(long clsID) throws SmaliDebuggerException {
        return this.getAllFields(clsID);
    }

    public void getFieldValueSync(long clsID, RuntimeField fld) throws SmaliDebuggerException {
        ArrayList<RuntimeField> list = new ArrayList<RuntimeField>(1);
        list.add(fld);
        this.getAllFieldValuesSync(clsID, list);
    }

    public void getAllFieldValuesSync(long thisID, List<RuntimeField> flds) throws SmaliDebuggerException {
        ArrayList ids = new ArrayList(flds.size());
        flds.forEach(f -> ids.add(f.getFieldID()));
        JDWP.Packet res = this.sendCommandSync(this.jdwp.objectReference().cmdGetValues().encode(thisID, ids));
        SmaliDebugger.tryThrowError(res);
        JDWP.ObjectReference.GetValues.GetValuesReplyData data = this.jdwp.objectReference().cmdGetValues().decode(res.getBuf(), 11);
        List values = data.values;
        for (int i = 0; i < values.size(); ++i) {
            JDWP.ObjectReference.GetValues.GetValuesReplyDataValues value = (JDWP.ObjectReference.GetValues.GetValuesReplyDataValues)values.get(i);
            flds.get(i).setValue(value.value.idOrValue).setType(RuntimeType.fromJdwpTag(value.value.tag));
        }
    }

    public Frame getCurrentFrame(long threadID) throws SmaliDebuggerException {
        return this.getCurrentFrameInternal(threadID);
    }

    public List<Frame> getFramesSync(long threadID) throws SmaliDebuggerException {
        return this.getAllFrames(threadID);
    }

    public List<Long> getAllThreadsSync() throws SmaliDebuggerException {
        return this.getAllThreads();
    }

    @Nullable
    public String getThreadNameSync(long threadID) throws SmaliDebuggerException {
        return this.sendThreadNameReq(threadID);
    }

    @Nullable
    public String getClassSignatureSync(long classID) throws SmaliDebuggerException {
        return this.getClassSignatureInternal(classID);
    }

    @Nullable
    public String getMethodSignatureSync(long classID, long methodID) throws SmaliDebuggerException {
        return this.getMethodSignatureInternal(classID, methodID);
    }

    public boolean errIsTypeMismatched(int errCode) {
        return errCode == 34;
    }

    public boolean errIsInvalidSlot(int errCode) {
        return errCode == 35;
    }

    public boolean errIsInvalidObject(int errCode) {
        return errCode == 20;
    }

    public void setClassListener(ClassListener listener) throws SmaliDebuggerException {
        if (this.clsListener != null) {
            if (listener != this.clsListener.listener) {
                this.unregisterEventSync(8, this.clsListener.prepareReqID);
                this.unregisterEventSync(9, this.clsListener.unloadReqID);
            }
        } else {
            this.clsListener = new ClassListenerInfo();
        }
        this.clsListener.reset(listener);
        this.regClassPrepareEvent(this.clsListener);
        this.regClassUnloadEvent(this.clsListener);
    }

    private void regClassUnloadEvent(final ClassListenerInfo info) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.eventRequest().cmdSet().newClassExcludeRequest((byte)9, (byte)0, "java.*"));
        SmaliDebugger.tryThrowError(res);
        info.unloadReqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
        this.eventListenerMap.put(info.unloadReqID, new EventListenerAdapter(){

            @Override
            void onClassUnload(JDWP.Event.Composite.ClassUnloadEvent event) {
                info.listener.onUnloaded(DbgUtils.classSigToRawFullName(event.signature));
            }
        });
    }

    private void regClassPrepareEvent(final ClassListenerInfo info) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.eventRequest().cmdSet().newClassExcludeRequest((byte)8, (byte)0, "java.*"));
        SmaliDebugger.tryThrowError(res);
        info.prepareReqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
        this.eventListenerMap.put(info.prepareReqID, new EventListenerAdapter(){

            @Override
            void onClassPrepare(JDWP.Event.Composite.ClassPrepareEvent event) {
                info.listener.onPrepared(DbgUtils.classSigToRawFullName(event.signature), event.typeID);
            }
        });
    }

    public void regClassPrepareEventForBreakpoint(String clsSig, final ClassPrepareListener l) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.buildClassMatchReqForBreakpoint(clsSig, 8));
        SmaliDebugger.tryThrowError(res);
        final int reqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
        this.eventListenerMap.put(reqID, new EventListenerAdapter(){

            @Override
            void onClassPrepare(JDWP.Event.Composite.ClassPrepareEvent event) {
                EVENT_LISTENER_QUEUE.execute(() -> {
                    try {
                        l.onPrepared(event.typeID);
                    }
                    finally {
                        SmaliDebugger.this.eventListenerMap.remove(reqID);
                        try {
                            SmaliDebugger.this.resume();
                        }
                        catch (SmaliDebuggerException e) {
                            LOG.error("Resume failed", (Throwable)e);
                        }
                    }
                });
            }
        });
    }

    public void regMethodEntryEventSync(String clsSig, final MethodEntryListener l) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.eventRequest().cmdSet().newClassMatchRequest((byte)40, (byte)2, clsSig));
        SmaliDebugger.tryThrowError(res);
        final int reqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
        this.eventListenerMap.put(reqID, new EventListenerAdapter(){

            @Override
            void onMethodEntry(JDWP.Event.Composite.MethodEntryEvent event) {
                EVENT_LISTENER_QUEUE.execute(() -> {
                    boolean removeListener = false;
                    try {
                        String sig = SmaliDebugger.this.getMethodSignatureInternal(event.location.classID, event.location.methodID);
                        removeListener = l.entry(sig);
                        if (removeListener) {
                            SmaliDebugger.this.sendCommand(SmaliDebugger.this.jdwp.eventRequest().cmdClear().encode((byte)40, reqID), SKIP_RESULT);
                            SmaliDebugger.this.onSuspended(event.thread, event.location.classID, event.location.methodID, -1L);
                            SmaliDebugger.this.eventListenerMap.remove(reqID);
                        }
                    }
                    catch (SmaliDebuggerException e) {
                        LOG.error("Method entry failed", (Throwable)e);
                    }
                    finally {
                        if (!removeListener) {
                            try {
                                SmaliDebugger.this.resume();
                            }
                            catch (SmaliDebuggerException e) {
                                LOG.error("Resume failed", (Throwable)e);
                            }
                        }
                    }
                });
            }
        });
    }

    private void unregisterEventSync(int eventKind, int reqID) throws SmaliDebuggerException {
        this.eventListenerMap.remove(reqID);
        JDWP.Packet rst = this.sendCommandSync(this.jdwp.eventRequest().cmdClear().encode((byte)eventKind, reqID));
        SmaliDebugger.tryThrowError(rst);
    }

    public String readObjectSignatureSync(RuntimeValue val) throws SmaliDebuggerException {
        long objID = this.readID(val);
        JDWP.Packet res = this.sendCommandSync(this.jdwp.objectReference().cmdReferenceType().encode(objID));
        SmaliDebugger.tryThrowError(res);
        JDWP.ObjectReference.ReferenceType.ReferenceTypeReplyData data = this.jdwp.objectReference().cmdReferenceType().decode(res.getBuf(), 11);
        res = this.sendCommandSync(this.jdwp.referenceType().cmdSignature().encode(data.typeID));
        SmaliDebugger.tryThrowError(res);
        JDWP.ReferenceType.Signature.SignatureReplyData sigData = this.jdwp.referenceType().cmdSignature().decode(res.getBuf(), 11);
        return sigData.signature;
    }

    public String readStringSync(RuntimeValue val) throws SmaliDebuggerException {
        return this.readStringSync(this.readID(val));
    }

    public String readStringSync(long id) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.stringReference().cmdValue().encode(id));
        SmaliDebugger.tryThrowError(res);
        JDWP.StringReference.Value.ValueReplyData strData = this.jdwp.stringReference().cmdValue().decode(res.getBuf(), 11);
        return strData.stringValue;
    }

    public boolean setValueSync(int runtimeRegNum, RuntimeType type, Object val, long threadID, long frameID) throws SmaliDebuggerException {
        if (type == RuntimeType.STRING) {
            long newID = this.createString((String)val);
            if (newID == -1L) {
                return false;
            }
            val = newID;
            type = RuntimeType.OBJECT;
        }
        List<JDWP.StackFrame.SetValues.SlotValueSetter> setters = this.buildRegValueSetter(type.getTag(), runtimeRegNum);
        JDWP.encodeAny((JDWP.ByteBuffer)setters.get((int)0).slotValue.idOrValue, (Object)val);
        JDWP.Packet res = this.sendCommandSync(this.jdwp.stackFrame().cmdSetValues().encode(threadID, frameID, setters));
        SmaliDebugger.tryThrowError(res);
        return this.jdwp.stackFrame().cmdSetValues().decode(res.getBuf(), 11);
    }

    public boolean setValueSync(long objID, long fldID, RuntimeType type, Object val) throws SmaliDebuggerException {
        if (type == RuntimeType.STRING) {
            long newID = this.createString((String)val);
            if (newID == -1L) {
                return false;
            }
            val = newID;
        }
        List<JDWP.ObjectReference.SetValues.FieldValueSetter> setters = this.buildFieldValueSetter();
        JDWP.ObjectReference.SetValues.FieldValueSetter setter = setters.get(0);
        setter.fieldID = fldID;
        JDWP.encodeAny((JDWP.ByteBuffer)setter.value.idOrValue, (Object)val);
        JDWP.Packet res = this.sendCommandSync(this.jdwp.objectReference().cmdSetValues().encode(objID, setters));
        SmaliDebugger.tryThrowError(res);
        return this.jdwp.objectReference().cmdSetValues().decode(res.getBuf(), 11);
    }

    public void getValueSync(long objID, RuntimeField fld) throws SmaliDebuggerException {
        List<Long> ids = this.fieldIdPool.get();
        ids.set(0, fld.getFieldID());
        JDWP.Packet res = this.sendCommandSync(this.jdwp.objectReference().cmdGetValues().encode(objID, ids));
        SmaliDebugger.tryThrowError(res);
        JDWP.ObjectReference.GetValues.GetValuesReplyData data = this.jdwp.objectReference().cmdGetValues().decode(res.getBuf(), 11);
        fld.setValue(((JDWP.ObjectReference.GetValues.GetValuesReplyDataValues)data.values.get((int)0)).value.idOrValue).setType(RuntimeType.fromJdwpTag(((JDWP.ObjectReference.GetValues.GetValuesReplyDataValues)data.values.get((int)0)).value.tag));
    }

    private long createString(String localStr) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.virtualMachine().cmdCreateString().encode(localStr));
        SmaliDebugger.tryThrowError(res);
        JDWP.VirtualMachine.CreateString.CreateStringReplyData id = this.jdwp.virtualMachine().cmdCreateString().decode(res.getBuf(), 11);
        return id.stringObject;
    }

    public long readID(RuntimeValue val) {
        return JDWP.decodeBySize((byte[])val.getRawVal().getBytes(), (int)0, (int)val.getRawVal().size());
    }

    public String readArraySignature(RuntimeValue val) throws SmaliDebuggerException {
        return this.readObjectSignatureSync(val);
    }

    public int readArrayLength(RuntimeValue val) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.arrayReference().cmdLength().encode(this.readID(val)));
        SmaliDebugger.tryThrowError(res);
        JDWP.ArrayReference.Length.LengthReplyData data = this.jdwp.arrayReference().cmdLength().decode(res.getBuf(), 11);
        return data.arrayLength;
    }

    public Map.Entry<Integer, List<Long>> readArray(RuntimeValue reg, int startIndex, int len) throws SmaliDebuggerException {
        AbstractMap.SimpleEntry<Integer, Object> ret;
        JDWP.Packet res;
        long id = this.readID(reg);
        if (len <= 0) {
            res = this.sendCommandSync(this.jdwp.arrayReference().cmdLength().encode(id));
            SmaliDebugger.tryThrowError(res);
            JDWP.ArrayReference.Length.LengthReplyData data = this.jdwp.arrayReference().cmdLength().decode(res.getBuf(), 11);
            len = Math.min(99, data.arrayLength);
            ret = new AbstractMap.SimpleEntry<Integer, Object>(data.arrayLength, null);
        } else {
            ret = new AbstractMap.SimpleEntry<Integer, Object>(0, null);
        }
        startIndex = Math.max(0, startIndex);
        res = this.sendCommandSync(this.jdwp.arrayReference().cmdGetValues().encode(id, startIndex, len));
        SmaliDebugger.tryThrowError(res);
        JDWP.ArrayReference.GetValues.GetValuesReplyData valData = this.jdwp.arrayReference().cmdGetValues().decode(res.getBuf(), 11);
        ret.setValue(valData.values.idOrValues);
        return ret;
    }

    public byte readByte(RuntimeValue val) {
        return JDWP.decodeByte((byte[])val.getRawVal().getBytes(), (int)0);
    }

    public char readChar(RuntimeValue val) {
        return JDWP.decodeChar((byte[])val.getRawVal().getBytes(), (int)0);
    }

    public short readShort(RuntimeValue val) {
        return JDWP.decodeShort((byte[])val.getRawVal().getBytes(), (int)0);
    }

    public int readInt(RuntimeValue val) {
        return JDWP.decodeInt((byte[])val.getRawVal().getBytes(), (int)0);
    }

    public float readFloat(RuntimeValue val) {
        return JDWP.decodeFloat((byte[])val.getRawVal().getBytes(), (int)0);
    }

    public long readAll(RuntimeValue val) {
        return JDWP.decodeBySize((byte[])val.getRawVal().getBytes(), (int)0, (int)Math.min(val.getRawVal().size(), 8));
    }

    public double readDouble(RuntimeValue val) {
        return JDWP.decodeDouble((byte[])val.getRawVal().getBytes(), (int)0);
    }

    @Nullable
    public RuntimeDebugInfo getRuntimeDebugInfo(long clsID, long mthID) throws SmaliDebuggerException {
        Map<Long, RuntimeDebugInfo> secMap = this.varMap.get(clsID);
        RuntimeDebugInfo info = null;
        if (secMap != null) {
            info = secMap.get(mthID);
        }
        if (info == null) {
            info = this.initDebugInfo(clsID, mthID);
        }
        return info;
    }

    private RuntimeDebugInfo initDebugInfo(long clsID, long mthID) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.method().cmdVariableTableWithGeneric.encode(clsID, mthID));
        SmaliDebugger.tryThrowError(res);
        JDWP.Method.VariableTableWithGeneric.VarTableWithGenericData data = this.jdwp.method().cmdVariableTableWithGeneric.decode(res.getBuf(), 11);
        if (this.varMap == Collections.EMPTY_MAP) {
            this.varMap = new ConcurrentHashMap<Long, Map<Long, RuntimeDebugInfo>>();
        }
        RuntimeDebugInfo info = new RuntimeDebugInfo(data);
        this.varMap.computeIfAbsent(clsID, k -> new HashMap()).put(mthID, info);
        return info;
    }

    private static JDWP initJDWP(OutputStream outputStream, InputStream inputStream) throws SmaliDebuggerException {
        try {
            SmaliDebugger.handShake(outputStream, inputStream);
            outputStream.write(JDWP.Suspend.encode().setPacketID(1).getBytes());
            JDWP.Packet res = SmaliDebugger.readPacket(inputStream);
            SmaliDebugger.tryThrowError(res);
            if (res.isReplyPacket() && res.getID() == 1) {
                outputStream.write(JDWP.IDSizes.encode().setPacketID(1).getBytes());
                res = SmaliDebugger.readPacket(inputStream);
                SmaliDebugger.tryThrowError(res);
                if (res.isReplyPacket() && res.getID() == 1) {
                    JDWP.IDSizes.IDSizesReplyData sizes = JDWP.IDSizes.decode((byte[])res.getBuf(), (int)11);
                    return new JDWP(sizes);
                }
            }
        }
        catch (IOException e) {
            throw new SmaliDebuggerException(e);
        }
        throw new SmaliDebuggerException("Failed to init JDWP.");
    }

    private static void handShake(OutputStream outputStream, InputStream inputStream) throws SmaliDebuggerException {
        byte[] buf;
        try {
            outputStream.write(JDWP.encodeHandShakePacket());
            buf = IOUtils.readNBytes(inputStream, 14);
        }
        catch (Exception e) {
            throw new SmaliDebuggerException("jdwp handshake failed", e);
        }
        if (buf == null || !JDWP.decodeHandShakePacket((byte[])buf)) {
            throw new SmaliDebuggerException("jdwp handshake bad reply");
        }
    }

    private JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData getMethodBySig(long classID, String sig) {
        List<JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData> methods = this.clsMethodMap.get(classID);
        if (methods != null) {
            for (JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData method : methods) {
                if (!sig.startsWith(method.name + "(") || !sig.endsWith(method.signature)) continue;
                return method;
            }
        }
        return null;
    }

    private int genID() {
        return this.idGenerator.getAndAdd(1);
    }

    private void decodingLoop() {
        Executors.newSingleThreadExecutor().execute(() -> {
            while (true) {
                boolean errFromCallback = false;
                try {
                    JDWP.Packet res = SmaliDebugger.readPacket(this.inputStream);
                    if (res == null) break;
                    this.suspendInfo.nextRound();
                    ICommandResult callback = this.callbackMap.remove(res.getID());
                    if (callback != null) {
                        if (callback == SKIP_RESULT) continue;
                        errFromCallback = true;
                        callback.onCommandReply(res);
                        continue;
                    }
                    if (res.getCommandSetID() == 64 && res.getCommandID() == 100) {
                        errFromCallback = true;
                        this.decodeCompositeEvents(res);
                        continue;
                    }
                    this.printUnexpectedID(res.getID());
                    continue;
                }
                catch (SmaliDebuggerException e) {
                    LOG.error("Error in debugger decoding loop", (Throwable)e);
                    if (!errFromCallback) break;
                    continue;
                }
                break;
            }
            this.suspendInfo.setTerminated();
            this.clearWaitingSyncQueue();
            this.suspendListener.onSuspendEvent(this.suspendInfo);
        });
    }

    private void sendCommand(JDWP.ByteBuffer buf, ICommandResult callback) throws SmaliDebuggerException {
        int id = this.genID();
        this.callbackMap.put(id, callback);
        try {
            this.outputStream.write(buf.setPacketID(id).getBytes());
        }
        catch (IOException e) {
            throw new SmaliDebuggerException(e);
        }
    }

    private JDWP.Packet sendCommandSync(JDWP.ByteBuffer buf) throws SmaliDebuggerException {
        SynchronousQueue<JDWP.Packet> store = this.syncQueuePool.get();
        this.sendCommand(buf, res -> {
            try {
                store.put(res);
            }
            catch (Exception e) {
                LOG.error("Command send failed", (Throwable)e);
            }
        });
        Integer id = this.syncQueueID.getAndAdd(1);
        try {
            this.syncQueueMap.put(id, Thread.currentThread());
            JDWP.Packet packet = store.take();
            return packet;
        }
        catch (InterruptedException e) {
            throw new SmaliDebuggerException(e);
        }
        finally {
            this.syncQueueMap.remove(id);
            this.syncQueuePool.put(store);
        }
    }

    private void clearWaitingSyncQueue() {
        this.syncQueueMap.keySet().forEach(k -> {
            Thread t = this.syncQueueMap.remove(k);
            if (t != null) {
                t.interrupt();
            }
        });
    }

    private void printUnexpectedID(int id) throws SmaliDebuggerException {
        throw new SmaliDebuggerException("Missing handler for this id: " + id);
    }

    private void decodeCompositeEvents(JDWP.Packet res) throws SmaliDebuggerException {
        JDWP.Event.Composite.EventData data = this.jdwp.event().cmdComposite().decode(res.getBuf(), 11);
        for (JDWP.EventRequestDecoder event : data.events) {
            EventListenerAdapter listener = this.eventListenerMap.get(event.getRequestID());
            if (listener == null) {
                LOG.error("Missing handler for id: {}", (Object)event.getRequestID());
                continue;
            }
            if (event instanceof JDWP.Event.Composite.VMStartEvent) {
                listener.onVMStart((JDWP.Event.Composite.VMStartEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.VMDeathEvent) {
                listener.onVMDeath((JDWP.Event.Composite.VMDeathEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.SingleStepEvent) {
                listener.onSingleStep((JDWP.Event.Composite.SingleStepEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.BreakpointEvent) {
                listener.onBreakpoint((JDWP.Event.Composite.BreakpointEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MethodEntryEvent) {
                listener.onMethodEntry((JDWP.Event.Composite.MethodEntryEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MethodExitEvent) {
                listener.onMethodExit((JDWP.Event.Composite.MethodExitEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MethodExitWithReturnValueEvent) {
                listener.onMethodExitWithReturnValue((JDWP.Event.Composite.MethodExitWithReturnValueEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MonitorContendedEnterEvent) {
                listener.onMonitorContendedEnter((JDWP.Event.Composite.MonitorContendedEnterEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MonitorContendedEnteredEvent) {
                listener.onMonitorContendedEntered((JDWP.Event.Composite.MonitorContendedEnteredEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MonitorWaitEvent) {
                listener.onMonitorWait((JDWP.Event.Composite.MonitorWaitEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.MonitorWaitedEvent) {
                listener.onMonitorWaited((JDWP.Event.Composite.MonitorWaitedEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.ExceptionEvent) {
                listener.onException((JDWP.Event.Composite.ExceptionEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.ThreadStartEvent) {
                listener.onThreadStart((JDWP.Event.Composite.ThreadStartEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.ThreadDeathEvent) {
                listener.onThreadDeath((JDWP.Event.Composite.ThreadDeathEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.ClassPrepareEvent) {
                listener.onClassPrepare((JDWP.Event.Composite.ClassPrepareEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.ClassUnloadEvent) {
                listener.onClassUnload((JDWP.Event.Composite.ClassUnloadEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.FieldAccessEvent) {
                listener.onFieldAccess((JDWP.Event.Composite.FieldAccessEvent)event);
                return;
            }
            if (event instanceof JDWP.Event.Composite.FieldModificationEvent) {
                listener.onFieldModification((JDWP.Event.Composite.FieldModificationEvent)event);
                return;
            }
            throw new SmaliDebuggerException("Unexpected event: " + event);
        }
    }

    private void sendStepRequest(long threadID, int depth) throws SmaliDebuggerException {
        List<JDWP.EventRequestEncoder> stepReq = this.buildStepRequest(threadID, 0, depth);
        JDWP.ByteBuffer stepEncodedBuf = this.jdwp.eventRequest().cmdSet().encode((byte)1, (byte)2, stepReq);
        this.stepReqPool.put(stepReq);
        this.sendCommand(stepEncodedBuf, res -> {
            SmaliDebugger.tryThrowError(res);
            int reqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
            this.eventListenerMap.put(reqID, this.stepListener);
        });
        this.resume();
    }

    public void resume() throws SmaliDebuggerException {
        this.sendCommand(JDWP.Resume.encode(), SKIP_RESULT);
    }

    public void suspend() throws SmaliDebuggerException {
        this.sendCommand(JDWP.Suspend.encode(), SKIP_RESULT);
    }

    public void setBreakpoint(RuntimeBreakpoint bp) throws SmaliDebuggerException {
        this.sendCommand(this.buildBreakpointRequest(bp), res -> {
            SmaliDebugger.tryThrowError(res);
            bp.reqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
            this.eventListenerMap.put(bp.reqID, new EventListenerAdapter(){

                @Override
                void onBreakpoint(JDWP.Event.Composite.BreakpointEvent event) {
                    SmaliDebugger.this.onSuspended(event.thread, event.location.classID, event.location.methodID, event.location.index);
                }
            });
        });
    }

    public long getClassID(String clsSig, boolean fetch) throws SmaliDebuggerException {
        block2: {
            JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData data;
            while ((data = this.classMap.get(clsSig)) == null) {
                if (fetch) {
                    this.getAllClasses();
                    fetch = false;
                    continue;
                }
                break block2;
            }
            return data.typeID;
        }
        return -1L;
    }

    public long getMethodID(long cid, String mthSig) throws SmaliDebuggerException {
        this.initClassCache(cid);
        JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData data = this.getMethodBySig(cid, mthSig);
        if (data != null) {
            return data.methodID;
        }
        return -1L;
    }

    public void initClassCache(long clsID) throws SmaliDebuggerException {
        this.initFields(clsID);
        this.initMethods(clsID);
    }

    public void removeBreakpoint(RuntimeBreakpoint bp) throws SmaliDebuggerException {
        this.sendCommand(this.jdwp.eventRequest().cmdClear().encode((byte)2, bp.reqID), SKIP_RESULT);
    }

    private JDWP.ByteBuffer buildBreakpointRequest(RuntimeBreakpoint bp) {
        JDWP.EventRequest.Set.LocationOnlyRequest req = this.jdwp.eventRequest().cmdSet().newLocationOnlyRequest();
        req.loc.classID = bp.clsID;
        req.loc.methodID = bp.mthID;
        req.loc.index = bp.offset;
        req.loc.tag = 1;
        ArrayList<JDWP.EventRequest.Set.LocationOnlyRequest> list = new ArrayList<JDWP.EventRequest.Set.LocationOnlyRequest>(1);
        list.add(req);
        return this.jdwp.eventRequest().cmdSet().encode((byte)2, (byte)2, list);
    }

    private JDWP.ByteBuffer buildClassMatchReqForBreakpoint(String cls, int eventKind) {
        ArrayList<Object> encoders = new ArrayList<Object>(2);
        JDWP.EventRequest.Set.ClassMatchRequest match = this.jdwp.eventRequest().cmdSet().newClassMatchRequest();
        encoders.add(match);
        encoders.add(this.oneOffEventReq);
        match.classPattern = cls;
        return this.jdwp.eventRequest().cmdSet().encode((byte)eventKind, (byte)2, encoders);
    }

    private List<JDWP.EventRequestEncoder> buildStepRequest(long threadID, int stepSize, int stepDepth) {
        List<JDWP.EventRequestEncoder> eventEncoders = this.stepReqPool.get();
        JDWP.EventRequest.Set.StepRequest req = (JDWP.EventRequest.Set.StepRequest)eventEncoders.get(0);
        req.size = stepSize;
        req.depth = stepDepth;
        req.thread = threadID;
        return eventEncoders;
    }

    private List<JDWP.ObjectReference.SetValues.FieldValueSetter> buildFieldValueSetter() {
        JDWP.ObjectReference.SetValues.FieldValueSetter setter = new JDWP.ObjectReference.SetValues.FieldValueSetter(this.jdwp.objectReference().cmdSetValues());
        setter.value = new JDWP.UntaggedValuePacket(this.jdwp);
        setter.value.idOrValue = new JDWP.ByteBuffer();
        ArrayList<JDWP.ObjectReference.SetValues.FieldValueSetter> setters = new ArrayList<JDWP.ObjectReference.SetValues.FieldValueSetter>(1);
        setters.add(setter);
        return setters;
    }

    private List<JDWP.StackFrame.SetValues.SlotValueSetter> buildRegValueSetter(int tag, int regNum) {
        ArrayList<JDWP.StackFrame.SetValues.SlotValueSetter> setters = new ArrayList<JDWP.StackFrame.SetValues.SlotValueSetter>(1);
        JDWP.StackFrame.SetValues.SlotValueSetter setter = new JDWP.StackFrame.SetValues.SlotValueSetter(this.jdwp.stackFrame().cmdSetValues());
        setters.add(setter);
        setter.slot = regNum;
        setter.slotValue = new JDWP.ValuePacket(this.jdwp);
        setter.slotValue.tag = tag;
        setter.slotValue.idOrValue = new JDWP.ByteBuffer();
        return setters;
    }

    private String getClassSignatureInternal(long id) throws SmaliDebuggerException {
        JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData data = this.classIDMap.get(id);
        if (data == null) {
            this.getAllClasses();
        }
        if ((data = this.classIDMap.get(id)) != null) {
            return data.signature;
        }
        return null;
    }

    private String getMethodSignatureInternal(long clsID, long mthID) throws SmaliDebuggerException {
        List mthData = this.clsMethodMap.get(clsID);
        if (mthData == null) {
            JDWP.Packet res = this.sendCommandSync(this.jdwp.referenceType().cmdMethodsWithGeneric().encode(clsID));
            SmaliDebugger.tryThrowError(res);
            JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData data = this.jdwp.referenceType().cmdMethodsWithGeneric().decode(res.getBuf(), 11);
            mthData = data.declared;
            this.clsMethodMap.put(clsID, mthData);
        }
        if (mthData != null) {
            for (JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericData data : mthData) {
                if (data.methodID != mthID) continue;
                return data.name + data.signature;
            }
        }
        return null;
    }

    private String sendThreadNameReq(long id) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.threadReference().cmdName().encode(id));
        SmaliDebugger.tryThrowError(res);
        JDWP.ThreadReference.Name.NameReplyData nameData = this.jdwp.threadReference().cmdName().decode(res.getBuf(), 11);
        return nameData.threadName;
    }

    private List<RuntimeField> getAllFields(long clsID) throws SmaliDebuggerException {
        this.initFields(clsID);
        List<JDWP.ReferenceType.FieldsWithGeneric.FieldsWithGenericData> flds = this.clsFieldMap.get(clsID);
        if (flds != null && flds.size() > 0) {
            ArrayList<RuntimeField> rfs = new ArrayList<RuntimeField>(flds.size());
            for (JDWP.ReferenceType.FieldsWithGeneric.FieldsWithGenericData fld : flds) {
                String type = fld.signature;
                if (fld.genericSignature != null && !fld.genericSignature.trim().isEmpty()) {
                    type = type + "<" + fld.genericSignature + ">";
                }
                rfs.add(new RuntimeField(fld.name, type, fld.fieldID, fld.modBits));
            }
            return rfs;
        }
        return Collections.emptyList();
    }

    public Frame getCurrentFrameInternal(long threadID) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.threadReference().cmdFrames().encode(threadID, 0, 1));
        SmaliDebugger.tryThrowError(res);
        JDWP.ThreadReference.Frames.FramesReplyData frameData = this.jdwp.threadReference().cmdFrames().decode(res.getBuf(), 11);
        JDWP.ThreadReference.Frames.FramesReplyDataFrames frame = (JDWP.ThreadReference.Frames.FramesReplyDataFrames)frameData.frames.get(0);
        return new Frame(frame.frameID, frame.location.classID, frame.location.methodID, frame.location.index);
    }

    private List<Frame> getAllFrames(long threadID) throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.threadReference().cmdFrames().encode(threadID, 0, -1));
        SmaliDebugger.tryThrowError(res);
        JDWP.ThreadReference.Frames.FramesReplyData frameData = this.jdwp.threadReference().cmdFrames().decode(res.getBuf(), 11);
        ArrayList<Frame> frames = new ArrayList<Frame>();
        for (JDWP.ThreadReference.Frames.FramesReplyDataFrames frame : frameData.frames) {
            frames.add(new Frame(frame.frameID, frame.location.classID, frame.location.methodID, frame.location.index));
        }
        return frames;
    }

    private List<Long> getAllThreads() throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.virtualMachine().cmdAllThreads().encode());
        SmaliDebugger.tryThrowError(res);
        JDWP.VirtualMachine.AllThreads.AllThreadsReplyData data = this.jdwp.virtualMachine().cmdAllThreads().decode(res.getBuf(), 11);
        ArrayList<Long> threads = new ArrayList<Long>(data.threads.size());
        for (JDWP.VirtualMachine.AllThreads.AllThreadsReplyDataThreads thread : data.threads) {
            threads.add(thread.thread);
        }
        return threads;
    }

    private void getAllClasses() throws SmaliDebuggerException {
        JDWP.Packet res = this.sendCommandSync(this.jdwp.virtualMachine().cmdAllClassesWithGeneric().encode());
        SmaliDebugger.tryThrowError(res);
        JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericReplyData classData = this.jdwp.virtualMachine().cmdAllClassesWithGeneric().decode(res.getBuf(), 11);
        for (JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData aClass : classData.classes) {
            this.classMap.put(DbgUtils.classSigToRawFullName(aClass.signature), aClass);
            this.classIDMap.put(aClass.typeID, aClass);
        }
    }

    private void initFields(long clsID) throws SmaliDebuggerException {
        if (this.clsFieldMap.get(clsID) == null) {
            JDWP.Packet res = this.sendCommandSync(this.jdwp.referenceType().cmdFieldsWithGeneric().encode(clsID));
            SmaliDebugger.tryThrowError(res);
            JDWP.ReferenceType.FieldsWithGeneric.FieldsWithGenericReplyData data = this.jdwp.referenceType().cmdFieldsWithGeneric().decode(res.getBuf(), 11);
            this.clsFieldMap.put(clsID, data.declared);
        }
    }

    private void initMethods(long clsID) throws SmaliDebuggerException {
        if (this.clsMethodMap.get(clsID) == null) {
            JDWP.Packet res = this.sendCommandSync(this.jdwp.referenceType().cmdMethodsWithGeneric().encode(clsID));
            SmaliDebugger.tryThrowError(res);
            JDWP.ReferenceType.MethodsWithGeneric.MethodsWithGenericReplyData data = this.jdwp.referenceType().cmdMethodsWithGeneric().decode(res.getBuf(), 11);
            this.clsMethodMap.put(clsID, data.declared);
        }
    }

    private void listenClassUnloadEvent() throws SmaliDebuggerException {
        this.sendCommand(this.jdwp.eventRequest().cmdSet().encode((byte)9, (byte)0, Collections.emptyList()), res -> {
            int reqID = this.jdwp.eventRequest().cmdSet().decodeRequestID(res.getBuf(), 11);
            this.eventListenerMap.put(reqID, new EventListenerAdapter(){

                @Override
                void onClassUnload(JDWP.Event.Composite.ClassUnloadEvent event) {
                    EVENT_LISTENER_QUEUE.execute(() -> {
                        System.out.printf("ClassUnloaded: %s%n", event.signature);
                        JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData clsData = (JDWP.VirtualMachine.AllClassesWithGeneric.AllClassesWithGenericData)SmaliDebugger.this.classMap.remove(event.signature);
                        if (clsData != null) {
                            SmaliDebugger.this.classIDMap.remove(clsData.typeID);
                            SmaliDebugger.this.clsFieldMap.remove(clsData.typeID);
                            SmaliDebugger.this.clsMethodMap.remove(clsData.typeID);
                            SmaliDebugger.this.varMap.remove(clsData.typeID);
                        }
                    });
                }
            });
        });
    }

    @Nullable
    private static JDWP.Packet readPacket(InputStream inputStream) throws SmaliDebuggerException {
        try {
            byte[] header = IOUtils.readNBytes(inputStream, 11);
            if (header == null) {
                return null;
            }
            int bodyLength = JDWP.getPacketLength((byte[])header, (int)0) - 11;
            if (bodyLength <= 0) {
                return JDWP.Packet.make((byte[])header);
            }
            byte[] body = IOUtils.readNBytes(inputStream, bodyLength);
            if (body == null) {
                throw new SmaliDebuggerException("Stream truncated");
            }
            return JDWP.Packet.make((byte[])SmaliDebugger.concatBytes(header, body));
        }
        catch (IOException e) {
            throw new SmaliDebuggerException("Read packer error", e);
        }
    }

    private static byte[] concatBytes(byte[] buf1, byte[] buf2) {
        byte[] tempBuf = new byte[buf1.length + buf2.length];
        System.arraycopy(buf1, 0, tempBuf, 0, buf1.length);
        System.arraycopy(buf2, 0, tempBuf, buf1.length, buf2.length);
        return tempBuf;
    }

    private static void tryThrowError(@Nullable JDWP.Packet res) throws SmaliDebuggerException {
        if (res == null) {
            throw new SmaliDebuggerException("Stream ended");
        }
        if (res.isError()) {
            throw new SmaliDebuggerException("(JDWP Error Code:" + res.getErrorCode() + ") " + res.getErrorText(), res.getErrorCode());
        }
    }

    public RuntimeBreakpoint makeBreakpoint(long cid, long mid, long offset) {
        RuntimeBreakpoint bp = new RuntimeBreakpoint();
        bp.clsID = cid;
        bp.mthID = mid;
        bp.offset = offset;
        return bp;
    }

    private RuntimeRegister buildRegister(int num, int tag, JDWP.ByteBuffer buf) throws SmaliDebuggerException {
        return new RuntimeRegister(num, RuntimeType.fromJdwpTag(tag), buf);
    }

    public static interface SuspendListener {
        public void onSuspendEvent(SuspendInfo var1);
    }

    public static interface ClassListener {
        public void onPrepared(String var1, long var2);

        public void onUnloaded(String var1);
    }

    public static interface ClassPrepareListener {
        public void onPrepared(long var1);
    }

    public static class Frame {
        private final long id;
        private final long clsID;
        private final long mthID;
        private final long index;

        private Frame(long id, long clsID, long mthID, long index) {
            this.id = id;
            this.clsID = clsID;
            this.mthID = mthID;
            this.index = index;
        }

        public long getID() {
            return this.id;
        }

        public long getClassID() {
            return this.clsID;
        }

        public long getMethodID() {
            return this.mthID;
        }

        public long getCodeIndex() {
            return this.index;
        }
    }

    public static class RuntimeDebugInfo {
        private final List<RuntimeVarInfo> infoList;

        private RuntimeDebugInfo(JDWP.Method.VariableTableWithGeneric.VarTableWithGenericData data) {
            this.infoList = new ArrayList<RuntimeVarInfo>(data.slots.size());
            for (JDWP.Method.VariableTableWithGeneric.VarWithGenericSlot slot : data.slots) {
                this.infoList.add(new RuntimeVarInfo(slot));
            }
        }

        public List<RuntimeVarInfo> getInfoList() {
            return this.infoList;
        }
    }

    public static class RuntimeVarInfo
    extends RegisterInfo {
        private final JDWP.Method.VariableTableWithGeneric.VarWithGenericSlot slot;

        private RuntimeVarInfo(JDWP.Method.VariableTableWithGeneric.VarWithGenericSlot slot) {
            this.slot = slot;
        }

        public String getName() {
            return this.slot.name;
        }

        public int getRegNum() {
            return this.slot.slot;
        }

        public String getType() {
            String gen = this.getSignature();
            if (gen == null || gen.isEmpty()) {
                return this.slot.signature;
            }
            return gen;
        }

        @NonNull
        public String getSignature() {
            return this.slot.genericSignature.trim();
        }

        public int getStartOffset() {
            return (int)this.slot.codeIndex;
        }

        public int getEndOffset() {
            return (int)(this.slot.codeIndex + (long)this.slot.length);
        }

        public boolean isMarkedAsParameter() {
            return false;
        }
    }

    public static class RuntimeRegister
    extends RuntimeValue {
        private final int num;

        private RuntimeRegister(int num, RuntimeType type, JDWP.ByteBuffer rawVal) {
            super(type, rawVal);
            this.num = num;
        }

        public int getRegNum() {
            return this.num;
        }
    }

    public static class RuntimeValue {
        protected JDWP.ByteBuffer rawVal;
        protected RuntimeType type;

        RuntimeValue(RuntimeType type, JDWP.ByteBuffer rawVal) {
            this.rawVal = rawVal;
            this.type = type;
        }

        public RuntimeType getType() {
            return this.type;
        }

        public void setType(RuntimeType type) {
            this.type = type;
        }

        private JDWP.ByteBuffer getRawVal() {
            return this.rawVal;
        }
    }

    public static class RuntimeBreakpoint {
        private long clsID;
        private long mthID;
        private long offset;
        private int reqID;

        public long getCodeOffset() {
            return this.offset;
        }
    }

    public static class RuntimeField
    extends RuntimeValue {
        private final String name;
        private final String fldType;
        private final long fieldID;
        private final int modBits;

        private RuntimeField(String name, String type, long fieldID, int modBits) {
            super(null, null);
            this.name = name;
            this.fldType = type;
            this.fieldID = fieldID;
            this.modBits = modBits;
        }

        public String getFieldType() {
            return this.fldType;
        }

        public String getName() {
            return this.name;
        }

        public long getFieldID() {
            return this.fieldID;
        }

        private RuntimeField setValue(JDWP.ByteBuffer rawVal) {
            this.rawVal = rawVal;
            return this;
        }

        public boolean isBelongToThis() {
            return !AccessFlags.hasFlag((int)this.modBits, (int)8) && !AccessFlags.hasFlag((int)this.modBits, (int)4096);
        }
    }

    private static interface ICommandResult {
        public void onCommandReply(JDWP.Packet var1) throws SmaliDebuggerException;
    }

    public static interface MethodEntryListener {
        public boolean entry(String var1);
    }

    private static class ClassListenerInfo {
        int prepareReqID;
        int unloadReqID;
        ClassListener listener;

        private ClassListenerInfo() {
        }

        void reset(ClassListener l) {
            this.listener = l;
            this.prepareReqID = -1;
            this.unloadReqID = -1;
        }
    }
}

