/******************************** -*- C -*- ****************************
 *
 *	Object Table declarations.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/



#ifndef GST_OOP_H
#define GST_OOP_H

#define NUM_CHAR_OBJECTS	256
#define NUM_BUILTIN_OBJECTS	3
#define FIRST_OOP_INDEX		(-NUM_CHAR_OBJECTS-NUM_BUILTIN_OBJECTS)
#define CHAR_OBJECT_BASE	FIRST_OOP_INDEX
#define BUILTIN_OBJECT_BASE	(-NUM_BUILTIN_OBJECTS)

/* The number of OOPs in the system.  This is exclusive of Character,
   True, False, and UndefinedObject (nil) oops, which are
   built-ins. */
#define INITIAL_OOP_TABLE_SIZE	(1024 * 128 + BUILTIN_OBJECT_BASE)
#define MAX_OOP_TABLE_SIZE	(sizeof(struct OOP) << 20)
#define MAX_OBJECT_DATA_SIZE	(sizeof(long) << 26)

/* The number of free OOPs under which we trigger GCs.  0 is not
   enough because _gst_scavenge might still need some oops in
   empty_context_stack!!! */
#define LOW_WATER_OOP_THRESHOLD	(1024 * 2)

#define smalltalkOOPIndex	0
#define processorOOPIndex	1
#define symbolTableOOPIndex	2
#define nilOOPIndex		(BUILTIN_OBJECT_BASE + 0)
#define trueOOPIndex		(BUILTIN_OBJECT_BASE + 1)
#define falseOOPIndex		(BUILTIN_OBJECT_BASE + 2)

/* Given a number of bytes "x", return the number of 32 bit words
   needed to represent that object, rounded up to the nearest 32 bit
   word boundary. */
#define ROUNDED_WORDS(x) \
  (((x) + sizeof(long) - 1) / sizeof(long))

/* Given a number of bytes "x", round it up to the next multiple of
   sizeof (long). */
#define ROUNDED_BYTES(x) \
  (((x) + sizeof(long) - 1) & ~(sizeof(long) - 1))

typedef struct gst_char_object
{
  OBJ_HEADER;
#if defined(WORDS_BIGENDIAN)
  gst_uchar dummy[3];		/* filler */
  gst_uchar charVal;
#else
  gst_uchar charVal;
  gst_uchar dummy[3];		/* filler */
#endif
}
gst_char_object;

struct gst_nil_object
{
  OBJ_HEADER;
};

struct gst_boolean_object
{
  OBJ_HEADER;
  OOP booleanValue;
};

struct memory_space
{
  char *space;			/* base of allocated storage */
  char *allocPtr;		/* new space ptr, starts low, goes up */
  char *maxPtr;			/* points to highest addr in heap */
  char *newAllocPtr;		/* new space ptr, starts low, goes up */
  unsigned long totalSize;	/* current allocated size */
};

/* This is the number of OOPs in the free list and in the full OOP
   table.  _gst_num_free_oops is only correct after a GC!*/
extern int _gst_num_free_oops, _gst_oop_table_size;

/* This vector holds the storage for all the Character objects in the
   system.  Since all character objects are unique, we pre-allocate
   space for 256 of them, and treat them as special built-ins when
   doing garbage collection.*/
extern gst_char_object _gst_char_object_table[];

/* This is "nil" object in the system.  That is, the single instance
   of the UndefinedObject class, which is called "nil". */
extern struct gst_nil_object _gst_nil_object;

/* These represent the two boolean objects in the system, true and
   false.  This is the object storage for those two objects.  
   false == &_gst_boolean_objects[0],
   true == &_gst_boolean_objects[1] */
extern struct gst_boolean_object _gst_boolean_objects[];

/* This is true to show a message whenever a GC happens. */
extern mst_Boolean _gst_gc_message;

/* This is true in the middle of a GC. */
extern mst_Boolean _gst_gc_running;

/* Objects that are at least this big (in bytes) are allocated outside
   the main heap, hoping to provide more locality of reference between
   small objects. */
extern int _gst_big_object_threshold;

/* If there is this much space used after a _gst_scavenge, we need to
   grow the object heap by _gst_space_grow_rate % next time we
   _gst_scavenge, so that the storage gets copied into the new, larger
   area. */
extern int _gst_grow_threshold_percent;

/* Grow the object heap by this percentage when the amount of space
   used exceeds _gst_grow_threshold_percent. */
extern int _gst_space_grow_rate;

/* Finds and returns an instance of the class CLASS_OOP.  Returns "nil"
   if there are no instances present. */
extern OOP _gst_find_an_instance (OOP class_oop) FN_PURE;

/* Execute a two-way become operation between OOP1 and OOP2. */
extern void _gst_swap_objects (OOP oop1,
		     OOP oop2);

/* Called after the fundamental class hierarchy has been defined, this
   function goes through and fixes up all the objects in the oop table
   that don't have a objClass (objClass == _gst_nil_oop).  It's a
   chicken-and-egg problem: the _gst_metaclass_class doesn't yet exist
   when the hierarchy is put together, so after it's created, we have
   to go back and fix all the metaclasses that we created. */
extern void _gst_fixup_metaclass_objects (void);

/* Mark & sweep the objects.  Starting from the root set, recursively
   mark objects as reachable, then compact memory, thus eliminating
   objects that are not reachable (sweeping them).  Then, check if the
   space has to be grown and, if so, readjust the collector's
   parameters. */
extern void _gst_scavenge (void);

/* Mark & sweep the new objects.  Starting from the root set,
   recursively mark objects as reachable, then include the actually
   reachable ones to the old generation. */
extern void _gst_minor_scavenge (void);

/* Set up so that the allocation starts SIZE bytes past the bottom of
   the object heap. */
extern void _gst_set_space_info (long int size);

/* Initialize the memory allocator.  The memory space is allocated,
   and the various garbage collection flags are set to their initial
   values. */
extern void _gst_init_mem (void);

/* Initialize an OOP table of SIZE bytes.  Initially, all the OOPs are
   on the OOP free list so that's just how we initialize them.  We do
   as much initialization as we can, but we're called before classses
   are defined, so things that have definite classes must wait until
   the classes are defined. */
extern void _gst_init_ooptable (long int size);

/* The transitive marker.  This function works in two ways: a) when
   OOP is NULL, it walks the list of pointers to OOPs at CUROOP
   (included) and finishing at ATENDOOP (excluded).  Each object in
   the list is then marked.  b) when oop is not NULL, it tail recurses
   telling itself to mark the pointers referred to by the object
   pointed to by oop. Note that a single function does both these
   jobs to allow a fast, tail-recursive implementation of
   single-object marking. 

   This function does not mark weak objects, for obvious reasons. */
extern void _gst_mark_an_oop_internal (register OOP oop,
			     register OOP * curOOP,
			     OOP * atEndOOP);

/* Fully initialize the builtin objects, possible after the respective
   classes have been created. */
extern void _gst_init_builtin_objects_classes (void);

/* Iterate through the OOP table.  Rebuild the free OOP list. */
extern void _gst_refresh_oop_free_list (void);

/* Create the registry of incubated objects. */
extern void _gst_inc_init_registry (void);

/* Grow the registry of incubated objects when it is full. */
extern void _gst_inc_grow_registry (void);

/* Allocate and return space for an object of SIZE bytes.  This
   basically means moving the allocation pointer for the current space
   up by SIZE bytes, and, if there isn't enough space left, flipping
   the garbage collector after memory is compacted.  The space is
   merely allocated; it is not initialized. 

   The pointer to the object data is returned, the OOP is
   stored in P_OOP. */
extern mst_Object _gst_alloc_obj (long unsigned int size,
				  OOP *p_oop);

/* Allocate and return space for an object of SIZE words, without
   creating an OOP.  This is a special operation that is only needed
   at bootstrap time, so it does not care about garbage collection. */
extern mst_Object _gst_alloc_words (long size);

extern mst_Object _gst_cur_space_addr (void) FN_PURE;

/* Checks to see if INDEX (a long index into the OOP table, 1 based
   due to being called from Smalltalk via a primitive) represents a
   valid OOP.  Returns true if so.  */
extern mst_Boolean _gst_oop_index_valid (long int index);

/* Grows the allocated memory to SPACESIZE bytes, if it's not there
   already.  Returns true if the operation was successful, false if
   the memory could not be allocated. */
extern mst_Boolean _gst_grow_to (long unsigned int spaceSize);

/* Grows the allocated memory to SPACESIZE bytes, if it's not there
   already.  Returns true if the operation was successful, false if
   the memory could not be allocated.  Should be called after the
   sweep has occurred so that things are contiguous.  Ensures that the
   OOP table pointers are fixed up to point to the new objects. */
extern mst_Boolean _gst_grow_memory_to (long unsigned int spaceSize);

/* Grow the OOP table to NEWSIZE pointers and initialize the newly
   created pointers. */
extern mst_Boolean _gst_realloc_ooptable (long int newSize);

/* Move OOP, which must be the last created object, to FixedSpace. */
extern mst_Boolean _gst_make_fixed_oop (OOP oop);

/* The OOP table.  This contains a pointer to the object, and some flag
   bits indicating whether the object is read-only, reachable and/or pooled.
   Some of the bits indicate the difference between the allocated length
   (stored in the object itself), and the real length, because variable
   byte objects may not be an even multiple of sizeof(PTR). */
extern struct OOP *_gst_oop_table, *_gst_all_oops_table;

/* These are the pointer to the first free OOP (they constitute a linked
   list whose "next" pointer is stored in the OBJECT field of the OOP)
   and a pointer to the last used OOP; OOPs following this are not 
   considered. */
extern OOP _gst_first_free_oop, _gst_last_used_oop;

/* This variable represents information about the memory space.  _gst_mem_space
   holds the required information: basically the pointer to the base and
   top of the space, and the pointers into it for allocation and copying.  */
extern struct memory_space _gst_mem_space;

#endif /* GST_OOP_H */
