"======================================================================
|
|   Smalltalk proxy class loader
|
|
 ======================================================================"


"======================================================================
|
| Copyright 2001, 2002 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

STParser subclass: #STClassLoader
       instanceVariableNames: 'loadedClasses proxies currentClass
			       currentCategory currentNamespace'
       classVariableNames: 'EvaluationMethods'
       poolDictionaries: 'STClassLoaderObjects'
       category: 'System-Compiler'
!

STClassLoader comment:
'This class creates non-executable proxies for the classes it loads in.
It does not work if classes are created dynamically, but otherwise it
does it job well.'!

!STClassLoader class methodsFor: 'accessing'!

initialize
    (EvaluationMethods := IdentityDictionary new)
	at: #subclass:instanceVariableNames:classVariableNames:poolDictionaries:category:
	put: #doSubclass:selector:arguments:;

	at: #variableSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category:
	put: #doSubclass:selector:arguments:;

	at: #variableWordSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category:
	put: #doSubclass:selector:arguments:;

	at: #variableByteSubclass:instanceVariableNames:classVariableNames:poolDictionaries:category:
	put: #doSubclass:selector:arguments:;

	at: #methodsFor:
	put: #doMethodsFor:selector:arguments:;

	at: #addSubspace:
	put: #doAddNamespace:selector:arguments:;

	at: #current:
	put: #doSetNamespace:selector:arguments:;

	at: #comment:
	put: #doComment:selector:arguments:;

	at: #instanceVariableNames:
	put: #doClassInstVars:selector:arguments:
! !

Namespace current: STInST!

!STClassLoader methodsFor: 'accessing'!

currentNamespace
    ^currentNamespace!

currentNamespace: ns
    currentNamespace := ns!

proxyForNamespace: anObject
    ^proxies at: anObject ifAbsentPut: [
	ProxyNamespace on: anObject for: self ]!

proxyForClass: anObject
    ^proxies at: anObject ifAbsentPut: [
	ProxyClass on: anObject for: self ]! !

!STClassLoader methodsFor: 'overrides'!

init: aStream
    super init: aStream.
    loadedClasses := OrderedCollection new.
    proxies := IdentityDictionary new.
    currentNamespace := self proxyForNamespace: Namespace current.
!

result
    "This is what #parseSmalltalk answers"
    ^loadedClasses
!

endMethodList
    currentClass := nil
!

compile: node
    currentClass methodDictionary
	at: node selector selector asSymbol
	put: (LoadedMethod
	    category: currentCategory
	    source: node source)
!

evaluate: node
    node body statements do: [ :each |
        each class = STExpressionNode ifTrue: [
            each expression class = STMessageNode ifTrue: [
		self evaluateStatement: each expression ] ]
    ].
    ^currentClass notNil
! !

!STClassLoader methodsFor: 'evaluating statements'!

evaluateStatement: node
    | receiver selector expressions |
    node message argumentCount = 0 ifTrue: [ ^self ].

    receiver := node receiver.
    selector := node message selector.
    expressions := node message expressions.

    EvaluationMethods at: selector ifPresent: [ :method |
	self
	    perform: method
	    with: receiver
	    with: selector
	    with: expressions ].
!

doSubclass: receiver selector: selector arguments: expressions
    | class arguments newClass |
    (expressions allSatisfy: [ :each | each class = STConstNode ])
	ifFalse: [ ^self ].

    class := self resolveClass: receiver.
    arguments := expressions collect: [ :each | each value ].
    newClass := class perform: selector withArguments: arguments asArray.
    loadedClasses add: newClass.
    proxies at: newClass put: newClass
!

doComment: receiver selector: selector arguments: expressions
    | class |
    (expressions allSatisfy: [ :each | each class = STConstNode ])
	ifFalse: [ ^self ].

    class := self resolveClass: receiver.
    class comment: expressions first value.
!

doClassInstVars: receiver selector: selector arguments: expressions
    | class |
    (expressions allSatisfy: [ :each | each class = STConstNode ])
	ifFalse: [ ^self ].

    receiver class = STMessageNode ifFalse: [ ^self ].
    receiver message selector = #class ifFalse: [ ^self ].

    class := self resolveClass: receiver.
    class instanceVariableNames: expressions first value.
!

doSetNamespace: receiver selector: selector arguments: expressions
    | ns |
    receiver class = STVariableNode ifFalse: [ ^self ].
    receiver id = 'Namespace' ifFalse: [ ^self ].

    ns := self resolveNamespace: expressions first.
    self currentNamespace: ns
!

doAddNamespace: receiver selector: selector arguments: expressions
    | root |
    (expressions allSatisfy: [ :each | each class = STConstNode ])
	ifFalse: [ ^self ].
    root := self resolveNamespace: receiver.
    root addSubspace: expressions first value.
!

doMethodsFor: receiver selector: selector arguments: expressions
    | class |
    (expressions allSatisfy: [ :each | each class = STConstNode ])
	ifFalse: [ ^self ].

    currentClass := self resolveClass: receiver.
    currentCategory := expressions first value
!

resolveClass: node
    | object |
    (node class = STMessageNode and: [ node message selector = #class ])
	ifTrue: [ ^(self resolveClass: node receiver) asMetaclass ].

    object := self resolveName: node.
    object isClass ifFalse: [ ^object ].

    ^self proxyForClass: object
!

resolveNamespace: node
    | object |
    object := self resolveName: node.
    object isNamespace ifFalse: [ ^object ].

    ^self proxyForNamespace: object
!

resolveName: node
    | current selectors |
    node class = STVariableNode
	ifTrue: [ ^self currentNamespace at: node id asSymbol ].

    current := node.
    selectors := OrderedCollection new.
    [ current class = STMessageNode ] whileTrue: [
	selectors addFirst: current message selector.
	current := current receiver
    ].
    selectors addFirst: current id asSymbol.

    ^selectors
	inject: self currentNamespace
	into: [ :current :each | current at: each ]
! !

STClassLoader initialize!
