/* grecs - Gray's Extensible Configuration System
   Copyright (C) 2007, 2008, 2009 Sergey Poznyakoff

   Grecs is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3 of the License, or (at your
   option) any later version.

   Grecs is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with Grecs. If not, see <http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdbool.h>
#include <gl_linked_list.h>

typedef struct {
    char *file;
    int line;
} grecs_locus_t;

extern grecs_locus_t grecs_locus;

enum grecs_data_type {
    grecs_type_void,
    grecs_type_string,
    grecs_type_short,
    grecs_type_ushort,
    grecs_type_int,
    grecs_type_uint,
    grecs_type_long,
    grecs_type_ulong,
    grecs_type_size,
/*    grecs_type_off,*/
    grecs_type_uintmax,
    grecs_type_intmax,
    grecs_type_time,
    grecs_type_bool,
    grecs_type_ipv4,
    grecs_type_cidr,
    grecs_type_host,
    grecs_type_sockaddr,
    grecs_type_section
};

#define GRECS_LIST 0x8000
#define GRECS_TYPE_MASK 0x00ff
#define GRECS_TYPE(c) ((c) & GRECS_TYPE_MASK)
#define GRECS_IS_LIST(c) ((c) & GRECS_LIST)

enum grecs_callback_command {
    grecs_callback_section_begin,
    grecs_callback_section_end,
    grecs_callback_set_value
};

#define GRECS_TYPE_STRING 0
#define GRECS_TYPE_LIST   1
#define GRECS_TYPE_ARRAY  2

typedef struct grecs_value {
    int type;
    union {
	gl_list_t list;
	const char *string;
	struct {
	    size_t c;
	    struct grecs_value *v;
	} arg;
    } v;
} grecs_value_t;

typedef int (*grecs_callback_fn) (
    enum grecs_callback_command cmd,
    grecs_locus_t *    /* locus */,
    void *             /* varptr */,
    grecs_value_t *    /* value */,
    void *             /* cb_data */
    );

struct grecs_keyword {
    const char *ident;
    const char *argname;
    const char *docstring;
    enum grecs_data_type type;
    void *varptr;
    size_t offset;
    grecs_callback_fn callback;
    void *callback_data;
    struct grecs_keyword *kwd;
};

struct grecs_sockaddr {
  int len;
  struct sockaddr *sa;
};

grecs_value_t *grecs_value_dup(grecs_value_t *input);

extern void grecs_print_diag(grecs_locus_t *, int, int, const char*);

void grecs_warning(grecs_locus_t *locus, int errcode, const char *fmt, ...)
  __attribute__ ((__format__ (__printf__, 3, 4)));
void grecs_error(grecs_locus_t *locus, int errcode, const char *fmt, ...)
  __attribute__ ((__format__ (__printf__, 3, 4)));
void grecs_set_keywords(struct grecs_keyword *kwd);
void grecs_gram_trace(int n);
void grecs_lex_trace (int n);

int grecs_lex_begin(const char*);  
void grecs_lex_end(void);
int grecs_parse (const char *name);

void grecs_line_begin (void);
void grecs_line_add (const char *text, size_t len);
char *grecs_line_finish (void);

extern int grecs_string_convert (void *target, enum grecs_data_type type,
				 const char *string, grecs_locus_t *locus);
extern void grecs_process_ident (struct grecs_keyword *kwp,
				 grecs_value_t *value,
				 void *base,
				 grecs_locus_t *locus);

extern grecs_locus_t grecs_current_locus;
extern int grecs_error_count;    
extern int grecs_default_port;

extern const char *grecs_preprocessor;
extern bool grecs_log_to_stderr;
extern void (*grecs_log_setup_hook) ();

size_t grecs_preproc_fill_buffer (char *buf, size_t size);
void grecs_preproc_add_include_dir (char *dir);
int grecs_preproc_init (const char *name);
void grecs_preproc_done (void);
int grecs_preproc_run (const char *config_file, const char *extpp);

FILE *grecs_preproc_extrn_start (const char *file, pid_t *ppid);
void grecs_preproc_extrn_shutdown (pid_t pid);

char *grecs_install_text (const char *str);
void grecs_destroy_text (void);

void grecs_include_path_setup (const char *dir, ...);
void grecs_include_path_setup_v (char **dirs);

const char *grecs_data_type_string (enum grecs_data_type type);
void grecs_format_docstring (FILE *stream, const char *docstring,
			     unsigned level);
void grecs_format_simple_statement (FILE *stream, struct grecs_keyword *kwp,
				    unsigned level);
void grecs_format_block_statement (FILE *stream, struct grecs_keyword *kwp,
				   unsigned level);
void grecs_format_statement_array (FILE *stream, struct grecs_keyword *kwp,
				   unsigned n,
				   unsigned level);



