/*  Motti -- a strategy game
    Copyright (C) 1999 Free Software Foundation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
/* This file manages thread creation.  When no thread support is
   compiled in, this just starts one game with the appropriate display
   type (X, curses).  */

#include <config.h>

#ifdef HAVE_LIBPTHREAD
#include <pthread.h>
#include <sched.h>
#endif

#include "thread.h"

#ifdef HAVE_LIBX11
#include <X11/Xlib.h>
#include <X11/Xutil.h>

/* Defined here globally, since every X thread needs to know this.  */
int global_argc;
char **global_argv;

static char *display_name_from_file = NULL;
#endif

#ifdef HAVE_LIBPTHREAD
/* Threads need to know each other connections, when generating X expose
   events or otherwise notifying of changes.  */
union conn_info *global_conn_info;
enum conn_type *global_conn_type;
#endif

struct thread_info {
  int n_conn;
  const char *connect_name;
};

/* Number of separate connections open.  */
int n_connect;
/* Number of separate X connections open.  */
int n_x_connect;

/* Mask array with a bit set for each player playing on corresponding
   display.  */
int *connect_mask;

#ifdef HAVE_LIBPTHREAD
static pthread_mutex_t nlock_mutex = PTHREAD_MUTEX_INITIALIZER;
static int n_lock = 0, write_pending = 0;
static pthread_cond_t r_wait = PTHREAD_COND_INITIALIZER;
static pthread_cond_t w_wait = PTHREAD_COND_INITIALIZER;
#endif

static void get_display_from_file ();

extern void
create_threads (type, connection)
     thread_func **type;
     const char **connection;
{
  register int i;
  struct thread_info *info;
#ifdef HAVE_LIBPTHREAD
  pthread_t null;	/* Discarded.  */
  pthread_attr_t attr;

  pthread_attr_init (&attr);
  pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);

  for (i = 0; i < n_connect; i++)
    get_read_lock ();	/* Make sure game map is under read lock until
			   everything is ready.  */

  for (i = 1; i < n_connect; i++)
    {
      info = (struct thread_info *) my_malloc (sizeof (info));
      info->connect_name = connection[i];
      info->n_conn = i;
      pthread_create (&null, &attr, type[i], info);
    }
  pthread_attr_destroy (&attr);
  pthread_detach (pthread_self ());
#endif /* HAVE_LIBPTHREAD */
  info = (struct thread_info *) my_malloc (sizeof (info));
  info->connect_name = connection[0];
  info->n_conn = 0;
#ifdef HAVE_LIBPTHREAD
  type[0] (info);
#else
  x_game_init (info);
#endif
}

/* This function searches for display name in certain files. */
static void
get_display_from_file ()
{
  XrmValue value;
  XrmDatabase displayDB = {0};
  char *str_type;
  
  XrmCombineFileDatabase ("~/.Xdefaults", &displayDB, False);
  XrmCombineFileDatabase (APPDEF, &displayDB, False);
  if (XrmGetResource (displayDB, "motti.display", "Motti.Display", &str_type,
		      &value))
    {
      display_name_from_file = (char *) my_malloc (value.size);
      memcpy (display_name_from_file, value.addr, value.size);
    }
  else
    display_name_from_file = "";
  XrmDestroyDatabase (displayDB);
}

extern void *
x_game_init (ptr)
     void *ptr;
{
  struct thread_info *info = ptr;
  struct x_info_struct x_info;
#ifdef HAVE_LIBPTHREAD
  static pthread_once_t display_name_init = PTHREAD_ONCE_INIT;
  static pthread_once_t xrm_init = PTHREAD_ONCE_INIT;
  static int x_inits_started;

  /* This is used to make sure the active thread doesn't do anything funny
     until all threads have been initialized.  */
  pthread_mutex_init (&x_info.init_lock, NULL);
  pthread_mutex_lock (&(x_info.init_lock));
  x_inits_started++;

  /* Called here, since create_threads couldn't care less if any of the
     threads was a X thread.  */
  pthread_once (&xrm_init, XrmInitialize);
#else
  XrmInitialize ();
#endif
  x_info.conn_num = info->n_conn;

#ifdef HAVE_LIBPTHREAD
  global_conn_info[info->n_conn].x_info = &x_info;
  global_conn_type[info->n_conn] = CONN_X;
#endif

  if (info->connect_name)
    {
      x_info.display = XOpenDisplay (info->connect_name);
#ifdef HAVE_LIBPTHREAD
      if (x_info.display)
	x_info.common_display = XOpenDisplay (info->connect_name);
#endif
    }
  else
    {
      if (!display_name_from_file)
#ifdef HAVE_LIBPTHREAD
	pthread_once (&display_name_init, get_display_from_file);
      x_info.display = XOpenDisplay (display_name_from_file);
      if (x_info.display)
	x_info.common_display = XOpenDisplay (display_name_from_file);
#else
      get_display_from_file ();
      x_info.display = XOpenDisplay (display_name_from_file);
#endif
    }
  if (!x_info.display)
    {
      if (n_connect == 1)
	die ("can't open display %s. try --curses for text mode",
	     info->connect_name);
      else
	die ("can't open display %s", info->connect_name);
    }
  create_DB (&x_info);
  if (!alloc_colors (&x_info))
    die ("can't allocate color on display %s", info->connect_name);
  open_windows (&x_info);
  free (info);

#ifdef HAVE_LIBPTHREAD
  /* Hack alert!  This display is used by the active thread to send events on
     screen updates.  We need here an separate display to avoid asynchronous
     access.  */
  XSynchronize (x_info.common_display, True);
  /* Hack again!  This state is used when no end window exists.  Endgame
     window gets a different value, when it is created.  */
  x_info.endgame_win = x_info.map_win;
  /* Make sure all threads have at least begun initialization.  */
  while (x_inits_started < n_x_connect)
    sched_yield ();
#endif
  event_loop (&x_info);
}

#ifdef HAVE_LIBPTHREAD
extern void
signal_end_game (except)
     const int except;
{
  register int i = 0;
#ifdef HAVE_LIBX11
  XEvent event;
  event.type = ClientMessage;
#endif

  for (; i < n_connect; i++)
    {
      if (i == except)
	continue;
#ifdef HAVE_LIBX11
      if (global_conn_type[i] == CONN_X)
	{
	  /* Make sure init is ready.  */
	  pthread_mutex_lock (&(global_conn_info[i].x_info->init_lock));
	  pthread_mutex_unlock (&(global_conn_info[i].x_info->init_lock));
	  event.xclient.display = global_conn_info[i].x_info->common_display;
	  event.xclient.window = global_conn_info[i].x_info->main_win;
	  event.xclient.format = 8;
	  event.xclient.data.b[0] = 1;
	  XSendEvent (event.xexpose.display, event.xexpose.window, False, 0,
		      &event);
	}
#endif
    }
}

extern void
expose_map (except, x, y, width, height)
     const int except, x, y, width, height;
{
  register int i;
#ifdef HAVE_LIBX11
  XEvent event;
  event.type = Expose;
#endif
  for (i = 0; i < n_connect; i++)
    {
      if (i == except)
	continue;
#ifdef HAVE_LIBX11
#ifdef HAVE_LIBCURSES
      if (global_conn_type[i] == CONN_X)
	{
#endif
	  /* Make sure init is ready.  */
	  pthread_mutex_lock (&(global_conn_info[i].x_info->init_lock));
	  pthread_mutex_unlock (&(global_conn_info[i].x_info->init_lock));
	  event.xexpose.x = x *
	    global_conn_info[i].x_info->db_tab.map_square_size;
	  event.xexpose.y = y *
	    global_conn_info[i].x_info->db_tab.map_square_size;
	  event.xexpose.width = width *
	    global_conn_info[i].x_info->db_tab.map_square_size;
	  event.xexpose.height = height *
	    global_conn_info[i].x_info->db_tab.map_square_size;
	  event.xexpose.display = global_conn_info[i].x_info->common_display;
	  event.xexpose.window = global_conn_info[i].x_info->map_win;
	  XSendEvent (event.xexpose.display, event.xexpose.window, False,
		      ExposureMask, &event);
#ifdef HAVE_LIBCURSES
	}
      else
	{
	  /* Curses can't use multiple threads, no need to ifdef that.  */
	  /* TODO: add stuff here.  */
	}
#endif
#endif
    }
}

extern void
expose_turnwin (except)
     const int except;
{
  register int i;
#ifdef HAVE_LIBX11
  XEvent event;
  event.type = Expose;
#endif
  for (i = 0; i < n_connect; i++)
    {
      if (i == except)
	continue;
#ifdef HAVE_LIBX11
#ifdef HAVE_LIBCURSES
      if (global_conn_type[i] == CONN_X)
	{
#endif
	  /* Make sure init is ready.  */
	  pthread_mutex_lock (&(global_conn_info[i].x_info->init_lock));
	  pthread_mutex_unlock (&(global_conn_info[i].x_info->init_lock));
	  event.xexpose.x = event.xexpose.y = 0;
	  event.xexpose.width = event.xexpose.height = 1;
	  event.xexpose.window = global_conn_info[i].x_info->turn_win;
	  event.xexpose.display = global_conn_info[i].x_info->common_display;
	  XSendEvent (event.xexpose.display, event.xexpose.window, False,
		      ExposureMask, &event);
#ifdef HAVE_LIBCURSES
	}
      else
	{
	  /* Same stuff as above...  */
	}
#endif
#endif
    }
}

/* Give visual clue about the number of attacks left.  */
extern void
expose_att_but (except)
     const int except;
{
  register int i;
#ifdef HAVE_LIBX11
  XEvent event;
  event.type = ClientMessage;
#endif
  for (i = 0; i < n_connect; i++)
    {
      if (i == except)
	continue;
#ifdef HAVE_LIBX11
#ifdef HAVE_LIBCURSES
      if (global_conn_type[i] == CONN_X)
	{
#endif
	  /* Make sure init is ready.  */
	  pthread_mutex_lock (&(global_conn_info[i].x_info->init_lock));
	  pthread_mutex_unlock (&(global_conn_info[i].x_info->init_lock));
	  event.xclient.display = global_conn_info[i].x_info->common_display;
	  event.xclient.window = global_conn_info[i].x_info->main_win;
	  event.xclient.format = 8;
	  event.xclient.data.b[0] = 0;
	  XSendEvent (event.xclient.display, event.xclient.window, False, 0,
		      &event);
#ifdef HAVE_LIBCURSES
	}
      else; /* Do something.  */
#endif
#endif
    }
}

/* Locks are used to synchronise map drawing and reading.  */
extern void
get_read_lock ()
{
  pthread_mutex_lock (&nlock_mutex);
  while (n_lock == -1 || write_pending)		/* Write lock.  */
    {
      pthread_cond_wait (&r_wait, &nlock_mutex);
      pthread_mutex_lock (&nlock_mutex);
    }
  n_lock++;
  pthread_mutex_unlock (&nlock_mutex);
}

extern void
release_read_lock ()
{
  pthread_mutex_lock (&nlock_mutex);
  if (--n_lock == 0 && write_pending)
    pthread_cond_signal (&w_wait);		/* Writes are urgent.  */
  pthread_mutex_unlock (&nlock_mutex);
}

extern void
get_write_lock ()
{
  pthread_mutex_lock (&nlock_mutex);
  while (n_lock != 0)
    {
      write_pending = 1;
      pthread_cond_wait (&w_wait, &nlock_mutex);
      pthread_mutex_lock (&nlock_mutex);
    }
  write_pending = 0;
  n_lock = -1;
  pthread_mutex_unlock (&nlock_mutex);
}

extern void
release_write_lock ()
{
  pthread_mutex_lock (&nlock_mutex);
  n_lock = 0;
  /* In motti, there's only one thread at time willing to write to the map.
     Therefore, no test on write_pending here.  */
  pthread_cond_broadcast (&r_wait);
  pthread_mutex_unlock (&nlock_mutex);
}
#endif	/* HAVE_LIBPTHREAD */
