/*****************************************************************************\
 * Copyright 2005, 2006 Niels Lohmann, Christian Gierds, Dennis Reinert      *
 *                                                                           *
 * This file is part of BPEL2oWFN.                                           *
 *                                                                           *
 * BPEL2oWFN is free software; you can redistribute it and/or modify it      *
 * under the terms of the GNU General Public License as published by the     *
 * Free Software Foundation; either version 2 of the License, or (at your    *
 * option) any later version.                                                *
 *                                                                           *
 * BPEL2oWFN is distributed in the hope that it will be useful, but WITHOUT  *
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or     *
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  *
 * more details.                                                             *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with BPEL2oWFN; if not, write to the Free Software Foundation, Inc., 51   *
 * Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.                      *
\****************************************************************************/

%{
/*!
 * \file bpel-syntax.cc
 *
 * \brief BPEL grammar (implementation)
 *
 * This file defines and implements the grammar of BPEL using standard 
 * BNF-rules to describe the originally XML-based syntax as it is specified in
 * the BPEL4WS 1.1 specification. All terminals are passed from the lexer
 * (implemented in \ref bpel-lexic.cc). Besides simple syntax-checking the
 * grammar is used to build the abstract syntax tree as it is defined in
 * bpel-abstract.k and implemented in \ref bpel-abstract.cc and
 * \ref bpel-abstract.h.
 * 
 * \author  
 *          - responsible: Niels Lohmann <nlohmann@informatik.hu-berlin.de>
 *          - last changes of: \$Author: gierds $
 *          
 * \date 
 *          - created: 2005/11/10
 *          - last changed: \$Date: 2006/04/05 16:00:30 $
 * 
 * \note    This file is part of the tool BPEL2oWFN and was created during the
 *          project "Tools4BPEL" at the Humboldt-Universit�t zu Berlin. See
 *          http://www.informatik.hu-berlin.de/top/tools4bpel for details.
 *
 * \note    This file was created using GNU Bison reading file bpel-syntax.yy.
 *          See http://www.gnu.org/software/bison/bison.html for details
 *
 * \version \$Revision: 1.169 $
 * 
 * \todo
 *          - add rules to ignored everything non-BPEL
 */
%}


%{	
/*!
 * \file bpel-syntax.h
 * \brief BPEL grammar (interface)
 *
 * See \ref bpel-syntax.c for more information.
 */
%}

// the terminal symbols (tokens)
%token K_ASSIGN K_CASE K_CATCH K_CATCHALL K_COMPENSATE K_COMPENSATIONHANDLER
%token K_COPY K_CORRELATION K_CORRELATIONS K_CORRELATIONSET K_CORRELATIONSETS
%token K_EMPTY K_EVENTHANDLERS K_FAULTHANDLERS K_FLOW K_FROM K_IMPORT K_INVOKE
%token K_LINK K_LINKS K_ONALARM K_ONMESSAGE K_OTHERWISE K_PARTNER K_PARTNERLINK
%token K_PARTNERLINKS K_PARTNERS K_PICK K_PROCESS K_RECEIVE K_REPLY K_SCOPE
%token K_SEQUENCE K_SOURCE K_SWITCH K_TARGET K_TERMINATE K_THROW K_TO
%token K_VARIABLE K_VARIABLES K_WAIT K_WHILE
%token X_OPEN X_SLASH X_CLOSE X_NEXT X_EQUALS
%token <yt_casestring> X_NAME
%token <yt_casestring> X_STRING


// the start symbol of the grammar
%start tProcess


// Bison generates a list of all used tokens in file "bpel-syntax.h" (for flex)
%token_table



%{



// options for Bison
#define YYDEBUG 1
#define YYERROR_VERBOSE 1  // for verbose error messages


// to avoid the message "parser stack overflow"
#define YYMAXDEPTH 1000000
#define YYINITDEPTH 10000


// generated by Kimwitu++
#include "bpel-kc-k.h" // phylum definitions
#include "bpel-kc-yystype.h" // data types for tokens and non-terminals

#include <map>
#include "exception.h"
#include "helpers.h"

// from flex
extern char* yytext;
extern int yylex();


// defined in "debug.h"
extern int yyerror(const char *);


using namespace kc;
using namespace std;

// symbol table
#include "symbol-table.h"

/// an instance of the symbol table
SymbolTable symTab = SymbolTable();
SymbolTableEntry* currentSymTabEntry;
unsigned int currentSymTabEntryKey = 0;

/// needed to tag handlers
integer currentScopeId;
/// pointer to the current surrounding Scope symbol table entry
STScope * currentSTScope = NULL;
/// pointer to the current surrounding Flow symbol table entry
STFlow * currentSTFlow = NULL;
/// pointer to the Process symbol table entry
STProcess * stProcess = NULL;

// manage attributes
#include "bpel-attributes.h"

/// an instance of the attribute manager
attributeManager att = attributeManager();

#include "check-symbols.h"

/// from check-symbols
SymbolManager symMan = SymbolManager();

/// needed to distinguish context of tPartnerLink
bool inPartners = false;

/// needed to check occurrence of links within whiles
bool inWhile = false;

/// needed to tag scopes
map <integer, integer> parent;

/// the root of the abstract syntax tree
tProcess TheProcess;

/// needed to find out if a FH has a <catchAll> element
bool hasCatchAll = false;

/// stack for checking for FaultHandler
stack<bool> isInFH;
/// stack for checking for CompensationHandler
stack< pair<bool,int> > isInCH;
int hasCompensate;

%}



/* the types of the non-terminal symbols */
%type <yt_activity_list> activity_list
%type <yt_activity> activity
%type <yt_integer> arbitraryAttributes
%type <yt_standardElements> standardElements
%type <yt_tAssign> tAssign
%type <yt_tCase_list> tCase_list
%type <yt_tCase> tCase
%type <yt_tCatch_list> tCatch_list
%type <yt_tCatch> tCatch
%type <yt_tCatchAll> tCatchAll
%type <yt_tCompensate> tCompensate
%type <yt_tCompensationHandler> tCompensationHandler
%type <yt_tCopy_list> tCopy_list
%type <yt_tCopy> tCopy
%type <yt_tCorrelation_list> tCorrelation_list
%type <yt_tCorrelation_list> tCorrelations
%type <yt_tCorrelation> tCorrelation
%type <yt_tCorrelationSet_list> tCorrelationSet_list
%type <yt_tCorrelationSet_list> tCorrelationSets
%type <yt_tCorrelationSet> tCorrelationSet
%type <yt_tEmpty> tEmpty
%type <yt_tEventHandlers> tEventHandlers
%type <yt_tFaultHandlers> tFaultHandlers
%type <yt_tFlow> tFlow
%type <yt_tFrom> tFrom
%type <yt_activity> tInvoke
%type <yt_tLink_list> tLink_list
%type <yt_tLink_list> tLinks
%type <yt_tLink> tLink
%type <yt_tOnAlarm_list> tOnAlarm_list
%type <yt_tOnAlarm> tOnAlarm
%type <yt_tOnMessage_list> tOnMessage_list
%type <yt_tOnMessage> tOnMessage
%type <yt_tOtherwise> tOtherwise
%type <yt_tPartner_list> tPartner_list
%type <yt_tPartner_list> tPartners
%type <yt_tPartner> tPartner
%type <yt_tPartnerLink_list> tPartnerLink_list
%type <yt_tPartnerLink_list> tPartnerLinks
%type <yt_tPartnerLink> tPartnerLink
%type <yt_tPick> tPick
%type <yt_tProcess> tProcess
%type <yt_tReceive> tReceive
%type <yt_tReply> tReply
%type <yt_tScope> tScope
%type <yt_tSequence> tSequence
%type <yt_tSource_list> tSource_list
%type <yt_tSource> tSource
%type <yt_tSwitch> tSwitch
%type <yt_tTarget_list> tTarget_list
%type <yt_tTarget> tTarget
%type <yt_tTerminate> tTerminate
%type <yt_tThrow> tThrow
%type <yt_tTo> tTo
%type <yt_tVariable_list> tVariable_list
%type <yt_tVariable_list> tVariables
%type <yt_tVariable> tVariable
%type <yt_tWait> tWait
%type <yt_tWhile> tWhile
%type <yt_integer> genSymTabEntry_Process
%type <yt_integer> genSymTabEntry_PartnerLink
%type <yt_integer> genSymTabEntry_Partner
%type <yt_integer> genSymTabEntry_FaultHandlers
%type <yt_integer> genSymTabEntry_Catch
%type <yt_integer> genSymTabEntry_CatchAll
%type <yt_integer> genSymTabEntry_CompensationHandler
%type <yt_integer> genSymTabEntry_EventHandlers
%type <yt_integer> genSymTabEntry_OnMessage
%type <yt_integer> genSymTabEntry_OnAlarm
%type <yt_integer> genSymTabEntry_Variable
%type <yt_integer> genSymTabEntry_CorrelationSet
%type <yt_integer> genSymTabEntry_Correlation
%type <yt_integer> genSymTabEntry_Empty
%type <yt_integer> genSymTabEntry_Invoke
%type <yt_integer> genSymTabEntry_Receive
%type <yt_integer> genSymTabEntry_Reply
%type <yt_integer> genSymTabEntry_Assign
%type <yt_integer> genSymTabEntry_From
%type <yt_integer> genSymTabEntry_To
%type <yt_integer> genSymTabEntry_Wait
%type <yt_integer> genSymTabEntry_Throw
%type <yt_integer> genSymTabEntry_Compensate
%type <yt_integer> genSymTabEntry_Terminate
%type <yt_integer> genSymTabEntry_Flow
%type <yt_integer> genSymTabEntry_Link
%type <yt_integer> genSymTabEntry_Switch
%type <yt_integer> genSymTabEntry_Case
%type <yt_integer> genSymTabEntry_While
%type <yt_integer> genSymTabEntry_Sequence
%type <yt_integer> genSymTabEntry_Scope
%type <yt_integer> genSymTabEntry_Pick
%type <yt_integer> genSymTabEntry_Target
%type <yt_integer> genSymTabEntry_Source

%type <yt_integer> descent_activity_list
%type <yt_integer> descent_case_list


%%

/******************************************************************************
  PROCESS
******************************************************************************/

tProcess:
  X_OPEN K_PROCESS genSymTabEntry_Process
  arbitraryAttributes
    { symTab.checkAttributes($3); //att.check($4, K_PROCESS);
      if(att.isAttributeValueEmpty($4, "suppressJoinFailure"))
      {
      	/// default attribute value
      	att.pushSJFStack($4, mkcasestring("no"));
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($4, att.read($4, "suppressJoinFailure"));      
      }      

      try
      {
	stProcess = dynamic_cast<STProcess*> (symTab.lookup($3));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
    
//CG      symMan.initialiseProcessScope($3);
      currentScopeId = $3;
      currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentScopeId->value));
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,0));
      hasCompensate = 0;
   }
  X_NEXT imports tPartnerLinks tPartners tVariables tCorrelationSets tFaultHandlers tCompensationHandler tEventHandlers
  activity
  X_NEXT X_SLASH K_PROCESS X_CLOSE
    { TheProcess = $$ = Process($8, $9, $10, $11, $12, $13, $14, StopInProcess(), $15);
//CG      symMan.quitScope();
      //symTab.traceST("\t\t\t\t HALLO " + string((att.read($4, "abstractProcess")->name)) + "\n");      
//      att.traceAM(string("tProcess: ") + ($$->suppressJoinFailure)->name + string("\n"));      
      att.popSJFStack(); symTab.popSJFStack();
      isInFH.pop();
      isInCH.pop();
      $$->id = $3;
      ((STProcess*)symTab.lookup($3))->hasEventHandler = (string($14->op_name()) == "userDefinedEventHandler");
      symTab.printSymbolTable(); // purely debugging
      symTab.traceST(symTab.readAttributeValue($3,"name") + "\n"); // for Niels
    }
;

genSymTabEntry_Process:
  { currentSymTabEntryKey = symTab.insert(K_PROCESS);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/* import other namespaces */
imports:
  /* empty */
| K_IMPORT
  { currentSymTabEntryKey = 0; /* no entry in SymbolTable */ } arbitraryAttributes
  X_SLASH X_NEXT imports
;

/*---------------------------------------------------------------------------*/


activity:
  tEmpty
    { $$ = activityEmpty($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tInvoke
    { $$ = $1;  }
/*
| tInvoke
    { $$ = activityInvoke($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
*/
| tReceive
    { $$ = activityReceive($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tReply
    { $$ = activityReply($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tAssign
    { $$ = activityAssign($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tWait
    { $$ = activityWait($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tThrow
    { $$ = activityThrow($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tTerminate
    { $$ = activityTerminate($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tFlow
    { $$ = activityFlow($1); $$->id = $1->id; 
      $$->dpe = $1->dpe ;
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tSwitch
    { $$ = activitySwitch($1); $$->id = $1->id; 
      $$->dpe = $1->dpe;
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tWhile
    { $$ = activityWhile($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tSequence
    { $$ = activitySequence($1); $$->id = $1->id; 
      $$->dpe = $1->dpe;
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tPick
    { $$ = activityPick($1); $$->id = $1->id; 
      $$->dpe = $1->dpe;
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tScope
    { $$ = activityScope($1); $$->id = $1->id; 
      $$->dpe = $1->dpe;
      $$->negativeControlFlow = $1->negativeControlFlow; }
| tCompensate
    { $$ = activityCompensate($1); $$->id = $1->id; 
      $$->negativeControlFlow = $1->negativeControlFlow; }
;


/******************************************************************************
  PARTNER LINKS
******************************************************************************/

tPartnerLinks:
  /* empty */
    { $$ = NiltPartnerLink_list(); }
| K_PARTNERLINKS X_NEXT tPartnerLink_list X_SLASH K_PARTNERLINKS X_NEXT
    { $$ = $3; } 
;

tPartnerLink_list:
  tPartnerLink X_NEXT
    { $$ = ConstPartnerLink_list($1, NiltPartnerLink_list()); }
| tPartnerLink X_NEXT tPartnerLink_list
    { $$ = ConstPartnerLink_list($1, $3); }
;

tPartnerLink:
  K_PARTNERLINK genSymTabEntry_PartnerLink
    arbitraryAttributes X_NEXT X_SLASH K_PARTNERLINK
    { 
      STPartnerLink * stPartnerLink = NULL;
      try
      {
	stPartnerLink = dynamic_cast<STPartnerLink*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      if(inPartners) {
        stPartnerLink->isInPartners = true;
      }
      else {
        stPartnerLink->isInPartners = false;      
      }
      symTab.checkAttributes($2);
      $$ = PartnerLink();
      $$->id = $2;
      if (inPartners) {
	stProcess->checkPartnerLink(symTab.readAttributeValue($2, "name"));
      }
      else
      {
	// cast to STProcess should always happen, since PartnerLinks can only be declared there
	(dynamic_cast<STProcess*>(currentSTScope))->addPartnerLink(stPartnerLink);
      }
    }
| K_PARTNERLINK genSymTabEntry_PartnerLink
    arbitraryAttributes X_SLASH
    { 
      STPartnerLink * stPartnerLink = NULL;
      try
      {
	stPartnerLink = dynamic_cast<STPartnerLink*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      if(inPartners) {
        stPartnerLink->isInPartners = true;
      }
      else {
        stPartnerLink->isInPartners = false;      
      }
      symTab.checkAttributes($2);
      $$ = PartnerLink();
      $$->id = $2;
      if (inPartners) {
	stProcess->checkPartnerLink(symTab.readAttributeValue($2, "name"));
      }
      else
      {
	// cast to STProcess should always happen, since PartnerLinks can only be declared there
	(dynamic_cast<STProcess*>(currentSTScope))->addPartnerLink(stPartnerLink);
      }
    }
;

genSymTabEntry_PartnerLink:
  { currentSymTabEntryKey = symTab.insert(K_PARTNERLINK);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  PARTNERS
******************************************************************************/

tPartners:
  /* empty */
    { $$ = NiltPartner_list(); }
| K_PARTNERS X_NEXT
    { inPartners = true; }
  tPartner_list X_SLASH K_PARTNERS X_NEXT
    { $$ = $4;
      inPartners = false; }
;

tPartner_list:
  tPartner X_NEXT
    { $$ = ConstPartner_list($1, NiltPartner_list()); }
| tPartner X_NEXT tPartner_list
    { $$ = ConstPartner_list($1, $3); }
;

tPartner:
  K_PARTNER genSymTabEntry_Partner arbitraryAttributes X_NEXT tPartnerLink_list X_SLASH K_PARTNER
    { symTab.checkAttributes($2);
      $$ = Partner($5);
      $$->id = $2;
    }
| K_PARTNER genSymTabEntry_Partner arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2);
      $$ = Partner(NiltPartnerLink_list());
      $$->id = $2;      
    }
;

genSymTabEntry_Partner:
  { currentSymTabEntryKey = symTab.insert(K_PARTNER);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  FAULT HANDLERS
******************************************************************************/

tFaultHandlers:
  /* empty */
    { currentSymTabEntryKey = symTab.insert(K_FAULTHANDLERS);
      $$ = implicitFaultHandler();
      $$->id = mkinteger(currentSymTabEntryKey);
      $$->inProcess = (currentScopeId->value == 1);
      $$->parentScopeId = currentScopeId; 
      $$->hasCatchAll = false;
    }
| K_FAULTHANDLERS genSymTabEntry_FaultHandlers X_NEXT 
    {
      symMan.startDPEinWhile();
      isInFH.push(true);
      hasCompensate = 0;
    }
  tCatch_list  
  tCatchAll 
  X_SLASH K_FAULTHANDLERS X_NEXT
    { $$ = userDefinedFaultHandler($5, $6);
      $$->id = $2;
      $$->inProcess = (currentScopeId->value == 1);
      $$->parentScopeId = currentScopeId;
      $$->hasCatchAll = hasCatchAll;
      isInFH.pop();
      hasCompensate = 0;
      symMan.endDPEinWhile();
    }
;

genSymTabEntry_FaultHandlers:
  { currentSymTabEntryKey = symTab.insert(K_FAULTHANDLERS);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tCatch_list:
  /* empty */
    { $$ = NiltCatch_list(); }
| tCatch X_NEXT tCatch_list
    { $$ = ConstCatch_list($1, $3); }
;

tCatch:
  K_CATCH genSymTabEntry_Catch
  arbitraryAttributes X_NEXT 
    {
      symTab.checkAttributes($2);
      STCatch * stCatch = NULL;
      try
      {
	stCatch = dynamic_cast<STCatch*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stCatch->faultVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "faultVariable"), currentSTScope, true);
    }
  activity X_NEXT X_SLASH K_CATCH
    { 

      $$ = Catch($6);
      $$->id = $2;
      $$->faultName = att.read($3, "faultName");
    }
;

genSymTabEntry_Catch:
  { currentSymTabEntryKey = symTab.insert(K_CATCH);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tCatchAll:
  /* empty */
    { hasCatchAll = false;
      $$ = NoCatchAll(); }
| K_CATCHALL genSymTabEntry_CatchAll
  arbitraryAttributes X_NEXT activity X_NEXT X_SLASH K_CATCHALL X_NEXT
    { hasCatchAll = true;
      $$ = CatchAll($5);
      $$->id = $2;      
   /* DR
      $$->faultName = att.read($3, "faultName");
      $$->faultVariable = att.read($3, "faultVariable");
   */
    }
;

genSymTabEntry_CatchAll:
  { currentSymTabEntryKey = symTab.insert(K_CATCHALL);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  COMPENSATION HANDLERS
******************************************************************************/

tCompensationHandler:
  /* empty */
    { currentSymTabEntryKey = symTab.insert(K_COMPENSATIONHANDLER);
      if (currentScopeId->value == 1)
        $$ = processCompensationHandler();
      else
        $$ = implicitCompensationHandler();
      $$->id = mkinteger(currentSymTabEntryKey);              
      $$->parentScopeId = currentScopeId; }
| K_COMPENSATIONHANDLER genSymTabEntry_CompensationHandler X_NEXT 
    {
      symMan.startDPEinWhile();
      isInCH.push(pair<bool,int>(true,hasCompensate));
      hasCompensate = 0;
    }
  activity 
  X_NEXT X_SLASH K_COMPENSATIONHANDLER X_NEXT
    { if (currentScopeId->value == 1)
        $$ = processCompensationHandler();
      else
        $$ = userDefinedCompensationHandler($5);

      switch ( hasCompensate ) {
        case 1 : $$->compensateWithoutScope = true;
                 $$->compensateWithScope = false;
                 break;
        case 2 : $$->compensateWithoutScope = false;
                 $$->compensateWithScope = true;
                 break;
        default: $$->compensateWithoutScope = false;
                 $$->compensateWithScope = false;
                 break;
      }
      hasCompensate = isInCH.top().second;
      isInCH.pop();
      
      $$->id = $2;
      $$->parentScopeId = currentScopeId; 
      symMan.endDPEinWhile();
    }
;

genSymTabEntry_CompensationHandler:
  { currentSymTabEntryKey = symTab.insert(K_COMPENSATIONHANDLER);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  EVENT HANDLERS
******************************************************************************/

tEventHandlers:
  /* empty */
    { currentSymTabEntryKey = symTab.insert(K_EVENTHANDLERS);
      $$ = implicitEventHandler();
      $$->id = mkinteger(currentSymTabEntryKey);      
      $$->parentScopeId = currentScopeId; }
| K_EVENTHANDLERS genSymTabEntry_EventHandlers X_NEXT 
    {
      symMan.startDPEinWhile();
    }
  tOnMessage_list 
  tOnAlarm_list 
  X_SLASH K_EVENTHANDLERS X_NEXT
    { $$ = userDefinedEventHandler($5, $6);
      $$->id = $2;    
      $$->parentScopeId = currentScopeId; 
      symMan.endDPEinWhile();
    }
;

genSymTabEntry_EventHandlers:
  { currentSymTabEntryKey = symTab.insert(K_EVENTHANDLERS);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tOnMessage_list:
  /* empty */
    { $$ = NiltOnMessage_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tOnMessage X_NEXT tOnMessage_list
    { $$ = ConstOnMessage_list($1, $3); 
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tOnAlarm_list:
  /* empty */
    { $$ = NiltOnAlarm_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tOnAlarm X_NEXT tOnAlarm_list
    { $$ = ConstOnAlarm_list($1, $3); 
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tOnMessage:
  K_ONMESSAGE genSymTabEntry_OnMessage
  arbitraryAttributes X_NEXT
    { symTab.checkAttributes($2); //att.check($3, K_ONMESSAGE);
      
//CG//      symMan.checkPartnerLink(att.read($3, "partnerLink")->name);
      symMan.resetDPEend();
    }
  tCorrelations activity X_NEXT X_SLASH K_ONMESSAGE
    { $$ = OnMessage($7);
      $$->id = $2;    
      STOnMessage * stOnMessage = NULL;
      try
      {
	stOnMessage = dynamic_cast<STOnMessage *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stOnMessage->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stOnMessage->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stOnMessage->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);
      $$->dpe = symMan.needsDPE();
    }
;

genSymTabEntry_OnMessage:
  { currentSymTabEntryKey = symTab.insert(K_ONMESSAGE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


tOnAlarm:
  K_ONALARM genSymTabEntry_OnAlarm
  arbitraryAttributes X_NEXT 
    { symTab.checkAttributes($2); //att.check($3, K_ONALARM);
      
      symMan.resetDPEend();
    }
  activity X_NEXT X_SLASH K_ONALARM 
    { $$ = OnAlarm($6);
      $$->id = $2;
      $$->For = att.read($3, "for");  // "for" is a keyword
      $$->until = att.read($3, "until");
      $$->dpe = symMan.needsDPE();
    }
;

genSymTabEntry_OnAlarm:
  { currentSymTabEntryKey = symTab.insert(K_ONALARM);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  VARIABLES
******************************************************************************/

tVariables:
  /* empty */
    { $$ = NiltVariable_list(); }
| K_VARIABLES X_NEXT tVariable_list X_SLASH K_VARIABLES X_NEXT
    { $$ = $3; }
;

tVariable_list:
  tVariable X_NEXT
    { $$ = ConstVariable_list($1, NiltVariable_list()); }
| tVariable X_NEXT tVariable_list
    { $$ = ConstVariable_list($1, $3); }
;

tVariable:
  K_VARIABLE genSymTabEntry_Variable
  arbitraryAttributes X_NEXT X_SLASH K_VARIABLE
    { symTab.checkAttributes($2); //att.check($3, K_VARIABLE);
      STVariable * stVar = NULL;
      try
      {
	stVar = dynamic_cast<STVariable *> (symTab.lookup(currentSymTabEntryKey));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }

      $$ = Variable();
      $$->id = $2;
      stVar->name = currentSTScope->addVariable(stVar);
    }
| K_VARIABLE genSymTabEntry_Variable
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_VARIABLE);
      STVariable * stVar = NULL;
      try
      {
	stVar = dynamic_cast<STVariable *> (symTab.lookup(currentSymTabEntryKey));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }

      $$ = Variable();
      $$->id = $2;
      stVar->name = currentSTScope->addVariable(stVar);
    }
;

genSymTabEntry_Variable:
  { currentSymTabEntryKey = symTab.insert(K_VARIABLE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  CORRELATION SETS
******************************************************************************/

tCorrelationSets:
  /* empty */
    { $$ = NiltCorrelationSet_list(); }
| K_CORRELATIONSETS X_NEXT tCorrelationSet_list X_SLASH K_CORRELATIONSETS X_NEXT
    { $$ = $3; }
;

tCorrelationSet_list:
  tCorrelationSet X_NEXT
    { $$ = ConstCorrelationSet_list($1, NiltCorrelationSet_list()); }
| tCorrelationSet X_NEXT tCorrelationSet_list
    { $$ = ConstCorrelationSet_list($1, $3); }
;

tCorrelationSet:
  K_CORRELATIONSET genSymTabEntry_CorrelationSet
  arbitraryAttributes X_NEXT X_SLASH K_CORRELATIONSET
    { symTab.checkAttributes($2); //att.check($3, K_CORRELATIONSET);
      $$ = CorrelationSet();
      $$->id = $2;      
    }
| K_CORRELATIONSET genSymTabEntry_CorrelationSet
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_CORRELATIONSET);
      $$ = CorrelationSet();
      $$->id = $2;      
    }
;

genSymTabEntry_CorrelationSet:
  { currentSymTabEntryKey = symTab.insert(K_CORRELATIONSET);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  CORRELATIONS
******************************************************************************/

tCorrelations:
  /* empty */
    { $$ = NiltCorrelation_list(); }
| K_CORRELATIONS X_NEXT tCorrelation_list X_SLASH K_CORRELATIONS X_NEXT
    { $$ = $3; }
;

tCorrelation_list:
  tCorrelation X_NEXT
    { $$ = ConstCorrelation_list($1, NiltCorrelation_list());
    }
| tCorrelation X_NEXT tCorrelation_list
    { $$ = ConstCorrelation_list($1, $3); }
;

tCorrelation:
  K_CORRELATION genSymTabEntry_Correlation
  arbitraryAttributes X_NEXT X_SLASH K_CORRELATION
    { symTab.checkAttributes($2);
      $$ = Correlation();
      $$->id = $2;      
    }
| K_CORRELATION genSymTabEntry_Correlation
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2);
      $$ = Correlation();
      $$->id = $2;      
    }
;

genSymTabEntry_Correlation:
  { currentSymTabEntryKey = symTab.insert(K_CORRELATION);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  EMPTY
******************************************************************************/

tEmpty:
  K_EMPTY genSymTabEntry_Empty  
  arbitraryAttributes 
    { symTab.checkAttributes($2); //att.check($3, K_EMPTY);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_EMPTY
    { $$ = Empty($6);
      att.traceAM(string("tEmpty: ") + (att.read($3, "suppressJoinFailure"))->name + string("\n"));
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; 
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
    }
| K_EMPTY genSymTabEntry_Empty
  arbitraryAttributes 
    { symTab.checkAttributes($2); //att.check($3, K_EMPTY);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Empty(noLinks);
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Empty:
  { currentSymTabEntryKey = symTab.insert(K_EMPTY);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  INVOKE
******************************************************************************/

tInvoke:
  K_INVOKE genSymTabEntry_Invoke 
  arbitraryAttributes 
    { symTab.checkAttributes($2); //att.check($3, K_INVOKE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    {
      // automatically create scope?
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,hasCompensate));
      parent[$2] = currentScopeId;
      kc::integer qad_id = symTab.nextId();
      parent[qad_id] = currentScopeId;
      currentScopeId = qad_id; 
    }
  standardElements 
  tCorrelations tCatch_list  tCatchAll tCompensationHandler X_SLASH K_INVOKE
    { 
      isInFH.pop();
      hasCompensate = isInCH.top().second;
      isInCH.pop();
      if ($9->length() > 0 
           || (string($10->op_name()) != "NoCatchAll") 
           || string($11->op_name()) != "implicitCompensationHandler")
      {
        //cerr << "embed in scope" << endl;
        currentSymTabEntryKey = symTab.insert(K_SCOPE);
        currentSymTabEntry = symTab.lookup(currentSymTabEntryKey); 
	parent[mkinteger(currentSymTabEntryKey)] = parent[currentScopeId];
	currentScopeId = mkinteger(currentSymTabEntryKey);
	
	currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentSymTabEntryKey));
	currentSTScope->parentScopeId = parent[currentScopeId]->value;
	(dynamic_cast<STScope *> (symTab.lookup(currentSTScope->parentScopeId)))->childScopes.push_back(currentSTScope);

	STInvoke * stInvoke = NULL;
	try
	{
	  stInvoke = dynamic_cast<STInvoke *> (symTab.lookup($2));
	}
	catch (bad_cast)
	{
	  throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
	}
	stInvoke->inputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "inputVariable"),currentSTScope);
	stInvoke->outputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "outputVariable"),currentSTScope);
	stInvoke->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
        stInvoke->isAsynchronousInvoke = (stInvoke->outputVariable == NULL);

	standardElements se =  StandardElements(NiltTarget_list(), NiltSource_list());
        tInvoke invoke = Invoke(se, $8);
        activity ai = activityInvoke(invoke);
        tFaultHandlers fh = userDefinedFaultHandler($9, $10);
        fh->hasCatchAll = hasCatchAll;
        tEventHandlers eh = implicitEventHandler();
        tScope scope = Scope($7, NiltVariable_list(), fh, $11, eh, StopInScope(), ai);

        scope->id = $7->parentId = currentScopeId; 
        invoke->id = ai->id = se->parentId = $2;

        fh->inProcess = false;
        fh->parentScopeId = scope->id;
	fh->id = symTab.nextId();
        $11->parentScopeId = scope->id;
        eh->parentScopeId = scope->id;

//CG        symMan.newScopeScope(scope->id);
//CG        symMan.quitScope();

        symTab.addAttribute(currentSymTabEntryKey, symTab.newAttribute(mkcasestring("name"), att.read($3, "name")));
        symTab.addAttribute(currentSymTabEntryKey, symTab.newAttribute(mkcasestring("joinCondition"), att.read($3, "joinCondition")));
        $7->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
        symTab.addAttribute(currentSymTabEntryKey, symTab.newAttribute(mkcasestring("suppressJoinFailure"), $7->suppressJoinFailure));
	att.popSJFStack();
	// inputVariable <=> input for invoke process !!!
        if (symTab.readAttributeValue($2, "inputVariable") != "")
        {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 false);
        }
        if (symTab.readAttributeValue($2, "outputVariable") != "")
        {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);
        }
	scope->dpe = invoke->dpe = mkinteger(0);
        // symMan.needsDPE();
        if ($7->hasTarget)
        {
	 symMan.remDPEstart();
	}
	if ($7->dpe->value > 0)
        {
          symMan.addDPEend();
        }
        scope->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
        invoke->negativeControlFlow = mkinteger(0);
        /*
        scope->id = $7->parentId = $2; 
        invoke->id = ai->id = se->parentId = symTab.nextId();
        */
        ((STScope*)symTab.lookup(currentSymTabEntryKey))->hasEventHandler = false;

        currentScopeId = scope->parentScopeId = parent[$2];
        currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentScopeId->value));

        $$ = activity(activityScope(scope));

        $$->id = scope->id;
        $$->negativeControlFlow = scope->negativeControlFlow;
      }
      else
      { 
        //cerr << "don't embed" << endl;

        //restore real scope ID
        currentScopeId = parent[$2];
        currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentScopeId->value));

	STInvoke * stInvoke = NULL;
	try
	{
	  stInvoke = dynamic_cast<STInvoke *> (symTab.lookup($2));
	}
	catch (bad_cast)
	{
	  throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
	}
	stInvoke->inputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "inputVariable"),currentSTScope);
	stInvoke->outputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "outputVariable"),currentSTScope);
	stInvoke->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
        stInvoke->isAsynchronousInvoke = (stInvoke->outputVariable == NULL);

	tInvoke invoke = Invoke($7, $8);

        $7->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
        att.popSJFStack(); symTab.popSJFStack();
	// inputVariable <=> input for invoke process !!!
        if (symTab.readAttributeValue($2, "inputVariable") != "")
        {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 false);
        }
        if (symTab.readAttributeValue($2, "outputVariable") != "")
        {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);
        }
        invoke->dpe = symMan.needsDPE();
        if ($7->hasTarget)
        {
	  symMan.remDPEstart();
	}
        if ($7->dpe->value > 0)
        {
          symMan.addDPEend();
        }
        invoke->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
        invoke->id = $7->parentId = $2; 

        $$ = activity(activityInvoke(invoke));
        $$->id = invoke->id;
        $$->negativeControlFlow = invoke->negativeControlFlow;
      }    
    }
| K_INVOKE genSymTabEntry_Invoke 
  arbitraryAttributes 
    { 
      symTab.checkAttributes($2); //att.check($3, K_INVOKE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { 
      impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;

      STInvoke * stInvoke = NULL;
      try
      {
	stInvoke = dynamic_cast<STInvoke *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stInvoke->inputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "inputVariable"),currentSTScope);
      stInvoke->outputVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "outputVariable"),currentSTScope);
      stInvoke->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stInvoke->isAsynchronousInvoke = (stInvoke->outputVariable == NULL);


      tInvoke invoke = Invoke(noLinks, NiltCorrelation_list());
      att.popSJFStack(); symTab.popSJFStack();
      // inputVariable <=> input for invoke process !!!
      if (symTab.readAttributeValue($2, "inputVariable") != "")
      {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 false);
      }
      if (symTab.readAttributeValue($2, "outputVariable") != "")
      {
	  stInvoke->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);
      }
      invoke->dpe = kc::mkinteger(0);
      invoke->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      invoke->id = $2; 

      $$ = activity(activityInvoke(invoke));
      $$->id = invoke->id;
      $$->negativeControlFlow = invoke->negativeControlFlow;
}
;

genSymTabEntry_Invoke:
  { currentSymTabEntryKey = symTab.insert(K_INVOKE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  RECEIVE
******************************************************************************/

tReceive:
  K_RECEIVE genSymTabEntry_Receive
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_RECEIVE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    {
    }
  standardElements tCorrelations X_SLASH K_RECEIVE
    { $$ = Receive($7, $8);
      // STReceive * symbolTableEntry = dynamic_cast<STReceive *> (symTab.lookup($2)); 
      STReceive * stReceive = NULL;
      try
      {
	stReceive = dynamic_cast<STReceive *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stReceive->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stReceive->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stReceive->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);


      $7->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      // symbolTableEntry->variable
      if ($7->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($7->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $7->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $7->parentId = $2; 
    }
| K_RECEIVE genSymTabEntry_Receive
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_RECEIVE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;

      STReceive * stReceive = NULL;
      try
      {
	stReceive = dynamic_cast<STReceive *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stReceive->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stReceive->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stReceive->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 true);

      $$ = Receive(noLinks, NiltCorrelation_list());
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; 
    }
;

genSymTabEntry_Receive:
  { currentSymTabEntryKey = symTab.insert(K_RECEIVE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  REPLY
******************************************************************************/

tReply:
  K_REPLY genSymTabEntry_Reply
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_REPLY);

      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT 
  standardElements 
  tCorrelations
  X_SLASH K_REPLY
    { $$ = Reply($6, $7);
      STReply * stReply = NULL;
      try
      {
	stReply = dynamic_cast<STReply*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stReply->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stReply->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stReply->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 false);

     $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_REPLY genSymTabEntry_Reply
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_REPLY);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      STReply * stReply = NULL;
      try
      {
	stReply = dynamic_cast<STReply*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stReply->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stReply->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));
      stReply->channelId = stProcess->addChannel(channelName(symTab.readAttributeValue($2, "portType"), 
								 symTab.readAttributeValue($2, "operation"), 
								 symTab.readAttributeValue($2, "partnerLink")),
								 false);

      $$ = Reply(noLinks, NiltCorrelation_list());
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Reply:
  { currentSymTabEntryKey = symTab.insert(K_REPLY);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  ASSIGN
******************************************************************************/

tAssign:
  K_ASSIGN genSymTabEntry_Assign
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_ASSIGN);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements tCopy_list  X_SLASH K_ASSIGN
    { $$ = Assign($6, $7);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
;

genSymTabEntry_Assign:
  { currentSymTabEntryKey = symTab.insert(K_ASSIGN);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tCopy_list:
  tCopy X_NEXT
    { $$ = ConstCopy_list($1, NiltCopy_list()); }
| tCopy X_NEXT tCopy_list
    { $$ = ConstCopy_list($1, $3); }
;

tCopy:
  K_COPY X_NEXT tFrom X_NEXT tTo X_NEXT X_SLASH K_COPY
    { currentSymTabEntryKey = symTab.insert(K_COPY);
      $$ = Copy($3, $5);
    }
; 

tFrom:
  K_FROM genSymTabEntry_From arbitraryAttributes X_NEXT X_SLASH K_FROM
    { symTab.checkAttributes($2); //att.check($3, K_FROM);
      STFromTo * stFrom = NULL;
      try
      {
	stFrom = dynamic_cast<STFromTo*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stFrom->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stFrom->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));

      $$ = From();
      $$->id = $2;      
    }
| K_FROM genSymTabEntry_From arbitraryAttributes X_CLOSE X_NAME X_OPEN X_SLASH K_FROM
    { symTab.checkAttributes($2, $5); //att.check($3, $5, K_FROM);
      STFromTo * stFrom = NULL;
      try
      {
	stFrom = dynamic_cast<STFromTo*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stFrom->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stFrom->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));

      $$ = From();
      $$->id = $2;      
      stFrom->literal = $5->name;
    }
| K_FROM genSymTabEntry_From arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_FROM);
      STFromTo * stFrom = NULL;
      try
      {
	stFrom = dynamic_cast<STFromTo*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stFrom->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stFrom->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));

      $$ = From();
      $$->id = $2;      
    }
;

genSymTabEntry_From:
  { currentSymTabEntryKey = symTab.insert(K_FROM);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tTo:
  K_TO genSymTabEntry_To arbitraryAttributes X_NEXT X_SLASH K_TO
    { symTab.checkAttributes($2); //att.check($3, K_TO);
      STFromTo * stTo = NULL;
      try
      {
	stTo = dynamic_cast<STFromTo*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stTo->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stTo->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));

      $$ = To();
      $$->id = $2;      
    }
| K_TO genSymTabEntry_To arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_TO);
      STFromTo * stTo = NULL;
      try
      {
	stTo = dynamic_cast<STFromTo*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stTo->variable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);
      stTo->partnerLink = stProcess->checkPartnerLink(symTab.readAttributeValue($2, "partnerLink"));

      $$ = To();
      $$->id = $2;      
    }
;

genSymTabEntry_To:
  { currentSymTabEntryKey = symTab.insert(K_TO);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  WAIT
******************************************************************************/

tWait:
  K_WAIT genSymTabEntry_Wait
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_WAIT);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_WAIT
    {
      if ( symTab.readAttributeValue($2, "for") != "" )
        $$ = WaitFor($6);
      else
        $$ = WaitUntil($6);

      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();      

      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_WAIT genSymTabEntry_Wait
  arbitraryAttributes
    {
      symTab.checkAttributes($2); //att.check($3, K_WAIT);

      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;

      if ( symTab.readAttributeValue($2, "for") != "" )
        $$ = WaitFor(noLinks);
      else
        $$ = WaitUntil(noLinks);

      att.popSJFStack(); symTab.popSJFStack();      

      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Wait:
  { currentSymTabEntryKey = symTab.insert(K_WAIT);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  THROW
******************************************************************************/

tThrow:
  K_THROW genSymTabEntry_Throw
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_THROW);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_THROW
    { symTab.checkAttributes($2); //att.check($3, K_THROW);
      STThrow * stThrow = NULL;
      try
      {
	stThrow = dynamic_cast<STThrow*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stThrow->faultVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);

      $$ = Throw($6);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();      
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_THROW genSymTabEntry_Throw
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_THROW);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      STThrow * stThrow = NULL;
      try
      {
	stThrow = dynamic_cast<STThrow*> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stThrow->faultVariable = currentSTScope->checkVariable(symTab.readAttributeValue($2, "variable"),currentSTScope);

      $$ = Throw(noLinks);
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Throw:
  { currentSymTabEntryKey = symTab.insert(K_THROW);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  COMPENSATE
******************************************************************************/

tCompensate:
  K_COMPENSATE genSymTabEntry_Compensate
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_COMPENSATE);

      // compensate only is allowed within Compensation- or FaultHandler
      if ( ! (isInCH.top().first || isInFH.top()) )
      {
        yyerror(string("The activity <compensate> is only allowed with in a Compensation of FaultHandler.\n").c_str());
      }
      if(string(att.read($3, "scope")->name) == "")
      {
        if ( hasCompensate > 0)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 1;
      }
      else
      {
        if ( hasCompensate == 1)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 2;
      }

      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_COMPENSATE
    { $$ = Compensate($6);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_COMPENSATE genSymTabEntry_Compensate
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_COMPENSATE);

      // compensate only is allowed within Compensation- or FaultHandler
      if ( ! (isInCH.top().first || isInFH.top()) )
      {
        yyerror(string("The activity <compensate> is only allowed with in a Compensation of FaultHandler.\n").c_str());
      }
      if(string(att.read($3, "scope")->name) == "")
      {
        if ( hasCompensate > 0)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 1;
      }
      else
      {
        if ( hasCompensate == 1)
        {
          yyerror(string("You may only define one <compensate> if you're using it without attributes.\n").c_str());
        }
        hasCompensate = 2;
      }

      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Compensate(noLinks);
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Compensate:
  { currentSymTabEntryKey = symTab.insert(K_COMPENSATE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  TERMINATE
******************************************************************************/

tTerminate:
  K_TERMINATE genSymTabEntry_Terminate
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_TERMINATE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT standardElements X_SLASH K_TERMINATE
    { $$ = Terminate($6);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; }
| K_TERMINATE genSymTabEntry_Terminate
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_TERMINATE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_SLASH
    { impl_standardElements_StandardElements *noLinks = StandardElements(NiltTarget_list(), NiltSource_list());
      noLinks->parentId = $2;
      $$ = Terminate(noLinks);
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = noLinks->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $2; }
;

genSymTabEntry_Terminate:
  { currentSymTabEntryKey = symTab.insert(K_TERMINATE);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

/******************************************************************************
  FLOW
******************************************************************************/

tFlow:
  K_FLOW genSymTabEntry_Flow
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_FLOW);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }   
  X_NEXT
    {
      STFlow * stFlow = NULL;
      try
      {
	stFlow = dynamic_cast<STFlow *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      if (currentSTFlow != NULL)
      {
	stFlow->parentFlowId = currentSTFlow->entryKey;
      }
      currentSTFlow = stFlow;
//CG      symMan.newFlowScope($2);
    } 
  standardElements tLinks activity_list X_SLASH K_FLOW
    { $$ = Flow($7, $8, $9);

      STFlow * stFlow = NULL;
      try
      {
	stFlow = dynamic_cast<STFlow *> (symTab.lookup($2));
	if (currentSTFlow->parentFlowId != 0)
	{
	  currentSTFlow = dynamic_cast<STFlow *> (symTab.lookup(currentSTFlow->parentFlowId));
	}
	else
	{
	  currentSTFlow = NULL;
	}
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stFlow->checkLinkUsage();

      $7->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      $$->dpe = mkinteger((symMan.needsDPE())->value);
      if ($7->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($7->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      if ($$->dpe->value > 0)
      {
	$7->dpe = mkinteger(1);
      }
      $$->negativeControlFlow = $7->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $7->parentId = $2;
//CG      symMan.checkLinks();
//CG      symMan.quitScope(); 
    }
;

genSymTabEntry_Flow:
  { currentSymTabEntryKey = symTab.insert(K_FLOW);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

activity_list:
  descent_activity_list activity X_NEXT
    { $$ = Consactivity_list($2, Nilactivity_list()); 
      for(int i = 0; i < $1->value; ++i)
      { 
	symMan.addDPEend();
      }
      $$->dpe = $2->dpe;
    }
| descent_activity_list activity X_NEXT activity_list
    { $$ = Consactivity_list($2, $4); 
      for(int i = 0; i < $1->value; ++i)
      { 
	symMan.addDPEend();
      }
      $$->dpe = mkinteger($2->dpe->value + $4->dpe->value);
    }
;

descent_activity_list:
    {
      $$ = symMan.needsDPE();
      symMan.resetDPEend();
    }
;

tLinks:
  /* empty */
    { $$ = NiltLink_list(); }
| K_LINKS arbitraryAttributes X_NEXT tLink_list X_SLASH K_LINKS X_NEXT
    { $$ = $4; }
;

tLink_list:
  tLink X_NEXT
    { $$ = ConstLink_list($1, NiltLink_list()); }
| tLink X_NEXT tLink_list
    { $$ = ConstLink_list($1, $3); }
;

tLink:
  K_LINK genSymTabEntry_Link
  arbitraryAttributes X_NEXT X_SLASH K_LINK
    { symTab.checkAttributes($2);
      $$ = Link();

      STLink * stLink = NULL;
      try
      {
	stLink = dynamic_cast<STLink *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stLink->name = currentSTFlow->addLink(stLink);

      $$->id = $2;
//CG      symMan.addLink(new csLink(symTab.readAttributeValue($2, "name"))); 
    }
| K_LINK genSymTabEntry_Link
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2);
      $$ = Link();

      STLink * stLink = NULL;
      try
      {
	stLink = dynamic_cast<STLink *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stLink->name = currentSTFlow->addLink(stLink);

      $$->id = $2;
//CG      symMan.addLink(new csLink(symTab.readAttributeValue($2, "name"))); 
    }
;

genSymTabEntry_Link:
  { currentSymTabEntryKey = symTab.insert(K_LINK);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  SWITCH
******************************************************************************/

tSwitch:
  K_SWITCH genSymTabEntry_Switch
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_SWITCH);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    } 
  X_NEXT 
  standardElements 
    {
      symMan.addDPEstart();
    }
  tCase_list 
  tOtherwise 
  X_SLASH K_SWITCH
    { $$ = Switch($6, $8, $9);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure",  (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      symMan.remDPEstart();
      $$->dpe = symMan.needsDPE();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      if ($$->dpe->value > 0)
      {
	$6->dpe = mkinteger(1);
      }
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2;
    }
;

genSymTabEntry_Switch:
  { currentSymTabEntryKey = symTab.insert(K_SWITCH);
    $$ = mkinteger(currentSymTabEntryKey);
  }
;

tCase_list:
  descent_case_list tCase X_NEXT
    { $$ = ConstCase_list($2, NiltCase_list()); 
      for(int i = 0; i < $1->value; ++i)
      { 
	symMan.addDPEend();
      }
      $$->dpe = $2->dpe;
    }
| descent_case_list tCase X_NEXT tCase_list
    { $$ = ConstCase_list($2, $4); 
      for(int i = 0; i < $1->value; ++i)
      { 
	symMan.addDPEend();
      }
      $$->dpe = mkinteger($2->dpe->value + $4->dpe->value);
    }
;

descent_case_list:
    {
      $$ = symMan.needsDPE();
      symMan.resetDPEend();
    }
;

tCase:
  K_CASE genSymTabEntry_Case
  arbitraryAttributes X_NEXT 
    { symTab.checkAttributes($2); //att.check($3, K_CASE);
      // since we descend, set DPE ends to 0
      // symMan.resetDPEend();
    }
  activity 
  X_NEXT X_SLASH K_CASE
    { $$ = Case($6);
      $$->id = $2;    
      $$->condition = att.read($3, "condition"); 
      $$->dpe = mkinteger((symMan.needsDPE())->value);
      
    }
;

genSymTabEntry_Case:
  { currentSymTabEntryKey = symTab.insert(K_CASE);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;

tOtherwise:
  /* If the otherwise branch is not explicitly specified, then an otherwise
     branch with an empty activity is deemed to be present. */
    { // creaty empty activit with id, without links etc.
      int otherwiseId = currentSymTabEntryKey = symTab.insert(K_OTHERWISE);
      currentSymTabEntry = symTab.lookup(currentSymTabEntryKey); 
      int emptyId = currentSymTabEntryKey = symTab.insert(K_EMPTY);
      currentSymTabEntry = symTab.lookup(currentSymTabEntryKey); 

      impl_standardElements_StandardElements* noLinks = StandardElements(NiltTarget_list(),NiltSource_list());
//      noLinks->dpe = kc::mkinteger(0);
      noLinks->parentId = kc::mkinteger(emptyId);
      impl_tEmpty_Empty* implicitEmpty = Empty(noLinks);
      implicitEmpty->id = kc::mkinteger(emptyId);
      implicitEmpty->negativeControlFlow = noLinks->negativeControlFlow = mkinteger(0);
//      implicitEmpty->dpe = mkinteger(0);
      impl_activity *otherwiseActivity = activityEmpty(implicitEmpty);
      otherwiseActivity->id = kc::mkinteger(emptyId);

      $$ = Otherwise(otherwiseActivity);
      $$->dpe = kc::mkinteger(0);
    }
| K_OTHERWISE X_NEXT 
    {
      // since we descend, set DPE ends to 0
      symMan.resetDPEend();
    }
  activity X_NEXT X_SLASH K_OTHERWISE X_NEXT
    { $$ = Otherwise($4); 
      $$->dpe = symMan.needsDPE();
    }
;


/******************************************************************************
  WHILE
******************************************************************************/

tWhile:
  K_WHILE genSymTabEntry_While
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_WHILE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }    
  X_NEXT 
  standardElements 
    { 
      symMan.startDPEinWhile();
    }
  activity 
  X_NEXT X_SLASH K_WHILE
    { // symTab.checkAttributes($3); att.check($3, K_WHILE);
      $$ = While($6, $8);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      $$->condition = att.read($3, "condition");
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; 
      symMan.endDPEinWhile();
    }
;

genSymTabEntry_While:
  { currentSymTabEntryKey = symTab.insert(K_WHILE);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  SEQUENCE
******************************************************************************/

tSequence:
  K_SEQUENCE genSymTabEntry_Sequence
  arbitraryAttributes 
    { symTab.checkAttributes($2); //att.check($3, K_SEQUENCE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }
  X_NEXT 
  standardElements 
  activity_list 
  X_SLASH 
  K_SEQUENCE
    { $$ = Sequence($6, $7);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $6->parentId = $2; 
      $$->dpe = mkinteger((symMan.needsDPE())->value);
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      if ($$->dpe->value > 0)
      {
	$6->dpe = mkinteger(1);
      }
    }
;

genSymTabEntry_Sequence:
  { currentSymTabEntryKey = symTab.insert(K_SEQUENCE);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  PICK
******************************************************************************/

tPick:
  K_PICK genSymTabEntry_Pick
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_PICK);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }
  X_NEXT 
  standardElements 
    {
      symMan.addDPEstart();
    }
  tOnMessage X_NEXT 
  tOnMessage_list 
  tOnAlarm_list 
  X_SLASH K_PICK
    { $$ = Pick($6, ConstOnMessage_list($8, $10), $11);
      $6->suppressJoinFailure = att.read($3, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      symMan.remDPEstart();
      $$->dpe = symMan.needsDPE();
      if ($6->hasTarget)
      {
	symMan.remDPEstart();
      }
      if ($6->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      if ($$->dpe->value > 0)
      {
	$6->dpe = mkinteger(1);
      }
      $$->id = $6->parentId = $2;
      $$->negativeControlFlow = $6->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
    }
;

genSymTabEntry_Pick:
  { currentSymTabEntryKey = symTab.insert(K_PICK);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  SCOPE
******************************************************************************/

tScope:
  K_SCOPE genSymTabEntry_Scope
  arbitraryAttributes
    { symTab.checkAttributes($2); //att.check($3, K_SCOPE);
      if(att.isAttributeValueEmpty($3, "suppressJoinFailure"))
      {
      	/// parent BPEL-element attribute value
      	att.pushSJFStack($3, (att.topSJFStack()).getSJFValue());
      }
      else
      {
        /// current BPEL-element attribute value
      	att.pushSJFStack($3, att.read($3, "suppressJoinFailure"));      
      }
    }  
  X_NEXT
    { 
//CG      symMan.newScopeScope($2);
//CG      symMan.setBlackListMode(true);
      isInFH.push(false);
      isInCH.push(pair<bool,int>(false,hasCompensate));
      parent[$2] = currentScopeId;
      currentScopeId = $2; 
    }
  standardElements 
    {
//CG      symMan.setBlackListMode(false);
      // should come after standardElements in order to prevent links from 
      // being added to enclosedLinks list of this scope
      currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentScopeId->value));
      currentSTScope->parentScopeId = parent[$2]->value;
      (dynamic_cast<STScope *> (symTab.lookup(currentSTScope->parentScopeId)))->childScopes.push_back(currentSTScope);
    }
  tVariables 
  tCorrelationSets 
  tFaultHandlers 
  tCompensationHandler 
  tEventHandlers 
  activity 
  X_NEXT X_SLASH K_SCOPE
    { isInFH.pop();
      hasCompensate = isInCH.top().second;
      isInCH.pop();
      $$ = Scope($7, $9, $11, $12, $13, StopInScope(), $14);
      $7->suppressJoinFailure = att.read($3, "suppressJoinFailure", (att.topSJFStack()).getSJFValue());
      att.popSJFStack(); symTab.popSJFStack();
      $$->negativeControlFlow = $7->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      $$->id = $7->parentId = $2;
      $$->parentScopeId = currentScopeId = parent[$2];
      currentSTScope = dynamic_cast<STScope *> (symTab.lookup(currentScopeId->value));
      $$->dpe = mkinteger((symMan.needsDPE())->value);
      if ($7->hasTarget)
      {
	symMan.remDPEstart();
      }
      ((STScope*)symTab.lookup($2))->hasEventHandler = (string($13->op_name()) == "userDefinedEventHandler");
      if ($7->dpe->value > 0)
      {
        symMan.addDPEend();
      }
      if ($$->dpe->value > 0)
      {
	$7->dpe = mkinteger(1);
      }
//CG      symMan.quitScope(); 
    }
;

genSymTabEntry_Scope:
  { currentSymTabEntryKey = symTab.insert(K_SCOPE);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;


/******************************************************************************
  STANDARD ELEMENTS
******************************************************************************/

standardElements:
  tTarget_list tSource_list
    { $$ = StandardElements($1, $2); 
      $$->dpe = $2->dpe;
      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
      if ($1->length() > 0)
      {
	symMan.addDPEstart();
	$$->hasTarget = true;
      }
    }
;

tTarget_list:
  /* empty */
    { $$ = NiltTarget_list(); 
    }
| tTarget X_NEXT tTarget_list
    { $$ = ConstTarget_list($1, $3); 
    }
;

tTarget:
  K_TARGET genSymTabEntry_Target
  arbitraryAttributes X_NEXT X_SLASH K_TARGET
    { symTab.checkAttributes($2); //att.check($3, K_TARGET);
      $$ = Target();
      $$->id = $2;      

      STSourceTarget * stTarget = NULL;
      try
      {
	stTarget = dynamic_cast<STSourceTarget *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stTarget->link = currentSTFlow->checkLink(symTab.readAttributeValue($2, "linkName"), $2->value, false);
      
      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
//CG      $$->linkID = symMan.checkLink(symTab.readAttributeValue($2, "linkName"), false);
}
| K_TARGET genSymTabEntry_Target
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_TARGET);
      $$ = Target();
      $$->id = $2;      

      STSourceTarget * stTarget = NULL;
      try
      {
	stTarget = dynamic_cast<STSourceTarget *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stTarget->link = currentSTFlow->checkLink(symTab.readAttributeValue($2, "linkName"), $2->value, false);

      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
//CG      $$->linkID = symMan.checkLink(symTab.readAttributeValue($2, "linkName"), false);
}
;

genSymTabEntry_Target:
  { currentSymTabEntryKey = symTab.insert(K_TARGET);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;

tSource_list:
  /* empty */
    { $$ = NiltSource_list(); 
      $$->dpe = kc::mkinteger(0);
    }
| tSource X_NEXT tSource_list
    { $$ = ConstSource_list($1, $3);
      $$->dpe = kc::mkinteger($1->dpe->value + $3->dpe->value);
    }
;

tSource:
  K_SOURCE genSymTabEntry_Source
  arbitraryAttributes X_NEXT X_SLASH K_SOURCE
    { symTab.checkAttributes($2); //att.check($3, K_SOURCE);
      $$ = Source();
      $$->id = $2;      

      STSourceTarget * stSource = NULL;
      try
      {
	stSource = dynamic_cast<STSourceTarget *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stSource->link = currentSTFlow->checkLink(symTab.readAttributeValue($2, "linkName"), $2->value, true);
      currentSTScope->addLink(stSource->link);

//CG      $$->linkID = symMan.checkLink(symTab.readAttributeValue($2, "linkName"), true); 
      symMan.addDPEend();
      $$->dpe = symMan.needsDPE();
      symMan.remDPEend();
      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
    }
| K_SOURCE genSymTabEntry_Source
  arbitraryAttributes X_SLASH
    { symTab.checkAttributes($2); //att.check($3, K_SOURCE);
      $$ = Source();
      $$->id = $2;      

      STSourceTarget * stSource = NULL;
      try
      {
	stSource = dynamic_cast<STSourceTarget *> (symTab.lookup($2));
      }
      catch (bad_cast)
      {
	throw Exception(CHECK_SYMBOLS_CAST_ERROR, "Could not cast correctly", pos(__FILE__, __LINE__, __FUNCTION__));
      }
      stSource->link = currentSTFlow->checkLink(symTab.readAttributeValue($2, "linkName"), $2->value, true);
      currentSTScope->addLink(stSource->link);

//CG      $$->linkID = symMan.checkLink(symTab.readAttributeValue($2, "linkName"), true);
      symMan.addDPEend();
      $$->dpe = symMan.needsDPE();
      symMan.remDPEend();
      $$->negativeControlFlow = mkinteger( ((int) isInFH.top()) + 2*((int) isInCH.top().first));
    }
;

genSymTabEntry_Source:
  { currentSymTabEntryKey = symTab.insert(K_SOURCE);
	$$ = mkinteger(currentSymTabEntryKey);
  }
;

/*---------------------------------------------------------------------------*/

arbitraryAttributes:
  /* empty */
    { $$ = att.nextId();
    }
| X_NAME X_EQUALS X_STRING
    {
      if(currentSymTabEntryKey > 0)
        symTab.addAttribute(currentSymTabEntryKey, symTab.newAttribute($1, $3));
    }
  arbitraryAttributes
    {
      att.define($1, $3);
      $$ = $5;
    }
;
