// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use rand::rngs::StdRng;
use rand::{Rng, SeedableRng};

#[derive(Debug, Clone)]
pub struct DistanceKDE {
    bins: &'static [f64],
    cdf: &'static [f64],
}

impl DistanceKDE {
    pub fn new(bins: &'static [f64], cdf: &'static [f64]) -> Self {
        Self { bins, cdf }
    }

    pub fn generate(&self, seed: u64) -> f64 {
        let mut rng = StdRng::seed_from_u64(seed);
        let u: f64 = rng.gen();

        let mut low = 0;
        let mut high = self.cdf.len() - 1;

        while low < high {
            let mid = (low + high) / 2;
            if self.cdf[mid] < u {
                low = mid + 1;
            } else {
                high = mid;
            }
        }

        self.bins[low]
    }
}

// Replace with real 100K values in your actual use
static CDF_VALUES: [f64; 100000] = [
    0.0,
    1.000010000100001e-5,
    2.000020000200002e-5,
    3.000030000300003e-5,
    4.000040000400004e-5,
    5.0000500005000046e-5,
    6.000060000600006e-5,
    7.000070000700007e-5,
    8.000080000800008e-5,
    9.000090000900009e-5,
    0.00010000100001000009,
    0.00011000110001100011,
    0.00012000120001200012,
    0.00013000130001300012,
    0.00014000140001400014,
    0.00015000150001500014,
    0.00016000160001600016,
    0.00017000170001700018,
    0.00018000180001800017,
    0.0001900019000190002,
    0.00020000200002000018,
    0.0002100021000210002,
    0.00022000220002200022,
    0.00023000230002300022,
    0.00024000240002400024,
    0.00025000250002500023,
    0.00026000260002600025,
    0.00027000270002700027,
    0.0002800028000280003,
    0.0002900029000290003,
    0.0003000030000300003,
    0.0003100031000310003,
    0.0003200032000320003,
    0.00033000330003300033,
    0.00034000340003400035,
    0.0003500035000350003,
    0.00036000360003600034,
    0.00037000370003700036,
    0.0003800038000380004,
    0.0003900039000390004,
    0.00040000400004000037,
    0.0004100041000410004,
    0.0004200042000420004,
    0.0004300043000430004,
    0.00044000440004400045,
    0.00045000450004500047,
    0.00046000460004600043,
    0.00047000470004700045,
    0.00048000480004800047,
    0.0004900049000490004,
    0.0005000050000500005,
    0.0005100051000510005,
    0.0005200052000520005,
    0.0005300053000530005,
    0.0005400054000540005,
    0.0005500055000550006,
    0.0005600056000560006,
    0.0005700057000570006,
    0.0005800058000580006,
    0.0005900059000590005,
    0.0006000060000600005,
    0.0006100061000610006,
    0.0006200062000620006,
    0.0006300063000630006,
    0.0006400064000640006,
    0.0006500065000650006,
    0.0006600066000660007,
    0.0006700067000670007,
    0.0006800068000680007,
    0.0006900069000690007,
    0.0007000070000700006,
    0.0007100071000710007,
    0.0007200072000720007,
    0.0007300073000730007,
    0.0007400074000740007,
    0.0007500075000750007,
    0.0007600076000760008,
    0.0007700077000770008,
    0.0007800078000780008,
    0.0007900079000790008,
    0.0008000080000800007,
    0.0008100081000810008,
    0.0008200082000820008,
    0.0008300083000830008,
    0.0008400084000840008,
    0.0008500085000850008,
    0.0008600086000860009,
    0.0008700087000870009,
    0.0008800088000880009,
    0.0008900089000890009,
    0.0009000090000900009,
    0.0009100091000910008,
    0.0009200092000920009,
    0.0009300093000930009,
    0.0009400094000940009,
    0.0009500095000950009,
    0.0009600096000960009,
    0.000970009700097001,
    0.0009800098000980009,
    0.000990009900099001,
    0.001000010000100001,
    0.001010010100101001,
    0.001020010200102001,
    0.001030010300103001,
    0.001040010400104001,
    0.0010500105001050011,
    0.001060010600106001,
    0.001070010700107001,
    0.001080010800108001,
    0.001090010900109001,
    0.0011000110001100011,
    0.001110011100111001,
    0.0011200112001120012,
    0.001130011300113001,
    0.0011400114001140012,
    0.001150011500115001,
    0.0011600116001160012,
    0.0011700117001170011,
    0.001180011800118001,
    0.0011900119001190012,
    0.001200012000120001,
    0.0012100121001210012,
    0.0012200122001220011,
    0.0012300123001230013,
    0.0012400124001240012,
    0.0012500125001250013,
    0.0012600126001260012,
    0.0012700127001270013,
    0.0012800128001280013,
    0.0012900129001290012,
    0.0013000130001300013,
    0.0013100131001310012,
    0.0013200132001320013,
    0.0013300133001330012,
    0.0013400134001340014,
    0.0013500135001350013,
    0.0013600136001360014,
    0.0013700137001370013,
    0.0013800138001380015,
    0.0013900139001390014,
    0.0014000140001400013,
    0.0014100141001410014,
    0.0014200142001420013,
    0.0014300143001430014,
    0.0014400144001440014,
    0.0014500145001450015,
    0.0014600146001460014,
    0.0014700147001470015,
    0.0014800148001480014,
    0.0014900149001490016,
    0.0015000150001500015,
    0.0015100151001510014,
    0.0015200152001520015,
    0.0015300153001530014,
    0.0015400154001540016,
    0.0015500155001550015,
    0.0015600156001560016,
    0.0015700157001570015,
    0.0015800158001580016,
    0.0015900159001590016,
    0.0016000160001600015,
    0.0016100161001610016,
    0.0016200162001620015,
    0.0016300163001630016,
    0.0016400164001640015,
    0.0016500165001650017,
    0.0016600166001660016,
    0.0016700167001670017,
    0.0016800168001680016,
    0.0016900169001690018,
    0.0017000170001700017,
    0.0017100171001710016,
    0.0017200172001720017,
    0.0017300173001730016,
    0.0017400174001740017,
    0.0017500175001750017,
    0.0017600176001760018,
    0.0017700177001770017,
    0.0017800178001780018,
    0.0017900179001790017,
    0.0018000180001800019,
    0.0018100181001810018,
    0.0018200182001820017,
    0.0018300183001830018,
    0.0018400184001840017,
    0.0018500185001850019,
    0.0018600186001860018,
    0.001870018700187002,
    0.0018800188001880018,
    0.001890018900189002,
    0.0019000190001900018,
    0.001910019100191002,
    0.0019200192001920019,
    0.0019300193001930018,
    0.001940019400194002,
    0.0019500195001950018,
    0.0019600196001960018,
    0.001970019700197002,
    0.001980019800198002,
    0.001990019900199002,
    0.002000020000200002,
    0.002010020100201002,
    0.002020020200202002,
    0.0020300203002030018,
    0.002040020400204002,
    0.002050020500205002,
    0.002060020600206002,
    0.002070020700207002,
    0.002080020800208002,
    0.002090020900209002,
    0.0021000210002100022,
    0.002110021100211002,
    0.002120021200212002,
    0.002130021300213002,
    0.002140021400214002,
    0.002150021500215002,
    0.002160021600216002,
    0.0021700217002170023,
    0.002180021800218002,
    0.002190021900219002,
    0.0022000220002200022,
    0.0022100221002210024,
    0.002220022200222002,
    0.002230022300223002,
    0.0022400224002240023,
    0.002250022500225002,
    0.002260022600226002,
    0.0022700227002270023,
    0.0022800228002280024,
    0.002290022900229002,
    0.002300023000230002,
    0.0023100231002310023,
    0.0023200232002320025,
    0.002330023300233002,
    0.0023400234002340023,
    0.0023500235002350024,
    0.002360023600236002,
    0.0023700237002370022,
    0.0023800238002380024,
    0.0023900239002390025,
    0.002400024000240002,
    0.0024100241002410023,
    0.0024200242002420025,
    0.0024300243002430026,
    0.0024400244002440023,
    0.0024500245002450024,
    0.0024600246002460025,
    0.0024700247002470022,
    0.0024800248002480024,
    0.0024900249002490025,
    0.0025000250002500026,
    0.0025100251002510023,
    0.0025200252002520024,
    0.0025300253002530026,
    0.0025400254002540027,
    0.0025500255002550024,
    0.0025600256002560025,
    0.0025700257002570026,
    0.0025800258002580023,
    0.0025900259002590025,
    0.0026000260002600026,
    0.0026100261002610027,
    0.0026200262002620024,
    0.0026300263002630025,
    0.0026400264002640027,
    0.002650026500265003,
    0.0026600266002660025,
    0.0026700267002670026,
    0.0026800268002680028,
    0.0026900269002690025,
    0.0027000270002700026,
    0.0027100271002710027,
    0.002720027200272003,
    0.0027300273002730025,
    0.0027400274002740027,
    0.002750027500275003,
    0.002760027600276003,
    0.0027700277002770026,
    0.0027800278002780027,
    0.002790027900279003,
    0.0028000280002800026,
    0.0028100281002810027,
    0.002820028200282003,
    0.002830028300283003,
    0.0028400284002840026,
    0.0028500285002850028,
    0.002860028600286003,
    0.002870028700287003,
    0.0028800288002880027,
    0.002890028900289003,
    0.002900029000290003,
    0.0029100291002910027,
    0.002920029200292003,
    0.002930029300293003,
    0.002940029400294003,
    0.0029500295002950028,
    0.002960029600296003,
    0.002970029700297003,
    0.002980029800298003,
    0.002990029900299003,
    0.003000030000300003,
    0.003010030100301003,
    0.003020030200302003,
    0.003030030300303003,
    0.003040030400304003,
    0.003050030500305003,
    0.003060030600306003,
    0.003070030700307003,
    0.003080030800308003,
    0.0030900309003090033,
    0.003100031000310003,
    0.003110031100311003,
    0.003120031200312003,
    0.003130031300313003,
    0.003140031400314003,
    0.003150031500315003,
    0.0031600316003160033,
    0.003170031700317003,
    0.003180031800318003,
    0.0031900319003190032,
    0.003200032000320003,
    0.003210032100321003,
    0.003220032200322003,
    0.0032300323003230033,
    0.003240032400324003,
    0.003250032500325003,
    0.0032600326003260033,
    0.0032700327003270034,
    0.003280032800328003,
    0.003290032900329003,
    0.0033000330003300033,
    0.003310033100331003,
    0.003320033200332003,
    0.0033300333003330033,
    0.0033400334003340034,
    0.003350033500335003,
    0.0033600336003360032,
    0.0033700337003370034,
    0.0033800338003380035,
    0.003390033900339003,
    0.0034000340003400033,
    0.0034100341003410035,
    0.003420034200342003,
    0.0034300343003430033,
    0.0034400344003440034,
    0.0034500345003450035,
    0.0034600346003460032,
    0.0034700347003470034,
    0.0034800348003480035,
    0.0034900349003490036,
    0.0035000350003500033,
    0.0035100351003510034,
    0.0035200352003520036,
    0.0035300353003530033,
    0.0035400354003540034,
    0.0035500355003550035,
    0.0035600356003560036,
    0.0035700357003570033,
    0.0035800358003580035,
    0.0035900359003590036,
    0.0036000360003600037,
    0.0036100361003610034,
    0.0036200362003620036,
    0.0036300363003630037,
    0.0036400364003640034,
    0.0036500365003650035,
    0.0036600366003660036,
    0.0036700367003670038,
    0.0036800368003680035,
    0.0036900369003690036,
    0.0037000370003700037,
    0.003710037100371004,
    0.0037200372003720035,
    0.0037300373003730037,
    0.003740037400374004,
    0.0037500375003750035,
    0.0037600376003760036,
    0.0037700377003770037,
    0.003780037800378004,
    0.0037900379003790036,
    0.0038000380003800037,
    0.003810038100381004,
    0.003820038200382004,
    0.0038300383003830036,
    0.0038400384003840038,
    0.003850038500385004,
    0.0038600386003860036,
    0.0038700387003870037,
    0.003880038800388004,
    0.003890038900389004,
    0.0039000390003900037,
    0.003910039100391004,
    0.0039200392003920035,
    0.003930039300393004,
    0.003940039400394004,
    0.003950039500395004,
    0.003960039600396004,
    0.003970039700397004,
    0.003980039800398004,
    0.0039900399003990035,
    0.004000040000400004,
    0.004010040100401004,
    0.004020040200402004,
    0.004030040300403004,
    0.004040040400404004,
    0.004050040500405004,
    0.0040600406004060036,
    0.004070040700407004,
    0.004080040800408004,
    0.004090040900409004,
    0.004100041000410004,
    0.004110041100411004,
    0.004120041200412004,
    0.0041300413004130045,
    0.004140041400414004,
    0.004150041500415004,
    0.004160041600416004,
    0.004170041700417004,
    0.004180041800418004,
    0.004190041900419004,
    0.0042000420004200045,
    0.004210042100421004,
    0.004220042200422004,
    0.004230042300423004,
    0.004240042400424004,
    0.004250042500425004,
    0.004260042600426004,
    0.0042700427004270045,
    0.004280042800428004,
    0.004290042900429004,
    0.004300043000430004,
    0.004310043100431004,
    0.004320043200432004,
    0.004330043300433004,
    0.0043400434004340046,
    0.004350043500435005,
    0.004360043600436004,
    0.004370043700437004,
    0.004380043800438004,
    0.004390043900439004,
    0.0044000440004400045,
    0.004410044100441005,
    0.004420044200442005,
    0.004430044300443004,
    0.004440044400444004,
    0.004450044500445004,
    0.004460044600446004,
    0.0044700447004470045,
    0.004480044800448005,
    0.004490044900449005,
    0.004500045000450004,
    0.004510045100451004,
    0.004520045200452004,
    0.004530045300453004,
    0.0045400454004540045,
    0.004550045500455005,
    0.004560045600456005,
    0.004570045700457005,
    0.004580045800458004,
    0.004590045900459004,
    0.004600046000460004,
    0.0046100461004610046,
    0.004620046200462005,
    0.004630046300463005,
    0.004640046400464005,
    0.004650046500465004,
    0.004660046600466004,
    0.0046700467004670045,
    0.004680046800468005,
    0.004690046900469005,
    0.004700047000470005,
    0.004710047100471005,
    0.004720047200472004,
    0.004730047300473004,
    0.0047400474004740045,
    0.004750047500475005,
    0.004760047600476005,
    0.004770047700477005,
    0.004780047800478005,
    0.004790047900479005,
    0.004800048000480004,
    0.0048100481004810045,
    0.004820048200482005,
    0.004830048300483005,
    0.004840048400484005,
    0.004850048500485005,
    0.004860048600486005,
    0.004870048700487004,
    0.0048800488004880046,
    0.004890048900489005,
    0.004900049000490005,
    0.004910049100491005,
    0.004920049200492005,
    0.004930049300493005,
    0.0049400494004940045,
    0.004950049500495005,
    0.004960049600496005,
    0.004970049700497005,
    0.004980049800498005,
    0.004990049900499005,
    0.005000050000500005,
    0.005010050100501005,
    0.005020050200502005,
    0.005030050300503005,
    0.005040050400504005,
    0.005050050500505005,
    0.005060050600506005,
    0.005070050700507005,
    0.005080050800508005,
    0.005090050900509005,
    0.005100051000510005,
    0.005110051100511005,
    0.005120051200512005,
    0.005130051300513005,
    0.005140051400514005,
    0.005150051500515005,
    0.005160051600516005,
    0.005170051700517005,
    0.005180051800518005,
    0.005190051900519005,
    0.005200052000520005,
    0.005210052100521005,
    0.0052200522005220054,
    0.005230052300523005,
    0.005240052400524005,
    0.005250052500525005,
    0.005260052600526005,
    0.005270052700527005,
    0.005280052800528005,
    0.0052900529005290055,
    0.005300053000530006,
    0.005310053100531005,
    0.005320053200532005,
    0.005330053300533005,
    0.005340053400534005,
    0.005350053500535005,
    0.0053600536005360055,
    0.005370053700537006,
    0.005380053800538005,
    0.005390053900539005,
    0.005400054000540005,
    0.005410054100541005,
    0.005420054200542005,
    0.0054300543005430055,
    0.005440054400544006,
    0.005450054500545005,
    0.005460054600546005,
    0.005470054700547005,
    0.005480054800548005,
    0.0054900549005490054,
    0.005500055000550006,
    0.005510055100551006,
    0.005520055200552006,
    0.005530055300553005,
    0.005540055400554005,
    0.005550055500555005,
    0.0055600556005560055,
    0.005570055700557006,
    0.005580055800558006,
    0.005590055900559006,
    0.005600056000560005,
    0.005610056100561005,
    0.005620056200562005,
    0.0056300563005630055,
    0.005640056400564006,
    0.005650056500565006,
    0.005660056600566006,
    0.005670056700567005,
    0.005680056800568005,
    0.005690056900569005,
    0.0057000570005700055,
    0.005710057100571006,
    0.005720057200572006,
    0.005730057300573006,
    0.005740057400574006,
    0.005750057500575005,
    0.0057600576005760054,
    0.005770057700577006,
    0.005780057800578006,
    0.005790057900579006,
    0.005800058000580006,
    0.005810058100581006,
    0.005820058200582005,
    0.0058300583005830055,
    0.005840058400584006,
    0.005850058500585006,
    0.005860058600586006,
    0.005870058700587006,
    0.005880058800588006,
    0.005890058900589005,
    0.0059000590005900055,
    0.005910059100591006,
    0.005920059200592006,
    0.005930059300593006,
    0.005940059400594006,
    0.005950059500595006,
    0.005960059600596006,
    0.0059700597005970055,
    0.005980059800598006,
    0.005990059900599006,
    0.006000060000600006,
    0.006010060100601006,
    0.006020060200602006,
    0.006030060300603006,
    0.006040060400604006,
    0.006050060500605006,
    0.006060060600606006,
    0.006070060700607006,
    0.006080060800608006,
    0.006090060900609006,
    0.006100061000610006,
    0.006110061100611006,
    0.006120061200612006,
    0.006130061300613006,
    0.006140061400614006,
    0.006150061500615006,
    0.006160061600616006,
    0.006170061700617006,
    0.0061800618006180065,
    0.006190061900619006,
    0.006200062000620006,
    0.006210062100621006,
    0.006220062200622006,
    0.006230062300623006,
    0.006240062400624006,
    0.0062500625006250065,
    0.006260062600626006,
    0.006270062700627006,
    0.006280062800628006,
    0.006290062900629006,
    0.006300063000630006,
    0.0063100631006310064,
    0.0063200632006320066,
    0.006330063300633006,
    0.006340063400634006,
    0.006350063500635006,
    0.006360063600636006,
    0.006370063700637006,
    0.0063800638006380065,
    0.006390063900639007,
    0.006400064000640006,
    0.006410064100641006,
    0.006420064200642006,
    0.006430064300643006,
    0.006440064400644006,
    0.0064500645006450065,
    0.006460064600646007,
    0.006470064700647007,
    0.006480064800648006,
    0.006490064900649006,
    0.006500065000650006,
    0.006510065100651006,
    0.0065200652006520065,
    0.006530065300653007,
    0.006540065400654007,
    0.006550065500655006,
    0.006560065600656006,
    0.006570065700657006,
    0.006580065800658006,
    0.0065900659006590066,
    0.006600066000660007,
    0.006610066100661007,
    0.006620066200662006,
    0.006630066300663006,
    0.006640066400664006,
    0.0066500665006650065,
    0.006660066600666007,
    0.006670066700667007,
    0.006680066800668007,
    0.006690066900669007,
    0.006700067000670006,
    0.006710067100671006,
    0.0067200672006720065,
    0.006730067300673007,
    0.006740067400674007,
    0.006750067500675007,
    0.006760067600676007,
    0.006770067700677006,
    0.006780067800678006,
    0.0067900679006790065,
    0.006800068000680007,
    0.006810068100681007,
    0.006820068200682007,
    0.006830068300683007,
    0.006840068400684006,
    0.006850068500685006,
    0.0068600686006860066,
    0.006870068700687007,
    0.006880068800688007,
    0.006890068900689007,
    0.006900069000690007,
    0.006910069100691007,
    0.0069200692006920065,
    0.006930069300693007,
    0.006940069400694007,
    0.006950069500695007,
    0.006960069600696007,
    0.006970069700697007,
    0.006980069800698007,
    0.0069900699006990065,
    0.007000070000700007,
    0.007010070100701007,
    0.007020070200702007,
    0.007030070300703007,
    0.007040070400704007,
    0.007050070500705007,
    0.0070600706007060065,
    0.007070070700707007,
    0.007080070800708007,
    0.007090070900709007,
    0.007100071000710007,
    0.007110071100711007,
    0.007120071200712007,
    0.007130071300713007,
    0.007140071400714007,
    0.007150071500715007,
    0.007160071600716007,
    0.007170071700717007,
    0.007180071800718007,
    0.007190071900719007,
    0.0072000720007200075,
    0.007210072100721007,
    0.007220072200722007,
    0.007230072300723007,
    0.007240072400724007,
    0.007250072500725007,
    0.007260072600726007,
    0.0072700727007270075,
    0.007280072800728007,
    0.007290072900729007,
    0.007300073000730007,
    0.007310073100731007,
    0.007320073200732007,
    0.007330073300733007,
    0.0073400734007340075,
    0.007350073500735008,
    0.007360073600736007,
    0.007370073700737007,
    0.007380073800738007,
    0.007390073900739007,
    0.007400074000740007,
    0.0074100741007410076,
    0.007420074200742008,
    0.007430074300743007,
    0.007440074400744007,
    0.007450074500745007,
    0.007460074600746007,
    0.0074700747007470075,
    0.007480074800748008,
    0.007490074900749008,
    0.007500075000750007,
    0.007510075100751007,
    0.007520075200752007,
    0.007530075300753007,
    0.0075400754007540075,
    0.007550075500755008,
    0.007560075600756008,
    0.007570075700757008,
    0.007580075800758007,
    0.007590075900759007,
    0.007600076000760007,
    0.0076100761007610075,
    0.007620076200762008,
    0.007630076300763008,
    0.007640076400764008,
    0.007650076500765007,
    0.007660076600766007,
    0.007670076700767007,
    0.0076800768007680075,
    0.007690076900769008,
    0.007700077000770008,
    0.007710077100771008,
    0.007720077200772007,
    0.007730077300773007,
    0.0077400774007740075,
    0.007750077500775008,
    0.007760077600776008,
    0.007770077700777008,
    0.007780077800778008,
    0.007790077900779007,
    0.007800078000780007,
    0.0078100781007810075,
    0.007820078200782008,
    0.007830078300783007,
    0.007840078400784007,
    0.007850078500785007,
    0.007860078600786007,
    0.007870078700787007,
    0.007880078800788008,
    0.007890078900789008,
    0.007900079000790008,
    0.007910079100791008,
    0.007920079200792008,
    0.007930079300793008,
    0.007940079400794008,
    0.007950079500795008,
    0.007960079600796009,
    0.007970079700797009,
    0.007980079800798007,
    0.007990079900799007,
    0.008000080000800007,
    0.008010080100801007,
    0.008020080200802008,
    0.008030080300803008,
    0.008040080400804008,
    0.008050080500805008,
    0.008060080600806008,
    0.008070080700807008,
    0.008080080800808008,
    0.008090080900809008,
    0.008100081000810009,
    0.008110081100811009,
    0.008120081200812007,
    0.008130081300813007,
    0.008140081400814007,
    0.008150081500815008,
    0.008160081600816008,
    0.008170081700817008,
    0.008180081800818008,
    0.008190081900819008,
    0.008200082000820008,
    0.008210082100821008,
    0.008220082200822008,
    0.008230082300823009,
    0.008240082400824009,
    0.008250082500825009,
    0.008260082600826009,
    0.008270082700827007,
    0.008280082800828007,
    0.008290082900829008,
    0.008300083000830008,
    0.008310083100831008,
    0.008320083200832008,
    0.008330083300833008,
    0.008340083400834008,
    0.008350083500835008,
    0.008360083600836008,
    0.008370083700837009,
    0.008380083800838009,
    0.008390083900839009,
    0.008400084000840009,
    0.008410084100841009,
    0.008420084200842008,
    0.008430084300843008,
    0.008440084400844008,
    0.008450084500845008,
    0.008460084600846008,
    0.008470084700847008,
    0.008480084800848008,
    0.008490084900849008,
    0.008500085000850009,
    0.008510085100851009,
    0.008520085200852009,
    0.008530085300853009,
    0.008540085400854009,
    0.00855008550085501,
    0.008560085600856008,
    0.008570085700857008,
    0.008580085800858008,
    0.008590085900859008,
    0.008600086000860008,
    0.008610086100861008,
    0.008620086200862008,
    0.008630086300863008,
    0.008640086400864009,
    0.008650086500865009,
    0.008660086600866009,
    0.008670086700867009,
    0.008680086800868009,
    0.00869008690086901,
    0.00870008700087001,
    0.008710087100871008,
    0.008720087200872008,
    0.008730087300873008,
    0.008740087400874008,
    0.008750087500875008,
    0.008760087600876008,
    0.008770087700877009,
    0.008780087800878009,
    0.008790087900879009,
    0.008800088000880009,
    0.008810088100881009,
    0.00882008820088201,
    0.00883008830088301,
    0.00884008840088401,
    0.00885008850088501,
    0.008860088600886008,
    0.008870088700887008,
    0.008880088800888008,
    0.008890088900889008,
    0.008900089000890008,
    0.008910089100891009,
    0.008920089200892009,
    0.008930089300893009,
    0.008940089400894009,
    0.008950089500895009,
    0.00896008960089601,
    0.00897008970089701,
    0.00898008980089801,
    0.00899008990089901,
    0.009000090000900008,
    0.009010090100901008,
    0.009020090200902008,
    0.009030090300903008,
    0.009040090400904009,
    0.009050090500905009,
    0.009060090600906009,
    0.009070090700907009,
    0.009080090800908009,
    0.00909009090090901,
    0.00910009100091001,
    0.00911009110091101,
    0.00912009120091201,
    0.00913009130091301,
    0.00914009140091401,
    0.009150091500915008,
    0.009160091600916008,
    0.009170091700917008,
    0.009180091800918009,
    0.009190091900919009,
    0.009200092000920009,
    0.009210092100921009,
    0.009220092200922009,
    0.00923009230092301,
    0.00924009240092401,
    0.00925009250092501,
    0.00926009260092601,
    0.00927009270092701,
    0.00928009280092801,
    0.009290092900929008,
    0.009300093000930008,
    0.009310093100931009,
    0.009320093200932009,
    0.009330093300933009,
    0.009340093400934009,
    0.009350093500935009,
    0.00936009360093601,
    0.00937009370093701,
    0.00938009380093801,
    0.00939009390093901,
    0.00940009400094001,
    0.00941009410094101,
    0.00942009420094201,
    0.00943009430094301,
    0.009440094400944008,
    0.009450094500945009,
    0.009460094600946009,
    0.009470094700947009,
    0.009480094800948009,
    0.009490094900949009,
    0.00950009500095001,
    0.00951009510095101,
    0.00952009520095201,
    0.00953009530095301,
    0.00954009540095401,
    0.00955009550095501,
    0.00956009560095601,
    0.00957009570095701,
    0.00958009580095801,
    0.009590095900959009,
    0.009600096000960009,
    0.009610096100961009,
    0.009620096200962009,
    0.00963009630096301,
    0.00964009640096401,
    0.00965009650096501,
    0.00966009660096601,
    0.00967009670096701,
    0.00968009680096801,
    0.00969009690096901,
    0.00970009700097001,
    0.00971009710097101,
    0.00972009720097201,
    0.009730097300973009,
    0.009740097400974009,
    0.009750097500975009,
    0.009760097600976009,
    0.00977009770097701,
    0.00978009780097801,
    0.00979009790097901,
    0.00980009800098001,
    0.00981009810098101,
    0.00982009820098201,
    0.00983009830098301,
    0.00984009840098401,
    0.00985009850098501,
    0.00986009860098601,
    0.00987009870098701,
    0.009880098800988009,
    0.009890098900989009,
    0.00990009900099001,
    0.00991009910099101,
    0.00992009920099201,
    0.00993009930099301,
    0.00994009940099401,
    0.00995009950099501,
    0.00996009960099601,
    0.00997009970099701,
    0.00998009980099801,
    0.00999009990099901,
    0.01000010000100001,
    0.01001010010100101,
    0.01002010020100201,
    0.010030100301003009,
    0.01004010040100401,
    0.01005010050100501,
    0.01006010060100601,
    0.01007010070100701,
    0.01008010080100801,
    0.01009010090100901,
    0.01010010100101001,
    0.01011010110101101,
    0.01012010120101201,
    0.01013010130101301,
    0.01014010140101401,
    0.01015010150101501,
    0.01016010160101601,
    0.01017010170101701,
    0.01018010180101801,
    0.01019010190101901,
    0.01020010200102001,
    0.01021010210102101,
    0.01022010220102201,
    0.01023010230102301,
    0.01024010240102401,
    0.01025010250102501,
    0.01026010260102601,
    0.01027010270102701,
    0.01028010280102801,
    0.01029010290102901,
    0.01030010300103001,
    0.010310103101031011,
    0.01032010320103201,
    0.01033010330103301,
    0.01034010340103401,
    0.01035010350103501,
    0.01036010360103601,
    0.01037010370103701,
    0.01038010380103801,
    0.01039010390103901,
    0.01040010400104001,
    0.01041010410104101,
    0.01042010420104201,
    0.01043010430104301,
    0.010440104401044011,
    0.010450104501045011,
    0.01046010460104601,
    0.01047010470104701,
    0.01048010480104801,
    0.01049010490104901,
    0.01050010500105001,
    0.01051010510105101,
    0.01052010520105201,
    0.01053010530105301,
    0.01054010540105401,
    0.01055010550105501,
    0.01056010560105601,
    0.01057010570105701,
    0.010580105801058011,
    0.010590105901059011,
    0.010600106001060011,
    0.01061010610106101,
    0.01062010620106201,
    0.01063010630106301,
    0.01064010640106401,
    0.01065010650106501,
    0.01066010660106601,
    0.01067010670106701,
    0.01068010680106801,
    0.01069010690106901,
    0.01070010700107001,
    0.010710107101071011,
    0.010720107201072011,
    0.010730107301073011,
    0.010740107401074011,
    0.010750107501075011,
    0.01076010760107601,
    0.01077010770107701,
    0.01078010780107801,
    0.01079010790107901,
    0.01080010800108001,
    0.01081010810108101,
    0.01082010820108201,
    0.01083010830108301,
    0.01084010840108401,
    0.010850108501085011,
    0.010860108601086011,
    0.010870108701087011,
    0.010880108801088011,
    0.010890108901089011,
    0.01090010900109001,
    0.01091010910109101,
    0.01092010920109201,
    0.01093010930109301,
    0.01094010940109401,
    0.01095010950109501,
    0.01096010960109601,
    0.01097010970109701,
    0.010980109801098011,
    0.010990109901099011,
    0.011000110001100011,
    0.011010110101101011,
    0.011020110201102011,
    0.011030110301103012,
    0.011040110401104012,
    0.01105011050110501,
    0.01106011060110601,
    0.01107011070110701,
    0.01108011080110801,
    0.01109011090110901,
    0.01110011100111001,
    0.01111011110111101,
    0.011120111201112011,
    0.011130111301113011,
    0.011140111401114011,
    0.011150111501115011,
    0.011160111601116011,
    0.011170111701117012,
    0.011180111801118012,
    0.011190111901119012,
    0.01120011200112001,
    0.01121011210112101,
    0.01122011220112201,
    0.01123011230112301,
    0.01124011240112401,
    0.011250112501125011,
    0.011260112601126011,
    0.011270112701127011,
    0.011280112801128011,
    0.011290112901129011,
    0.011300113001130012,
    0.011310113101131012,
    0.011320113201132012,
    0.011330113301133012,
    0.01134011340113401,
    0.01135011350113501,
    0.01136011360113601,
    0.01137011370113701,
    0.01138011380113801,
    0.011390113901139011,
    0.011400114001140011,
    0.011410114101141011,
    0.011420114201142011,
    0.011430114301143011,
    0.011440114401144012,
    0.011450114501145012,
    0.011460114601146012,
    0.011470114701147012,
    0.011480114801148012,
    0.01149011490114901,
    0.01150011500115001,
    0.01151011510115101,
    0.011520115201152011,
    0.011530115301153011,
    0.011540115401154011,
    0.011550115501155011,
    0.011560115601156011,
    0.011570115701157012,
    0.011580115801158012,
    0.011590115901159012,
    0.011600116001160012,
    0.011610116101161012,
    0.011620116201162012,
    0.01163011630116301,
    0.01164011640116401,
    0.01165011650116501,
    0.011660116601166011,
    0.011670116701167011,
    0.011680116801168011,
    0.011690116901169011,
    0.011700117001170011,
    0.011710117101171012,
    0.011720117201172012,
    0.011730117301173012,
    0.011740117401174012,
    0.011750117501175012,
    0.011760117601176012,
    0.011770117701177012,
    0.01178011780117801,
    0.011790117901179011,
    0.011800118001180011,
    0.011810118101181011,
    0.011820118201182011,
    0.011830118301183011,
    0.011840118401184012,
    0.011850118501185012,
    0.011860118601186012,
    0.011870118701187012,
    0.011880118801188012,
    0.011890118901189012,
    0.011900119001190012,
    0.011910119101191012,
    0.011920119201192013,
    0.011930119301193011,
    0.011940119401194011,
    0.011950119501195011,
    0.011960119601196011,
    0.011970119701197011,
    0.011980119801198012,
    0.011990119901199012,
    0.012000120001200012,
    0.012010120101201012,
    0.012020120201202012,
    0.012030120301203012,
    0.012040120401204012,
    0.012050120501205012,
    0.012060120601206013,
    0.012070120701207011,
    0.012080120801208011,
    0.012090120901209011,
    0.012100121001210011,
    0.012110121101211012,
    0.012120121201212012,
    0.012130121301213012,
    0.012140121401214012,
    0.012150121501215012,
    0.012160121601216012,
    0.012170121701217012,
    0.012180121801218012,
    0.012190121901219013,
    0.012200122001220013,
    0.012210122101221013,
    0.012220122201222011,
    0.012230122301223011,
    0.012240122401224011,
    0.012250122501225012,
    0.012260122601226012,
    0.012270122701227012,
    0.012280122801228012,
    0.012290122901229012,
    0.012300123001230012,
    0.012310123101231012,
    0.012320123201232012,
    0.012330123301233013,
    0.012340123401234013,
    0.012350123501235013,
    0.012360123601236013,
    0.012370123701237011,
    0.012380123801238012,
    0.012390123901239012,
    0.012400124001240012,
    0.012410124101241012,
    0.012420124201242012,
    0.012430124301243012,
    0.012440124401244012,
    0.012450124501245012,
    0.012460124601246013,
    0.012470124701247013,
    0.012480124801248013,
    0.012490124901249013,
    0.012500125001250013,
    0.012510125101251011,
    0.012520125201252012,
    0.012530125301253012,
    0.012540125401254012,
    0.012550125501255012,
    0.012560125601256012,
    0.012570125701257012,
    0.012580125801258012,
    0.012590125901259012,
    0.012600126001260013,
    0.012610126101261013,
    0.012620126201262013,
    0.012630126301263013,
    0.012640126401264013,
    0.012650126501265013,
    0.012660126601266012,
    0.012670126701267012,
    0.012680126801268012,
    0.012690126901269012,
    0.012700127001270012,
    0.012710127101271012,
    0.012720127201272012,
    0.012730127301273013,
    0.012740127401274013,
    0.012750127501275013,
    0.012760127601276013,
    0.012770127701277013,
    0.012780127801278013,
    0.012790127901279013,
    0.012800128001280012,
    0.012810128101281012,
    0.012820128201282012,
    0.012830128301283012,
    0.012840128401284012,
    0.012850128501285012,
    0.012860128601286012,
    0.012870128701287013,
    0.012880128801288013,
    0.012890128901289013,
    0.012900129001290013,
    0.012910129101291013,
    0.012920129201292013,
    0.012930129301293013,
    0.012940129401294014,
    0.012950129501295012,
    0.012960129601296012,
    0.012970129701297012,
    0.012980129801298012,
    0.012990129901299012,
    0.013000130001300013,
    0.013010130101301013,
    0.013020130201302013,
    0.013030130301303013,
    0.013040130401304013,
    0.013050130501305013,
    0.013060130601306013,
    0.013070130701307013,
    0.013080130801308014,
    0.013090130901309014,
    0.013100131001310012,
    0.013110131101311012,
    0.013120131201312012,
    0.013130131301313012,
    0.013140131401314013,
    0.013150131501315013,
    0.013160131601316013,
    0.013170131701317013,
    0.013180131801318013,
    0.013190131901319013,
    0.013200132001320013,
    0.013210132101321014,
    0.013220132201322014,
    0.013230132301323014,
    0.013240132401324012,
    0.013250132501325012,
    0.013260132601326012,
    0.013270132701327013,
    0.013280132801328013,
    0.013290132901329013,
    0.013300133001330013,
    0.013310133101331013,
    0.013320133201332013,
    0.013330133301333013,
    0.013340133401334013,
    0.013350133501335014,
    0.013360133601336014,
    0.013370133701337014,
    0.013380133801338014,
    0.013390133901339012,
    0.013400134001340012,
    0.013410134101341013,
    0.013420134201342013,
    0.013430134301343013,
    0.013440134401344013,
    0.013450134501345013,
    0.013460134601346013,
    0.013470134701347013,
    0.013480134801348014,
    0.013490134901349014,
    0.013500135001350014,
    0.013510135101351014,
    0.013520135201352014,
    0.013530135301353014,
    0.013540135401354013,
    0.013550135501355013,
    0.013560135601356013,
    0.013570135701357013,
    0.013580135801358013,
    0.013590135901359013,
    0.013600136001360013,
    0.013610136101361013,
    0.013620136201362014,
    0.013630136301363014,
    0.013640136401364014,
    0.013650136501365014,
    0.013660136601366014,
    0.013670136701367014,
    0.013680136801368013,
    0.013690136901369013,
    0.013700137001370013,
    0.013710137101371013,
    0.013720137201372013,
    0.013730137301373013,
    0.013740137401374013,
    0.013750137501375014,
    0.013760137601376014,
    0.013770137701377014,
    0.013780137801378014,
    0.013790137901379014,
    0.013800138001380014,
    0.013810138101381014,
    0.013820138201382014,
    0.013830138301383013,
    0.013840138401384013,
    0.013850138501385013,
    0.013860138601386013,
    0.013870138701387013,
    0.013880138801388013,
    0.013890138901389014,
    0.013900139001390014,
    0.013910139101391014,
    0.013920139201392014,
    0.013930139301393014,
    0.013940139401394014,
    0.013950139501395014,
    0.013960139601396014,
    0.013970139701397015,
    0.013980139801398013,
    0.013990139901399013,
    0.014000140001400013,
    0.014010140101401013,
    0.014020140201402014,
    0.014030140301403014,
    0.014040140401404014,
    0.014050140501405014,
    0.014060140601406014,
    0.014070140701407014,
    0.014080140801408014,
    0.014090140901409014,
    0.014100141001410015,
    0.014110141101411015,
    0.014120141201412013,
    0.014130141301413013,
    0.014140141401414013,
    0.014150141501415013,
    0.014160141601416014,
    0.014170141701417014,
    0.014180141801418014,
    0.014190141901419014,
    0.014200142001420014,
    0.014210142101421014,
    0.014220142201422014,
    0.014230142301423014,
    0.014240142401424015,
    0.014250142501425015,
    0.014260142601426015,
    0.014270142701427013,
    0.014280142801428013,
    0.014290142901429014,
    0.014300143001430014,
    0.014310143101431014,
    0.014320143201432014,
    0.014330143301433014,
    0.014340143401434014,
    0.014350143501435014,
    0.014360143601436014,
    0.014370143701437015,
    0.014380143801438015,
    0.014390143901439015,
    0.014400144001440015,
    0.014410144101441013,
    0.014420144201442013,
    0.014430144301443014,
    0.014440144401444014,
    0.014450144501445014,
    0.014460144601446014,
    0.014470144701447014,
    0.014480144801448014,
    0.014490144901449014,
    0.014500145001450014,
    0.014510145101451015,
    0.014520145201452015,
    0.014530145301453015,
    0.014540145401454015,
    0.014550145501455015,
    0.014560145601456013,
    0.014570145701457014,
    0.014580145801458014,
    0.014590145901459014,
    0.014600146001460014,
    0.014610146101461014,
    0.014620146201462014,
    0.014630146301463014,
    0.014640146401464015,
    0.014650146501465015,
    0.014660146601466015,
    0.014670146701467015,
    0.014680146801468015,
    0.014690146901469015,
    0.014700147001470015,
    0.014710147101471014,
    0.014720147201472014,
    0.014730147301473014,
    0.014740147401474014,
    0.014750147501475014,
    0.014760147601476014,
    0.014770147701477014,
    0.014780147801478015,
    0.014790147901479015,
    0.014800148001480015,
    0.014810148101481015,
    0.014820148201482015,
    0.014830148301483015,
    0.014840148401484015,
    0.014850148501485014,
    0.014860148601486014,
    0.014870148701487014,
    0.014880148801488014,
    0.014890148901489014,
    0.014900149001490014,
    0.014910149101491015,
    0.014920149201492015,
    0.014930149301493015,
    0.014940149401494015,
    0.014950149501495015,
    0.014960149601496015,
    0.014970149701497015,
    0.014980149801498015,
    0.014990149901499016,
    0.015000150001500014,
    0.015010150101501014,
    0.015020150201502014,
    0.015030150301503014,
    0.015040150401504014,
    0.015050150501505015,
    0.015060150601506015,
    0.015070150701507015,
    0.015080150801508015,
    0.015090150901509015,
    0.015100151001510015,
    0.015110151101511015,
    0.015120151201512015,
    0.015130151301513016,
    0.015140151401514016,
    0.015150151501515014,
    0.015160151601516014,
    0.015170151701517014,
    0.015180151801518015,
    0.015190151901519015,
    0.015200152001520015,
    0.015210152101521015,
    0.015220152201522015,
    0.015230152301523015,
    0.015240152401524015,
    0.015250152501525015,
    0.015260152601526016,
    0.015270152701527016,
    0.015280152801528016,
    0.015290152901529014,
    0.015300153001530014,
    0.015310153101531014,
    0.015320153201532015,
    0.015330153301533015,
    0.015340153401534015,
    0.015350153501535015,
    0.015360153601536015,
    0.015370153701537015,
    0.015380153801538015,
    0.015390153901539015,
    0.015400154001540016,
    0.015410154101541016,
    0.015420154201542016,
    0.015430154301543016,
    0.015440154401544014,
    0.015450154501545015,
    0.015460154601546015,
    0.015470154701547015,
    0.015480154801548015,
    0.015490154901549015,
    0.015500155001550015,
    0.015510155101551015,
    0.015520155201552015,
    0.015530155301553016,
    0.015540155401554016,
    0.015550155501555016,
    0.015560155601556016,
    0.015570155701557016,
    0.015580155801558014,
    0.015590155901559015,
    0.015600156001560015,
    0.015610156101561015,
    0.015620156201562015,
    0.015630156301563017,
    0.015640156401564017,
    0.015650156501565017,
    0.015660156601566014,
    0.015670156701567014,
    0.015680156801568014,
    0.015690156901569014,
    0.015700157001570014,
    0.015710157101571014,
    0.015720157201572015,
    0.015730157301573015,
    0.015740157401574015,
    0.015750157501575015,
    0.015760157601576015,
    0.015770157701577015,
    0.015780157801578015,
    0.015790157901579015,
    0.015800158001580016,
    0.015810158101581016,
    0.015820158201582016,
    0.015830158301583016,
    0.015840158401584016,
    0.015850158501585016,
    0.015860158601586016,
    0.015870158701587016,
    0.015880158801588017,
    0.015890158901589017,
    0.015900159001590017,
    0.015910159101591017,
    0.015920159201592017,
    0.015930159301593017,
    0.015940159401594017,
    0.015950159501595014,
    0.015960159601596014,
    0.015970159701597014,
    0.015980159801598014,
    0.015990159901599015,
    0.016000160001600015,
    0.016010160101601015,
    0.016020160201602015,
    0.016030160301603015,
    0.016040160401604015,
    0.016050160501605015,
    0.016060160601606015,
    0.016070160701607016,
    0.016080160801608016,
    0.016090160901609016,
    0.016100161001610016,
    0.016110161101611016,
    0.016120161201612016,
    0.016130161301613016,
    0.016140161401614016,
    0.016150161501615017,
    0.016160161601616017,
    0.016170161701617017,
    0.016180161801618017,
    0.016190161901619017,
    0.016200162001620017,
    0.016210162101621017,
    0.016220162201622017,
    0.016230162301623018,
    0.016240162401624014,
    0.016250162501625014,
    0.016260162601626015,
    0.016270162701627015,
    0.016280162801628015,
    0.016290162901629015,
    0.016300163001630015,
    0.016310163101631015,
    0.016320163201632015,
    0.016330163301633015,
    0.016340163401634016,
    0.016350163501635016,
    0.016360163601636016,
    0.016370163701637016,
    0.016380163801638016,
    0.016390163901639016,
    0.016400164001640016,
    0.016410164101641016,
    0.016420164201642017,
    0.016430164301643017,
    0.016440164401644017,
    0.016450164501645017,
    0.016460164601646017,
    0.016470164701647017,
    0.016480164801648017,
    0.016490164901649017,
    0.016500165001650018,
    0.016510165101651018,
    0.016520165201652018,
    0.016530165301653018,
    0.016540165401654015,
    0.016550165501655015,
    0.016560165601656015,
    0.016570165701657015,
    0.016580165801658015,
    0.016590165901659015,
    0.016600166001660015,
    0.016610166101661016,
    0.016620166201662016,
    0.016630166301663016,
    0.016640166401664016,
    0.016650166501665016,
    0.016660166601666016,
    0.016670166701667016,
    0.016680166801668016,
    0.016690166901669017,
    0.016700167001670017,
    0.016710167101671017,
    0.016720167201672017,
    0.016730167301673017,
    0.016740167401674017,
    0.016750167501675017,
    0.016760167601676017,
    0.016770167701677018,
    0.016780167801678018,
    0.016790167901679018,
    0.016800168001680018,
    0.016810168101681018,
    0.016820168201682018,
    0.016830168301683015,
    0.016840168401684015,
    0.016850168501685015,
    0.016860168601686015,
    0.016870168701687015,
    0.016880168801688016,
    0.016890168901689016,
    0.016900169001690016,
    0.016910169101691016,
    0.016920169201692016,
    0.016930169301693016,
    0.016940169401694016,
    0.016950169501695016,
    0.016960169601696017,
    0.016970169701697017,
    0.016980169801698017,
    0.016990169901699017,
    0.017000170001700017,
    0.017010170101701017,
    0.017020170201702017,
    0.017030170301703017,
    0.017040170401704018,
    0.017050170501705018,
    0.017060170601706018,
    0.017070170701707018,
    0.017080170801708018,
    0.017090170901709018,
    0.01710017100171002,
    0.01711017110171102,
    0.017120171201712015,
    0.017130171301713015,
    0.017140171401714015,
    0.017150171501715016,
    0.017160171601716016,
    0.017170171701717016,
    0.017180171801718016,
    0.017190171901719016,
    0.017200172001720016,
    0.017210172101721016,
    0.017220172201722016,
    0.017230172301723017,
    0.017240172401724017,
    0.017250172501725017,
    0.017260172601726017,
    0.017270172701727017,
    0.017280172801728017,
    0.017290172901729017,
    0.017300173001730017,
    0.017310173101731018,
    0.017320173201732018,
    0.017330173301733018,
    0.017340173401734018,
    0.017350173501735018,
    0.017360173601736018,
    0.01737017370173702,
    0.01738017380173802,
    0.01739017390173902,
    0.01740017400174002,
    0.017410174101741015,
    0.017420174201742016,
    0.017430174301743016,
    0.017440174401744016,
    0.017450174501745016,
    0.017460174601746016,
    0.017470174701747016,
    0.017480174801748016,
    0.017490174901749016,
    0.017500175001750017,
    0.017510175101751017,
    0.017520175201752017,
    0.017530175301753017,
    0.017540175401754017,
    0.017550175501755017,
    0.017560175601756017,
    0.017570175701757017,
    0.017580175801758018,
    0.017590175901759018,
    0.017600176001760018,
    0.017610176101761018,
    0.017620176201762018,
    0.017630176301763018,
    0.01764017640176402,
    0.01765017650176502,
    0.01766017660176602,
    0.01767017670176702,
    0.01768017680176802,
    0.01769017690176902,
    0.01770017700177002,
    0.017710177101771016,
    0.017720177201772016,
    0.017730177301773016,
    0.017740177401774016,
    0.017750177501775016,
    0.017760177601776016,
    0.017770177701777017,
    0.017780177801778017,
    0.017790177901779017,
    0.017800178001780017,
    0.017810178101781017,
    0.017820178201782017,
    0.017830178301783017,
    0.017840178401784017,
    0.017850178501785018,
    0.017860178601786018,
    0.017870178701787018,
    0.017880178801788018,
    0.017890178901789018,
    0.017900179001790018,
    0.01791017910179102,
    0.01792017920179202,
    0.01793017930179302,
    0.01794017940179402,
    0.01795017950179502,
    0.01796017960179602,
    0.01797017970179702,
    0.01798017980179802,
    0.01799017990179902,
    0.018000180001800016,
    0.018010180101801016,
    0.018020180201802016,
    0.018030180301803016,
    0.018040180401804017,
    0.018050180501805017,
    0.018060180601806017,
    0.018070180701807017,
    0.018080180801808017,
    0.018090180901809017,
    0.018100181001810017,
    0.018110181101811017,
    0.018120181201812018,
    0.018130181301813018,
    0.018140181401814018,
    0.018150181501815018,
    0.018160181601816018,
    0.018170181701817018,
    0.01818018180181802,
    0.01819018190181902,
    0.01820018200182002,
    0.01821018210182102,
    0.01822018220182202,
    0.01823018230182302,
    0.01824018240182402,
    0.01825018250182502,
    0.01826018260182602,
    0.01827018270182702,
    0.01828018280182802,
    0.018290182901829016,
    0.018300183001830016,
    0.018310183101831017,
    0.018320183201832017,
    0.018330183301833017,
    0.018340183401834017,
    0.018350183501835017,
    0.018360183601836017,
    0.018370183701837017,
    0.018380183801838017,
    0.018390183901839018,
    0.018400184001840018,
    0.018410184101841018,
    0.018420184201842018,
    0.018430184301843018,
    0.018440184401844018,
    0.01845018450184502,
    0.01846018460184602,
    0.01847018470184702,
    0.01848018480184802,
    0.01849018490184902,
    0.01850018500185002,
    0.01851018510185102,
    0.01852018520185202,
    0.01853018530185302,
    0.01854018540185402,
    0.01855018550185502,
    0.01856018560185602,
    0.01857018570185702,
    0.018580185801858017,
    0.018590185901859017,
    0.018600186001860017,
    0.018610186101861017,
    0.018620186201862017,
    0.018630186301863017,
    0.018640186401864017,
    0.018650186501865017,
    0.018660186601866018,
    0.018670186701867018,
    0.018680186801868018,
    0.018690186901869018,
    0.018700187001870018,
    0.018710187101871018,
    0.01872018720187202,
    0.01873018730187302,
    0.01874018740187402,
    0.01875018750187502,
    0.01876018760187602,
    0.01877018770187702,
    0.01878018780187802,
    0.01879018790187902,
    0.01880018800188002,
    0.01881018810188102,
    0.01882018820188202,
    0.01883018830188302,
    0.01884018840188402,
    0.01885018850188502,
    0.01886018860188602,
    0.01887018870188702,
    0.018880188801888017,
    0.018890188901889017,
    0.018900189001890017,
    0.018910189101891017,
    0.018920189201892017,
    0.018930189301893018,
    0.018940189401894018,
    0.018950189501895018,
    0.018960189601896018,
    0.018970189701897018,
    0.018980189801898018,
    0.01899018990189902,
    0.01900019000190002,
    0.01901019010190102,
    0.01902019020190202,
    0.01903019030190302,
    0.01904019040190402,
    0.01905019050190502,
    0.01906019060190602,
    0.01907019070190702,
    0.01908019080190802,
    0.01909019090190902,
    0.01910019100191002,
    0.01911019110191102,
    0.01912019120191202,
    0.01913019130191302,
    0.01914019140191402,
    0.01915019150191502,
    0.01916019160191602,
    0.019170191701917017,
    0.019180191801918017,
    0.019190191901919017,
    0.019200192001920018,
    0.019210192101921018,
    0.019220192201922018,
    0.019230192301923018,
    0.019240192401924018,
    0.019250192501925018,
    0.01926019260192602,
    0.01927019270192702,
    0.01928019280192802,
    0.01929019290192902,
    0.01930019300193002,
    0.01931019310193102,
    0.01932019320193202,
    0.01933019330193302,
    0.01934019340193402,
    0.01935019350193502,
    0.01936019360193602,
    0.01937019370193702,
    0.01938019380193802,
    0.01939019390193902,
    0.01940019400194002,
    0.01941019410194102,
    0.01942019420194202,
    0.01943019430194302,
    0.01944019440194402,
    0.01945019450194502,
    0.019460194601946017,
    0.019470194701947018,
    0.019480194801948018,
    0.019490194901949018,
    0.019500195001950018,
    0.019510195101951018,
    0.019520195201952018,
    0.01953019530195302,
    0.01954019540195402,
    0.01955019550195502,
    0.01956019560195602,
    0.01957019570195702,
    0.01958019580195802,
    0.01959019590195902,
    0.01960019600196002,
    0.01961019610196102,
    0.01962019620196202,
    0.01963019630196302,
    0.01964019640196402,
    0.01965019650196502,
    0.01966019660196602,
    0.01967019670196702,
    0.01968019680196802,
    0.01969019690196902,
    0.01970019700197002,
    0.01971019710197102,
    0.01972019720197202,
    0.01973019730197302,
    0.01974019740197402,
    0.019750197501975018,
    0.019760197601976018,
    0.019770197701977018,
    0.019780197801978018,
    0.019790197901979018,
    0.01980019800198002,
    0.01981019810198102,
    0.01982019820198202,
    0.01983019830198302,
    0.01984019840198402,
    0.01985019850198502,
    0.01986019860198602,
    0.01987019870198702,
    0.01988019880198802,
    0.01989019890198902,
    0.01990019900199002,
    0.01991019910199102,
    0.01992019920199202,
    0.01993019930199302,
    0.01994019940199402,
    0.01995019950199502,
    0.01996019960199602,
    0.01997019970199702,
    0.01998019980199802,
    0.01999019990199902,
    0.02000020000200002,
    0.02001020010200102,
    0.02002020020200202,
    0.02003020030200302,
    0.02004020040200402,
    0.020050200502005018,
    0.020060200602006018,
    0.02007020070200702,
    0.02008020080200802,
    0.02009020090200902,
    0.02010020100201002,
    0.02011020110201102,
    0.02012020120201202,
    0.02013020130201302,
    0.02014020140201402,
    0.02015020150201502,
    0.02016020160201602,
    0.02017020170201702,
    0.02018020180201802,
    0.02019020190201902,
    0.02020020200202002,
    0.02021020210202102,
    0.02022020220202202,
    0.02023020230202302,
    0.02024020240202402,
    0.02025020250202502,
    0.02026020260202602,
    0.02027020270202702,
    0.02028020280202802,
    0.02029020290202902,
    0.02030020300203002,
    0.02031020310203102,
    0.02032020320203202,
    0.02033020330203302,
    0.02034020340203402,
    0.02035020350203502,
    0.02036020360203602,
    0.02037020370203702,
    0.02038020380203802,
    0.02039020390203902,
    0.02040020400204002,
    0.02041020410204102,
    0.02042020420204202,
    0.02043020430204302,
    0.02044020440204402,
    0.02045020450204502,
    0.02046020460204602,
    0.02047020470204702,
    0.02048020480204802,
    0.02049020490204902,
    0.02050020500205002,
    0.02051020510205102,
    0.02052020520205202,
    0.02053020530205302,
    0.02054020540205402,
    0.02055020550205502,
    0.02056020560205602,
    0.02057020570205702,
    0.02058020580205802,
    0.02059020590205902,
    0.02060020600206002,
    0.020610206102061022,
    0.020620206202062022,
    0.02063020630206302,
    0.02064020640206402,
    0.02065020650206502,
    0.02066020660206602,
    0.02067020670206702,
    0.02068020680206802,
    0.02069020690206902,
    0.02070020700207002,
    0.02071020710207102,
    0.02072020720207202,
    0.02073020730207302,
    0.02074020740207402,
    0.02075020750207502,
    0.02076020760207602,
    0.02077020770207702,
    0.02078020780207802,
    0.02079020790207902,
    0.02080020800208002,
    0.02081020810208102,
    0.02082020820208202,
    0.02083020830208302,
    0.02084020840208402,
    0.02085020850208502,
    0.02086020860208602,
    0.02087020870208702,
    0.020880208802088022,
    0.020890208902089022,
    0.020900209002090022,
    0.020910209102091022,
    0.02092020920209202,
    0.02093020930209302,
    0.02094020940209402,
    0.02095020950209502,
    0.02096020960209602,
    0.02097020970209702,
    0.02098020980209802,
    0.02099020990209902,
    0.02100021000210002,
    0.02101021010210102,
    0.02102021020210202,
    0.02103021030210302,
    0.02104021040210402,
    0.02105021050210502,
    0.02106021060210602,
    0.02107021070210702,
    0.02108021080210802,
    0.02109021090210902,
    0.02110021100211002,
    0.02111021110211102,
    0.02112021120211202,
    0.02113021130211302,
    0.02114021140211402,
    0.021150211502115022,
    0.021160211602116022,
    0.021170211702117022,
    0.021180211802118022,
    0.021190211902119022,
    0.021200212002120022,
    0.021210212102121023,
    0.02122021220212202,
    0.02123021230212302,
    0.02124021240212402,
    0.02125021250212502,
    0.02126021260212602,
    0.02127021270212702,
    0.02128021280212802,
    0.02129021290212902,
    0.02130021300213002,
    0.02131021310213102,
    0.02132021320213202,
    0.02133021330213302,
    0.02134021340213402,
    0.02135021350213502,
    0.02136021360213602,
    0.02137021370213702,
    0.02138021380213802,
    0.02139021390213902,
    0.02140021400214002,
    0.02141021410214102,
    0.021420214202142022,
    0.021430214302143022,
    0.021440214402144022,
    0.021450214502145022,
    0.021460214602146022,
    0.021470214702147022,
    0.021480214802148023,
    0.021490214902149023,
    0.021500215002150023,
    0.02151021510215102,
    0.02152021520215202,
    0.02153021530215302,
    0.02154021540215402,
    0.02155021550215502,
    0.02156021560215602,
    0.02157021570215702,
    0.02158021580215802,
    0.02159021590215902,
    0.02160021600216002,
    0.02161021610216102,
    0.02162021620216202,
    0.02163021630216302,
    0.02164021640216402,
    0.02165021650216502,
    0.02166021660216602,
    0.02167021670216702,
    0.02168021680216802,
    0.021690216902169022,
    0.021700217002170022,
    0.021710217102171022,
    0.021720217202172022,
    0.021730217302173022,
    0.021740217402174022,
    0.021750217502175023,
    0.021760217602176023,
    0.021770217702177023,
    0.021780217802178023,
    0.021790217902179023,
    0.02180021800218002,
    0.02181021810218102,
    0.02182021820218202,
    0.02183021830218302,
    0.02184021840218402,
    0.02185021850218502,
    0.02186021860218602,
    0.02187021870218702,
    0.02188021880218802,
    0.02189021890218902,
    0.02190021900219002,
    0.02191021910219102,
    0.02192021920219202,
    0.02193021930219302,
    0.02194021940219402,
    0.02195021950219502,
    0.021960219602196022,
    0.021970219702197022,
    0.021980219802198022,
    0.021990219902199022,
    0.022000220002200022,
    0.022010220102201022,
    0.022020220202202023,
    0.022030220302203023,
    0.022040220402204023,
    0.022050220502205023,
    0.022060220602206023,
    0.022070220702207023,
    0.022080220802208023,
    0.02209022090220902,
    0.02210022100221002,
    0.02211022110221102,
    0.02212022120221202,
    0.02213022130221302,
    0.02214022140221402,
    0.02215022150221502,
    0.02216022160221602,
    0.02217022170221702,
    0.02218022180221802,
    0.02219022190221902,
    0.02220022200222002,
    0.02221022210222102,
    0.02222022220222202,
    0.022230222302223022,
    0.022240222402224022,
    0.022250222502225022,
    0.022260222602226022,
    0.022270222702227022,
    0.022280222802228022,
    0.022290222902229023,
    0.022300223002230023,
    0.022310223102231023,
    0.022320223202232023,
    0.022330223302233023,
    0.022340223402234023,
    0.022350223502235023,
    0.022360223602236023,
    0.022370223702237024,
    0.022380223802238024,
    0.02239022390223902,
    0.02240022400224002,
    0.02241022410224102,
    0.02242022420224202,
    0.02243022430224302,
    0.02244022440224402,
    0.02245022450224502,
    0.02246022460224602,
    0.02247022470224702,
    0.02248022480224802,
    0.02249022490224902,
    0.022500225002250022,
    0.022510225102251022,
    0.022520225202252022,
    0.022530225302253022,
    0.022540225402254022,
    0.022550225502255022,
    0.022560225602256023,
    0.022570225702257023,
    0.022580225802258023,
    0.022590225902259023,
    0.022600226002260023,
    0.022610226102261023,
    0.022620226202262023,
    0.022630226302263023,
    0.022640226402264024,
    0.022650226502265024,
    0.022660226602266024,
    0.022670226702267024,
    0.02268022680226802,
    0.02269022690226902,
    0.02270022700227002,
    0.02271022710227102,
    0.02272022720227202,
    0.02273022730227302,
    0.02274022740227402,
    0.02275022750227502,
    0.02276022760227602,
    0.022770227702277022,
    0.022780227802278022,
    0.022790227902279022,
    0.022800228002280022,
    0.022810228102281022,
    0.022820228202282022,
    0.022830228302283023,
    0.022840228402284023,
    0.022850228502285023,
    0.022860228602286023,
    0.022870228702287023,
    0.022880228802288023,
    0.022890228902289023,
    0.022900229002290023,
    0.022910229102291024,
    0.022920229202292024,
    0.022930229302293024,
    0.022940229402294024,
    0.022950229502295024,
    0.022960229602296024,
    0.02297022970229702,
    0.02298022980229802,
    0.02299022990229902,
    0.02300023000230002,
    0.02301023010230102,
    0.02302023020230202,
    0.02303023030230302,
    0.023040230402304022,
    0.023050230502305022,
    0.023060230602306022,
    0.023070230702307022,
    0.023080230802308022,
    0.023090230902309022,
    0.023100231002310023,
    0.023110231102311023,
    0.023120231202312023,
    0.023130231302313023,
    0.023140231402314023,
    0.023150231502315023,
    0.023160231602316023,
    0.023170231702317023,
    0.023180231802318024,
    0.023190231902319024,
    0.023200232002320024,
    0.023210232102321024,
    0.023220232202322024,
    0.023230232302323024,
    0.023240232402324024,
    0.023250232502325024,
    0.02326023260232602,
    0.02327023270232702,
    0.02328023280232802,
    0.02329023290232902,
    0.02330023300233002,
    0.023310233102331022,
    0.023320233202332022,
    0.023330233302333022,
    0.023340233402334022,
    0.023350233502335022,
    0.023360233602336022,
    0.023370233702337023,
    0.023380233802338023,
    0.023390233902339023,
    0.023400234002340023,
    0.023410234102341023,
    0.023420234202342023,
    0.023430234302343023,
    0.023440234402344023,
    0.023450234502345024,
    0.023460234602346024,
    0.023470234702347024,
    0.023480234802348024,
    0.023490234902349024,
    0.023500235002350024,
    0.023510235102351024,
    0.023520235202352024,
    0.023530235302353025,
    0.023540235402354025,
    0.023550235502355025,
    0.02356023560235602,
    0.02357023570235702,
    0.023580235802358022,
    0.023590235902359022,
    0.023600236002360022,
    0.023610236102361022,
    0.023620236202362022,
    0.023630236302363022,
    0.023640236402364023,
    0.023650236502365023,
    0.023660236602366023,
    0.023670236702367023,
    0.023680236802368023,
    0.023690236902369023,
    0.023700237002370023,
    0.023710237102371023,
    0.023720237202372024,
    0.023730237302373024,
    0.023740237402374024,
    0.023750237502375024,
    0.023760237602376024,
    0.023770237702377024,
    0.023780237802378024,
    0.023790237902379024,
    0.023800238002380025,
    0.023810238102381025,
    0.023820238202382025,
    0.023830238302383025,
    0.023840238402384025,
    0.023850238502385022,
    0.023860238602386022,
    0.023870238702387022,
    0.023880238802388022,
    0.023890238902389022,
    0.023900239002390022,
    0.023910239102391023,
    0.023920239202392023,
    0.023930239302393023,
    0.023940239402394023,
    0.023950239502395023,
    0.023960239602396023,
    0.023970239702397023,
    0.023980239802398023,
    0.023990239902399024,
    0.024000240002400024,
    0.024010240102401024,
    0.024020240202402024,
    0.024030240302403024,
    0.024040240402404024,
    0.024050240502405024,
    0.024060240602406024,
    0.024070240702407025,
    0.024080240802408025,
    0.024090240902409025,
    0.024100241002410025,
    0.024110241102411025,
    0.024120241202412025,
    0.024130241302413025,
    0.024140241402414022,
    0.024150241502415022,
    0.024160241602416022,
    0.024170241702417022,
    0.024180241802418023,
    0.024190241902419023,
    0.024200242002420023,
    0.024210242102421023,
    0.024220242202422023,
    0.024230242302423023,
    0.024240242402424023,
    0.024250242502425023,
    0.024260242602426024,
    0.024270242702427024,
    0.024280242802428024,
    0.024290242902429024,
    0.024300243002430024,
    0.024310243102431024,
    0.024320243202432024,
    0.024330243302433024,
    0.024340243402434025,
    0.024350243502435025,
    0.024360243602436025,
    0.024370243702437025,
    0.024380243802438025,
    0.024390243902439025,
    0.024400244002440025,
    0.024410244102441025,
    0.024420244202442026,
    0.024430244302443022,
    0.024440244402444022,
    0.024450244502445023,
    0.024460244602446023,
    0.024470244702447023,
    0.024480244802448023,
    0.024490244902449023,
    0.024500245002450023,
    0.024510245102451023,
    0.024520245202452023,
    0.024530245302453024,
    0.024540245402454024,
    0.024550245502455024,
    0.024560245602456024,
    0.024570245702457024,
    0.024580245802458024,
    0.024590245902459024,
    0.024600246002460024,
    0.024610246102461025,
    0.024620246202462025,
    0.024630246302463025,
    0.024640246402464025,
    0.024650246502465025,
    0.024660246602466025,
    0.024670246702467025,
    0.024680246802468025,
    0.024690246902469026,
    0.024700247002470026,
    0.024710247102471026,
    0.024720247202472026,
    0.024730247302473023,
    0.024740247402474023,
    0.024750247502475023,
    0.024760247602476023,
    0.024770247702477023,
    0.024780247802478023,
    0.024790247902479023,
    0.024800248002480024,
    0.024810248102481024,
    0.024820248202482024,
    0.024830248302483024,
    0.024840248402484024,
    0.024850248502485024,
    0.024860248602486024,
    0.024870248702487024,
    0.024880248802488025,
    0.024890248902489025,
    0.024900249002490025,
    0.024910249102491025,
    0.024920249202492025,
    0.024930249302493025,
    0.024940249402494025,
    0.024950249502495025,
    0.024960249602496026,
    0.024970249702497026,
    0.024980249802498026,
    0.024990249902499026,
    0.025000250002500026,
    0.025010250102501026,
    0.025020250202502023,
    0.025030250302503023,
    0.025040250402504023,
    0.025050250502505023,
    0.025060250602506023,
    0.025070250702507024,
    0.025080250802508024,
    0.025090250902509024,
    0.025100251002510024,
    0.025110251102511024,
    0.025120251202512024,
    0.025130251302513024,
    0.025140251402514024,
    0.025150251502515025,
    0.025160251602516025,
    0.025170251702517025,
    0.025180251802518025,
    0.025190251902519025,
    0.025200252002520025,
    0.025210252102521025,
    0.025220252202522025,
    0.025230252302523026,
    0.025240252402524026,
    0.025250252502525026,
    0.025260252602526026,
    0.025270252702527026,
    0.025280252802528026,
    0.025290252902529026,
    0.025300253002530027,
    0.025310253102531023,
    0.025320253202532023,
    0.025330253302533023,
    0.025340253402534024,
    0.025350253502535024,
    0.025360253602536024,
    0.025370253702537024,
    0.025380253802538024,
    0.025390253902539024,
    0.025400254002540024,
    0.025410254102541024,
    0.025420254202542025,
    0.025430254302543025,
    0.025440254402544025,
    0.025450254502545025,
    0.025460254602546025,
    0.025470254702547025,
    0.025480254802548025,
    0.025490254902549025,
    0.025500255002550026,
    0.025510255102551026,
    0.025520255202552026,
    0.025530255302553026,
    0.025540255402554026,
    0.025550255502555026,
    0.025560255602556026,
    0.025570255702557027,
    0.025580255802558027,
    0.025590255902559027,
    0.025600256002560023,
    0.025610256102561024,
    0.025620256202562024,
    0.025630256302563024,
    0.025640256402564024,
    0.025650256502565024,
    0.025660256602566024,
    0.025670256702567024,
    0.025680256802568024,
    0.025690256902569025,
    0.025700257002570025,
    0.025710257102571025,
    0.025720257202572025,
    0.025730257302573025,
    0.025740257402574025,
    0.025750257502575025,
    0.025760257602576025,
    0.025770257702577026,
    0.025780257802578026,
    0.025790257902579026,
    0.025800258002580026,
    0.025810258102581026,
    0.025820258202582026,
    0.025830258302583026,
    0.025840258402584027,
    0.025850258502585027,
    0.025860258602586027,
    0.025870258702587027,
    0.025880258802588027,
    0.025890258902589027,
    0.025900259002590024,
    0.025910259102591024,
    0.025920259202592024,
    0.025930259302593024,
    0.025940259402594024,
    0.025950259502595024,
    0.025960259602596025,
    0.025970259702597025,
    0.025980259802598025,
    0.025990259902599025,
    0.026000260002600025,
    0.026010260102601025,
    0.026020260202602025,
    0.026030260302603025,
    0.026040260402604026,
    0.026050260502605026,
    0.026060260602606026,
    0.026070260702607026,
    0.026080260802608026,
    0.026090260902609026,
    0.026100261002610026,
    0.026110261102611027,
    0.026120261202612027,
    0.026130261302613027,
    0.026140261402614027,
    0.026150261502615027,
    0.026160261602616027,
    0.026170261702617027,
    0.026180261802618027,
    0.026190261902619024,
    0.026200262002620024,
    0.026210262102621024,
    0.026220262202622024,
    0.026230262302623025,
    0.026240262402624025,
    0.026250262502625025,
    0.026260262602626025,
    0.026270262702627025,
    0.026280262802628025,
    0.026290262902629025,
    0.026300263002630025,
    0.026310263102631026,
    0.026320263202632026,
    0.026330263302633026,
    0.026340263402634026,
    0.026350263502635026,
    0.026360263602636026,
    0.026370263702637026,
    0.026380263802638027,
    0.026390263902639027,
    0.026400264002640027,
    0.026410264102641027,
    0.026420264202642027,
    0.026430264302643027,
    0.026440264402644027,
    0.026450264502645027,
    0.026460264602646028,
    0.026470264702647028,
    0.026480264802648024,
    0.026490264902649024,
    0.026500265002650025,
    0.026510265102651025,
    0.026520265202652025,
    0.026530265302653025,
    0.026540265402654025,
    0.026550265502655025,
    0.026560265602656025,
    0.026570265702657025,
    0.026580265802658026,
    0.026590265902659026,
    0.026600266002660026,
    0.026610266102661026,
    0.026620266202662026,
    0.026630266302663026,
    0.026640266402664026,
    0.026650266502665027,
    0.026660266602666027,
    0.026670266702667027,
    0.026680266802668027,
    0.026690266902669027,
    0.026700267002670027,
    0.026710267102671027,
    0.026720267202672027,
    0.026730267302673028,
    0.026740267402674028,
    0.026750267502675028,
    0.026760267602676028,
    0.026770267702677028,
    0.026780267802678025,
    0.026790267902679025,
    0.026800268002680025,
    0.026810268102681025,
    0.026820268202682025,
    0.026830268302683025,
    0.026840268402684025,
    0.026850268502685026,
    0.026860268602686026,
    0.026870268702687026,
    0.026880268802688026,
    0.026890268902689026,
    0.026900269002690026,
    0.026910269102691026,
    0.026920269202692027,
    0.026930269302693027,
    0.026940269402694027,
    0.026950269502695027,
    0.026960269602696027,
    0.026970269702697027,
    0.026980269802698027,
    0.026990269902699027,
    0.027000270002700028,
    0.027010270102701028,
    0.027020270202702028,
    0.027030270302703028,
    0.027040270402704028,
    0.027050270502705028,
    0.02706027060270603,
    0.027070270702707025,
    0.027080270802708025,
    0.027090270902709025,
    0.027100271002710025,
    0.027110271102711025,
    0.027120271202712026,
    0.027130271302713026,
    0.027140271402714026,
    0.027150271502715026,
    0.027160271602716026,
    0.027170271702717026,
    0.027180271802718026,
    0.027190271902719026,
    0.027200272002720027,
    0.027210272102721027,
    0.027220272202722027,
    0.027230272302723027,
    0.027240272402724027,
    0.027250272502725027,
    0.027260272602726027,
    0.027270272702727028,
    0.027280272802728028,
    0.027290272902729028,
    0.027300273002730028,
    0.027310273102731028,
    0.027320273202732028,
    0.02733027330273303,
    0.02734027340273403,
    0.02735027350273503,
    0.027360273602736025,
    0.027370273702737025,
    0.027380273802738025,
    0.027390273902739026,
    0.027400274002740026,
    0.027410274102741026,
    0.027420274202742026,
    0.027430274302743026,
    0.027440274402744026,
    0.027450274502745026,
    0.027460274602746026,
    0.027470274702747027,
    0.027480274802748027,
    0.027490274902749027,
    0.027500275002750027,
    0.027510275102751027,
    0.027520275202752027,
    0.027530275302753027,
    0.027540275402754028,
    0.027550275502755028,
    0.027560275602756028,
    0.027570275702757028,
    0.027580275802758028,
    0.027590275902759028,
    0.02760027600276003,
    0.02761027610276103,
    0.02762027620276203,
    0.02763027630276303,
    0.02764027640276403,
    0.027650276502765025,
    0.027660276602766026,
    0.027670276702767026,
    0.027680276802768026,
    0.027690276902769026,
    0.027700277002770026,
    0.027710277102771026,
    0.027720277202772026,
    0.027730277302773026,
    0.027740277402774027,
    0.027750277502775027,
    0.027760277602776027,
    0.027770277702777027,
    0.027780277802778027,
    0.027790277902779027,
    0.027800278002780027,
    0.027810278102781028,
    0.027820278202782028,
    0.027830278302783028,
    0.027840278402784028,
    0.027850278502785028,
    0.027860278602786028,
    0.02787027870278703,
    0.02788027880278803,
    0.02789027890278903,
    0.02790027900279003,
    0.02791027910279103,
    0.02792027920279203,
    0.02793027930279303,
    0.02794027940279403,
    0.027950279502795026,
    0.027960279602796026,
    0.027970279702797026,
    0.027980279802798026,
    0.027990279902799026,
    0.028000280002800026,
    0.028010280102801027,
    0.028020280202802027,
    0.028030280302803027,
    0.028040280402804027,
    0.028050280502805027,
    0.028060280602806027,
    0.028070280702807027,
    0.028080280802808028,
    0.028090280902809028,
    0.028100281002810028,
    0.028110281102811028,
    0.028120281202812028,
    0.028130281302813028,
    0.02814028140281403,
    0.02815028150281503,
    0.02816028160281603,
    0.02817028170281703,
    0.02818028180281803,
    0.02819028190281903,
    0.02820028200282003,
    0.02821028210282103,
    0.02822028220282203,
    0.02823028230282303,
    0.028240282402824026,
    0.028250282502825026,
    0.028260282602826026,
    0.028270282702827026,
    0.028280282802828027,
    0.028290282902829027,
    0.028300283002830027,
    0.028310283102831027,
    0.028320283202832027,
    0.028330283302833027,
    0.028340283402834027,
    0.028350283502835028,
    0.028360283602836028,
    0.028370283702837028,
    0.028380283802838028,
    0.028390283902839028,
    0.028400284002840028,
    0.02841028410284103,
    0.02842028420284203,
    0.02843028430284303,
    0.02844028440284403,
    0.02845028450284503,
    0.02846028460284603,
    0.02847028470284703,
    0.02848028480284803,
    0.02849028490284903,
    0.02850028500285003,
    0.02851028510285103,
    0.02852028520285203,
    0.028530285302853026,
    0.028540285402854026,
    0.028550285502855027,
    0.028560285602856027,
    0.028570285702857027,
    0.028580285802858027,
    0.028590285902859027,
    0.028600286002860027,
    0.028610286102861027,
    0.028620286202862028,
    0.028630286302863028,
    0.028640286402864028,
    0.028650286502865028,
    0.028660286602866028,
    0.028670286702867028,
    0.02868028680286803,
    0.02869028690286903,
    0.02870028700287003,
    0.02871028710287103,
    0.02872028720287203,
    0.02873028730287303,
    0.02874028740287403,
    0.02875028750287503,
    0.02876028760287603,
    0.02877028770287703,
    0.02878028780287803,
    0.02879028790287903,
    0.02880028800288003,
    0.02881028810288103,
    0.028820288202882027,
    0.028830288302883027,
    0.028840288402884027,
    0.028850288502885027,
    0.028860288602886027,
    0.028870288702887027,
    0.028880288802888027,
    0.028890288902889028,
    0.028900289002890028,
    0.028910289102891028,
    0.028920289202892028,
    0.028930289302893028,
    0.028940289402894028,
    0.02895028950289503,
    0.02896028960289603,
    0.02897028970289703,
    0.02898028980289803,
    0.02899028990289903,
    0.02900029000290003,
    0.02901029010290103,
    0.02902029020290203,
    0.02903029030290303,
    0.02904029040290403,
    0.02905029050290503,
    0.02906029060290603,
    0.02907029070290703,
    0.02908029080290803,
    0.02909029090290903,
    0.02910029100291003,
    0.02911029110291103,
    0.029120291202912027,
    0.029130291302913027,
    0.029140291402914027,
    0.029150291502915027,
    0.029160291602916028,
    0.029170291702917028,
    0.029180291802918028,
    0.029190291902919028,
    0.029200292002920028,
    0.029210292102921028,
    0.02922029220292203,
    0.02923029230292303,
    0.02924029240292403,
    0.02925029250292503,
    0.02926029260292603,
    0.02927029270292703,
    0.02928029280292803,
    0.02929029290292903,
    0.02930029300293003,
    0.02931029310293103,
    0.02932029320293203,
    0.02933029330293303,
    0.02934029340293403,
    0.02935029350293503,
    0.02936029360293603,
    0.02937029370293703,
    0.02938029380293803,
    0.02939029390293903,
    0.02940029400294003,
    0.029410294102941027,
    0.029420294202942027,
    0.029430294302943028,
    0.029440294402944028,
    0.029450294502945028,
    0.029460294602946028,
    0.029470294702947028,
    0.029480294802948028,
    0.02949029490294903,
    0.02950029500295003,
    0.02951029510295103,
    0.02952029520295203,
    0.02953029530295303,
    0.02954029540295403,
    0.02955029550295503,
    0.02956029560295603,
    0.02957029570295703,
    0.02958029580295803,
    0.02959029590295903,
    0.02960029600296003,
    0.02961029610296103,
    0.02962029620296203,
    0.02963029630296303,
    0.02964029640296403,
    0.02965029650296503,
    0.02966029660296603,
    0.02967029670296703,
    0.02968029680296803,
    0.02969029690296903,
    0.029700297002970028,
    0.029710297102971028,
    0.029720297202972028,
    0.029730297302973028,
    0.029740297402974028,
    0.029750297502975028,
    0.02976029760297603,
    0.02977029770297703,
    0.02978029780297803,
    0.02979029790297903,
    0.02980029800298003,
    0.02981029810298103,
    0.02982029820298203,
    0.02983029830298303,
    0.02984029840298403,
    0.02985029850298503,
    0.02986029860298603,
    0.02987029870298703,
    0.02988029880298803,
    0.02989029890298903,
    0.02990029900299003,
    0.02991029910299103,
    0.02992029920299203,
    0.02993029930299303,
    0.02994029940299403,
    0.02995029950299503,
    0.02996029960299603,
    0.02997029970299703,
    0.02998029980299803,
    0.029990299902999028,
    0.030000300003000028,
    0.030010300103001028,
    0.030020300203002028,
    0.03003030030300303,
    0.03004030040300403,
    0.03005030050300503,
    0.03006030060300603,
    0.03007030070300703,
    0.03008030080300803,
    0.03009030090300903,
    0.03010030100301003,
    0.03011030110301103,
    0.03012030120301203,
    0.03013030130301303,
    0.03014030140301403,
    0.03015030150301503,
    0.03016030160301603,
    0.03017030170301703,
    0.03018030180301803,
    0.03019030190301903,
    0.03020030200302003,
    0.03021030210302103,
    0.03022030220302203,
    0.03023030230302303,
    0.03024030240302403,
    0.03025030250302503,
    0.03026030260302603,
    0.03027030270302703,
    0.03028030280302803,
    0.030290302903029028,
    0.03030030300303003,
    0.03031030310303103,
    0.03032030320303203,
    0.03033030330303303,
    0.03034030340303403,
    0.03035030350303503,
    0.03036030360303603,
    0.03037030370303703,
    0.03038030380303803,
    0.03039030390303903,
    0.03040030400304003,
    0.03041030410304103,
    0.03042030420304203,
    0.03043030430304303,
    0.03044030440304403,
    0.03045030450304503,
    0.03046030460304603,
    0.03047030470304703,
    0.03048030480304803,
    0.03049030490304903,
    0.03050030500305003,
    0.03051030510305103,
    0.03052030520305203,
    0.03053030530305303,
    0.03054030540305403,
    0.03055030550305503,
    0.03056030560305603,
    0.030570305703057032,
    0.03058030580305803,
    0.03059030590305903,
    0.03060030600306003,
    0.03061030610306103,
    0.03062030620306203,
    0.03063030630306303,
    0.03064030640306403,
    0.03065030650306503,
    0.03066030660306603,
    0.03067030670306703,
    0.03068030680306803,
    0.03069030690306903,
    0.03070030700307003,
    0.03071030710307103,
    0.03072030720307203,
    0.03073030730307303,
    0.03074030740307403,
    0.03075030750307503,
    0.03076030760307603,
    0.03077030770307703,
    0.03078030780307803,
    0.03079030790307903,
    0.03080030800308003,
    0.03081030810308103,
    0.03082030820308203,
    0.03083030830308303,
    0.030840308403084032,
    0.030850308503085032,
    0.030860308603086032,
    0.03087030870308703,
    0.03088030880308803,
    0.03089030890308903,
    0.03090030900309003,
    0.03091030910309103,
    0.03092030920309203,
    0.03093030930309303,
    0.03094030940309403,
    0.03095030950309503,
    0.03096030960309603,
    0.03097030970309703,
    0.03098030980309803,
    0.03099030990309903,
    0.03100031000310003,
    0.03101031010310103,
    0.03102031020310203,
    0.03103031030310303,
    0.03104031040310403,
    0.03105031050310503,
    0.03106031060310603,
    0.03107031070310703,
    0.03108031080310803,
    0.03109031090310903,
    0.03110031100311003,
    0.03111031110311103,
    0.031120311203112032,
    0.031130311303113032,
    0.031140311403114032,
    0.031150311503115032,
    0.03116031160311603,
    0.03117031170311703,
    0.03118031180311803,
    0.03119031190311903,
    0.03120031200312003,
    0.03121031210312103,
    0.03122031220312203,
    0.03123031230312303,
    0.03124031240312403,
    0.03125031250312503,
    0.031260312603126034,
    0.03127031270312703,
    0.031280312803128034,
    0.03129031290312903,
    0.031300313003130034,
    0.03131031310313103,
    0.03132031320313203,
    0.03133031330313303,
    0.03134031340313403,
    0.03135031350313503,
    0.03136031360313603,
    0.03137031370313703,
    0.03138031380313803,
    0.03139031390313903,
    0.03140031400314003,
    0.03141031410314103,
    0.03142031420314203,
    0.03143031430314303,
    0.03144031440314403,
    0.03145031450314503,
    0.03146031460314603,
    0.03147031470314703,
    0.03148031480314803,
    0.03149031490314903,
    0.03150031500315003,
    0.03151031510315103,
    0.03152031520315203,
    0.031530315303153034,
    0.03154031540315403,
    0.031550315503155034,
    0.03156031560315603,
    0.031570315703157034,
    0.03158031580315803,
    0.031590315903159034,
    0.03160031600316003,
    0.03161031610316103,
    0.03162031620316203,
    0.03163031630316303,
    0.03164031640316403,
    0.03165031650316503,
    0.03166031660316603,
    0.03167031670316703,
    0.03168031680316803,
    0.03169031690316903,
    0.03170031700317003,
    0.03171031710317103,
    0.03172031720317203,
    0.03173031730317303,
    0.03174031740317403,
    0.03175031750317503,
    0.03176031760317603,
    0.03177031770317703,
    0.03178031780317803,
    0.03179031790317903,
    0.031800318003180034,
    0.03181031810318103,
    0.031820318203182034,
    0.03183031830318303,
    0.031840318403184034,
    0.03185031850318503,
    0.031860318603186034,
    0.03187031870318703,
    0.031880318803188035,
    0.03189031890318903,
    0.03190031900319003,
    0.03191031910319103,
    0.03192031920319203,
    0.03193031930319303,
    0.03194031940319403,
    0.03195031950319503,
    0.03196031960319603,
    0.03197031970319703,
    0.03198031980319803,
    0.03199031990319903,
    0.03200032000320003,
    0.03201032010320103,
    0.03202032020320203,
    0.03203032030320303,
    0.03204032040320403,
    0.03205032050320503,
    0.03206032060320603,
    0.032070320703207034,
    0.03208032080320803,
    0.032090320903209034,
    0.03210032100321003,
    0.032110321103211034,
    0.03212032120321203,
    0.032130321303213034,
    0.03214032140321403,
    0.032150321503215035,
    0.03216032160321603,
    0.032170321703217035,
    0.03218032180321803,
    0.03219032190321903,
    0.03220032200322003,
    0.03221032210322103,
    0.03222032220322203,
    0.03223032230322303,
    0.03224032240322403,
    0.03225032250322503,
    0.03226032260322603,
    0.03227032270322703,
    0.03228032280322803,
    0.03229032290322903,
    0.03230032300323003,
    0.03231032310323103,
    0.03232032320323203,
    0.03233032330323303,
    0.032340323403234034,
    0.03235032350323503,
    0.032360323603236034,
    0.03237032370323703,
    0.032380323803238034,
    0.03239032390323903,
    0.032400324003240034,
    0.03241032410324103,
    0.032420324203242035,
    0.03243032430324303,
    0.032440324403244035,
    0.03245032450324503,
    0.032460324603246035,
    0.03247032470324703,
    0.03248032480324803,
    0.03249032490324903,
    0.03250032500325003,
    0.03251032510325103,
    0.03252032520325203,
    0.03253032530325303,
    0.03254032540325403,
    0.03255032550325503,
    0.03256032560325603,
    0.03257032570325703,
    0.03258032580325803,
    0.03259032590325903,
    0.03260032600326003,
    0.032610326103261034,
    0.03262032620326203,
    0.032630326303263034,
    0.03264032640326403,
    0.032650326503265034,
    0.03266032660326603,
    0.032670326703267034,
    0.03268032680326803,
    0.032690326903269035,
    0.03270032700327003,
    0.032710327103271035,
    0.03272032720327203,
    0.032730327303273035,
    0.03274032740327403,
    0.032750327503275035,
    0.03276032760327603,
    0.03277032770327703,
    0.03278032780327803,
    0.03279032790327903,
    0.03280032800328003,
    0.03281032810328103,
    0.03282032820328203,
    0.03283032830328303,
    0.03284032840328403,
    0.03285032850328503,
    0.03286032860328603,
    0.03287032870328703,
    0.032880328803288034,
    0.03289032890328903,
    0.032900329003290034,
    0.03291032910329103,
    0.032920329203292034,
    0.03293032930329303,
    0.032940329403294034,
    0.03295032950329503,
    0.032960329603296035,
    0.03297032970329703,
    0.032980329803298035,
    0.03299032990329903,
    0.033000330003300035,
    0.03301033010330103,
    0.033020330203302035,
    0.03303033030330303,
    0.033040330403304036,
    0.03305033050330503,
    0.033060330603306036,
    0.03307033070330703,
    0.03308033080330803,
    0.03309033090330903,
    0.03310033100331003,
    0.03311033110331103,
    0.03312033120331203,
    0.03313033130331303,
    0.03314033140331403,
    0.033150331503315034,
    0.03316033160331603,
    0.033170331703317034,
    0.03318033180331803,
    0.033190331903319034,
    0.03320033200332003,
    0.033210332103321034,
    0.03322033220332203,
    0.033230332303323035,
    0.03324033240332403,
    0.033250332503325035,
    0.03326033260332603,
    0.033270332703327035,
    0.03328033280332803,
    0.033290332903329035,
    0.03330033300333003,
    0.033310333103331036,
    0.03332033320333203,
    0.033330333303333036,
    0.03334033340333403,
    0.033350333503335036,
    0.03336033360333603,
    0.03337033370333703,
    0.03338033380333803,
    0.03339033390333903,
    0.03340033400334003,
    0.03341033410334103,
    0.033420334203342034,
    0.03343033430334303,
    0.033440334403344034,
    0.03345033450334503,
    0.033460334603346034,
    0.03347033470334703,
    0.033480334803348034,
    0.03349033490334903,
    0.033500335003350035,
    0.03351033510335103,
    0.033520335203352035,
    0.03353033530335303,
    0.033540335403354035,
    0.03355033550335503,
    0.033560335603356035,
    0.03357033570335703,
    0.033580335803358036,
    0.03359033590335903,
    0.033600336003360036,
    0.03361033610336103,
    0.033620336203362036,
    0.03363033630336303,
    0.033640336403364036,
    0.03365033650336503,
    0.03366033660336603,
    0.03367033670336703,
    0.03368033680336803,
    0.033690336903369034,
    0.03370033700337003,
    0.033710337103371034,
    0.03372033720337203,
    0.033730337303373034,
    0.03374033740337403,
    0.033750337503375034,
    0.03376033760337603,
    0.033770337703377035,
    0.03378033780337803,
    0.033790337903379035,
    0.03380033800338003,
    0.033810338103381035,
    0.03382033820338203,
    0.033830338303383035,
    0.03384033840338403,
    0.033850338503385036,
    0.03386033860338603,
    0.033870338703387036,
    0.03388033880338803,
    0.033890338903389036,
    0.03390033900339003,
    0.033910339103391036,
    0.03392033920339203,
    0.03393033930339304,
    0.03394033940339403,
    0.03395033950339503,
    0.033960339603396034,
    0.03397033970339703,
    0.033980339803398034,
    0.03399033990339903,
    0.034000340003400034,
    0.03401034010340103,
    0.034020340203402034,
    0.03403034030340303,
    0.034040340403404035,
    0.03405034050340503,
    0.034060340603406035,
    0.03407034070340703,
    0.034080340803408035,
    0.03409034090340903,
    0.034100341003410035,
    0.03411034110341103,
    0.034120341203412036,
    0.03413034130341303,
    0.034140341403414036,
    0.03415034150341503,
    0.034160341603416036,
    0.03417034170341703,
    0.034180341803418036,
    0.03419034190341903,
    0.03420034200342004,
    0.03421034210342103,
    0.03422034220342204,
    0.034230342303423034,
    0.03424034240342403,
    0.034250342503425034,
    0.03426034260342603,
    0.034270342703427034,
    0.03428034280342803,
    0.034290342903429034,
    0.03430034300343003,
    0.034310343103431035,
    0.03432034320343203,
    0.034330343303433035,
    0.03434034340343403,
    0.034350343503435035,
    0.03436034360343603,
    0.034370343703437035,
    0.03438034380343803,
    0.034390343903439036,
    0.03440034400344003,
    0.034410344103441036,
    0.03442034420344203,
    0.034430344303443036,
    0.03444034440344403,
    0.034450344503445036,
    0.03446034460344603,
    0.03447034470344704,
    0.03448034480344803,
    0.03449034490344904,
    0.034500345003450034,
    0.03451034510345104,
    0.034520345203452034,
    0.03453034530345303,
    0.034540345403454034,
    0.03455034550345503,
    0.034560345603456034,
    0.03457034570345703,
    0.034580345803458035,
    0.03459034590345903,
    0.034600346003460035,
    0.03461034610346103,
    0.034620346203462035,
    0.03463034630346303,
    0.034640346403464035,
    0.03465034650346503,
    0.034660346603466036,
    0.03467034670346703,
    0.034680346803468036,
    0.03469034690346903,
    0.034700347003470036,
    0.03471034710347103,
    0.034720347203472036,
    0.03473034730347303,
    0.03474034740347404,
    0.03475034750347503,
    0.03476034760347604,
    0.034770347703477034,
    0.03478034780347804,
    0.034790347903479034,
    0.03480034800348004,
    0.034810348103481034,
    0.03482034820348203,
    0.034830348303483034,
    0.03484034840348403,
    0.034850348503485035,
    0.03486034860348603,
    0.034870348703487035,
    0.03488034880348803,
    0.034890348903489035,
    0.03490034900349003,
    0.034910349103491035,
    0.03492034920349203,
    0.034930349303493036,
    0.03494034940349403,
    0.034950349503495036,
    0.03496034960349603,
    0.034970349703497036,
    0.03498034980349803,
    0.034990349903499036,
    0.03500035000350003,
    0.03501035010350104,
    0.03502035020350203,
    0.03503035030350304,
    0.035040350403504034,
    0.03505035050350504,
    0.035060350603506034,
    0.03507035070350704,
    0.035080350803508034,
    0.03509035090350904,
    0.035100351003510034,
    0.03511035110351103,
    0.035120351203512035,
    0.03513035130351303,
    0.035140351403514035,
    0.03515035150351503,
    0.035160351603516035,
    0.03517035170351703,
    0.035180351803518035,
    0.03519035190351903,
    0.035200352003520036,
    0.03521035210352103,
    0.035220352203522036,
    0.03523035230352303,
    0.035240352403524036,
    0.03525035250352503,
    0.035260352603526036,
    0.03527035270352703,
    0.03528035280352804,
    0.03529035290352903,
    0.03530035300353004,
    0.035310353103531034,
    0.03532035320353204,
    0.035330353303533034,
    0.03534035340353404,
    0.035350353503535034,
    0.03536035360353604,
    0.035370353703537034,
    0.03538035380353804,
    0.035390353903539035,
    0.03540035400354004,
    0.035410354103541035,
    0.03542035420354203,
    0.035430354303543035,
    0.03544035440354403,
    0.035450354503545035,
    0.03546035460354603,
    0.035470354703547036,
    0.03548035480354803,
    0.035490354903549036,
    0.03550035500355003,
    0.035510355103551036,
    0.03552035520355203,
    0.035530355303553036,
    0.03554035540355403,
    0.03555035550355504,
    0.03556035560355603,
    0.03557035570355704,
    0.035580355803558034,
    0.03559035590355904,
    0.035600356003560034,
    0.03561035610356104,
    0.035620356203562034,
    0.03563035630356304,
    0.035640356403564034,
    0.03565035650356504,
    0.035660356603566035,
    0.03567035670356704,
    0.035680356803568035,
    0.03569035690356904,
    0.035700357003570035,
    0.03571035710357103,
    0.035720357203572035,
    0.03573035730357303,
    0.035740357403574036,
    0.03575035750357503,
    0.035760357603576036,
    0.03577035770357703,
    0.035780357803578036,
    0.03579035790357903,
    0.035800358003580036,
    0.03581035810358103,
    0.03582035820358204,
    0.03583035830358303,
    0.03584035840358404,
    0.035850358503585034,
    0.03586035860358604,
    0.035870358703587034,
    0.03588035880358804,
    0.035890358903589034,
    0.03590035900359004,
    0.035910359103591034,
    0.03592035920359204,
    0.035930359303593035,
    0.03594035940359404,
    0.035950359503595035,
    0.03596035960359604,
    0.035970359703597035,
    0.03598035980359804,
    0.035990359903599035,
    0.03600036000360003,
    0.036010360103601036,
    0.03602036020360203,
    0.036030360303603036,
    0.03604036040360403,
    0.036050360503605036,
    0.03606036060360603,
    0.036070360703607036,
    0.03608036080360803,
    0.03609036090360904,
    0.03610036100361003,
    0.03611036110361104,
    0.036120361203612034,
    0.03613036130361304,
    0.036140361403614034,
    0.03615036150361504,
    0.036160361603616034,
    0.03617036170361704,
    0.036180361803618034,
    0.03619036190361904,
    0.036200362003620035,
    0.03621036210362104,
    0.036220362203622035,
    0.03623036230362304,
    0.036240362403624035,
    0.03625036250362504,
    0.036260362603626035,
    0.03627036270362704,
    0.036280362803628036,
    0.03629036290362903,
    0.036300363003630036,
    0.03631036310363103,
    0.036320363203632036,
    0.03633036330363303,
    0.036340363403634036,
    0.03635036350363503,
    0.03636036360363604,
    0.03637036370363703,
    0.03638036380363804,
    0.036390363903639034,
    0.03640036400364004,
    0.036410364103641034,
    0.03642036420364204,
    0.036430364303643034,
    0.03644036440364404,
    0.036450364503645034,
    0.03646036460364604,
    0.036470364703647035,
    0.03648036480364804,
    0.036490364903649035,
    0.03650036500365004,
    0.036510365103651035,
    0.03652036520365204,
    0.036530365303653035,
    0.03654036540365404,
    0.036550365503655036,
    0.03656036560365604,
    0.036570365703657036,
    0.03658036580365803,
    0.036590365903659036,
    0.03660036600366003,
    0.036610366103661036,
    0.03662036620366203,
    0.03663036630366304,
    0.03664036640366403,
    0.03665036650366504,
    0.036660366603666034,
    0.03667036670366704,
    0.036680366803668034,
    0.03669036690366904,
    0.036700367003670034,
    0.03671036710367104,
    0.036720367203672034,
    0.03673036730367304,
    0.036740367403674035,
    0.03675036750367504,
    0.036760367603676035,
    0.03677036770367704,
    0.036780367803678035,
    0.03679036790367904,
    0.036800368003680035,
    0.03681036810368104,
    0.036820368203682036,
    0.03683036830368304,
    0.036840368403684036,
    0.03685036850368504,
    0.036860368603686036,
    0.03687036870368703,
    0.036880368803688036,
    0.03689036890368903,
    0.03690036900369004,
    0.03691036910369103,
    0.03692036920369204,
    0.036930369303693034,
    0.03694036940369404,
    0.036950369503695034,
    0.03696036960369604,
    0.036970369703697034,
    0.03698036980369804,
    0.036990369903699034,
    0.03700037000370004,
    0.037010370103701035,
    0.03702037020370204,
    0.037030370303703035,
    0.03704037040370404,
    0.037050370503705035,
    0.03706037060370604,
    0.037070370703707035,
    0.03708037080370804,
    0.037090370903709036,
    0.03710037100371004,
    0.037110371103711036,
    0.03712037120371204,
    0.037130371303713036,
    0.03714037140371404,
    0.037150371503715036,
    0.03716037160371603,
    0.03717037170371704,
    0.03718037180371803,
    0.03719037190371904,
    0.037200372003720034,
    0.03721037210372104,
    0.037220372203722034,
    0.03723037230372304,
    0.037240372403724034,
    0.03725037250372504,
    0.037260372603726034,
    0.03727037270372704,
    0.037280372803728035,
    0.03729037290372904,
    0.037300373003730035,
    0.03731037310373104,
    0.037320373203732035,
    0.03733037330373304,
    0.037340373403734035,
    0.03735037350373504,
    0.037360373603736036,
    0.03737037370373704,
    0.037380373803738036,
    0.03739037390373904,
    0.037400374003740036,
    0.03741037410374104,
    0.037420374203742036,
    0.03743037430374304,
    0.03744037440374404,
    0.03745037450374503,
    0.03746037460374604,
    0.037470374703747034,
    0.03748037480374804,
    0.037490374903749034,
    0.03750037500375004,
    0.037510375103751034,
    0.03752037520375204,
    0.037530375303753034,
    0.03754037540375404,
    0.037550375503755035,
    0.03756037560375604,
    0.037570375703757035,
    0.03758037580375804,
    0.037590375903759035,
    0.03760037600376004,
    0.037610376103761035,
    0.03762037620376204,
    0.037630376303763036,
    0.03764037640376404,
    0.037650376503765036,
    0.03766037660376604,
    0.037670376703767036,
    0.03768037680376804,
    0.037690376903769036,
    0.03770037700377004,
    0.03771037710377104,
    0.03772037720377204,
    0.03773037730377304,
    0.03774037740377404,
    0.03775037750377504,
    0.037760377603776034,
    0.03777037770377704,
    0.037780377803778034,
    0.03779037790377904,
    0.037800378003780034,
    0.03781037810378104,
    0.037820378203782035,
    0.03783037830378304,
    0.037840378403784035,
    0.03785037850378504,
    0.037860378603786035,
    0.03787037870378704,
    0.037880378803788035,
    0.03789037890378904,
    0.037900379003790036,
    0.03791037910379104,
    0.037920379203792036,
    0.03793037930379304,
    0.037940379403794036,
    0.03795037950379504,
    0.037960379603796036,
    0.03797037970379704,
    0.03798037980379804,
    0.03799037990379904,
    0.03800038000380004,
    0.03801038010380104,
    0.03802038020380204,
    0.03803038030380304,
    0.03804038040380404,
    0.038050380503805034,
    0.03806038060380604,
    0.038070380703807034,
    0.03808038080380804,
    0.038090380903809035,
    0.03810038100381004,
    0.038110381103811035,
    0.03812038120381204,
    0.038130381303813035,
    0.03814038140381404,
    0.038150381503815035,
    0.03816038160381604,
    0.038170381703817036,
    0.03818038180381804,
    0.038190381903819036,
    0.03820038200382004,
    0.038210382103821036,
    0.03822038220382204,
    0.038230382303823036,
    0.03824038240382404,
    0.03825038250382504,
    0.03826038260382604,
    0.03827038270382704,
    0.03828038280382804,
    0.03829038290382904,
    0.03830038300383004,
    0.03831038310383104,
    0.03832038320383204,
    0.03833038330383304,
    0.038340383403834034,
    0.03835038350383504,
    0.038360383603836035,
    0.03837038370383704,
    0.038380383803838035,
    0.03839038390383904,
    0.038400384003840035,
    0.03841038410384104,
    0.038420384203842035,
    0.03843038430384304,
    0.038440384403844036,
    0.03845038450384504,
    0.038460384603846036,
    0.03847038470384704,
    0.038480384803848036,
    0.03849038490384904,
    0.038500385003850036,
    0.03851038510385104,
    0.03852038520385204,
    0.03853038530385304,
    0.03854038540385404,
    0.03855038550385504,
    0.03856038560385604,
    0.03857038570385704,
    0.03858038580385804,
    0.03859038590385904,
    0.03860038600386004,
    0.03861038610386104,
    0.03862038620386204,
    0.038630386303863035,
    0.03864038640386404,
    0.038650386503865035,
    0.03866038660386604,
    0.038670386703867035,
    0.03868038680386804,
    0.038690386903869035,
    0.03870038700387004,
    0.038710387103871036,
    0.03872038720387204,
    0.038730387303873036,
    0.03874038740387404,
    0.038750387503875036,
    0.03876038760387604,
    0.038770387703877036,
    0.03878038780387804,
    0.03879038790387904,
    0.03880038800388004,
    0.03881038810388104,
    0.03882038820388204,
    0.03883038830388304,
    0.03884038840388404,
    0.03885038850388504,
    0.03886038860388604,
    0.03887038870388704,
    0.03888038880388804,
    0.03889038890388904,
    0.03890038900389004,
    0.03891038910389104,
    0.038920389203892035,
    0.03893038930389304,
    0.038940389403894035,
    0.03895038950389504,
    0.038960389603896035,
    0.03897038970389704,
    0.038980389803898036,
    0.03899038990389904,
    0.039000390003900036,
    0.03901039010390104,
    0.039020390203902036,
    0.03903039030390304,
    0.039040390403904036,
    0.03905039050390504,
    0.03906039060390604,
    0.03907039070390704,
    0.03908039080390804,
    0.03909039090390904,
    0.03910039100391004,
    0.03911039110391104,
    0.03912039120391204,
    0.03913039130391304,
    0.03914039140391404,
    0.03915039150391504,
    0.03916039160391604,
    0.03917039170391704,
    0.03918039180391804,
    0.03919039190391904,
    0.03920039200392004,
    0.039210392103921035,
    0.03922039220392204,
    0.039230392303923035,
    0.03924039240392404,
    0.039250392503925036,
    0.03926039260392604,
    0.039270392703927036,
    0.03928039280392804,
    0.039290392903929036,
    0.03930039300393004,
    0.039310393103931036,
    0.03932039320393204,
    0.03933039330393304,
    0.03934039340393404,
    0.03935039350393504,
    0.03936039360393604,
    0.03937039370393704,
    0.03938039380393804,
    0.03939039390393904,
    0.03940039400394004,
    0.03941039410394104,
    0.03942039420394204,
    0.03943039430394304,
    0.03944039440394404,
    0.03945039450394504,
    0.03946039460394604,
    0.03947039470394704,
    0.03948039480394804,
    0.03949039490394904,
    0.039500395003950035,
    0.03951039510395104,
    0.039520395203952036,
    0.03953039530395304,
    0.039540395403954036,
    0.03955039550395504,
    0.039560395603956036,
    0.03957039570395704,
    0.039580395803958036,
    0.03959039590395904,
    0.03960039600396004,
    0.03961039610396104,
    0.03962039620396204,
    0.03963039630396304,
    0.03964039640396404,
    0.03965039650396504,
    0.03966039660396604,
    0.03967039670396704,
    0.03968039680396804,
    0.03969039690396904,
    0.03970039700397004,
    0.03971039710397104,
    0.03972039720397204,
    0.03973039730397304,
    0.03974039740397404,
    0.03975039750397504,
    0.03976039760397604,
    0.03977039770397704,
    0.03978039780397804,
    0.039790397903979036,
    0.03980039800398004,
    0.039810398103981036,
    0.03982039820398204,
    0.039830398303983036,
    0.03984039840398404,
    0.039850398503985036,
    0.03986039860398604,
    0.03987039870398704,
    0.03988039880398804,
    0.03989039890398904,
    0.03990039900399004,
    0.03991039910399104,
    0.03992039920399204,
    0.03993039930399304,
    0.03994039940399404,
    0.03995039950399504,
    0.03996039960399604,
    0.03997039970399704,
    0.03998039980399804,
    0.03999039990399904,
    0.04000040000400004,
    0.04001040010400104,
    0.04002040020400204,
    0.04003040030400304,
    0.04004040040400404,
    0.04005040050400504,
    0.04006040060400604,
    0.04007040070400704,
    0.04008040080400804,
    0.04009040090400904,
    0.040100401004010036,
    0.04011040110401104,
    0.040120401204012036,
    0.04013040130401304,
    0.04014040140401404,
    0.04015040150401504,
    0.04016040160401604,
    0.04017040170401704,
    0.04018040180401804,
    0.04019040190401904,
    0.04020040200402004,
    0.04021040210402104,
    0.04022040220402204,
    0.04023040230402304,
    0.04024040240402404,
    0.04025040250402504,
    0.04026040260402604,
    0.04027040270402704,
    0.04028040280402804,
    0.04029040290402904,
    0.04030040300403004,
    0.04031040310403104,
    0.04032040320403204,
    0.04033040330403304,
    0.04034040340403404,
    0.04035040350403504,
    0.04036040360403604,
    0.04037040370403704,
    0.04038040380403804,
    0.040390403904039036,
    0.04040040400404004,
    0.04041040410404104,
    0.04042040420404204,
    0.04043040430404304,
    0.04044040440404404,
    0.04045040450404504,
    0.04046040460404604,
    0.04047040470404704,
    0.04048040480404804,
    0.04049040490404904,
    0.04050040500405004,
    0.04051040510405104,
    0.04052040520405204,
    0.04053040530405304,
    0.04054040540405404,
    0.04055040550405504,
    0.04056040560405604,
    0.04057040570405704,
    0.04058040580405804,
    0.04059040590405904,
    0.04060040600406004,
    0.04061040610406104,
    0.04062040620406204,
    0.04063040630406304,
    0.04064040640406404,
    0.04065040650406504,
    0.04066040660406604,
    0.04067040670406704,
    0.04068040680406804,
    0.04069040690406904,
    0.04070040700407004,
    0.04071040710407104,
    0.04072040720407204,
    0.04073040730407304,
    0.04074040740407404,
    0.04075040750407504,
    0.04076040760407604,
    0.04077040770407704,
    0.04078040780407804,
    0.04079040790407904,
    0.04080040800408004,
    0.04081040810408104,
    0.04082040820408204,
    0.04083040830408304,
    0.04084040840408404,
    0.04085040850408504,
    0.04086040860408604,
    0.04087040870408704,
    0.04088040880408804,
    0.04089040890408904,
    0.04090040900409004,
    0.04091040910409104,
    0.04092040920409204,
    0.04093040930409304,
    0.04094040940409404,
    0.040950409504095044,
    0.04096040960409604,
    0.04097040970409704,
    0.04098040980409804,
    0.04099040990409904,
    0.04100041000410004,
    0.04101041010410104,
    0.04102041020410204,
    0.04103041030410304,
    0.04104041040410404,
    0.04105041050410504,
    0.04106041060410604,
    0.04107041070410704,
    0.04108041080410804,
    0.04109041090410904,
    0.04110041100411004,
    0.04111041110411104,
    0.04112041120411204,
    0.04113041130411304,
    0.04114041140411404,
    0.04115041150411504,
    0.04116041160411604,
    0.04117041170411704,
    0.04118041180411804,
    0.04119041190411904,
    0.04120041200412004,
    0.04121041210412104,
    0.041220412204122044,
    0.04123041230412304,
    0.041240412404124044,
    0.04125041250412504,
    0.04126041260412604,
    0.04127041270412704,
    0.04128041280412804,
    0.04129041290412904,
    0.04130041300413004,
    0.04131041310413104,
    0.04132041320413204,
    0.04133041330413304,
    0.04134041340413404,
    0.04135041350413504,
    0.04136041360413604,
    0.04137041370413704,
    0.04138041380413804,
    0.04139041390413904,
    0.04140041400414004,
    0.04141041410414104,
    0.04142041420414204,
    0.04143041430414304,
    0.04144041440414404,
    0.04145041450414504,
    0.04146041460414604,
    0.04147041470414704,
    0.04148041480414804,
    0.041490414904149044,
    0.04150041500415004,
    0.041510415104151044,
    0.04152041520415204,
    0.041530415304153044,
    0.04154041540415404,
    0.04155041550415504,
    0.04156041560415604,
    0.04157041570415704,
    0.04158041580415804,
    0.04159041590415904,
    0.04160041600416004,
    0.04161041610416104,
    0.04162041620416204,
    0.04163041630416304,
    0.04164041640416404,
    0.04165041650416504,
    0.04166041660416604,
    0.04167041670416704,
    0.04168041680416804,
    0.04169041690416904,
    0.04170041700417004,
    0.04171041710417104,
    0.04172041720417204,
    0.04173041730417304,
    0.04174041740417404,
    0.04175041750417504,
    0.041760417604176044,
    0.04177041770417704,
    0.041780417804178044,
    0.04179041790417904,
    0.041800418004180044,
    0.04181041810418104,
    0.041820418204182044,
    0.04183041830418304,
    0.04184041840418404,
    0.04185041850418504,
    0.04186041860418604,
    0.04187041870418704,
    0.04188041880418804,
    0.04189041890418904,
    0.04190041900419004,
    0.04191041910419104,
    0.04192041920419204,
    0.04193041930419304,
    0.04194041940419404,
    0.04195041950419504,
    0.04196041960419604,
    0.04197041970419704,
    0.04198041980419804,
    0.04199041990419904,
    0.04200042000420004,
    0.04201042010420104,
    0.04202042020420204,
    0.042030420304203044,
    0.04204042040420404,
    0.042050420504205044,
    0.04206042060420604,
    0.042070420704207044,
    0.04208042080420804,
    0.042090420904209044,
    0.04210042100421004,
    0.042110421104211045,
    0.04212042120421204,
    0.042130421304213045,
    0.04214042140421404,
    0.04215042150421504,
    0.04216042160421604,
    0.04217042170421704,
    0.04218042180421804,
    0.04219042190421904,
    0.04220042200422004,
    0.04221042210422104,
    0.04222042220422204,
    0.04223042230422304,
    0.04224042240422404,
    0.04225042250422504,
    0.04226042260422604,
    0.04227042270422704,
    0.04228042280422804,
    0.04229042290422904,
    0.042300423004230044,
    0.04231042310423104,
    0.042320423204232044,
    0.04233042330423304,
    0.042340423404234044,
    0.04235042350423504,
    0.042360423604236044,
    0.04237042370423704,
    0.042380423804238045,
    0.04239042390423904,
    0.042400424004240045,
    0.04241042410424104,
    0.042420424204242045,
    0.04243042430424304,
    0.04244042440424404,
    0.04245042450424504,
    0.04246042460424604,
    0.04247042470424704,
    0.04248042480424804,
    0.04249042490424904,
    0.04250042500425004,
    0.04251042510425104,
    0.04252042520425204,
    0.04253042530425304,
    0.04254042540425404,
    0.04255042550425504,
    0.04256042560425604,
    0.042570425704257044,
    0.04258042580425804,
    0.042590425904259044,
    0.04260042600426004,
    0.042610426104261044,
    0.04262042620426204,
    0.042630426304263044,
    0.04264042640426404,
    0.042650426504265045,
    0.04266042660426604,
    0.042670426704267045,
    0.04268042680426804,
    0.042690426904269045,
    0.04270042700427004,
    0.042710427104271045,
    0.04272042720427204,
    0.04273042730427304,
    0.04274042740427404,
    0.04275042750427504,
    0.04276042760427604,
    0.04277042770427704,
    0.04278042780427804,
    0.04279042790427904,
    0.04280042800428004,
    0.04281042810428104,
    0.04282042820428204,
    0.04283042830428304,
    0.042840428404284044,
    0.04285042850428504,
    0.042860428604286044,
    0.04287042870428704,
    0.042880428804288044,
    0.04289042890428904,
    0.042900429004290044,
    0.04291042910429104,
    0.042920429204292045,
    0.04293042930429304,
    0.042940429404294045,
    0.04295042950429504,
    0.042960429604296045,
    0.04297042970429704,
    0.042980429804298045,
    0.04299042990429904,
    0.043000430004300046,
    0.04301043010430104,
    0.04302043020430204,
    0.04303043030430304,
    0.04304043040430404,
    0.04305043050430504,
    0.04306043060430604,
    0.04307043070430704,
    0.04308043080430804,
    0.04309043090430904,
    0.04310043100431004,
    0.043110431104311044,
    0.04312043120431204,
    0.043130431304313044,
    0.04314043140431404,
    0.043150431504315044,
    0.04316043160431604,
    0.043170431704317044,
    0.04318043180431804,
    0.043190431904319045,
    0.04320043200432004,
    0.043210432104321045,
    0.04322043220432204,
    0.043230432304323045,
    0.04324043240432404,
    0.043250432504325045,
    0.04326043260432604,
    0.043270432704327046,
    0.04328043280432804,
    0.043290432904329046,
    0.04330043300433004,
    0.04331043310433104,
    0.04332043320433204,
    0.04333043330433304,
    0.04334043340433404,
    0.04335043350433504,
    0.04336043360433604,
    0.04337043370433704,
    0.043380433804338044,
    0.04339043390433904,
    0.043400434004340044,
    0.04341043410434104,
    0.043420434204342044,
    0.04343043430434304,
    0.043440434404344044,
    0.04345043450434504,
    0.043460434604346045,
    0.04347043470434704,
    0.043480434804348045,
    0.04349043490434904,
    0.043500435004350045,
    0.04351043510435104,
    0.043520435204352045,
    0.04353043530435304,
    0.043540435404354046,
    0.04355043550435504,
    0.043560435604356046,
    0.04357043570435704,
    0.043580435804358046,
    0.04359043590435904,
    0.04360043600436004,
    0.04361043610436104,
    0.04362043620436204,
    0.04363043630436304,
    0.04364043640436404,
    0.043650436504365044,
    0.04366043660436604,
    0.043670436704367044,
    0.04368043680436804,
    0.043690436904369044,
    0.04370043700437004,
    0.043710437104371044,
    0.04372043720437204,
    0.043730437304373045,
    0.04374043740437404,
    0.043750437504375045,
    0.04376043760437604,
    0.043770437704377045,
    0.04378043780437804,
    0.043790437904379045,
    0.04380043800438004,
    0.043810438104381046,
    0.04382043820438204,
    0.043830438304383046,
    0.04384043840438404,
    0.043850438504385046,
    0.04386043860438604,
    0.043870438704387046,
    0.04388043880438804,
    0.04389043890438904,
    0.04390043900439004,
    0.04391043910439104,
    0.043920439204392044,
    0.04393043930439304,
    0.043940439404394044,
    0.04395043950439504,
    0.043960439604396044,
    0.04397043970439704,
    0.043980439804398044,
    0.04399043990439904,
    0.044000440004400045,
    0.04401044010440104,
    0.044020440204402045,
    0.04403044030440304,
    0.044040440404404045,
    0.04405044050440504,
    0.044060440604406045,
    0.04407044070440704,
    0.044080440804408046,
    0.04409044090440904,
    0.044100441004410046,
    0.04411044110441104,
    0.044120441204412046,
    0.04413044130441304,
    0.044140441404414046,
    0.04415044150441504,
    0.04416044160441605,
    0.04417044170441704,
    0.04418044180441804,
    0.044190441904419044,
    0.04420044200442004,
    0.044210442104421044,
    0.04422044220442204,
    0.044230442304423044,
    0.04424044240442404,
    0.044250442504425044,
    0.04426044260442604,
    0.044270442704427045,
    0.04428044280442804,
    0.044290442904429045,
    0.04430044300443004,
    0.044310443104431045,
    0.04432044320443204,
    0.044330443304433045,
    0.04434044340443404,
    0.044350443504435046,
    0.04436044360443604,
    0.044370443704437046,
    0.04438044380443804,
    0.044390443904439046,
    0.04440044400444004,
    0.044410444104441046,
    0.04442044420444204,
    0.04443044430444305,
    0.04444044440444404,
    0.04445044450444505,
    0.044460444604446044,
    0.04447044470444705,
    0.044480444804448044,
    0.04449044490444904,
    0.044500445004450044,
    0.04451044510445104,
    0.044520445204452044,
    0.04453044530445304,
    0.044540445404454045,
    0.04455044550445504,
    0.044560445604456045,
    0.04457044570445704,
    0.044580445804458045,
    0.04459044590445904,
    0.044600446004460045,
    0.04461044610446104,
    0.044620446204462046,
    0.04463044630446304,
    0.044640446404464046,
    0.04465044650446504,
    0.044660446604466046,
    0.04467044670446704,
    0.044680446804468046,
    0.04469044690446904,
    0.04470044700447005,
    0.04471044710447104,
    0.04472044720447205,
    0.044730447304473044,
    0.04474044740447405,
    0.044750447504475044,
    0.04476044760447605,
    0.044770447704477044,
    0.04478044780447804,
    0.044790447904479044,
    0.04480044800448004,
    0.044810448104481045,
    0.04482044820448204,
    0.044830448304483045,
    0.04484044840448404,
    0.044850448504485045,
    0.04486044860448604,
    0.044870448704487045,
    0.04488044880448804,
    0.044890448904489046,
    0.04490044900449004,
    0.044910449104491046,
    0.04492044920449204,
    0.044930449304493046,
    0.04494044940449404,
    0.044950449504495046,
    0.04496044960449604,
    0.04497044970449705,
    0.04498044980449804,
    0.04499044990449905,
    0.045000450004500044,
    0.04501045010450105,
    0.045020450204502044,
    0.04503045030450305,
    0.045040450404504044,
    0.04505045050450505,
    0.045060450604506044,
    0.04507045070450704,
    0.045080450804508045,
    0.04509045090450904,
    0.045100451004510045,
    0.04511045110451104,
    0.045120451204512045,
    0.04513045130451304,
    0.045140451404514045,
    0.04515045150451504,
    0.045160451604516046,
    0.04517045170451704,
    0.045180451804518046,
    0.04519045190451904,
    0.045200452004520046,
    0.04521045210452104,
    0.045220452204522046,
    0.04523045230452304,
    0.04524045240452405,
    0.04525045250452504,
    0.04526045260452605,
    0.045270452704527044,
    0.04528045280452805,
    0.045290452904529044,
    0.04530045300453005,
    0.045310453104531044,
    0.04532045320453205,
    0.045330453304533044,
    0.04534045340453405,
    0.045350453504535045,
    0.04536045360453604,
    0.045370453704537045,
    0.04538045380453804,
    0.045390453904539045,
    0.04540045400454004,
    0.045410454104541045,
    0.04542045420454204,
    0.045430454304543046,
    0.04544045440454404,
    0.045450454504545046,
    0.04546045460454604,
    0.045470454704547046,
    0.04548045480454804,
    0.045490454904549046,
    0.04550045500455004,
    0.04551045510455105,
    0.04552045520455204,
    0.04553045530455305,
    0.045540455404554044,
    0.04555045550455505,
    0.045560455604556044,
    0.04557045570455705,
    0.045580455804558044,
    0.04559045590455905,
    0.045600456004560044,
    0.04561045610456105,
    0.045620456204562045,
    0.04563045630456305,
    0.045640456404564045,
    0.04565045650456504,
    0.045660456604566045,
    0.04567045670456704,
    0.045680456804568045,
    0.04569045690456904,
    0.045700457004570046,
    0.04571045710457104,
    0.045720457204572046,
    0.04573045730457304,
    0.045740457404574046,
    0.04575045750457504,
    0.045760457604576046,
    0.04577045770457704,
    0.04578045780457805,
    0.04579045790457904,
    0.04580045800458005,
    0.045810458104581044,
    0.04582045820458205,
    0.045830458304583044,
    0.04584045840458405,
    0.045850458504585044,
    0.04586045860458605,
    0.045870458704587044,
    0.04588045880458805,
    0.045890458904589045,
    0.04590045900459005,
    0.045910459104591045,
    0.04592045920459205,
    0.045930459304593045,
    0.04594045940459404,
    0.045950459504595045,
    0.04596045960459604,
    0.045970459704597046,
    0.04598045980459804,
    0.045990459904599046,
    0.04600046000460004,
    0.046010460104601046,
    0.04602046020460204,
    0.046030460304603046,
    0.04604046040460404,
    0.04605046050460505,
    0.04606046060460604,
    0.04607046070460705,
    0.046080460804608044,
    0.04609046090460905,
    0.046100461004610044,
    0.04611046110461105,
    0.046120461204612044,
    0.04613046130461305,
    0.046140461404614044,
    0.04615046150461505,
    0.046160461604616045,
    0.04617046170461705,
    0.046180461804618045,
    0.04619046190461905,
    0.046200462004620045,
    0.04621046210462105,
    0.046220462204622045,
    0.04623046230462304,
    0.046240462404624046,
    0.04625046250462504,
    0.046260462604626046,
    0.04627046270462704,
    0.046280462804628046,
    0.04629046290462904,
    0.046300463004630046,
    0.04631046310463104,
    0.04632046320463205,
    0.04633046330463304,
    0.04634046340463405,
    0.046350463504635044,
    0.04636046360463605,
    0.046370463704637044,
    0.04638046380463805,
    0.046390463904639044,
    0.04640046400464005,
    0.046410464104641044,
    0.04642046420464205,
    0.046430464304643045,
    0.04644046440464405,
    0.046450464504645045,
    0.04646046460464605,
    0.046470464704647045,
    0.04648046480464805,
    0.046490464904649045,
    0.04650046500465005,
    0.046510465104651046,
    0.04652046520465204,
    0.046530465304653046,
    0.04654046540465404,
    0.046550465504655046,
    0.04656046560465604,
    0.046570465704657046,
    0.04658046580465804,
    0.04659046590465905,
    0.04660046600466004,
    0.04661046610466105,
    0.046620466204662044,
    0.04663046630466305,
    0.046640466404664044,
    0.04665046650466505,
    0.046660466604666044,
    0.04667046670466705,
    0.046680466804668044,
    0.04669046690466905,
    0.046700467004670045,
    0.04671046710467105,
    0.046720467204672045,
    0.04673046730467305,
    0.046740467404674045,
    0.04675046750467505,
    0.046760467604676045,
    0.04677046770467705,
    0.046780467804678046,
    0.04679046790467905,
    0.046800468004680046,
    0.04681046810468105,
    0.046820468204682046,
    0.04683046830468304,
    0.046840468404684046,
    0.04685046850468504,
    0.04686046860468605,
    0.04687046870468704,
    0.04688046880468805,
    0.046890468904689044,
    0.04690046900469005,
    0.046910469104691044,
    0.04692046920469205,
    0.046930469304693044,
    0.04694046940469405,
    0.046950469504695044,
    0.04696046960469605,
    0.046970469704697045,
    0.04698046980469805,
    0.046990469904699045,
    0.04700047000470005,
    0.047010470104701045,
    0.04702047020470205,
    0.047030470304703045,
    0.04704047040470405,
    0.047050470504705046,
    0.04706047060470605,
    0.047070470704707046,
    0.04708047080470805,
    0.047090470904709046,
    0.04710047100471005,
    0.047110471104711046,
    0.04712047120471204,
    0.04713047130471305,
    0.04714047140471404,
    0.04715047150471505,
    0.047160471604716044,
    0.04717047170471705,
    0.047180471804718044,
    0.04719047190471905,
    0.047200472004720044,
    0.04721047210472105,
    0.047220472204722044,
    0.04723047230472305,
    0.047240472404724045,
    0.04725047250472505,
    0.047260472604726045,
    0.04727047270472705,
    0.047280472804728045,
    0.04729047290472905,
    0.047300473004730045,
    0.04731047310473105,
    0.047320473204732046,
    0.04733047330473305,
    0.047340473404734046,
    0.04735047350473505,
    0.047360473604736046,
    0.04737047370473705,
    0.047380473804738046,
    0.04739047390473905,
    0.04740047400474005,
    0.04741047410474104,
    0.04742047420474205,
    0.047430474304743044,
    0.04744047440474405,
    0.047450474504745044,
    0.04746047460474605,
    0.047470474704747044,
    0.04748047480474805,
    0.047490474904749044,
    0.04750047500475005,
    0.047510475104751045,
    0.04752047520475205,
    0.047530475304753045,
    0.04754047540475405,
    0.047550475504755045,
    0.04756047560475605,
    0.047570475704757045,
    0.04758047580475805,
    0.047590475904759046,
    0.04760047600476005,
    0.047610476104761046,
    0.04762047620476205,
    0.047630476304763046,
    0.04764047640476405,
    0.047650476504765046,
    0.04766047660476605,
    0.04767047670476705,
    0.04768047680476805,
    0.04769047690476905,
    0.047700477004770044,
    0.04771047710477105,
    0.047720477204772044,
    0.04773047730477305,
    0.047740477404774044,
    0.04775047750477505,
    0.047760477604776044,
    0.04777047770477705,
    0.047780477804778045,
    0.04779047790477905,
    0.047800478004780045,
    0.04781047810478105,
    0.047820478204782045,
    0.04783047830478305,
    0.047840478404784045,
    0.04785047850478505,
    0.047860478604786046,
    0.04787047870478705,
    0.047880478804788046,
    0.04789047890478905,
    0.047900479004790046,
    0.04791047910479105,
    0.047920479204792046,
    0.04793047930479305,
    0.04794047940479405,
    0.04795047950479505,
    0.04796047960479605,
    0.04797047970479705,
    0.04798047980479805,
    0.047990479904799044,
    0.04800048000480005,
    0.048010480104801044,
    0.04802048020480205,
    0.048030480304803044,
    0.04804048040480405,
    0.048050480504805045,
    0.04806048060480605,
    0.048070480704807045,
    0.04808048080480805,
    0.048090480904809045,
    0.04810048100481005,
    0.048110481104811045,
    0.04812048120481205,
    0.048130481304813046,
    0.04814048140481405,
    0.048150481504815046,
    0.04816048160481605,
    0.048170481704817046,
    0.04818048180481805,
    0.048190481904819046,
    0.04820048200482005,
    0.04821048210482105,
    0.04822048220482205,
    0.04823048230482305,
    0.04824048240482405,
    0.04825048250482505,
    0.04826048260482605,
    0.04827048270482705,
    0.048280482804828044,
    0.04829048290482905,
    0.048300483004830044,
    0.04831048310483105,
    0.048320483204832045,
    0.04833048330483305,
    0.048340483404834045,
    0.04835048350483505,
    0.048360483604836045,
    0.04837048370483705,
    0.048380483804838045,
    0.04839048390483905,
    0.048400484004840046,
    0.04841048410484105,
    0.048420484204842046,
    0.04843048430484305,
    0.048440484404844046,
    0.04845048450484505,
    0.048460484604846046,
    0.04847048470484705,
    0.04848048480484805,
    0.04849048490484905,
    0.04850048500485005,
    0.04851048510485105,
    0.04852048520485205,
    0.04853048530485305,
    0.04854048540485405,
    0.04855048550485505,
    0.04856048560485605,
    0.048570485704857044,
    0.04858048580485805,
    0.048590485904859045,
    0.04860048600486005,
    0.048610486104861045,
    0.04862048620486205,
    0.048630486304863045,
    0.04864048640486405,
    0.048650486504865045,
    0.04866048660486605,
    0.048670486704867046,
    0.04868048680486805,
    0.048690486904869046,
    0.04870048700487005,
    0.048710487104871046,
    0.04872048720487205,
    0.048730487304873046,
    0.04874048740487405,
    0.04875048750487505,
    0.04876048760487605,
    0.04877048770487705,
    0.04878048780487805,
    0.04879048790487905,
    0.04880048800488005,
    0.04881048810488105,
    0.04882048820488205,
    0.04883048830488305,
    0.04884048840488405,
    0.04885048850488505,
    0.048860488604886045,
    0.04887048870488705,
    0.048880488804888045,
    0.04889048890488905,
    0.048900489004890045,
    0.04891048910489105,
    0.048920489204892045,
    0.04893048930489305,
    0.048940489404894046,
    0.04895048950489505,
    0.048960489604896046,
    0.04897048970489705,
    0.048980489804898046,
    0.04899048990489905,
    0.049000490004900046,
    0.04901049010490105,
    0.04902049020490205,
    0.04903049030490305,
    0.04904049040490405,
    0.04905049050490505,
    0.04906049060490605,
    0.04907049070490705,
    0.04908049080490805,
    0.04909049090490905,
    0.04910049100491005,
    0.04911049110491105,
    0.04912049120491205,
    0.04913049130491305,
    0.04914049140491405,
    0.04915049150491505,
    0.04916049160491605,
    0.049170491704917045,
    0.04918049180491805,
    0.049190491904919045,
    0.04920049200492005,
    0.049210492104921046,
    0.04922049220492205,
    0.049230492304923046,
    0.04924049240492405,
    0.049250492504925046,
    0.04926049260492605,
    0.049270492704927046,
    0.04928049280492805,
    0.04929049290492905,
    0.04930049300493005,
    0.04931049310493105,
    0.04932049320493205,
    0.04933049330493305,
    0.04934049340493405,
    0.04935049350493505,
    0.04936049360493605,
    0.04937049370493705,
    0.04938049380493805,
    0.04939049390493905,
    0.04940049400494005,
    0.04941049410494105,
    0.04942049420494205,
    0.04943049430494305,
    0.04944049440494405,
    0.04945049450494505,
    0.049460494604946045,
    0.04947049470494705,
    0.049480494804948046,
    0.04949049490494905,
    0.049500495004950046,
    0.04951049510495105,
    0.049520495204952046,
    0.04953049530495305,
    0.049540495404954046,
    0.04955049550495505,
    0.04956049560495605,
    0.04957049570495705,
    0.04958049580495805,
    0.04959049590495905,
    0.04960049600496005,
    0.04961049610496105,
    0.04962049620496205,
    0.04963049630496305,
    0.04964049640496405,
    0.04965049650496505,
    0.04966049660496605,
    0.04967049670496705,
    0.04968049680496805,
    0.04969049690496905,
    0.04970049700497005,
    0.04971049710497105,
    0.04972049720497205,
    0.04973049730497305,
    0.04974049740497405,
    0.049750497504975046,
    0.04976049760497605,
    0.049770497704977046,
    0.04978049780497805,
    0.049790497904979046,
    0.04980049800498005,
    0.049810498104981046,
    0.04982049820498205,
    0.04983049830498305,
    0.04984049840498405,
    0.04985049850498505,
    0.04986049860498605,
    0.04987049870498705,
    0.04988049880498805,
    0.04989049890498905,
    0.04990049900499005,
    0.04991049910499105,
    0.04992049920499205,
    0.04993049930499305,
    0.04994049940499405,
    0.04995049950499505,
    0.04996049960499605,
    0.04997049970499705,
    0.04998049980499805,
    0.04999049990499905,
    0.05000050000500005,
    0.05001050010500105,
    0.05002050020500205,
    0.05003050030500305,
    0.050040500405004046,
    0.05005050050500505,
    0.050060500605006046,
    0.05007050070500705,
    0.050080500805008046,
    0.05009050090500905,
    0.05010050100501005,
    0.05011050110501105,
    0.05012050120501205,
    0.05013050130501305,
    0.05014050140501405,
    0.05015050150501505,
    0.05016050160501605,
    0.05017050170501705,
    0.05018050180501805,
    0.05019050190501905,
    0.05020050200502005,
    0.05021050210502105,
    0.05022050220502205,
    0.05023050230502305,
    0.05024050240502405,
    0.05025050250502505,
    0.05026050260502605,
    0.05027050270502705,
    0.05028050280502805,
    0.05029050290502905,
    0.05030050300503005,
    0.05031050310503105,
    0.05032050320503205,
    0.050330503305033046,
    0.05034050340503405,
    0.050350503505035046,
    0.05036050360503605,
    0.05037050370503705,
    0.05038050380503805,
    0.05039050390503905,
    0.05040050400504005,
    0.05041050410504105,
    0.05042050420504205,
    0.05043050430504305,
    0.05044050440504405,
    0.05045050450504505,
    0.05046050460504605,
    0.05047050470504705,
    0.05048050480504805,
    0.05049050490504905,
    0.05050050500505005,
    0.05051050510505105,
    0.05052050520505205,
    0.05053050530505305,
    0.05054050540505405,
    0.05055050550505505,
    0.05056050560505605,
    0.05057050570505705,
    0.05058050580505805,
    0.05059050590505905,
    0.05060050600506005,
    0.05061050610506105,
    0.050620506205062046,
    0.05063050630506305,
    0.05064050640506405,
    0.05065050650506505,
    0.05066050660506605,
    0.05067050670506705,
    0.05068050680506805,
    0.05069050690506905,
    0.05070050700507005,
    0.05071050710507105,
    0.05072050720507205,
    0.05073050730507305,
    0.05074050740507405,
    0.05075050750507505,
    0.05076050760507605,
    0.05077050770507705,
    0.05078050780507805,
    0.05079050790507905,
    0.05080050800508005,
    0.05081050810508105,
    0.05082050820508205,
    0.05083050830508305,
    0.05084050840508405,
    0.05085050850508505,
    0.05086050860508605,
    0.05087050870508705,
    0.05088050880508805,
    0.05089050890508905,
    0.05090050900509005,
    0.05091050910509105,
    0.05092050920509205,
    0.05093050930509305,
    0.05094050940509405,
    0.05095050950509505,
    0.05096050960509605,
    0.05097050970509705,
    0.05098050980509805,
    0.05099050990509905,
    0.05100051000510005,
    0.05101051010510105,
    0.05102051020510205,
    0.05103051030510305,
    0.05104051040510405,
    0.05105051050510505,
    0.05106051060510605,
    0.05107051070510705,
    0.05108051080510805,
    0.05109051090510905,
    0.05110051100511005,
    0.05111051110511105,
    0.05112051120511205,
    0.05113051130511305,
    0.05114051140511405,
    0.05115051150511505,
    0.05116051160511605,
    0.05117051170511705,
    0.051180511805118054,
    0.05119051190511905,
    0.05120051200512005,
    0.05121051210512105,
    0.05122051220512205,
    0.05123051230512305,
    0.05124051240512405,
    0.05125051250512505,
    0.05126051260512605,
    0.05127051270512705,
    0.05128051280512805,
    0.05129051290512905,
    0.05130051300513005,
    0.05131051310513105,
    0.05132051320513205,
    0.05133051330513305,
    0.05134051340513405,
    0.05135051350513505,
    0.05136051360513605,
    0.05137051370513705,
    0.05138051380513805,
    0.05139051390513905,
    0.05140051400514005,
    0.05141051410514105,
    0.05142051420514205,
    0.05143051430514305,
    0.05144051440514405,
    0.051450514505145054,
    0.05146051460514605,
    0.051470514705147054,
    0.05148051480514805,
    0.051490514905149054,
    0.05150051500515005,
    0.05151051510515105,
    0.05152051520515205,
    0.05153051530515305,
    0.05154051540515405,
    0.05155051550515505,
    0.05156051560515605,
    0.05157051570515705,
    0.05158051580515805,
    0.05159051590515905,
    0.05160051600516005,
    0.05161051610516105,
    0.05162051620516205,
    0.05163051630516305,
    0.05164051640516405,
    0.05165051650516505,
    0.05166051660516605,
    0.05167051670516705,
    0.05168051680516805,
    0.05169051690516905,
    0.05170051700517005,
    0.05171051710517105,
    0.051720517205172054,
    0.05173051730517305,
    0.051740517405174054,
    0.05175051750517505,
    0.051760517605176054,
    0.05177051770517705,
    0.051780517805178054,
    0.05179051790517905,
    0.05180051800518005,
    0.05181051810518105,
    0.05182051820518205,
    0.05183051830518305,
    0.05184051840518405,
    0.05185051850518505,
    0.05186051860518605,
    0.05187051870518705,
    0.05188051880518805,
    0.05189051890518905,
    0.05190051900519005,
    0.05191051910519105,
    0.05192051920519205,
    0.05193051930519305,
    0.05194051940519405,
    0.05195051950519505,
    0.05196051960519605,
    0.05197051970519705,
    0.05198051980519805,
    0.051990519905199054,
    0.05200052000520005,
    0.052010520105201054,
    0.05202052020520205,
    0.052030520305203054,
    0.05204052040520405,
    0.052050520505205054,
    0.05206052060520605,
    0.052070520705207055,
    0.05208052080520805,
    0.05209052090520905,
    0.05210052100521005,
    0.05211052110521105,
    0.05212052120521205,
    0.05213052130521305,
    0.05214052140521405,
    0.05215052150521505,
    0.05216052160521605,
    0.05217052170521705,
    0.05218052180521805,
    0.05219052190521905,
    0.05220052200522005,
    0.05221052210522105,
    0.05222052220522205,
    0.05223052230522305,
    0.05224052240522405,
    0.05225052250522505,
    0.052260522605226054,
    0.05227052270522705,
    0.052280522805228054,
    0.05229052290522905,
    0.052300523005230054,
    0.05231052310523105,
    0.052320523205232054,
    0.05233052330523305,
    0.052340523405234055,
    0.05235052350523505,
    0.052360523605236055,
    0.05237052370523705,
    0.05238052380523805,
    0.05239052390523905,
    0.05240052400524005,
    0.05241052410524105,
    0.05242052420524205,
    0.05243052430524305,
    0.05244052440524405,
    0.05245052450524505,
    0.05246052460524605,
    0.05247052470524705,
    0.05248052480524805,
    0.05249052490524905,
    0.05250052500525005,
    0.05251052510525105,
    0.05252052520525205,
    0.052530525305253054,
    0.05254052540525405,
    0.052550525505255054,
    0.05256052560525605,
    0.052570525705257054,
    0.05258052580525805,
    0.052590525905259054,
    0.05260052600526005,
    0.052610526105261055,
    0.05262052620526205,
    0.052630526305263055,
    0.05264052640526405,
    0.052650526505265055,
    0.05266052660526605,
    0.05267052670526705,
    0.05268052680526805,
    0.05269052690526905,
    0.05270052700527005,
    0.05271052710527105,
    0.05272052720527205,
    0.05273052730527305,
    0.05274052740527405,
    0.05275052750527505,
    0.05276052760527605,
    0.05277052770527705,
    0.05278052780527805,
    0.05279052790527905,
    0.052800528005280054,
    0.05281052810528105,
    0.052820528205282054,
    0.05283052830528305,
    0.052840528405284054,
    0.05285052850528505,
    0.052860528605286054,
    0.05287052870528705,
    0.052880528805288055,
    0.05289052890528905,
    0.052900529005290055,
    0.05291052910529105,
    0.052920529205292055,
    0.05293052930529305,
    0.052940529405294055,
    0.05295052950529505,
    0.05296052960529605,
    0.05297052970529705,
    0.05298052980529805,
    0.05299052990529905,
    0.05300053000530005,
    0.05301053010530105,
    0.05302053020530205,
    0.05303053030530305,
    0.05304053040530405,
    0.05305053050530505,
    0.05306053060530605,
    0.053070530705307054,
    0.05308053080530805,
    0.053090530905309054,
    0.05310053100531005,
    0.053110531105311054,
    0.05312053120531205,
    0.053130531305313054,
    0.05314053140531405,
    0.053150531505315055,
    0.05316053160531605,
    0.053170531705317055,
    0.05318053180531805,
    0.053190531905319055,
    0.05320053200532005,
    0.053210532105321055,
    0.05322053220532205,
    0.053230532305323056,
    0.05324053240532405,
    0.05325053250532505,
    0.05326053260532605,
    0.05327053270532705,
    0.05328053280532805,
    0.05329053290532905,
    0.05330053300533005,
    0.05331053310533105,
    0.05332053320533205,
    0.05333053330533305,
    0.053340533405334054,
    0.05335053350533505,
    0.053360533605336054,
    0.05337053370533705,
    0.053380533805338054,
    0.05339053390533905,
    0.053400534005340054,
    0.05341053410534105,
    0.053420534205342055,
    0.05343053430534305,
    0.053440534405344055,
    0.05345053450534505,
    0.053460534605346055,
    0.05347053470534705,
    0.053480534805348055,
    0.05349053490534905,
    0.053500535005350056,
    0.05351053510535105,
    0.053520535205352056,
    0.05353053530535305,
    0.053540535405354056,
    0.05355053550535505,
    0.05356053560535605,
    0.05357053570535705,
    0.05358053580535805,
    0.05359053590535905,
    0.05360053600536005,
    0.053610536105361054,
    0.05362053620536205,
    0.053630536305363054,
    0.05364053640536405,
    0.053650536505365054,
    0.05366053660536605,
    0.053670536705367054,
    0.05368053680536805,
    0.053690536905369055,
    0.05370053700537005,
    0.053710537105371055,
    0.05372053720537205,
    0.053730537305373055,
    0.05374053740537405,
    0.053750537505375055,
    0.05376053760537605,
    0.053770537705377056,
    0.05378053780537805,
    0.053790537905379056,
    0.05380053800538005,
    0.053810538105381056,
    0.05382053820538205,
    0.053830538305383056,
    0.05384053840538405,
    0.05385053850538505,
    0.05386053860538605,
    0.05387053870538705,
    0.053880538805388054,
    0.05389053890538905,
    0.053900539005390054,
    0.05391053910539105,
    0.053920539205392054,
    0.05393053930539305,
    0.053940539405394054,
    0.05395053950539505,
    0.053960539605396055,
    0.05397053970539705,
    0.053980539805398055,
    0.05399053990539905,
    0.054000540005400055,
    0.05401054010540105,
    0.054020540205402055,
    0.05403054030540305,
    0.054040540405404056,
    0.05405054050540505,
    0.054060540605406056,
    0.05407054070540705,
    0.054080540805408056,
    0.05409054090540905,
    0.054100541005410056,
    0.05411054110541105,
    0.05412054120541206,
    0.05413054130541305,
    0.05414054140541405,
    0.054150541505415054,
    0.05416054160541605,
    0.054170541705417054,
    0.05418054180541805,
    0.054190541905419054,
    0.05420054200542005,
    0.054210542105421054,
    0.05422054220542205,
    0.054230542305423055,
    0.05424054240542405,
    0.054250542505425055,
    0.05426054260542605,
    0.054270542705427055,
    0.05428054280542805,
    0.054290542905429055,
    0.05430054300543005,
    0.054310543105431056,
    0.05432054320543205,
    0.054330543305433056,
    0.05434054340543405,
    0.054350543505435056,
    0.05436054360543605,
    0.054370543705437056,
    0.05438054380543805,
    0.05439054390543906,
    0.05440054400544005,
    0.05441054410544106,
    0.054420544205442054,
    0.05443054430544305,
    0.054440544405444054,
    0.05445054450544505,
    0.054460544605446054,
    0.05447054470544705,
    0.054480544805448054,
    0.05449054490544905,
    0.054500545005450055,
    0.05451054510545105,
    0.054520545205452055,
    0.05453054530545305,
    0.054540545405454055,
    0.05455054550545505,
    0.054560545605456055,
    0.05457054570545705,
    0.054580545805458056,
    0.05459054590545905,
    0.054600546005460056,
    0.05461054610546105,
    0.054620546205462056,
    0.05463054630546305,
    0.054640546405464056,
    0.05465054650546505,
    0.05466054660546606,
    0.05467054670546705,
    0.05468054680546806,
    0.054690546905469054,
    0.05470054700547006,
    0.054710547105471054,
    0.05472054720547205,
    0.054730547305473054,
    0.05474054740547405,
    0.054750547505475054,
    0.05476054760547605,
    0.054770547705477055,
    0.05478054780547805,
    0.054790547905479055,
    0.05480054800548005,
    0.054810548105481055,
    0.05482054820548205,
    0.054830548305483055,
    0.05484054840548405,
    0.054850548505485056,
    0.05486054860548605,
    0.054870548705487056,
    0.05488054880548805,
    0.054890548905489056,
    0.05490054900549005,
    0.054910549105491056,
    0.05492054920549205,
    0.05493054930549306,
    0.05494054940549405,
    0.05495054950549506,
    0.054960549605496054,
    0.05497054970549706,
    0.054980549805498054,
    0.05499054990549906,
    0.055000550005500054,
    0.05501055010550105,
    0.055020550205502054,
    0.05503055030550305,
    0.055040550405504055,
    0.05505055050550505,
    0.055060550605506055,
    0.05507055070550705,
    0.055080550805508055,
    0.05509055090550905,
    0.055100551005510055,
    0.05511055110551105,
    0.055120551205512056,
    0.05513055130551305,
    0.055140551405514056,
    0.05515055150551505,
    0.055160551605516056,
    0.05517055170551705,
    0.055180551805518056,
    0.05519055190551905,
    0.05520055200552006,
    0.05521055210552105,
    0.05522055220552206,
    0.055230552305523054,
    0.05524055240552406,
    0.055250552505525054,
    0.05526055260552606,
    0.055270552705527054,
    0.05528055280552806,
    0.055290552905529054,
    0.05530055300553005,
    0.055310553105531055,
    0.05532055320553205,
    0.055330553305533055,
    0.05534055340553405,
    0.055350553505535055,
    0.05536055360553605,
    0.055370553705537055,
    0.05538055380553805,
    0.055390553905539056,
    0.05540055400554005,
    0.055410554105541056,
    0.05542055420554205,
    0.055430554305543056,
    0.05544055440554405,
    0.055450554505545056,
    0.05546055460554605,
    0.05547055470554706,
    0.05548055480554805,
    0.05549055490554906,
    0.055500555005550054,
    0.05551055510555106,
    0.055520555205552054,
    0.05553055530555306,
    0.055540555405554054,
    0.05555055550555506,
    0.055560555605556054,
    0.05557055570555706,
    0.055580555805558055,
    0.05559055590555905,
    0.055600556005560055,
    0.05561055610556105,
    0.055620556205562055,
    0.05563055630556305,
    0.055640556405564055,
    0.05565055650556505,
    0.055660556605566056,
    0.05567055670556705,
    0.055680556805568056,
    0.05569055690556905,
    0.055700557005570056,
    0.05571055710557105,
    0.055720557205572056,
    0.05573055730557305,
    0.05574055740557406,
    0.05575055750557505,
    0.05576055760557606,
    0.055770557705577053,
    0.05578055780557806,
    0.055790557905579054,
    0.05580055800558006,
    0.055810558105581054,
    0.05582055820558206,
    0.055830558305583054,
    0.05584055840558406,
    0.055850558505585055,
    0.05586055860558606,
    0.055870558705587055,
    0.05588055880558806,
    0.055890558905589055,
    0.05590055900559005,
    0.055910559105591055,
    0.05592055920559205,
    0.055930559305593056,
    0.05594055940559405,
    0.055950559505595056,
    0.05596055960559605,
    0.055970559705597056,
    0.05598055980559805,
    0.055990559905599056,
    0.05600056000560005,
    0.05601056010560106,
    0.05602056020560205,
    0.05603056030560306,
    0.056040560405604053,
    0.05605056050560506,
    0.056060560605606054,
    0.05607056070560706,
    0.056080560805608054,
    0.05609056090560906,
    0.056100561005610054,
    0.05611056110561106,
    0.056120561205612055,
    0.05613056130561306,
    0.056140561405614055,
    0.05615056150561506,
    0.056160561605616055,
    0.05617056170561706,
    0.056180561805618055,
    0.05619056190561905,
    0.056200562005620056,
    0.05621056210562105,
    0.056220562205622056,
    0.05623056230562305,
    0.056240562405624056,
    0.05625056250562505,
    0.056260562605626056,
    0.05627056270562705,
    0.05628056280562806,
    0.05629056290562905,
    0.05630056300563006,
    0.056310563105631053,
    0.05632056320563206,
    0.056330563305633054,
    0.05634056340563406,
    0.056350563505635054,
    0.05636056360563606,
    0.056370563705637054,
    0.05638056380563806,
    0.056390563905639055,
    0.05640056400564006,
    0.056410564105641055,
    0.05642056420564206,
    0.056430564305643055,
    0.05644056440564406,
    0.056450564505645055,
    0.05646056460564606,
    0.056470564705647056,
    0.05648056480564805,
    0.056490564905649056,
    0.05650056500565005,
    0.056510565105651056,
    0.05652056520565205,
    0.056530565305653056,
    0.05654056540565405,
    0.05655056550565506,
    0.05656056560565605,
    0.05657056570565706,
    0.056580565805658053,
    0.05659056590565906,
    0.056600566005660054,
    0.05661056610566106,
    0.056620566205662054,
    0.05663056630566306,
    0.056640566405664054,
    0.05665056650566506,
    0.056660566605666055,
    0.05667056670566706,
    0.056680566805668055,
    0.05669056690566906,
    0.056700567005670055,
    0.05671056710567106,
    0.056720567205672055,
    0.05673056730567306,
    0.056740567405674056,
    0.05675056750567506,
    0.056760567605676056,
    0.05677056770567705,
    0.056780567805678056,
    0.05679056790567905,
    0.056800568005680056,
    0.05681056810568105,
    0.05682056820568206,
    0.05683056830568305,
    0.05684056840568406,
    0.056850568505685053,
    0.05686056860568606,
    0.056870568705687054,
    0.05688056880568806,
    0.056890568905689054,
    0.05690056900569006,
    0.056910569105691054,
    0.05692056920569206,
    0.056930569305693055,
    0.05694056940569406,
    0.056950569505695055,
    0.05696056960569606,
    0.056970569705697055,
    0.05698056980569806,
    0.056990569905699055,
    0.05700057000570006,
    0.057010570105701056,
    0.05702057020570206,
    0.057030570305703056,
    0.05704057040570406,
    0.057050570505705056,
    0.05706057060570605,
    0.057070570705707056,
    0.05708057080570805,
    0.05709057090570906,
    0.05710057100571005,
    0.05711057110571106,
    0.057120571205712053,
    0.05713057130571306,
    0.057140571405714054,
    0.05715057150571506,
    0.057160571605716054,
    0.05717057170571706,
    0.057180571805718054,
    0.05719057190571906,
    0.057200572005720055,
    0.05721057210572106,
    0.057220572205722055,
    0.05723057230572306,
    0.057240572405724055,
    0.05725057250572506,
    0.057260572605726055,
    0.05727057270572706,
    0.057280572805728056,
    0.05729057290572906,
    0.057300573005730056,
    0.05731057310573106,
    0.057320573205732056,
    0.05733057330573306,
    0.057340573405734056,
    0.05735057350573505,
    0.05736057360573606,
    0.05737057370573705,
    0.05738057380573806,
    0.057390573905739053,
    0.05740057400574006,
    0.057410574105741054,
    0.05742057420574206,
    0.057430574305743054,
    0.05744057440574406,
    0.057450574505745054,
    0.05746057460574606,
    0.057470574705747055,
    0.05748057480574806,
    0.057490574905749055,
    0.05750057500575006,
    0.057510575105751055,
    0.05752057520575206,
    0.057530575305753055,
    0.05754057540575406,
    0.057550575505755056,
    0.05756057560575606,
    0.057570575705757056,
    0.05758057580575806,
    0.057590575905759056,
    0.05760057600576006,
    0.057610576105761056,
    0.05762057620576206,
    0.05763057630576306,
    0.05764057640576405,
    0.05765057650576506,
    0.057660576605766053,
    0.05767057670576706,
    0.057680576805768054,
    0.05769057690576906,
    0.057700577005770054,
    0.05771057710577106,
    0.057720577205772054,
    0.05773057730577306,
    0.057740577405774055,
    0.05775057750577506,
    0.057760577605776055,
    0.05777057770577706,
    0.057780577805778055,
    0.05779057790577906,
    0.057800578005780055,
    0.05781057810578106,
    0.057820578205782056,
    0.05783057830578306,
    0.057840578405784056,
    0.05785057850578506,
    0.057860578605786056,
    0.05787057870578706,
    0.057880578805788056,
    0.05789057890578906,
    0.05790057900579006,
    0.05791057910579106,
    0.05792057920579206,
    0.057930579305793053,
    0.05794057940579406,
    0.057950579505795054,
    0.05796057960579606,
    0.057970579705797054,
    0.05798057980579806,
    0.057990579905799054,
    0.05800058000580006,
    0.058010580105801055,
    0.05802058020580206,
    0.058030580305803055,
    0.05804058040580406,
    0.058050580505805055,
    0.05806058060580606,
    0.058070580705807055,
    0.05808058080580806,
    0.058090580905809056,
    0.05810058100581006,
    0.058110581105811056,
    0.05812058120581206,
    0.058130581305813056,
    0.05814058140581406,
    0.058150581505815056,
    0.05816058160581606,
    0.05817058170581706,
    0.05818058180581806,
    0.05819058190581906,
    0.05820058200582006,
    0.05821058210582106,
    0.05822058220582206,
    0.05823058230582306,
    0.058240582405824054,
    0.05825058250582506,
    0.058260582605826054,
    0.05827058270582706,
    0.058280582805828055,
    0.05829058290582906,
    0.058300583005830055,
    0.05831058310583106,
    0.058320583205832055,
    0.05833058330583306,
    0.058340583405834055,
    0.05835058350583506,
    0.058360583605836056,
    0.05837058370583706,
    0.058380583805838056,
    0.05839058390583906,
    0.058400584005840056,
    0.05841058410584106,
    0.058420584205842056,
    0.05843058430584306,
    0.05844058440584406,
    0.05845058450584506,
    0.05846058460584606,
    0.05847058470584706,
    0.05848058480584806,
    0.05849058490584906,
    0.05850058500585006,
    0.05851058510585106,
    0.05852058520585206,
    0.058530585305853054,
    0.05854058540585406,
    0.058550585505855055,
    0.05856058560585606,
    0.058570585705857055,
    0.05858058580585806,
    0.058590585905859055,
    0.05860058600586006,
    0.058610586105861055,
    0.05862058620586206,
    0.058630586305863056,
    0.05864058640586406,
    0.058650586505865056,
    0.05866058660586606,
    0.058670586705867056,
    0.05868058680586806,
    0.058690586905869056,
    0.05870058700587006,
    0.05871058710587106,
    0.05872058720587206,
    0.05873058730587306,
    0.05874058740587406,
    0.05875058750587506,
    0.05876058760587606,
    0.05877058770587706,
    0.05878058780587806,
    0.05879058790587906,
    0.05880058800588006,
    0.05881058810588106,
    0.058820588205882054,
    0.05883058830588306,
    0.058840588405884055,
    0.05885058850588506,
    0.058860588605886055,
    0.05887058870588706,
    0.058880588805888055,
    0.05889058890588906,
    0.058900589005890056,
    0.05891058910589106,
    0.058920589205892056,
    0.05893058930589306,
    0.058940589405894056,
    0.05895058950589506,
    0.058960589605896056,
    0.05897058970589706,
    0.05898058980589806,
    0.05899058990589906,
    0.05900059000590006,
    0.05901059010590106,
    0.05902059020590206,
    0.05903059030590306,
    0.05904059040590406,
    0.05905059050590506,
    0.05906059060590606,
    0.05907059070590706,
    0.05908059080590806,
    0.05909059090590906,
    0.05910059100591006,
    0.059110591105911055,
    0.05912059120591206,
    0.059130591305913055,
    0.05914059140591406,
    0.059150591505915055,
    0.05916059160591606,
    0.059170591705917056,
    0.05918059180591806,
    0.059190591905919056,
    0.05920059200592006,
    0.059210592105921056,
    0.05922059220592206,
    0.059230592305923056,
    0.05924059240592406,
    0.05925059250592506,
    0.05926059260592606,
    0.05927059270592706,
    0.05928059280592806,
    0.05929059290592906,
    0.05930059300593006,
    0.05931059310593106,
    0.05932059320593206,
    0.05933059330593306,
    0.05934059340593406,
    0.05935059350593506,
    0.05936059360593606,
    0.05937059370593706,
    0.05938059380593806,
    0.05939059390593906,
    0.059400594005940055,
    0.05941059410594106,
    0.059420594205942055,
    0.05943059430594306,
    0.059440594405944056,
    0.05945059450594506,
    0.059460594605946056,
    0.05947059470594706,
    0.059480594805948056,
    0.05949059490594906,
    0.059500595005950056,
    0.05951059510595106,
    0.05952059520595206,
    0.05953059530595306,
    0.05954059540595406,
    0.05955059550595506,
    0.05956059560595606,
    0.05957059570595706,
    0.05958059580595806,
    0.05959059590595906,
    0.05960059600596006,
    0.05961059610596106,
    0.05962059620596206,
    0.05963059630596306,
    0.05964059640596406,
    0.05965059650596506,
    0.05966059660596606,
    0.05967059670596706,
    0.05968059680596806,
    0.059690596905969055,
    0.05970059700597006,
    0.059710597105971056,
    0.05972059720597206,
    0.059730597305973056,
    0.05974059740597406,
    0.059750597505975056,
    0.05976059760597606,
    0.059770597705977056,
    0.05978059780597806,
    0.05979059790597906,
    0.05980059800598006,
    0.05981059810598106,
    0.05982059820598206,
    0.05983059830598306,
    0.05984059840598406,
    0.05985059850598506,
    0.05986059860598606,
    0.05987059870598706,
    0.05988059880598806,
    0.05989059890598906,
    0.05990059900599006,
    0.05991059910599106,
    0.05992059920599206,
    0.05993059930599306,
    0.05994059940599406,
    0.05995059950599506,
    0.05996059960599606,
    0.05997059970599706,
    0.059980599805998056,
    0.05999059990599906,
    0.060000600006000056,
    0.06001060010600106,
    0.060020600206002056,
    0.06003060030600306,
    0.060040600406004056,
    0.06005060050600506,
    0.06006060060600606,
    0.06007060070600706,
    0.06008060080600806,
    0.06009060090600906,
    0.06010060100601006,
    0.06011060110601106,
    0.06012060120601206,
    0.06013060130601306,
    0.06014060140601406,
    0.06015060150601506,
    0.06016060160601606,
    0.06017060170601706,
    0.06018060180601806,
    0.06019060190601906,
    0.06020060200602006,
    0.06021060210602106,
    0.06022060220602206,
    0.06023060230602306,
    0.06024060240602406,
    0.06025060250602506,
    0.06026060260602606,
    0.060270602706027056,
    0.06028060280602806,
    0.060290602906029056,
    0.06030060300603006,
    0.060310603106031056,
    0.06032060320603206,
    0.06033060330603306,
    0.06034060340603406,
    0.06035060350603506,
    0.06036060360603606,
    0.06037060370603706,
    0.06038060380603806,
    0.06039060390603906,
    0.06040060400604006,
    0.06041060410604106,
    0.06042060420604206,
    0.06043060430604306,
    0.06044060440604406,
    0.06045060450604506,
    0.06046060460604606,
    0.06047060470604706,
    0.06048060480604806,
    0.06049060490604906,
    0.06050060500605006,
    0.06051060510605106,
    0.06052060520605206,
    0.06053060530605306,
    0.06054060540605406,
    0.06055060550605506,
    0.06056060560605606,
    0.06057060570605706,
    0.060580605806058056,
    0.06059060590605906,
    0.06060060600606006,
    0.06061060610606106,
    0.06062060620606206,
    0.06063060630606306,
    0.06064060640606406,
    0.06065060650606506,
    0.06066060660606606,
    0.06067060670606706,
    0.06068060680606806,
    0.06069060690606906,
    0.06070060700607006,
    0.06071060710607106,
    0.06072060720607206,
    0.06073060730607306,
    0.06074060740607406,
    0.06075060750607506,
    0.06076060760607606,
    0.06077060770607706,
    0.06078060780607806,
    0.06079060790607906,
    0.06080060800608006,
    0.06081060810608106,
    0.06082060820608206,
    0.06083060830608306,
    0.06084060840608406,
    0.06085060850608506,
    0.06086060860608606,
    0.06087060870608706,
    0.06088060880608806,
    0.06089060890608906,
    0.06090060900609006,
    0.06091060910609106,
    0.06092060920609206,
    0.06093060930609306,
    0.06094060940609406,
    0.06095060950609506,
    0.06096060960609606,
    0.06097060970609706,
    0.06098060980609806,
    0.06099060990609906,
    0.06100061000610006,
    0.06101061010610106,
    0.06102061020610206,
    0.06103061030610306,
    0.06104061040610406,
    0.06105061050610506,
    0.06106061060610606,
    0.06107061070610706,
    0.06108061080610806,
    0.06109061090610906,
    0.06110061100611006,
    0.06111061110611106,
    0.06112061120611206,
    0.06113061130611306,
    0.061140611406114063,
    0.06115061150611506,
    0.06116061160611606,
    0.06117061170611706,
    0.06118061180611806,
    0.06119061190611906,
    0.06120061200612006,
    0.06121061210612106,
    0.06122061220612206,
    0.06123061230612306,
    0.06124061240612406,
    0.06125061250612506,
    0.06126061260612606,
    0.06127061270612706,
    0.06128061280612806,
    0.06129061290612906,
    0.06130061300613006,
    0.06131061310613106,
    0.06132061320613206,
    0.06133061330613306,
    0.06134061340613406,
    0.06135061350613506,
    0.06136061360613606,
    0.06137061370613706,
    0.06138061380613806,
    0.06139061390613906,
    0.06140061400614006,
    0.061410614106141063,
    0.06142061420614206,
    0.061430614306143064,
    0.06144061440614406,
    0.06145061450614506,
    0.06146061460614606,
    0.06147061470614706,
    0.06148061480614806,
    0.06149061490614906,
    0.06150061500615006,
    0.06151061510615106,
    0.06152061520615206,
    0.06153061530615306,
    0.06154061540615406,
    0.06155061550615506,
    0.06156061560615606,
    0.06157061570615706,
    0.06158061580615806,
    0.06159061590615906,
    0.06160061600616006,
    0.06161061610616106,
    0.06162061620616206,
    0.06163061630616306,
    0.06164061640616406,
    0.06165061650616506,
    0.06166061660616606,
    0.06167061670616706,
    0.061680616806168063,
    0.06169061690616906,
    0.061700617006170064,
    0.06171061710617106,
    0.061720617206172064,
    0.06173061730617306,
    0.06174061740617406,
    0.06175061750617506,
    0.06176061760617606,
    0.06177061770617706,
    0.06178061780617806,
    0.06179061790617906,
    0.06180061800618006,
    0.06181061810618106,
    0.06182061820618206,
    0.06183061830618306,
    0.06184061840618406,
    0.06185061850618506,
    0.06186061860618606,
    0.06187061870618706,
    0.06188061880618806,
    0.06189061890618906,
    0.06190061900619006,
    0.06191061910619106,
    0.06192061920619206,
    0.06193061930619306,
    0.06194061940619406,
    0.06195061950619506,
    0.06196061960619606,
    0.061970619706197064,
    0.06198061980619806,
    0.061990619906199064,
    0.06200062000620006,
    0.062010620106201064,
    0.06202062020620206,
    0.06203062030620306,
    0.06204062040620406,
    0.06205062050620506,
    0.06206062060620606,
    0.06207062070620706,
    0.06208062080620806,
    0.06209062090620906,
    0.06210062100621006,
    0.06211062110621106,
    0.06212062120621206,
    0.06213062130621306,
    0.06214062140621406,
    0.06215062150621506,
    0.06216062160621606,
    0.06217062170621706,
    0.06218062180621806,
    0.06219062190621906,
    0.06220062200622006,
    0.06221062210622106,
    0.06222062220622206,
    0.06223062230622306,
    0.062240622406224064,
    0.06225062250622506,
    0.062260622606226064,
    0.06227062270622706,
    0.062280622806228064,
    0.06229062290622906,
    0.062300623006230064,
    0.06231062310623106,
    0.06232062320623206,
    0.06233062330623306,
    0.06234062340623406,
    0.06235062350623506,
    0.06236062360623606,
    0.06237062370623706,
    0.06238062380623806,
    0.06239062390623906,
    0.06240062400624006,
    0.06241062410624106,
    0.06242062420624206,
    0.06243062430624306,
    0.06244062440624406,
    0.06245062450624506,
    0.06246062460624606,
    0.06247062470624706,
    0.06248062480624806,
    0.06249062490624906,
    0.06250062500625006,
    0.06251062510625106,
    0.06252062520625207,
    0.06253062530625306,
    0.06254062540625406,
    0.06255062550625506,
    0.06256062560625607,
    0.06257062570625706,
    0.06258062580625806,
    0.06259062590625906,
    0.06260062600626007,
    0.06261062610626106,
    0.06262062620626206,
    0.06263062630626307,
    0.06264062640626405,
    0.06265062650626506,
    0.06266062660626606,
    0.06267062670626707,
    0.06268062680626806,
    0.06269062690626906,
    0.06270062700627006,
    0.06271062710627107,
    0.06272062720627206,
    0.06273062730627306,
    0.06274062740627406,
    0.06275062750627507,
    0.06276062760627606,
    0.06277062770627706,
    0.06278062780627806,
    0.06279062790627907,
    0.06280062800628006,
    0.06281062810628106,
    0.06282062820628206,
    0.06283062830628307,
    0.06284062840628406,
    0.06285062850628506,
    0.06286062860628606,
    0.06287062870628707,
    0.06288062880628806,
    0.06289062890628906,
    0.06290062900629007,
    0.06291062910629105,
    0.06292062920629206,
    0.06293062930629306,
    0.06294062940629407,
    0.06295062950629506,
    0.06296062960629606,
    0.06297062970629706,
    0.06298062980629807,
    0.06299062990629906,
    0.06300063000630006,
    0.06301063010630106,
    0.06302063020630207,
    0.06303063030630306,
    0.06304063040630406,
    0.06305063050630506,
    0.06306063060630607,
    0.06307063070630706,
    0.06308063080630806,
    0.06309063090630906,
    0.06310063100631007,
    0.06311063110631106,
    0.06312063120631206,
    0.06313063130631306,
    0.06314063140631407,
    0.06315063150631506,
    0.06316063160631606,
    0.06317063170631707,
    0.06318063180631807,
    0.06319063190631906,
    0.06320063200632006,
    0.06321063210632107,
    0.06322063220632206,
    0.06323063230632306,
    0.06324063240632406,
    0.06325063250632507,
    0.06326063260632606,
    0.06327063270632706,
    0.06328063280632806,
    0.06329063290632907,
    0.06330063300633006,
    0.06331063310633106,
    0.06332063320633206,
    0.06333063330633307,
    0.06334063340633406,
    0.06335063350633506,
    0.06336063360633606,
    0.06337063370633707,
    0.06338063380633806,
    0.06339063390633906,
    0.06340063400634006,
    0.06341063410634107,
    0.06342063420634206,
    0.06343063430634306,
    0.06344063440634407,
    0.06345063450634507,
    0.06346063460634606,
    0.06347063470634706,
    0.06348063480634807,
    0.06349063490634906,
    0.06350063500635006,
    0.06351063510635106,
    0.06352063520635207,
    0.06353063530635306,
    0.06354063540635406,
    0.06355063550635506,
    0.06356063560635607,
    0.06357063570635706,
    0.06358063580635806,
    0.06359063590635906,
    0.06360063600636007,
    0.06361063610636106,
    0.06362063620636206,
    0.06363063630636306,
    0.06364063640636407,
    0.06365063650636506,
    0.06366063660636606,
    0.06367063670636706,
    0.06368063680636807,
    0.06369063690636906,
    0.06370063700637006,
    0.06371063710637107,
    0.06372063720637207,
    0.06373063730637306,
    0.06374063740637406,
    0.06375063750637507,
    0.06376063760637607,
    0.06377063770637706,
    0.06378063780637806,
    0.06379063790637907,
    0.06380063800638006,
    0.06381063810638106,
    0.06382063820638206,
    0.06383063830638307,
    0.06384063840638406,
    0.06385063850638506,
    0.06386063860638606,
    0.06387063870638707,
    0.06388063880638806,
    0.06389063890638906,
    0.06390063900639006,
    0.06391063910639107,
    0.06392063920639206,
    0.06393063930639306,
    0.06394063940639406,
    0.06395063950639507,
    0.06396063960639606,
    0.06397063970639706,
    0.06398063980639807,
    0.06399063990639907,
    0.06400064000640006,
    0.06401064010640106,
    0.06402064020640207,
    0.06403064030640307,
    0.06404064040640406,
    0.06405064050640506,
    0.06406064060640607,
    0.06407064070640707,
    0.06408064080640806,
    0.06409064090640906,
    0.06410064100641007,
    0.06411064110641106,
    0.06412064120641206,
    0.06413064130641306,
    0.06414064140641407,
    0.06415064150641506,
    0.06416064160641606,
    0.06417064170641706,
    0.06418064180641807,
    0.06419064190641906,
    0.06420064200642006,
    0.06421064210642106,
    0.06422064220642207,
    0.06423064230642306,
    0.06424064240642406,
    0.06425064250642507,
    0.06426064260642607,
    0.06427064270642706,
    0.06428064280642806,
    0.06429064290642907,
    0.06430064300643007,
    0.06431064310643106,
    0.06432064320643206,
    0.06433064330643307,
    0.06434064340643407,
    0.06435064350643506,
    0.06436064360643606,
    0.06437064370643707,
    0.06438064380643806,
    0.06439064390643906,
    0.06440064400644006,
    0.06441064410644107,
    0.06442064420644206,
    0.06443064430644306,
    0.06444064440644406,
    0.06445064450644507,
    0.06446064460644606,
    0.06447064470644706,
    0.06448064480644806,
    0.06449064490644907,
    0.06450064500645006,
    0.06451064510645106,
    0.06452064520645207,
    0.06453064530645307,
    0.06454064540645406,
    0.06455064550645506,
    0.06456064560645607,
    0.06457064570645707,
    0.06458064580645806,
    0.06459064590645906,
    0.06460064600646007,
    0.06461064610646107,
    0.06462064620646206,
    0.06463064630646306,
    0.06464064640646407,
    0.06465064650646507,
    0.06466064660646606,
    0.06467064670646706,
    0.06468064680646807,
    0.06469064690646906,
    0.06470064700647006,
    0.06471064710647106,
    0.06472064720647207,
    0.06473064730647306,
    0.06474064740647406,
    0.06475064750647506,
    0.06476064760647607,
    0.06477064770647706,
    0.06478064780647806,
    0.06479064790647907,
    0.06480064800648007,
    0.06481064810648106,
    0.06482064820648206,
    0.06483064830648307,
    0.06484064840648407,
    0.06485064850648506,
    0.06486064860648606,
    0.06487064870648707,
    0.06488064880648807,
    0.06489064890648906,
    0.06490064900649006,
    0.06491064910649107,
    0.06492064920649207,
    0.06493064930649306,
    0.06494064940649406,
    0.06495064950649507,
    0.06496064960649606,
    0.06497064970649706,
    0.06498064980649806,
    0.06499064990649907,
    0.06500065000650006,
    0.06501065010650106,
    0.06502065020650206,
    0.06503065030650307,
    0.06504065040650406,
    0.06505065050650506,
    0.06506065060650607,
    0.06507065070650707,
    0.06508065080650806,
    0.06509065090650906,
    0.06510065100651007,
    0.06511065110651107,
    0.06512065120651206,
    0.06513065130651306,
    0.06514065140651407,
    0.06515065150651507,
    0.06516065160651606,
    0.06517065170651706,
    0.06518065180651807,
    0.06519065190651907,
    0.06520065200652006,
    0.06521065210652106,
    0.06522065220652207,
    0.06523065230652307,
    0.06524065240652406,
    0.06525065250652506,
    0.06526065260652607,
    0.06527065270652706,
    0.06528065280652806,
    0.06529065290652906,
    0.06530065300653007,
    0.06531065310653106,
    0.06532065320653206,
    0.06533065330653307,
    0.06534065340653407,
    0.06535065350653506,
    0.06536065360653606,
    0.06537065370653707,
    0.06538065380653807,
    0.06539065390653906,
    0.06540065400654006,
    0.06541065410654107,
    0.06542065420654207,
    0.06543065430654306,
    0.06544065440654406,
    0.06545065450654507,
    0.06546065460654607,
    0.06547065470654706,
    0.06548065480654806,
    0.06549065490654907,
    0.06550065500655007,
    0.06551065510655106,
    0.06552065520655206,
    0.06553065530655307,
    0.06554065540655406,
    0.06555065550655506,
    0.06556065560655606,
    0.06557065570655707,
    0.06558065580655806,
    0.06559065590655906,
    0.06560065600656007,
    0.06561065610656107,
    0.06562065620656206,
    0.06563065630656306,
    0.06564065640656407,
    0.06565065650656507,
    0.06566065660656606,
    0.06567065670656706,
    0.06568065680656807,
    0.06569065690656907,
    0.06570065700657006,
    0.06571065710657106,
    0.06572065720657207,
    0.06573065730657307,
    0.06574065740657406,
    0.06575065750657506,
    0.06576065760657607,
    0.06577065770657707,
    0.06578065780657806,
    0.06579065790657906,
    0.06580065800658007,
    0.06581065810658107,
    0.06582065820658206,
    0.06583065830658306,
    0.06584065840658407,
    0.06585065850658506,
    0.06586065860658606,
    0.06587065870658707,
    0.06588065880658807,
    0.06589065890658906,
    0.06590065900659006,
    0.06591065910659107,
    0.06592065920659207,
    0.06593065930659306,
    0.06594065940659406,
    0.06595065950659507,
    0.06596065960659607,
    0.06597065970659706,
    0.06598065980659806,
    0.06599065990659907,
    0.06600066000660007,
    0.06601066010660106,
    0.06602066020660206,
    0.06603066030660307,
    0.06604066040660407,
    0.06605066050660506,
    0.06606066060660606,
    0.06607066070660707,
    0.06608066080660807,
    0.06609066090660906,
    0.06610066100661006,
    0.06611066110661107,
    0.06612066120661207,
    0.06613066130661306,
    0.06614066140661407,
    0.06615066150661507,
    0.06616066160661606,
    0.06617066170661706,
    0.06618066180661807,
    0.06619066190661907,
    0.06620066200662006,
    0.06621066210662106,
    0.06622066220662207,
    0.06623066230662307,
    0.06624066240662406,
    0.06625066250662506,
    0.06626066260662607,
    0.06627066270662707,
    0.06628066280662806,
    0.06629066290662906,
    0.06630066300663007,
    0.06631066310663107,
    0.06632066320663206,
    0.06633066330663306,
    0.06634066340663407,
    0.06635066350663507,
    0.06636066360663606,
    0.06637066370663706,
    0.06638066380663807,
    0.06639066390663907,
    0.06640066400664006,
    0.06641066410664107,
    0.06642066420664207,
    0.06643066430664306,
    0.06644066440664406,
    0.06645066450664507,
    0.06646066460664607,
    0.06647066470664706,
    0.06648066480664806,
    0.06649066490664907,
    0.06650066500665007,
    0.06651066510665106,
    0.06652066520665206,
    0.06653066530665307,
    0.06654066540665407,
    0.06655066550665506,
    0.06656066560665606,
    0.06657066570665707,
    0.06658066580665807,
    0.06659066590665906,
    0.06660066600666006,
    0.06661066610666107,
    0.06662066620666207,
    0.06663066630666306,
    0.06664066640666406,
    0.06665066650666507,
    0.06666066660666607,
    0.06667066670666706,
    0.06668066680666807,
    0.06669066690666907,
    0.06670066700667007,
    0.06671066710667106,
    0.06672066720667207,
    0.06673066730667307,
    0.06674066740667406,
    0.06675066750667506,
    0.06676066760667607,
    0.06677066770667707,
    0.06678066780667806,
    0.06679066790667906,
    0.06680066800668007,
    0.06681066810668107,
    0.06682066820668206,
    0.06683066830668306,
    0.06684066840668407,
    0.06685066850668507,
    0.06686066860668606,
    0.06687066870668706,
    0.06688066880668807,
    0.06689066890668907,
    0.06690066900669006,
    0.06691066910669106,
    0.06692066920669207,
    0.06693066930669307,
    0.06694066940669406,
    0.06695066950669507,
    0.06696066960669607,
    0.06697066970669707,
    0.06698066980669806,
    0.06699066990669907,
    0.06700067000670007,
    0.06701067010670106,
    0.06702067020670206,
    0.06703067030670307,
    0.06704067040670407,
    0.06705067050670506,
    0.06706067060670606,
    0.06707067070670707,
    0.06708067080670807,
    0.06709067090670906,
    0.06710067100671006,
    0.06711067110671107,
    0.06712067120671207,
    0.06713067130671306,
    0.06714067140671406,
    0.06715067150671507,
    0.06716067160671607,
    0.06717067170671706,
    0.06718067180671806,
    0.06719067190671907,
    0.06720067200672007,
    0.06721067210672106,
    0.06722067220672207,
    0.06723067230672307,
    0.06724067240672407,
    0.06725067250672506,
    0.06726067260672607,
    0.06727067270672707,
    0.06728067280672807,
    0.06729067290672906,
    0.06730067300673007,
    0.06731067310673107,
    0.06732067320673206,
    0.06733067330673306,
    0.06734067340673407,
    0.06735067350673507,
    0.06736067360673606,
    0.06737067370673706,
    0.06738067380673807,
    0.06739067390673907,
    0.06740067400674006,
    0.06741067410674106,
    0.06742067420674207,
    0.06743067430674307,
    0.06744067440674406,
    0.06745067450674506,
    0.06746067460674607,
    0.06747067470674707,
    0.06748067480674806,
    0.06749067490674907,
    0.06750067500675007,
    0.06751067510675107,
    0.06752067520675206,
    0.06753067530675307,
    0.06754067540675407,
    0.06755067550675507,
    0.06756067560675606,
    0.06757067570675707,
    0.06758067580675807,
    0.06759067590675906,
    0.06760067600676006,
    0.06761067610676107,
    0.06762067620676207,
    0.06763067630676306,
    0.06764067640676406,
    0.06765067650676507,
    0.06766067660676607,
    0.06767067670676706,
    0.06768067680676806,
    0.06769067690676907,
    0.06770067700677007,
    0.06771067710677106,
    0.06772067720677206,
    0.06773067730677307,
    0.06774067740677407,
    0.06775067750677506,
    0.06776067760677607,
    0.06777067770677707,
    0.06778067780677807,
    0.06779067790677906,
    0.06780067800678007,
    0.06781067810678107,
    0.06782067820678207,
    0.06783067830678306,
    0.06784067840678407,
    0.06785067850678507,
    0.06786067860678607,
    0.06787067870678706,
    0.06788067880678807,
    0.06789067890678907,
    0.06790067900679006,
    0.06791067910679106,
    0.06792067920679207,
    0.06793067930679307,
    0.06794067940679406,
    0.06795067950679506,
    0.06796067960679607,
    0.06797067970679707,
    0.06798067980679806,
    0.06799067990679906,
    0.06800068000680007,
    0.06801068010680107,
    0.06802068020680206,
    0.06803068030680307,
    0.06804068040680407,
    0.06805068050680507,
    0.06806068060680606,
    0.06807068070680707,
    0.06808068080680807,
    0.06809068090680907,
    0.06810068100681006,
    0.06811068110681107,
    0.06812068120681207,
    0.06813068130681307,
    0.06814068140681406,
    0.06815068150681507,
    0.06816068160681607,
    0.06817068170681706,
    0.06818068180681806,
    0.06819068190681907,
    0.06820068200682007,
    0.06821068210682106,
    0.06822068220682206,
    0.06823068230682307,
    0.06824068240682407,
    0.06825068250682506,
    0.06826068260682606,
    0.06827068270682707,
    0.06828068280682807,
    0.06829068290682906,
    0.06830068300683007,
    0.06831068310683107,
    0.06832068320683207,
    0.06833068330683306,
    0.06834068340683407,
    0.06835068350683507,
    0.06836068360683607,
    0.06837068370683706,
    0.06838068380683807,
    0.06839068390683907,
    0.06840068400684007,
    0.06841068410684106,
    0.06842068420684207,
    0.06843068430684307,
    0.06844068440684407,
    0.06845068450684506,
    0.06846068460684607,
    0.06847068470684707,
    0.06848068480684806,
    0.06849068490684906,
    0.06850068500685007,
    0.06851068510685107,
    0.06852068520685206,
    0.06853068530685306,
    0.06854068540685407,
    0.06855068550685507,
    0.06856068560685606,
    0.06857068570685707,
    0.06858068580685807,
    0.06859068590685907,
    0.06860068600686006,
    0.06861068610686107,
    0.06862068620686207,
    0.06863068630686307,
    0.06864068640686406,
    0.06865068650686507,
    0.06866068660686607,
    0.06867068670686707,
    0.06868068680686806,
    0.06869068690686907,
    0.06870068700687007,
    0.06871068710687107,
    0.06872068720687206,
    0.06873068730687307,
    0.06874068740687407,
    0.06875068750687507,
    0.06876068760687606,
    0.06877068770687707,
    0.06878068780687807,
    0.06879068790687906,
    0.06880068800688006,
    0.06881068810688107,
    0.06882068820688207,
    0.06883068830688306,
    0.06884068840688407,
    0.06885068850688507,
    0.06886068860688607,
    0.06887068870688706,
    0.06888068880688807,
    0.06889068890688907,
    0.06890068900689007,
    0.06891068910689106,
    0.06892068920689207,
    0.06893068930689307,
    0.06894068940689407,
    0.06895068950689506,
    0.06896068960689607,
    0.06897068970689707,
    0.06898068980689807,
    0.06899068990689906,
    0.06900069000690007,
    0.06901069010690107,
    0.06902069020690207,
    0.06903069030690306,
    0.06904069040690407,
    0.06905069050690507,
    0.06906069060690606,
    0.06907069070690706,
    0.06908069080690807,
    0.06909069090690907,
    0.06910069100691006,
    0.06911069110691107,
    0.06912069120691207,
    0.06913069130691307,
    0.06914069140691406,
    0.06915069150691507,
    0.06916069160691607,
    0.06917069170691707,
    0.06918069180691806,
    0.06919069190691907,
    0.06920069200692007,
    0.06921069210692107,
    0.06922069220692206,
    0.06923069230692307,
    0.06924069240692407,
    0.06925069250692507,
    0.06926069260692606,
    0.06927069270692707,
    0.06928069280692807,
    0.06929069290692907,
    0.06930069300693006,
    0.06931069310693107,
    0.06932069320693207,
    0.06933069330693307,
    0.06934069340693406,
    0.06935069350693507,
    0.06936069360693607,
    0.06937069370693706,
    0.06938069380693807,
    0.06939069390693907,
    0.06940069400694007,
    0.06941069410694106,
    0.06942069420694207,
    0.06943069430694307,
    0.06944069440694407,
    0.06945069450694506,
    0.06946069460694607,
    0.06947069470694707,
    0.06948069480694807,
    0.06949069490694906,
    0.06950069500695007,
    0.06951069510695107,
    0.06952069520695207,
    0.06953069530695306,
    0.06954069540695407,
    0.06955069550695507,
    0.06956069560695607,
    0.06957069570695706,
    0.06958069580695807,
    0.06959069590695907,
    0.06960069600696007,
    0.06961069610696106,
    0.06962069620696207,
    0.06963069630696307,
    0.06964069640696406,
    0.06965069650696507,
    0.06966069660696607,
    0.06967069670696707,
    0.06968069680696806,
    0.06969069690696907,
    0.06970069700697007,
    0.06971069710697107,
    0.06972069720697206,
    0.06973069730697307,
    0.06974069740697407,
    0.06975069750697507,
    0.06976069760697606,
    0.06977069770697707,
    0.06978069780697807,
    0.06979069790697907,
    0.06980069800698006,
    0.06981069810698107,
    0.06982069820698207,
    0.06983069830698307,
    0.06984069840698406,
    0.06985069850698507,
    0.06986069860698607,
    0.06987069870698707,
    0.06988069880698806,
    0.06989069890698907,
    0.06990069900699007,
    0.06991069910699108,
    0.06992069920699207,
    0.06993069930699307,
    0.06994069940699407,
    0.06995069950699506,
    0.06996069960699607,
    0.06997069970699707,
    0.06998069980699807,
    0.06999069990699906,
    0.07000070000700007,
    0.07001070010700107,
    0.07002070020700207,
    0.07003070030700306,
    0.07004070040700407,
    0.07005070050700507,
    0.07006070060700607,
    0.07007070070700706,
    0.07008070080700807,
    0.07009070090700907,
    0.07010070100701007,
    0.07011070110701106,
    0.07012070120701207,
    0.07013070130701307,
    0.07014070140701407,
    0.07015070150701506,
    0.07016070160701607,
    0.07017070170701707,
    0.07018070180701808,
    0.07019070190701907,
    0.07020070200702007,
    0.07021070210702107,
    0.07022070220702206,
    0.07023070230702307,
    0.07024070240702407,
    0.07025070250702507,
    0.07026070260702606,
    0.07027070270702707,
    0.07028070280702807,
    0.07029070290702907,
    0.07030070300703006,
    0.07031070310703107,
    0.07032070320703207,
    0.07033070330703307,
    0.07034070340703406,
    0.07035070350703507,
    0.07036070360703607,
    0.07037070370703707,
    0.07038070380703806,
    0.07039070390703907,
    0.07040070400704007,
    0.07041070410704107,
    0.07042070420704206,
    0.07043070430704307,
    0.07044070440704407,
    0.07045070450704508,
    0.07046070460704607,
    0.07047070470704707,
    0.07048070480704807,
    0.07049070490704908,
    0.07050070500705007,
    0.07051070510705107,
    0.07052070520705207,
    0.07053070530705306,
    0.07054070540705407,
    0.07055070550705507,
    0.07056070560705607,
    0.07057070570705706,
    0.07058070580705807,
    0.07059070590705907,
    0.07060070600706007,
    0.07061070610706106,
    0.07062070620706207,
    0.07063070630706307,
    0.07064070640706407,
    0.07065070650706506,
    0.07066070660706607,
    0.07067070670706707,
    0.07068070680706807,
    0.07069070690706906,
    0.07070070700707007,
    0.07071070710707107,
    0.07072070720707208,
    0.07073070730707307,
    0.07074070740707407,
    0.07075070750707507,
    0.07076070760707608,
    0.07077070770707707,
    0.07078070780707807,
    0.07079070790707907,
    0.07080070800708008,
    0.07081070810708107,
    0.07082070820708207,
    0.07083070830708307,
    0.07084070840708406,
    0.07085070850708507,
    0.07086070860708607,
    0.07087070870708707,
    0.07088070880708806,
    0.07089070890708907,
    0.07090070900709007,
    0.07091070910709107,
    0.07092070920709206,
    0.07093070930709307,
    0.07094070940709407,
    0.07095070950709507,
    0.07096070960709606,
    0.07097070970709707,
    0.07098070980709807,
    0.07099070990709908,
    0.07100071000710007,
    0.07101071010710107,
    0.07102071020710207,
    0.07103071030710308,
    0.07104071040710407,
    0.07105071050710507,
    0.07106071060710607,
    0.07107071070710708,
    0.07108071080710807,
    0.07109071090710907,
    0.07110071100711007,
    0.07111071110711106,
    0.07112071120711207,
    0.07113071130711307,
    0.07114071140711407,
    0.07115071150711506,
    0.07116071160711607,
    0.07117071170711707,
    0.07118071180711807,
    0.07119071190711906,
    0.07120071200712007,
    0.07121071210712107,
    0.07122071220712207,
    0.07123071230712306,
    0.07124071240712407,
    0.07125071250712507,
    0.07126071260712608,
    0.07127071270712707,
    0.07128071280712807,
    0.07129071290712907,
    0.07130071300713008,
    0.07131071310713107,
    0.07132071320713207,
    0.07133071330713307,
    0.07134071340713408,
    0.07135071350713507,
    0.07136071360713607,
    0.07137071370713707,
    0.07138071380713808,
    0.07139071390713907,
    0.07140071400714007,
    0.07141071410714107,
    0.07142071420714206,
    0.07143071430714307,
    0.07144071440714407,
    0.07145071450714507,
    0.07146071460714606,
    0.07147071470714707,
    0.07148071480714807,
    0.07149071490714907,
    0.07150071500715006,
    0.07151071510715107,
    0.07152071520715207,
    0.07153071530715308,
    0.07154071540715407,
    0.07155071550715507,
    0.07156071560715607,
    0.07157071570715708,
    0.07158071580715807,
    0.07159071590715907,
    0.07160071600716007,
    0.07161071610716108,
    0.07162071620716207,
    0.07163071630716307,
    0.07164071640716407,
    0.07165071650716508,
    0.07166071660716607,
    0.07167071670716707,
    0.07168071680716807,
    0.07169071690716906,
    0.07170071700717007,
    0.07171071710717107,
    0.07172071720717207,
    0.07173071730717306,
    0.07174071740717407,
    0.07175071750717507,
    0.07176071760717607,
    0.07177071770717706,
    0.07178071780717807,
    0.07179071790717907,
    0.07180071800718008,
    0.07181071810718107,
    0.07182071820718207,
    0.07183071830718307,
    0.07184071840718408,
    0.07185071850718507,
    0.07186071860718607,
    0.07187071870718707,
    0.07188071880718808,
    0.07189071890718907,
    0.07190071900719007,
    0.07191071910719107,
    0.07192071920719208,
    0.07193071930719307,
    0.07194071940719407,
    0.07195071950719507,
    0.07196071960719608,
    0.07197071970719707,
    0.07198071980719807,
    0.07199071990719907,
    0.07200072000720006,
    0.07201072010720107,
    0.07202072020720207,
    0.07203072030720307,
    0.07204072040720406,
    0.07205072050720507,
    0.07206072060720607,
    0.07207072070720708,
    0.07208072080720807,
    0.07209072090720907,
    0.07210072100721007,
    0.07211072110721108,
    0.07212072120721207,
    0.07213072130721307,
    0.07214072140721407,
    0.07215072150721508,
    0.07216072160721607,
    0.07217072170721707,
    0.07218072180721807,
    0.07219072190721908,
    0.07220072200722007,
    0.07221072210722107,
    0.07222072220722207,
    0.07223072230722308,
    0.07224072240722407,
    0.07225072250722507,
    0.07226072260722607,
    0.07227072270722706,
    0.07228072280722807,
    0.07229072290722907,
    0.07230072300723007,
    0.07231072310723106,
    0.07232072320723207,
    0.07233072330723307,
    0.07234072340723408,
    0.07235072350723507,
    0.07236072360723607,
    0.07237072370723707,
    0.07238072380723808,
    0.07239072390723907,
    0.07240072400724007,
    0.07241072410724107,
    0.07242072420724208,
    0.07243072430724307,
    0.07244072440724407,
    0.07245072450724507,
    0.07246072460724608,
    0.07247072470724707,
    0.07248072480724807,
    0.07249072490724907,
    0.07250072500725008,
    0.07251072510725107,
    0.07252072520725207,
    0.07253072530725307,
    0.07254072540725408,
    0.07255072550725507,
    0.07256072560725607,
    0.07257072570725707,
    0.07258072580725806,
    0.07259072590725907,
    0.07260072600726007,
    0.07261072610726108,
    0.07262072620726207,
    0.07263072630726307,
    0.07264072640726407,
    0.07265072650726508,
    0.07266072660726607,
    0.07267072670726707,
    0.07268072680726807,
    0.07269072690726908,
    0.07270072700727007,
    0.07271072710727107,
    0.07272072720727207,
    0.07273072730727308,
    0.07274072740727407,
    0.07275072750727507,
    0.07276072760727607,
    0.07277072770727708,
    0.07278072780727807,
    0.07279072790727907,
    0.07280072800728007,
    0.07281072810728108,
    0.07282072820728207,
    0.07283072830728307,
    0.07284072840728407,
    0.07285072850728508,
    0.07286072860728607,
    0.07287072870728707,
    0.07288072880728808,
    0.07289072890728907,
    0.07290072900729007,
    0.07291072910729107,
    0.07292072920729208,
    0.07293072930729307,
    0.07294072940729407,
    0.07295072950729507,
    0.07296072960729608,
    0.07297072970729707,
    0.07298072980729807,
    0.07299072990729907,
    0.07300073000730008,
    0.07301073010730107,
    0.07302073020730207,
    0.07303073030730307,
    0.07304073040730408,
    0.07305073050730507,
    0.07306073060730607,
    0.07307073070730707,
    0.07308073080730808,
    0.07309073090730907,
    0.07310073100731007,
    0.07311073110731107,
    0.07312073120731208,
    0.07313073130731307,
    0.07314073140731407,
    0.07315073150731508,
    0.07316073160731607,
    0.07317073170731707,
    0.07318073180731807,
    0.07319073190731908,
    0.07320073200732007,
    0.07321073210732107,
    0.07322073220732207,
    0.07323073230732308,
    0.07324073240732407,
    0.07325073250732507,
    0.07326073260732607,
    0.07327073270732708,
    0.07328073280732807,
    0.07329073290732907,
    0.07330073300733007,
    0.07331073310733108,
    0.07332073320733207,
    0.07333073330733307,
    0.07334073340733407,
    0.07335073350733508,
    0.07336073360733607,
    0.07337073370733707,
    0.07338073380733807,
    0.07339073390733908,
    0.07340073400734007,
    0.07341073410734107,
    0.07342073420734208,
    0.07343073430734308,
    0.07344073440734407,
    0.07345073450734507,
    0.07346073460734608,
    0.07347073470734707,
    0.07348073480734807,
    0.07349073490734907,
    0.07350073500735008,
    0.07351073510735107,
    0.07352073520735207,
    0.07353073530735307,
    0.07354073540735408,
    0.07355073550735507,
    0.07356073560735607,
    0.07357073570735707,
    0.07358073580735808,
    0.07359073590735907,
    0.07360073600736007,
    0.07361073610736107,
    0.07362073620736208,
    0.07363073630736307,
    0.07364073640736407,
    0.07365073650736507,
    0.07366073660736608,
    0.07367073670736707,
    0.07368073680736807,
    0.07369073690736908,
    0.07370073700737008,
    0.07371073710737107,
    0.07372073720737207,
    0.07373073730737308,
    0.07374073740737407,
    0.07375073750737507,
    0.07376073760737607,
    0.07377073770737708,
    0.07378073780737807,
    0.07379073790737907,
    0.07380073800738007,
    0.07381073810738108,
    0.07382073820738207,
    0.07383073830738307,
    0.07384073840738407,
    0.07385073850738508,
    0.07386073860738607,
    0.07387073870738707,
    0.07388073880738807,
    0.07389073890738908,
    0.07390073900739007,
    0.07391073910739107,
    0.07392073920739207,
    0.07393073930739308,
    0.07394073940739407,
    0.07395073950739507,
    0.07396073960739608,
    0.07397073970739708,
    0.07398073980739807,
    0.07399073990739907,
    0.07400074000740008,
    0.07401074010740108,
    0.07402074020740207,
    0.07403074030740307,
    0.07404074040740408,
    0.07405074050740507,
    0.07406074060740607,
    0.07407074070740707,
    0.07408074080740808,
    0.07409074090740907,
    0.07410074100741007,
    0.07411074110741107,
    0.07412074120741208,
    0.07413074130741307,
    0.07414074140741407,
    0.07415074150741507,
    0.07416074160741608,
    0.07417074170741707,
    0.07418074180741807,
    0.07419074190741907,
    0.07420074200742008,
    0.07421074210742107,
    0.07422074220742207,
    0.07423074230742308,
    0.07424074240742408,
    0.07425074250742507,
    0.07426074260742607,
    0.07427074270742708,
    0.07428074280742808,
    0.07429074290742907,
    0.07430074300743007,
    0.07431074310743108,
    0.07432074320743207,
    0.07433074330743307,
    0.07434074340743407,
    0.07435074350743508,
    0.07436074360743607,
    0.07437074370743707,
    0.07438074380743807,
    0.07439074390743908,
    0.07440074400744007,
    0.07441074410744107,
    0.07442074420744207,
    0.07443074430744308,
    0.07444074440744407,
    0.07445074450744507,
    0.07446074460744607,
    0.07447074470744708,
    0.07448074480744807,
    0.07449074490744907,
    0.07450074500745008,
    0.07451074510745108,
    0.07452074520745207,
    0.07453074530745307,
    0.07454074540745408,
    0.07455074550745508,
    0.07456074560745607,
    0.07457074570745707,
    0.07458074580745808,
    0.07459074590745908,
    0.07460074600746007,
    0.07461074610746107,
    0.07462074620746208,
    0.07463074630746307,
    0.07464074640746407,
    0.07465074650746507,
    0.07466074660746608,
    0.07467074670746707,
    0.07468074680746807,
    0.07469074690746907,
    0.07470074700747008,
    0.07471074710747107,
    0.07472074720747207,
    0.07473074730747307,
    0.07474074740747408,
    0.07475074750747507,
    0.07476074760747607,
    0.07477074770747708,
    0.07478074780747808,
    0.07479074790747907,
    0.07480074800748007,
    0.07481074810748108,
    0.07482074820748208,
    0.07483074830748307,
    0.07484074840748407,
    0.07485074850748508,
    0.07486074860748608,
    0.07487074870748707,
    0.07488074880748807,
    0.07489074890748908,
    0.07490074900749007,
    0.07491074910749107,
    0.07492074920749207,
    0.07493074930749308,
    0.07494074940749407,
    0.07495074950749507,
    0.07496074960749607,
    0.07497074970749708,
    0.07498074980749807,
    0.07499074990749907,
    0.07500075000750007,
    0.07501075010750108,
    0.07502075020750207,
    0.07503075030750307,
    0.07504075040750408,
    0.07505075050750508,
    0.07506075060750607,
    0.07507075070750707,
    0.07508075080750808,
    0.07509075090750908,
    0.07510075100751007,
    0.07511075110751107,
    0.07512075120751208,
    0.07513075130751308,
    0.07514075140751407,
    0.07515075150751507,
    0.07516075160751608,
    0.07517075170751708,
    0.07518075180751807,
    0.07519075190751907,
    0.07520075200752008,
    0.07521075210752107,
    0.07522075220752207,
    0.07523075230752307,
    0.07524075240752408,
    0.07525075250752507,
    0.07526075260752607,
    0.07527075270752707,
    0.07528075280752808,
    0.07529075290752907,
    0.07530075300753007,
    0.07531075310753108,
    0.07532075320753208,
    0.07533075330753307,
    0.07534075340753407,
    0.07535075350753508,
    0.07536075360753608,
    0.07537075370753707,
    0.07538075380753807,
    0.07539075390753908,
    0.07540075400754008,
    0.07541075410754107,
    0.07542075420754207,
    0.07543075430754308,
    0.07544075440754408,
    0.07545075450754507,
    0.07546075460754607,
    0.07547075470754708,
    0.07548075480754808,
    0.07549075490754907,
    0.07550075500755007,
    0.07551075510755108,
    0.07552075520755207,
    0.07553075530755307,
    0.07554075540755407,
    0.07555075550755508,
    0.07556075560755607,
    0.07557075570755707,
    0.07558075580755808,
    0.07559075590755908,
    0.07560075600756007,
    0.07561075610756107,
    0.07562075620756208,
    0.07563075630756308,
    0.07564075640756407,
    0.07565075650756507,
    0.07566075660756608,
    0.07567075670756708,
    0.07568075680756807,
    0.07569075690756907,
    0.07570075700757008,
    0.07571075710757108,
    0.07572075720757207,
    0.07573075730757307,
    0.07574075740757408,
    0.07575075750757508,
    0.07576075760757607,
    0.07577075770757707,
    0.07578075780757808,
    0.07579075790757907,
    0.07580075800758007,
    0.07581075810758107,
    0.07582075820758208,
    0.07583075830758307,
    0.07584075840758407,
    0.07585075850758508,
    0.07586075860758608,
    0.07587075870758707,
    0.07588075880758807,
    0.07589075890758908,
    0.07590075900759008,
    0.07591075910759107,
    0.07592075920759207,
    0.07593075930759308,
    0.07594075940759408,
    0.07595075950759507,
    0.07596075960759607,
    0.07597075970759708,
    0.07598075980759808,
    0.07599075990759907,
    0.07600076000760007,
    0.07601076010760108,
    0.07602076020760208,
    0.07603076030760307,
    0.07604076040760407,
    0.07605076050760508,
    0.07606076060760608,
    0.07607076070760707,
    0.07608076080760807,
    0.07609076090760908,
    0.07610076100761007,
    0.07611076110761107,
    0.07612076120761208,
    0.07613076130761308,
    0.07614076140761407,
    0.07615076150761507,
    0.07616076160761608,
    0.07617076170761708,
    0.07618076180761807,
    0.07619076190761907,
    0.07620076200762008,
    0.07621076210762108,
    0.07622076220762207,
    0.07623076230762307,
    0.07624076240762408,
    0.07625076250762508,
    0.07626076260762607,
    0.07627076270762707,
    0.07628076280762808,
    0.07629076290762908,
    0.07630076300763007,
    0.07631076310763107,
    0.07632076320763208,
    0.07633076330763308,
    0.07634076340763407,
    0.07635076350763507,
    0.07636076360763608,
    0.07637076370763707,
    0.07638076380763807,
    0.07639076390763908,
    0.07640076400764008,
    0.07641076410764107,
    0.07642076420764207,
    0.07643076430764308,
    0.07644076440764408,
    0.07645076450764507,
    0.07646076460764607,
    0.07647076470764708,
    0.07648076480764808,
    0.07649076490764907,
    0.07650076500765007,
    0.07651076510765108,
    0.07652076520765208,
    0.07653076530765307,
    0.07654076540765407,
    0.07655076550765508,
    0.07656076560765608,
    0.07657076570765707,
    0.07658076580765807,
    0.07659076590765908,
    0.07660076600766008,
    0.07661076610766107,
    0.07662076620766207,
    0.07663076630766308,
    0.07664076640766408,
    0.07665076650766507,
    0.07666076660766608,
    0.07667076670766708,
    0.07668076680766807,
    0.07669076690766907,
    0.07670076700767008,
    0.07671076710767108,
    0.07672076720767207,
    0.07673076730767307,
    0.07674076740767408,
    0.07675076750767508,
    0.07676076760767607,
    0.07677076770767707,
    0.07678076780767808,
    0.07679076790767908,
    0.07680076800768007,
    0.07681076810768107,
    0.07682076820768208,
    0.07683076830768308,
    0.07684076840768407,
    0.07685076850768507,
    0.07686076860768608,
    0.07687076870768708,
    0.07688076880768807,
    0.07689076890768907,
    0.07690076900769008,
    0.07691076910769108,
    0.07692076920769207,
    0.07693076930769308,
    0.07694076940769408,
    0.07695076950769507,
    0.07696076960769607,
    0.07697076970769708,
    0.07698076980769808,
    0.07699076990769907,
    0.07700077000770007,
    0.07701077010770108,
    0.07702077020770208,
    0.07703077030770307,
    0.07704077040770407,
    0.07705077050770508,
    0.07706077060770608,
    0.07707077070770707,
    0.07708077080770807,
    0.07709077090770908,
    0.07710077100771008,
    0.07711077110771107,
    0.07712077120771207,
    0.07713077130771308,
    0.07714077140771408,
    0.07715077150771507,
    0.07716077160771607,
    0.07717077170771708,
    0.07718077180771808,
    0.07719077190771907,
    0.07720077200772008,
    0.07721077210772108,
    0.07722077220772208,
    0.07723077230772307,
    0.07724077240772408,
    0.07725077250772508,
    0.07726077260772607,
    0.07727077270772707,
    0.07728077280772808,
    0.07729077290772908,
    0.07730077300773007,
    0.07731077310773107,
    0.07732077320773208,
    0.07733077330773308,
    0.07734077340773407,
    0.07735077350773507,
    0.07736077360773608,
    0.07737077370773708,
    0.07738077380773807,
    0.07739077390773907,
    0.07740077400774008,
    0.07741077410774108,
    0.07742077420774207,
    0.07743077430774307,
    0.07744077440774408,
    0.07745077450774508,
    0.07746077460774607,
    0.07747077470774708,
    0.07748077480774808,
    0.07749077490774908,
    0.07750077500775007,
    0.07751077510775108,
    0.07752077520775208,
    0.07753077530775308,
    0.07754077540775407,
    0.07755077550775508,
    0.07756077560775608,
    0.07757077570775707,
    0.07758077580775807,
    0.07759077590775908,
    0.07760077600776008,
    0.07761077610776107,
    0.07762077620776207,
    0.07763077630776308,
    0.07764077640776408,
    0.07765077650776507,
    0.07766077660776607,
    0.07767077670776708,
    0.07768077680776808,
    0.07769077690776907,
    0.07770077700777007,
    0.07771077710777108,
    0.07772077720777208,
    0.07773077730777307,
    0.07774077740777408,
    0.07775077750777508,
    0.07776077760777608,
    0.07777077770777707,
    0.07778077780777808,
    0.07779077790777908,
    0.07780077800778008,
    0.07781077810778107,
    0.07782077820778208,
    0.07783077830778308,
    0.07784077840778407,
    0.07785077850778507,
    0.07786077860778608,
    0.07787077870778708,
    0.07788077880778807,
    0.07789077890778907,
    0.07790077900779008,
    0.07791077910779108,
    0.07792077920779207,
    0.07793077930779307,
    0.07794077940779408,
    0.07795077950779508,
    0.07796077960779607,
    0.07797077970779707,
    0.07798077980779808,
    0.07799077990779908,
    0.07800078000780007,
    0.07801078010780108,
    0.07802078020780208,
    0.07803078030780308,
    0.07804078040780407,
    0.07805078050780508,
    0.07806078060780608,
    0.07807078070780708,
    0.07808078080780807,
    0.07809078090780908,
    0.07810078100781008,
    0.07811078110781108,
    0.07812078120781207,
    0.07813078130781308,
    0.07814078140781408,
    0.07815078150781507,
    0.07816078160781607,
    0.07817078170781708,
    0.07818078180781808,
    0.07819078190781907,
    0.07820078200782007,
    0.07821078210782108,
    0.07822078220782208,
    0.07823078230782307,
    0.07824078240782407,
    0.07825078250782508,
    0.07826078260782608,
    0.07827078270782707,
    0.07828078280782808,
    0.07829078290782908,
    0.07830078300783008,
    0.07831078310783107,
    0.07832078320783208,
    0.07833078330783308,
    0.07834078340783408,
    0.07835078350783507,
    0.07836078360783608,
    0.07837078370783708,
    0.07838078380783808,
    0.07839078390783907,
    0.07840078400784008,
    0.07841078410784108,
    0.07842078420784207,
    0.07843078430784307,
    0.07844078440784408,
    0.07845078450784508,
    0.07846078460784607,
    0.07847078470784707,
    0.07848078480784808,
    0.07849078490784908,
    0.07850078500785007,
    0.07851078510785107,
    0.07852078520785208,
    0.07853078530785308,
    0.07854078540785407,
    0.07855078550785508,
    0.07856078560785608,
    0.07857078570785708,
    0.07858078580785807,
    0.07859078590785908,
    0.07860078600786008,
    0.07861078610786108,
    0.07862078620786207,
    0.07863078630786308,
    0.07864078640786408,
    0.07865078650786508,
    0.07866078660786607,
    0.07867078670786708,
    0.07868078680786808,
    0.07869078690786908,
    0.07870078700787007,
    0.07871078710787108,
    0.07872078720787208,
    0.07873078730787307,
    0.07874078740787407,
    0.07875078750787508,
    0.07876078760787608,
    0.07877078770787707,
    0.07878078780787807,
    0.07879078790787908,
    0.07880078800788008,
    0.07881078810788107,
    0.07882078820788208,
    0.07883078830788308,
    0.07884078840788408,
    0.07885078850788507,
    0.07886078860788608,
    0.07887078870788708,
    0.07888078880788808,
    0.07889078890788907,
    0.07890078900789008,
    0.07891078910789108,
    0.07892078920789208,
    0.07893078930789307,
    0.07894078940789408,
    0.07895078950789508,
    0.07896078960789608,
    0.07897078970789707,
    0.07898078980789808,
    0.07899078990789908,
    0.07900079000790007,
    0.07901079010790107,
    0.07902079020790208,
    0.07903079030790308,
    0.07904079040790407,
    0.07905079050790507,
    0.07906079060790608,
    0.07907079070790708,
    0.07908079080790807,
    0.07909079090790908,
    0.07910079100791008,
    0.07911079110791108,
    0.07912079120791207,
    0.07913079130791308,
    0.07914079140791408,
    0.07915079150791508,
    0.07916079160791607,
    0.07917079170791708,
    0.07918079180791808,
    0.07919079190791908,
    0.07920079200792007,
    0.07921079210792108,
    0.07922079220792208,
    0.07923079230792308,
    0.07924079240792407,
    0.07925079250792508,
    0.07926079260792608,
    0.07927079270792708,
    0.07928079280792807,
    0.07929079290792908,
    0.07930079300793008,
    0.07931079310793107,
    0.07932079320793207,
    0.07933079330793308,
    0.07934079340793408,
    0.07935079350793507,
    0.07936079360793608,
    0.07937079370793708,
    0.07938079380793808,
    0.07939079390793907,
    0.07940079400794008,
    0.07941079410794108,
    0.07942079420794208,
    0.07943079430794307,
    0.07944079440794408,
    0.07945079450794508,
    0.07946079460794608,
    0.07947079470794707,
    0.07948079480794808,
    0.07949079490794908,
    0.07950079500795008,
    0.07951079510795107,
    0.07952079520795208,
    0.07953079530795308,
    0.07954079540795408,
    0.07955079550795507,
    0.07956079560795608,
    0.07957079570795708,
    0.07958079580795807,
    0.07959079590795907,
    0.07960079600796008,
    0.07961079610796108,
    0.07962079620796207,
    0.07963079630796308,
    0.07964079640796408,
    0.07965079650796508,
    0.07966079660796607,
    0.07967079670796708,
    0.07968079680796808,
    0.07969079690796908,
    0.07970079700797007,
    0.07971079710797108,
    0.07972079720797208,
    0.07973079730797308,
    0.07974079740797407,
    0.07975079750797508,
    0.07976079760797608,
    0.07977079770797708,
    0.07978079780797807,
    0.07979079790797908,
    0.07980079800798008,
    0.07981079810798108,
    0.07982079820798207,
    0.07983079830798308,
    0.07984079840798408,
    0.07985079850798509,
    0.07986079860798607,
    0.07987079870798708,
    0.07988079880798808,
    0.07989079890798907,
    0.07990079900799008,
    0.07991079910799108,
    0.07992079920799208,
    0.07993079930799307,
    0.07994079940799408,
    0.07995079950799508,
    0.07996079960799608,
    0.07997079970799707,
    0.07998079980799808,
    0.07999079990799908,
    0.08000080000800008,
    0.08001080010800107,
    0.08002080020800208,
    0.08003080030800308,
    0.08004080040800408,
    0.08005080050800507,
    0.08006080060800608,
    0.08007080070800708,
    0.08008080080800808,
    0.08009080090800907,
    0.08010080100801008,
    0.08011080110801108,
    0.08012080120801209,
    0.08013080130801307,
    0.08014080140801408,
    0.08015080150801508,
    0.08016080160801609,
    0.08017080170801708,
    0.08018080180801808,
    0.08019080190801908,
    0.08020080200802007,
    0.08021080210802108,
    0.08022080220802208,
    0.08023080230802308,
    0.08024080240802407,
    0.08025080250802508,
    0.08026080260802608,
    0.08027080270802708,
    0.08028080280802807,
    0.08029080290802908,
    0.08030080300803008,
    0.08031080310803108,
    0.08032080320803207,
    0.08033080330803308,
    0.08034080340803408,
    0.08035080350803508,
    0.08036080360803607,
    0.08037080370803708,
    0.08038080380803808,
    0.08039080390803909,
    0.08040080400804007,
    0.08041080410804108,
    0.08042080420804208,
    0.08043080430804309,
    0.08044080440804408,
    0.08045080450804508,
    0.08046080460804608,
    0.08047080470804707,
    0.08048080480804808,
    0.08049080490804908,
    0.08050080500805008,
    0.08051080510805107,
    0.08052080520805208,
    0.08053080530805308,
    0.08054080540805408,
    0.08055080550805507,
    0.08056080560805608,
    0.08057080570805708,
    0.08058080580805808,
    0.08059080590805907,
    0.08060080600806008,
    0.08061080610806108,
    0.08062080620806208,
    0.08063080630806307,
    0.08064080640806408,
    0.08065080650806508,
    0.08066080660806609,
    0.08067080670806707,
    0.08068080680806808,
    0.08069080690806908,
    0.08070080700807009,
    0.08071080710807108,
    0.08072080720807208,
    0.08073080730807308,
    0.08074080740807409,
    0.08075080750807508,
    0.08076080760807608,
    0.08077080770807708,
    0.08078080780807807,
    0.08079080790807908,
    0.08080080800808008,
    0.08081080810808108,
    0.08082080820808207,
    0.08083080830808308,
    0.08084080840808408,
    0.08085080850808508,
    0.08086080860808607,
    0.08087080870808708,
    0.08088080880808808,
    0.08089080890808908,
    0.08090080900809007,
    0.08091080910809108,
    0.08092080920809208,
    0.08093080930809309,
    0.08094080940809407,
    0.08095080950809508,
    0.08096080960809608,
    0.08097080970809709,
    0.08098080980809808,
    0.08099080990809908,
    0.08100081000810008,
    0.08101081010810109,
    0.08102081020810208,
    0.08103081030810308,
    0.08104081040810408,
    0.08105081050810507,
    0.08106081060810608,
    0.08107081070810708,
    0.08108081080810808,
    0.08109081090810907,
    0.08110081100811008,
    0.08111081110811108,
    0.08112081120811208,
    0.08113081130811307,
    0.08114081140811408,
    0.08115081150811508,
    0.08116081160811608,
    0.08117081170811707,
    0.08118081180811808,
    0.08119081190811908,
    0.08120081200812009,
    0.08121081210812107,
    0.08122081220812208,
    0.08123081230812308,
    0.08124081240812409,
    0.08125081250812508,
    0.08126081260812608,
    0.08127081270812708,
    0.08128081280812809,
    0.08129081290812908,
    0.08130081300813008,
    0.08131081310813108,
    0.08132081320813209,
    0.08133081330813308,
    0.08134081340813408,
    0.08135081350813508,
    0.08136081360813607,
    0.08137081370813708,
    0.08138081380813808,
    0.08139081390813908,
    0.08140081400814007,
    0.08141081410814108,
    0.08142081420814208,
    0.08143081430814308,
    0.08144081440814407,
    0.08145081450814508,
    0.08146081460814608,
    0.08147081470814709,
    0.08148081480814807,
    0.08149081490814908,
    0.08150081500815008,
    0.08151081510815109,
    0.08152081520815208,
    0.08153081530815308,
    0.08154081540815408,
    0.08155081550815509,
    0.08156081560815608,
    0.08157081570815708,
    0.08158081580815808,
    0.08159081590815909,
    0.08160081600816008,
    0.08161081610816108,
    0.08162081620816208,
    0.08163081630816307,
    0.08164081640816408,
    0.08165081650816508,
    0.08166081660816608,
    0.08167081670816707,
    0.08168081680816808,
    0.08169081690816908,
    0.08170081700817008,
    0.08171081710817107,
    0.08172081720817208,
    0.08173081730817308,
    0.08174081740817409,
    0.08175081750817507,
    0.08176081760817608,
    0.08177081770817708,
    0.08178081780817809,
    0.08179081790817908,
    0.08180081800818008,
    0.08181081810818108,
    0.08182081820818209,
    0.08183081830818308,
    0.08184081840818408,
    0.08185081850818508,
    0.08186081860818609,
    0.08187081870818708,
    0.08188081880818808,
    0.08189081890818908,
    0.08190081900819009,
    0.08191081910819108,
    0.08192081920819208,
    0.08193081930819308,
    0.08194081940819407,
    0.08195081950819508,
    0.08196081960819608,
    0.08197081970819708,
    0.08198081980819807,
    0.08199081990819908,
    0.08200082000820008,
    0.08201082010820109,
    0.08202082020820207,
    0.08203082030820308,
    0.08204082040820408,
    0.08205082050820509,
    0.08206082060820608,
    0.08207082070820708,
    0.08208082080820808,
    0.08209082090820909,
    0.08210082100821008,
    0.08211082110821108,
    0.08212082120821208,
    0.08213082130821309,
    0.08214082140821408,
    0.08215082150821508,
    0.08216082160821608,
    0.08217082170821709,
    0.08218082180821808,
    0.08219082190821908,
    0.08220082200822008,
    0.08221082210822109,
    0.08222082220822208,
    0.08223082230822308,
    0.08224082240822408,
    0.08225082250822507,
    0.08226082260822608,
    0.08227082270822708,
    0.08228082280822809,
    0.08229082290822907,
    0.08230082300823008,
    0.08231082310823108,
    0.08232082320823209,
    0.08233082330823308,
    0.08234082340823408,
    0.08235082350823508,
    0.08236082360823609,
    0.08237082370823708,
    0.08238082380823808,
    0.08239082390823908,
    0.08240082400824009,
    0.08241082410824108,
    0.08242082420824208,
    0.08243082430824308,
    0.08244082440824409,
    0.08245082450824508,
    0.08246082460824608,
    0.08247082470824708,
    0.08248082480824809,
    0.08249082490824908,
    0.08250082500825008,
    0.08251082510825108,
    0.08252082520825207,
    0.08253082530825308,
    0.08254082540825408,
    0.08255082550825509,
    0.08256082560825607,
    0.08257082570825708,
    0.08258082580825808,
    0.08259082590825909,
    0.08260082600826008,
    0.08261082610826108,
    0.08262082620826208,
    0.08263082630826309,
    0.08264082640826408,
    0.08265082650826508,
    0.08266082660826608,
    0.08267082670826709,
    0.08268082680826808,
    0.08269082690826908,
    0.08270082700827008,
    0.08271082710827109,
    0.08272082720827208,
    0.08273082730827308,
    0.08274082740827408,
    0.08275082750827509,
    0.08276082760827608,
    0.08277082770827708,
    0.08278082780827808,
    0.08279082790827909,
    0.08280082800828008,
    0.08281082810828108,
    0.08282082820828209,
    0.08283082830828307,
    0.08284082840828408,
    0.08285082850828508,
    0.08286082860828609,
    0.08287082870828708,
    0.08288082880828808,
    0.08289082890828908,
    0.08290082900829009,
    0.08291082910829108,
    0.08292082920829208,
    0.08293082930829308,
    0.08294082940829409,
    0.08295082950829508,
    0.08296082960829608,
    0.08297082970829708,
    0.08298082980829809,
    0.08299082990829908,
    0.08300083000830008,
    0.08301083010830108,
    0.08302083020830209,
    0.08303083030830308,
    0.08304083040830408,
    0.08305083050830508,
    0.08306083060830609,
    0.08307083070830708,
    0.08308083080830808,
    0.08309083090830909,
    0.08310083100831007,
    0.08311083110831108,
    0.08312083120831208,
    0.08313083130831309,
    0.08314083140831408,
    0.08315083150831508,
    0.08316083160831608,
    0.08317083170831709,
    0.08318083180831808,
    0.08319083190831908,
    0.08320083200832008,
    0.08321083210832109,
    0.08322083220832208,
    0.08323083230832308,
    0.08324083240832408,
    0.08325083250832509,
    0.08326083260832608,
    0.08327083270832708,
    0.08328083280832808,
    0.08329083290832909,
    0.08330083300833008,
    0.08331083310833108,
    0.08332083320833208,
    0.08333083330833309,
    0.08334083340833408,
    0.08335083350833508,
    0.08336083360833609,
    0.08337083370833709,
    0.08338083380833808,
    0.08339083390833908,
    0.08340083400834009,
    0.08341083410834108,
    0.08342083420834208,
    0.08343083430834308,
    0.08344083440834409,
    0.08345083450834508,
    0.08346083460834608,
    0.08347083470834708,
    0.08348083480834809,
    0.08349083490834908,
    0.08350083500835008,
    0.08351083510835108,
    0.08352083520835209,
    0.08353083530835308,
    0.08354083540835408,
    0.08355083550835508,
    0.08356083560835609,
    0.08357083570835708,
    0.08358083580835808,
    0.08359083590835908,
    0.08360083600836009,
    0.08361083610836108,
    0.08362083620836208,
    0.08363083630836309,
    0.08364083640836409,
    0.08365083650836508,
    0.08366083660836608,
    0.08367083670836709,
    0.08368083680836808,
    0.08369083690836908,
    0.08370083700837008,
    0.08371083710837109,
    0.08372083720837208,
    0.08373083730837308,
    0.08374083740837408,
    0.08375083750837509,
    0.08376083760837608,
    0.08377083770837708,
    0.08378083780837808,
    0.08379083790837909,
    0.08380083800838008,
    0.08381083810838108,
    0.08382083820838208,
    0.08383083830838309,
    0.08384083840838408,
    0.08385083850838508,
    0.08386083860838608,
    0.08387083870838709,
    0.08388083880838808,
    0.08389083890838908,
    0.08390083900839009,
    0.08391083910839109,
    0.08392083920839208,
    0.08393083930839308,
    0.08394083940839409,
    0.08395083950839509,
    0.08396083960839608,
    0.08397083970839708,
    0.08398083980839809,
    0.08399083990839908,
    0.08400084000840008,
    0.08401084010840108,
    0.08402084020840209,
    0.08403084030840308,
    0.08404084040840408,
    0.08405084050840508,
    0.08406084060840609,
    0.08407084070840708,
    0.08408084080840808,
    0.08409084090840908,
    0.08410084100841009,
    0.08411084110841108,
    0.08412084120841208,
    0.08413084130841308,
    0.08414084140841409,
    0.08415084150841508,
    0.08416084160841608,
    0.08417084170841709,
    0.08418084180841809,
    0.08419084190841908,
    0.08420084200842008,
    0.08421084210842109,
    0.08422084220842209,
    0.08423084230842308,
    0.08424084240842408,
    0.08425084250842509,
    0.08426084260842609,
    0.08427084270842708,
    0.08428084280842808,
    0.08429084290842909,
    0.08430084300843008,
    0.08431084310843108,
    0.08432084320843208,
    0.08433084330843309,
    0.08434084340843408,
    0.08435084350843508,
    0.08436084360843608,
    0.08437084370843709,
    0.08438084380843808,
    0.08439084390843908,
    0.08440084400844008,
    0.08441084410844109,
    0.08442084420844208,
    0.08443084430844308,
    0.08444084440844409,
    0.08445084450844509,
    0.08446084460844608,
    0.08447084470844708,
    0.08448084480844809,
    0.08449084490844909,
    0.08450084500845008,
    0.08451084510845108,
    0.08452084520845209,
    0.08453084530845309,
    0.08454084540845408,
    0.08455084550845508,
    0.08456084560845609,
    0.08457084570845708,
    0.08458084580845808,
    0.08459084590845908,
    0.08460084600846009,
    0.08461084610846108,
    0.08462084620846208,
    0.08463084630846308,
    0.08464084640846409,
    0.08465084650846508,
    0.08466084660846608,
    0.08467084670846708,
    0.08468084680846809,
    0.08469084690846908,
    0.08470084700847008,
    0.08471084710847109,
    0.08472084720847209,
    0.08473084730847308,
    0.08474084740847408,
    0.08475084750847509,
    0.08476084760847609,
    0.08477084770847708,
    0.08478084780847808,
    0.08479084790847909,
    0.08480084800848009,
    0.08481084810848108,
    0.08482084820848208,
    0.08483084830848309,
    0.08484084840848409,
    0.08485084850848508,
    0.08486084860848608,
    0.08487084870848709,
    0.08488084880848808,
    0.08489084890848908,
    0.08490084900849008,
    0.08491084910849109,
    0.08492084920849208,
    0.08493084930849308,
    0.08494084940849408,
    0.08495084950849509,
    0.08496084960849608,
    0.08497084970849708,
    0.08498084980849809,
    0.08499084990849909,
    0.08500085000850008,
    0.08501085010850108,
    0.08502085020850209,
    0.08503085030850309,
    0.08504085040850408,
    0.08505085050850508,
    0.08506085060850609,
    0.08507085070850709,
    0.08508085080850808,
    0.08509085090850908,
    0.08510085100851009,
    0.08511085110851109,
    0.08512085120851208,
    0.08513085130851308,
    0.08514085140851409,
    0.08515085150851508,
    0.08516085160851608,
    0.08517085170851708,
    0.08518085180851809,
    0.08519085190851908,
    0.08520085200852008,
    0.08521085210852108,
    0.08522085220852209,
    0.08523085230852308,
    0.08524085240852408,
    0.08525085250852509,
    0.08526085260852609,
    0.08527085270852708,
    0.08528085280852808,
    0.08529085290852909,
    0.08530085300853009,
    0.08531085310853108,
    0.08532085320853208,
    0.08533085330853309,
    0.08534085340853409,
    0.08535085350853508,
    0.08536085360853608,
    0.08537085370853709,
    0.08538085380853809,
    0.08539085390853908,
    0.08540085400854008,
    0.08541085410854109,
    0.08542085420854209,
    0.08543085430854308,
    0.08544085440854408,
    0.08545085450854509,
    0.08546085460854608,
    0.08547085470854708,
    0.08548085480854808,
    0.08549085490854909,
    0.08550085500855008,
    0.08551085510855108,
    0.08552085520855209,
    0.08553085530855309,
    0.08554085540855408,
    0.08555085550855508,
    0.08556085560855609,
    0.08557085570855709,
    0.08558085580855808,
    0.08559085590855908,
    0.08560085600856009,
    0.08561085610856109,
    0.08562085620856208,
    0.08563085630856308,
    0.08564085640856409,
    0.08565085650856509,
    0.08566085660856608,
    0.08567085670856708,
    0.08568085680856809,
    0.08569085690856909,
    0.08570085700857008,
    0.08571085710857108,
    0.08572085720857209,
    0.08573085730857308,
    0.08574085740857408,
    0.08575085750857508,
    0.08576085760857609,
    0.08577085770857708,
    0.08578085780857808,
    0.08579085790857909,
    0.08580085800858009,
    0.08581085810858108,
    0.08582085820858208,
    0.08583085830858309,
    0.08584085840858409,
    0.08585085850858508,
    0.08586085860858608,
    0.08587085870858709,
    0.08588085880858809,
    0.08589085890858908,
    0.08590085900859008,
    0.08591085910859109,
    0.08592085920859209,
    0.08593085930859308,
    0.08594085940859408,
    0.08595085950859509,
    0.08596085960859609,
    0.08597085970859708,
    0.08598085980859808,
    0.08599085990859909,
    0.08600086000860009,
    0.08601086010860108,
    0.08602086020860208,
    0.08603086030860309,
    0.08604086040860408,
    0.08605086050860508,
    0.08606086060860609,
    0.08607086070860709,
    0.08608086080860808,
    0.08609086090860908,
    0.08610086100861009,
    0.08611086110861109,
    0.08612086120861208,
    0.08613086130861308,
    0.08614086140861409,
    0.08615086150861509,
    0.08616086160861608,
    0.08617086170861708,
    0.08618086180861809,
    0.08619086190861909,
    0.08620086200862008,
    0.08621086210862108,
    0.08622086220862209,
    0.08623086230862309,
    0.08624086240862408,
    0.08625086250862508,
    0.08626086260862609,
    0.08627086270862709,
    0.08628086280862808,
    0.08629086290862908,
    0.08630086300863009,
    0.08631086310863108,
    0.08632086320863208,
    0.08633086330863309,
    0.08634086340863409,
    0.08635086350863508,
    0.08636086360863608,
    0.08637086370863709,
    0.08638086380863809,
    0.08639086390863908,
    0.08640086400864008,
    0.08641086410864109,
    0.08642086420864209,
    0.08643086430864308,
    0.08644086440864408,
    0.08645086450864509,
    0.08646086460864609,
    0.08647086470864708,
    0.08648086480864808,
    0.08649086490864909,
    0.08650086500865009,
    0.08651086510865108,
    0.08652086520865208,
    0.08653086530865309,
    0.08654086540865409,
    0.08655086550865508,
    0.08656086560865608,
    0.08657086570865709,
    0.08658086580865809,
    0.08659086590865908,
    0.08660086600866009,
    0.08661086610866109,
    0.08662086620866208,
    0.08663086630866308,
    0.08664086640866409,
    0.08665086650866509,
    0.08666086660866608,
    0.08667086670866708,
    0.08668086680866809,
    0.08669086690866909,
    0.08670086700867008,
    0.08671086710867108,
    0.08672086720867209,
    0.08673086730867309,
    0.08674086740867408,
    0.08675086750867508,
    0.08676086760867609,
    0.08677086770867709,
    0.08678086780867808,
    0.08679086790867908,
    0.08680086800868009,
    0.08681086810868109,
    0.08682086820868208,
    0.08683086830868308,
    0.08684086840868409,
    0.08685086850868509,
    0.08686086860868608,
    0.08687086870868709,
    0.08688086880868809,
    0.08689086890868909,
    0.08690086900869008,
    0.08691086910869109,
    0.08692086920869209,
    0.08693086930869308,
    0.08694086940869408,
    0.08695086950869509,
    0.08696086960869609,
    0.08697086970869708,
    0.08698086980869808,
    0.08699086990869909,
    0.08700087000870009,
    0.08701087010870108,
    0.08702087020870208,
    0.08703087030870309,
    0.08704087040870409,
    0.08705087050870508,
    0.08706087060870608,
    0.08707087070870709,
    0.08708087080870809,
    0.08709087090870908,
    0.08710087100871008,
    0.08711087110871109,
    0.08712087120871209,
    0.08713087130871308,
    0.08714087140871409,
    0.08715087150871509,
    0.08716087160871609,
    0.08717087170871708,
    0.08718087180871809,
    0.08719087190871909,
    0.08720087200872008,
    0.08721087210872108,
    0.08722087220872209,
    0.08723087230872309,
    0.08724087240872408,
    0.08725087250872508,
    0.08726087260872609,
    0.08727087270872709,
    0.08728087280872808,
    0.08729087290872908,
    0.08730087300873009,
    0.08731087310873109,
    0.08732087320873208,
    0.08733087330873308,
    0.08734087340873409,
    0.08735087350873509,
    0.08736087360873608,
    0.08737087370873708,
    0.08738087380873809,
    0.08739087390873909,
    0.08740087400874008,
    0.08741087410874109,
    0.08742087420874209,
    0.08743087430874309,
    0.08744087440874408,
    0.08745087450874509,
    0.08746087460874609,
    0.08747087470874709,
    0.08748087480874808,
    0.08749087490874909,
    0.08750087500875009,
    0.08751087510875108,
    0.08752087520875208,
    0.08753087530875309,
    0.08754087540875409,
    0.08755087550875508,
    0.08756087560875608,
    0.08757087570875709,
    0.08758087580875809,
    0.08759087590875908,
    0.08760087600876008,
    0.08761087610876109,
    0.08762087620876209,
    0.08763087630876308,
    0.08764087640876408,
    0.08765087650876509,
    0.08766087660876609,
    0.08767087670876708,
    0.08768087680876809,
    0.08769087690876909,
    0.08770087700877009,
    0.08771087710877108,
    0.08772087720877209,
    0.08773087730877309,
    0.08774087740877409,
    0.08775087750877508,
    0.08776087760877609,
    0.08777087770877709,
    0.08778087780877808,
    0.08779087790877908,
    0.08780087800878009,
    0.08781087810878109,
    0.08782087820878208,
    0.08783087830878308,
    0.08784087840878409,
    0.08785087850878509,
    0.08786087860878608,
    0.08787087870878708,
    0.08788087880878809,
    0.08789087890878909,
    0.08790087900879008,
    0.08791087910879108,
    0.08792087920879209,
    0.08793087930879309,
    0.08794087940879408,
    0.08795087950879509,
    0.08796087960879609,
    0.08797087970879709,
    0.08798087980879808,
    0.08799087990879909,
    0.08800088000880009,
    0.08801088010880109,
    0.08802088020880208,
    0.08803088030880309,
    0.08804088040880409,
    0.0880508805088051,
    0.08806088060880608,
    0.08807088070880709,
    0.08808088080880809,
    0.08809088090880908,
    0.08810088100881008,
    0.08811088110881109,
    0.08812088120881209,
    0.08813088130881308,
    0.08814088140881408,
    0.08815088150881509,
    0.08816088160881609,
    0.08817088170881708,
    0.08818088180881808,
    0.08819088190881909,
    0.08820088200882009,
    0.08821088210882108,
    0.08822088220882209,
    0.08823088230882309,
    0.08824088240882409,
    0.08825088250882508,
    0.08826088260882609,
    0.08827088270882709,
    0.08828088280882809,
    0.08829088290882908,
    0.08830088300883009,
    0.08831088310883109,
    0.0883208832088321,
    0.08833088330883308,
    0.08834088340883409,
    0.08835088350883509,
    0.08836088360883608,
    0.08837088370883708,
    0.08838088380883809,
    0.08839088390883909,
    0.08840088400884008,
    0.08841088410884108,
    0.08842088420884209,
    0.08843088430884309,
    0.08844088440884408,
    0.08845088450884508,
    0.08846088460884609,
    0.08847088470884709,
    0.08848088480884808,
    0.08849088490884909,
    0.08850088500885009,
    0.08851088510885109,
    0.08852088520885208,
    0.08853088530885309,
    0.08854088540885409,
    0.08855088550885509,
    0.08856088560885608,
    0.08857088570885709,
    0.08858088580885809,
    0.0885908859088591,
    0.08860088600886008,
    0.08861088610886109,
    0.08862088620886209,
    0.0886308863088631,
    0.08864088640886408,
    0.08865088650886509,
    0.08866088660886609,
    0.08867088670886708,
    0.08868088680886808,
    0.08869088690886909,
    0.08870088700887009,
    0.08871088710887108,
    0.08872088720887208,
    0.08873088730887309,
    0.08874088740887409,
    0.08875088750887508,
    0.08876088760887609,
    0.08877088770887709,
    0.08878088780887809,
    0.08879088790887908,
    0.08880088800888009,
    0.08881088810888109,
    0.08882088820888209,
    0.08883088830888308,
    0.08884088840888409,
    0.08885088850888509,
    0.0888608886088861,
    0.08887088870888708,
    0.08888088880888809,
    0.08889088890888909,
    0.0889008890088901,
    0.08891088910889108,
    0.08892088920889209,
    0.08893088930889309,
    0.0889408894088941,
    0.08895088950889508,
    0.08896088960889609,
    0.08897088970889709,
    0.08898088980889808,
    0.08899088990889908,
    0.08900089000890009,
    0.08901089010890109,
    0.08902089020890208,
    0.08903089030890309,
    0.08904089040890409,
    0.08905089050890509,
    0.08906089060890608,
    0.08907089070890709,
    0.08908089080890809,
    0.08909089090890909,
    0.08910089100891008,
    0.08911089110891109,
    0.08912089120891209,
    0.0891308913089131,
    0.08914089140891408,
    0.08915089150891509,
    0.08916089160891609,
    0.0891708917089171,
    0.08918089180891808,
    0.08919089190891909,
    0.08920089200892009,
    0.0892108921089211,
    0.08922089220892208,
    0.08923089230892309,
    0.08924089240892409,
    0.08925089250892508,
    0.08926089260892608,
    0.08927089270892709,
    0.08928089280892809,
    0.08929089290892908,
    0.08930089300893009,
    0.08931089310893109,
    0.08932089320893209,
    0.08933089330893308,
    0.08934089340893409,
    0.08935089350893509,
    0.08936089360893609,
    0.08937089370893708,
    0.08938089380893809,
    0.08939089390893909,
    0.0894008940089401,
    0.08941089410894108,
    0.08942089420894209,
    0.08943089430894309,
    0.0894408944089441,
    0.08945089450894508,
    0.08946089460894609,
    0.08947089470894709,
    0.0894808948089481,
    0.08949089490894908,
    0.08950089500895009,
    0.08951089510895109,
    0.0895208952089521,
    0.08953089530895308,
    0.08954089540895409,
    0.08955089550895509,
    0.08956089560895608,
    0.08957089570895709,
    0.08958089580895809,
    0.08959089590895909,
    0.08960089600896008,
    0.08961089610896109,
    0.08962089620896209,
    0.08963089630896309,
    0.08964089640896408,
    0.08965089650896509,
    0.08966089660896609,
    0.0896708967089671,
    0.08968089680896808,
    0.08969089690896909,
    0.08970089700897009,
    0.0897108971089711,
    0.08972089720897208,
    0.08973089730897309,
    0.08974089740897409,
    0.0897508975089751,
    0.08976089760897608,
    0.08977089770897709,
    0.08978089780897809,
    0.0897908979089791,
    0.08980089800898008,
    0.08981089810898109,
    0.08982089820898209,
    0.08983089830898308,
    0.08984089840898409,
    0.08985089850898509,
    0.08986089860898609,
    0.08987089870898708,
    0.08988089880898809,
    0.08989089890898909,
    0.08990089900899009,
    0.08991089910899108,
    0.08992089920899209,
    0.08993089930899309,
    0.0899408994089941,
    0.08995089950899508,
    0.08996089960899609,
    0.08997089970899709,
    0.0899808998089981,
    0.08999089990899908,
    0.09000090000900009,
    0.09001090010900109,
    0.0900209002090021,
    0.09003090030900308,
    0.09004090040900409,
    0.09005090050900509,
    0.0900609006090061,
    0.09007090070900708,
    0.09008090080900809,
    0.09009090090900909,
    0.0901009010090101,
    0.09011090110901109,
    0.09012090120901209,
    0.09013090130901309,
    0.09014090140901408,
    0.09015090150901509,
    0.09016090160901609,
    0.09017090170901709,
    0.09018090180901808,
    0.09019090190901909,
    0.09020090200902009,
    0.0902109021090211,
    0.09022090220902208,
    0.09023090230902309,
    0.09024090240902409,
    0.0902509025090251,
    0.09026090260902608,
    0.09027090270902709,
    0.09028090280902809,
    0.0902909029090291,
    0.09030090300903008,
    0.09031090310903109,
    0.09032090320903209,
    0.0903309033090331,
    0.09034090340903408,
    0.09035090350903509,
    0.09036090360903609,
    0.0903709037090371,
    0.09038090380903809,
    0.09039090390903909,
    0.09040090400904009,
    0.09041090410904108,
    0.09042090420904209,
    0.09043090430904309,
    0.09044090440904409,
    0.09045090450904508,
    0.09046090460904609,
    0.09047090470904709,
    0.0904809048090481,
    0.09049090490904908,
    0.09050090500905009,
    0.09051090510905109,
    0.0905209052090521,
    0.09053090530905308,
    0.09054090540905409,
    0.09055090550905509,
    0.0905609056090561,
    0.09057090570905708,
    0.09058090580905809,
    0.09059090590905909,
    0.0906009060090601,
    0.09061090610906108,
    0.09062090620906209,
    0.09063090630906309,
    0.0906409064090641,
    0.09065090650906509,
    0.09066090660906609,
    0.09067090670906709,
    0.0906809068090681,
    0.09069090690906909,
    0.09070090700907009,
    0.09071090710907109,
    0.09072090720907208,
    0.09073090730907309,
    0.09074090740907409,
    0.0907509075090751,
    0.09076090760907608,
    0.09077090770907709,
    0.09078090780907809,
    0.0907909079090791,
    0.09080090800908008,
    0.09081090810908109,
    0.09082090820908209,
    0.0908309083090831,
    0.09084090840908408,
    0.09085090850908509,
    0.09086090860908609,
    0.0908709087090871,
    0.09088090880908808,
    0.09089090890908909,
    0.09090090900909009,
    0.0909109091090911,
    0.09092090920909209,
    0.09093090930909309,
    0.09094090940909409,
    0.0909509095090951,
    0.09096090960909609,
    0.09097090970909709,
    0.09098090980909809,
    0.09099090990909908,
    0.09100091000910009,
    0.09101091010910109,
    0.0910209102091021,
    0.09103091030910308,
    0.09104091040910409,
    0.09105091050910509,
    0.0910609106091061,
    0.09107091070910708,
    0.09108091080910809,
    0.09109091090910909,
    0.0911009110091101,
    0.09111091110911108,
    0.09112091120911209,
    0.09113091130911309,
    0.0911409114091141,
    0.09115091150911508,
    0.09116091160911609,
    0.09117091170911709,
    0.0911809118091181,
    0.09119091190911909,
    0.09120091200912009,
    0.09121091210912109,
    0.0912209122091221,
    0.09123091230912309,
    0.09124091240912409,
    0.09125091250912509,
    0.0912609126091261,
    0.09127091270912709,
    0.09128091280912809,
    0.0912909129091291,
    0.09130091300913008,
    0.09131091310913109,
    0.09132091320913209,
    0.0913309133091331,
    0.09134091340913408,
    0.09135091350913509,
    0.09136091360913609,
    0.0913709137091371,
    0.09138091380913808,
    0.09139091390913909,
    0.09140091400914009,
    0.0914109141091411,
    0.09142091420914208,
    0.09143091430914309,
    0.09144091440914409,
    0.0914509145091451,
    0.09146091460914609,
    0.09147091470914709,
    0.09148091480914809,
    0.0914909149091491,
    0.09150091500915009,
    0.09151091510915109,
    0.09152091520915209,
    0.0915309153091531,
    0.09154091540915409,
    0.09155091550915509,
    0.0915609156091561,
    0.0915709157091571,
    0.09158091580915809,
    0.09159091590915909,
    0.0916009160091601,
    0.09161091610916108,
    0.09162091620916209,
    0.09163091630916309,
    0.0916409164091641,
    0.09165091650916508,
    0.09166091660916609,
    0.09167091670916709,
    0.0916809168091681,
    0.09169091690916908,
    0.09170091700917009,
    0.09171091710917109,
    0.0917209172091721,
    0.09173091730917309,
    0.09174091740917409,
    0.09175091750917509,
    0.0917609176091761,
    0.09177091770917709,
    0.09178091780917809,
    0.09179091790917909,
    0.0918009180091801,
    0.09181091810918109,
    0.09182091820918209,
    0.0918309183091831,
    0.0918409184091841,
    0.09185091850918509,
    0.09186091860918609,
    0.0918709187091871,
    0.09188091880918808,
    0.09189091890918909,
    0.09190091900919009,
    0.0919109191091911,
    0.09192091920919208,
    0.09193091930919309,
    0.09194091940919409,
    0.0919509195091951,
    0.09196091960919608,
    0.09197091970919709,
    0.09198091980919809,
    0.0919909199091991,
    0.09200092000920009,
    0.09201092010920109,
    0.09202092020920209,
    0.0920309203092031,
    0.09204092040920409,
    0.09205092050920509,
    0.09206092060920609,
    0.0920709207092071,
    0.09208092080920809,
    0.09209092090920909,
    0.0921009210092101,
    0.0921109211092111,
    0.09212092120921209,
    0.09213092130921309,
    0.0921409214092141,
    0.0921509215092151,
    0.09216092160921609,
    0.09217092170921709,
    0.0921809218092181,
    0.09219092190921908,
    0.09220092200922009,
    0.09221092210922109,
    0.0922209222092221,
    0.09223092230922308,
    0.09224092240922409,
    0.09225092250922509,
    0.0922609226092261,
    0.09227092270922709,
    0.09228092280922809,
    0.09229092290922909,
    0.0923009230092301,
    0.09231092310923109,
    0.09232092320923209,
    0.09233092330923309,
    0.0923409234092341,
    0.09235092350923509,
    0.09236092360923609,
    0.0923709237092371,
    0.0923809238092381,
    0.09239092390923909,
    0.09240092400924009,
    0.0924109241092411,
    0.0924209242092421,
    0.09243092430924309,
    0.09244092440924409,
    0.0924509245092451,
    0.09246092460924608,
    0.09247092470924709,
    0.09248092480924809,
    0.0924909249092491,
    0.09250092500925008,
    0.09251092510925109,
    0.09252092520925209,
    0.0925309253092531,
    0.09254092540925409,
    0.09255092550925509,
    0.09256092560925609,
    0.0925709257092571,
    0.09258092580925809,
    0.09259092590925909,
    0.09260092600926009,
    0.0926109261092611,
    0.09262092620926209,
    0.09263092630926309,
    0.0926409264092641,
    0.0926509265092651,
    0.09266092660926609,
    0.09267092670926709,
    0.0926809268092681,
    0.0926909269092691,
    0.09270092700927009,
    0.09271092710927109,
    0.0927209272092721,
    0.0927309273092731,
    0.09274092740927409,
    0.09275092750927509,
    0.0927609276092761,
    0.09277092770927708,
    0.09278092780927809,
    0.09279092790927909,
    0.0928009280092801,
    0.09281092810928109,
    0.09282092820928209,
    0.09283092830928309,
    0.0928409284092841,
    0.09285092850928509,
    0.09286092860928609,
    0.09287092870928709,
    0.0928809288092881,
    0.09289092890928909,
    0.09290092900929009,
    0.0929109291092911,
    0.0929209292092921,
    0.09293092930929309,
    0.09294092940929409,
    0.0929509295092951,
    0.0929609296092961,
    0.09297092970929709,
    0.09298092980929809,
    0.0929909299092991,
    0.0930009300093001,
    0.09301093010930109,
    0.09302093020930209,
    0.0930309303093031,
    0.09304093040930408,
    0.09305093050930509,
    0.09306093060930609,
    0.0930709307093071,
    0.09308093080930809,
    0.09309093090930909,
    0.09310093100931009,
    0.0931109311093111,
    0.09312093120931209,
    0.09313093130931309,
    0.09314093140931409,
    0.0931509315093151,
    0.09316093160931609,
    0.09317093170931709,
    0.0931809318093181,
    0.0931909319093191,
    0.09320093200932009,
    0.09321093210932109,
    0.0932209322093221,
    0.0932309323093231,
    0.09324093240932409,
    0.09325093250932509,
    0.0932609326093261,
    0.0932709327093271,
    0.09328093280932809,
    0.09329093290932909,
    0.0933009330093301,
    0.0933109331093311,
    0.09332093320933209,
    0.09333093330933309,
    0.0933409334093341,
    0.09335093350933509,
    0.09336093360933609,
    0.09337093370933709,
    0.0933809338093381,
    0.09339093390933909,
    0.09340093400934009,
    0.09341093410934109,
    0.0934209342093421,
    0.09343093430934309,
    0.09344093440934409,
    0.0934509345093451,
    0.0934609346093461,
    0.09347093470934709,
    0.09348093480934809,
    0.0934909349093491,
    0.0935009350093501,
    0.09351093510935109,
    0.09352093520935209,
    0.0935309353093531,
    0.0935409354093541,
    0.09355093550935509,
    0.09356093560935609,
    0.0935709357093571,
    0.0935809358093581,
    0.09359093590935909,
    0.09360093600936009,
    0.0936109361093611,
    0.0936209362093621,
    0.09363093630936309,
    0.09364093640936409,
    0.0936509365093651,
    0.09366093660936609,
    0.09367093670936709,
    0.09368093680936809,
    0.0936909369093691,
    0.09370093700937009,
    0.09371093710937109,
    0.0937209372093721,
    0.0937309373093731,
    0.09374093740937409,
    0.09375093750937509,
    0.0937609376093761,
    0.0937709377093771,
    0.09378093780937809,
    0.09379093790937909,
    0.0938009380093801,
    0.0938109381093811,
    0.09382093820938209,
    0.09383093830938309,
    0.0938409384093841,
    0.0938509385093851,
    0.09386093860938609,
    0.09387093870938709,
    0.0938809388093881,
    0.0938909389093891,
    0.09390093900939009,
    0.09391093910939109,
    0.0939209392093921,
    0.09393093930939309,
    0.09394093940939409,
    0.09395093950939509,
    0.0939609396093961,
    0.09397093970939709,
    0.09398093980939809,
    0.0939909399093991,
    0.0940009400094001,
    0.09401094010940109,
    0.09402094020940209,
    0.0940309403094031,
    0.0940409404094041,
    0.09405094050940509,
    0.09406094060940609,
    0.0940709407094071,
    0.0940809408094081,
    0.09409094090940909,
    0.09410094100941009,
    0.0941109411094111,
    0.0941209412094121,
    0.09413094130941309,
    0.09414094140941409,
    0.0941509415094151,
    0.0941609416094161,
    0.09417094170941709,
    0.09418094180941809,
    0.0941909419094191,
    0.0942009420094201,
    0.09421094210942109,
    0.09422094220942209,
    0.0942309423094231,
    0.09424094240942409,
    0.09425094250942509,
    0.0942609426094261,
    0.0942709427094271,
    0.09428094280942809,
    0.09429094290942909,
    0.0943009430094301,
    0.0943109431094311,
    0.09432094320943209,
    0.09433094330943309,
    0.0943409434094341,
    0.0943509435094351,
    0.09436094360943609,
    0.09437094370943709,
    0.0943809438094381,
    0.0943909439094391,
    0.09440094400944009,
    0.09441094410944109,
    0.0944209442094421,
    0.0944309443094431,
    0.09444094440944409,
    0.09445094450944509,
    0.0944609446094461,
    0.0944709447094471,
    0.09448094480944809,
    0.09449094490944909,
    0.0945009450094501,
    0.09451094510945109,
    0.09452094520945209,
    0.0945309453094531,
    0.0945409454094541,
    0.09455094550945509,
    0.09456094560945609,
    0.0945709457094571,
    0.0945809458094581,
    0.09459094590945909,
    0.09460094600946009,
    0.0946109461094611,
    0.0946209462094621,
    0.09463094630946309,
    0.09464094640946409,
    0.0946509465094651,
    0.0946609466094661,
    0.09467094670946709,
    0.09468094680946809,
    0.0946909469094691,
    0.0947009470094701,
    0.09471094710947109,
    0.09472094720947209,
    0.0947309473094731,
    0.0947409474094741,
    0.09475094750947509,
    0.09476094760947609,
    0.0947709477094771,
    0.0947809478094781,
    0.09479094790947909,
    0.0948009480094801,
    0.0948109481094811,
    0.09482094820948209,
    0.09483094830948309,
    0.0948409484094841,
    0.0948509485094851,
    0.09486094860948609,
    0.09487094870948709,
    0.0948809488094881,
    0.0948909489094891,
    0.09490094900949009,
    0.09491094910949109,
    0.0949209492094921,
    0.0949309493094931,
    0.09494094940949409,
    0.09495094950949509,
    0.0949609496094961,
    0.0949709497094971,
    0.09498094980949809,
    0.09499094990949909,
    0.0950009500095001,
    0.0950109501095011,
    0.09502095020950209,
    0.09503095030950309,
    0.0950409504095041,
    0.0950509505095051,
    0.09506095060950609,
    0.0950709507095071,
    0.0950809508095081,
    0.09509095090950909,
    0.09510095100951009,
    0.0951109511095111,
    0.0951209512095121,
    0.09513095130951309,
    0.09514095140951409,
    0.0951509515095151,
    0.0951609516095161,
    0.09517095170951709,
    0.09518095180951809,
    0.0951909519095191,
    0.0952009520095201,
    0.09521095210952109,
    0.09522095220952209,
    0.0952309523095231,
    0.0952409524095241,
    0.09525095250952509,
    0.09526095260952609,
    0.0952709527095271,
    0.0952809528095281,
    0.09529095290952909,
    0.09530095300953009,
    0.0953109531095311,
    0.0953209532095321,
    0.09533095330953309,
    0.0953409534095341,
    0.0953509535095351,
    0.0953609536095361,
    0.09537095370953709,
    0.0953809538095381,
    0.0953909539095391,
    0.09540095400954009,
    0.09541095410954109,
    0.0954209542095421,
    0.0954309543095431,
    0.09544095440954409,
    0.09545095450954509,
    0.0954609546095461,
    0.0954709547095471,
    0.09548095480954809,
    0.09549095490954909,
    0.0955009550095501,
    0.0955109551095511,
    0.09552095520955209,
    0.09553095530955309,
    0.0955409554095541,
    0.0955509555095551,
    0.09556095560955609,
    0.09557095570955709,
    0.0955809558095581,
    0.0955909559095591,
    0.09560095600956009,
    0.0956109561095611,
    0.0956209562095621,
    0.0956309563095631,
    0.09564095640956409,
    0.0956509565095651,
    0.0956609566095661,
    0.0956709567095671,
    0.09568095680956809,
    0.0956909569095691,
    0.0957009570095701,
    0.09571095710957109,
    0.09572095720957209,
    0.0957309573095731,
    0.0957409574095741,
    0.09575095750957509,
    0.09576095760957609,
    0.0957709577095771,
    0.0957809578095781,
    0.09579095790957909,
    0.09580095800958009,
    0.0958109581095811,
    0.0958209582095821,
    0.09583095830958309,
    0.09584095840958409,
    0.0958509585095851,
    0.0958609586095861,
    0.09587095870958709,
    0.0958809588095881,
    0.0958909589095891,
    0.0959009590095901,
    0.09591095910959109,
    0.0959209592095921,
    0.0959309593095931,
    0.0959409594095941,
    0.09595095950959509,
    0.0959609596095961,
    0.0959709597095971,
    0.09598095980959809,
    0.09599095990959909,
    0.0960009600096001,
    0.0960109601096011,
    0.09602096020960209,
    0.09603096030960309,
    0.0960409604096041,
    0.0960509605096051,
    0.09606096060960609,
    0.09607096070960709,
    0.0960809608096081,
    0.0960909609096091,
    0.09610096100961009,
    0.09611096110961109,
    0.0961209612096121,
    0.0961309613096131,
    0.09614096140961409,
    0.0961509615096151,
    0.0961609616096161,
    0.0961709617096171,
    0.09618096180961809,
    0.0961909619096191,
    0.0962009620096201,
    0.0962109621096211,
    0.09622096220962209,
    0.0962309623096231,
    0.0962409624096241,
    0.0962509625096251,
    0.09626096260962609,
    0.0962709627096271,
    0.0962809628096281,
    0.09629096290962909,
    0.09630096300963009,
    0.0963109631096311,
    0.0963209632096321,
    0.09633096330963309,
    0.09634096340963409,
    0.0963509635096351,
    0.0963609636096361,
    0.09637096370963709,
    0.09638096380963809,
    0.0963909639096391,
    0.0964009640096401,
    0.09641096410964109,
    0.0964209642096421,
    0.0964309643096431,
    0.0964409644096441,
    0.09645096450964509,
    0.0964609646096461,
    0.0964709647096471,
    0.0964809648096481,
    0.09649096490964909,
    0.0965009650096501,
    0.0965109651096511,
    0.0965209652096521,
    0.09653096530965309,
    0.0965409654096541,
    0.0965509655096551,
    0.09656096560965609,
    0.09657096570965709,
    0.0965809658096581,
    0.0965909659096591,
    0.09660096600966009,
    0.09661096610966109,
    0.0966209662096621,
    0.0966309663096631,
    0.09664096640966409,
    0.09665096650966509,
    0.0966609666096661,
    0.0966709667096671,
    0.09668096680966809,
    0.0966909669096691,
    0.0967009670096701,
    0.0967109671096711,
    0.09672096720967209,
    0.0967309673096731,
    0.0967409674096741,
    0.0967509675096751,
    0.09676096760967609,
    0.0967709677096771,
    0.0967809678096781,
    0.0967909679096791,
    0.09680096800968009,
    0.0968109681096811,
    0.0968209682096821,
    0.0968309683096831,
    0.09684096840968409,
    0.0968509685096851,
    0.0968609686096861,
    0.09687096870968709,
    0.09688096880968809,
    0.0968909689096891,
    0.0969009690096901,
    0.09691096910969109,
    0.09692096920969209,
    0.0969309693096931,
    0.0969409694096941,
    0.09695096950969509,
    0.0969609696096961,
    0.0969709697096971,
    0.0969809698096981,
    0.09699096990969909,
    0.0970009700097001,
    0.0970109701097011,
    0.0970209702097021,
    0.09703097030970309,
    0.0970409704097041,
    0.0970509705097051,
    0.0970609706097061,
    0.09707097070970709,
    0.0970809708097081,
    0.0970909709097091,
    0.0971009710097101,
    0.09711097110971109,
    0.0971209712097121,
    0.0971309713097131,
    0.09714097140971409,
    0.09715097150971509,
    0.0971609716097161,
    0.0971709717097171,
    0.09718097180971809,
    0.09719097190971909,
    0.0972009720097201,
    0.0972109721097211,
    0.09722097220972209,
    0.0972309723097231,
    0.0972409724097241,
    0.0972509725097251,
    0.09726097260972609,
    0.0972709727097271,
    0.0972809728097281,
    0.0972909729097291,
    0.09730097300973009,
    0.0973109731097311,
    0.0973209732097321,
    0.0973309733097331,
    0.09734097340973409,
    0.0973509735097351,
    0.0973609736097361,
    0.0973709737097371,
    0.09738097380973809,
    0.0973909739097391,
    0.0974009740097401,
    0.0974109741097411,
    0.09742097420974209,
    0.0974309743097431,
    0.0974409744097441,
    0.09745097450974509,
    0.09746097460974609,
    0.0974709747097471,
    0.0974809748097481,
    0.09749097490974909,
    0.0975009750097501,
    0.0975109751097511,
    0.0975209752097521,
    0.09753097530975309,
    0.0975409754097541,
    0.0975509755097551,
    0.0975609756097561,
    0.09757097570975709,
    0.0975809758097581,
    0.0975909759097591,
    0.0976009760097601,
    0.09761097610976109,
    0.0976209762097621,
    0.0976309763097631,
    0.0976409764097641,
    0.09765097650976509,
    0.0976609766097661,
    0.0976709767097671,
    0.0976809768097681,
    0.09769097690976909,
    0.0977009770097701,
    0.0977109771097711,
    0.09772097720977209,
    0.09773097730977309,
    0.0977409774097741,
    0.0977509775097751,
    0.09776097760977609,
    0.0977709777097771,
    0.0977809778097781,
    0.0977909779097791,
    0.09780097800978009,
    0.0978109781097811,
    0.0978209782097821,
    0.0978309783097831,
    0.09784097840978409,
    0.0978509785097851,
    0.0978609786097861,
    0.0978709787097871,
    0.09788097880978809,
    0.0978909789097891,
    0.0979009790097901,
    0.0979109791097911,
    0.09792097920979209,
    0.0979309793097931,
    0.0979409794097941,
    0.0979509795097951,
    0.09796097960979609,
    0.0979709797097971,
    0.0979809798097981,
    0.0979909799097991,
    0.09800098000980009,
    0.0980109801098011,
    0.0980209802098021,
    0.09803098030980309,
    0.0980409804098041,
    0.0980509805098051,
    0.0980609806098061,
    0.09807098070980709,
    0.0980809808098081,
    0.0980909809098091,
    0.0981009810098101,
    0.09811098110981109,
    0.0981209812098121,
    0.0981309813098131,
    0.0981409814098141,
    0.09815098150981509,
    0.0981609816098161,
    0.0981709817098171,
    0.0981809818098181,
    0.09819098190981909,
    0.0982009820098201,
    0.0982109821098211,
    0.0982209822098221,
    0.09823098230982309,
    0.0982409824098241,
    0.0982509825098251,
    0.0982609826098261,
    0.09827098270982709,
    0.0982809828098281,
    0.0982909829098291,
    0.0983009830098301,
    0.0983109831098311,
    0.0983209832098321,
    0.0983309833098331,
    0.09834098340983409,
    0.0983509835098351,
    0.0983609836098361,
    0.0983709837098371,
    0.09838098380983809,
    0.0983909839098391,
    0.0984009840098401,
    0.0984109841098411,
    0.09842098420984209,
    0.0984309843098431,
    0.0984409844098441,
    0.0984509845098451,
    0.09846098460984609,
    0.0984709847098471,
    0.0984809848098481,
    0.0984909849098491,
    0.09850098500985009,
    0.0985109851098511,
    0.0985209852098521,
    0.0985309853098531,
    0.09854098540985409,
    0.0985509855098551,
    0.0985609856098561,
    0.0985709857098571,
    0.0985809858098581,
    0.0985909859098591,
    0.0986009860098601,
    0.09861098610986109,
    0.0986209862098621,
    0.0986309863098631,
    0.0986409864098641,
    0.09865098650986509,
    0.0986609866098661,
    0.0986709867098671,
    0.0986809868098681,
    0.09869098690986909,
    0.0987009870098701,
    0.0987109871098711,
    0.0987209872098721,
    0.09873098730987309,
    0.0987409874098741,
    0.0987509875098751,
    0.0987609876098761,
    0.09877098770987709,
    0.0987809878098781,
    0.0987909879098791,
    0.0988009880098801,
    0.09881098810988109,
    0.0988209882098821,
    0.0988309883098831,
    0.0988409884098841,
    0.0988509885098851,
    0.0988609886098861,
    0.0988709887098871,
    0.0988809888098881,
    0.0988909889098891,
    0.0989009890098901,
    0.0989109891098911,
    0.09892098920989209,
    0.0989309893098931,
    0.0989409894098941,
    0.0989509895098951,
    0.09896098960989609,
    0.0989709897098971,
    0.0989809898098981,
    0.0989909899098991,
    0.09900099000990009,
    0.0990109901099011,
    0.0990209902099021,
    0.0990309903099031,
    0.09904099040990409,
    0.0990509905099051,
    0.0990609906099061,
    0.0990709907099071,
    0.09908099080990809,
    0.0990909909099091,
    0.0991009910099101,
    0.0991109911099111,
    0.0991209912099121,
    0.0991309913099131,
    0.0991409914099141,
    0.0991509915099151,
    0.0991609916099161,
    0.0991709917099171,
    0.0991809918099181,
    0.09919099190991909,
    0.0992009920099201,
    0.0992109921099211,
    0.0992209922099221,
    0.09923099230992309,
    0.0992409924099241,
    0.0992509925099251,
    0.0992609926099261,
    0.09927099270992709,
    0.0992809928099281,
    0.0992909929099291,
    0.0993009930099301,
    0.09931099310993109,
    0.0993209932099321,
    0.0993309933099331,
    0.0993409934099341,
    0.09935099350993509,
    0.0993609936099361,
    0.0993709937099371,
    0.0993809938099381,
    0.0993909939099391,
    0.0994009940099401,
    0.0994109941099411,
    0.0994209942099421,
    0.0994309943099431,
    0.0994409944099441,
    0.0994509945099451,
    0.0994609946099461,
    0.0994709947099471,
    0.0994809948099481,
    0.0994909949099491,
    0.09950099500995009,
    0.0995109951099511,
    0.0995209952099521,
    0.0995309953099531,
    0.09954099540995409,
    0.0995509955099551,
    0.0995609956099561,
    0.0995709957099571,
    0.09958099580995809,
    0.0995909959099591,
    0.0996009960099601,
    0.0996109961099611,
    0.09962099620996209,
    0.0996309963099631,
    0.0996409964099641,
    0.0996509965099651,
    0.0996609966099661,
    0.0996709967099671,
    0.0996809968099681,
    0.0996909969099691,
    0.0997009970099701,
    0.0997109971099711,
    0.0997209972099721,
    0.0997309973099731,
    0.0997409974099741,
    0.0997509975099751,
    0.0997609976099761,
    0.09977099770997709,
    0.0997809978099781,
    0.0997909979099791,
    0.0998009980099801,
    0.09981099810998109,
    0.0998209982099821,
    0.0998309983099831,
    0.0998409984099841,
    0.09985099850998509,
    0.0998609986099861,
    0.0998709987099871,
    0.0998809988099881,
    0.09989099890998909,
    0.0999009990099901,
    0.0999109991099911,
    0.0999209992099921,
    0.0999309993099931,
    0.0999409994099941,
    0.0999509995099951,
    0.0999609996099961,
    0.0999709997099971,
    0.0999809998099981,
    0.0999909999099991,
    0.1000010000100001,
    0.1000110001100011,
    0.1000210002100021,
    0.1000310003100031,
    0.1000410004100041,
    0.1000510005100051,
    0.1000610006100061,
    0.1000710007100071,
    0.10008100081000809,
    0.1000910009100091,
    0.1001010010100101,
    0.1001110011100111,
    0.10012100121001209,
    0.1001310013100131,
    0.1001410014100141,
    0.1001510015100151,
    0.10016100161001609,
    0.1001710017100171,
    0.1001810018100181,
    0.1001910019100191,
    0.1002010020100201,
    0.1002110021100211,
    0.1002210022100221,
    0.1002310023100231,
    0.1002410024100241,
    0.1002510025100251,
    0.1002610026100261,
    0.1002710027100271,
    0.1002810028100281,
    0.1002910029100291,
    0.1003010030100301,
    0.1003110031100311,
    0.1003210032100321,
    0.1003310033100331,
    0.1003410034100341,
    0.1003510035100351,
    0.1003610036100361,
    0.1003710037100371,
    0.1003810038100381,
    0.10039100391003909,
    0.1004010040100401,
    0.1004110041100411,
    0.1004210042100421,
    0.10043100431004309,
    0.1004410044100441,
    0.1004510045100451,
    0.1004610046100461,
    0.1004710047100471,
    0.1004810048100481,
    0.1004910049100491,
    0.1005010050100501,
    0.1005110051100511,
    0.1005210052100521,
    0.1005310053100531,
    0.1005410054100541,
    0.1005510055100551,
    0.1005610056100561,
    0.1005710057100571,
    0.1005810058100581,
    0.1005910059100591,
    0.1006010060100601,
    0.1006110061100611,
    0.1006210062100621,
    0.1006310063100631,
    0.1006410064100641,
    0.1006510065100651,
    0.10066100661006609,
    0.1006710067100671,
    0.1006810068100681,
    0.1006910069100691,
    0.10070100701007009,
    0.1007110071100711,
    0.1007210072100721,
    0.1007310073100731,
    0.1007410074100741,
    0.1007510075100751,
    0.1007610076100761,
    0.1007710077100771,
    0.1007810078100781,
    0.1007910079100791,
    0.1008010080100801,
    0.1008110081100811,
    0.1008210082100821,
    0.1008310083100831,
    0.1008410084100841,
    0.1008510085100851,
    0.1008610086100861,
    0.1008710087100871,
    0.1008810088100881,
    0.1008910089100891,
    0.1009010090100901,
    0.1009110091100911,
    0.1009210092100921,
    0.1009310093100931,
    0.1009410094100941,
    0.1009510095100951,
    0.1009610096100961,
    0.10097100971009709,
    0.1009810098100981,
    0.1009910099100991,
    0.1010010100101001,
    0.1010110101101011,
    0.1010210102101021,
    0.1010310103101031,
    0.1010410104101041,
    0.1010510105101051,
    0.1010610106101061,
    0.1010710107101071,
    0.1010810108101081,
    0.1010910109101091,
    0.1011010110101101,
    0.1011110111101111,
    0.1011210112101121,
    0.1011310113101131,
    0.1011410114101141,
    0.1011510115101151,
    0.1011610116101161,
    0.1011710117101171,
    0.1011810118101181,
    0.1011910119101191,
    0.1012010120101201,
    0.1012110121101211,
    0.1012210122101221,
    0.1012310123101231,
    0.10124101241012409,
    0.1012510125101251,
    0.1012610126101261,
    0.1012710127101271,
    0.1012810128101281,
    0.1012910129101291,
    0.1013010130101301,
    0.1013110131101311,
    0.1013210132101321,
    0.1013310133101331,
    0.1013410134101341,
    0.1013510135101351,
    0.1013610136101361,
    0.1013710137101371,
    0.1013810138101381,
    0.1013910139101391,
    0.1014010140101401,
    0.1014110141101411,
    0.1014210142101421,
    0.1014310143101431,
    0.1014410144101441,
    0.1014510145101451,
    0.1014610146101461,
    0.1014710147101471,
    0.1014810148101481,
    0.1014910149101491,
    0.1015010150101501,
    0.1015110151101511,
    0.1015210152101521,
    0.1015310153101531,
    0.1015410154101541,
    0.1015510155101551,
    0.1015610156101561,
    0.1015710157101571,
    0.1015810158101581,
    0.1015910159101591,
    0.1016010160101601,
    0.1016110161101611,
    0.1016210162101621,
    0.1016310163101631,
    0.1016410164101641,
    0.1016510165101651,
    0.1016610166101661,
    0.1016710167101671,
    0.1016810168101681,
    0.1016910169101691,
    0.1017010170101701,
    0.1017110171101711,
    0.1017210172101721,
    0.1017310173101731,
    0.1017410174101741,
    0.1017510175101751,
    0.1017610176101761,
    0.1017710177101771,
    0.1017810178101781,
    0.1017910179101791,
    0.1018010180101801,
    0.1018110181101811,
    0.1018210182101821,
    0.1018310183101831,
    0.1018410184101841,
    0.1018510185101851,
    0.1018610186101861,
    0.1018710187101871,
    0.1018810188101881,
    0.1018910189101891,
    0.1019010190101901,
    0.1019110191101911,
    0.1019210192101921,
    0.1019310193101931,
    0.1019410194101941,
    0.1019510195101951,
    0.1019610196101961,
    0.1019710197101971,
    0.1019810198101981,
    0.1019910199101991,
    0.1020010200102001,
    0.1020110201102011,
    0.1020210202102021,
    0.1020310203102031,
    0.1020410204102041,
    0.1020510205102051,
    0.1020610206102061,
    0.1020710207102071,
    0.1020810208102081,
    0.10209102091020911,
    0.1021010210102101,
    0.1021110211102111,
    0.1021210212102121,
    0.1021310213102131,
    0.1021410214102141,
    0.1021510215102151,
    0.1021610216102161,
    0.1021710217102171,
    0.1021810218102181,
    0.1021910219102191,
    0.1022010220102201,
    0.1022110221102211,
    0.1022210222102221,
    0.1022310223102231,
    0.1022410224102241,
    0.1022510225102251,
    0.1022610226102261,
    0.1022710227102271,
    0.1022810228102281,
    0.1022910229102291,
    0.1023010230102301,
    0.1023110231102311,
    0.1023210232102321,
    0.1023310233102331,
    0.1023410234102341,
    0.1023510235102351,
    0.10236102361023611,
    0.1023710237102371,
    0.1023810238102381,
    0.1023910239102391,
    0.1024010240102401,
    0.1024110241102411,
    0.1024210242102421,
    0.1024310243102431,
    0.1024410244102441,
    0.1024510245102451,
    0.1024610246102461,
    0.1024710247102471,
    0.1024810248102481,
    0.1024910249102491,
    0.1025010250102501,
    0.1025110251102511,
    0.1025210252102521,
    0.1025310253102531,
    0.1025410254102541,
    0.1025510255102551,
    0.1025610256102561,
    0.1025710257102571,
    0.1025810258102581,
    0.1025910259102591,
    0.1026010260102601,
    0.1026110261102611,
    0.1026210262102621,
    0.10263102631026311,
    0.1026410264102641,
    0.1026510265102651,
    0.1026610266102661,
    0.10267102671026711,
    0.1026810268102681,
    0.1026910269102691,
    0.1027010270102701,
    0.1027110271102711,
    0.1027210272102721,
    0.1027310273102731,
    0.1027410274102741,
    0.1027510275102751,
    0.1027610276102761,
    0.1027710277102771,
    0.1027810278102781,
    0.1027910279102791,
    0.1028010280102801,
    0.1028110281102811,
    0.1028210282102821,
    0.1028310283102831,
    0.1028410284102841,
    0.1028510285102851,
    0.1028610286102861,
    0.1028710287102871,
    0.1028810288102881,
    0.1028910289102891,
    0.10290102901029011,
    0.1029110291102911,
    0.1029210292102921,
    0.1029310293102931,
    0.10294102941029411,
    0.1029510295102951,
    0.1029610296102961,
    0.1029710297102971,
    0.10298102981029811,
    0.1029910299102991,
    0.1030010300103001,
    0.1030110301103011,
    0.1030210302103021,
    0.1030310303103031,
    0.1030410304103041,
    0.1030510305103051,
    0.1030610306103061,
    0.1030710307103071,
    0.1030810308103081,
    0.1030910309103091,
    0.1031010310103101,
    0.1031110311103111,
    0.1031210312103121,
    0.1031310313103131,
    0.1031410314103141,
    0.1031510315103151,
    0.1031610316103161,
    0.10317103171031711,
    0.1031810318103181,
    0.1031910319103191,
    0.1032010320103201,
    0.10321103211032111,
    0.1032210322103221,
    0.1032310323103231,
    0.1032410324103241,
    0.10325103251032511,
    0.1032610326103261,
    0.1032710327103271,
    0.1032810328103281,
    0.1032910329103291,
    0.1033010330103301,
    0.1033110331103311,
    0.1033210332103321,
    0.1033310333103331,
    0.1033410334103341,
    0.1033510335103351,
    0.1033610336103361,
    0.1033710337103371,
    0.1033810338103381,
    0.1033910339103391,
    0.1034010340103401,
    0.1034110341103411,
    0.1034210342103421,
    0.1034310343103431,
    0.10344103441034411,
    0.1034510345103451,
    0.1034610346103461,
    0.1034710347103471,
    0.10348103481034811,
    0.1034910349103491,
    0.1035010350103501,
    0.1035110351103511,
    0.10352103521035211,
    0.1035310353103531,
    0.1035410354103541,
    0.1035510355103551,
    0.10356103561035611,
    0.1035710357103571,
    0.1035810358103581,
    0.1035910359103591,
    0.1036010360103601,
    0.1036110361103611,
    0.1036210362103621,
    0.1036310363103631,
    0.1036410364103641,
    0.1036510365103651,
    0.1036610366103661,
    0.1036710367103671,
    0.1036810368103681,
    0.1036910369103691,
    0.1037010370103701,
    0.10371103711037111,
    0.1037210372103721,
    0.1037310373103731,
    0.1037410374103741,
    0.10375103751037511,
    0.1037610376103761,
    0.1037710377103771,
    0.1037810378103781,
    0.10379103791037911,
    0.1038010380103801,
    0.1038110381103811,
    0.1038210382103821,
    0.10383103831038311,
    0.1038410384103841,
    0.1038510385103851,
    0.1038610386103861,
    0.1038710387103871,
    0.1038810388103881,
    0.1038910389103891,
    0.1039010390103901,
    0.1039110391103911,
    0.1039210392103921,
    0.1039310393103931,
    0.1039410394103941,
    0.1039510395103951,
    0.1039610396103961,
    0.1039710397103971,
    0.10398103981039811,
    0.1039910399103991,
    0.1040010400104001,
    0.1040110401104011,
    0.10402104021040211,
    0.1040310403104031,
    0.1040410404104041,
    0.1040510405104051,
    0.10406104061040611,
    0.1040710407104071,
    0.1040810408104081,
    0.1040910409104091,
    0.10410104101041011,
    0.1041110411104111,
    0.1041210412104121,
    0.1041310413104131,
    0.10414104141041411,
    0.1041510415104151,
    0.1041610416104161,
    0.1041710417104171,
    0.1041810418104181,
    0.1041910419104191,
    0.1042010420104201,
    0.1042110421104211,
    0.1042210422104221,
    0.1042310423104231,
    0.1042410424104241,
    0.10425104251042511,
    0.1042610426104261,
    0.1042710427104271,
    0.1042810428104281,
    0.10429104291042911,
    0.1043010430104301,
    0.1043110431104311,
    0.1043210432104321,
    0.10433104331043311,
    0.1043410434104341,
    0.1043510435104351,
    0.1043610436104361,
    0.10437104371043711,
    0.1043810438104381,
    0.1043910439104391,
    0.1044010440104401,
    0.10441104411044111,
    0.1044210442104421,
    0.1044310443104431,
    0.1044410444104441,
    0.1044510445104451,
    0.1044610446104461,
    0.1044710447104471,
    0.1044810448104481,
    0.1044910449104491,
    0.1045010450104501,
    0.1045110451104511,
    0.10452104521045211,
    0.1045310453104531,
    0.1045410454104541,
    0.1045510455104551,
    0.10456104561045611,
    0.1045710457104571,
    0.1045810458104581,
    0.1045910459104591,
    0.10460104601046011,
    0.1046110461104611,
    0.1046210462104621,
    0.1046310463104631,
    0.10464104641046411,
    0.1046510465104651,
    0.1046610466104661,
    0.1046710467104671,
    0.10468104681046811,
    0.1046910469104691,
    0.1047010470104701,
    0.1047110471104711,
    0.10472104721047211,
    0.1047310473104731,
    0.1047410474104741,
    0.1047510475104751,
    0.1047610476104761,
    0.1047710477104771,
    0.1047810478104781,
    0.10479104791047911,
    0.1048010480104801,
    0.1048110481104811,
    0.1048210482104821,
    0.10483104831048311,
    0.1048410484104841,
    0.1048510485104851,
    0.1048610486104861,
    0.10487104871048711,
    0.1048810488104881,
    0.1048910489104891,
    0.1049010490104901,
    0.10491104911049111,
    0.1049210492104921,
    0.1049310493104931,
    0.1049410494104941,
    0.10495104951049511,
    0.1049610496104961,
    0.1049710497104971,
    0.1049810498104981,
    0.10499104991049911,
    0.1050010500105001,
    0.1050110501105011,
    0.1050210502105021,
    0.10503105031050311,
    0.1050410504105041,
    0.1050510505105051,
    0.10506105061050611,
    0.1050710507105071,
    0.1050810508105081,
    0.1050910509105091,
    0.10510105101051011,
    0.1051110511105111,
    0.1051210512105121,
    0.1051310513105131,
    0.10514105141051411,
    0.1051510515105151,
    0.1051610516105161,
    0.1051710517105171,
    0.10518105181051811,
    0.1051910519105191,
    0.1052010520105201,
    0.1052110521105211,
    0.10522105221052211,
    0.1052310523105231,
    0.1052410524105241,
    0.1052510525105251,
    0.10526105261052611,
    0.1052710527105271,
    0.1052810528105281,
    0.1052910529105291,
    0.10530105301053011,
    0.1053110531105311,
    0.1053210532105321,
    0.10533105331053311,
    0.1053410534105341,
    0.1053510535105351,
    0.1053610536105361,
    0.10537105371053711,
    0.1053810538105381,
    0.1053910539105391,
    0.1054010540105401,
    0.10541105411054111,
    0.1054210542105421,
    0.1054310543105431,
    0.1054410544105441,
    0.10545105451054511,
    0.1054610546105461,
    0.1054710547105471,
    0.1054810548105481,
    0.10549105491054911,
    0.1055010550105501,
    0.1055110551105511,
    0.1055210552105521,
    0.10553105531055311,
    0.1055410554105541,
    0.1055510555105551,
    0.1055610556105561,
    0.10557105571055711,
    0.1055810558105581,
    0.1055910559105591,
    0.10560105601056011,
    0.10561105611056111,
    0.1056210562105621,
    0.1056310563105631,
    0.10564105641056411,
    0.1056510565105651,
    0.1056610566105661,
    0.1056710567105671,
    0.10568105681056811,
    0.1056910569105691,
    0.1057010570105701,
    0.1057110571105711,
    0.10572105721057211,
    0.1057310573105731,
    0.1057410574105741,
    0.1057510575105751,
    0.10576105761057611,
    0.1057710577105771,
    0.1057810578105781,
    0.1057910579105791,
    0.10580105801058011,
    0.1058110581105811,
    0.1058210582105821,
    0.1058310583105831,
    0.10584105841058411,
    0.1058510585105851,
    0.1058610586105861,
    0.10587105871058711,
    0.10588105881058811,
    0.1058910589105891,
    0.1059010590105901,
    0.10591105911059111,
    0.1059210592105921,
    0.1059310593105931,
    0.1059410594105941,
    0.10595105951059511,
    0.1059610596105961,
    0.1059710597105971,
    0.1059810598105981,
    0.10599105991059911,
    0.1060010600106001,
    0.1060110601106011,
    0.1060210602106021,
    0.10603106031060311,
    0.1060410604106041,
    0.1060510605106051,
    0.1060610606106061,
    0.10607106071060711,
    0.1060810608106081,
    0.1060910609106091,
    0.1061010610106101,
    0.10611106111061111,
    0.1061210612106121,
    0.1061310613106131,
    0.10614106141061411,
    0.10615106151061511,
    0.1061610616106161,
    0.1061710617106171,
    0.10618106181061811,
    0.10619106191061911,
    0.1062010620106201,
    0.1062110621106211,
    0.10622106221062211,
    0.1062310623106231,
    0.1062410624106241,
    0.1062510625106251,
    0.10626106261062611,
    0.1062710627106271,
    0.1062810628106281,
    0.1062910629106291,
    0.10630106301063011,
    0.1063110631106311,
    0.1063210632106321,
    0.1063310633106331,
    0.10634106341063411,
    0.1063510635106351,
    0.1063610636106361,
    0.1063710637106371,
    0.10638106381063811,
    0.1063910639106391,
    0.1064010640106401,
    0.10641106411064111,
    0.10642106421064211,
    0.1064310643106431,
    0.1064410644106441,
    0.10645106451064511,
    0.10646106461064611,
    0.1064710647106471,
    0.1064810648106481,
    0.10649106491064911,
    0.1065010650106501,
    0.1065110651106511,
    0.1065210652106521,
    0.10653106531065311,
    0.1065410654106541,
    0.1065510655106551,
    0.1065610656106561,
    0.10657106571065711,
    0.1065810658106581,
    0.1065910659106591,
    0.1066010660106601,
    0.10661106611066111,
    0.1066210662106621,
    0.1066310663106631,
    0.1066410664106641,
    0.10665106651066511,
    0.1066610666106661,
    0.1066710667106671,
    0.10668106681066811,
    0.10669106691066911,
    0.1067010670106701,
    0.1067110671106711,
    0.10672106721067211,
    0.10673106731067311,
    0.1067410674106741,
    0.1067510675106751,
    0.10676106761067611,
    0.10677106771067711,
    0.1067810678106781,
    0.1067910679106791,
    0.10680106801068011,
    0.1068110681106811,
    0.1068210682106821,
    0.1068310683106831,
    0.10684106841068411,
    0.1068510685106851,
    0.1068610686106861,
    0.1068710687106871,
    0.10688106881068811,
    0.1068910689106891,
    0.1069010690106901,
    0.1069110691106911,
    0.10692106921069211,
    0.1069310693106931,
    0.1069410694106941,
    0.10695106951069511,
    0.10696106961069611,
    0.1069710697106971,
    0.1069810698106981,
    0.10699106991069911,
    0.10700107001070011,
    0.1070110701107011,
    0.1070210702107021,
    0.10703107031070311,
    0.10704107041070411,
    0.1070510705107051,
    0.1070610706107061,
    0.10707107071070711,
    0.10708107081070811,
    0.1070910709107091,
    0.1071010710107101,
    0.10711107111071111,
    0.1071210712107121,
    0.1071310713107131,
    0.1071410714107141,
    0.10715107151071511,
    0.1071610716107161,
    0.1071710717107171,
    0.1071810718107181,
    0.10719107191071911,
    0.1072010720107201,
    0.1072110721107211,
    0.10722107221072211,
    0.10723107231072311,
    0.1072410724107241,
    0.1072510725107251,
    0.10726107261072611,
    0.10727107271072711,
    0.1072810728107281,
    0.1072910729107291,
    0.10730107301073011,
    0.10731107311073111,
    0.1073210732107321,
    0.1073310733107331,
    0.10734107341073411,
    0.10735107351073511,
    0.1073610736107361,
    0.1073710737107371,
    0.10738107381073811,
    0.1073910739107391,
    0.1074010740107401,
    0.1074110741107411,
    0.10742107421074211,
    0.1074310743107431,
    0.1074410744107441,
    0.1074510745107451,
    0.10746107461074611,
    0.1074710747107471,
    0.1074810748107481,
    0.10749107491074911,
    0.10750107501075011,
    0.1075110751107511,
    0.1075210752107521,
    0.10753107531075311,
    0.10754107541075411,
    0.1075510755107551,
    0.1075610756107561,
    0.10757107571075711,
    0.10758107581075811,
    0.1075910759107591,
    0.1076010760107601,
    0.10761107611076111,
    0.10762107621076211,
    0.1076310763107631,
    0.1076410764107641,
    0.10765107651076511,
    0.10766107661076611,
    0.1076710767107671,
    0.1076810768107681,
    0.10769107691076911,
    0.1077010770107701,
    0.1077110771107711,
    0.1077210772107721,
    0.10773107731077311,
    0.1077410774107741,
    0.1077510775107751,
    0.10776107761077611,
    0.10777107771077711,
    0.1077810778107781,
    0.1077910779107791,
    0.10780107801078011,
    0.10781107811078111,
    0.1078210782107821,
    0.1078310783107831,
    0.10784107841078411,
    0.10785107851078511,
    0.1078610786107861,
    0.1078710787107871,
    0.10788107881078811,
    0.10789107891078911,
    0.1079010790107901,
    0.1079110791107911,
    0.10792107921079211,
    0.10793107931079311,
    0.1079410794107941,
    0.1079510795107951,
    0.10796107961079611,
    0.1079710797107971,
    0.1079810798107981,
    0.1079910799107991,
    0.10800108001080011,
    0.1080110801108011,
    0.1080210802108021,
    0.10803108031080311,
    0.10804108041080411,
    0.1080510805108051,
    0.1080610806108061,
    0.10807108071080711,
    0.10808108081080811,
    0.1080910809108091,
    0.1081010810108101,
    0.10811108111081111,
    0.10812108121081211,
    0.1081310813108131,
    0.1081410814108141,
    0.10815108151081511,
    0.10816108161081611,
    0.1081710817108171,
    0.1081810818108181,
    0.10819108191081911,
    0.10820108201082011,
    0.1082110821108211,
    0.1082210822108221,
    0.10823108231082311,
    0.10824108241082411,
    0.1082510825108251,
    0.1082610826108261,
    0.10827108271082711,
    0.1082810828108281,
    0.1082910829108291,
    0.10830108301083011,
    0.10831108311083111,
    0.1083210832108321,
    0.1083310833108331,
    0.10834108341083411,
    0.10835108351083511,
    0.1083610836108361,
    0.1083710837108371,
    0.10838108381083811,
    0.10839108391083911,
    0.1084010840108401,
    0.1084110841108411,
    0.10842108421084211,
    0.10843108431084311,
    0.1084410844108441,
    0.1084510845108451,
    0.10846108461084611,
    0.10847108471084711,
    0.1084810848108481,
    0.1084910849108491,
    0.10850108501085011,
    0.10851108511085111,
    0.1085210852108521,
    0.1085310853108531,
    0.10854108541085411,
    0.1085510855108551,
    0.1085610856108561,
    0.10857108571085711,
    0.10858108581085811,
    0.1085910859108591,
    0.1086010860108601,
    0.10861108611086111,
    0.10862108621086211,
    0.1086310863108631,
    0.1086410864108641,
    0.10865108651086511,
    0.10866108661086611,
    0.1086710867108671,
    0.1086810868108681,
    0.10869108691086911,
    0.10870108701087011,
    0.1087110871108711,
    0.1087210872108721,
    0.10873108731087311,
    0.10874108741087411,
    0.1087510875108751,
    0.1087610876108761,
    0.10877108771087711,
    0.10878108781087811,
    0.1087910879108791,
    0.1088010880108801,
    0.10881108811088111,
    0.10882108821088211,
    0.1088310883108831,
    0.10884108841088411,
    0.10885108851088511,
    0.1088610886108861,
    0.1088710887108871,
    0.10888108881088811,
    0.10889108891088911,
    0.1089010890108901,
    0.1089110891108911,
    0.10892108921089211,
    0.10893108931089311,
    0.1089410894108941,
    0.1089510895108951,
    0.10896108961089611,
    0.10897108971089711,
    0.1089810898108981,
    0.1089910899108991,
    0.10900109001090011,
    0.10901109011090111,
    0.1090210902109021,
    0.1090310903109031,
    0.10904109041090411,
    0.10905109051090511,
    0.1090610906109061,
    0.1090710907109071,
    0.10908109081090811,
    0.10909109091090911,
    0.1091010910109101,
    0.10911109111091111,
    0.10912109121091211,
    0.1091310913109131,
    0.1091410914109141,
    0.10915109151091511,
    0.10916109161091611,
    0.1091710917109171,
    0.1091810918109181,
    0.10919109191091911,
    0.10920109201092011,
    0.1092110921109211,
    0.1092210922109221,
    0.10923109231092311,
    0.10924109241092411,
    0.1092510925109251,
    0.1092610926109261,
    0.10927109271092711,
    0.10928109281092811,
    0.1092910929109291,
    0.1093010930109301,
    0.10931109311093111,
    0.10932109321093211,
    0.1093310933109331,
    0.1093410934109341,
    0.10935109351093511,
    0.10936109361093611,
    0.1093710937109371,
    0.10938109381093811,
    0.10939109391093911,
    0.10940109401094011,
    0.1094110941109411,
    0.10942109421094211,
    0.10943109431094311,
    0.1094410944109441,
    0.1094510945109451,
    0.10946109461094611,
    0.10947109471094711,
    0.1094810948109481,
    0.1094910949109491,
    0.10950109501095011,
    0.10951109511095111,
    0.1095210952109521,
    0.1095310953109531,
    0.10954109541095411,
    0.10955109551095511,
    0.1095610956109561,
    0.1095710957109571,
    0.10958109581095811,
    0.10959109591095911,
    0.1096010960109601,
    0.1096110961109611,
    0.10962109621096211,
    0.10963109631096311,
    0.1096410964109641,
    0.10965109651096511,
    0.10966109661096611,
    0.10967109671096711,
    0.1096810968109681,
    0.10969109691096911,
    0.10970109701097011,
    0.10971109711097111,
    0.1097210972109721,
    0.10973109731097311,
    0.10974109741097411,
    0.1097510975109751,
    0.1097610976109761,
    0.10977109771097711,
    0.10978109781097811,
    0.1097910979109791,
    0.1098010980109801,
    0.10981109811098111,
    0.10982109821098211,
    0.1098310983109831,
    0.1098410984109841,
    0.10985109851098511,
    0.10986109861098611,
    0.1098710987109871,
    0.1098810988109881,
    0.10989109891098911,
    0.10990109901099011,
    0.1099110991109911,
    0.10992109921099211,
    0.10993109931099311,
    0.10994109941099411,
    0.1099510995109951,
    0.10996109961099611,
    0.10997109971099711,
    0.10998109981099811,
    0.1099910999109991,
    0.11000110001100011,
    0.11001110011100111,
    0.1100211002110021,
    0.1100311003110031,
    0.11004110041100411,
    0.11005110051100511,
    0.1100611006110061,
    0.1100711007110071,
    0.11008110081100811,
    0.11009110091100911,
    0.1101011010110101,
    0.1101111011110111,
    0.11012110121101211,
    0.11013110131101311,
    0.1101411014110141,
    0.1101511015110151,
    0.11016110161101611,
    0.11017110171101711,
    0.1101811018110181,
    0.11019110191101911,
    0.11020110201102011,
    0.11021110211102111,
    0.1102211022110221,
    0.11023110231102311,
    0.11024110241102411,
    0.11025110251102511,
    0.1102611026110261,
    0.11027110271102711,
    0.11028110281102811,
    0.11029110291102912,
    0.1103011030110301,
    0.11031110311103111,
    0.11032110321103211,
    0.1103311033110331,
    0.1103411034110341,
    0.11035110351103511,
    0.11036110361103611,
    0.1103711037110371,
    0.1103811038110381,
    0.11039110391103911,
    0.11040110401104011,
    0.1104111041110411,
    0.1104211042110421,
    0.11043110431104311,
    0.11044110441104411,
    0.1104511045110451,
    0.11046110461104611,
    0.11047110471104711,
    0.11048110481104811,
    0.1104911049110491,
    0.11050110501105011,
    0.11051110511105111,
    0.11052110521105211,
    0.1105311053110531,
    0.11054110541105411,
    0.11055110551105511,
    0.11056110561105612,
    0.1105711057110571,
    0.11058110581105811,
    0.11059110591105911,
    0.1106011060110601,
    0.1106111061110611,
    0.11062110621106211,
    0.11063110631106311,
    0.1106411064110641,
    0.1106511065110651,
    0.11066110661106611,
    0.11067110671106711,
    0.1106811068110681,
    0.1106911069110691,
    0.11070110701107011,
    0.11071110711107111,
    0.1107211072110721,
    0.11073110731107311,
    0.11074110741107411,
    0.11075110751107511,
    0.1107611076110761,
    0.11077110771107711,
    0.11078110781107811,
    0.11079110791107911,
    0.1108011080110801,
    0.11081110811108111,
    0.11082110821108211,
    0.11083110831108312,
    0.1108411084110841,
    0.11085110851108511,
    0.11086110861108611,
    0.11087110871108712,
    0.1108811088110881,
    0.11089110891108911,
    0.11090110901109011,
    0.1109111091110911,
    0.1109211092110921,
    0.11093110931109311,
    0.11094110941109411,
    0.1109511095110951,
    0.1109611096110961,
    0.11097110971109711,
    0.11098110981109811,
    0.1109911099110991,
    0.11100111001110011,
    0.11101111011110111,
    0.11102111021110211,
    0.1110311103111031,
    0.11104111041110411,
    0.11105111051110511,
    0.11106111061110611,
    0.1110711107111071,
    0.11108111081110811,
    0.11109111091110911,
    0.11110111101111012,
    0.1111111111111111,
    0.11112111121111211,
    0.11113111131111311,
    0.11114111141111412,
    0.1111511115111151,
    0.11116111161111611,
    0.11117111171111711,
    0.1111811118111181,
    0.1111911119111191,
    0.11120111201112011,
    0.11121111211112111,
    0.1112211122111221,
    0.1112311123111231,
    0.11124111241112411,
    0.11125111251112511,
    0.1112611126111261,
    0.11127111271112711,
    0.11128111281112811,
    0.11129111291112911,
    0.1113011130111301,
    0.11131111311113111,
    0.11132111321113211,
    0.11133111331113311,
    0.1113411134111341,
    0.11135111351113511,
    0.11136111361113611,
    0.11137111371113712,
    0.1113811138111381,
    0.11139111391113911,
    0.11140111401114011,
    0.11141111411114112,
    0.1114211142111421,
    0.11143111431114311,
    0.11144111441114411,
    0.11145111451114512,
    0.1114611146111461,
    0.11147111471114711,
    0.11148111481114811,
    0.1114911149111491,
    0.1115011150111501,
    0.11151111511115111,
    0.11152111521115211,
    0.1115311153111531,
    0.11154111541115411,
    0.11155111551115511,
    0.11156111561115611,
    0.1115711157111571,
    0.11158111581115811,
    0.11159111591115911,
    0.11160111601116011,
    0.1116111161111611,
    0.11162111621116211,
    0.11163111631116311,
    0.11164111641116412,
    0.1116511165111651,
    0.11166111661116611,
    0.11167111671116711,
    0.11168111681116812,
    0.1116911169111691,
    0.11170111701117011,
    0.11171111711117111,
    0.11172111721117212,
    0.1117311173111731,
    0.11174111741117411,
    0.11175111751117511,
    0.11176111761117612,
    0.1117711177111771,
    0.11178111781117811,
    0.11179111791117911,
    0.1118011180111801,
    0.11181111811118111,
    0.11182111821118211,
    0.11183111831118311,
    0.1118411184111841,
    0.11185111851118511,
    0.11186111861118611,
    0.11187111871118711,
    0.1118811188111881,
    0.11189111891118911,
    0.11190111901119011,
    0.11191111911119112,
    0.1119211192111921,
    0.11193111931119311,
    0.11194111941119411,
    0.11195111951119512,
    0.1119611196111961,
    0.11197111971119711,
    0.11198111981119811,
    0.11199111991119912,
    0.1120011200112001,
    0.11201112011120111,
    0.11202112021120211,
    0.11203112031120312,
    0.1120411204112041,
    0.11205112051120511,
    0.11206112061120611,
    0.1120711207112071,
    0.11208112081120811,
    0.11209112091120911,
    0.11210112101121011,
    0.1121111211112111,
    0.11212112121121211,
    0.11213112131121311,
    0.11214112141121411,
    0.1121511215112151,
    0.11216112161121611,
    0.11217112171121711,
    0.11218112181121812,
    0.1121911219112191,
    0.11220112201122011,
    0.11221112211122111,
    0.11222112221122212,
    0.1122311223112231,
    0.11224112241122411,
    0.11225112251122511,
    0.11226112261122612,
    0.1122711227112271,
    0.11228112281122811,
    0.11229112291122911,
    0.11230112301123012,
    0.1123111231112311,
    0.11232112321123211,
    0.11233112331123311,
    0.11234112341123412,
    0.11235112351123511,
    0.11236112361123611,
    0.11237112371123711,
    0.1123811238112381,
    0.11239112391123911,
    0.11240112401124011,
    0.11241112411124111,
    0.1124211242112421,
    0.11243112431124311,
    0.11244112441124411,
    0.11245112451124512,
    0.1124611246112461,
    0.11247112471124711,
    0.11248112481124811,
    0.11249112491124912,
    0.1125011250112501,
    0.11251112511125111,
    0.11252112521125211,
    0.11253112531125312,
    0.1125411254112541,
    0.11255112551125511,
    0.11256112561125611,
    0.11257112571125712,
    0.1125811258112581,
    0.11259112591125911,
    0.11260112601126011,
    0.11261112611126112,
    0.11262112621126211,
    0.11263112631126311,
    0.11264112641126411,
    0.1126511265112651,
    0.11266112661126611,
    0.11267112671126711,
    0.11268112681126811,
    0.1126911269112691,
    0.11270112701127011,
    0.11271112711127111,
    0.11272112721127212,
    0.1127311273112731,
    0.11274112741127411,
    0.11275112751127511,
    0.11276112761127612,
    0.1127711277112771,
    0.11278112781127811,
    0.11279112791127911,
    0.11280112801128012,
    0.1128111281112811,
    0.11282112821128211,
    0.11283112831128311,
    0.11284112841128412,
    0.1128511285112851,
    0.11286112861128611,
    0.11287112871128711,
    0.11288112881128812,
    0.11289112891128911,
    0.11290112901129011,
    0.11291112911129111,
    0.11292112921129212,
    0.11293112931129311,
    0.11294112941129411,
    0.11295112951129511,
    0.1129611296112961,
    0.11297112971129711,
    0.11298112981129811,
    0.11299112991129912,
    0.1130011300113001,
    0.11301113011130111,
    0.11302113021130211,
    0.11303113031130312,
    0.1130411304113041,
    0.11305113051130511,
    0.11306113061130611,
    0.11307113071130712,
    0.1130811308113081,
    0.11309113091130911,
    0.11310113101131011,
    0.11311113111131112,
    0.1131211312113121,
    0.11313113131131311,
    0.11314113141131411,
    0.11315113151131512,
    0.11316113161131611,
    0.11317113171131711,
    0.11318113181131811,
    0.11319113191131912,
    0.11320113201132011,
    0.11321113211132111,
    0.11322113221132211,
    0.1132311323113231,
    0.11324113241132411,
    0.11325113251132511,
    0.11326113261132612,
    0.1132711327113271,
    0.11328113281132811,
    0.11329113291132911,
    0.11330113301133012,
    0.1133111331113311,
    0.11332113321133211,
    0.11333113331133311,
    0.11334113341133412,
    0.1133511335113351,
    0.11336113361133611,
    0.11337113371133711,
    0.11338113381133812,
    0.1133911339113391,
    0.11340113401134011,
    0.11341113411134111,
    0.11342113421134212,
    0.11343113431134311,
    0.11344113441134411,
    0.11345113451134511,
    0.11346113461134612,
    0.11347113471134711,
    0.11348113481134811,
    0.11349113491134911,
    0.11350113501135012,
    0.11351113511135111,
    0.11352113521135211,
    0.11353113531135312,
    0.1135411354113541,
    0.11355113551135511,
    0.11356113561135611,
    0.11357113571135712,
    0.1135811358113581,
    0.11359113591135911,
    0.11360113601136011,
    0.11361113611136112,
    0.1136211362113621,
    0.11363113631136311,
    0.11364113641136411,
    0.11365113651136512,
    0.1136611366113661,
    0.11367113671136711,
    0.11368113681136811,
    0.11369113691136912,
    0.11370113701137011,
    0.11371113711137111,
    0.11372113721137211,
    0.11373113731137312,
    0.11374113741137411,
    0.11375113751137511,
    0.11376113761137611,
    0.11377113771137712,
    0.11378113781137811,
    0.11379113791137911,
    0.11380113801138012,
    0.11381113811138112,
    0.11382113821138211,
    0.11383113831138311,
    0.11384113841138412,
    0.1138511385113851,
    0.11386113861138611,
    0.11387113871138711,
    0.11388113881138812,
    0.1138911389113891,
    0.11390113901139011,
    0.11391113911139111,
    0.11392113921139212,
    0.1139311393113931,
    0.11394113941139411,
    0.11395113951139511,
    0.11396113961139612,
    0.11397113971139711,
    0.11398113981139811,
    0.11399113991139911,
    0.11400114001140012,
    0.11401114011140111,
    0.11402114021140211,
    0.11403114031140311,
    0.11404114041140412,
    0.11405114051140511,
    0.11406114061140611,
    0.11407114071140712,
    0.11408114081140812,
    0.11409114091140911,
    0.11410114101141011,
    0.11411114111141112,
    0.1141211412114121,
    0.11413114131141311,
    0.11414114141141411,
    0.11415114151141512,
    0.1141611416114161,
    0.11417114171141711,
    0.11418114181141811,
    0.11419114191141912,
    0.1142011420114201,
    0.11421114211142111,
    0.11422114221142211,
    0.11423114231142312,
    0.11424114241142411,
    0.11425114251142511,
    0.11426114261142611,
    0.11427114271142712,
    0.11428114281142811,
    0.11429114291142911,
    0.11430114301143011,
    0.11431114311143112,
    0.11432114321143211,
    0.11433114331143311,
    0.11434114341143412,
    0.11435114351143512,
    0.11436114361143611,
    0.11437114371143711,
    0.11438114381143812,
    0.11439114391143912,
    0.11440114401144011,
    0.11441114411144111,
    0.11442114421144212,
    0.1144311443114431,
    0.11444114441144411,
    0.11445114451144511,
    0.11446114461144612,
    0.1144711447114471,
    0.11448114481144811,
    0.11449114491144911,
    0.11450114501145012,
    0.11451114511145111,
    0.11452114521145211,
    0.11453114531145311,
    0.11454114541145412,
    0.11455114551145511,
    0.11456114561145611,
    0.11457114571145711,
    0.11458114581145812,
    0.11459114591145911,
    0.11460114601146011,
    0.11461114611146112,
    0.11462114621146212,
    0.11463114631146311,
    0.11464114641146411,
    0.11465114651146512,
    0.11466114661146612,
    0.11467114671146711,
    0.11468114681146811,
    0.11469114691146912,
    0.1147011470114701,
    0.11471114711147111,
    0.11472114721147211,
    0.11473114731147312,
    0.1147411474114741,
    0.11475114751147511,
    0.11476114761147611,
    0.11477114771147712,
    0.11478114781147811,
    0.11479114791147911,
    0.11480114801148011,
    0.11481114811148112,
    0.11482114821148211,
    0.11483114831148311,
    0.11484114841148411,
    0.11485114851148512,
    0.11486114861148611,
    0.11487114871148711,
    0.11488114881148812,
    0.11489114891148912,
    0.11490114901149011,
    0.11491114911149111,
    0.11492114921149212,
    0.11493114931149312,
    0.11494114941149411,
    0.11495114951149511,
    0.11496114961149612,
    0.11497114971149712,
    0.11498114981149811,
    0.11499114991149911,
    0.11500115001150012,
    0.1150111501115011,
    0.11502115021150211,
    0.11503115031150311,
    0.11504115041150412,
    0.11505115051150511,
    0.11506115061150611,
    0.11507115071150711,
    0.11508115081150812,
    0.11509115091150911,
    0.11510115101151011,
    0.11511115111151111,
    0.11512115121151212,
    0.11513115131151311,
    0.11514115141151411,
    0.11515115151151512,
    0.11516115161151612,
    0.11517115171151711,
    0.11518115181151811,
    0.11519115191151912,
    0.11520115201152012,
    0.11521115211152111,
    0.11522115221152211,
    0.11523115231152312,
    0.11524115241152412,
    0.11525115251152511,
    0.11526115261152611,
    0.11527115271152712,
    0.1152811528115281,
    0.11529115291152911,
    0.11530115301153011,
    0.11531115311153112,
    0.11532115321153211,
    0.11533115331153311,
    0.11534115341153411,
    0.11535115351153512,
    0.11536115361153611,
    0.11537115371153711,
    0.11538115381153811,
    0.11539115391153912,
    0.11540115401154011,
    0.11541115411154111,
    0.11542115421154212,
    0.11543115431154312,
    0.11544115441154411,
    0.11545115451154511,
    0.11546115461154612,
    0.11547115471154712,
    0.11548115481154811,
    0.11549115491154911,
    0.11550115501155012,
    0.11551115511155112,
    0.11552115521155211,
    0.11553115531155311,
    0.11554115541155412,
    0.11555115551155512,
    0.11556115561155611,
    0.11557115571155711,
    0.11558115581155812,
    0.11559115591155911,
    0.11560115601156011,
    0.11561115611156111,
    0.11562115621156212,
    0.11563115631156311,
    0.11564115641156411,
    0.11565115651156511,
    0.11566115661156612,
    0.11567115671156711,
    0.11568115681156811,
    0.11569115691156912,
    0.11570115701157012,
    0.11571115711157111,
    0.11572115721157211,
    0.11573115731157312,
    0.11574115741157412,
    0.11575115751157511,
    0.11576115761157611,
    0.11577115771157712,
    0.11578115781157812,
    0.11579115791157911,
    0.11580115801158011,
    0.11581115811158112,
    0.11582115821158212,
    0.11583115831158311,
    0.11584115841158411,
    0.11585115851158512,
    0.11586115861158611,
    0.11587115871158711,
    0.11588115881158811,
    0.11589115891158912,
    0.11590115901159011,
    0.11591115911159111,
    0.11592115921159211,
    0.11593115931159312,
    0.11594115941159411,
    0.11595115951159511,
    0.11596115961159612,
    0.11597115971159712,
    0.11598115981159811,
    0.11599115991159911,
    0.11600116001160012,
    0.11601116011160112,
    0.11602116021160211,
    0.11603116031160311,
    0.11604116041160412,
    0.11605116051160512,
    0.11606116061160611,
    0.11607116071160711,
    0.11608116081160812,
    0.11609116091160912,
    0.11610116101161011,
    0.11611116111161111,
    0.11612116121161212,
    0.11613116131161312,
    0.11614116141161411,
    0.11615116151161511,
    0.11616116161161612,
    0.11617116171161711,
    0.11618116181161811,
    0.11619116191161911,
    0.11620116201162012,
    0.11621116211162111,
    0.11622116221162211,
    0.11623116231162312,
    0.11624116241162412,
    0.11625116251162511,
    0.11626116261162611,
    0.11627116271162712,
    0.11628116281162812,
    0.11629116291162911,
    0.11630116301163011,
    0.11631116311163112,
    0.11632116321163212,
    0.11633116331163311,
    0.11634116341163411,
    0.11635116351163512,
    0.11636116361163612,
    0.11637116371163711,
    0.11638116381163811,
    0.11639116391163912,
    0.11640116401164012,
    0.11641116411164111,
    0.11642116421164211,
    0.11643116431164312,
    0.11644116441164412,
    0.11645116451164511,
    0.11646116461164611,
    0.11647116471164712,
    0.11648116481164811,
    0.11649116491164911,
    0.11650116501165012,
    0.11651116511165112,
    0.11652116521165211,
    0.11653116531165311,
    0.11654116541165412,
    0.11655116551165512,
    0.11656116561165611,
    0.11657116571165711,
    0.11658116581165812,
    0.11659116591165912,
    0.11660116601166011,
    0.11661116611166111,
    0.11662116621166212,
    0.11663116631166312,
    0.11664116641166411,
    0.11665116651166511,
    0.11666116661166612,
    0.11667116671166712,
    0.11668116681166811,
    0.11669116691166911,
    0.11670116701167012,
    0.11671116711167112,
    0.11672116721167211,
    0.11673116731167311,
    0.11674116741167412,
    0.11675116751167511,
    0.11676116761167611,
    0.11677116771167712,
    0.11678116781167812,
    0.11679116791167911,
    0.11680116801168011,
    0.11681116811168112,
    0.11682116821168212,
    0.11683116831168311,
    0.11684116841168411,
    0.11685116851168512,
    0.11686116861168612,
    0.11687116871168711,
    0.11688116881168811,
    0.11689116891168912,
    0.11690116901169012,
    0.11691116911169111,
    0.11692116921169211,
    0.11693116931169312,
    0.11694116941169412,
    0.11695116951169511,
    0.11696116961169611,
    0.11697116971169712,
    0.11698116981169812,
    0.11699116991169911,
    0.11700117001170011,
    0.11701117011170112,
    0.11702117021170212,
    0.11703117031170311,
    0.11704117041170412,
    0.11705117051170512,
    0.11706117061170611,
    0.11707117071170711,
    0.11708117081170812,
    0.11709117091170912,
    0.11710117101171011,
    0.11711117111171111,
    0.11712117121171212,
    0.11713117131171312,
    0.11714117141171411,
    0.11715117151171511,
    0.11716117161171612,
    0.11717117171171712,
    0.11718117181171811,
    0.11719117191171911,
    0.11720117201172012,
    0.11721117211172112,
    0.11722117221172211,
    0.11723117231172311,
    0.11724117241172412,
    0.11725117251172512,
    0.11726117261172611,
    0.11727117271172711,
    0.11728117281172812,
    0.11729117291172912,
    0.11730117301173011,
    0.11731117311173112,
    0.11732117321173212,
    0.11733117331173311,
    0.11734117341173411,
    0.11735117351173512,
    0.11736117361173612,
    0.11737117371173711,
    0.11738117381173811,
    0.11739117391173912,
    0.11740117401174012,
    0.11741117411174111,
    0.11742117421174211,
    0.11743117431174312,
    0.11744117441174412,
    0.11745117451174511,
    0.11746117461174611,
    0.11747117471174712,
    0.11748117481174812,
    0.11749117491174911,
    0.11750117501175011,
    0.11751117511175112,
    0.11752117521175212,
    0.11753117531175311,
    0.11754117541175411,
    0.11755117551175512,
    0.11756117561175612,
    0.11757117571175711,
    0.11758117581175812,
    0.11759117591175912,
    0.11760117601176012,
    0.11761117611176111,
    0.11762117621176212,
    0.11763117631176312,
    0.11764117641176411,
    0.11765117651176511,
    0.11766117661176612,
    0.11767117671176712,
    0.11768117681176811,
    0.11769117691176911,
    0.11770117701177012,
    0.11771117711177112,
    0.11772117721177211,
    0.11773117731177311,
    0.11774117741177412,
    0.11775117751177512,
    0.11776117761177611,
    0.11777117771177711,
    0.11778117781177812,
    0.11779117791177912,
    0.11780117801178011,
    0.11781117811178111,
    0.11782117821178212,
    0.11783117831178312,
    0.11784117841178411,
    0.11785117851178512,
    0.11786117861178612,
    0.11787117871178712,
    0.11788117881178811,
    0.11789117891178912,
    0.11790117901179012,
    0.11791117911179111,
    0.11792117921179211,
    0.11793117931179312,
    0.11794117941179412,
    0.11795117951179511,
    0.11796117961179611,
    0.11797117971179712,
    0.11798117981179812,
    0.11799117991179911,
    0.11800118001180011,
    0.11801118011180112,
    0.11802118021180212,
    0.11803118031180311,
    0.11804118041180411,
    0.11805118051180512,
    0.11806118061180612,
    0.11807118071180711,
    0.11808118081180811,
    0.11809118091180912,
    0.11810118101181012,
    0.11811118111181111,
    0.11812118121181212,
    0.11813118131181312,
    0.11814118141181412,
    0.11815118151181511,
    0.11816118161181612,
    0.11817118171181712,
    0.11818118181181812,
    0.11819118191181911,
    0.11820118201182012,
    0.11821118211182112,
    0.11822118221182211,
    0.11823118231182311,
    0.11824118241182412,
    0.11825118251182512,
    0.11826118261182611,
    0.11827118271182711,
    0.11828118281182812,
    0.11829118291182912,
    0.11830118301183011,
    0.11831118311183111,
    0.11832118321183212,
    0.11833118331183312,
    0.11834118341183411,
    0.11835118351183511,
    0.11836118361183612,
    0.11837118371183712,
    0.11838118381183811,
    0.11839118391183912,
    0.11840118401184012,
    0.11841118411184112,
    0.11842118421184211,
    0.11843118431184312,
    0.11844118441184412,
    0.11845118451184512,
    0.11846118461184611,
    0.11847118471184712,
    0.11848118481184812,
    0.11849118491184912,
    0.11850118501185011,
    0.11851118511185112,
    0.11852118521185212,
    0.11853118531185311,
    0.11854118541185411,
    0.11855118551185512,
    0.11856118561185612,
    0.11857118571185711,
    0.11858118581185811,
    0.11859118591185912,
    0.11860118601186012,
    0.11861118611186111,
    0.11862118621186211,
    0.11863118631186312,
    0.11864118641186412,
    0.11865118651186511,
    0.11866118661186612,
    0.11867118671186712,
    0.11868118681186812,
    0.11869118691186911,
    0.11870118701187012,
    0.11871118711187112,
    0.11872118721187212,
    0.11873118731187311,
    0.11874118741187412,
    0.11875118751187512,
    0.11876118761187612,
    0.11877118771187711,
    0.11878118781187812,
    0.11879118791187912,
    0.11880118801188011,
    0.11881118811188111,
    0.11882118821188212,
    0.11883118831188312,
    0.11884118841188411,
    0.11885118851188511,
    0.11886118861188612,
    0.11887118871188712,
    0.11888118881188811,
    0.11889118891188911,
    0.11890118901189012,
    0.11891118911189112,
    0.11892118921189211,
    0.11893118931189312,
    0.11894118941189412,
    0.11895118951189512,
    0.11896118961189611,
    0.11897118971189712,
    0.11898118981189812,
    0.11899118991189912,
    0.11900119001190011,
    0.11901119011190112,
    0.11902119021190212,
    0.11903119031190312,
    0.11904119041190411,
    0.11905119051190512,
    0.11906119061190612,
    0.11907119071190712,
    0.11908119081190811,
    0.11909119091190912,
    0.11910119101191012,
    0.11911119111191111,
    0.11912119121191211,
    0.11913119131191312,
    0.11914119141191412,
    0.11915119151191511,
    0.11916119161191611,
    0.11917119171191712,
    0.11918119181191812,
    0.11919119191191911,
    0.11920119201192012,
    0.11921119211192112,
    0.11922119221192212,
    0.11923119231192311,
    0.11924119241192412,
    0.11925119251192512,
    0.11926119261192612,
    0.11927119271192711,
    0.11928119281192812,
    0.11929119291192912,
    0.11930119301193012,
    0.11931119311193111,
    0.11932119321193212,
    0.11933119331193312,
    0.11934119341193412,
    0.11935119351193511,
    0.11936119361193612,
    0.11937119371193712,
    0.11938119381193811,
    0.11939119391193911,
    0.11940119401194012,
    0.11941119411194112,
    0.11942119421194211,
    0.11943119431194311,
    0.11944119441194412,
    0.11945119451194512,
    0.11946119461194611,
    0.11947119471194712,
    0.11948119481194812,
    0.11949119491194912,
    0.11950119501195011,
    0.11951119511195112,
    0.11952119521195212,
    0.11953119531195312,
    0.11954119541195411,
    0.11955119551195512,
    0.11956119561195612,
    0.11957119571195712,
    0.11958119581195811,
    0.11959119591195912,
    0.11960119601196012,
    0.11961119611196112,
    0.11962119621196211,
    0.11963119631196312,
    0.11964119641196412,
    0.11965119651196512,
    0.11966119661196611,
    0.11967119671196712,
    0.11968119681196812,
    0.11969119691196911,
    0.11970119701197011,
    0.11971119711197112,
    0.11972119721197212,
    0.11973119731197311,
    0.11974119741197412,
    0.11975119751197512,
    0.11976119761197612,
    0.11977119771197711,
    0.11978119781197812,
    0.11979119791197912,
    0.11980119801198012,
    0.11981119811198111,
    0.11982119821198212,
    0.11983119831198312,
    0.11984119841198412,
    0.11985119851198511,
    0.11986119861198612,
    0.11987119871198712,
    0.11988119881198812,
    0.11989119891198911,
    0.11990119901199012,
    0.11991119911199112,
    0.11992119921199212,
    0.11993119931199311,
    0.11994119941199412,
    0.11995119951199512,
    0.11996119961199611,
    0.11997119971199711,
    0.11998119981199812,
    0.11999119991199912,
    0.12000120001200011,
    0.12001120011200112,
    0.12002120021200212,
    0.12003120031200312,
    0.12004120041200411,
    0.12005120051200512,
    0.12006120061200612,
    0.12007120071200712,
    0.12008120081200811,
    0.12009120091200912,
    0.12010120101201012,
    0.12011120111201112,
    0.12012120121201211,
    0.12013120131201312,
    0.12014120141201412,
    0.12015120151201512,
    0.12016120161201611,
    0.12017120171201712,
    0.12018120181201812,
    0.12019120191201912,
    0.12020120201202011,
    0.12021120211202112,
    0.12022120221202212,
    0.12023120231202312,
    0.12024120241202411,
    0.12025120251202512,
    0.12026120261202612,
    0.12027120271202711,
    0.12028120281202812,
    0.12029120291202912,
    0.12030120301203012,
    0.12031120311203111,
    0.12032120321203212,
    0.12033120331203312,
    0.12034120341203412,
    0.12035120351203511,
    0.12036120361203612,
    0.12037120371203712,
    0.12038120381203812,
    0.12039120391203911,
    0.12040120401204012,
    0.12041120411204112,
    0.12042120421204212,
    0.12043120431204311,
    0.12044120441204412,
    0.12045120451204512,
    0.12046120461204612,
    0.12047120471204711,
    0.12048120481204812,
    0.12049120491204912,
    0.12050120501205012,
    0.12051120511205111,
    0.12052120521205212,
    0.12053120531205312,
    0.12054120541205411,
    0.12055120551205512,
    0.12056120561205612,
    0.12057120571205712,
    0.12058120581205811,
    0.12059120591205912,
    0.12060120601206012,
    0.12061120611206112,
    0.12062120621206211,
    0.12063120631206312,
    0.12064120641206412,
    0.12065120651206512,
    0.12066120661206611,
    0.12067120671206712,
    0.12068120681206812,
    0.12069120691206912,
    0.12070120701207011,
    0.12071120711207112,
    0.12072120721207212,
    0.12073120731207312,
    0.12074120741207411,
    0.12075120751207512,
    0.12076120761207612,
    0.12077120771207712,
    0.12078120781207811,
    0.12079120791207912,
    0.12080120801208012,
    0.12081120811208113,
    0.12082120821208212,
    0.12083120831208312,
    0.12084120841208412,
    0.12085120851208511,
    0.12086120861208612,
    0.12087120871208712,
    0.12088120881208812,
    0.12089120891208911,
    0.12090120901209012,
    0.12091120911209112,
    0.12092120921209212,
    0.12093120931209311,
    0.12094120941209412,
    0.12095120951209512,
    0.12096120961209612,
    0.12097120971209711,
    0.12098120981209812,
    0.12099120991209912,
    0.12100121001210012,
    0.12101121011210111,
    0.12102121021210212,
    0.12103121031210312,
    0.12104121041210412,
    0.12105121051210511,
    0.12106121061210612,
    0.12107121071210712,
    0.12108121081210813,
    0.12109121091210912,
    0.12110121101211012,
    0.12111121111211112,
    0.12112121121211213,
    0.12113121131211312,
    0.12114121141211412,
    0.12115121151211512,
    0.12116121161211611,
    0.12117121171211712,
    0.12118121181211812,
    0.12119121191211912,
    0.12120121201212011,
    0.12121121211212112,
    0.12122121221212212,
    0.12123121231212312,
    0.12124121241212411,
    0.12125121251212512,
    0.12126121261212612,
    0.12127121271212712,
    0.12128121281212811,
    0.12129121291212912,
    0.12130121301213012,
    0.12131121311213112,
    0.12132121321213211,
    0.12133121331213312,
    0.12134121341213412,
    0.12135121351213513,
    0.12136121361213612,
    0.12137121371213712,
    0.12138121381213812,
    0.12139121391213913,
    0.12140121401214012,
    0.12141121411214112,
    0.12142121421214212,
    0.12143121431214311,
    0.12144121441214412,
    0.12145121451214512,
    0.12146121461214612,
    0.12147121471214711,
    0.12148121481214812,
    0.12149121491214912,
    0.12150121501215012,
    0.12151121511215111,
    0.12152121521215212,
    0.12153121531215312,
    0.12154121541215412,
    0.12155121551215511,
    0.12156121561215612,
    0.12157121571215712,
    0.12158121581215812,
    0.12159121591215911,
    0.12160121601216012,
    0.12161121611216112,
    0.12162121621216213,
    0.12163121631216312,
    0.12164121641216412,
    0.12165121651216512,
    0.12166121661216613,
    0.12167121671216712,
    0.12168121681216812,
    0.12169121691216912,
    0.12170121701217013,
    0.12171121711217112,
    0.12172121721217212,
    0.12173121731217312,
    0.12174121741217411,
    0.12175121751217512,
    0.12176121761217612,
    0.12177121771217712,
    0.12178121781217811,
    0.12179121791217912,
    0.12180121801218012,
    0.12181121811218112,
    0.12182121821218211,
    0.12183121831218312,
    0.12184121841218412,
    0.12185121851218512,
    0.12186121861218611,
    0.12187121871218712,
    0.12188121881218812,
    0.12189121891218913,
    0.12190121901219012,
    0.12191121911219112,
    0.12192121921219212,
    0.12193121931219313,
    0.12194121941219412,
    0.12195121951219512,
    0.12196121961219612,
    0.12197121971219713,
    0.12198121981219812,
    0.12199121991219912,
    0.12200122001220012,
    0.12201122011220111,
    0.12202122021220212,
    0.12203122031220312,
    0.12204122041220412,
    0.12205122051220511,
    0.12206122061220612,
    0.12207122071220712,
    0.12208122081220812,
    0.12209122091220911,
    0.12210122101221012,
    0.12211122111221112,
    0.12212122121221212,
    0.12213122131221311,
    0.12214122141221412,
    0.12215122151221512,
    0.12216122161221613,
    0.12217122171221712,
    0.12218122181221812,
    0.12219122191221912,
    0.12220122201222013,
    0.12221122211222112,
    0.12222122221222212,
    0.12223122231222312,
    0.12224122241222413,
    0.12225122251222512,
    0.12226122261222612,
    0.12227122271222712,
    0.12228122281222813,
    0.12229122291222912,
    0.12230122301223012,
    0.12231122311223112,
    0.12232122321223211,
    0.12233122331223312,
    0.12234122341223412,
    0.12235122351223512,
    0.12236122361223611,
    0.12237122371223712,
    0.12238122381223812,
    0.12239122391223912,
    0.12240122401224011,
    0.12241122411224112,
    0.12242122421224212,
    0.12243122431224313,
    0.12244122441224412,
    0.12245122451224512,
    0.12246122461224612,
    0.12247122471224713,
    0.12248122481224812,
    0.12249122491224912,
    0.12250122501225012,
    0.12251122511225113,
    0.12252122521225212,
    0.12253122531225312,
    0.12254122541225412,
    0.12255122551225513,
    0.12256122561225612,
    0.12257122571225712,
    0.12258122581225812,
    0.12259122591225911,
    0.12260122601226012,
    0.12261122611226112,
    0.12262122621226212,
    0.12263122631226311,
    0.12264122641226412,
    0.12265122651226512,
    0.12266122661226612,
    0.12267122671226711,
    0.12268122681226812,
    0.12269122691226912,
    0.12270122701227013,
    0.12271122711227112,
    0.12272122721227212,
    0.12273122731227312,
    0.12274122741227413,
    0.12275122751227512,
    0.12276122761227612,
    0.12277122771227712,
    0.12278122781227813,
    0.12279122791227912,
    0.12280122801228012,
    0.12281122811228112,
    0.12282122821228213,
    0.12283122831228312,
    0.12284122841228412,
    0.12285122851228512,
    0.12286122861228613,
    0.12287122871228712,
    0.12288122881228812,
    0.12289122891228912,
    0.12290122901229011,
    0.12291122911229112,
    0.12292122921229212,
    0.12293122931229312,
    0.12294122941229411,
    0.12295122951229512,
    0.12296122961229612,
    0.12297122971229713,
    0.12298122981229812,
    0.12299122991229912,
    0.12300123001230012,
    0.12301123011230113,
    0.12302123021230212,
    0.12303123031230312,
    0.12304123041230412,
    0.12305123051230513,
    0.12306123061230612,
    0.12307123071230712,
    0.12308123081230812,
    0.12309123091230913,
    0.12310123101231012,
    0.12311123111231112,
    0.12312123121231212,
    0.12313123131231313,
    0.12314123141231412,
    0.12315123151231512,
    0.12316123161231612,
    0.12317123171231713,
    0.12318123181231812,
    0.12319123191231912,
    0.12320123201232012,
    0.12321123211232111,
    0.12322123221232212,
    0.12323123231232312,
    0.12324123241232413,
    0.12325123251232512,
    0.12326123261232612,
    0.12327123271232712,
    0.12328123281232813,
    0.12329123291232912,
    0.12330123301233012,
    0.12331123311233112,
    0.12332123321233213,
    0.12333123331233312,
    0.12334123341233412,
    0.12335123351233512,
    0.12336123361233613,
    0.12337123371233712,
    0.12338123381233812,
    0.12339123391233912,
    0.12340123401234013,
    0.12341123411234112,
    0.12342123421234212,
    0.12343123431234312,
    0.12344123441234413,
    0.12345123451234512,
    0.12346123461234612,
    0.12347123471234712,
    0.12348123481234811,
    0.12349123491234912,
    0.12350123501235012,
    0.12351123511235113,
    0.12352123521235212,
    0.12353123531235312,
    0.12354123541235412,
    0.12355123551235513,
    0.12356123561235612,
    0.12357123571235712,
    0.12358123581235812,
    0.12359123591235913,
    0.12360123601236012,
    0.12361123611236112,
    0.12362123621236212,
    0.12363123631236313,
    0.12364123641236412,
    0.12365123651236512,
    0.12366123661236612,
    0.12367123671236713,
    0.12368123681236812,
    0.12369123691236912,
    0.12370123701237012,
    0.12371123711237113,
    0.12372123721237212,
    0.12373123731237312,
    0.12374123741237412,
    0.12375123751237513,
    0.12376123761237612,
    0.12377123771237712,
    0.12378123781237813,
    0.12379123791237912,
    0.12380123801238012,
    0.12381123811238112,
    0.12382123821238213,
    0.12383123831238312,
    0.12384123841238412,
    0.12385123851238512,
    0.12386123861238613,
    0.12387123871238712,
    0.12388123881238812,
    0.12389123891238912,
    0.12390123901239013,
    0.12391123911239112,
    0.12392123921239212,
    0.12393123931239312,
    0.12394123941239413,
    0.12395123951239512,
    0.12396123961239612,
    0.12397123971239712,
    0.12398123981239813,
    0.12399123991239912,
    0.12400124001240012,
    0.12401124011240112,
    0.12402124021240213,
    0.12403124031240312,
    0.12404124041240412,
    0.12405124051240513,
    0.12406124061240612,
    0.12407124071240712,
    0.12408124081240812,
    0.12409124091240913,
    0.12410124101241012,
    0.12411124111241112,
    0.12412124121241212,
    0.12413124131241313,
    0.12414124141241412,
    0.12415124151241512,
    0.12416124161241612,
    0.12417124171241713,
    0.12418124181241812,
    0.12419124191241912,
    0.12420124201242012,
    0.12421124211242113,
    0.12422124221242212,
    0.12423124231242312,
    0.12424124241242412,
    0.12425124251242513,
    0.12426124261242612,
    0.12427124271242712,
    0.12428124281242812,
    0.12429124291242913,
    0.12430124301243012,
    0.12431124311243112,
    0.12432124321243213,
    0.12433124331243313,
    0.12434124341243412,
    0.12435124351243512,
    0.12436124361243613,
    0.12437124371243712,
    0.12438124381243812,
    0.12439124391243912,
    0.12440124401244013,
    0.12441124411244112,
    0.12442124421244212,
    0.12443124431244312,
    0.12444124441244413,
    0.12445124451244512,
    0.12446124461244612,
    0.12447124471244712,
    0.12448124481244813,
    0.12449124491244912,
    0.12450124501245012,
    0.12451124511245112,
    0.12452124521245213,
    0.12453124531245312,
    0.12454124541245412,
    0.12455124551245512,
    0.12456124561245613,
    0.12457124571245712,
    0.12458124581245812,
    0.12459124591245913,
    0.12460124601246013,
    0.12461124611246112,
    0.12462124621246212,
    0.12463124631246313,
    0.12464124641246412,
    0.12465124651246512,
    0.12466124661246612,
    0.12467124671246713,
    0.12468124681246812,
    0.12469124691246912,
    0.12470124701247012,
    0.12471124711247113,
    0.12472124721247212,
    0.12473124731247312,
    0.12474124741247412,
    0.12475124751247513,
    0.12476124761247612,
    0.12477124771247712,
    0.12478124781247812,
    0.12479124791247913,
    0.12480124801248012,
    0.12481124811248112,
    0.12482124821248212,
    0.12483124831248313,
    0.12484124841248412,
    0.12485124851248512,
    0.12486124861248613,
    0.12487124871248713,
    0.12488124881248812,
    0.12489124891248912,
    0.12490124901249013,
    0.12491124911249113,
    0.12492124921249212,
    0.12493124931249312,
    0.12494124941249413,
    0.12495124951249512,
    0.12496124961249612,
    0.12497124971249712,
    0.12498124981249813,
    0.12499124991249912,
    0.12500125001250012,
    0.1250112501125011,
    0.12502125021250213,
    0.12503125031250312,
    0.12504125041250413,
    0.12505125051250512,
    0.12506125061250611,
    0.12507125071250713,
    0.12508125081250812,
    0.1250912509125091,
    0.12510125101251013,
    0.12511125111251112,
    0.12512125121251214,
    0.12513125131251313,
    0.12514125141251412,
    0.12515125151251513,
    0.12516125161251612,
    0.1251712517125171,
    0.12518125181251813,
    0.12519125191251912,
    0.12520125201252014,
    0.12521125211252113,
    0.12522125221252212,
    0.12523125231252313,
    0.12524125241252412,
    0.1252512525125251,
    0.12526125261252613,
    0.12527125271252712,
    0.1252812528125281,
    0.12529125291252913,
    0.12530125301253012,
    0.12531125311253113,
    0.12532125321253212,
    0.12533125331253311,
    0.12534125341253413,
    0.12535125351253512,
    0.1253612536125361,
    0.12537125371253713,
    0.12538125381253812,
    0.12539125391253914,
    0.12540125401254013,
    0.12541125411254112,
    0.12542125421254213,
    0.12543125431254312,
    0.1254412544125441,
    0.12545125451254513,
    0.12546125461254612,
    0.12547125471254714,
    0.12548125481254813,
    0.12549125491254912,
    0.12550125501255013,
    0.12551125511255112,
    0.1255212552125521,
    0.12553125531255313,
    0.12554125541255412,
    0.1255512555125551,
    0.12556125561255613,
    0.12557125571255712,
    0.12558125581255813,
    0.12559125591255912,
    0.12560125601256011,
    0.12561125611256113,
    0.12562125621256212,
    0.1256312563125631,
    0.12564125641256413,
    0.12565125651256512,
    0.12566125661256614,
    0.12567125671256713,
    0.12568125681256812,
    0.12569125691256913,
    0.12570125701257012,
    0.1257112571125711,
    0.12572125721257213,
    0.12573125731257312,
    0.12574125741257414,
    0.12575125751257513,
    0.12576125761257612,
    0.12577125771257713,
    0.12578125781257812,
    0.1257912579125791,
    0.12580125801258013,
    0.12581125811258112,
    0.1258212582125821,
    0.12583125831258313,
    0.12584125841258412,
    0.12585125851258513,
    0.12586125861258612,
    0.12587125871258711,
    0.12588125881258813,
    0.12589125891258912,
    0.1259012590125901,
    0.12591125911259113,
    0.12592125921259212,
    0.12593125931259314,
    0.12594125941259413,
    0.12595125951259512,
    0.12596125961259613,
    0.12597125971259712,
    0.1259812598125981,
    0.12599125991259913,
    0.12600126001260012,
    0.12601126011260114,
    0.12602126021260213,
    0.12603126031260312,
    0.12604126041260413,
    0.12605126051260512,
    0.1260612606126061,
    0.12607126071260713,
    0.12608126081260812,
    0.12609126091260914,
    0.12610126101261013,
    0.12611126111261112,
    0.12612126121261213,
    0.12613126131261312,
    0.12614126141261411,
    0.12615126151261513,
    0.12616126161261612,
    0.1261712617126171,
    0.12618126181261813,
    0.12619126191261912,
    0.12620126201262014,
    0.12621126211262113,
    0.12622126221262212,
    0.12623126231262313,
    0.12624126241262412,
    0.1262512625126251,
    0.12626126261262613,
    0.12627126271262712,
    0.12628126281262814,
    0.12629126291262913,
    0.12630126301263012,
    0.12631126311263113,
    0.12632126321263212,
    0.1263312633126331,
    0.12634126341263413,
    0.12635126351263512,
    0.12636126361263614,
    0.12637126371263713,
    0.12638126381263812,
    0.12639126391263913,
    0.12640126401264012,
    0.12641126411264111,
    0.12642126421264213,
    0.12643126431264312,
    0.1264412644126441,
    0.12645126451264513,
    0.12646126461264612,
    0.12647126471264714,
    0.12648126481264813,
    0.12649126491264912,
    0.12650126501265013,
    0.12651126511265112,
    0.1265212652126521,
    0.12653126531265313,
    0.12654126541265412,
    0.12655126551265514,
    0.12656126561265613,
    0.12657126571265712,
    0.12658126581265813,
    0.12659126591265912,
    0.1266012660126601,
    0.12661126611266113,
    0.12662126621266212,
    0.12663126631266314,
    0.12664126641266413,
    0.12665126651266512,
    0.12666126661266613,
    0.12667126671266712,
    0.12668126681266811,
    0.12669126691266913,
    0.12670126701267012,
    0.1267112671126711,
    0.12672126721267213,
    0.12673126731267312,
    0.12674126741267414,
    0.12675126751267513,
    0.12676126761267612,
    0.12677126771267713,
    0.12678126781267812,
    0.1267912679126791,
    0.12680126801268013,
    0.12681126811268112,
    0.12682126821268214,
    0.12683126831268313,
    0.12684126841268412,
    0.12685126851268513,
    0.12686126861268612,
    0.1268712687126871,
    0.12688126881268813,
    0.12689126891268912,
    0.12690126901269014,
    0.12691126911269113,
    0.12692126921269212,
    0.12693126931269313,
    0.12694126941269412,
    0.12695126951269511,
    0.12696126961269613,
    0.12697126971269712,
    0.1269812698126981,
    0.12699126991269913,
    0.12700127001270012,
    0.12701127011270114,
    0.12702127021270213,
    0.12703127031270312,
    0.12704127041270413,
    0.12705127051270512,
    0.1270612706127061,
    0.12707127071270713,
    0.12708127081270812,
    0.12709127091270914,
    0.12710127101271013,
    0.12711127111271112,
    0.12712127121271213,
    0.12713127131271312,
    0.1271412714127141,
    0.12715127151271513,
    0.12716127161271612,
    0.12717127171271714,
    0.12718127181271813,
    0.12719127191271912,
    0.12720127201272013,
    0.12721127211272112,
    0.12722127221272211,
    0.12723127231272313,
    0.12724127241272412,
    0.12725127251272514,
    0.12726127261272613,
    0.12727127271272712,
    0.12728127281272814,
    0.12729127291272913,
    0.12730127301273012,
    0.12731127311273113,
    0.12732127321273212,
    0.1273312733127331,
    0.12734127341273413,
    0.12735127351273512,
    0.12736127361273614,
    0.12737127371273713,
    0.12738127381273812,
    0.12739127391273913,
    0.12740127401274012,
    0.1274112741127411,
    0.12742127421274213,
    0.12743127431274312,
    0.12744127441274414,
    0.12745127451274513,
    0.12746127461274612,
    0.12747127471274713,
    0.12748127481274812,
    0.12749127491274911,
    0.12750127501275013,
    0.12751127511275112,
    0.12752127521275214,
    0.12753127531275313,
    0.12754127541275412,
    0.12755127551275514,
    0.12756127561275613,
    0.12757127571275712,
    0.12758127581275813,
    0.12759127591275912,
    0.1276012760127601,
    0.12761127611276113,
    0.12762127621276212,
    0.12763127631276314,
    0.12764127641276413,
    0.12765127651276512,
    0.12766127661276613,
    0.12767127671276712,
    0.1276812768127681,
    0.12769127691276913,
    0.12770127701277012,
    0.12771127711277114,
    0.12772127721277213,
    0.12773127731277312,
    0.12774127741277413,
    0.12775127751277512,
    0.12776127761277611,
    0.12777127771277713,
    0.12778127781277812,
    0.12779127791277914,
    0.12780127801278013,
    0.12781127811278112,
    0.12782127821278214,
    0.12783127831278313,
    0.12784127841278412,
    0.12785127851278513,
    0.12786127861278612,
    0.1278712787127871,
    0.12788127881278813,
    0.12789127891278912,
    0.12790127901279014,
    0.12791127911279113,
    0.12792127921279212,
    0.12793127931279313,
    0.12794127941279412,
    0.1279512795127951,
    0.12796127961279613,
    0.12797127971279712,
    0.12798127981279814,
    0.12799127991279913,
    0.12800128001280012,
    0.12801128011280113,
    0.12802128021280212,
    0.12803128031280311,
    0.12804128041280413,
    0.12805128051280512,
    0.12806128061280614,
    0.12807128071280713,
    0.12808128081280812,
    0.12809128091280914,
    0.12810128101281013,
    0.12811128111281112,
    0.12812128121281213,
    0.12813128131281312,
    0.12814128141281414,
    0.12815128151281513,
    0.12816128161281612,
    0.12817128171281714,
    0.12818128181281813,
    0.12819128191281912,
    0.12820128201282013,
    0.12821128211282112,
    0.1282212822128221,
    0.12823128231282313,
    0.12824128241282412,
    0.12825128251282514,
    0.12826128261282613,
    0.12827128271282712,
    0.12828128281282813,
    0.12829128291282912,
    0.12830128301283011,
    0.12831128311283113,
    0.12832128321283212,
    0.12833128331283314,
    0.12834128341283413,
    0.12835128351283512,
    0.12836128361283614,
    0.12837128371283713,
    0.12838128381283812,
    0.12839128391283913,
    0.12840128401284012,
    0.12841128411284114,
    0.12842128421284213,
    0.12843128431284312,
    0.12844128441284414,
    0.12845128451284513,
    0.12846128461284612,
    0.12847128471284713,
    0.12848128481284812,
    0.1284912849128491,
    0.12850128501285013,
    0.12851128511285112,
    0.12852128521285214,
    0.12853128531285313,
    0.12854128541285412,
    0.12855128551285513,
    0.12856128561285612,
    0.12857128571285711,
    0.12858128581285813,
    0.12859128591285912,
    0.12860128601286014,
    0.12861128611286113,
    0.12862128621286212,
    0.12863128631286314,
    0.12864128641286413,
    0.12865128651286512,
    0.12866128661286613,
    0.12867128671286712,
    0.12868128681286814,
    0.12869128691286913,
    0.12870128701287012,
    0.12871128711287114,
    0.12872128721287213,
    0.12873128731287312,
    0.12874128741287413,
    0.12875128751287512,
    0.1287612876128761,
    0.12877128771287713,
    0.12878128781287812,
    0.12879128791287914,
    0.12880128801288013,
    0.12881128811288112,
    0.12882128821288213,
    0.12883128831288312,
    0.12884128841288411,
    0.12885128851288513,
    0.12886128861288612,
    0.12887128871288714,
    0.12888128881288813,
    0.12889128891288912,
    0.12890128901289014,
    0.12891128911289113,
    0.12892128921289212,
    0.12893128931289313,
    0.12894128941289412,
    0.12895128951289514,
    0.12896128961289613,
    0.12897128971289712,
    0.12898128981289814,
    0.12899128991289913,
    0.12900129001290012,
    0.12901129011290113,
    0.12902129021290212,
    0.1290312903129031,
    0.12904129041290413,
    0.12905129051290512,
    0.12906129061290614,
    0.12907129071290713,
    0.12908129081290812,
    0.12909129091290913,
    0.12910129101291012,
    0.12911129111291111,
    0.12912129121291213,
    0.12913129131291312,
    0.12914129141291414,
    0.12915129151291513,
    0.12916129161291612,
    0.12917129171291714,
    0.12918129181291813,
    0.12919129191291912,
    0.12920129201292013,
    0.12921129211292112,
    0.12922129221292214,
    0.12923129231292313,
    0.12924129241292412,
    0.12925129251292514,
    0.12926129261292613,
    0.12927129271292712,
    0.12928129281292813,
    0.12929129291292912,
    0.12930129301293014,
    0.12931129311293113,
    0.12932129321293212,
    0.12933129331293314,
    0.12934129341293413,
    0.12935129351293512,
    0.12936129361293613,
    0.12937129371293712,
    0.12938129381293811,
    0.12939129391293913,
    0.12940129401294012,
    0.12941129411294114,
    0.12942129421294213,
    0.12943129431294312,
    0.12944129441294414,
    0.12945129451294513,
    0.12946129461294612,
    0.12947129471294713,
    0.12948129481294812,
    0.12949129491294914,
    0.12950129501295013,
    0.12951129511295112,
    0.12952129521295214,
    0.12953129531295313,
    0.12954129541295412,
    0.12955129551295513,
    0.12956129561295612,
    0.12957129571295714,
    0.12958129581295813,
    0.12959129591295912,
    0.12960129601296014,
    0.12961129611296113,
    0.12962129621296212,
    0.12963129631296313,
    0.12964129641296412,
    0.12965129651296511,
    0.12966129661296613,
    0.12967129671296712,
    0.12968129681296814,
    0.12969129691296913,
    0.12970129701297012,
    0.12971129711297114,
    0.12972129721297213,
    0.12973129731297312,
    0.12974129741297413,
    0.12975129751297512,
    0.12976129761297614,
    0.12977129771297713,
    0.12978129781297812,
    0.12979129791297914,
    0.12980129801298013,
    0.12981129811298112,
    0.12982129821298213,
    0.12983129831298312,
    0.12984129841298414,
    0.12985129851298513,
    0.12986129861298612,
    0.12987129871298714,
    0.12988129881298813,
    0.12989129891298912,
    0.12990129901299013,
    0.12991129911299112,
    0.12992129921299211,
    0.12993129931299313,
    0.12994129941299412,
    0.12995129951299514,
    0.12996129961299613,
    0.12997129971299712,
    0.12998129981299814,
    0.12999129991299913,
    0.13000130001300012,
    0.13001130011300113,
    0.13002130021300212,
    0.13003130031300314,
    0.13004130041300413,
    0.13005130051300512,
    0.13006130061300614,
    0.13007130071300713,
    0.13008130081300812,
    0.13009130091300913,
    0.13010130101301012,
    0.13011130111301114,
    0.13012130121301213,
    0.13013130131301312,
    0.13014130141301414,
    0.13015130151301513,
    0.13016130161301612,
    0.13017130171301713,
    0.13018130181301812,
    0.13019130191301914,
    0.13020130201302013,
    0.13021130211302112,
    0.13022130221302214,
    0.13023130231302313,
    0.13024130241302412,
    0.13025130251302514,
    0.13026130261302613,
    0.13027130271302712,
    0.13028130281302813,
    0.13029130291302912,
    0.13030130301303014,
    0.13031130311303113,
    0.13032130321303212,
    0.13033130331303314,
    0.13034130341303413,
    0.13035130351303512,
    0.13036130361303613,
    0.13037130371303712,
    0.13038130381303814,
    0.13039130391303913,
    0.13040130401304012,
    0.13041130411304114,
    0.13042130421304213,
    0.13043130431304312,
    0.13044130441304413,
    0.13045130451304512,
    0.13046130461304614,
    0.13047130471304713,
    0.13048130481304812,
    0.13049130491304914,
    0.13050130501305013,
    0.13051130511305112,
    0.13052130521305214,
    0.13053130531305313,
    0.13054130541305412,
    0.13055130551305513,
    0.13056130561305612,
    0.13057130571305714,
    0.13058130581305813,
    0.13059130591305912,
    0.13060130601306014,
    0.13061130611306113,
    0.13062130621306212,
    0.13063130631306313,
    0.13064130641306412,
    0.13065130651306514,
    0.13066130661306613,
    0.13067130671306712,
    0.13068130681306814,
    0.13069130691306913,
    0.13070130701307012,
    0.13071130711307113,
    0.13072130721307212,
    0.13073130731307314,
    0.13074130741307413,
    0.13075130751307512,
    0.13076130761307614,
    0.13077130771307713,
    0.13078130781307812,
    0.13079130791307914,
    0.13080130801308013,
    0.13081130811308112,
    0.13082130821308213,
    0.13083130831308312,
    0.13084130841308414,
    0.13085130851308513,
    0.13086130861308612,
    0.13087130871308714,
    0.13088130881308813,
    0.13089130891308912,
    0.13090130901309013,
    0.13091130911309112,
    0.13092130921309214,
    0.13093130931309313,
    0.13094130941309412,
    0.13095130951309514,
    0.13096130961309613,
    0.13097130971309712,
    0.13098130981309813,
    0.13099130991309912,
    0.13100131001310014,
    0.13101131011310113,
    0.13102131021310212,
    0.13103131031310314,
    0.13104131041310413,
    0.13105131051310512,
    0.13106131061310614,
    0.13107131071310713,
    0.13108131081310812,
    0.13109131091310913,
    0.13110131101311012,
    0.13111131111311114,
    0.13112131121311213,
    0.13113131131311312,
    0.13114131141311414,
    0.13115131151311513,
    0.13116131161311612,
    0.13117131171311713,
    0.13118131181311812,
    0.13119131191311914,
    0.13120131201312013,
    0.13121131211312112,
    0.13122131221312214,
    0.13123131231312313,
    0.13124131241312412,
    0.13125131251312513,
    0.13126131261312612,
    0.13127131271312714,
    0.13128131281312813,
    0.13129131291312912,
    0.13130131301313014,
    0.13131131311313113,
    0.13132131321313212,
    0.13133131331313314,
    0.13134131341313413,
    0.13135131351313514,
    0.13136131361313613,
    0.13137131371313712,
    0.13138131381313814,
    0.13139131391313913,
    0.13140131401314012,
    0.13141131411314114,
    0.13142131421314213,
    0.13143131431314312,
    0.13144131441314413,
    0.13145131451314512,
    0.13146131461314614,
    0.13147131471314713,
    0.13148131481314812,
    0.13149131491314914,
    0.13150131501315013,
    0.13151131511315112,
    0.13152131521315213,
    0.13153131531315312,
    0.13154131541315414,
    0.13155131551315513,
    0.13156131561315612,
    0.13157131571315714,
    0.13158131581315813,
    0.13159131591315912,
    0.13160131601316014,
    0.13161131611316113,
    0.13162131621316214,
    0.13163131631316313,
    0.13164131641316412,
    0.13165131651316514,
    0.13166131661316613,
    0.13167131671316712,
    0.13168131681316814,
    0.13169131691316913,
    0.13170131701317012,
    0.13171131711317113,
    0.13172131721317212,
    0.13173131731317314,
    0.13174131741317413,
    0.13175131751317512,
    0.13176131761317614,
    0.13177131771317713,
    0.13178131781317812,
    0.13179131791317913,
    0.13180131801318012,
    0.13181131811318114,
    0.13182131821318213,
    0.13183131831318312,
    0.13184131841318414,
    0.13185131851318513,
    0.13186131861318612,
    0.13187131871318714,
    0.13188131881318813,
    0.13189131891318914,
    0.13190131901319013,
    0.13191131911319112,
    0.13192131921319214,
    0.13193131931319313,
    0.13194131941319412,
    0.13195131951319514,
    0.13196131961319613,
    0.13197131971319712,
    0.13198131981319813,
    0.13199131991319912,
    0.13200132001320014,
    0.13201132011320113,
    0.13202132021320212,
    0.13203132031320314,
    0.13204132041320413,
    0.13205132051320512,
    0.13206132061320613,
    0.13207132071320712,
    0.13208132081320814,
    0.13209132091320913,
    0.13210132101321012,
    0.13211132111321114,
    0.13212132121321213,
    0.13213132131321312,
    0.13214132141321414,
    0.13215132151321513,
    0.13216132161321614,
    0.13217132171321713,
    0.13218132181321812,
    0.13219132191321914,
    0.13220132201322013,
    0.13221132211322112,
    0.13222132221322214,
    0.13223132231322313,
    0.13224132241322414,
    0.13225132251322513,
    0.13226132261322612,
    0.13227132271322714,
    0.13228132281322813,
    0.13229132291322912,
    0.13230132301323014,
    0.13231132311323113,
    0.13232132321323212,
    0.13233132331323313,
    0.13234132341323412,
    0.13235132351323514,
    0.13236132361323613,
    0.13237132371323712,
    0.13238132381323814,
    0.13239132391323913,
    0.13240132401324012,
    0.13241132411324114,
    0.13242132421324213,
    0.13243132431324314,
    0.13244132441324413,
    0.13245132451324512,
    0.13246132461324614,
    0.13247132471324713,
    0.13248132481324812,
    0.13249132491324914,
    0.13250132501325013,
    0.13251132511325114,
    0.13252132521325213,
    0.13253132531325312,
    0.13254132541325414,
    0.13255132551325513,
    0.13256132561325612,
    0.13257132571325714,
    0.13258132581325813,
    0.13259132591325912,
    0.13260132601326013,
    0.13261132611326112,
    0.13262132621326214,
    0.13263132631326313,
    0.13264132641326412,
    0.13265132651326514,
    0.13266132661326613,
    0.13267132671326712,
    0.13268132681326814,
    0.13269132691326913,
    0.13270132701327014,
    0.13271132711327113,
    0.13272132721327212,
    0.13273132731327314,
    0.13274132741327413,
    0.13275132751327512,
    0.13276132761327614,
    0.13277132771327713,
    0.13278132781327814,
    0.13279132791327913,
    0.13280132801328012,
    0.13281132811328114,
    0.13282132821328213,
    0.13283132831328312,
    0.13284132841328414,
    0.13285132851328513,
    0.13286132861328612,
    0.13287132871328713,
    0.13288132881328812,
    0.13289132891328914,
    0.13290132901329013,
    0.13291132911329112,
    0.13292132921329214,
    0.13293132931329313,
    0.13294132941329412,
    0.13295132951329514,
    0.13296132961329613,
    0.13297132971329714,
    0.13298132981329813,
    0.13299132991329912,
    0.13300133001330014,
    0.13301133011330113,
    0.13302133021330212,
    0.13303133031330314,
    0.13304133041330413,
    0.13305133051330514,
    0.13306133061330613,
    0.13307133071330712,
    0.13308133081330814,
    0.13309133091330913,
    0.13310133101331012,
    0.13311133111331114,
    0.13312133121331213,
    0.13313133131331312,
    0.13314133141331413,
    0.13315133151331512,
    0.13316133161331614,
    0.13317133171331713,
    0.13318133181331812,
    0.13319133191331914,
    0.13320133201332013,
    0.13321133211332112,
    0.13322133221332214,
    0.13323133231332313,
    0.13324133241332414,
    0.13325133251332513,
    0.13326133261332612,
    0.13327133271332714,
    0.13328133281332813,
    0.13329133291332912,
    0.13330133301333014,
    0.13331133311333113,
    0.13332133321333214,
    0.13333133331333313,
    0.13334133341333412,
    0.13335133351333514,
    0.13336133361333613,
    0.13337133371333712,
    0.13338133381333814,
    0.13339133391333913,
    0.13340133401334014,
    0.13341133411334113,
    0.13342133421334212,
    0.13343133431334314,
    0.13344133441334413,
    0.13345133451334512,
    0.13346133461334614,
    0.13347133471334713,
    0.13348133481334812,
    0.13349133491334914,
    0.13350133501335013,
    0.13351133511335114,
    0.13352133521335213,
    0.13353133531335312,
    0.13354133541335414,
    0.13355133551335513,
    0.13356133561335612,
    0.13357133571335714,
    0.13358133581335813,
    0.13359133591335914,
    0.13360133601336013,
    0.13361133611336112,
    0.13362133621336214,
    0.13363133631336313,
    0.13364133641336412,
    0.13365133651336514,
    0.13366133661336613,
    0.13367133671336714,
    0.13368133681336813,
    0.13369133691336912,
    0.13370133701337014,
    0.13371133711337113,
    0.13372133721337212,
    0.13373133731337314,
    0.13374133741337413,
    0.13375133751337512,
    0.13376133761337614,
    0.13377133771337713,
    0.13378133781337814,
    0.13379133791337913,
    0.13380133801338012,
    0.13381133811338114,
    0.13382133821338213,
    0.13383133831338312,
    0.13384133841338414,
    0.13385133851338513,
    0.13386133861338614,
    0.13387133871338713,
    0.13388133881338812,
    0.13389133891338914,
    0.13390133901339013,
    0.13391133911339112,
    0.13392133921339214,
    0.13393133931339313,
    0.13394133941339414,
    0.13395133951339513,
    0.13396133961339612,
    0.13397133971339714,
    0.13398133981339813,
    0.13399133991339912,
    0.13400134001340014,
    0.13401134011340113,
    0.13402134021340212,
    0.13403134031340314,
    0.13404134041340413,
    0.13405134051340514,
    0.13406134061340613,
    0.13407134071340712,
    0.13408134081340814,
    0.13409134091340913,
    0.13410134101341012,
    0.13411134111341114,
    0.13412134121341213,
    0.13413134131341314,
    0.13414134141341413,
    0.13415134151341512,
    0.13416134161341614,
    0.13417134171341713,
    0.13418134181341812,
    0.13419134191341914,
    0.13420134201342013,
    0.13421134211342114,
    0.13422134221342213,
    0.13423134231342312,
    0.13424134241342414,
    0.13425134251342513,
    0.13426134261342612,
    0.13427134271342714,
    0.13428134281342813,
    0.13429134291342915,
    0.13430134301343014,
    0.13431134311343113,
    0.13432134321343214,
    0.13433134331343313,
    0.13434134341343412,
    0.13435134351343514,
    0.13436134361343613,
    0.13437134371343712,
    0.13438134381343814,
    0.13439134391343913,
    0.13440134401344014,
    0.13441134411344113,
    0.13442134421344212,
    0.13443134431344314,
    0.13444134441344413,
    0.13445134451344512,
    0.13446134461344614,
    0.13447134471344713,
    0.13448134481344814,
    0.13449134491344913,
    0.13450134501345012,
    0.13451134511345114,
    0.13452134521345213,
    0.13453134531345312,
    0.13454134541345414,
    0.13455134551345513,
    0.13456134561345615,
    0.13457134571345714,
    0.13458134581345813,
    0.13459134591345914,
    0.13460134601346013,
    0.13461134611346112,
    0.13462134621346214,
    0.13463134631346313,
    0.13464134641346412,
    0.13465134651346514,
    0.13466134661346613,
    0.13467134671346714,
    0.13468134681346813,
    0.13469134691346912,
    0.13470134701347014,
    0.13471134711347113,
    0.13472134721347212,
    0.13473134731347314,
    0.13474134741347413,
    0.13475134751347514,
    0.13476134761347613,
    0.13477134771347712,
    0.13478134781347814,
    0.13479134791347913,
    0.13480134801348012,
    0.13481134811348114,
    0.13482134821348213,
    0.13483134831348315,
    0.13484134841348414,
    0.13485134851348513,
    0.13486134861348614,
    0.13487134871348713,
    0.13488134881348812,
    0.13489134891348914,
    0.13490134901349013,
    0.13491134911349112,
    0.13492134921349214,
    0.13493134931349313,
    0.13494134941349414,
    0.13495134951349513,
    0.13496134961349612,
    0.13497134971349714,
    0.13498134981349813,
    0.13499134991349912,
    0.13500135001350014,
    0.13501135011350113,
    0.13502135021350214,
    0.13503135031350313,
    0.13504135041350412,
    0.13505135051350514,
    0.13506135061350613,
    0.13507135071350712,
    0.13508135081350814,
    0.13509135091350913,
    0.13510135101351015,
    0.13511135111351114,
    0.13512135121351213,
    0.13513135131351314,
    0.13514135141351413,
    0.13515135151351512,
    0.13516135161351614,
    0.13517135171351713,
    0.13518135181351812,
    0.13519135191351914,
    0.13520135201352013,
    0.13521135211352114,
    0.13522135221352213,
    0.13523135231352312,
    0.13524135241352414,
    0.13525135251352513,
    0.13526135261352612,
    0.13527135271352714,
    0.13528135281352813,
    0.13529135291352914,
    0.13530135301353013,
    0.13531135311353112,
    0.13532135321353214,
    0.13533135331353313,
    0.13534135341353412,
    0.13535135351353514,
    0.13536135361353613,
    0.13537135371353715,
    0.13538135381353814,
    0.13539135391353913,
    0.13540135401354014,
    0.13541135411354113,
    0.13542135421354212,
    0.13543135431354314,
    0.13544135441354413,
    0.13545135451354515,
    0.13546135461354614,
    0.13547135471354713,
    0.13548135481354814,
    0.13549135491354913,
    0.13550135501355012,
    0.13551135511355114,
    0.13552135521355213,
    0.13553135531355312,
    0.13554135541355414,
    0.13555135551355513,
    0.13556135561355614,
    0.13557135571355713,
    0.13558135581355812,
    0.13559135591355914,
    0.13560135601356013,
    0.13561135611356112,
    0.13562135621356214,
    0.13563135631356313,
    0.13564135641356415,
    0.13565135651356514,
    0.13566135661356613,
    0.13567135671356714,
    0.13568135681356813,
    0.13569135691356912,
    0.13570135701357014,
    0.13571135711357113,
    0.13572135721357215,
    0.13573135731357314,
    0.13574135741357413,
    0.13575135751357514,
    0.13576135761357613,
    0.13577135771357712,
    0.13578135781357814,
    0.13579135791357913,
    0.13580135801358012,
    0.13581135811358114,
    0.13582135821358213,
    0.13583135831358314,
    0.13584135841358413,
    0.13585135851358512,
    0.13586135861358614,
    0.13587135871358713,
    0.13588135881358812,
    0.13589135891358914,
    0.13590135901359013,
    0.13591135911359115,
    0.13592135921359214,
    0.13593135931359313,
    0.13594135941359414,
    0.13595135951359513,
    0.13596135961359612,
    0.13597135971359714,
    0.13598135981359813,
    0.13599135991359915,
    0.13600136001360014,
    0.13601136011360113,
    0.13602136021360214,
    0.13603136031360313,
    0.13604136041360412,
    0.13605136051360514,
    0.13606136061360613,
    0.13607136071360712,
    0.13608136081360814,
    0.13609136091360913,
    0.13610136101361014,
    0.13611136111361113,
    0.13612136121361212,
    0.13613136131361314,
    0.13614136141361413,
    0.13615136151361512,
    0.13616136161361614,
    0.13617136171361713,
    0.13618136181361815,
    0.13619136191361914,
    0.13620136201362013,
    0.13621136211362114,
    0.13622136221362213,
    0.13623136231362312,
    0.13624136241362414,
    0.13625136251362513,
    0.13626136261362615,
    0.13627136271362714,
    0.13628136281362813,
    0.13629136291362914,
    0.13630136301363013,
    0.13631136311363112,
    0.13632136321363214,
    0.13633136331363313,
    0.13634136341363412,
    0.13635136351363514,
    0.13636136361363613,
    0.13637136371363714,
    0.13638136381363813,
    0.13639136391363912,
    0.13640136401364014,
    0.13641136411364113,
    0.13642136421364212,
    0.13643136431364314,
    0.13644136441364413,
    0.13645136451364515,
    0.13646136461364614,
    0.13647136471364713,
    0.13648136481364814,
    0.13649136491364913,
    0.13650136501365012,
    0.13651136511365114,
    0.13652136521365213,
    0.13653136531365315,
    0.13654136541365414,
    0.13655136551365513,
    0.13656136561365614,
    0.13657136571365713,
    0.13658136581365812,
    0.13659136591365914,
    0.13660136601366013,
    0.13661136611366115,
    0.13662136621366214,
    0.13663136631366313,
    0.13664136641366414,
    0.13665136651366513,
    0.13666136661366612,
    0.13667136671366714,
    0.13668136681366813,
    0.13669136691366912,
    0.13670136701367014,
    0.13671136711367113,
    0.13672136721367215,
    0.13673136731367314,
    0.13674136741367413,
    0.13675136751367514,
    0.13676136761367613,
    0.13677136771367712,
    0.13678136781367814,
    0.13679136791367913,
    0.13680136801368015,
    0.13681136811368114,
    0.13682136821368213,
    0.13683136831368314,
    0.13684136841368413,
    0.13685136851368512,
    0.13686136861368614,
    0.13687136871368713,
    0.13688136881368815,
    0.13689136891368914,
    0.13690136901369013,
    0.13691136911369114,
    0.13692136921369213,
    0.13693136931369312,
    0.13694136941369414,
    0.13695136951369513,
    0.13696136961369612,
    0.13697136971369714,
    0.13698136981369813,
    0.13699136991369915,
    0.13700137001370014,
    0.13701137011370113,
    0.13702137021370214,
    0.13703137031370313,
    0.13704137041370412,
    0.13705137051370514,
    0.13706137061370613,
    0.13707137071370715,
    0.13708137081370814,
    0.13709137091370913,
    0.13710137101371014,
    0.13711137111371113,
    0.13712137121371212,
    0.13713137131371314,
    0.13714137141371413,
    0.13715137151371515,
    0.13716137161371614,
    0.13717137171371713,
    0.13718137181371814,
    0.13719137191371913,
    0.13720137201372012,
    0.13721137211372114,
    0.13722137221372213,
    0.13723137231372312,
    0.13724137241372414,
    0.13725137251372513,
    0.13726137261372615,
    0.13727137271372714,
    0.13728137281372813,
    0.13729137291372914,
    0.13730137301373013,
    0.13731137311373112,
    0.13732137321373214,
    0.13733137331373313,
    0.13734137341373415,
    0.13735137351373514,
    0.13736137361373613,
    0.13737137371373714,
    0.13738137381373813,
    0.13739137391373912,
    0.13740137401374014,
    0.13741137411374113,
    0.13742137421374215,
    0.13743137431374314,
    0.13744137441374413,
    0.13745137451374514,
    0.13746137461374613,
    0.13747137471374712,
    0.13748137481374814,
    0.13749137491374913,
    0.13750137501375015,
    0.13751137511375114,
    0.13752137521375213,
    0.13753137531375315,
    0.13754137541375414,
    0.13755137551375513,
    0.13756137561375614,
    0.13757137571375713,
    0.13758137581375812,
    0.13759137591375914,
    0.13760137601376013,
    0.13761137611376115,
    0.13762137621376214,
    0.13763137631376313,
    0.13764137641376414,
    0.13765137651376513,
    0.13766137661376612,
    0.13767137671376714,
    0.13768137681376813,
    0.13769137691376915,
    0.13770137701377014,
    0.13771137711377113,
    0.13772137721377214,
    0.13773137731377313,
    0.13774137741377412,
    0.13775137751377514,
    0.13776137761377613,
    0.13777137771377715,
    0.13778137781377814,
    0.13779137791377913,
    0.13780137801378015,
    0.13781137811378114,
    0.13782137821378213,
    0.13783137831378314,
    0.13784137841378413,
    0.13785137851378512,
    0.13786137861378614,
    0.13787137871378713,
    0.13788137881378815,
    0.13789137891378914,
    0.13790137901379013,
    0.13791137911379114,
    0.13792137921379213,
    0.13793137931379312,
    0.13794137941379414,
    0.13795137951379513,
    0.13796137961379615,
    0.13797137971379714,
    0.13798137981379813,
    0.13799137991379914,
    0.13800138001380013,
    0.13801138011380112,
    0.13802138021380214,
    0.13803138031380313,
    0.13804138041380415,
    0.13805138051380514,
    0.13806138061380613,
    0.13807138071380715,
    0.13808138081380814,
    0.13809138091380913,
    0.13810138101381014,
    0.13811138111381113,
    0.13812138121381212,
    0.13813138131381314,
    0.13814138141381413,
    0.13815138151381515,
    0.13816138161381614,
    0.13817138171381713,
    0.13818138181381814,
    0.13819138191381913,
    0.13820138201382012,
    0.13821138211382114,
    0.13822138221382213,
    0.13823138231382315,
    0.13824138241382414,
    0.13825138251382513,
    0.13826138261382614,
    0.13827138271382713,
    0.13828138281382812,
    0.13829138291382914,
    0.13830138301383013,
    0.13831138311383115,
    0.13832138321383214,
    0.13833138331383313,
    0.13834138341383415,
    0.13835138351383514,
    0.13836138361383613,
    0.13837138371383714,
    0.13838138381383813,
    0.13839138391383912,
    0.13840138401384014,
    0.13841138411384113,
    0.13842138421384215,
    0.13843138431384314,
    0.13844138441384413,
    0.13845138451384514,
    0.13846138461384613,
    0.13847138471384712,
    0.13848138481384814,
    0.13849138491384913,
    0.13850138501385015,
    0.13851138511385114,
    0.13852138521385213,
    0.13853138531385314,
    0.13854138541385413,
    0.13855138551385512,
    0.13856138561385614,
    0.13857138571385713,
    0.13858138581385815,
    0.13859138591385914,
    0.13860138601386013,
    0.13861138611386115,
    0.13862138621386214,
    0.13863138631386313,
    0.13864138641386414,
    0.13865138651386513,
    0.13866138661386615,
    0.13867138671386714,
    0.13868138681386813,
    0.13869138691386915,
    0.13870138701387014,
    0.13871138711387113,
    0.13872138721387214,
    0.13873138731387313,
    0.13874138741387412,
    0.13875138751387514,
    0.13876138761387613,
    0.13877138771387715,
    0.13878138781387814,
    0.13879138791387913,
    0.13880138801388014,
    0.13881138811388113,
    0.13882138821388212,
    0.13883138831388314,
    0.13884138841388413,
    0.13885138851388515,
    0.13886138861388614,
    0.13887138871388713,
    0.13888138881388815,
    0.13889138891388914,
    0.13890138901389013,
    0.13891138911389114,
    0.13892138921389213,
    0.13893138931389315,
    0.13894138941389414,
    0.13895138951389513,
    0.13896138961389615,
    0.13897138971389714,
    0.13898138981389813,
    0.13899138991389914,
    0.13900139001390013,
    0.13901139011390112,
    0.13902139021390214,
    0.13903139031390313,
    0.13904139041390415,
    0.13905139051390514,
    0.13906139061390613,
    0.13907139071390714,
    0.13908139081390813,
    0.13909139091390912,
    0.13910139101391014,
    0.13911139111391113,
    0.13912139121391215,
    0.13913139131391314,
    0.13914139141391413,
    0.13915139151391515,
    0.13916139161391614,
    0.13917139171391713,
    0.13918139181391814,
    0.13919139191391913,
    0.13920139201392015,
    0.13921139211392114,
    0.13922139221392213,
    0.13923139231392315,
    0.13924139241392414,
    0.13925139251392513,
    0.13926139261392614,
    0.13927139271392713,
    0.13928139281392812,
    0.13929139291392914,
    0.13930139301393013,
    0.13931139311393115,
    0.13932139321393214,
    0.13933139331393313,
    0.13934139341393414,
    0.13935139351393513,
    0.13936139361393612,
    0.13937139371393714,
    0.13938139381393813,
    0.13939139391393915,
    0.13940139401394014,
    0.13941139411394113,
    0.13942139421394215,
    0.13943139431394314,
    0.13944139441394413,
    0.13945139451394514,
    0.13946139461394613,
    0.13947139471394715,
    0.13948139481394814,
    0.13949139491394913,
    0.13950139501395015,
    0.13951139511395114,
    0.13952139521395213,
    0.13953139531395314,
    0.13954139541395413,
    0.13955139551395515,
    0.13956139561395614,
    0.13957139571395713,
    0.13958139581395815,
    0.13959139591395914,
    0.13960139601396013,
    0.13961139611396114,
    0.13962139621396213,
    0.13963139631396312,
    0.13964139641396414,
    0.13965139651396513,
    0.13966139661396615,
    0.13967139671396714,
    0.13968139681396813,
    0.13969139691396915,
    0.13970139701397014,
    0.13971139711397113,
    0.13972139721397214,
    0.13973139731397313,
    0.13974139741397415,
    0.13975139751397514,
    0.13976139761397613,
    0.13977139771397715,
    0.13978139781397814,
    0.13979139791397913,
    0.13980139801398014,
    0.13981139811398113,
    0.13982139821398215,
    0.13983139831398314,
    0.13984139841398413,
    0.13985139851398515,
    0.13986139861398614,
    0.13987139871398713,
    0.13988139881398814,
    0.13989139891398913,
    0.13990139901399012,
    0.13991139911399114,
    0.13992139921399213,
    0.13993139931399315,
    0.13994139941399414,
    0.13995139951399513,
    0.13996139961399615,
    0.13997139971399714,
    0.13998139981399813,
    0.13999139991399914,
    0.14000140001400013,
    0.14001140011400115,
    0.14002140021400214,
    0.14003140031400313,
    0.14004140041400415,
    0.14005140051400514,
    0.14006140061400613,
    0.14007140071400714,
    0.14008140081400813,
    0.14009140091400915,
    0.14010140101401014,
    0.14011140111401113,
    0.14012140121401215,
    0.14013140131401314,
    0.14014140141401413,
    0.14015140151401514,
    0.14016140161401613,
    0.14017140171401712,
    0.14018140181401814,
    0.14019140191401913,
    0.14020140201402015,
    0.14021140211402114,
    0.14022140221402213,
    0.14023140231402315,
    0.14024140241402414,
    0.14025140251402513,
    0.14026140261402614,
    0.14027140271402713,
    0.14028140281402815,
    0.14029140291402914,
    0.14030140301403013,
    0.14031140311403115,
    0.14032140321403214,
    0.14033140331403313,
    0.14034140341403414,
    0.14035140351403513,
    0.14036140361403615,
    0.14037140371403714,
    0.14038140381403813,
    0.14039140391403915,
    0.14040140401404014,
    0.14041140411404113,
    0.14042140421404214,
    0.14043140431404313,
    0.14044140441404412,
    0.14045140451404514,
    0.14046140461404613,
    0.14047140471404715,
    0.14048140481404814,
    0.14049140491404913,
    0.14050140501405015,
    0.14051140511405114,
    0.14052140521405213,
    0.14053140531405314,
    0.14054140541405413,
    0.14055140551405515,
    0.14056140561405614,
    0.14057140571405713,
    0.14058140581405815,
    0.14059140591405914,
    0.14060140601406013,
    0.14061140611406114,
    0.14062140621406213,
    0.14063140631406315,
    0.14064140641406414,
    0.14065140651406513,
    0.14066140661406615,
    0.14067140671406714,
    0.14068140681406813,
    0.14069140691406914,
    0.14070140701407013,
    0.14071140711407115,
    0.14072140721407214,
    0.14073140731407313,
    0.14074140741407415,
    0.14075140751407514,
    0.14076140761407613,
    0.14077140771407715,
    0.14078140781407814,
    0.14079140791407913,
    0.14080140801408014,
    0.14081140811408113,
    0.14082140821408215,
    0.14083140831408314,
    0.14084140841408413,
    0.14085140851408515,
    0.14086140861408614,
    0.14087140871408713,
    0.14088140881408814,
    0.14089140891408913,
    0.14090140901409015,
    0.14091140911409114,
    0.14092140921409213,
    0.14093140931409315,
    0.14094140941409414,
    0.14095140951409513,
    0.14096140961409614,
    0.14097140971409713,
    0.14098140981409815,
    0.14099140991409914,
    0.14100141001410013,
    0.14101141011410115,
    0.14102141021410214,
    0.14103141031410313,
    0.14104141041410415,
    0.14105141051410514,
    0.14106141061410613,
    0.14107141071410714,
    0.14108141081410813,
    0.14109141091410915,
    0.14110141101411014,
    0.14111141111411113,
    0.14112141121411215,
    0.14113141131411314,
    0.14114141141411413,
    0.14115141151411514,
    0.14116141161411613,
    0.14117141171411715,
    0.14118141181411814,
    0.14119141191411913,
    0.14120141201412015,
    0.14121141211412114,
    0.14122141221412213,
    0.14123141231412314,
    0.14124141241412413,
    0.14125141251412515,
    0.14126141261412614,
    0.14127141271412713,
    0.14128141281412815,
    0.14129141291412914,
    0.14130141301413013,
    0.14131141311413115,
    0.14132141321413214,
    0.14133141331413313,
    0.14134141341413414,
    0.14135141351413513,
    0.14136141361413615,
    0.14137141371413714,
    0.14138141381413813,
    0.14139141391413915,
    0.14140141401414014,
    0.14141141411414113,
    0.14142141421414214,
    0.14143141431414313,
    0.14144141441414415,
    0.14145141451414514,
    0.14146141461414613,
    0.14147141471414715,
    0.14148141481414814,
    0.14149141491414913,
    0.14150141501415014,
    0.14151141511415113,
    0.14152141521415215,
    0.14153141531415314,
    0.14154141541415413,
    0.14155141551415515,
    0.14156141561415614,
    0.14157141571415713,
    0.14158141581415815,
    0.14159141591415914,
    0.14160141601416015,
    0.14161141611416114,
    0.14162141621416213,
    0.14163141631416315,
    0.14164141641416414,
    0.14165141651416513,
    0.14166141661416615,
    0.14167141671416714,
    0.14168141681416813,
    0.14169141691416914,
    0.14170141701417013,
    0.14171141711417115,
    0.14172141721417214,
    0.14173141731417313,
    0.14174141741417415,
    0.14175141751417514,
    0.14176141761417613,
    0.14177141771417714,
    0.14178141781417813,
    0.14179141791417915,
    0.14180141801418014,
    0.14181141811418113,
    0.14182141821418215,
    0.14183141831418314,
    0.14184141841418413,
    0.14185141851418515,
    0.14186141861418614,
    0.14187141871418715,
    0.14188141881418814,
    0.14189141891418913,
    0.14190141901419015,
    0.14191141911419114,
    0.14192141921419213,
    0.14193141931419315,
    0.14194141941419414,
    0.14195141951419513,
    0.14196141961419614,
    0.14197141971419713,
    0.14198141981419815,
    0.14199141991419914,
    0.14200142001420013,
    0.14201142011420115,
    0.14202142021420214,
    0.14203142031420313,
    0.14204142041420414,
    0.14205142051420513,
    0.14206142061420615,
    0.14207142071420714,
    0.14208142081420813,
    0.14209142091420915,
    0.14210142101421014,
    0.14211142111421113,
    0.14212142121421215,
    0.14213142131421314,
    0.14214142141421415,
    0.14215142151421514,
    0.14216142161421613,
    0.14217142171421715,
    0.14218142181421814,
    0.14219142191421913,
    0.14220142201422015,
    0.14221142211422114,
    0.14222142221422213,
    0.14223142231422314,
    0.14224142241422413,
    0.14225142251422515,
    0.14226142261422614,
    0.14227142271422713,
    0.14228142281422815,
    0.14229142291422914,
    0.14230142301423013,
    0.14231142311423114,
    0.14232142321423213,
    0.14233142331423315,
    0.14234142341423414,
    0.14235142351423513,
    0.14236142361423615,
    0.14237142371423714,
    0.14238142381423813,
    0.14239142391423915,
    0.14240142401424014,
    0.14241142411424115,
    0.14242142421424214,
    0.14243142431424313,
    0.14244142441424415,
    0.14245142451424514,
    0.14246142461424613,
    0.14247142471424715,
    0.14248142481424814,
    0.14249142491424913,
    0.14250142501425014,
    0.14251142511425113,
    0.14252142521425215,
    0.14253142531425314,
    0.14254142541425413,
    0.14255142551425515,
    0.14256142561425614,
    0.14257142571425713,
    0.14258142581425814,
    0.14259142591425913,
    0.14260142601426015,
    0.14261142611426114,
    0.14262142621426213,
    0.14263142631426315,
    0.14264142641426414,
    0.14265142651426513,
    0.14266142661426615,
    0.14267142671426714,
    0.14268142681426815,
    0.14269142691426914,
    0.14270142701427013,
    0.14271142711427115,
    0.14272142721427214,
    0.14273142731427313,
    0.14274142741427415,
    0.14275142751427514,
    0.14276142761427615,
    0.14277142771427714,
    0.14278142781427813,
    0.14279142791427915,
    0.14280142801428014,
    0.14281142811428113,
    0.14282142821428215,
    0.14283142831428314,
    0.14284142841428413,
    0.14285142851428514,
    0.14286142861428613,
    0.14287142871428715,
    0.14288142881428814,
    0.14289142891428913,
    0.14290142901429015,
    0.14291142911429114,
    0.14292142921429213,
    0.14293142931429315,
    0.14294142941429414,
    0.14295142951429515,
    0.14296142961429614,
    0.14297142971429713,
    0.14298142981429815,
    0.14299142991429914,
    0.14300143001430013,
    0.14301143011430115,
    0.14302143021430214,
    0.14303143031430315,
    0.14304143041430414,
    0.14305143051430513,
    0.14306143061430615,
    0.14307143071430714,
    0.14308143081430813,
    0.14309143091430915,
    0.14310143101431014,
    0.14311143111431113,
    0.14312143121431214,
    0.14313143131431313,
    0.14314143141431415,
    0.14315143151431514,
    0.14316143161431613,
    0.14317143171431715,
    0.14318143181431814,
    0.14319143191431913,
    0.14320143201432015,
    0.14321143211432114,
    0.14322143221432215,
    0.14323143231432314,
    0.14324143241432413,
    0.14325143251432515,
    0.14326143261432614,
    0.14327143271432713,
    0.14328143281432815,
    0.14329143291432914,
    0.14330143301433015,
    0.14331143311433114,
    0.14332143321433213,
    0.14333143331433315,
    0.14334143341433414,
    0.14335143351433513,
    0.14336143361433615,
    0.14337143371433714,
    0.14338143381433813,
    0.14339143391433914,
    0.14340143401434013,
    0.14341143411434115,
    0.14342143421434214,
    0.14343143431434313,
    0.14344143441434415,
    0.14345143451434514,
    0.14346143461434613,
    0.14347143471434715,
    0.14348143481434814,
    0.14349143491434915,
    0.14350143501435014,
    0.14351143511435113,
    0.14352143521435215,
    0.14353143531435314,
    0.14354143541435413,
    0.14355143551435515,
    0.14356143561435614,
    0.14357143571435715,
    0.14358143581435814,
    0.14359143591435913,
    0.14360143601436015,
    0.14361143611436114,
    0.14362143621436213,
    0.14363143631436315,
    0.14364143641436414,
    0.14365143651436516,
    0.14366143661436614,
    0.14367143671436713,
    0.14368143681436815,
    0.14369143691436914,
    0.14370143701437013,
    0.14371143711437115,
    0.14372143721437214,
    0.14373143731437313,
    0.14374143741437415,
    0.14375143751437514,
    0.14376143761437615,
    0.14377143771437714,
    0.14378143781437813,
    0.14379143791437915,
    0.14380143801438014,
    0.14381143811438113,
    0.14382143821438215,
    0.14383143831438314,
    0.14384143841438415,
    0.14385143851438514,
    0.14386143861438613,
    0.14387143871438715,
    0.14388143881438814,
    0.14389143891438913,
    0.14390143901439015,
    0.14391143911439114,
    0.14392143921439216,
    0.14393143931439314,
    0.14394143941439413,
    0.14395143951439515,
    0.14396143961439614,
    0.14397143971439713,
    0.14398143981439815,
    0.14399143991439914,
    0.14400144001440013,
    0.14401144011440115,
    0.14402144021440214,
    0.14403144031440315,
    0.14404144041440414,
    0.14405144051440513,
    0.14406144061440615,
    0.14407144071440714,
    0.14408144081440813,
    0.14409144091440915,
    0.14410144101441014,
    0.14411144111441115,
    0.14412144121441214,
    0.14413144131441313,
    0.14414144141441415,
    0.14415144151441514,
    0.14416144161441613,
    0.14417144171441715,
    0.14418144181441814,
    0.14419144191441916,
    0.14420144201442014,
    0.14421144211442113,
    0.14422144221442215,
    0.14423144231442314,
    0.14424144241442413,
    0.14425144251442515,
    0.14426144261442614,
    0.14427144271442713,
    0.14428144281442815,
    0.14429144291442914,
    0.14430144301443015,
    0.14431144311443114,
    0.14432144321443213,
    0.14433144331443315,
    0.14434144341443414,
    0.14435144351443513,
    0.14436144361443615,
    0.14437144371443714,
    0.14438144381443815,
    0.14439144391443914,
    0.14440144401444013,
    0.14441144411444115,
    0.14442144421444214,
    0.14443144431444313,
    0.14444144441444415,
    0.14445144451444514,
    0.14446144461444616,
    0.14447144471444714,
    0.14448144481444813,
    0.14449144491444915,
    0.14450144501445014,
    0.14451144511445113,
    0.14452144521445215,
    0.14453144531445314,
    0.14454144541445413,
    0.14455144551445515,
    0.14456144561445614,
    0.14457144571445715,
    0.14458144581445814,
    0.14459144591445913,
    0.14460144601446015,
    0.14461144611446114,
    0.14462144621446213,
    0.14463144631446315,
    0.14464144641446414,
    0.14465144651446515,
    0.14466144661446614,
    0.14467144671446713,
    0.14468144681446815,
    0.14469144691446914,
    0.14470144701447013,
    0.14471144711447115,
    0.14472144721447214,
    0.14473144731447316,
    0.14474144741447414,
    0.14475144751447513,
    0.14476144761447615,
    0.14477144771447714,
    0.14478144781447813,
    0.14479144791447915,
    0.14480144801448014,
    0.14481144811448116,
    0.14482144821448215,
    0.14483144831448314,
    0.14484144841448415,
    0.14485144851448514,
    0.14486144861448613,
    0.14487144871448715,
    0.14488144881448814,
    0.14489144891448913,
    0.14490144901449015,
    0.14491144911449114,
    0.14492144921449215,
    0.14493144931449314,
    0.14494144941449413,
    0.14495144951449515,
    0.14496144961449614,
    0.14497144971449713,
    0.14498144981449815,
    0.14499144991449914,
    0.14500145001450016,
    0.14501145011450114,
    0.14502145021450213,
    0.14503145031450315,
    0.14504145041450414,
    0.14505145051450513,
    0.14506145061450615,
    0.14507145071450714,
    0.14508145081450816,
    0.14509145091450915,
    0.14510145101451014,
    0.14511145111451115,
    0.14512145121451214,
    0.14513145131451313,
    0.14514145141451415,
    0.14515145151451514,
    0.14516145161451613,
    0.14517145171451715,
    0.14518145181451814,
    0.14519145191451915,
    0.14520145201452014,
    0.14521145211452113,
    0.14522145221452215,
    0.14523145231452314,
    0.14524145241452413,
    0.14525145251452515,
    0.14526145261452614,
    0.14527145271452716,
    0.14528145281452814,
    0.14529145291452913,
    0.14530145301453015,
    0.14531145311453114,
    0.14532145321453213,
    0.14533145331453315,
    0.14534145341453414,
    0.14535145351453516,
    0.14536145361453615,
    0.14537145371453714,
    0.14538145381453815,
    0.14539145391453914,
    0.14540145401454013,
    0.14541145411454115,
    0.14542145421454214,
    0.14543145431454313,
    0.14544145441454415,
    0.14545145451454514,
    0.14546145461454615,
    0.14547145471454714,
    0.14548145481454813,
    0.14549145491454915,
    0.14550145501455014,
    0.14551145511455113,
    0.14552145521455215,
    0.14553145531455314,
    0.14554145541455416,
    0.14555145551455514,
    0.14556145561455613,
    0.14557145571455715,
    0.14558145581455814,
    0.14559145591455913,
    0.14560145601456015,
    0.14561145611456114,
    0.14562145621456216,
    0.14563145631456315,
    0.14564145641456414,
    0.14565145651456515,
    0.14566145661456614,
    0.14567145671456713,
    0.14568145681456815,
    0.14569145691456914,
    0.14570145701457016,
    0.14571145711457115,
    0.14572145721457214,
    0.14573145731457315,
    0.14574145741457414,
    0.14575145751457513,
    0.14576145761457615,
    0.14577145771457714,
    0.14578145781457813,
    0.14579145791457915,
    0.14580145801458014,
    0.14581145811458116,
    0.14582145821458214,
    0.14583145831458313,
    0.14584145841458415,
    0.14585145851458514,
    0.14586145861458613,
    0.14587145871458715,
    0.14588145881458814,
    0.14589145891458916,
    0.14590145901459015,
    0.14591145911459114,
    0.14592145921459215,
    0.14593145931459314,
    0.14594145941459413,
    0.14595145951459515,
    0.14596145961459614,
    0.14597145971459716,
    0.14598145981459815,
    0.14599145991459914,
    0.14600146001460015,
    0.14601146011460114,
    0.14602146021460213,
    0.14603146031460315,
    0.14604146041460414,
    0.14605146051460513,
    0.14606146061460615,
    0.14607146071460714,
    0.14608146081460816,
    0.14609146091460914,
    0.14610146101461013,
    0.14611146111461115,
    0.14612146121461214,
    0.14613146131461313,
    0.14614146141461415,
    0.14615146151461514,
    0.14616146161461616,
    0.14617146171461715,
    0.14618146181461814,
    0.14619146191461915,
    0.14620146201462014,
    0.14621146211462113,
    0.14622146221462215,
    0.14623146231462314,
    0.14624146241462416,
    0.14625146251462515,
    0.14626146261462614,
    0.14627146271462715,
    0.14628146281462814,
    0.14629146291462913,
    0.14630146301463015,
    0.14631146311463114,
    0.14632146321463213,
    0.14633146331463315,
    0.14634146341463414,
    0.14635146351463516,
    0.14636146361463614,
    0.14637146371463713,
    0.14638146381463815,
    0.14639146391463914,
    0.14640146401464013,
    0.14641146411464115,
    0.14642146421464214,
    0.14643146431464316,
    0.14644146441464415,
    0.14645146451464514,
    0.14646146461464615,
    0.14647146471464714,
    0.14648146481464813,
    0.14649146491464915,
    0.14650146501465014,
    0.14651146511465116,
    0.14652146521465215,
    0.14653146531465314,
    0.14654146541465415,
    0.14655146551465514,
    0.14656146561465613,
    0.14657146571465715,
    0.14658146581465814,
    0.14659146591465913,
    0.14660146601466015,
    0.14661146611466114,
    0.14662146621466216,
    0.14663146631466314,
    0.14664146641466413,
    0.14665146651466515,
    0.14666146661466614,
    0.14667146671466713,
    0.14668146681466815,
    0.14669146691466914,
    0.14670146701467016,
    0.14671146711467115,
    0.14672146721467214,
    0.14673146731467315,
    0.14674146741467414,
    0.14675146751467513,
    0.14676146761467615,
    0.14677146771467714,
    0.14678146781467816,
    0.14679146791467915,
    0.14680146801468014,
    0.14681146811468115,
    0.14682146821468214,
    0.14683146831468313,
    0.14684146841468415,
    0.14685146851468514,
    0.14686146861468616,
    0.14687146871468715,
    0.14688146881468814,
    0.14689146891468916,
    0.14690146901469014,
    0.14691146911469113,
    0.14692146921469215,
    0.14693146931469314,
    0.14694146941469413,
    0.14695146951469515,
    0.14696146961469614,
    0.14697146971469716,
    0.14698146981469815,
    0.14699146991469914,
    0.14700147001470015,
    0.14701147011470114,
    0.14702147021470213,
    0.14703147031470315,
    0.14704147041470414,
    0.14705147051470516,
    0.14706147061470615,
    0.14707147071470714,
    0.14708147081470815,
    0.14709147091470914,
    0.14710147101471013,
    0.14711147111471115,
    0.14712147121471214,
    0.14713147131471316,
    0.14714147141471415,
    0.14715147151471514,
    0.14716147161471616,
    0.14717147171471714,
    0.14718147181471813,
    0.14719147191471915,
    0.14720147201472014,
    0.14721147211472113,
    0.14722147221472215,
    0.14723147231472314,
    0.14724147241472416,
    0.14725147251472515,
    0.14726147261472614,
    0.14727147271472715,
    0.14728147281472814,
    0.14729147291472913,
    0.14730147301473015,
    0.14731147311473114,
    0.14732147321473216,
    0.14733147331473315,
    0.14734147341473414,
    0.14735147351473515,
    0.14736147361473614,
    0.14737147371473713,
    0.14738147381473815,
    0.14739147391473914,
    0.14740147401474016,
    0.14741147411474115,
    0.14742147421474214,
    0.14743147431474316,
    0.14744147441474414,
    0.14745147451474513,
    0.14746147461474615,
    0.14747147471474714,
    0.14748147481474813,
    0.14749147491474915,
    0.14750147501475014,
    0.14751147511475116,
    0.14752147521475215,
    0.14753147531475314,
    0.14754147541475415,
    0.14755147551475514,
    0.14756147561475613,
    0.14757147571475715,
    0.14758147581475814,
    0.14759147591475916,
    0.14760147601476015,
    0.14761147611476114,
    0.14762147621476215,
    0.14763147631476314,
    0.14764147641476413,
    0.14765147651476515,
    0.14766147661476614,
    0.14767147671476716,
    0.14768147681476815,
    0.14769147691476914,
    0.14770147701477016,
    0.14771147711477114,
    0.14772147721477213,
    0.14773147731477315,
    0.14774147741477414,
    0.14775147751477513,
    0.14776147761477615,
    0.14777147771477714,
    0.14778147781477816,
    0.14779147791477915,
    0.14780147801478014,
    0.14781147811478115,
    0.14782147821478214,
    0.14783147831478313,
    0.14784147841478415,
    0.14785147851478514,
    0.14786147861478616,
    0.14787147871478715,
    0.14788147881478814,
    0.14789147891478915,
    0.14790147901479014,
    0.14791147911479113,
    0.14792147921479215,
    0.14793147931479314,
    0.14794147941479416,
    0.14795147951479515,
    0.14796147961479614,
    0.14797147971479716,
    0.14798147981479814,
    0.14799147991479913,
    0.14800148001480015,
    0.14801148011480114,
    0.14802148021480216,
    0.14803148031480315,
    0.14804148041480414,
    0.14805148051480516,
    0.14806148061480615,
    0.14807148071480714,
    0.14808148081480815,
    0.14809148091480914,
    0.14810148101481013,
    0.14811148111481115,
    0.14812148121481214,
    0.14813148131481316,
    0.14814148141481415,
    0.14815148151481514,
    0.14816148161481615,
    0.14817148171481714,
    0.14818148181481813,
    0.14819148191481915,
    0.14820148201482014,
    0.14821148211482116,
    0.14822148221482215,
    0.14823148231482314,
    0.14824148241482416,
    0.14825148251482514,
    0.14826148261482613,
    0.14827148271482715,
    0.14828148281482814,
    0.14829148291482916,
    0.14830148301483015,
    0.14831148311483114,
    0.14832148321483216,
    0.14833148331483315,
    0.14834148341483414,
    0.14835148351483515,
    0.14836148361483614,
    0.14837148371483713,
    0.14838148381483815,
    0.14839148391483914,
    0.14840148401484016,
    0.14841148411484115,
    0.14842148421484214,
    0.14843148431484315,
    0.14844148441484414,
    0.14845148451484513,
    0.14846148461484615,
    0.14847148471484714,
    0.14848148481484816,
    0.14849148491484915,
    0.14850148501485014,
    0.14851148511485115,
    0.14852148521485214,
    0.14853148531485313,
    0.14854148541485415,
    0.14855148551485514,
    0.14856148561485616,
    0.14857148571485715,
    0.14858148581485814,
    0.14859148591485916,
    0.14860148601486015,
    0.14861148611486114,
    0.14862148621486215,
    0.14863148631486314,
    0.14864148641486413,
    0.14865148651486515,
    0.14866148661486614,
    0.14867148671486716,
    0.14868148681486815,
    0.14869148691486914,
    0.14870148701487015,
    0.14871148711487114,
    0.14872148721487213,
    0.14873148731487315,
    0.14874148741487414,
    0.14875148751487516,
    0.14876148761487615,
    0.14877148771487714,
    0.14878148781487815,
    0.14879148791487914,
    0.14880148801488013,
    0.14881148811488115,
    0.14882148821488214,
    0.14883148831488316,
    0.14884148841488415,
    0.14885148851488514,
    0.14886148861488616,
    0.14887148871488715,
    0.14888148881488814,
    0.14889148891488915,
    0.14890148901489014,
    0.14891148911489116,
    0.14892148921489215,
    0.14893148931489314,
    0.14894148941489416,
    0.14895148951489515,
    0.14896148961489614,
    0.14897148971489715,
    0.14898148981489814,
    0.14899148991489913,
    0.14900149001490015,
    0.14901149011490114,
    0.14902149021490216,
    0.14903149031490315,
    0.14904149041490414,
    0.14905149051490515,
    0.14906149061490614,
    0.14907149071490713,
    0.14908149081490815,
    0.14909149091490914,
    0.14910149101491016,
    0.14911149111491115,
    0.14912149121491214,
    0.14913149131491316,
    0.14914149141491415,
    0.14915149151491514,
    0.14916149161491615,
    0.14917149171491714,
    0.14918149181491816,
    0.14919149191491915,
    0.14920149201492014,
    0.14921149211492116,
    0.14922149221492215,
    0.14923149231492314,
    0.14924149241492415,
    0.14925149251492514,
    0.14926149261492613,
    0.14927149271492715,
    0.14928149281492814,
    0.14929149291492916,
    0.14930149301493015,
    0.14931149311493114,
    0.14932149321493215,
    0.14933149331493314,
    0.14934149341493413,
    0.14935149351493515,
    0.14936149361493614,
    0.14937149371493716,
    0.14938149381493815,
    0.14939149391493914,
    0.14940149401494016,
    0.14941149411494115,
    0.14942149421494214,
    0.14943149431494315,
    0.14944149441494414,
    0.14945149451494516,
    0.14946149461494615,
    0.14947149471494714,
    0.14948149481494816,
    0.14949149491494915,
    0.14950149501495014,
    0.14951149511495115,
    0.14952149521495214,
    0.14953149531495313,
    0.14954149541495415,
    0.14955149551495514,
    0.14956149561495616,
    0.14957149571495715,
    0.14958149581495814,
    0.14959149591495915,
    0.14960149601496014,
    0.14961149611496113,
    0.14962149621496215,
    0.14963149631496314,
    0.14964149641496416,
    0.14965149651496515,
    0.14966149661496614,
    0.14967149671496716,
    0.14968149681496815,
    0.14969149691496914,
    0.14970149701497015,
    0.14971149711497114,
    0.14972149721497216,
    0.14973149731497315,
    0.14974149741497414,
    0.14975149751497516,
    0.14976149761497615,
    0.14977149771497714,
    0.14978149781497815,
    0.14979149791497914,
    0.14980149801498013,
    0.14981149811498115,
    0.14982149821498214,
    0.14983149831498316,
    0.14984149841498415,
    0.14985149851498514,
    0.14986149861498615,
    0.14987149871498714,
    0.14988149881498813,
    0.14989149891498915,
    0.14990149901499014,
    0.14991149911499116,
    0.14992149921499215,
    0.14993149931499314,
    0.14994149941499416,
    0.14995149951499515,
    0.14996149961499614,
    0.14997149971499715,
    0.14998149981499814,
    0.14999149991499916,
    0.15000150001500015,
    0.15001150011500114,
    0.15002150021500216,
    0.15003150031500315,
    0.15004150041500414,
    0.15005150051500515,
    0.15006150061500614,
    0.15007150071500716,
    0.15008150081500815,
    0.15009150091500914,
    0.15010150101501016,
    0.15011150111501115,
    0.15012150121501214,
    0.15013150131501315,
    0.15014150141501414,
    0.15015150151501513,
    0.15016150161501615,
    0.15017150171501714,
    0.15018150181501816,
    0.15019150191501915,
    0.15020150201502014,
    0.15021150211502116,
    0.15022150221502215,
    0.15023150231502314,
    0.15024150241502415,
    0.15025150251502514,
    0.15026150261502616,
    0.15027150271502715,
    0.15028150281502814,
    0.15029150291502916,
    0.15030150301503015,
    0.15031150311503114,
    0.15032150321503215,
    0.15033150331503314,
    0.15034150341503416,
    0.15035150351503515,
    0.15036150361503614,
    0.15037150371503716,
    0.15038150381503815,
    0.15039150391503914,
    0.15040150401504015,
    0.15041150411504114,
    0.15042150421504213,
    0.15043150431504315,
    0.15044150441504414,
    0.15045150451504516,
    0.15046150461504615,
    0.15047150471504714,
    0.15048150481504816,
    0.15049150491504915,
    0.15050150501505014,
    0.15051150511505115,
    0.15052150521505214,
    0.15053150531505316,
    0.15054150541505415,
    0.15055150551505514,
    0.15056150561505616,
    0.15057150571505715,
    0.15058150581505814,
    0.15059150591505915,
    0.15060150601506014,
    0.15061150611506116,
    0.15062150621506215,
    0.15063150631506314,
    0.15064150641506416,
    0.15065150651506515,
    0.15066150661506614,
    0.15067150671506715,
    0.15068150681506814,
    0.15069150691506913,
    0.15070150701507015,
    0.15071150711507114,
    0.15072150721507216,
    0.15073150731507315,
    0.15074150741507414,
    0.15075150751507516,
    0.15076150761507615,
    0.15077150771507714,
    0.15078150781507815,
    0.15079150791507914,
    0.15080150801508016,
    0.15081150811508115,
    0.15082150821508214,
    0.15083150831508316,
    0.15084150841508415,
    0.15085150851508514,
    0.15086150861508615,
    0.15087150871508714,
    0.15088150881508816,
    0.15089150891508915,
    0.15090150901509014,
    0.15091150911509116,
    0.15092150921509215,
    0.15093150931509314,
    0.15094150941509415,
    0.15095150951509514,
    0.15096150961509616,
    0.15097150971509715,
    0.15098150981509814,
    0.15099150991509916,
    0.15100151001510015,
    0.15101151011510114,
    0.15102151021510216,
    0.15103151031510315,
    0.15104151041510414,
    0.15105151051510515,
    0.15106151061510614,
    0.15107151071510716,
    0.15108151081510815,
    0.15109151091510914,
    0.15110151101511016,
    0.15111151111511115,
    0.15112151121511214,
    0.15113151131511315,
    0.15114151141511414,
    0.15115151151511516,
    0.15116151161511615,
    0.15117151171511714,
    0.15118151181511816,
    0.15119151191511915,
    0.15120151201512014,
    0.15121151211512115,
    0.15122151221512214,
    0.15123151231512316,
    0.15124151241512415,
    0.15125151251512514,
    0.15126151261512616,
    0.15127151271512715,
    0.15128151281512814,
    0.15129151291512916,
    0.15130151301513015,
    0.15131151311513114,
    0.15132151321513215,
    0.15133151331513314,
    0.15134151341513416,
    0.15135151351513515,
    0.15136151361513614,
    0.15137151371513716,
    0.15138151381513815,
    0.15139151391513914,
    0.15140151401514015,
    0.15141151411514114,
    0.15142151421514216,
    0.15143151431514315,
    0.15144151441514414,
    0.15145151451514516,
    0.15146151461514615,
    0.15147151471514714,
    0.15148151481514815,
    0.15149151491514914,
    0.15150151501515016,
    0.15151151511515115,
    0.15152151521515214,
    0.15153151531515316,
    0.15154151541515415,
    0.15155151551515514,
    0.15156151561515616,
    0.15157151571515715,
    0.15158151581515814,
    0.15159151591515915,
    0.15160151601516014,
    0.15161151611516116,
    0.15162151621516215,
    0.15163151631516314,
    0.15164151641516416,
    0.15165151651516515,
    0.15166151661516614,
    0.15167151671516715,
    0.15168151681516814,
    0.15169151691516916,
    0.15170151701517015,
    0.15171151711517114,
    0.15172151721517216,
    0.15173151731517315,
    0.15174151741517414,
    0.15175151751517515,
    0.15176151761517614,
    0.15177151771517716,
    0.15178151781517815,
    0.15179151791517914,
    0.15180151801518016,
    0.15181151811518115,
    0.15182151821518214,
    0.15183151831518316,
    0.15184151841518415,
    0.15185151851518514,
    0.15186151861518615,
    0.15187151871518714,
    0.15188151881518816,
    0.15189151891518915,
    0.15190151901519014,
    0.15191151911519116,
    0.15192151921519215,
    0.15193151931519314,
    0.15194151941519415,
    0.15195151951519514,
    0.15196151961519616,
    0.15197151971519715,
    0.15198151981519814,
    0.15199151991519916,
    0.15200152001520015,
    0.15201152011520114,
    0.15202152021520215,
    0.15203152031520314,
    0.15204152041520416,
    0.15205152051520515,
    0.15206152061520614,
    0.15207152071520716,
    0.15208152081520815,
    0.15209152091520914,
    0.15210152101521016,
    0.15211152111521115,
    0.15212152121521216,
    0.15213152131521315,
    0.15214152141521414,
    0.15215152151521516,
    0.15216152161521615,
    0.15217152171521714,
    0.15218152181521816,
    0.15219152191521915,
    0.15220152201522014,
    0.15221152211522115,
    0.15222152221522214,
    0.15223152231522316,
    0.15224152241522415,
    0.15225152251522514,
    0.15226152261522616,
    0.15227152271522715,
    0.15228152281522814,
    0.15229152291522915,
    0.15230152301523014,
    0.15231152311523116,
    0.15232152321523215,
    0.15233152331523314,
    0.15234152341523416,
    0.15235152351523515,
    0.15236152361523614,
    0.15237152371523716,
    0.15238152381523815,
    0.15239152391523916,
    0.15240152401524015,
    0.15241152411524114,
    0.15242152421524216,
    0.15243152431524315,
    0.15244152441524414,
    0.15245152451524516,
    0.15246152461524615,
    0.15247152471524714,
    0.15248152481524815,
    0.15249152491524914,
    0.15250152501525016,
    0.15251152511525115,
    0.15252152521525214,
    0.15253152531525316,
    0.15254152541525415,
    0.15255152551525514,
    0.15256152561525615,
    0.15257152571525714,
    0.15258152581525816,
    0.15259152591525915,
    0.15260152601526014,
    0.15261152611526116,
    0.15262152621526215,
    0.15263152631526314,
    0.15264152641526416,
    0.15265152651526515,
    0.15266152661526616,
    0.15267152671526715,
    0.15268152681526814,
    0.15269152691526916,
    0.15270152701527015,
    0.15271152711527114,
    0.15272152721527216,
    0.15273152731527315,
    0.15274152741527414,
    0.15275152751527515,
    0.15276152761527614,
    0.15277152771527716,
    0.15278152781527815,
    0.15279152791527914,
    0.15280152801528016,
    0.15281152811528115,
    0.15282152821528214,
    0.15283152831528315,
    0.15284152841528414,
    0.15285152851528516,
    0.15286152861528615,
    0.15287152871528714,
    0.15288152881528816,
    0.15289152891528915,
    0.15290152901529014,
    0.15291152911529116,
    0.15292152921529215,
    0.15293152931529316,
    0.15294152941529415,
    0.15295152951529514,
    0.15296152961529616,
    0.15297152971529715,
    0.15298152981529814,
    0.15299152991529916,
    0.15300153001530015,
    0.15301153011530116,
    0.15302153021530215,
    0.15303153031530314,
    0.15304153041530416,
    0.15305153051530515,
    0.15306153061530614,
    0.15307153071530716,
    0.15308153081530815,
    0.15309153091530914,
    0.15310153101531015,
    0.15311153111531114,
    0.15312153121531216,
    0.15313153131531315,
    0.15314153141531414,
    0.15315153151531516,
    0.15316153161531615,
    0.15317153171531714,
    0.15318153181531816,
    0.15319153191531915,
    0.15320153201532016,
    0.15321153211532115,
    0.15322153221532214,
    0.15323153231532316,
    0.15324153241532415,
    0.15325153251532514,
    0.15326153261532616,
    0.15327153271532715,
    0.15328153281532816,
    0.15329153291532915,
    0.15330153301533014,
    0.15331153311533116,
    0.15332153321533215,
    0.15333153331533314,
    0.15334153341533416,
    0.15335153351533515,
    0.15336153361533614,
    0.15337153371533715,
    0.15338153381533814,
    0.15339153391533916,
    0.15340153401534015,
    0.15341153411534114,
    0.15342153421534216,
    0.15343153431534315,
    0.15344153441534414,
    0.15345153451534516,
    0.15346153461534615,
    0.15347153471534716,
    0.15348153481534815,
    0.15349153491534914,
    0.15350153501535016,
    0.15351153511535115,
    0.15352153521535214,
    0.15353153531535316,
    0.15354153541535415,
    0.15355153551535516,
    0.15356153561535615,
    0.15357153571535714,
    0.15358153581535816,
    0.15359153591535915,
    0.15360153601536014,
    0.15361153611536116,
    0.15362153621536215,
    0.15363153631536314,
    0.15364153641536415,
    0.15365153651536514,
    0.15366153661536616,
    0.15367153671536715,
    0.15368153681536814,
    0.15369153691536916,
    0.15370153701537015,
    0.15371153711537114,
    0.15372153721537216,
    0.15373153731537315,
    0.15374153741537416,
    0.15375153751537515,
    0.15376153761537614,
    0.15377153771537716,
    0.15378153781537815,
    0.15379153791537914,
    0.15380153801538016,
    0.15381153811538115,
    0.15382153821538216,
    0.15383153831538315,
    0.15384153841538414,
    0.15385153851538516,
    0.15386153861538615,
    0.15387153871538714,
    0.15388153881538816,
    0.15389153891538915,
    0.15390153901539014,
    0.15391153911539115,
    0.15392153921539214,
    0.15393153931539316,
    0.15394153941539415,
    0.15395153951539514,
    0.15396153961539616,
    0.15397153971539715,
    0.15398153981539814,
    0.15399153991539916,
    0.15400154001540015,
    0.15401154011540116,
    0.15402154021540215,
    0.15403154031540314,
    0.15404154041540416,
    0.15405154051540515,
    0.15406154061540614,
    0.15407154071540716,
    0.15408154081540815,
    0.15409154091540916,
    0.15410154101541015,
    0.15411154111541114,
    0.15412154121541216,
    0.15413154131541315,
    0.15414154141541414,
    0.15415154151541516,
    0.15416154161541615,
    0.15417154171541717,
    0.15418154181541815,
    0.15419154191541914,
    0.15420154201542016,
    0.15421154211542115,
    0.15422154221542214,
    0.15423154231542316,
    0.15424154241542415,
    0.15425154251542514,
    0.15426154261542616,
    0.15427154271542715,
    0.15428154281542816,
    0.15429154291542915,
    0.15430154301543014,
    0.15431154311543116,
    0.15432154321543215,
    0.15433154331543314,
    0.15434154341543416,
    0.15435154351543515,
    0.15436154361543616,
    0.15437154371543715,
    0.15438154381543814,
    0.15439154391543916,
    0.15440154401544015,
    0.15441154411544114,
    0.15442154421544216,
    0.15443154431544315,
    0.15444154441544417,
    0.15445154451544515,
    0.15446154461544614,
    0.15447154471544716,
    0.15448154481544815,
    0.15449154491544914,
    0.15450154501545016,
    0.15451154511545115,
    0.15452154521545214,
    0.15453154531545316,
    0.15454154541545415,
    0.15455154551545516,
    0.15456154561545615,
    0.15457154571545714,
    0.15458154581545816,
    0.15459154591545915,
    0.15460154601546014,
    0.15461154611546116,
    0.15462154621546215,
    0.15463154631546316,
    0.15464154641546415,
    0.15465154651546514,
    0.15466154661546616,
    0.15467154671546715,
    0.15468154681546814,
    0.15469154691546916,
    0.15470154701547015,
    0.15471154711547117,
    0.15472154721547215,
    0.15473154731547314,
    0.15474154741547416,
    0.15475154751547515,
    0.15476154761547614,
    0.15477154771547716,
    0.15478154781547815,
    0.15479154791547914,
    0.15480154801548016,
    0.15481154811548115,
    0.15482154821548216,
    0.15483154831548315,
    0.15484154841548414,
    0.15485154851548516,
    0.15486154861548615,
    0.15487154871548714,
    0.15488154881548816,
    0.15489154891548915,
    0.15490154901549016,
    0.15491154911549115,
    0.15492154921549214,
    0.15493154931549316,
    0.15494154941549415,
    0.15495154951549514,
    0.15496154961549616,
    0.15497154971549715,
    0.15498154981549817,
    0.15499154991549915,
    0.15500155001550014,
    0.15501155011550116,
    0.15502155021550215,
    0.15503155031550314,
    0.15504155041550416,
    0.15505155051550515,
    0.15506155061550617,
    0.15507155071550716,
    0.15508155081550815,
    0.15509155091550916,
    0.15510155101551015,
    0.15511155111551114,
    0.15512155121551216,
    0.15513155131551315,
    0.15514155141551414,
    0.15515155151551516,
    0.15516155161551615,
    0.15517155171551716,
    0.15518155181551815,
    0.15519155191551914,
    0.15520155201552016,
    0.15521155211552115,
    0.15522155221552214,
    0.15523155231552316,
    0.15524155241552415,
    0.15525155251552517,
    0.15526155261552615,
    0.15527155271552714,
    0.15528155281552816,
    0.15529155291552915,
    0.15530155301553014,
    0.15531155311553116,
    0.15532155321553215,
    0.15533155331553317,
    0.15534155341553416,
    0.15535155351553515,
    0.15536155361553616,
    0.15537155371553715,
    0.15538155381553814,
    0.15539155391553916,
    0.15540155401554015,
    0.15541155411554114,
    0.15542155421554216,
    0.15543155431554315,
    0.15544155441554416,
    0.15545155451554515,
    0.15546155461554614,
    0.15547155471554716,
    0.15548155481554815,
    0.15549155491554914,
    0.15550155501555016,
    0.15551155511555115,
    0.15552155521555217,
    0.15553155531555315,
    0.15554155541555414,
    0.15555155551555516,
    0.15556155561555615,
    0.15557155571555714,
    0.15558155581555816,
    0.15559155591555915,
    0.15560155601556017,
    0.15561155611556116,
    0.15562155621556215,
    0.15563155631556316,
    0.15564155641556415,
    0.15565155651556514,
    0.15566155661556616,
    0.15567155671556715,
    0.15568155681556814,
    0.15569155691556916,
    0.15570155701557015,
    0.15571155711557116,
    0.15572155721557215,
    0.15573155731557314,
    0.15574155741557416,
    0.15575155751557515,
    0.15576155761557614,
    0.15577155771557716,
    0.15578155781557815,
    0.15579155791557917,
    0.15580155801558015,
    0.15581155811558114,
    0.15582155821558216,
    0.15583155831558315,
    0.15584155841558414,
    0.15585155851558516,
    0.15586155861558615,
    0.15587155871558717,
    0.15588155881558816,
    0.15589155891558915,
    0.15590155901559016,
    0.15591155911559115,
    0.15592155921559214,
    0.15593155931559316,
    0.15594155941559415,
    0.15595155951559514,
    0.15596155961559616,
    0.15597155971559715,
    0.15598155981559816,
    0.15599155991559915,
    0.15600156001560014,
    0.15601156011560116,
    0.15602156021560215,
    0.15603156031560314,
    0.15604156041560416,
    0.15605156051560515,
    0.15606156061560617,
    0.15607156071560715,
    0.15608156081560814,
    0.15609156091560916,
    0.15610156101561015,
    0.15611156111561114,
    0.15612156121561216,
    0.15613156131561315,
    0.15614156141561417,
    0.15615156151561516,
    0.15616156161561615,
    0.15617156171561716,
    0.15618156181561815,
    0.15619156191561914,
    0.15620156201562016,
    0.15621156211562115,
    0.15622156221562217,
    0.15623156231562316,
    0.15624156241562415,
    0.15625156251562516,
    0.15626156261562615,
    0.15627156271562714,
    0.15628156281562816,
    0.15629156291562915,
    0.15630156301563014,
    0.15631156311563116,
    0.15632156321563215,
    0.15633156331563317,
    0.15634156341563415,
    0.15635156351563514,
    0.15636156361563616,
    0.15637156371563715,
    0.15638156381563814,
    0.15639156391563916,
    0.15640156401564015,
    0.15641156411564117,
    0.15642156421564216,
    0.15643156431564315,
    0.15644156441564416,
    0.15645156451564515,
    0.15646156461564614,
    0.15647156471564716,
    0.15648156481564815,
    0.15649156491564917,
    0.15650156501565016,
    0.15651156511565115,
    0.15652156521565216,
    0.15653156531565315,
    0.15654156541565414,
    0.15655156551565516,
    0.15656156561565615,
    0.15657156571565714,
    0.15658156581565816,
    0.15659156591565915,
    0.15660156601566017,
    0.15661156611566115,
    0.15662156621566214,
    0.15663156631566316,
    0.15664156641566415,
    0.15665156651566514,
    0.15666156661566616,
    0.15667156671566715,
    0.15668156681566817,
    0.15669156691566916,
    0.15670156701567015,
    0.15671156711567116,
    0.15672156721567215,
    0.15673156731567314,
    0.15674156741567416,
    0.15675156751567515,
    0.15676156761567617,
    0.15677156771567716,
    0.15678156781567815,
    0.15679156791567916,
    0.15680156801568015,
    0.15681156811568114,
    0.15682156821568216,
    0.15683156831568315,
    0.15684156841568414,
    0.15685156851568516,
    0.15686156861568615,
    0.15687156871568717,
    0.15688156881568815,
    0.15689156891568914,
    0.15690156901569016,
    0.15691156911569115,
    0.15692156921569214,
    0.15693156931569316,
    0.15694156941569415,
    0.15695156951569517,
    0.15696156961569616,
    0.15697156971569715,
    0.15698156981569816,
    0.15699156991569915,
    0.15700157001570014,
    0.15701157011570116,
    0.15702157021570215,
    0.15703157031570317,
    0.15704157041570416,
    0.15705157051570515,
    0.15706157061570616,
    0.15707157071570715,
    0.15708157081570814,
    0.15709157091570916,
    0.15710157101571015,
    0.15711157111571117,
    0.15712157121571216,
    0.15713157131571315,
    0.15714157141571417,
    0.15715157151571515,
    0.15716157161571614,
    0.15717157171571716,
    0.15718157181571815,
    0.15719157191571914,
    0.15720157201572016,
    0.15721157211572115,
    0.15722157221572217,
    0.15723157231572316,
    0.15724157241572415,
    0.15725157251572516,
    0.15726157261572615,
    0.15727157271572714,
    0.15728157281572816,
    0.15729157291572915,
    0.15730157301573017,
    0.15731157311573116,
    0.15732157321573215,
    0.15733157331573316,
    0.15734157341573415,
    0.15735157351573514,
    0.15736157361573616,
    0.15737157371573715,
    0.15738157381573817,
    0.15739157391573916,
    0.15740157401574015,
    0.15741157411574117,
    0.15742157421574215,
    0.15743157431574314,
    0.15744157441574416,
    0.15745157451574515,
    0.15746157461574614,
    0.15747157471574716,
    0.15748157481574815,
    0.15749157491574917,
    0.15750157501575016,
    0.15751157511575115,
    0.15752157521575216,
    0.15753157531575315,
    0.15754157541575414,
    0.15755157551575516,
    0.15756157561575615,
    0.15757157571575717,
    0.15758157581575816,
    0.15759157591575915,
    0.15760157601576016,
    0.15761157611576115,
    0.15762157621576214,
    0.15763157631576316,
    0.15764157641576415,
    0.15765157651576517,
    0.15766157661576616,
    0.15767157671576715,
    0.15768157681576817,
    0.15769157691576915,
    0.15770157701577014,
    0.15771157711577116,
    0.15772157721577215,
    0.15773157731577314,
    0.15774157741577416,
    0.15775157751577515,
    0.15776157761577617,
    0.15777157771577716,
    0.15778157781577815,
    0.15779157791577916,
    0.15780157801578015,
    0.15781157811578114,
    0.15782157821578216,
    0.15783157831578315,
    0.15784157841578417,
    0.15785157851578516,
    0.15786157861578615,
    0.15787157871578716,
    0.15788157881578815,
    0.15789157891578914,
    0.15790157901579016,
    0.15791157911579115,
    0.15792157921579217,
    0.15793157931579316,
    0.15794157941579415,
    0.15795157951579517,
    0.15796157961579615,
    0.15797157971579714,
    0.15798157981579816,
    0.15799157991579915,
    0.15800158001580014,
    0.15801158011580116,
    0.15802158021580215,
    0.15803158031580317,
    0.15804158041580416,
    0.15805158051580515,
    0.15806158061580616,
    0.15807158071580715,
    0.15808158081580814,
    0.15809158091580916,
    0.15810158101581015,
    0.15811158111581117,
    0.15812158121581216,
    0.15813158131581315,
    0.15814158141581416,
    0.15815158151581515,
    0.15816158161581614,
    0.15817158171581716,
    0.15818158181581815,
    0.15819158191581917,
    0.15820158201582016,
    0.15821158211582115,
    0.15822158221582217,
    0.15823158231582315,
    0.15824158241582414,
    0.15825158251582516,
    0.15826158261582615,
    0.15827158271582717,
    0.15828158281582816,
    0.15829158291582915,
    0.15830158301583017,
    0.15831158311583116,
    0.15832158321583215,
    0.15833158331583316,
    0.15834158341583415,
    0.15835158351583514,
    0.15836158361583616,
    0.15837158371583715,
    0.15838158381583817,
    0.15839158391583916,
    0.15840158401584015,
    0.15841158411584116,
    0.15842158421584215,
    0.15843158431584314,
    0.15844158441584416,
    0.15845158451584515,
    0.15846158461584617,
    0.15847158471584716,
    0.15848158481584815,
    0.15849158491584917,
    0.15850158501585015,
    0.15851158511585114,
    0.15852158521585216,
    0.15853158531585315,
    0.15854158541585417,
    0.15855158551585516,
    0.15856158561585615,
    0.15857158571585717,
    0.15858158581585816,
    0.15859158591585915,
    0.15860158601586016,
    0.15861158611586115,
    0.15862158621586214,
    0.15863158631586316,
    0.15864158641586415,
    0.15865158651586517,
    0.15866158661586616,
    0.15867158671586715,
    0.15868158681586816,
    0.15869158691586915,
    0.15870158701587014,
    0.15871158711587116,
    0.15872158721587215,
    0.15873158731587317,
    0.15874158741587416,
    0.15875158751587515,
    0.15876158761587617,
    0.15877158771587715,
    0.15878158781587814,
    0.15879158791587916,
    0.15880158801588015,
    0.15881158811588117,
    0.15882158821588216,
    0.15883158831588315,
    0.15884158841588417,
    0.15885158851588516,
    0.15886158861588615,
    0.15887158871588716,
    0.15888158881588815,
    0.15889158891588914,
    0.15890158901589016,
    0.15891158911589115,
    0.15892158921589217,
    0.15893158931589316,
    0.15894158941589415,
    0.15895158951589516,
    0.15896158961589615,
    0.15897158971589714,
    0.15898158981589816,
    0.15899158991589915,
    0.15900159001590017,
    0.15901159011590116,
    0.15902159021590215,
    0.15903159031590317,
    0.15904159041590415,
    0.15905159051590514,
    0.15906159061590616,
    0.15907159071590715,
    0.15908159081590817,
    0.15909159091590916,
    0.15910159101591015,
    0.15911159111591117,
    0.15912159121591216,
    0.15913159131591315,
    0.15914159141591416,
    0.15915159151591515,
    0.15916159161591614,
    0.15917159171591716,
    0.15918159181591815,
    0.15919159191591917,
    0.15920159201592016,
    0.15921159211592115,
    0.15922159221592216,
    0.15923159231592315,
    0.15924159241592414,
    0.15925159251592516,
    0.15926159261592615,
    0.15927159271592717,
    0.15928159281592816,
    0.15929159291592915,
    0.15930159301593017,
    0.15931159311593115,
    0.15932159321593214,
    0.15933159331593316,
    0.15934159341593415,
    0.15935159351593517,
    0.15936159361593616,
    0.15937159371593715,
    0.15938159381593817,
    0.15939159391593916,
    0.15940159401594015,
    0.15941159411594116,
    0.15942159421594215,
    0.15943159431594317,
    0.15944159441594416,
    0.15945159451594515,
    0.15946159461594617,
    0.15947159471594716,
    0.15948159481594815,
    0.15949159491594916,
    0.15950159501595015,
    0.15951159511595114,
    0.15952159521595216,
    0.15953159531595315,
    0.15954159541595417,
    0.15955159551595516,
    0.15956159561595615,
    0.15957159571595717,
    0.15958159581595815,
    0.15959159591595914,
    0.15960159601596016,
    0.15961159611596115,
    0.15962159621596217,
    0.15963159631596316,
    0.15964159641596415,
    0.15965159651596517,
    0.15966159661596616,
    0.15967159671596715,
    0.15968159681596816,
    0.15969159691596915,
    0.15970159701597017,
    0.15971159711597116,
    0.15972159721597215,
    0.15973159731597317,
    0.15974159741597416,
    0.15975159751597515,
    0.15976159761597616,
    0.15977159771597715,
    0.15978159781597814,
    0.15979159791597916,
    0.15980159801598015,
    0.15981159811598117,
    0.15982159821598216,
    0.15983159831598315,
    0.15984159841598417,
    0.15985159851598515,
    0.15986159861598614,
    0.15987159871598716,
    0.15988159881598815,
    0.15989159891598917,
    0.15990159901599016,
    0.15991159911599115,
    0.15992159921599217,
    0.15993159931599316,
    0.15994159941599415,
    0.15995159951599516,
    0.15996159961599615,
    0.15997159971599717,
    0.15998159981599816,
    0.15999159991599915,
    0.16000160001600017,
    0.16001160011600116,
    0.16002160021600215,
    0.16003160031600316,
    0.16004160041600415,
    0.16005160051600514,
    0.16006160061600616,
    0.16007160071600715,
    0.16008160081600817,
    0.16009160091600916,
    0.16010160101601015,
    0.16011160111601117,
    0.16012160121601215,
    0.16013160131601314,
    0.16014160141601416,
    0.16015160151601515,
    0.16016160161601617,
    0.16017160171601716,
    0.16018160181601815,
    0.16019160191601917,
    0.16020160201602016,
    0.16021160211602115,
    0.16022160221602216,
    0.16023160231602315,
    0.16024160241602417,
    0.16025160251602516,
    0.16026160261602615,
    0.16027160271602717,
    0.16028160281602816,
    0.16029160291602915,
    0.16030160301603016,
    0.16031160311603115,
    0.16032160321603217,
    0.16033160331603316,
    0.16034160341603415,
    0.16035160351603517,
    0.16036160361603616,
    0.16037160371603715,
    0.16038160381603817,
    0.16039160391603915,
    0.16040160401604014,
    0.16041160411604116,
    0.16042160421604215,
    0.16043160431604317,
    0.16044160441604416,
    0.16045160451604515,
    0.16046160461604617,
    0.16047160471604716,
    0.16048160481604815,
    0.16049160491604916,
    0.16050160501605015,
    0.16051160511605117,
    0.16052160521605216,
    0.16053160531605315,
    0.16054160541605417,
    0.16055160551605516,
    0.16056160561605615,
    0.16057160571605716,
    0.16058160581605815,
    0.16059160591605917,
    0.16060160601606016,
    0.16061160611606115,
    0.16062160621606217,
    0.16063160631606316,
    0.16064160641606415,
    0.16065160651606517,
    0.16066160661606615,
    0.16067160671606714,
    0.16068160681606816,
    0.16069160691606915,
    0.16070160701607017,
    0.16071160711607116,
    0.16072160721607215,
    0.16073160731607317,
    0.16074160741607416,
    0.16075160751607515,
    0.16076160761607616,
    0.16077160771607715,
    0.16078160781607817,
    0.16079160791607916,
    0.16080160801608015,
    0.16081160811608117,
    0.16082160821608216,
    0.16083160831608315,
    0.16084160841608416,
    0.16085160851608515,
    0.16086160861608617,
    0.16087160871608716,
    0.16088160881608815,
    0.16089160891608917,
    0.16090160901609016,
    0.16091160911609115,
    0.16092160921609217,
    0.16093160931609315,
    0.16094160941609414,
    0.16095160951609516,
    0.16096160961609615,
    0.16097160971609717,
    0.16098160981609816,
    0.16099160991609915,
    0.16100161001610017,
    0.16101161011610116,
    0.16102161021610215,
    0.16103161031610316,
    0.16104161041610415,
    0.16105161051610517,
    0.16106161061610616,
    0.16107161071610715,
    0.16108161081610817,
    0.16109161091610916,
    0.16110161101611015,
    0.16111161111611116,
    0.16112161121611215,
    0.16113161131611317,
    0.16114161141611416,
    0.16115161151611515,
    0.16116161161611617,
    0.16117161171611716,
    0.16118161181611815,
    0.16119161191611917,
    0.16120161201612015,
    0.16121161211612114,
    0.16122161221612216,
    0.16123161231612315,
    0.16124161241612417,
    0.16125161251612516,
    0.16126161261612615,
    0.16127161271612717,
    0.16128161281612816,
    0.16129161291612915,
    0.16130161301613016,
    0.16131161311613115,
    0.16132161321613217,
    0.16133161331613316,
    0.16134161341613415,
    0.16135161351613517,
    0.16136161361613616,
    0.16137161371613715,
    0.16138161381613816,
    0.16139161391613915,
    0.16140161401614017,
    0.16141161411614116,
    0.16142161421614215,
    0.16143161431614317,
    0.16144161441614416,
    0.16145161451614515,
    0.16146161461614617,
    0.16147161471614715,
    0.16148161481614817,
    0.16149161491614916,
    0.16150161501615015,
    0.16151161511615117,
    0.16152161521615216,
    0.16153161531615315,
    0.16154161541615417,
    0.16155161551615516,
    0.16156161561615615,
    0.16157161571615716,
    0.16158161581615815,
    0.16159161591615917,
    0.16160161601616016,
    0.16161161611616115,
    0.16162161621616217,
    0.16163161631616316,
    0.16164161641616415,
    0.16165161651616516,
    0.16166161661616615,
    0.16167161671616717,
    0.16168161681616816,
    0.16169161691616915,
    0.16170161701617017,
    0.16171161711617116,
    0.16172161721617215,
    0.16173161731617317,
    0.16174161741617415,
    0.16175161751617517,
    0.16176161761617616,
    0.16177161771617715,
    0.16178161781617817,
    0.16179161791617916,
    0.16180161801618015,
    0.16181161811618117,
    0.16182161821618216,
    0.16183161831618315,
    0.16184161841618416,
    0.16185161851618515,
    0.16186161861618617,
    0.16187161871618716,
    0.16188161881618815,
    0.16189161891618917,
    0.16190161901619016,
    0.16191161911619115,
    0.16192161921619216,
    0.16193161931619315,
    0.16194161941619417,
    0.16195161951619516,
    0.16196161961619615,
    0.16197161971619717,
    0.16198161981619816,
    0.16199161991619915,
    0.16200162001620017,
    0.16201162011620115,
    0.16202162021620217,
    0.16203162031620316,
    0.16204162041620415,
    0.16205162051620517,
    0.16206162061620616,
    0.16207162071620715,
    0.16208162081620817,
    0.16209162091620916,
    0.16210162101621015,
    0.16211162111621116,
    0.16212162121621215,
    0.16213162131621317,
    0.16214162141621416,
    0.16215162151621515,
    0.16216162161621617,
    0.16217162171621716,
    0.16218162181621815,
    0.16219162191621916,
    0.16220162201622015,
    0.16221162211622117,
    0.16222162221622216,
    0.16223162231622315,
    0.16224162241622417,
    0.16225162251622516,
    0.16226162261622615,
    0.16227162271622717,
    0.16228162281622815,
    0.16229162291622917,
    0.16230162301623016,
    0.16231162311623115,
    0.16232162321623217,
    0.16233162331623316,
    0.16234162341623415,
    0.16235162351623517,
    0.16236162361623616,
    0.16237162371623717,
    0.16238162381623816,
    0.16239162391623915,
    0.16240162401624017,
    0.16241162411624116,
    0.16242162421624215,
    0.16243162431624317,
    0.16244162441624416,
    0.16245162451624515,
    0.16246162461624616,
    0.16247162471624715,
    0.16248162481624817,
    0.16249162491624916,
    0.16250162501625015,
    0.16251162511625117,
    0.16252162521625216,
    0.16253162531625315,
    0.16254162541625417,
    0.16255162551625515,
    0.16256162561625617,
    0.16257162571625716,
    0.16258162581625815,
    0.16259162591625917,
    0.16260162601626016,
    0.16261162611626115,
    0.16262162621626217,
    0.16263162631626316,
    0.16264162641626417,
    0.16265162651626516,
    0.16266162661626615,
    0.16267162671626717,
    0.16268162681626816,
    0.16269162691626915,
    0.16270162701627017,
    0.16271162711627116,
    0.16272162721627215,
    0.16273162731627316,
    0.16274162741627415,
    0.16275162751627517,
    0.16276162761627616,
    0.16277162771627715,
    0.16278162781627817,
    0.16279162791627916,
    0.16280162801628015,
    0.16281162811628117,
    0.16282162821628215,
    0.16283162831628317,
    0.16284162841628416,
    0.16285162851628515,
    0.16286162861628617,
    0.16287162871628716,
    0.16288162881628815,
    0.16289162891628917,
    0.16290162901629016,
    0.16291162911629117,
    0.16292162921629216,
    0.16293162931629315,
    0.16294162941629417,
    0.16295162951629516,
    0.16296162961629615,
    0.16297162971629717,
    0.16298162981629816,
    0.16299162991629915,
    0.16300163001630016,
    0.16301163011630115,
    0.16302163021630217,
    0.16303163031630316,
    0.16304163041630415,
    0.16305163051630517,
    0.16306163061630616,
    0.16307163071630715,
    0.16308163081630817,
    0.16309163091630915,
    0.16310163101631017,
    0.16311163111631116,
    0.16312163121631215,
    0.16313163131631317,
    0.16314163141631416,
    0.16315163151631515,
    0.16316163161631617,
    0.16317163171631716,
    0.16318163181631817,
    0.16319163191631916,
    0.16320163201632015,
    0.16321163211632117,
    0.16322163221632216,
    0.16323163231632315,
    0.16324163241632417,
    0.16325163251632516,
    0.16326163261632615,
    0.16327163271632716,
    0.16328163281632815,
    0.16329163291632917,
    0.16330163301633016,
    0.16331163311633115,
    0.16332163321633217,
    0.16333163331633316,
    0.16334163341633415,
    0.16335163351633517,
    0.16336163361633615,
    0.16337163371633717,
    0.16338163381633816,
    0.16339163391633915,
    0.16340163401634017,
    0.16341163411634116,
    0.16342163421634215,
    0.16343163431634317,
    0.16344163441634416,
    0.16345163451634517,
    0.16346163461634616,
    0.16347163471634715,
    0.16348163481634817,
    0.16349163491634916,
    0.16350163501635015,
    0.16351163511635117,
    0.16352163521635216,
    0.16353163531635317,
    0.16354163541635416,
    0.16355163551635515,
    0.16356163561635617,
    0.16357163571635716,
    0.16358163581635815,
    0.16359163591635917,
    0.16360163601636016,
    0.16361163611636115,
    0.16362163621636217,
    0.16363163631636315,
    0.16364163641636417,
    0.16365163651636516,
    0.16366163661636615,
    0.16367163671636717,
    0.16368163681636816,
    0.16369163691636915,
    0.16370163701637017,
    0.16371163711637116,
    0.16372163721637217,
    0.16373163731637316,
    0.16374163741637415,
    0.16375163751637517,
    0.16376163761637616,
    0.16377163771637715,
    0.16378163781637817,
    0.16379163791637916,
    0.16380163801638017,
    0.16381163811638116,
    0.16382163821638215,
    0.16383163831638317,
    0.16384163841638416,
    0.16385163851638515,
    0.16386163861638617,
    0.16387163871638716,
    0.16388163881638815,
    0.16389163891638917,
    0.16390163901639015,
    0.16391163911639117,
    0.16392163921639216,
    0.16393163931639315,
    0.16394163941639417,
    0.16395163951639516,
    0.16396163961639615,
    0.16397163971639717,
    0.16398163981639816,
    0.16399163991639917,
    0.16400164001640016,
    0.16401164011640115,
    0.16402164021640217,
    0.16403164031640316,
    0.16404164041640415,
    0.16405164051640517,
    0.16406164061640616,
    0.16407164071640717,
    0.16408164081640816,
    0.16409164091640915,
    0.16410164101641017,
    0.16411164111641116,
    0.16412164121641215,
    0.16413164131641317,
    0.16414164141641416,
    0.16415164151641515,
    0.16416164161641617,
    0.16417164171641715,
    0.16418164181641817,
    0.16419164191641916,
    0.16420164201642015,
    0.16421164211642117,
    0.16422164221642216,
    0.16423164231642315,
    0.16424164241642417,
    0.16425164251642516,
    0.16426164261642617,
    0.16427164271642716,
    0.16428164281642815,
    0.16429164291642917,
    0.16430164301643016,
    0.16431164311643115,
    0.16432164321643217,
    0.16433164331643316,
    0.16434164341643417,
    0.16435164351643516,
    0.16436164361643615,
    0.16437164371643717,
    0.16438164381643816,
    0.16439164391643915,
    0.16440164401644017,
    0.16441164411644116,
    0.16442164421644218,
    0.16443164431644317,
    0.16444164441644415,
    0.16445164451644517,
    0.16446164461644616,
    0.16447164471644715,
    0.16448164481644817,
    0.16449164491644916,
    0.16450164501645015,
    0.16451164511645117,
    0.16452164521645216,
    0.16453164531645317,
    0.16454164541645416,
    0.16455164551645515,
    0.16456164561645617,
    0.16457164571645716,
    0.16458164581645815,
    0.16459164591645917,
    0.16460164601646016,
    0.16461164611646117,
    0.16462164621646216,
    0.16463164631646315,
    0.16464164641646417,
    0.16465164651646516,
    0.16466164661646615,
    0.16467164671646717,
    0.16468164681646816,
    0.16469164691646918,
    0.16470164701647017,
    0.16471164711647115,
    0.16472164721647217,
    0.16473164731647316,
    0.16474164741647415,
    0.16475164751647517,
    0.16476164761647616,
    0.16477164771647715,
    0.16478164781647817,
    0.16479164791647916,
    0.16480164801648017,
    0.16481164811648116,
    0.16482164821648215,
    0.16483164831648317,
    0.16484164841648416,
    0.16485164851648515,
    0.16486164861648617,
    0.16487164871648716,
    0.16488164881648817,
    0.16489164891648916,
    0.16490164901649015,
    0.16491164911649117,
    0.16492164921649216,
    0.16493164931649315,
    0.16494164941649417,
    0.16495164951649516,
    0.16496164961649618,
    0.16497164971649717,
    0.16498164981649815,
    0.16499164991649917,
    0.16500165001650016,
    0.16501165011650115,
    0.16502165021650217,
    0.16503165031650316,
    0.16504165041650415,
    0.16505165051650517,
    0.16506165061650616,
    0.16507165071650717,
    0.16508165081650816,
    0.16509165091650915,
    0.16510165101651017,
    0.16511165111651116,
    0.16512165121651215,
    0.16513165131651317,
    0.16514165141651416,
    0.16515165151651517,
    0.16516165161651616,
    0.16517165171651715,
    0.16518165181651817,
    0.16519165191651916,
    0.16520165201652015,
    0.16521165211652117,
    0.16522165221652216,
    0.16523165231652318,
    0.16524165241652417,
    0.16525165251652515,
    0.16526165261652617,
    0.16527165271652716,
    0.16528165281652815,
    0.16529165291652917,
    0.16530165301653016,
    0.16531165311653115,
    0.16532165321653217,
    0.16533165331653316,
    0.16534165341653417,
    0.16535165351653516,
    0.16536165361653615,
    0.16537165371653717,
    0.16538165381653816,
    0.16539165391653915,
    0.16540165401654017,
    0.16541165411654116,
    0.16542165421654217,
    0.16543165431654316,
    0.16544165441654415,
    0.16545165451654517,
    0.16546165461654616,
    0.16547165471654715,
    0.16548165481654817,
    0.16549165491654916,
    0.16550165501655018,
    0.16551165511655117,
    0.16552165521655215,
    0.16553165531655317,
    0.16554165541655416,
    0.16555165551655515,
    0.16556165561655617,
    0.16557165571655716,
    0.16558165581655818,
    0.16559165591655917,
    0.16560165601656016,
    0.16561165611656117,
    0.16562165621656216,
    0.16563165631656315,
    0.16564165641656417,
    0.16565165651656516,
    0.16566165661656615,
    0.16567165671656717,
    0.16568165681656816,
    0.16569165691656917,
    0.16570165701657016,
    0.16571165711657115,
    0.16572165721657217,
    0.16573165731657316,
    0.16574165741657415,
    0.16575165751657517,
    0.16576165761657616,
    0.16577165771657718,
    0.16578165781657817,
    0.16579165791657915,
    0.16580165801658017,
    0.16581165811658116,
    0.16582165821658215,
    0.16583165831658317,
    0.16584165841658416,
    0.16585165851658518,
    0.16586165861658617,
    0.16587165871658716,
    0.16588165881658817,
    0.16589165891658916,
    0.16590165901659015,
    0.16591165911659117,
    0.16592165921659216,
    0.16593165931659315,
    0.16594165941659417,
    0.16595165951659516,
    0.16596165961659617,
    0.16597165971659716,
    0.16598165981659815,
    0.16599165991659917,
    0.16600166001660016,
    0.16601166011660115,
    0.16602166021660217,
    0.16603166031660316,
    0.16604166041660418,
    0.16605166051660517,
    0.16606166061660615,
    0.16607166071660717,
    0.16608166081660816,
    0.16609166091660915,
    0.16610166101661017,
    0.16611166111661116,
    0.16612166121661218,
    0.16613166131661317,
    0.16614166141661416,
    0.16615166151661517,
    0.16616166161661616,
    0.16617166171661715,
    0.16618166181661817,
    0.16619166191661916,
    0.16620166201662015,
    0.16621166211662117,
    0.16622166221662216,
    0.16623166231662317,
    0.16624166241662416,
    0.16625166251662515,
    0.16626166261662617,
    0.16627166271662716,
    0.16628166281662815,
    0.16629166291662917,
    0.16630166301663016,
    0.16631166311663118,
    0.16632166321663217,
    0.16633166331663315,
    0.16634166341663417,
    0.16635166351663516,
    0.16636166361663615,
    0.16637166371663717,
    0.16638166381663816,
    0.16639166391663918,
    0.16640166401664017,
    0.16641166411664116,
    0.16642166421664217,
    0.16643166431664316,
    0.16644166441664415,
    0.16645166451664517,
    0.16646166461664616,
    0.16647166471664718,
    0.16648166481664817,
    0.16649166491664916,
    0.16650166501665017,
    0.16651166511665116,
    0.16652166521665215,
    0.16653166531665317,
    0.16654166541665416,
    0.16655166551665515,
    0.16656166561665617,
    0.16657166571665716,
    0.16658166581665818,
    0.16659166591665917,
    0.16660166601666015,
    0.16661166611666117,
    0.16662166621666216,
    0.16663166631666315,
    0.16664166641666417,
    0.16665166651666516,
    0.16666166661666618,
    0.16667166671666717,
    0.16668166681666816,
    0.16669166691666917,
    0.16670166701667016,
    0.16671166711667115,
    0.16672166721667217,
    0.16673166731667316,
    0.16674166741667418,
    0.16675166751667517,
    0.16676166761667616,
    0.16677166771667717,
    0.16678166781667816,
    0.16679166791667915,
    0.16680166801668017,
    0.16681166811668116,
    0.16682166821668215,
    0.16683166831668317,
    0.16684166841668416,
    0.16685166851668518,
    0.16686166861668617,
    0.16687166871668715,
    0.16688166881668817,
    0.16689166891668916,
    0.16690166901669015,
    0.16691166911669117,
    0.16692166921669216,
    0.16693166931669318,
    0.16694166941669417,
    0.16695166951669516,
    0.16696166961669617,
    0.16697166971669716,
    0.16698166981669815,
    0.16699166991669917,
    0.16700167001670016,
    0.16701167011670118,
    0.16702167021670217,
    0.16703167031670316,
    0.16704167041670417,
    0.16705167051670516,
    0.16706167061670615,
    0.16707167071670717,
    0.16708167081670816,
    0.16709167091670915,
    0.16710167101671017,
    0.16711167111671116,
    0.16712167121671218,
    0.16713167131671317,
    0.16714167141671415,
    0.16715167151671517,
    0.16716167161671616,
    0.16717167171671715,
    0.16718167181671817,
    0.16719167191671916,
    0.16720167201672018,
    0.16721167211672117,
    0.16722167221672216,
    0.16723167231672317,
    0.16724167241672416,
    0.16725167251672515,
    0.16726167261672617,
    0.16727167271672716,
    0.16728167281672818,
    0.16729167291672917,
    0.16730167301673016,
    0.16731167311673117,
    0.16732167321673216,
    0.16733167331673315,
    0.16734167341673417,
    0.16735167351673516,
    0.16736167361673615,
    0.16737167371673717,
    0.16738167381673816,
    0.16739167391673918,
    0.16740167401674017,
    0.16741167411674115,
    0.16742167421674217,
    0.16743167431674316,
    0.16744167441674415,
    0.16745167451674517,
    0.16746167461674616,
    0.16747167471674718,
    0.16748167481674817,
    0.16749167491674916,
    0.16750167501675017,
    0.16751167511675116,
    0.16752167521675215,
    0.16753167531675317,
    0.16754167541675416,
    0.16755167551675518,
    0.16756167561675617,
    0.16757167571675716,
    0.16758167581675817,
    0.16759167591675916,
    0.16760167601676015,
    0.16761167611676117,
    0.16762167621676216,
    0.16763167631676318,
    0.16764167641676417,
    0.16765167651676516,
    0.16766167661676618,
    0.16767167671676717,
    0.16768167681676815,
    0.16769167691676917,
    0.16770167701677016,
    0.16771167711677115,
    0.16772167721677217,
    0.16773167731677316,
    0.16774167741677418,
    0.16775167751677517,
    0.16776167761677616,
    0.16777167771677717,
    0.16778167781677816,
    0.16779167791677915,
    0.16780167801678017,
    0.16781167811678116,
    0.16782167821678218,
    0.16783167831678317,
    0.16784167841678416,
    0.16785167851678517,
    0.16786167861678616,
    0.16787167871678715,
    0.16788167881678817,
    0.16789167891678916,
    0.16790167901679018,
    0.16791167911679117,
    0.16792167921679216,
    0.16793167931679318,
    0.16794167941679417,
    0.16795167951679515,
    0.16796167961679617,
    0.16797167971679716,
    0.16798167981679815,
    0.16799167991679917,
    0.16800168001680016,
    0.16801168011680118,
    0.16802168021680217,
    0.16803168031680316,
    0.16804168041680417,
    0.16805168051680516,
    0.16806168061680615,
    0.16807168071680717,
    0.16808168081680816,
    0.16809168091680918,
    0.16810168101681017,
    0.16811168111681116,
    0.16812168121681217,
    0.16813168131681316,
    0.16814168141681415,
    0.16815168151681517,
    0.16816168161681616,
    0.16817168171681718,
    0.16818168181681817,
    0.16819168191681916,
    0.16820168201682018,
    0.16821168211682117,
    0.16822168221682215,
    0.16823168231682317,
    0.16824168241682416,
    0.16825168251682515,
    0.16826168261682617,
    0.16827168271682716,
    0.16828168281682818,
    0.16829168291682917,
    0.16830168301683016,
    0.16831168311683117,
    0.16832168321683216,
    0.16833168331683315,
    0.16834168341683417,
    0.16835168351683516,
    0.16836168361683618,
    0.16837168371683717,
    0.16838168381683816,
    0.16839168391683917,
    0.16840168401684016,
    0.16841168411684115,
    0.16842168421684217,
    0.16843168431684316,
    0.16844168441684418,
    0.16845168451684517,
    0.16846168461684616,
    0.16847168471684718,
    0.16848168481684817,
    0.16849168491684915,
    0.16850168501685017,
    0.16851168511685116,
    0.16852168521685218,
    0.16853168531685317,
    0.16854168541685416,
    0.16855168551685518,
    0.16856168561685617,
    0.16857168571685716,
    0.16858168581685817,
    0.16859168591685916,
    0.16860168601686015,
    0.16861168611686117,
    0.16862168621686216,
    0.16863168631686318,
    0.16864168641686417,
    0.16865168651686516,
    0.16866168661686617,
    0.16867168671686716,
    0.16868168681686815,
    0.16869168691686917,
    0.16870168701687016,
    0.16871168711687118,
    0.16872168721687217,
    0.16873168731687316,
    0.16874168741687418,
    0.16875168751687517,
    0.16876168761687615,
    0.16877168771687717,
    0.16878168781687816,
    0.16879168791687918,
    0.16880168801688017,
    0.16881168811688116,
    0.16882168821688218,
    0.16883168831688317,
    0.16884168841688416,
    0.16885168851688517,
    0.16886168861688616,
    0.16887168871688715,
    0.16888168881688817,
    0.16889168891688916,
    0.16890168901689018,
    0.16891168911689117,
    0.16892168921689216,
    0.16893168931689317,
    0.16894168941689416,
    0.16895168951689515,
    0.16896168961689617,
    0.16897168971689716,
    0.16898168981689818,
    0.16899168991689917,
    0.16900169001690016,
    0.16901169011690118,
    0.16902169021690217,
    0.16903169031690315,
    0.16904169041690417,
    0.16905169051690516,
    0.16906169061690618,
    0.16907169071690717,
    0.16908169081690816,
    0.16909169091690918,
    0.16910169101691017,
    0.16911169111691116,
    0.16912169121691217,
    0.16913169131691316,
    0.16914169141691415,
    0.16915169151691517,
    0.16916169161691616,
    0.16917169171691718,
    0.16918169181691817,
    0.16919169191691916,
    0.16920169201692017,
    0.16921169211692116,
    0.16922169221692215,
    0.16923169231692317,
    0.16924169241692416,
    0.16925169251692518,
    0.16926169261692617,
    0.16927169271692716,
    0.16928169281692818,
    0.16929169291692917,
    0.16930169301693015,
    0.16931169311693117,
    0.16932169321693216,
    0.16933169331693318,
    0.16934169341693417,
    0.16935169351693516,
    0.16936169361693618,
    0.16937169371693717,
    0.16938169381693816,
    0.16939169391693917,
    0.16940169401694016,
    0.16941169411694115,
    0.16942169421694217,
    0.16943169431694316,
    0.16944169441694418,
    0.16945169451694517,
    0.16946169461694616,
    0.16947169471694717,
    0.16948169481694816,
    0.16949169491694915,
    0.16950169501695017,
    0.16951169511695116,
    0.16952169521695218,
    0.16953169531695317,
    0.16954169541695416,
    0.16955169551695518,
    0.16956169561695617,
    0.16957169571695715,
    0.16958169581695817,
    0.16959169591695916,
    0.16960169601696018,
    0.16961169611696117,
    0.16962169621696216,
    0.16963169631696318,
    0.16964169641696417,
    0.16965169651696516,
    0.16966169661696617,
    0.16967169671696716,
    0.16968169681696818,
    0.16969169691696917,
    0.16970169701697016,
    0.16971169711697118,
    0.16972169721697217,
    0.16973169731697316,
    0.16974169741697417,
    0.16975169751697516,
    0.16976169761697615,
    0.16977169771697717,
    0.16978169781697816,
    0.16979169791697918,
    0.16980169801698017,
    0.16981169811698116,
    0.16982169821698218,
    0.16983169831698317,
    0.16984169841698415,
    0.16985169851698517,
    0.16986169861698616,
    0.16987169871698718,
    0.16988169881698817,
    0.16989169891698916,
    0.16990169901699018,
    0.16991169911699117,
    0.16992169921699216,
    0.16993169931699317,
    0.16994169941699416,
    0.16995169951699518,
    0.16996169961699617,
    0.16997169971699716,
    0.16998169981699818,
    0.16999169991699917,
    0.17000170001700016,
    0.17001170011700117,
    0.17002170021700216,
    0.17003170031700315,
    0.17004170041700417,
    0.17005170051700516,
    0.17006170061700618,
    0.17007170071700717,
    0.17008170081700816,
    0.17009170091700918,
    0.17010170101701017,
    0.17011170111701115,
    0.17012170121701217,
    0.17013170131701316,
    0.17014170141701418,
    0.17015170151701517,
    0.17016170161701616,
    0.17017170171701718,
    0.17018170181701817,
    0.17019170191701916,
    0.17020170201702017,
    0.17021170211702116,
    0.17022170221702218,
    0.17023170231702317,
    0.17024170241702416,
    0.17025170251702518,
    0.17026170261702617,
    0.17027170271702716,
    0.17028170281702817,
    0.17029170291702916,
    0.17030170301703015,
    0.17031170311703117,
    0.17032170321703216,
    0.17033170331703318,
    0.17034170341703417,
    0.17035170351703516,
    0.17036170361703618,
    0.17037170371703717,
    0.17038170381703815,
    0.17039170391703917,
    0.17040170401704016,
    0.17041170411704118,
    0.17042170421704217,
    0.17043170431704316,
    0.17044170441704418,
    0.17045170451704517,
    0.17046170461704616,
    0.17047170471704717,
    0.17048170481704816,
    0.17049170491704918,
    0.17050170501705017,
    0.17051170511705116,
    0.17052170521705218,
    0.17053170531705317,
    0.17054170541705416,
    0.17055170551705517,
    0.17056170561705616,
    0.17057170571705715,
    0.17058170581705817,
    0.17059170591705916,
    0.17060170601706018,
    0.17061170611706117,
    0.17062170621706216,
    0.17063170631706318,
    0.17064170641706417,
    0.17065170651706515,
    0.17066170661706617,
    0.17067170671706716,
    0.17068170681706818,
    0.17069170691706917,
    0.17070170701707016,
    0.17071170711707118,
    0.17072170721707217,
    0.17073170731707316,
    0.17074170741707417,
    0.17075170751707516,
    0.17076170761707618,
    0.17077170771707717,
    0.17078170781707816,
    0.17079170791707918,
    0.17080170801708017,
    0.17081170811708116,
    0.17082170821708217,
    0.17083170831708316,
    0.17084170841708418,
    0.17085170851708517,
    0.17086170861708616,
    0.17087170871708718,
    0.17088170881708817,
    0.17089170891708916,
    0.17090170901709018,
    0.17091170911709117,
    0.17092170921709215,
    0.17093170931709317,
    0.17094170941709416,
    0.17095170951709518,
    0.17096170961709617,
    0.17097170971709716,
    0.17098170981709818,
    0.17099170991709917,
    0.17100171001710016,
    0.17101171011710117,
    0.17102171021710216,
    0.17103171031710318,
    0.17104171041710417,
    0.17105171051710516,
    0.17106171061710618,
    0.17107171071710717,
    0.17108171081710816,
    0.17109171091710917,
    0.17110171101711016,
    0.17111171111711118,
    0.17112171121711217,
    0.17113171131711316,
    0.17114171141711418,
    0.17115171151711517,
    0.17116171161711616,
    0.17117171171711718,
    0.17118171181711817,
    0.17119171191711915,
    0.17120171201712017,
    0.17121171211712116,
    0.17122171221712218,
    0.17123171231712317,
    0.17124171241712416,
    0.17125171251712518,
    0.17126171261712617,
    0.17127171271712716,
    0.17128171281712817,
    0.17129171291712916,
    0.17130171301713018,
    0.17131171311713117,
    0.17132171321713216,
    0.17133171331713318,
    0.17134171341713417,
    0.17135171351713516,
    0.17136171361713617,
    0.17137171371713716,
    0.17138171381713818,
    0.17139171391713917,
    0.17140171401714016,
    0.17141171411714118,
    0.17142171421714217,
    0.17143171431714316,
    0.17144171441714418,
    0.17145171451714517,
    0.17146171461714615,
    0.17147171471714717,
    0.17148171481714816,
    0.17149171491714918,
    0.17150171501715017,
    0.17151171511715116,
    0.17152171521715218,
    0.17153171531715317,
    0.17154171541715416,
    0.17155171551715517,
    0.17156171561715616,
    0.17157171571715718,
    0.17158171581715817,
    0.17159171591715916,
    0.17160171601716018,
    0.17161171611716117,
    0.17162171621716216,
    0.17163171631716317,
    0.17164171641716416,
    0.17165171651716518,
    0.17166171661716617,
    0.17167171671716716,
    0.17168171681716818,
    0.17169171691716917,
    0.17170171701717016,
    0.17171171711717118,
    0.17172171721717217,
    0.17173171731717318,
    0.17174171741717417,
    0.17175171751717516,
    0.17176171761717618,
    0.17177171771717717,
    0.17178171781717816,
    0.17179171791717918,
    0.17180171801718017,
    0.17181171811718116,
    0.17182171821718217,
    0.17183171831718316,
    0.17184171841718418,
    0.17185171851718517,
    0.17186171861718616,
    0.17187171871718718,
    0.17188171881718817,
    0.17189171891718916,
    0.17190171901719017,
    0.17191171911719116,
    0.17192171921719218,
    0.17193171931719317,
    0.17194171941719416,
    0.17195171951719518,
    0.17196171961719617,
    0.17197171971719716,
    0.17198171981719818,
    0.17199171991719917,
    0.17200172001720018,
    0.17201172011720117,
    0.17202172021720216,
    0.17203172031720318,
    0.17204172041720417,
    0.17205172051720516,
    0.17206172061720618,
    0.17207172071720717,
    0.17208172081720816,
    0.17209172091720917,
    0.17210172101721016,
    0.17211172111721118,
    0.17212172121721217,
    0.17213172131721316,
    0.17214172141721418,
    0.17215172151721517,
    0.17216172161721616,
    0.17217172171721717,
    0.17218172181721816,
    0.17219172191721918,
    0.17220172201722017,
    0.17221172211722116,
    0.17222172221722218,
    0.17223172231722317,
    0.17224172241722416,
    0.17225172251722518,
    0.17226172261722617,
    0.17227172271722718,
    0.17228172281722817,
    0.17229172291722916,
    0.17230172301723018,
    0.17231172311723117,
    0.17232172321723216,
    0.17233172331723318,
    0.17234172341723417,
    0.17235172351723516,
    0.17236172361723617,
    0.17237172371723716,
    0.17238172381723818,
    0.17239172391723917,
    0.17240172401724016,
    0.17241172411724118,
    0.17242172421724217,
    0.17243172431724316,
    0.17244172441724417,
    0.17245172451724516,
    0.17246172461724618,
    0.17247172471724717,
    0.17248172481724816,
    0.17249172491724918,
    0.17250172501725017,
    0.17251172511725116,
    0.17252172521725218,
    0.17253172531725317,
    0.17254172541725418,
    0.17255172551725517,
    0.17256172561725616,
    0.17257172571725718,
    0.17258172581725817,
    0.17259172591725916,
    0.17260172601726018,
    0.17261172611726117,
    0.17262172621726216,
    0.17263172631726317,
    0.17264172641726416,
    0.17265172651726518,
    0.17266172661726617,
    0.17267172671726716,
    0.17268172681726818,
    0.17269172691726917,
    0.17270172701727016,
    0.17271172711727117,
    0.17272172721727216,
    0.17273172731727318,
    0.17274172741727417,
    0.17275172751727516,
    0.17276172761727618,
    0.17277172771727717,
    0.17278172781727816,
    0.17279172791727918,
    0.17280172801728017,
    0.17281172811728118,
    0.17282172821728217,
    0.17283172831728316,
    0.17284172841728418,
    0.17285172851728517,
    0.17286172861728616,
    0.17287172871728718,
    0.17288172881728817,
    0.17289172891728918,
    0.17290172901729017,
    0.17291172911729116,
    0.17292172921729218,
    0.17293172931729317,
    0.17294172941729416,
    0.17295172951729518,
    0.17296172961729617,
    0.17297172971729716,
    0.17298172981729817,
    0.17299172991729916,
    0.17300173001730018,
    0.17301173011730117,
    0.17302173021730216,
    0.17303173031730318,
    0.17304173041730417,
    0.17305173051730516,
    0.17306173061730618,
    0.17307173071730717,
    0.17308173081730818,
    0.17309173091730917,
    0.17310173101731016,
    0.17311173111731118,
    0.17312173121731217,
    0.17313173131731316,
    0.17314173141731418,
    0.17315173151731517,
    0.17316173161731618,
    0.17317173171731717,
    0.17318173181731816,
    0.17319173191731918,
    0.17320173201732017,
    0.17321173211732116,
    0.17322173221732218,
    0.17323173231732317,
    0.17324173241732416,
    0.17325173251732517,
    0.17326173261732616,
    0.17327173271732718,
    0.17328173281732817,
    0.17329173291732916,
    0.17330173301733018,
    0.17331173311733117,
    0.17332173321733216,
    0.17333173331733318,
    0.17334173341733417,
    0.17335173351733518,
    0.17336173361733617,
    0.17337173371733716,
    0.17338173381733818,
    0.17339173391733917,
    0.17340173401734016,
    0.17341173411734118,
    0.17342173421734217,
    0.17343173431734318,
    0.17344173441734417,
    0.17345173451734516,
    0.17346173461734618,
    0.17347173471734717,
    0.17348173481734816,
    0.17349173491734918,
    0.17350173501735017,
    0.17351173511735116,
    0.17352173521735217,
    0.17353173531735316,
    0.17354173541735418,
    0.17355173551735517,
    0.17356173561735616,
    0.17357173571735718,
    0.17358173581735817,
    0.17359173591735916,
    0.17360173601736018,
    0.17361173611736117,
    0.17362173621736218,
    0.17363173631736317,
    0.17364173641736416,
    0.17365173651736518,
    0.17366173661736617,
    0.17367173671736716,
    0.17368173681736818,
    0.17369173691736917,
    0.17370173701737018,
    0.17371173711737117,
    0.17372173721737216,
    0.17373173731737318,
    0.17374173741737417,
    0.17375173751737516,
    0.17376173761737618,
    0.17377173771737717,
    0.17378173781737818,
    0.17379173791737917,
    0.17380173801738016,
    0.17381173811738118,
    0.17382173821738217,
    0.17383173831738316,
    0.17384173841738418,
    0.17385173851738517,
    0.17386173861738616,
    0.17387173871738718,
    0.17388173881738817,
    0.17389173891738918,
    0.17390173901739017,
    0.17391173911739116,
    0.17392173921739218,
    0.17393173931739317,
    0.17394173941739416,
    0.17395173951739518,
    0.17396173961739617,
    0.17397173971739718,
    0.17398173981739817,
    0.17399173991739916,
    0.17400174001740018,
    0.17401174011740117,
    0.17402174021740216,
    0.17403174031740318,
    0.17404174041740417,
    0.17405174051740518,
    0.17406174061740617,
    0.17407174071740716,
    0.17408174081740818,
    0.17409174091740917,
    0.17410174101741016,
    0.17411174111741118,
    0.17412174121741217,
    0.17413174131741316,
    0.17414174141741418,
    0.17415174151741517,
    0.17416174161741618,
    0.17417174171741717,
    0.17418174181741816,
    0.17419174191741918,
    0.17420174201742017,
    0.17421174211742116,
    0.17422174221742218,
    0.17423174231742317,
    0.17424174241742418,
    0.17425174251742517,
    0.17426174261742616,
    0.17427174271742718,
    0.17428174281742817,
    0.17429174291742916,
    0.17430174301743018,
    0.17431174311743117,
    0.17432174321743218,
    0.17433174331743317,
    0.17434174341743416,
    0.17435174351743518,
    0.17436174361743617,
    0.17437174371743716,
    0.17438174381743818,
    0.17439174391743917,
    0.17440174401744016,
    0.17441174411744118,
    0.17442174421744217,
    0.17443174431744318,
    0.17444174441744417,
    0.17445174451744516,
    0.17446174461744618,
    0.17447174471744717,
    0.17448174481744816,
    0.17449174491744918,
    0.17450174501745017,
    0.17451174511745118,
    0.17452174521745217,
    0.17453174531745316,
    0.17454174541745418,
    0.17455174551745517,
    0.17456174561745616,
    0.17457174571745718,
    0.17458174581745817,
    0.17459174591745918,
    0.17460174601746017,
    0.17461174611746116,
    0.17462174621746218,
    0.17463174631746317,
    0.17464174641746416,
    0.17465174651746518,
    0.17466174661746617,
    0.17467174671746716,
    0.17468174681746818,
    0.17469174691746917,
    0.17470174701747018,
    0.17471174711747117,
    0.17472174721747216,
    0.17473174731747318,
    0.17474174741747417,
    0.17475174751747516,
    0.17476174761747618,
    0.17477174771747717,
    0.17478174781747818,
    0.17479174791747917,
    0.17480174801748016,
    0.17481174811748118,
    0.17482174821748217,
    0.17483174831748316,
    0.17484174841748418,
    0.17485174851748517,
    0.17486174861748618,
    0.17487174871748717,
    0.17488174881748816,
    0.17489174891748918,
    0.17490174901749017,
    0.17491174911749116,
    0.17492174921749218,
    0.17493174931749317,
    0.17494174941749419,
    0.17495174951749518,
    0.17496174961749617,
    0.17497174971749718,
    0.17498174981749817,
    0.17499174991749916,
    0.17500175001750018,
    0.17501175011750117,
    0.17502175021750216,
    0.17503175031750318,
    0.17504175041750417,
    0.17505175051750518,
    0.17506175061750617,
    0.17507175071750716,
    0.17508175081750818,
    0.17509175091750917,
    0.17510175101751016,
    0.17511175111751118,
    0.17512175121751217,
    0.17513175131751318,
    0.17514175141751417,
    0.17515175151751516,
    0.17516175161751618,
    0.17517175171751717,
    0.17518175181751816,
    0.17519175191751918,
    0.17520175201752017,
    0.17521175211752119,
    0.17522175221752218,
    0.17523175231752317,
    0.17524175241752418,
    0.17525175251752517,
    0.17526175261752616,
    0.17527175271752718,
    0.17528175281752817,
    0.17529175291752916,
    0.17530175301753018,
    0.17531175311753117,
    0.17532175321753218,
    0.17533175331753317,
    0.17534175341753416,
    0.17535175351753518,
    0.17536175361753617,
    0.17537175371753716,
    0.17538175381753818,
    0.17539175391753917,
    0.17540175401754018,
    0.17541175411754117,
    0.17542175421754216,
    0.17543175431754318,
    0.17544175441754417,
    0.17545175451754516,
    0.17546175461754618,
    0.17547175471754717,
    0.17548175481754819,
    0.17549175491754918,
    0.17550175501755017,
    0.17551175511755118,
    0.17552175521755217,
    0.17553175531755316,
    0.17554175541755418,
    0.17555175551755517,
    0.17556175561755616,
    0.17557175571755718,
    0.17558175581755817,
    0.17559175591755918,
    0.17560175601756017,
    0.17561175611756116,
    0.17562175621756218,
    0.17563175631756317,
    0.17564175641756416,
    0.17565175651756518,
    0.17566175661756617,
    0.17567175671756718,
    0.17568175681756817,
    0.17569175691756916,
    0.17570175701757018,
    0.17571175711757117,
    0.17572175721757216,
    0.17573175731757318,
    0.17574175741757417,
    0.17575175751757519,
    0.17576175761757618,
    0.17577175771757717,
    0.17578175781757818,
    0.17579175791757917,
    0.17580175801758016,
    0.17581175811758118,
    0.17582175821758217,
    0.1758317583175832,
    0.17584175841758418,
    0.17585175851758517,
    0.17586175861758618,
    0.17587175871758717,
    0.17588175881758816,
    0.17589175891758918,
    0.17590175901759017,
    0.17591175911759116,
    0.17592175921759218,
    0.17593175931759317,
    0.17594175941759418,
    0.17595175951759517,
    0.17596175961759616,
    0.17597175971759718,
    0.17598175981759817,
    0.17599175991759916,
    0.17600176001760018,
    0.17601176011760117,
    0.17602176021760219,
    0.17603176031760318,
    0.17604176041760417,
    0.17605176051760518,
    0.17606176061760617,
    0.17607176071760716,
    0.17608176081760818,
    0.17609176091760917,
    0.1761017610176102,
    0.17611176111761118,
    0.17612176121761217,
    0.17613176131761318,
    0.17614176141761417,
    0.17615176151761516,
    0.17616176161761618,
    0.17617176171761717,
    0.17618176181761816,
    0.17619176191761918,
    0.17620176201762017,
    0.17621176211762118,
    0.17622176221762217,
    0.17623176231762316,
    0.17624176241762418,
    0.17625176251762517,
    0.17626176261762616,
    0.17627176271762718,
    0.17628176281762817,
    0.17629176291762919,
    0.17630176301763018,
    0.17631176311763117,
    0.17632176321763218,
    0.17633176331763317,
    0.17634176341763416,
    0.17635176351763518,
    0.17636176361763617,
    0.1763717637176372,
    0.17638176381763818,
    0.17639176391763917,
    0.17640176401764018,
    0.17641176411764117,
    0.17642176421764216,
    0.17643176431764318,
    0.17644176441764417,
    0.17645176451764516,
    0.17646176461764618,
    0.17647176471764717,
    0.17648176481764818,
    0.17649176491764917,
    0.17650176501765016,
    0.17651176511765118,
    0.17652176521765217,
    0.17653176531765316,
    0.17654176541765418,
    0.17655176551765517,
    0.17656176561765619,
    0.17657176571765718,
    0.17658176581765817,
    0.17659176591765918,
    0.17660176601766017,
    0.17661176611766116,
    0.17662176621766218,
    0.17663176631766317,
    0.1766417664176642,
    0.17665176651766518,
    0.17666176661766617,
    0.17667176671766718,
    0.17668176681766817,
    0.17669176691766916,
    0.17670176701767018,
    0.17671176711767117,
    0.17672176721767216,
    0.17673176731767318,
    0.17674176741767417,
    0.17675176751767518,
    0.17676176761767617,
    0.17677176771767716,
    0.17678176781767818,
    0.17679176791767917,
    0.17680176801768016,
    0.17681176811768118,
    0.17682176821768217,
    0.17683176831768319,
    0.17684176841768418,
    0.17685176851768517,
    0.17686176861768618,
    0.17687176871768717,
    0.17688176881768816,
    0.17689176891768918,
    0.17690176901769017,
    0.1769117691176912,
    0.17692176921769218,
    0.17693176931769317,
    0.17694176941769418,
    0.17695176951769517,
    0.17696176961769616,
    0.17697176971769718,
    0.17698176981769817,
    0.1769917699176992,
    0.17700177001770018,
    0.17701177011770117,
    0.17702177021770218,
    0.17703177031770317,
    0.17704177041770416,
    0.17705177051770518,
    0.17706177061770617,
    0.17707177071770716,
    0.17708177081770818,
    0.17709177091770917,
    0.17710177101771019,
    0.17711177111771118,
    0.17712177121771217,
    0.17713177131771318,
    0.17714177141771417,
    0.17715177151771516,
    0.17716177161771618,
    0.17717177171771717,
    0.1771817718177182,
    0.17719177191771918,
    0.17720177201772017,
    0.17721177211772118,
    0.17722177221772217,
    0.17723177231772316,
    0.17724177241772418,
    0.17725177251772517,
    0.1772617726177262,
    0.17727177271772718,
    0.17728177281772817,
    0.17729177291772918,
    0.17730177301773017,
    0.17731177311773116,
    0.17732177321773218,
    0.17733177331773317,
    0.17734177341773416,
    0.17735177351773518,
    0.17736177361773617,
    0.17737177371773719,
    0.17738177381773818,
    0.17739177391773917,
    0.17740177401774018,
    0.17741177411774117,
    0.17742177421774216,
    0.17743177431774318,
    0.17744177441774417,
    0.1774517745177452,
    0.17746177461774618,
    0.17747177471774717,
    0.17748177481774818,
    0.17749177491774917,
    0.17750177501775016,
    0.17751177511775118,
    0.17752177521775217,
    0.1775317753177532,
    0.17754177541775418,
    0.17755177551775517,
    0.17756177561775618,
    0.17757177571775717,
    0.17758177581775816,
    0.17759177591775918,
    0.17760177601776017,
    0.17761177611776116,
    0.17762177621776218,
    0.17763177631776317,
    0.17764177641776419,
    0.17765177651776518,
    0.17766177661776617,
    0.17767177671776718,
    0.17768177681776817,
    0.17769177691776916,
    0.17770177701777018,
    0.17771177711777117,
    0.1777217772177722,
    0.17773177731777318,
    0.17774177741777417,
    0.17775177751777518,
    0.17776177761777617,
    0.17777177771777716,
    0.17778177781777818,
    0.17779177791777917,
    0.1778017780177802,
    0.17781177811778118,
    0.17782177821778217,
    0.17783177831778318,
    0.17784177841778417,
    0.17785177851778516,
    0.17786177861778618,
    0.17787177871778717,
    0.1778817788177882,
    0.17789177891778918,
    0.17790177901779017,
    0.17791177911779119,
    0.17792177921779218,
    0.17793177931779317,
    0.17794177941779418,
    0.17795177951779517,
    0.17796177961779616,
    0.17797177971779718,
    0.17798177981779817,
    0.1779917799177992,
    0.17800178001780018,
    0.17801178011780117,
    0.17802178021780218,
    0.17803178031780317,
    0.17804178041780416,
    0.17805178051780518,
    0.17806178061780617,
    0.1780717807178072,
    0.17808178081780818,
    0.17809178091780917,
    0.17810178101781018,
    0.17811178111781117,
    0.17812178121781216,
    0.17813178131781318,
    0.17814178141781417,
    0.1781517815178152,
    0.17816178161781618,
    0.17817178171781717,
    0.17818178181781819,
    0.17819178191781918,
    0.17820178201782017,
    0.17821178211782118,
    0.17822178221782217,
    0.17823178231782316,
    0.17824178241782418,
    0.17825178251782517,
    0.1782617826178262,
    0.17827178271782718,
    0.17828178281782817,
    0.17829178291782918,
    0.17830178301783017,
    0.17831178311783116,
    0.17832178321783218,
    0.17833178331783317,
    0.1783417834178342,
    0.17835178351783518,
    0.17836178361783617,
    0.17837178371783718,
    0.17838178381783817,
    0.17839178391783916,
    0.17840178401784018,
    0.17841178411784117,
    0.1784217842178422,
    0.17843178431784318,
    0.17844178441784417,
    0.17845178451784519,
    0.17846178461784618,
    0.17847178471784717,
    0.17848178481784818,
    0.17849178491784917,
    0.17850178501785016,
    0.17851178511785118,
    0.17852178521785217,
    0.1785317853178532,
    0.17854178541785418,
    0.17855178551785517,
    0.17856178561785618,
    0.17857178571785717,
    0.17858178581785816,
    0.17859178591785918,
    0.17860178601786017,
    0.1786117861178612,
    0.17862178621786218,
    0.17863178631786317,
    0.17864178641786418,
    0.17865178651786517,
    0.17866178661786616,
    0.17867178671786718,
    0.17868178681786817,
    0.1786917869178692,
    0.17870178701787018,
    0.17871178711787117,
    0.17872178721787219,
    0.17873178731787318,
    0.17874178741787417,
    0.17875178751787518,
    0.17876178761787617,
    0.17877178771787716,
    0.17878178781787818,
    0.17879178791787917,
    0.1788017880178802,
    0.17881178811788118,
    0.17882178821788217,
    0.17883178831788318,
    0.17884178841788417,
    0.17885178851788516,
    0.17886178861788618,
    0.17887178871788717,
    0.1788817888178882,
    0.17889178891788918,
    0.17890178901789017,
    0.17891178911789118,
    0.17892178921789217,
    0.17893178931789316,
    0.17894178941789418,
    0.17895178951789517,
    0.1789617896178962,
    0.17897178971789718,
    0.17898178981789817,
    0.17899178991789919,
    0.17900179001790018,
    0.17901179011790117,
    0.17902179021790218,
    0.17903179031790317,
    0.1790417904179042,
    0.17905179051790518,
    0.17906179061790617,
    0.1790717907179072,
    0.17908179081790818,
    0.17909179091790917,
    0.17910179101791018,
    0.17911179111791117,
    0.17912179121791216,
    0.17913179131791318,
    0.17914179141791417,
    0.1791517915179152,
    0.17916179161791618,
    0.17917179171791717,
    0.17918179181791818,
    0.17919179191791917,
    0.17920179201792016,
    0.17921179211792118,
    0.17922179221792217,
    0.1792317923179232,
    0.17924179241792418,
    0.17925179251792517,
    0.17926179261792619,
    0.17927179271792718,
    0.17928179281792817,
    0.17929179291792918,
    0.17930179301793017,
    0.1793117931179312,
    0.17932179321793218,
    0.17933179331793317,
    0.1793417934179342,
    0.17935179351793518,
    0.17936179361793617,
    0.17937179371793718,
    0.17938179381793817,
    0.17939179391793916,
    0.17940179401794018,
    0.17941179411794117,
    0.1794217942179422,
    0.17943179431794318,
    0.17944179441794417,
    0.17945179451794518,
    0.17946179461794617,
    0.17947179471794716,
    0.17948179481794818,
    0.17949179491794917,
    0.1795017950179502,
    0.17951179511795118,
    0.17952179521795217,
    0.17953179531795319,
    0.17954179541795418,
    0.17955179551795517,
    0.17956179561795618,
    0.17957179571795717,
    0.1795817958179582,
    0.17959179591795918,
    0.17960179601796017,
    0.1796117961179612,
    0.17962179621796218,
    0.17963179631796317,
    0.17964179641796418,
    0.17965179651796517,
    0.17966179661796616,
    0.17967179671796718,
    0.17968179681796817,
    0.1796917969179692,
    0.17970179701797018,
    0.17971179711797117,
    0.17972179721797218,
    0.17973179731797317,
    0.17974179741797416,
    0.17975179751797518,
    0.17976179761797617,
    0.1797717977179772,
    0.17978179781797818,
    0.17979179791797917,
    0.17980179801798019,
    0.17981179811798118,
    0.17982179821798217,
    0.17983179831798318,
    0.17984179841798417,
    0.1798517985179852,
    0.17986179861798618,
    0.17987179871798717,
    0.1798817988179882,
    0.17989179891798918,
    0.17990179901799017,
    0.17991179911799118,
    0.17992179921799217,
    0.1799317993179932,
    0.17994179941799418,
    0.17995179951799517,
    0.1799617996179962,
    0.17997179971799718,
    0.17998179981799817,
    0.17999179991799918,
    0.18000180001800017,
    0.18001180011800116,
    0.18002180021800218,
    0.18003180031800317,
    0.1800418004180042,
    0.18005180051800518,
    0.18006180061800617,
    0.18007180071800719,
    0.18008180081800818,
    0.18009180091800917,
    0.18010180101801018,
    0.18011180111801117,
    0.1801218012180122,
    0.18013180131801318,
    0.18014180141801417,
    0.1801518015180152,
    0.18016180161801618,
    0.18017180171801717,
    0.18018180181801818,
    0.18019180191801917,
    0.1802018020180202,
    0.18021180211802118,
    0.18022180221802217,
    0.1802318023180232,
    0.18024180241802418,
    0.18025180251802517,
    0.18026180261802618,
    0.18027180271802717,
    0.18028180281802816,
    0.18029180291802918,
    0.18030180301803017,
    0.1803118031180312,
    0.18032180321803218,
    0.18033180331803317,
    0.18034180341803419,
    0.18035180351803518,
    0.18036180361803617,
    0.18037180371803718,
    0.18038180381803817,
    0.1803918039180392,
    0.18040180401804018,
    0.18041180411804117,
    0.1804218042180422,
    0.18043180431804318,
    0.18044180441804417,
    0.18045180451804518,
    0.18046180461804617,
    0.1804718047180472,
    0.18048180481804818,
    0.18049180491804917,
    0.1805018050180502,
    0.18051180511805118,
    0.18052180521805217,
    0.18053180531805318,
    0.18054180541805417,
    0.18055180551805516,
    0.18056180561805618,
    0.18057180571805717,
    0.1805818058180582,
    0.18059180591805918,
    0.18060180601806017,
    0.18061180611806119,
    0.18062180621806218,
    0.18063180631806317,
    0.18064180641806418,
    0.18065180651806517,
    0.1806618066180662,
    0.18067180671806718,
    0.18068180681806817,
    0.1806918069180692,
    0.18070180701807018,
    0.18071180711807117,
    0.18072180721807218,
    0.18073180731807317,
    0.1807418074180742,
    0.18075180751807518,
    0.18076180761807617,
    0.1807718077180772,
    0.18078180781807818,
    0.18079180791807917,
    0.18080180801808018,
    0.18081180811808117,
    0.18082180821808216,
    0.18083180831808318,
    0.18084180841808417,
    0.1808518085180852,
    0.18086180861808618,
    0.18087180871808717,
    0.18088180881808819,
    0.18089180891808918,
    0.18090180901809017,
    0.18091180911809118,
    0.18092180921809217,
    0.1809318093180932,
    0.18094180941809418,
    0.18095180951809517,
    0.1809618096180962,
    0.18097180971809718,
    0.18098180981809817,
    0.18099180991809918,
    0.18100181001810017,
    0.1810118101181012,
    0.18102181021810218,
    0.18103181031810317,
    0.1810418104181042,
    0.18105181051810518,
    0.18106181061810617,
    0.18107181071810718,
    0.18108181081810817,
    0.1810918109181092,
    0.18110181101811018,
    0.18111181111811117,
    0.1811218112181122,
    0.18113181131811318,
    0.18114181141811417,
    0.18115181151811519,
    0.18116181161811618,
    0.18117181171811717,
    0.18118181181811818,
    0.18119181191811917,
    0.1812018120181202,
    0.18121181211812118,
    0.18122181221812217,
    0.1812318123181232,
    0.18124181241812418,
    0.18125181251812517,
    0.18126181261812618,
    0.18127181271812717,
    0.1812818128181282,
    0.18129181291812918,
    0.18130181301813017,
    0.1813118131181312,
    0.18132181321813218,
    0.18133181331813317,
    0.18134181341813418,
    0.18135181351813517,
    0.1813618136181362,
    0.18137181371813718,
    0.18138181381813817,
    0.1813918139181392,
    0.18140181401814018,
    0.18141181411814117,
    0.18142181421814219,
    0.18143181431814318,
    0.18144181441814416,
    0.18145181451814518,
    0.18146181461814617,
    0.1814718147181472,
    0.18148181481814818,
    0.18149181491814917,
    0.1815018150181502,
    0.18151181511815118,
    0.18152181521815217,
    0.18153181531815318,
    0.18154181541815417,
    0.1815518155181552,
    0.18156181561815618,
    0.18157181571815717,
    0.1815818158181582,
    0.18159181591815918,
    0.18160181601816017,
    0.18161181611816118,
    0.18162181621816217,
    0.1816318163181632,
    0.18164181641816418,
    0.18165181651816517,
    0.1816618166181662,
    0.18167181671816718,
    0.18168181681816817,
    0.18169181691816919,
    0.18170181701817018,
    0.18171181711817116,
    0.18172181721817218,
    0.18173181731817317,
    0.1817418174181742,
    0.18175181751817518,
    0.18176181761817617,
    0.1817718177181772,
    0.18178181781817818,
    0.18179181791817917,
    0.18180181801818018,
    0.18181181811818117,
    0.1818218182181822,
    0.18183181831818318,
    0.18184181841818417,
    0.1818518185181852,
    0.18186181861818618,
    0.18187181871818717,
    0.18188181881818818,
    0.18189181891818917,
    0.1819018190181902,
    0.18191181911819118,
    0.18192181921819217,
    0.1819318193181932,
    0.18194181941819418,
    0.18195181951819517,
    0.18196181961819619,
    0.18197181971819718,
    0.18198181981819816,
    0.18199181991819918,
    0.18200182001820017,
    0.1820118201182012,
    0.18202182021820218,
    0.18203182031820317,
    0.1820418204182042,
    0.18205182051820518,
    0.18206182061820617,
    0.18207182071820718,
    0.18208182081820817,
    0.1820918209182092,
    0.18210182101821018,
    0.18211182111821117,
    0.1821218212182122,
    0.18213182131821318,
    0.18214182141821417,
    0.18215182151821518,
    0.18216182161821617,
    0.1821718217182172,
    0.18218182181821818,
    0.18219182191821917,
    0.1822018220182202,
    0.18221182211822118,
    0.18222182221822217,
    0.18223182231822319,
    0.18224182241822418,
    0.1822518225182252,
    0.18226182261822618,
    0.18227182271822717,
    0.1822818228182282,
    0.18229182291822918,
    0.18230182301823017,
    0.1823118231182312,
    0.18232182321823218,
    0.18233182331823317,
    0.18234182341823418,
    0.18235182351823517,
    0.1823618236182362,
    0.18237182371823718,
    0.18238182381823817,
    0.1823918239182392,
    0.18240182401824018,
    0.18241182411824117,
    0.18242182421824218,
    0.18243182431824317,
    0.1824418244182442,
    0.18245182451824518,
    0.18246182461824617,
    0.1824718247182472,
    0.18248182481824818,
    0.18249182491824917,
    0.18250182501825019,
    0.18251182511825118,
    0.1825218252182522,
    0.18253182531825318,
    0.18254182541825417,
    0.1825518255182552,
    0.18256182561825618,
    0.18257182571825717,
    0.1825818258182582,
    0.18259182591825918,
    0.18260182601826017,
    0.18261182611826118,
    0.18262182621826217,
    0.1826318263182632,
    0.18264182641826418,
    0.18265182651826517,
    0.1826618266182662,
    0.18267182671826718,
    0.18268182681826817,
    0.18269182691826918,
    0.18270182701827017,
    0.1827118271182712,
    0.18272182721827218,
    0.18273182731827317,
    0.1827418274182742,
    0.18275182751827518,
    0.18276182761827617,
    0.18277182771827719,
    0.18278182781827818,
    0.1827918279182792,
    0.18280182801828018,
    0.18281182811828117,
    0.1828218282182822,
    0.18283182831828318,
    0.18284182841828417,
    0.1828518285182852,
    0.18286182861828618,
    0.18287182871828717,
    0.18288182881828818,
    0.18289182891828917,
    0.1829018290182902,
    0.18291182911829118,
    0.18292182921829217,
    0.1829318293182932,
    0.18294182941829418,
    0.18295182951829517,
    0.18296182961829618,
    0.18297182971829717,
    0.1829818298182982,
    0.18299182991829918,
    0.18300183001830017,
    0.1830118301183012,
    0.18302183021830218,
    0.18303183031830317,
    0.18304183041830419,
    0.18305183051830518,
    0.1830618306183062,
    0.18307183071830718,
    0.18308183081830817,
    0.1830918309183092,
    0.18310183101831018,
    0.18311183111831117,
    0.1831218312183122,
    0.18313183131831318,
    0.1831418314183142,
    0.18315183151831518,
    0.18316183161831617,
    0.1831718317183172,
    0.18318183181831818,
    0.18319183191831917,
    0.1832018320183202,
    0.18321183211832118,
    0.18322183221832217,
    0.18323183231832318,
    0.18324183241832417,
    0.1832518325183252,
    0.18326183261832618,
    0.18327183271832717,
    0.1832818328183282,
    0.18329183291832918,
    0.18330183301833017,
    0.18331183311833119,
    0.18332183321833218,
    0.1833318333183332,
    0.18334183341833418,
    0.18335183351833517,
    0.1833618336183362,
    0.18337183371833718,
    0.18338183381833817,
    0.1833918339183392,
    0.18340183401834018,
    0.1834118341183412,
    0.18342183421834218,
    0.18343183431834317,
    0.1834418344183442,
    0.18345183451834518,
    0.18346183461834617,
    0.1834718347183472,
    0.18348183481834818,
    0.18349183491834917,
    0.18350183501835018,
    0.18351183511835117,
    0.1835218352183522,
    0.18353183531835318,
    0.18354183541835417,
    0.1835518355183552,
    0.18356183561835618,
    0.18357183571835717,
    0.18358183581835819,
    0.18359183591835918,
    0.1836018360183602,
    0.18361183611836118,
    0.18362183621836217,
    0.1836318363183632,
    0.18364183641836418,
    0.18365183651836517,
    0.1836618366183662,
    0.18367183671836718,
    0.1836818368183682,
    0.18369183691836918,
    0.18370183701837017,
    0.1837118371183712,
    0.18372183721837218,
    0.18373183731837317,
    0.1837418374183742,
    0.18375183751837518,
    0.18376183761837617,
    0.18377183771837718,
    0.18378183781837817,
    0.1837918379183792,
    0.18380183801838018,
    0.18381183811838117,
    0.1838218382183822,
    0.18383183831838318,
    0.18384183841838417,
    0.18385183851838519,
    0.18386183861838618,
    0.1838718387183872,
    0.18388183881838818,
    0.18389183891838917,
    0.1839018390183902,
    0.18391183911839118,
    0.18392183921839217,
    0.1839318393183932,
    0.18394183941839418,
    0.1839518395183952,
    0.18396183961839618,
    0.18397183971839717,
    0.1839818398183982,
    0.18399183991839918,
    0.18400184001840017,
    0.1840118401184012,
    0.18402184021840218,
    0.18403184031840317,
    0.18404184041840418,
    0.18405184051840517,
    0.1840618406184062,
    0.18407184071840718,
    0.18408184081840817,
    0.1840918409184092,
    0.18410184101841018,
    0.18411184111841117,
    0.18412184121841219,
    0.18413184131841318,
    0.1841418414184142,
    0.18415184151841518,
    0.18416184161841617,
    0.1841718417184172,
    0.18418184181841818,
    0.18419184191841917,
    0.1842018420184202,
    0.18421184211842118,
    0.1842218422184222,
    0.18423184231842318,
    0.18424184241842417,
    0.1842518425184252,
    0.18426184261842618,
    0.18427184271842717,
    0.1842818428184282,
    0.18429184291842918,
    0.1843018430184302,
    0.18431184311843118,
    0.18432184321843217,
    0.1843318433184332,
    0.18434184341843418,
    0.18435184351843517,
    0.1843618436184362,
    0.18437184371843718,
    0.18438184381843817,
    0.18439184391843919,
    0.18440184401844018,
    0.1844118441184412,
    0.18442184421844218,
    0.18443184431844317,
    0.1844418444184442,
    0.18445184451844518,
    0.18446184461844617,
    0.1844718447184472,
    0.18448184481844818,
    0.1844918449184492,
    0.18450184501845018,
    0.18451184511845117,
    0.1845218452184522,
    0.18453184531845318,
    0.18454184541845417,
    0.1845518455184552,
    0.18456184561845618,
    0.1845718457184572,
    0.18458184581845818,
    0.18459184591845917,
    0.1846018460184602,
    0.18461184611846118,
    0.18462184621846217,
    0.1846318463184632,
    0.18464184641846418,
    0.18465184651846517,
    0.18466184661846619,
    0.18467184671846718,
    0.1846818468184682,
    0.18469184691846918,
    0.18470184701847017,
    0.1847118471184712,
    0.18472184721847218,
    0.18473184731847317,
    0.1847418474184742,
    0.18475184751847518,
    0.1847618476184762,
    0.18477184771847718,
    0.18478184781847817,
    0.1847918479184792,
    0.18480184801848018,
    0.18481184811848117,
    0.1848218482184822,
    0.18483184831848318,
    0.1848418484184842,
    0.18485184851848518,
    0.18486184861848617,
    0.1848718487184872,
    0.18488184881848818,
    0.18489184891848917,
    0.1849018490184902,
    0.18491184911849118,
    0.18492184921849217,
    0.18493184931849319,
    0.18494184941849418,
    0.1849518495184952,
    0.18496184961849618,
    0.18497184971849717,
    0.1849818498184982,
    0.18499184991849918,
    0.18500185001850017,
    0.1850118501185012,
    0.18502185021850218,
    0.1850318503185032,
    0.18504185041850418,
    0.18505185051850517,
    0.1850618506185062,
    0.18507185071850718,
    0.18508185081850817,
    0.1850918509185092,
    0.18510185101851018,
    0.1851118511185112,
    0.18512185121851218,
    0.18513185131851317,
    0.1851418514185142,
    0.18515185151851518,
    0.18516185161851617,
    0.1851718517185172,
    0.18518185181851818,
    0.1851918519185192,
    0.18520185201852019,
    0.18521185211852118,
    0.1852218522185222,
    0.18523185231852318,
    0.18524185241852417,
    0.1852518525185252,
    0.18526185261852618,
    0.18527185271852717,
    0.1852818528185282,
    0.18529185291852918,
    0.1853018530185302,
    0.18531185311853118,
    0.18532185321853217,
    0.1853318533185332,
    0.18534185341853418,
    0.18535185351853517,
    0.1853618536185362,
    0.18537185371853718,
    0.1853818538185382,
    0.18539185391853918,
    0.18540185401854017,
    0.1854118541185412,
    0.18542185421854218,
    0.18543185431854317,
    0.1854418544185442,
    0.18545185451854518,
    0.1854618546185462,
    0.18547185471854719,
    0.18548185481854818,
    0.1854918549185492,
    0.18550185501855018,
    0.18551185511855117,
    0.1855218552185522,
    0.18553185531855318,
    0.18554185541855417,
    0.1855518555185552,
    0.18556185561855618,
    0.1855718557185572,
    0.18558185581855818,
    0.18559185591855917,
    0.1856018560185602,
    0.18561185611856118,
    0.18562185621856217,
    0.1856318563185632,
    0.18564185641856418,
    0.1856518565185652,
    0.18566185661856618,
    0.18567185671856717,
    0.1856818568185682,
    0.18569185691856918,
    0.18570185701857017,
    0.1857118571185712,
    0.18572185721857218,
    0.1857318573185732,
    0.18574185741857419,
    0.18575185751857518,
    0.1857618576185762,
    0.18577185771857718,
    0.18578185781857817,
    0.1857918579185792,
    0.18580185801858018,
    0.18581185811858117,
    0.1858218582185822,
    0.18583185831858318,
    0.1858418584185842,
    0.18585185851858518,
    0.18586185861858617,
    0.1858718587185872,
    0.18588185881858818,
    0.18589185891858917,
    0.1859018590185902,
    0.18591185911859118,
    0.1859218592185922,
    0.18593185931859318,
    0.18594185941859417,
    0.1859518595185952,
    0.18596185961859618,
    0.18597185971859717,
    0.1859818598185982,
    0.18599185991859918,
    0.1860018600186002,
    0.18601186011860119,
    0.18602186021860218,
    0.1860318603186032,
    0.18604186041860418,
    0.18605186051860517,
    0.1860618606186062,
    0.18607186071860718,
    0.18608186081860817,
    0.1860918609186092,
    0.18610186101861018,
    0.1861118611186112,
    0.18612186121861218,
    0.18613186131861317,
    0.1861418614186142,
    0.18615186151861518,
    0.18616186161861617,
    0.1861718617186172,
    0.18618186181861818,
    0.1861918619186192,
    0.18620186201862018,
    0.18621186211862117,
    0.1862218622186222,
    0.18623186231862318,
    0.18624186241862417,
    0.1862518625186252,
    0.18626186261862618,
    0.1862718627186272,
    0.18628186281862819,
    0.18629186291862918,
    0.1863018630186302,
    0.18631186311863118,
    0.18632186321863217,
    0.1863318633186332,
    0.18634186341863418,
    0.1863518635186352,
    0.1863618636186362,
    0.18637186371863718,
    0.1863818638186382,
    0.18639186391863918,
    0.18640186401864017,
    0.1864118641186412,
    0.18642186421864218,
    0.18643186431864317,
    0.1864418644186442,
    0.18645186451864518,
    0.1864618646186462,
    0.18647186471864718,
    0.18648186481864817,
    0.1864918649186492,
    0.18650186501865018,
    0.18651186511865117,
    0.1865218652186522,
    0.18653186531865318,
    0.1865418654186542,
    0.18655186551865519,
    0.18656186561865618,
    0.1865718657186572,
    0.18658186581865818,
    0.18659186591865917,
    0.1866018660186602,
    0.18661186611866118,
    0.1866218662186622,
    0.1866318663186632,
    0.18664186641866418,
    0.1866518665186652,
    0.18666186661866618,
    0.18667186671866717,
    0.1866818668186682,
    0.18669186691866918,
    0.18670186701867017,
    0.1867118671186712,
    0.18672186721867218,
    0.1867318673186732,
    0.18674186741867418,
    0.18675186751867517,
    0.1867618676186762,
    0.18677186771867718,
    0.18678186781867817,
    0.1867918679186792,
    0.18680186801868018,
    0.1868118681186812,
    0.18682186821868219,
    0.18683186831868318,
    0.1868418684186842,
    0.18685186851868518,
    0.18686186861868617,
    0.1868718687186872,
    0.18688186881868818,
    0.1868918689186892,
    0.1869018690186902,
    0.18691186911869118,
    0.1869218692186922,
    0.18693186931869318,
    0.18694186941869417,
    0.1869518695186952,
    0.18696186961869618,
    0.18697186971869717,
    0.1869818698186982,
    0.18699186991869918,
    0.1870018700187002,
    0.18701187011870118,
    0.18702187021870217,
    0.1870318703187032,
    0.18704187041870418,
    0.18705187051870517,
    0.1870618706187062,
    0.18707187071870718,
    0.1870818708187082,
    0.18709187091870919,
    0.18710187101871018,
    0.1871118711187112,
    0.18712187121871218,
    0.18713187131871317,
    0.1871418714187142,
    0.18715187151871518,
    0.1871618716187162,
    0.1871718717187172,
    0.18718187181871818,
    0.1871918719187192,
    0.18720187201872018,
    0.18721187211872117,
    0.1872218722187222,
    0.18723187231872318,
    0.1872418724187242,
    0.1872518725187252,
    0.18726187261872618,
    0.1872718727187272,
    0.18728187281872818,
    0.18729187291872917,
    0.1873018730187302,
    0.18731187311873118,
    0.18732187321873217,
    0.1873318733187332,
    0.18734187341873418,
    0.1873518735187352,
    0.18736187361873619,
    0.18737187371873718,
    0.1873818738187382,
    0.18739187391873918,
    0.18740187401874017,
    0.1874118741187412,
    0.18742187421874218,
    0.1874318743187432,
    0.1874418744187442,
    0.18745187451874518,
    0.1874618746187462,
    0.18747187471874718,
    0.18748187481874817,
    0.1874918749187492,
    0.18750187501875018,
    0.1875118751187512,
    0.1875218752187522,
    0.18753187531875318,
    0.1875418754187542,
    0.18755187551875518,
    0.18756187561875617,
    0.1875718757187572,
    0.18758187581875818,
    0.18759187591875917,
    0.1876018760187602,
    0.18761187611876118,
    0.1876218762187622,
    0.18763187631876319,
    0.18764187641876418,
    0.1876518765187652,
    0.18766187661876618,
    0.18767187671876717,
    0.1876818768187682,
    0.18769187691876918,
    0.1877018770187702,
    0.1877118771187712,
    0.18772187721877218,
    0.1877318773187732,
    0.18774187741877418,
    0.18775187751877517,
    0.1877618776187762,
    0.18777187771877718,
    0.1877818778187782,
    0.1877918779187792,
    0.18780187801878018,
    0.1878118781187812,
    0.18782187821878218,
    0.18783187831878317,
    0.1878418784187842,
    0.18785187851878518,
    0.18786187861878617,
    0.1878718787187872,
    0.18788187881878818,
    0.1878918789187892,
    0.18790187901879019,
    0.18791187911879118,
    0.1879218792187922,
    0.18793187931879318,
    0.18794187941879417,
    0.1879518795187952,
    0.18796187961879618,
    0.1879718797187972,
    0.1879818798187982,
    0.18799187991879918,
    0.1880018800188002,
    0.18801188011880118,
    0.18802188021880217,
    0.1880318803188032,
    0.18804188041880418,
    0.1880518805188052,
    0.1880618806188062,
    0.18807188071880718,
    0.1880818808188082,
    0.18809188091880918,
    0.18810188101881017,
    0.1881118811188112,
    0.18812188121881218,
    0.18813188131881317,
    0.1881418814188142,
    0.18815188151881518,
    0.1881618816188162,
    0.18817188171881719,
    0.18818188181881818,
    0.1881918819188192,
    0.18820188201882018,
    0.18821188211882117,
    0.1882218822188222,
    0.18823188231882318,
    0.1882418824188242,
    0.1882518825188252,
    0.18826188261882618,
    0.1882718827188272,
    0.18828188281882818,
    0.18829188291882917,
    0.1883018830188302,
    0.18831188311883118,
    0.1883218832188322,
    0.1883318833188332,
    0.18834188341883418,
    0.1883518835188352,
    0.18836188361883618,
    0.18837188371883717,
    0.1883818838188382,
    0.18839188391883918,
    0.1884018840188402,
    0.1884118841188412,
    0.18842188421884218,
    0.1884318843188432,
    0.18844188441884419,
    0.18845188451884518,
    0.1884618846188462,
    0.18847188471884718,
    0.18848188481884817,
    0.1884918849188492,
    0.18850188501885018,
    0.1885118851188512,
    0.1885218852188522,
    0.18853188531885318,
    0.1885418854188542,
    0.18855188551885518,
    0.18856188561885617,
    0.1885718857188572,
    0.18858188581885818,
    0.1885918859188592,
    0.1886018860188602,
    0.18861188611886118,
    0.1886218862188622,
    0.18863188631886318,
    0.18864188641886417,
    0.1886518865188652,
    0.18866188661886618,
    0.1886718867188672,
    0.1886818868188682,
    0.18869188691886918,
    0.1887018870188702,
    0.18871188711887119,
    0.18872188721887218,
    0.1887318873188732,
    0.18874188741887418,
    0.18875188751887517,
    0.1887618876188762,
    0.18877188771887718,
    0.1887818878188782,
    0.1887918879188792,
    0.18880188801888018,
    0.1888118881188812,
    0.18882188821888218,
    0.18883188831888317,
    0.1888418884188842,
    0.18885188851888518,
    0.1888618886188862,
    0.1888718887188872,
    0.18888188881888818,
    0.1888918889188892,
    0.18890188901889018,
    0.18891188911889117,
    0.1889218892188922,
    0.18893188931889318,
    0.1889418894188942,
    0.1889518895188952,
    0.18896188961889618,
    0.1889718897188972,
    0.18898188981889819,
    0.18899188991889918,
    0.1890018900189002,
    0.18901189011890118,
    0.18902189021890217,
    0.1890318903189032,
    0.18904189041890418,
    0.1890518905189052,
    0.1890618906189062,
    0.18907189071890718,
    0.1890818908189082,
    0.18909189091890918,
    0.18910189101891017,
    0.1891118911189112,
    0.18912189121891218,
    0.1891318913189132,
    0.1891418914189142,
    0.18915189151891518,
    0.1891618916189162,
    0.18917189171891718,
    0.18918189181891817,
    0.1891918919189192,
    0.18920189201892018,
    0.1892118921189212,
    0.1892218922189222,
    0.18923189231892318,
    0.1892418924189242,
    0.18925189251892519,
    0.18926189261892618,
    0.1892718927189272,
    0.18928189281892818,
    0.1892918929189292,
    0.1893018930189302,
    0.18931189311893118,
    0.1893218932189322,
    0.1893318933189332,
    0.18934189341893418,
    0.1893518935189352,
    0.18936189361893618,
    0.18937189371893717,
    0.1893818938189382,
    0.18939189391893918,
    0.1894018940189402,
    0.1894118941189412,
    0.18942189421894218,
    0.1894318943189432,
    0.18944189441894418,
    0.18945189451894517,
    0.1894618946189462,
    0.18947189471894718,
    0.1894818948189482,
    0.1894918949189492,
    0.18950189501895018,
    0.1895118951189512,
    0.18952189521895219,
    0.18953189531895318,
    0.1895418954189542,
    0.18955189551895518,
    0.1895618956189562,
    0.1895718957189572,
    0.18958189581895818,
    0.1895918959189592,
    0.1896018960189602,
    0.18961189611896118,
    0.1896218962189622,
    0.18963189631896318,
    0.18964189641896417,
    0.1896518965189652,
    0.18966189661896618,
    0.1896718967189672,
    0.1896818968189682,
    0.18969189691896918,
    0.1897018970189702,
    0.18971189711897118,
    0.18972189721897217,
    0.1897318973189732,
    0.18974189741897418,
    0.1897518975189752,
    0.1897618976189762,
    0.18977189771897718,
    0.1897818978189782,
    0.18979189791897919,
    0.18980189801898018,
    0.1898118981189812,
    0.18982189821898218,
    0.1898318983189832,
    0.1898418984189842,
    0.18985189851898518,
    0.1898618986189862,
    0.1898718987189872,
    0.18988189881898818,
    0.1898918989189892,
    0.18990189901899018,
    0.18991189911899117,
    0.1899218992189922,
    0.18993189931899318,
    0.1899418994189942,
    0.1899518995189952,
    0.18996189961899618,
    0.1899718997189972,
    0.18998189981899818,
    0.18999189991899917,
    0.1900019000190002,
    0.19001190011900118,
    0.1900219002190022,
    0.1900319003190032,
    0.19004190041900418,
    0.1900519005190052,
    0.19006190061900619,
    0.19007190071900718,
    0.1900819008190082,
    0.19009190091900918,
    0.1901019010190102,
    0.1901119011190112,
    0.19012190121901218,
    0.1901319013190132,
    0.1901419014190142,
    0.19015190151901518,
    0.1901619016190162,
    0.19017190171901718,
    0.19018190181901817,
    0.1901919019190192,
    0.19020190201902018,
    0.1902119021190212,
    0.1902219022190222,
    0.19023190231902318,
    0.1902419024190242,
    0.19025190251902518,
    0.19026190261902617,
    0.1902719027190272,
    0.19028190281902818,
    0.1902919029190292,
    0.1903019030190302,
    0.19031190311903118,
    0.1903219032190322,
    0.19033190331903319,
    0.19034190341903418,
    0.1903519035190352,
    0.19036190361903618,
    0.1903719037190372,
    0.1903819038190382,
    0.19039190391903918,
    0.1904019040190402,
    0.1904119041190412,
    0.19042190421904218,
    0.1904319043190432,
    0.19044190441904418,
    0.1904519045190452,
    0.1904619046190462,
    0.19047190471904718,
    0.1904819048190482,
    0.1904919049190492,
    0.19050190501905018,
    0.1905119051190512,
    0.19052190521905218,
    0.19053190531905317,
    0.1905419054190542,
    0.19055190551905518,
    0.1905619056190562,
    0.1905719057190572,
    0.19058190581905818,
    0.1905919059190592,
    0.19060190601906019,
    0.19061190611906118,
    0.1906219062190622,
    0.19063190631906318,
    0.1906419064190642,
    0.1906519065190652,
    0.19066190661906618,
    0.1906719067190672,
    0.1906819068190682,
    0.19069190691906918,
    0.1907019070190702,
    0.19071190711907118,
    0.1907219072190722,
    0.1907319073190732,
    0.19074190741907418,
    0.1907519075190752,
    0.1907619076190762,
    0.19077190771907718,
    0.1907819078190782,
    0.19079190791907918,
    0.19080190801908017,
    0.1908119081190812,
    0.19082190821908218,
    0.1908319083190832,
    0.1908419084190842,
    0.19085190851908518,
    0.1908619086190862,
    0.19087190871908719,
    0.19088190881908818,
    0.1908919089190892,
    0.19090190901909018,
    0.1909119091190912,
    0.1909219092190922,
    0.19093190931909318,
    0.1909419094190942,
    0.1909519095190952,
    0.19096190961909618,
    0.1909719097190972,
    0.19098190981909818,
    0.1909919099190992,
    0.1910019100191002,
    0.19101191011910118,
    0.1910219102191022,
    0.1910319103191032,
    0.19104191041910418,
    0.1910519105191052,
    0.19106191061910618,
    0.19107191071910717,
    0.1910819108191082,
    0.19109191091910918,
    0.1911019110191102,
    0.1911119111191112,
    0.19112191121911218,
    0.1911319113191132,
    0.19114191141911419,
    0.19115191151911518,
    0.1911619116191162,
    0.19117191171911718,
    0.1911819118191182,
    0.1911919119191192,
    0.19120191201912018,
    0.1912119121191212,
    0.1912219122191222,
    0.19123191231912318,
    0.1912419124191242,
    0.19125191251912518,
    0.1912619126191262,
    0.1912719127191272,
    0.19128191281912818,
    0.1912919129191292,
    0.1913019130191302,
    0.19131191311913118,
    0.1913219132191322,
    0.19133191331913318,
    0.1913419134191342,
    0.1913519135191352,
    0.19136191361913618,
    0.1913719137191372,
    0.1913819138191382,
    0.19139191391913918,
    0.1914019140191402,
    0.19141191411914119,
    0.19142191421914218,
    0.1914319143191432,
    0.19144191441914418,
    0.1914519145191452,
    0.1914619146191462,
    0.19147191471914718,
    0.1914819148191482,
    0.1914919149191492,
    0.19150191501915018,
    0.1915119151191512,
    0.19152191521915218,
    0.1915319153191532,
    0.1915419154191542,
    0.19155191551915518,
    0.1915619156191562,
    0.1915719157191572,
    0.19158191581915818,
    0.1915919159191592,
    0.19160191601916018,
    0.1916119161191612,
    0.1916219162191622,
    0.19163191631916318,
    0.1916419164191642,
    0.1916519165191652,
    0.19166191661916618,
    0.1916719167191672,
    0.19168191681916819,
    0.19169191691916918,
    0.1917019170191702,
    0.19171191711917118,
    0.1917219172191722,
    0.1917319173191732,
    0.19174191741917418,
    0.1917519175191752,
    0.1917619176191762,
    0.19177191771917718,
    0.1917819178191782,
    0.19179191791917918,
    0.1918019180191802,
    0.1918119181191812,
    0.19182191821918218,
    0.1918319183191832,
    0.1918419184191842,
    0.19185191851918518,
    0.1918619186191862,
    0.19187191871918718,
    0.1918819188191882,
    0.1918919189191892,
    0.19190191901919018,
    0.1919119191191912,
    0.1919219192191922,
    0.19193191931919318,
    0.1919419194191942,
    0.19195191951919519,
    0.19196191961919618,
    0.1919719197191972,
    0.19198191981919818,
    0.1919919199191992,
    0.1920019200192002,
    0.19201192011920118,
    0.1920219202192022,
    0.1920319203192032,
    0.19204192041920418,
    0.1920519205192052,
    0.19206192061920618,
    0.1920719207192072,
    0.1920819208192082,
    0.19209192091920918,
    0.1921019210192102,
    0.1921119211192112,
    0.19212192121921218,
    0.1921319213192132,
    0.19214192141921418,
    0.1921519215192152,
    0.1921619216192162,
    0.19217192171921718,
    0.1921819218192182,
    0.1921919219192192,
    0.19220192201922018,
    0.1922119221192212,
    0.19222192221922219,
    0.19223192231922318,
    0.1922419224192242,
    0.19225192251922518,
    0.1922619226192262,
    0.1922719227192272,
    0.19228192281922818,
    0.1922919229192292,
    0.1923019230192302,
    0.19231192311923118,
    0.1923219232192322,
    0.19233192331923318,
    0.1923419234192342,
    0.1923519235192352,
    0.19236192361923618,
    0.1923719237192372,
    0.1923819238192382,
    0.19239192391923918,
    0.1924019240192402,
    0.19241192411924118,
    0.1924219242192422,
    0.1924319243192432,
    0.19244192441924418,
    0.1924519245192452,
    0.1924619246192462,
    0.19247192471924718,
    0.1924819248192482,
    0.19249192491924919,
    0.1925019250192502,
    0.1925119251192512,
    0.19252192521925218,
    0.1925319253192532,
    0.1925419254192542,
    0.19255192551925518,
    0.1925619256192562,
    0.1925719257192572,
    0.19258192581925818,
    0.1925919259192592,
    0.19260192601926018,
    0.1926119261192612,
    0.1926219262192622,
    0.19263192631926318,
    0.1926419264192642,
    0.1926519265192652,
    0.19266192661926618,
    0.1926719267192672,
    0.19268192681926818,
    0.1926919269192692,
    0.1927019270192702,
    0.19271192711927118,
    0.1927219272192722,
    0.1927319273192732,
    0.19274192741927418,
    0.1927519275192752,
    0.19276192761927619,
    0.1927719277192772,
    0.1927819278192782,
    0.19279192791927918,
    0.1928019280192802,
    0.1928119281192812,
    0.19282192821928218,
    0.1928319283192832,
    0.1928419284192842,
    0.19285192851928518,
    0.1928619286192862,
    0.19287192871928718,
    0.1928819288192882,
    0.1928919289192892,
    0.19290192901929018,
    0.1929119291192912,
    0.1929219292192922,
    0.19293192931929318,
    0.1929419294192942,
    0.19295192951929518,
    0.1929619296192962,
    0.1929719297192972,
    0.19298192981929818,
    0.1929919299192992,
    0.1930019300193002,
    0.19301193011930118,
    0.1930219302193022,
    0.19303193031930319,
    0.1930419304193042,
    0.1930519305193052,
    0.19306193061930618,
    0.1930719307193072,
    0.1930819308193082,
    0.19309193091930918,
    0.1931019310193102,
    0.1931119311193112,
    0.19312193121931218,
    0.1931319313193132,
    0.19314193141931418,
    0.1931519315193152,
    0.1931619316193162,
    0.19317193171931718,
    0.1931819318193182,
    0.1931919319193192,
    0.19320193201932018,
    0.1932119321193212,
    0.19322193221932218,
    0.1932319323193232,
    0.1932419324193242,
    0.19325193251932518,
    0.1932619326193262,
    0.1932719327193272,
    0.19328193281932818,
    0.1932919329193292,
    0.19330193301933019,
    0.1933119331193312,
    0.1933219332193322,
    0.19333193331933318,
    0.1933419334193342,
    0.1933519335193352,
    0.19336193361933618,
    0.1933719337193372,
    0.1933819338193382,
    0.19339193391933918,
    0.1934019340193402,
    0.19341193411934118,
    0.1934219342193422,
    0.1934319343193432,
    0.19344193441934418,
    0.1934519345193452,
    0.1934619346193462,
    0.19347193471934718,
    0.1934819348193482,
    0.19349193491934918,
    0.1935019350193502,
    0.1935119351193512,
    0.19352193521935218,
    0.1935319353193532,
    0.1935419354193542,
    0.19355193551935518,
    0.1935619356193562,
    0.19357193571935719,
    0.1935819358193582,
    0.1935919359193592,
    0.19360193601936018,
    0.1936119361193612,
    0.1936219362193622,
    0.19363193631936318,
    0.1936419364193642,
    0.1936519365193652,
    0.1936619366193662,
    0.1936719367193672,
    0.19368193681936818,
    0.1936919369193692,
    0.1937019370193702,
    0.19371193711937118,
    0.1937219372193722,
    0.1937319373193732,
    0.19374193741937418,
    0.1937519375193752,
    0.19376193761937618,
    0.1937719377193772,
    0.1937819378193782,
    0.19379193791937918,
    0.1938019380193802,
    0.1938119381193812,
    0.19382193821938218,
    0.1938319383193832,
    0.19384193841938419,
    0.1938519385193852,
    0.1938619386193862,
    0.19387193871938718,
    0.1938819388193882,
    0.1938919389193892,
    0.19390193901939018,
    0.1939119391193912,
    0.1939219392193922,
    0.1939319393193932,
    0.1939419394193942,
    0.19395193951939518,
    0.1939619396193962,
    0.1939719397193972,
    0.19398193981939818,
    0.1939919399193992,
    0.1940019400194002,
    0.19401194011940118,
    0.1940219402194022,
    0.19403194031940318,
    0.1940419404194042,
    0.1940519405194052,
    0.19406194061940618,
    0.1940719407194072,
    0.1940819408194082,
    0.19409194091940918,
    0.1941019410194102,
    0.19411194111941119,
    0.1941219412194122,
    0.1941319413194132,
    0.19414194141941418,
    0.1941519415194152,
    0.1941619416194162,
    0.19417194171941718,
    0.1941819418194182,
    0.1941919419194192,
    0.1942019420194202,
    0.1942119421194212,
    0.19422194221942218,
    0.1942319423194232,
    0.1942419424194242,
    0.19425194251942518,
    0.1942619426194262,
    0.1942719427194272,
    0.19428194281942818,
    0.1942919429194292,
    0.19430194301943018,
    0.1943119431194312,
    0.1943219432194322,
    0.19433194331943318,
    0.1943419434194342,
    0.1943519435194352,
    0.19436194361943618,
    0.1943719437194372,
    0.19438194381943819,
    0.1943919439194392,
    0.1944019440194402,
    0.19441194411944118,
    0.1944219442194422,
    0.1944319443194432,
    0.19444194441944418,
    0.1944519445194452,
    0.1944619446194462,
    0.1944719447194472,
    0.1944819448194482,
    0.19449194491944918,
    0.1945019450194502,
    0.1945119451194512,
    0.19452194521945218,
    0.1945319453194532,
    0.1945419454194542,
    0.1945519455194552,
    0.1945619456194562,
    0.19457194571945718,
    0.1945819458194582,
    0.1945919459194592,
    0.19460194601946018,
    0.1946119461194612,
    0.1946219462194622,
    0.19463194631946318,
    0.1946419464194642,
    0.19465194651946519,
    0.1946619466194662,
    0.1946719467194672,
    0.19468194681946818,
    0.1946919469194692,
    0.1947019470194702,
    0.19471194711947118,
    0.1947219472194722,
    0.1947319473194732,
    0.1947419474194742,
    0.1947519475194752,
    0.19476194761947618,
    0.1947719477194772,
    0.1947819478194782,
    0.19479194791947918,
    0.1948019480194802,
    0.1948119481194812,
    0.1948219482194822,
    0.1948319483194832,
    0.19484194841948418,
    0.1948519485194852,
    0.1948619486194862,
    0.19487194871948718,
    0.1948819488194882,
    0.1948919489194892,
    0.19490194901949018,
    0.1949119491194912,
    0.19492194921949219,
    0.1949319493194932,
    0.1949419494194942,
    0.19495194951949518,
    0.1949619496194962,
    0.1949719497194972,
    0.19498194981949818,
    0.1949919499194992,
    0.1950019500195002,
    0.1950119501195012,
    0.1950219502195022,
    0.19503195031950318,
    0.1950419504195042,
    0.1950519505195052,
    0.19506195061950618,
    0.1950719507195072,
    0.1950819508195082,
    0.1950919509195092,
    0.1951019510195102,
    0.19511195111951118,
    0.1951219512195122,
    0.1951319513195132,
    0.19514195141951418,
    0.1951519515195152,
    0.1951619516195162,
    0.19517195171951718,
    0.1951819518195182,
    0.19519195191951919,
    0.1952019520195202,
    0.1952119521195212,
    0.19522195221952218,
    0.1952319523195232,
    0.1952419524195242,
    0.19525195251952518,
    0.1952619526195262,
    0.1952719527195272,
    0.1952819528195282,
    0.1952919529195292,
    0.19530195301953018,
    0.1953119531195312,
    0.1953219532195322,
    0.19533195331953318,
    0.1953419534195342,
    0.1953519535195352,
    0.1953619536195362,
    0.1953719537195372,
    0.19538195381953818,
    0.1953919539195392,
    0.1954019540195402,
    0.19541195411954118,
    0.1954219542195422,
    0.1954319543195432,
    0.19544195441954418,
    0.1954519545195452,
    0.19546195461954619,
    0.1954719547195472,
    0.1954819548195482,
    0.19549195491954918,
    0.1955019550195502,
    0.1955119551195512,
    0.19552195521955218,
    0.1955319553195532,
    0.1955419554195542,
    0.1955519555195552,
    0.1955619556195562,
    0.19557195571955718,
    0.1955819558195582,
    0.1955919559195592,
    0.19560195601956018,
    0.1956119561195612,
    0.1956219562195622,
    0.1956319563195632,
    0.1956419564195642,
    0.19565195651956518,
    0.1956619566195662,
    0.1956719567195672,
    0.19568195681956818,
    0.1956919569195692,
    0.1957019570195702,
    0.1957119571195712,
    0.1957219572195722,
    0.19573195731957319,
    0.1957419574195742,
    0.1957519575195752,
    0.19576195761957618,
    0.1957719577195772,
    0.1957819578195782,
    0.19579195791957918,
    0.1958019580195802,
    0.1958119581195812,
    0.1958219582195822,
    0.1958319583195832,
    0.19584195841958418,
    0.1958519585195852,
    0.1958619586195862,
    0.19587195871958718,
    0.1958819588195882,
    0.1958919589195892,
    0.1959019590195902,
    0.1959119591195912,
    0.19592195921959218,
    0.1959319593195932,
    0.1959419594195942,
    0.19595195951959518,
    0.1959619596195962,
    0.1959719597195972,
    0.1959819598195982,
    0.1959919599195992,
    0.19600196001960019,
    0.1960119601196012,
    0.1960219602196022,
    0.19603196031960318,
    0.1960419604196042,
    0.1960519605196052,
    0.19606196061960618,
    0.1960719607196072,
    0.1960819608196082,
    0.1960919609196092,
    0.1961019610196102,
    0.19611196111961118,
    0.1961219612196122,
    0.1961319613196132,
    0.19614196141961418,
    0.1961519615196152,
    0.1961619616196162,
    0.1961719617196172,
    0.1961819618196182,
    0.19619196191961918,
    0.1962019620196202,
    0.1962119621196212,
    0.19622196221962218,
    0.1962319623196232,
    0.1962419624196242,
    0.1962519625196252,
    0.1962619626196262,
    0.19627196271962719,
    0.1962819628196282,
    0.1962919629196292,
    0.19630196301963018,
    0.1963119631196312,
    0.1963219632196322,
    0.19633196331963318,
    0.1963419634196342,
    0.1963519635196352,
    0.1963619636196362,
    0.1963719637196372,
    0.19638196381963818,
    0.1963919639196392,
    0.1964019640196402,
    0.19641196411964118,
    0.1964219642196422,
    0.1964319643196432,
    0.1964419644196442,
    0.1964519645196452,
    0.19646196461964618,
    0.1964719647196472,
    0.1964819648196482,
    0.19649196491964918,
    0.1965019650196502,
    0.1965119651196512,
    0.1965219652196522,
    0.1965319653196532,
    0.19654196541965419,
    0.1965519655196552,
    0.1965619656196562,
    0.19657196571965718,
    0.1965819658196582,
    0.1965919659196592,
    0.1966019660196602,
    0.1966119661196612,
    0.1966219662196622,
    0.1966319663196632,
    0.1966419664196642,
    0.19665196651966518,
    0.1966619666196662,
    0.1966719667196672,
    0.19668196681966818,
    0.1966919669196692,
    0.1967019670196702,
    0.1967119671196712,
    0.1967219672196722,
    0.19673196731967318,
    0.1967419674196742,
    0.1967519675196752,
    0.19676196761967618,
    0.1967719677196772,
    0.1967819678196782,
    0.1967919679196792,
    0.1968019680196802,
    0.19681196811968119,
    0.1968219682196822,
    0.1968319683196832,
    0.19684196841968418,
    0.1968519685196852,
    0.1968619686196862,
    0.1968719687196872,
    0.1968819688196882,
    0.1968919689196892,
    0.1969019690196902,
    0.1969119691196912,
    0.19692196921969218,
    0.1969319693196932,
    0.1969419694196942,
    0.19695196951969518,
    0.1969619696196962,
    0.1969719697196972,
    0.1969819698196982,
    0.1969919699196992,
    0.19700197001970018,
    0.1970119701197012,
    0.1970219702197022,
    0.19703197031970318,
    0.1970419704197042,
    0.1970519705197052,
    0.1970619706197062,
    0.1970719707197072,
    0.19708197081970819,
    0.1970919709197092,
    0.1971019710197102,
    0.19711197111971118,
    0.1971219712197122,
    0.1971319713197132,
    0.1971419714197142,
    0.1971519715197152,
    0.1971619716197162,
    0.1971719717197172,
    0.1971819718197182,
    0.19719197191971918,
    0.1972019720197202,
    0.1972119721197212,
    0.19722197221972218,
    0.1972319723197232,
    0.1972419724197242,
    0.1972519725197252,
    0.1972619726197262,
    0.19727197271972718,
    0.1972819728197282,
    0.1972919729197292,
    0.19730197301973018,
    0.1973119731197312,
    0.1973219732197322,
    0.1973319733197332,
    0.1973419734197342,
    0.19735197351973519,
    0.1973619736197362,
    0.1973719737197372,
    0.19738197381973818,
    0.1973919739197392,
    0.1974019740197402,
    0.1974119741197412,
    0.1974219742197422,
    0.1974319743197432,
    0.1974419744197442,
    0.1974519745197452,
    0.19746197461974618,
    0.1974719747197472,
    0.1974819748197482,
    0.19749197491974918,
    0.1975019750197502,
    0.1975119751197512,
    0.1975219752197522,
    0.1975319753197532,
    0.19754197541975418,
    0.1975519755197552,
    0.1975619756197562,
    0.19757197571975718,
    0.1975819758197582,
    0.1975919759197592,
    0.1976019760197602,
    0.1976119761197612,
    0.19762197621976219,
    0.1976319763197632,
    0.1976419764197642,
    0.19765197651976518,
    0.1976619766197662,
    0.1976719767197672,
    0.1976819768197682,
    0.1976919769197692,
    0.1977019770197702,
    0.1977119771197712,
    0.1977219772197722,
    0.19773197731977318,
    0.1977419774197742,
    0.1977519775197752,
    0.1977619776197762,
    0.1977719777197772,
    0.1977819778197782,
    0.1977919779197792,
    0.1978019780197802,
    0.19781197811978118,
    0.1978219782197822,
    0.1978319783197832,
    0.19784197841978418,
    0.1978519785197852,
    0.1978619786197862,
    0.1978719787197872,
    0.1978819788197882,
    0.19789197891978919,
    0.1979019790197902,
    0.1979119791197912,
    0.19792197921979218,
    0.1979319793197932,
    0.1979419794197942,
    0.1979519795197952,
    0.1979619796197962,
    0.1979719797197972,
    0.1979819798197982,
    0.1979919799197992,
    0.19800198001980018,
    0.1980119801198012,
    0.1980219802198022,
    0.1980319803198032,
    0.1980419804198042,
    0.1980519805198052,
    0.1980619806198062,
    0.1980719807198072,
    0.19808198081980818,
    0.1980919809198092,
    0.1981019810198102,
    0.19811198111981118,
    0.1981219812198122,
    0.1981319813198132,
    0.1981419814198142,
    0.1981519815198152,
    0.19816198161981619,
    0.1981719817198172,
    0.1981819818198182,
    0.19819198191981918,
    0.1982019820198202,
    0.1982119821198212,
    0.1982219822198222,
    0.1982319823198232,
    0.1982419824198242,
    0.1982519825198252,
    0.1982619826198262,
    0.19827198271982718,
    0.1982819828198282,
    0.1982919829198292,
    0.1983019830198302,
    0.1983119831198312,
    0.1983219832198322,
    0.1983319833198332,
    0.1983419834198342,
    0.19835198351983518,
    0.1983619836198362,
    0.1983719837198372,
    0.19838198381983818,
    0.1983919839198392,
    0.1984019840198402,
    0.1984119841198412,
    0.1984219842198422,
    0.19843198431984319,
    0.1984419844198442,
    0.1984519845198452,
    0.19846198461984618,
    0.1984719847198472,
    0.1984819848198482,
    0.1984919849198492,
    0.1985019850198502,
    0.1985119851198512,
    0.1985219852198522,
    0.1985319853198532,
    0.19854198541985418,
    0.1985519855198552,
    0.1985619856198562,
    0.1985719857198572,
    0.1985819858198582,
    0.1985919859198592,
    0.1986019860198602,
    0.1986119861198612,
    0.19862198621986218,
    0.1986319863198632,
    0.1986419864198642,
    0.1986519865198652,
    0.1986619866198662,
    0.1986719867198672,
    0.1986819868198682,
    0.1986919869198692,
    0.19870198701987019,
    0.1987119871198712,
    0.1987219872198722,
    0.19873198731987318,
    0.1987419874198742,
    0.1987519875198752,
    0.1987619876198762,
    0.1987719877198772,
    0.1987819878198782,
    0.1987919879198792,
    0.1988019880198802,
    0.19881198811988118,
    0.1988219882198822,
    0.1988319883198832,
    0.1988419884198842,
    0.1988519885198852,
    0.1988619886198862,
    0.1988719887198872,
    0.1988819888198882,
    0.19889198891988918,
    0.1989019890198902,
    0.1989119891198912,
    0.1989219892198922,
    0.1989319893198932,
    0.1989419894198942,
    0.1989519895198952,
    0.1989619896198962,
    0.19897198971989719,
    0.1989819898198982,
    0.1989919899198992,
    0.19900199001990018,
    0.1990119901199012,
    0.1990219902199022,
    0.1990319903199032,
    0.1990419904199042,
    0.1990519905199052,
    0.1990619906199062,
    0.1990719907199072,
    0.19908199081990818,
    0.1990919909199092,
    0.1991019910199102,
    0.1991119911199112,
    0.1991219912199122,
    0.1991319913199132,
    0.1991419914199142,
    0.1991519915199152,
    0.19916199161991618,
    0.1991719917199172,
    0.1991819918199182,
    0.1991919919199192,
    0.1992019920199202,
    0.1992119921199212,
    0.1992219922199222,
    0.1992319923199232,
    0.19924199241992419,
    0.1992519925199252,
    0.1992619926199262,
    0.19927199271992718,
    0.1992819928199282,
    0.1992919929199292,
    0.1993019930199302,
    0.1993119931199312,
    0.1993219932199322,
    0.1993319933199332,
    0.1993419934199342,
    0.19935199351993518,
    0.1993619936199362,
    0.1993719937199372,
    0.1993819938199382,
    0.1993919939199392,
    0.1994019940199402,
    0.1994119941199412,
    0.1994219942199422,
    0.19943199431994318,
    0.1994419944199442,
    0.1994519945199452,
    0.1994619946199462,
    0.1994719947199472,
    0.1994819948199482,
    0.1994919949199492,
    0.1995019950199502,
    0.19951199511995119,
    0.1995219952199522,
    0.1995319953199532,
    0.19954199541995418,
    0.1995519955199552,
    0.1995619956199562,
    0.1995719957199572,
    0.1995819958199582,
    0.1995919959199592,
    0.1996019960199602,
    0.1996119961199612,
    0.19962199621996218,
    0.1996319963199632,
    0.1996419964199642,
    0.1996519965199652,
    0.1996619966199662,
    0.1996719967199672,
    0.1996819968199682,
    0.1996919969199692,
    0.19970199701997018,
    0.1997119971199712,
    0.1997219972199722,
    0.1997319973199732,
    0.1997419974199742,
    0.1997519975199752,
    0.1997619976199762,
    0.1997719977199772,
    0.19978199781997819,
    0.1997919979199792,
    0.1998019980199802,
    0.1998119981199812,
    0.1998219982199822,
    0.1998319983199832,
    0.1998419984199842,
    0.1998519985199852,
    0.1998619986199862,
    0.1998719987199872,
    0.1998819988199882,
    0.19989199891998918,
    0.1999019990199902,
    0.1999119991199912,
    0.1999219992199922,
    0.1999319993199932,
    0.1999419994199942,
    0.1999519995199952,
    0.1999619996199962,
    0.19997199971999718,
    0.1999819998199982,
    0.1999919999199992,
    0.2000020000200002,
    0.2000120001200012,
    0.2000220002200022,
    0.2000320003200032,
    0.2000420004200042,
    0.20005200052000519,
    0.2000620006200062,
    0.2000720007200072,
    0.2000820008200082,
    0.2000920009200092,
    0.2001020010200102,
    0.2001120011200112,
    0.2001220012200122,
    0.2001320013200132,
    0.2001420014200142,
    0.2001520015200152,
    0.20016200162001618,
    0.2001720017200172,
    0.2001820018200182,
    0.2001920019200192,
    0.2002020020200202,
    0.2002120021200212,
    0.2002220022200222,
    0.2002320023200232,
    0.20024200242002418,
    0.2002520025200252,
    0.2002620026200262,
    0.2002720027200272,
    0.2002820028200282,
    0.2002920029200292,
    0.2003020030200302,
    0.2003120031200312,
    0.20032200322003219,
    0.2003320033200332,
    0.2003420034200342,
    0.2003520035200352,
    0.2003620036200362,
    0.2003720037200372,
    0.2003820038200382,
    0.2003920039200392,
    0.2004020040200402,
    0.2004120041200412,
    0.2004220042200422,
    0.20043200432004318,
    0.2004420044200442,
    0.2004520045200452,
    0.2004620046200462,
    0.2004720047200472,
    0.2004820048200482,
    0.2004920049200492,
    0.2005020050200502,
    0.20051200512005118,
    0.2005220052200522,
    0.2005320053200532,
    0.2005420054200542,
    0.2005520055200552,
    0.2005620056200562,
    0.2005720057200572,
    0.2005820058200582,
    0.20059200592005919,
    0.2006020060200602,
    0.2006120061200612,
    0.2006220062200622,
    0.2006320063200632,
    0.2006420064200642,
    0.2006520065200652,
    0.2006620066200662,
    0.2006720067200672,
    0.2006820068200682,
    0.2006920069200692,
    0.2007020070200702,
    0.2007120071200712,
    0.2007220072200722,
    0.2007320073200732,
    0.2007420074200742,
    0.2007520075200752,
    0.2007620076200762,
    0.2007720077200772,
    0.20078200782007818,
    0.2007920079200792,
    0.2008020080200802,
    0.2008120081200812,
    0.2008220082200822,
    0.2008320083200832,
    0.2008420084200842,
    0.2008520085200852,
    0.20086200862008619,
    0.2008720087200872,
    0.2008820088200882,
    0.2008920089200892,
    0.2009020090200902,
    0.2009120091200912,
    0.2009220092200922,
    0.2009320093200932,
    0.2009420094200942,
    0.2009520095200952,
    0.2009620096200962,
    0.2009720097200972,
    0.2009820098200982,
    0.2009920099200992,
    0.2010020100201002,
    0.2010120101201012,
    0.2010220102201022,
    0.2010320103201032,
    0.2010420104201042,
    0.20105201052010518,
    0.2010620106201062,
    0.2010720107201072,
    0.2010820108201082,
    0.2010920109201092,
    0.2011020110201102,
    0.2011120111201112,
    0.2011220112201122,
    0.20113201132011319,
    0.2011420114201142,
    0.2011520115201152,
    0.2011620116201162,
    0.2011720117201172,
    0.2011820118201182,
    0.2011920119201192,
    0.2012020120201202,
    0.2012120121201212,
    0.2012220122201222,
    0.2012320123201232,
    0.2012420124201242,
    0.2012520125201252,
    0.2012620126201262,
    0.2012720127201272,
    0.2012820128201282,
    0.2012920129201292,
    0.2013020130201302,
    0.2013120131201312,
    0.20132201322013218,
    0.2013320133201332,
    0.2013420134201342,
    0.2013520135201352,
    0.2013620136201362,
    0.2013720137201372,
    0.2013820138201382,
    0.2013920139201392,
    0.20140201402014019,
    0.2014120141201412,
    0.2014220142201422,
    0.2014320143201432,
    0.2014420144201442,
    0.2014520145201452,
    0.2014620146201462,
    0.2014720147201472,
    0.2014820148201482,
    0.2014920149201492,
    0.2015020150201502,
    0.2015120151201512,
    0.2015220152201522,
    0.2015320153201532,
    0.2015420154201542,
    0.2015520155201552,
    0.2015620156201562,
    0.2015720157201572,
    0.2015820158201582,
    0.20159201592015918,
    0.2016020160201602,
    0.2016120161201612,
    0.2016220162201622,
    0.2016320163201632,
    0.2016420164201642,
    0.2016520165201652,
    0.2016620166201662,
    0.20167201672016719,
    0.2016820168201682,
    0.2016920169201692,
    0.2017020170201702,
    0.2017120171201712,
    0.2017220172201722,
    0.2017320173201732,
    0.2017420174201742,
    0.2017520175201752,
    0.2017620176201762,
    0.2017720177201772,
    0.2017820178201782,
    0.2017920179201792,
    0.2018020180201802,
    0.2018120181201812,
    0.2018220182201822,
    0.2018320183201832,
    0.2018420184201842,
    0.2018520185201852,
    0.2018620186201862,
    0.2018720187201872,
    0.2018820188201882,
    0.2018920189201892,
    0.2019020190201902,
    0.2019120191201912,
    0.2019220192201922,
    0.2019320193201932,
    0.20194201942019419,
    0.2019520195201952,
    0.2019620196201962,
    0.2019720197201972,
    0.2019820198201982,
    0.2019920199201992,
    0.2020020200202002,
    0.2020120201202012,
    0.2020220202202022,
    0.2020320203202032,
    0.2020420204202042,
    0.2020520205202052,
    0.2020620206202062,
    0.2020720207202072,
    0.2020820208202082,
    0.2020920209202092,
    0.2021020210202102,
    0.2021120211202112,
    0.2021220212202122,
    0.2021320213202132,
    0.2021420214202142,
    0.2021520215202152,
    0.2021620216202162,
    0.2021720217202172,
    0.2021820218202182,
    0.2021920219202192,
    0.2022020220202202,
    0.20221202212022119,
    0.2022220222202222,
    0.2022320223202232,
    0.2022420224202242,
    0.2022520225202252,
    0.2022620226202262,
    0.2022720227202272,
    0.2022820228202282,
    0.2022920229202292,
    0.2023020230202302,
    0.2023120231202312,
    0.2023220232202322,
    0.2023320233202332,
    0.2023420234202342,
    0.2023520235202352,
    0.2023620236202362,
    0.2023720237202372,
    0.2023820238202382,
    0.2023920239202392,
    0.2024020240202402,
    0.2024120241202412,
    0.2024220242202422,
    0.2024320243202432,
    0.2024420244202442,
    0.2024520245202452,
    0.2024620246202462,
    0.2024720247202472,
    0.20248202482024819,
    0.2024920249202492,
    0.2025020250202502,
    0.2025120251202512,
    0.2025220252202522,
    0.2025320253202532,
    0.2025420254202542,
    0.2025520255202552,
    0.2025620256202562,
    0.2025720257202572,
    0.2025820258202582,
    0.2025920259202592,
    0.2026020260202602,
    0.2026120261202612,
    0.2026220262202622,
    0.2026320263202632,
    0.2026420264202642,
    0.2026520265202652,
    0.2026620266202662,
    0.2026720267202672,
    0.2026820268202682,
    0.2026920269202692,
    0.2027020270202702,
    0.2027120271202712,
    0.2027220272202722,
    0.2027320273202732,
    0.2027420274202742,
    0.2027520275202752,
    0.2027620276202762,
    0.2027720277202772,
    0.2027820278202782,
    0.2027920279202792,
    0.2028020280202802,
    0.2028120281202812,
    0.2028220282202822,
    0.2028320283202832,
    0.2028420284202842,
    0.2028520285202852,
    0.2028620286202862,
    0.2028720287202872,
    0.2028820288202882,
    0.2028920289202892,
    0.2029020290202902,
    0.2029120291202912,
    0.2029220292202922,
    0.2029320293202932,
    0.2029420294202942,
    0.2029520295202952,
    0.2029620296202962,
    0.2029720297202972,
    0.2029820298202982,
    0.2029920299202992,
    0.2030020300203002,
    0.2030120301203012,
    0.2030220302203022,
    0.2030320303203032,
    0.2030420304203042,
    0.2030520305203052,
    0.2030620306203062,
    0.2030720307203072,
    0.2030820308203082,
    0.2030920309203092,
    0.2031020310203102,
    0.2031120311203112,
    0.2031220312203122,
    0.2031320313203132,
    0.2031420314203142,
    0.2031520315203152,
    0.2031620316203162,
    0.2031720317203172,
    0.2031820318203182,
    0.2031920319203192,
    0.2032020320203202,
    0.2032120321203212,
    0.2032220322203222,
    0.2032320323203232,
    0.2032420324203242,
    0.2032520325203252,
    0.2032620326203262,
    0.2032720327203272,
    0.2032820328203282,
    0.2032920329203292,
    0.2033020330203302,
    0.2033120331203312,
    0.2033220332203322,
    0.2033320333203332,
    0.2033420334203342,
    0.2033520335203352,
    0.2033620336203362,
    0.2033720337203372,
    0.2033820338203382,
    0.2033920339203392,
    0.2034020340203402,
    0.2034120341203412,
    0.2034220342203422,
    0.2034320343203432,
    0.2034420344203442,
    0.2034520345203452,
    0.2034620346203462,
    0.2034720347203472,
    0.2034820348203482,
    0.2034920349203492,
    0.2035020350203502,
    0.2035120351203512,
    0.2035220352203522,
    0.2035320353203532,
    0.2035420354203542,
    0.2035520355203552,
    0.2035620356203562,
    0.2035720357203572,
    0.2035820358203582,
    0.2035920359203592,
    0.2036020360203602,
    0.2036120361203612,
    0.2036220362203622,
    0.2036320363203632,
    0.2036420364203642,
    0.2036520365203652,
    0.2036620366203662,
    0.2036720367203672,
    0.2036820368203682,
    0.2036920369203692,
    0.2037020370203702,
    0.2037120371203712,
    0.2037220372203722,
    0.2037320373203732,
    0.2037420374203742,
    0.2037520375203752,
    0.2037620376203762,
    0.2037720377203772,
    0.2037820378203782,
    0.2037920379203792,
    0.2038020380203802,
    0.2038120381203812,
    0.2038220382203822,
    0.2038320383203832,
    0.2038420384203842,
    0.2038520385203852,
    0.2038620386203862,
    0.2038720387203872,
    0.2038820388203882,
    0.2038920389203892,
    0.2039020390203902,
    0.20391203912039121,
    0.2039220392203922,
    0.2039320393203932,
    0.2039420394203942,
    0.2039520395203952,
    0.2039620396203962,
    0.2039720397203972,
    0.2039820398203982,
    0.2039920399203992,
    0.2040020400204002,
    0.2040120401204012,
    0.2040220402204022,
    0.2040320403204032,
    0.2040420404204042,
    0.2040520405204052,
    0.2040620406204062,
    0.2040720407204072,
    0.2040820408204082,
    0.2040920409204092,
    0.2041020410204102,
    0.2041120411204112,
    0.2041220412204122,
    0.2041320413204132,
    0.2041420414204142,
    0.2041520415204152,
    0.2041620416204162,
    0.2041720417204172,
    0.20418204182041821,
    0.2041920419204192,
    0.2042020420204202,
    0.2042120421204212,
    0.2042220422204222,
    0.2042320423204232,
    0.2042420424204242,
    0.2042520425204252,
    0.2042620426204262,
    0.2042720427204272,
    0.2042820428204282,
    0.2042920429204292,
    0.2043020430204302,
    0.2043120431204312,
    0.2043220432204322,
    0.2043320433204332,
    0.2043420434204342,
    0.2043520435204352,
    0.2043620436204362,
    0.2043720437204372,
    0.2043820438204382,
    0.2043920439204392,
    0.2044020440204402,
    0.2044120441204412,
    0.2044220442204422,
    0.2044320443204432,
    0.2044420444204442,
    0.20445204452044521,
    0.2044620446204462,
    0.2044720447204472,
    0.2044820448204482,
    0.2044920449204492,
    0.2045020450204502,
    0.2045120451204512,
    0.2045220452204522,
    0.2045320453204532,
    0.2045420454204542,
    0.2045520455204552,
    0.2045620456204562,
    0.2045720457204572,
    0.2045820458204582,
    0.2045920459204592,
    0.2046020460204602,
    0.2046120461204612,
    0.2046220462204622,
    0.2046320463204632,
    0.2046420464204642,
    0.2046520465204652,
    0.2046620466204662,
    0.2046720467204672,
    0.2046820468204682,
    0.2046920469204692,
    0.2047020470204702,
    0.2047120471204712,
    0.20472204722047221,
    0.2047320473204732,
    0.2047420474204742,
    0.2047520475204752,
    0.2047620476204762,
    0.2047720477204772,
    0.2047820478204782,
    0.2047920479204792,
    0.2048020480204802,
    0.2048120481204812,
    0.2048220482204822,
    0.2048320483204832,
    0.2048420484204842,
    0.2048520485204852,
    0.2048620486204862,
    0.2048720487204872,
    0.2048820488204882,
    0.2048920489204892,
    0.2049020490204902,
    0.2049120491204912,
    0.2049220492204922,
    0.2049320493204932,
    0.2049420494204942,
    0.2049520495204952,
    0.2049620496204962,
    0.2049720497204972,
    0.2049820498204982,
    0.20499204992049921,
    0.2050020500205002,
    0.2050120501205012,
    0.2050220502205022,
    0.2050320503205032,
    0.2050420504205042,
    0.2050520505205052,
    0.2050620506205062,
    0.20507205072050722,
    0.2050820508205082,
    0.2050920509205092,
    0.2051020510205102,
    0.2051120511205112,
    0.2051220512205122,
    0.2051320513205132,
    0.2051420514205142,
    0.2051520515205152,
    0.2051620516205162,
    0.2051720517205172,
    0.2051820518205182,
    0.2051920519205192,
    0.2052020520205202,
    0.2052120521205212,
    0.2052220522205222,
    0.2052320523205232,
    0.2052420524205242,
    0.2052520525205252,
    0.20526205262052621,
    0.2052720527205272,
    0.2052820528205282,
    0.2052920529205292,
    0.2053020530205302,
    0.2053120531205312,
    0.2053220532205322,
    0.2053320533205332,
    0.20534205342053422,
    0.2053520535205352,
    0.2053620536205362,
    0.2053720537205372,
    0.2053820538205382,
    0.2053920539205392,
    0.2054020540205402,
    0.2054120541205412,
    0.2054220542205422,
    0.2054320543205432,
    0.2054420544205442,
    0.2054520545205452,
    0.2054620546205462,
    0.2054720547205472,
    0.2054820548205482,
    0.2054920549205492,
    0.2055020550205502,
    0.2055120551205512,
    0.2055220552205522,
    0.20553205532055321,
    0.2055420554205542,
    0.2055520555205552,
    0.2055620556205562,
    0.2055720557205572,
    0.2055820558205582,
    0.2055920559205592,
    0.2056020560205602,
    0.20561205612056122,
    0.2056220562205622,
    0.2056320563205632,
    0.2056420564205642,
    0.2056520565205652,
    0.2056620566205662,
    0.2056720567205672,
    0.2056820568205682,
    0.2056920569205692,
    0.2057020570205702,
    0.2057120571205712,
    0.2057220572205722,
    0.2057320573205732,
    0.2057420574205742,
    0.2057520575205752,
    0.2057620576205762,
    0.2057720577205772,
    0.2057820578205782,
    0.2057920579205792,
    0.20580205802058021,
    0.2058120581205812,
    0.2058220582205822,
    0.2058320583205832,
    0.2058420584205842,
    0.2058520585205852,
    0.2058620586205862,
    0.2058720587205872,
    0.20588205882058822,
    0.2058920589205892,
    0.2059020590205902,
    0.2059120591205912,
    0.2059220592205922,
    0.2059320593205932,
    0.2059420594205942,
    0.2059520595205952,
    0.20596205962059622,
    0.2059720597205972,
    0.2059820598205982,
    0.2059920599205992,
    0.2060020600206002,
    0.2060120601206012,
    0.2060220602206022,
    0.2060320603206032,
    0.2060420604206042,
    0.2060520605206052,
    0.2060620606206062,
    0.20607206072060721,
    0.2060820608206082,
    0.2060920609206092,
    0.2061020610206102,
    0.2061120611206112,
    0.2061220612206122,
    0.2061320613206132,
    0.2061420614206142,
    0.20615206152061522,
    0.2061620616206162,
    0.2061720617206172,
    0.2061820618206182,
    0.2061920619206192,
    0.2062020620206202,
    0.2062120621206212,
    0.2062220622206222,
    0.20623206232062322,
    0.2062420624206242,
    0.2062520625206252,
    0.2062620626206262,
    0.2062720627206272,
    0.2062820628206282,
    0.2062920629206292,
    0.2063020630206302,
    0.2063120631206312,
    0.2063220632206322,
    0.2063320633206332,
    0.20634206342063421,
    0.2063520635206352,
    0.2063620636206362,
    0.2063720637206372,
    0.2063820638206382,
    0.2063920639206392,
    0.2064020640206402,
    0.2064120641206412,
    0.20642206422064222,
    0.2064320643206432,
    0.2064420644206442,
    0.2064520645206452,
    0.2064620646206462,
    0.2064720647206472,
    0.2064820648206482,
    0.2064920649206492,
    0.20650206502065022,
    0.2065120651206512,
    0.2065220652206522,
    0.2065320653206532,
    0.2065420654206542,
    0.2065520655206552,
    0.2065620656206562,
    0.2065720657206572,
    0.2065820658206582,
    0.2065920659206592,
    0.2066020660206602,
    0.20661206612066121,
    0.2066220662206622,
    0.2066320663206632,
    0.2066420664206642,
    0.2066520665206652,
    0.2066620666206662,
    0.2066720667206672,
    0.2066820668206682,
    0.20669206692066922,
    0.2067020670206702,
    0.2067120671206712,
    0.2067220672206722,
    0.2067320673206732,
    0.2067420674206742,
    0.2067520675206752,
    0.2067620676206762,
    0.20677206772067722,
    0.2067820678206782,
    0.2067920679206792,
    0.2068020680206802,
    0.2068120681206812,
    0.2068220682206822,
    0.2068320683206832,
    0.2068420684206842,
    0.2068520685206852,
    0.2068620686206862,
    0.2068720687206872,
    0.20688206882068821,
    0.2068920689206892,
    0.2069020690206902,
    0.2069120691206912,
    0.2069220692206922,
    0.2069320693206932,
    0.2069420694206942,
    0.2069520695206952,
    0.20696206962069622,
    0.2069720697206972,
    0.2069820698206982,
    0.2069920699206992,
    0.2070020700207002,
    0.2070120701207012,
    0.2070220702207022,
    0.2070320703207032,
    0.20704207042070422,
    0.2070520705207052,
    0.2070620706207062,
    0.2070720707207072,
    0.2070820708207082,
    0.2070920709207092,
    0.2071020710207102,
    0.2071120711207112,
    0.20712207122071222,
    0.2071320713207132,
    0.2071420714207142,
    0.20715207152071521,
    0.2071620716207162,
    0.2071720717207172,
    0.2071820718207182,
    0.2071920719207192,
    0.2072020720207202,
    0.2072120721207212,
    0.2072220722207222,
    0.20723207232072322,
    0.2072420724207242,
    0.2072520725207252,
    0.2072620726207262,
    0.2072720727207272,
    0.2072820728207282,
    0.2072920729207292,
    0.2073020730207302,
    0.20731207312073122,
    0.2073220732207322,
    0.2073320733207332,
    0.2073420734207342,
    0.2073520735207352,
    0.2073620736207362,
    0.2073720737207372,
    0.2073820738207382,
    0.20739207392073922,
    0.2074020740207402,
    0.2074120741207412,
    0.20742207422074221,
    0.2074320743207432,
    0.2074420744207442,
    0.2074520745207452,
    0.2074620746207462,
    0.2074720747207472,
    0.2074820748207482,
    0.2074920749207492,
    0.20750207502075022,
    0.2075120751207512,
    0.2075220752207522,
    0.2075320753207532,
    0.2075420754207542,
    0.2075520755207552,
    0.2075620756207562,
    0.2075720757207572,
    0.20758207582075822,
    0.2075920759207592,
    0.2076020760207602,
    0.2076120761207612,
    0.2076220762207622,
    0.2076320763207632,
    0.2076420764207642,
    0.2076520765207652,
    0.20766207662076622,
    0.2076720767207672,
    0.2076820768207682,
    0.20769207692076921,
    0.2077020770207702,
    0.2077120771207712,
    0.2077220772207722,
    0.2077320773207732,
    0.2077420774207742,
    0.2077520775207752,
    0.2077620776207762,
    0.20777207772077722,
    0.2077820778207782,
    0.2077920779207792,
    0.2078020780207802,
    0.2078120781207812,
    0.2078220782207822,
    0.2078320783207832,
    0.2078420784207842,
    0.20785207852078522,
    0.2078620786207862,
    0.2078720787207872,
    0.2078820788207882,
    0.2078920789207892,
    0.2079020790207902,
    0.2079120791207912,
    0.2079220792207922,
    0.20793207932079322,
    0.2079420794207942,
    0.2079520795207952,
    0.20796207962079621,
    0.2079720797207972,
    0.2079820798207982,
    0.2079920799207992,
    0.2080020800208002,
    0.20801208012080122,
    0.2080220802208022,
    0.2080320803208032,
    0.20804208042080422,
    0.2080520805208052,
    0.2080620806208062,
    0.2080720807208072,
    0.2080820808208082,
    0.2080920809208092,
    0.2081020810208102,
    0.2081120811208112,
    0.20812208122081222,
    0.2081320813208132,
    0.2081420814208142,
    0.2081520815208152,
    0.2081620816208162,
    0.2081720817208172,
    0.2081820818208182,
    0.2081920819208192,
    0.20820208202082022,
    0.2082120821208212,
    0.2082220822208222,
    0.20823208232082321,
    0.2082420824208242,
    0.2082520825208252,
    0.2082620826208262,
    0.2082720827208272,
    0.20828208282082822,
    0.2082920829208292,
    0.2083020830208302,
    0.20831208312083122,
    0.2083220832208322,
    0.2083320833208332,
    0.2083420834208342,
    0.2083520835208352,
    0.2083620836208362,
    0.2083720837208372,
    0.2083820838208382,
    0.20839208392083922,
    0.2084020840208402,
    0.2084120841208412,
    0.2084220842208422,
    0.2084320843208432,
    0.2084420844208442,
    0.2084520845208452,
    0.2084620846208462,
    0.20847208472084722,
    0.2084820848208482,
    0.2084920849208492,
    0.20850208502085021,
    0.2085120851208512,
    0.2085220852208522,
    0.2085320853208532,
    0.2085420854208542,
    0.20855208552085522,
    0.2085620856208562,
    0.2085720857208572,
    0.20858208582085822,
    0.2085920859208592,
    0.2086020860208602,
    0.2086120861208612,
    0.2086220862208622,
    0.2086320863208632,
    0.2086420864208642,
    0.2086520865208652,
    0.20866208662086622,
    0.2086720867208672,
    0.2086820868208682,
    0.2086920869208692,
    0.2087020870208702,
    0.2087120871208712,
    0.2087220872208722,
    0.2087320873208732,
    0.20874208742087422,
    0.2087520875208752,
    0.2087620876208762,
    0.20877208772087721,
    0.2087820878208782,
    0.2087920879208792,
    0.2088020880208802,
    0.2088120881208812,
    0.20882208822088222,
    0.2088320883208832,
    0.2088420884208842,
    0.20885208852088522,
    0.2088620886208862,
    0.2088720887208872,
    0.2088820888208882,
    0.2088920889208892,
    0.2089020890208902,
    0.2089120891208912,
    0.2089220892208922,
    0.20893208932089322,
    0.2089420894208942,
    0.2089520895208952,
    0.2089620896208962,
    0.2089720897208972,
    0.2089820898208982,
    0.2089920899208992,
    0.2090020900209002,
    0.20901209012090122,
    0.2090220902209022,
    0.2090320903209032,
    0.20904209042090421,
    0.2090520905209052,
    0.2090620906209062,
    0.2090720907209072,
    0.2090820908209082,
    0.20909209092090922,
    0.2091020910209102,
    0.2091120911209112,
    0.20912209122091222,
    0.2091320913209132,
    0.2091420914209142,
    0.2091520915209152,
    0.2091620916209162,
    0.20917209172091722,
    0.2091820918209182,
    0.2091920919209192,
    0.20920209202092022,
    0.2092120921209212,
    0.2092220922209222,
    0.2092320923209232,
    0.2092420924209242,
    0.2092520925209252,
    0.2092620926209262,
    0.2092720927209272,
    0.20928209282092822,
    0.2092920929209292,
    0.2093020930209302,
    0.20931209312093121,
    0.2093220932209322,
    0.2093320933209332,
    0.2093420934209342,
    0.2093520935209352,
    0.20936209362093622,
    0.2093720937209372,
    0.2093820938209382,
    0.20939209392093922,
    0.2094020940209402,
    0.2094120941209412,
    0.2094220942209422,
    0.2094320943209432,
    0.20944209442094422,
    0.2094520945209452,
    0.2094620946209462,
    0.20947209472094722,
    0.2094820948209482,
    0.2094920949209492,
    0.2095020950209502,
    0.2095120951209512,
    0.2095220952209522,
    0.2095320953209532,
    0.2095420954209542,
    0.20955209552095522,
    0.2095620956209562,
    0.2095720957209572,
    0.20958209582095821,
    0.2095920959209592,
    0.2096020960209602,
    0.2096120961209612,
    0.2096220962209622,
    0.20963209632096322,
    0.2096420964209642,
    0.2096520965209652,
    0.20966209662096622,
    0.2096720967209672,
    0.2096820968209682,
    0.2096920969209692,
    0.2097020970209702,
    0.20971209712097122,
    0.2097220972209722,
    0.2097320973209732,
    0.20974209742097422,
    0.2097520975209752,
    0.2097620976209762,
    0.2097720977209772,
    0.2097820978209782,
    0.2097920979209792,
    0.2098020980209802,
    0.2098120981209812,
    0.20982209822098222,
    0.2098320983209832,
    0.2098420984209842,
    0.20985209852098521,
    0.2098620986209862,
    0.2098720987209872,
    0.2098820988209882,
    0.2098920989209892,
    0.20990209902099022,
    0.2099120991209912,
    0.2099220992209922,
    0.20993209932099322,
    0.2099420994209942,
    0.2099520995209952,
    0.2099620996209962,
    0.2099720997209972,
    0.20998209982099822,
    0.2099920999209992,
    0.2100021000210002,
    0.21001210012100122,
    0.2100221002210022,
    0.2100321003210032,
    0.2100421004210042,
    0.2100521005210052,
    0.21006210062100622,
    0.2100721007210072,
    0.2100821008210082,
    0.21009210092100922,
    0.2101021010210102,
    0.2101121011210112,
    0.21012210122101221,
    0.2101321013210132,
    0.2101421014210142,
    0.2101521015210152,
    0.2101621016210162,
    0.21017210172101722,
    0.2101821018210182,
    0.2101921019210192,
    0.21020210202102022,
    0.2102121021210212,
    0.2102221022210222,
    0.2102321023210232,
    0.2102421024210242,
    0.21025210252102522,
    0.2102621026210262,
    0.2102721027210272,
    0.21028210282102822,
    0.2102921029210292,
    0.2103021030210302,
    0.2103121031210312,
    0.2103221032210322,
    0.21033210332103322,
    0.2103421034210342,
    0.2103521035210352,
    0.21036210362103622,
    0.2103721037210372,
    0.2103821038210382,
    0.21039210392103921,
    0.2104021040210402,
    0.2104121041210412,
    0.2104221042210422,
    0.2104321043210432,
    0.21044210442104422,
    0.2104521045210452,
    0.2104621046210462,
    0.21047210472104722,
    0.2104821048210482,
    0.2104921049210492,
    0.2105021050210502,
    0.2105121051210512,
    0.21052210522105222,
    0.2105321053210532,
    0.2105421054210542,
    0.21055210552105522,
    0.2105621056210562,
    0.2105721057210572,
    0.2105821058210582,
    0.2105921059210592,
    0.21060210602106022,
    0.2106121061210612,
    0.2106221062210622,
    0.21063210632106322,
    0.2106421064210642,
    0.2106521065210652,
    0.21066210662106621,
    0.2106721067210672,
    0.2106821068210682,
    0.2106921069210692,
    0.2107021070210702,
    0.21071210712107122,
    0.2107221072210722,
    0.2107321073210732,
    0.21074210742107422,
    0.2107521075210752,
    0.2107621076210762,
    0.2107721077210772,
    0.2107821078210782,
    0.21079210792107922,
    0.2108021080210802,
    0.2108121081210812,
    0.21082210822108222,
    0.2108321083210832,
    0.2108421084210842,
    0.2108521085210852,
    0.2108621086210862,
    0.21087210872108722,
    0.2108821088210882,
    0.2108921089210892,
    0.21090210902109022,
    0.2109121091210912,
    0.2109221092210922,
    0.21093210932109321,
    0.2109421094210942,
    0.2109521095210952,
    0.2109621096210962,
    0.2109721097210972,
    0.21098210982109822,
    0.2109921099210992,
    0.2110021100211002,
    0.21101211012110122,
    0.2110221102211022,
    0.2110321103211032,
    0.2110421104211042,
    0.2110521105211052,
    0.21106211062110622,
    0.2110721107211072,
    0.2110821108211082,
    0.21109211092110922,
    0.2111021110211102,
    0.2111121111211112,
    0.2111221112211122,
    0.2111321113211132,
    0.21114211142111422,
    0.2111521115211152,
    0.2111621116211162,
    0.21117211172111722,
    0.2111821118211182,
    0.2111921119211192,
    0.21120211202112021,
    0.2112121121211212,
    0.21122211222112222,
    0.2112321123211232,
    0.2112421124211242,
    0.21125211252112522,
    0.2112621126211262,
    0.2112721127211272,
    0.21128211282112822,
    0.2112921129211292,
    0.2113021130211302,
    0.2113121131211312,
    0.2113221132211322,
    0.21133211332113322,
    0.2113421134211342,
    0.2113521135211352,
    0.21136211362113622,
    0.2113721137211372,
    0.2113821138211382,
    0.2113921139211392,
    0.2114021140211402,
    0.21141211412114122,
    0.2114221142211422,
    0.2114321143211432,
    0.21144211442114422,
    0.2114521145211452,
    0.2114621146211462,
    0.21147211472114721,
    0.2114821148211482,
    0.21149211492114922,
    0.2115021150211502,
    0.2115121151211512,
    0.21152211522115222,
    0.2115321153211532,
    0.2115421154211542,
    0.21155211552115522,
    0.2115621156211562,
    0.2115721157211572,
    0.2115821158211582,
    0.2115921159211592,
    0.21160211602116022,
    0.2116121161211612,
    0.2116221162211622,
    0.21163211632116322,
    0.2116421164211642,
    0.2116521165211652,
    0.2116621166211662,
    0.2116721167211672,
    0.21168211682116822,
    0.2116921169211692,
    0.2117021170211702,
    0.21171211712117122,
    0.2117221172211722,
    0.2117321173211732,
    0.21174211742117421,
    0.2117521175211752,
    0.21176211762117622,
    0.2117721177211772,
    0.2117821178211782,
    0.21179211792117922,
    0.2118021180211802,
    0.2118121181211812,
    0.21182211822118222,
    0.2118321183211832,
    0.2118421184211842,
    0.2118521185211852,
    0.2118621186211862,
    0.21187211872118722,
    0.2118821188211882,
    0.2118921189211892,
    0.21190211902119022,
    0.2119121191211912,
    0.2119221192211922,
    0.2119321193211932,
    0.2119421194211942,
    0.21195211952119522,
    0.2119621196211962,
    0.2119721197211972,
    0.21198211982119822,
    0.2119921199211992,
    0.2120021200212002,
    0.21201212012120121,
    0.2120221202212022,
    0.21203212032120322,
    0.2120421204212042,
    0.2120521205212052,
    0.21206212062120622,
    0.2120721207212072,
    0.2120821208212082,
    0.21209212092120922,
    0.2121021210212102,
    0.21211212112121122,
    0.2121221212212122,
    0.2121321213212132,
    0.21214212142121422,
    0.2121521215212152,
    0.2121621216212162,
    0.21217212172121722,
    0.2121821218212182,
    0.2121921219212192,
    0.2122021220212202,
    0.2122121221212212,
    0.21222212222122222,
    0.2122321223212232,
    0.2122421224212242,
    0.21225212252122522,
    0.2122621226212262,
    0.2122721227212272,
    0.21228212282122821,
    0.2122921229212292,
    0.21230212302123022,
    0.2123121231212312,
    0.2123221232212322,
    0.21233212332123322,
    0.2123421234212342,
    0.2123521235212352,
    0.21236212362123622,
    0.2123721237212372,
    0.21238212382123822,
    0.2123921239212392,
    0.2124021240212402,
    0.21241212412124122,
    0.2124221242212422,
    0.2124321243212432,
    0.21244212442124422,
    0.2124521245212452,
    0.2124621246212462,
    0.2124721247212472,
    0.2124821248212482,
    0.21249212492124922,
    0.2125021250212502,
    0.2125121251212512,
    0.21252212522125222,
    0.2125321253212532,
    0.2125421254212542,
    0.21255212552125521,
    0.2125621256212562,
    0.21257212572125722,
    0.2125821258212582,
    0.2125921259212592,
    0.21260212602126022,
    0.2126121261212612,
    0.2126221262212622,
    0.21263212632126322,
    0.2126421264212642,
    0.21265212652126522,
    0.2126621266212662,
    0.2126721267212672,
    0.21268212682126822,
    0.2126921269212692,
    0.2127021270212702,
    0.21271212712127122,
    0.2127221272212722,
    0.2127321273212732,
    0.2127421274212742,
    0.2127521275212752,
    0.21276212762127622,
    0.2127721277212772,
    0.2127821278212782,
    0.21279212792127922,
    0.2128021280212802,
    0.2128121281212812,
    0.21282212822128221,
    0.2128321283212832,
    0.21284212842128422,
    0.2128521285212852,
    0.2128621286212862,
    0.21287212872128722,
    0.2128821288212882,
    0.2128921289212892,
    0.21290212902129022,
    0.2129121291212912,
    0.21292212922129222,
    0.2129321293212932,
    0.2129421294212942,
    0.21295212952129522,
    0.2129621296212962,
    0.2129721297212972,
    0.21298212982129822,
    0.2129921299212992,
    0.2130021300213002,
    0.2130121301213012,
    0.2130221302213022,
    0.21303213032130322,
    0.2130421304213042,
    0.2130521305213052,
    0.21306213062130622,
    0.2130721307213072,
    0.2130821308213082,
    0.21309213092130921,
    0.2131021310213102,
    0.21311213112131122,
    0.2131221312213122,
    0.2131321313213132,
    0.21314213142131422,
    0.2131521315213152,
    0.2131621316213162,
    0.21317213172131722,
    0.2131821318213182,
    0.21319213192131922,
    0.2132021320213202,
    0.2132121321213212,
    0.21322213222132222,
    0.2132321323213232,
    0.2132421324213242,
    0.21325213252132522,
    0.2132621326213262,
    0.21327213272132722,
    0.2132821328213282,
    0.2132921329213292,
    0.21330213302133022,
    0.2133121331213312,
    0.2133221332213322,
    0.21333213332133322,
    0.2133421334213342,
    0.2133521335213352,
    0.21336213362133621,
    0.2133721337213372,
    0.21338213382133822,
    0.2133921339213392,
    0.2134021340213402,
    0.21341213412134122,
    0.2134221342213422,
    0.2134321343213432,
    0.21344213442134422,
    0.2134521345213452,
    0.21346213462134622,
    0.2134721347213472,
    0.2134821348213482,
    0.21349213492134922,
    0.2135021350213502,
    0.2135121351213512,
    0.21352213522135222,
    0.2135321353213532,
    0.21354213542135422,
    0.2135521355213552,
    0.2135621356213562,
    0.21357213572135722,
    0.2135821358213582,
    0.2135921359213592,
    0.21360213602136022,
    0.2136121361213612,
    0.2136221362213622,
    0.21363213632136321,
    0.2136421364213642,
    0.21365213652136522,
    0.2136621366213662,
    0.2136721367213672,
    0.21368213682136822,
    0.2136921369213692,
    0.2137021370213702,
    0.21371213712137122,
    0.2137221372213722,
    0.21373213732137322,
    0.2137421374213742,
    0.2137521375213752,
    0.21376213762137622,
    0.2137721377213772,
    0.2137821378213782,
    0.21379213792137922,
    0.2138021380213802,
    0.21381213812138122,
    0.2138221382213822,
    0.2138321383213832,
    0.21384213842138422,
    0.2138521385213852,
    0.2138621386213862,
    0.21387213872138722,
    0.2138821388213882,
    0.2138921389213892,
    0.21390213902139021,
    0.2139121391213912,
    0.21392213922139222,
    0.2139321393213932,
    0.2139421394213942,
    0.21395213952139522,
    0.2139621396213962,
    0.2139721397213972,
    0.21398213982139822,
    0.2139921399213992,
    0.21400214002140022,
    0.2140121401214012,
    0.2140221402214022,
    0.21403214032140322,
    0.2140421404214042,
    0.2140521405214052,
    0.21406214062140622,
    0.2140721407214072,
    0.21408214082140822,
    0.2140921409214092,
    0.2141021410214102,
    0.21411214112141122,
    0.2141221412214122,
    0.2141321413214132,
    0.21414214142141422,
    0.2141521415214152,
    0.21416214162141622,
    0.21417214172141721,
    0.2141821418214182,
    0.21419214192141922,
    0.2142021420214202,
    0.2142121421214212,
    0.21422214222142222,
    0.2142321423214232,
    0.2142421424214242,
    0.21425214252142522,
    0.2142621426214262,
    0.21427214272142722,
    0.2142821428214282,
    0.2142921429214292,
    0.21430214302143022,
    0.2143121431214312,
    0.2143221432214322,
    0.21433214332143322,
    0.2143421434214342,
    0.21435214352143522,
    0.2143621436214362,
    0.2143721437214372,
    0.21438214382143822,
    0.2143921439214392,
    0.2144021440214402,
    0.21441214412144122,
    0.2144221442214422,
    0.21443214432144322,
    0.21444214442144421,
    0.2144521445214452,
    0.21446214462144622,
    0.2144721447214472,
    0.2144821448214482,
    0.21449214492144922,
    0.2145021450214502,
    0.2145121451214512,
    0.21452214522145222,
    0.2145321453214532,
    0.21454214542145422,
    0.2145521455214552,
    0.2145621456214562,
    0.21457214572145722,
    0.2145821458214582,
    0.2145921459214592,
    0.21460214602146022,
    0.2146121461214612,
    0.21462214622146222,
    0.2146321463214632,
    0.2146421464214642,
    0.21465214652146522,
    0.2146621466214662,
    0.2146721467214672,
    0.21468214682146822,
    0.2146921469214692,
    0.21470214702147022,
    0.21471214712147121,
    0.2147221472214722,
    0.21473214732147322,
    0.2147421474214742,
    0.2147521475214752,
    0.21476214762147622,
    0.2147721477214772,
    0.2147821478214782,
    0.21479214792147922,
    0.2148021480214802,
    0.21481214812148122,
    0.2148221482214822,
    0.2148321483214832,
    0.21484214842148422,
    0.2148521485214852,
    0.2148621486214862,
    0.21487214872148722,
    0.2148821488214882,
    0.21489214892148922,
    0.2149021490214902,
    0.2149121491214912,
    0.21492214922149222,
    0.2149321493214932,
    0.2149421494214942,
    0.21495214952149522,
    0.2149621496214962,
    0.21497214972149722,
    0.21498214982149821,
    0.2149921499214992,
    0.21500215002150022,
    0.2150121501215012,
    0.2150221502215022,
    0.21503215032150322,
    0.2150421504215042,
    0.2150521505215052,
    0.21506215062150622,
    0.2150721507215072,
    0.21508215082150822,
    0.2150921509215092,
    0.2151021510215102,
    0.21511215112151122,
    0.2151221512215122,
    0.2151321513215132,
    0.21514215142151422,
    0.2151521515215152,
    0.21516215162151622,
    0.2151721517215172,
    0.2151821518215182,
    0.21519215192151922,
    0.2152021520215202,
    0.2152121521215212,
    0.21522215222152222,
    0.2152321523215232,
    0.21524215242152422,
    0.21525215252152521,
    0.2152621526215262,
    0.21527215272152722,
    0.2152821528215282,
    0.2152921529215292,
    0.21530215302153022,
    0.2153121531215312,
    0.21532215322153223,
    0.21533215332153322,
    0.2153421534215342,
    0.21535215352153522,
    0.2153621536215362,
    0.2153721537215372,
    0.21538215382153822,
    0.2153921539215392,
    0.2154021540215402,
    0.21541215412154122,
    0.2154221542215422,
    0.21543215432154322,
    0.2154421544215442,
    0.2154521545215452,
    0.21546215462154622,
    0.2154721547215472,
    0.2154821548215482,
    0.21549215492154922,
    0.2155021550215502,
    0.21551215512155122,
    0.21552215522155221,
    0.2155321553215532,
    0.21554215542155422,
    0.2155521555215552,
    0.2155621556215562,
    0.21557215572155722,
    0.2155821558215582,
    0.21559215592155923,
    0.21560215602156022,
    0.2156121561215612,
    0.21562215622156222,
    0.2156321563215632,
    0.2156421564215642,
    0.21565215652156522,
    0.2156621566215662,
    0.2156721567215672,
    0.21568215682156822,
    0.2156921569215692,
    0.21570215702157022,
    0.2157121571215712,
    0.2157221572215722,
    0.21573215732157322,
    0.2157421574215742,
    0.2157521575215752,
    0.21576215762157622,
    0.2157721577215772,
    0.21578215782157822,
    0.21579215792157921,
    0.2158021580215802,
    0.21581215812158122,
    0.2158221582215822,
    0.2158321583215832,
    0.21584215842158422,
    0.2158521585215852,
    0.21586215862158623,
    0.21587215872158722,
    0.2158821588215882,
    0.21589215892158922,
    0.2159021590215902,
    0.2159121591215912,
    0.21592215922159222,
    0.2159321593215932,
    0.2159421594215942,
    0.21595215952159522,
    0.2159621596215962,
    0.21597215972159722,
    0.2159821598215982,
    0.2159921599215992,
    0.21600216002160022,
    0.2160121601216012,
    0.2160221602216022,
    0.21603216032160322,
    0.2160421604216042,
    0.21605216052160522,
    0.21606216062160621,
    0.2160721607216072,
    0.21608216082160822,
    0.2160921609216092,
    0.2161021610216102,
    0.21611216112161122,
    0.2161221612216122,
    0.21613216132161323,
    0.21614216142161422,
    0.2161521615216152,
    0.21616216162161622,
    0.2161721617216172,
    0.2161821618216182,
    0.21619216192161922,
    0.2162021620216202,
    0.21621216212162123,
    0.21622216222162222,
    0.2162321623216232,
    0.21624216242162422,
    0.2162521625216252,
    0.2162621626216262,
    0.21627216272162722,
    0.2162821628216282,
    0.2162921629216292,
    0.21630216302163022,
    0.2163121631216312,
    0.21632216322163222,
    0.21633216332163321,
    0.2163421634216342,
    0.21635216352163522,
    0.2163621636216362,
    0.2163721637216372,
    0.21638216382163822,
    0.2163921639216392,
    0.21640216402164023,
    0.21641216412164122,
    0.2164221642216422,
    0.21643216432164322,
    0.2164421644216442,
    0.2164521645216452,
    0.21646216462164622,
    0.2164721647216472,
    0.21648216482164823,
    0.21649216492164922,
    0.2165021650216502,
    0.21651216512165122,
    0.2165221652216522,
    0.2165321653216532,
    0.21654216542165422,
    0.2165521655216552,
    0.2165621656216562,
    0.21657216572165722,
    0.2165821658216582,
    0.21659216592165922,
    0.21660216602166021,
    0.2166121661216612,
    0.21662216622166222,
    0.2166321663216632,
    0.2166421664216642,
    0.21665216652166522,
    0.2166621666216662,
    0.21667216672166723,
    0.21668216682166822,
    0.2166921669216692,
    0.21670216702167022,
    0.2167121671216712,
    0.2167221672216722,
    0.21673216732167322,
    0.2167421674216742,
    0.21675216752167523,
    0.21676216762167622,
    0.2167721677216772,
    0.21678216782167822,
    0.2167921679216792,
    0.2168021680216802,
    0.21681216812168122,
    0.2168221682216822,
    0.2168321683216832,
    0.21684216842168422,
    0.2168521685216852,
    0.21686216862168622,
    0.21687216872168721,
    0.2168821688216882,
    0.21689216892168922,
    0.2169021690216902,
    0.2169121691216912,
    0.21692216922169222,
    0.2169321693216932,
    0.21694216942169423,
    0.21695216952169522,
    0.2169621696216962,
    0.21697216972169722,
    0.2169821698216982,
    0.2169921699216992,
    0.21700217002170022,
    0.2170121701217012,
    0.21702217022170223,
    0.21703217032170322,
    0.2170421704217042,
    0.21705217052170522,
    0.2170621706217062,
    0.2170721707217072,
    0.21708217082170822,
    0.2170921709217092,
    0.2171021710217102,
    0.21711217112171122,
    0.2171221712217122,
    0.21713217132171322,
    0.21714217142171421,
    0.2171521715217152,
    0.21716217162171622,
    0.2171721717217172,
    0.2171821718217182,
    0.21719217192171922,
    0.2172021720217202,
    0.21721217212172123,
    0.21722217222172222,
    0.2172321723217232,
    0.21724217242172422,
    0.2172521725217252,
    0.2172621726217262,
    0.21727217272172722,
    0.2172821728217282,
    0.21729217292172923,
    0.21730217302173022,
    0.2173121731217312,
    0.21732217322173222,
    0.2173321733217332,
    0.2173421734217342,
    0.21735217352173522,
    0.2173621736217362,
    0.21737217372173723,
    0.21738217382173822,
    0.2173921739217392,
    0.21740217402174022,
    0.21741217412174121,
    0.2174221742217422,
    0.21743217432174322,
    0.2174421744217442,
    0.2174521745217452,
    0.21746217462174622,
    0.2174721747217472,
    0.21748217482174823,
    0.21749217492174922,
    0.2175021750217502,
    0.21751217512175122,
    0.2175221752217522,
    0.2175321753217532,
    0.21754217542175422,
    0.2175521755217552,
    0.21756217562175623,
    0.21757217572175722,
    0.2175821758217582,
    0.21759217592175922,
    0.2176021760217602,
    0.2176121761217612,
    0.21762217622176222,
    0.2176321763217632,
    0.21764217642176423,
    0.21765217652176522,
    0.2176621766217662,
    0.21767217672176722,
    0.21768217682176821,
    0.2176921769217692,
    0.21770217702177022,
    0.2177121771217712,
    0.2177221772217722,
    0.21773217732177322,
    0.2177421774217742,
    0.21775217752177523,
    0.21776217762177622,
    0.2177721777217772,
    0.21778217782177822,
    0.2177921779217792,
    0.2178021780217802,
    0.21781217812178122,
    0.2178221782217822,
    0.21783217832178323,
    0.21784217842178422,
    0.2178521785217852,
    0.21786217862178622,
    0.2178721787217872,
    0.2178821788217882,
    0.21789217892178922,
    0.2179021790217902,
    0.21791217912179123,
    0.21792217922179222,
    0.2179321793217932,
    0.21794217942179422,
    0.21795217952179521,
    0.2179621796217962,
    0.21797217972179722,
    0.2179821798217982,
    0.2179921799217992,
    0.21800218002180022,
    0.2180121801218012,
    0.21802218022180223,
    0.21803218032180322,
    0.2180421804218042,
    0.21805218052180522,
    0.2180621806218062,
    0.2180721807218072,
    0.21808218082180822,
    0.2180921809218092,
    0.21810218102181023,
    0.21811218112181122,
    0.2181221812218122,
    0.21813218132181322,
    0.2181421814218142,
    0.2181521815218152,
    0.21816218162181622,
    0.2181721817218172,
    0.21818218182181823,
    0.21819218192181922,
    0.2182021820218202,
    0.21821218212182122,
    0.21822218222182221,
    0.2182321823218232,
    0.21824218242182422,
    0.2182521825218252,
    0.2182621826218262,
    0.21827218272182722,
    0.2182821828218282,
    0.21829218292182923,
    0.21830218302183022,
    0.2183121831218312,
    0.21832218322183222,
    0.2183321833218332,
    0.2183421834218342,
    0.21835218352183522,
    0.2183621836218362,
    0.21837218372183723,
    0.21838218382183822,
    0.2183921839218392,
    0.21840218402184022,
    0.2184121841218412,
    0.2184221842218422,
    0.21843218432184322,
    0.2184421844218442,
    0.21845218452184523,
    0.21846218462184622,
    0.2184721847218472,
    0.21848218482184822,
    0.21849218492184921,
    0.2185021850218502,
    0.21851218512185122,
    0.2185221852218522,
    0.21853218532185323,
    0.21854218542185422,
    0.2185521855218552,
    0.21856218562185623,
    0.21857218572185722,
    0.2185821858218582,
    0.21859218592185922,
    0.2186021860218602,
    0.2186121861218612,
    0.21862218622186222,
    0.2186321863218632,
    0.21864218642186423,
    0.21865218652186522,
    0.2186621866218662,
    0.21867218672186722,
    0.2186821868218682,
    0.2186921869218692,
    0.21870218702187022,
    0.2187121871218712,
    0.21872218722187223,
    0.21873218732187322,
    0.2187421874218742,
    0.21875218752187522,
    0.21876218762187621,
    0.2187721877218772,
    0.21878218782187822,
    0.2187921879218792,
    0.21880218802188023,
    0.21881218812188122,
    0.2188221882218822,
    0.21883218832188323,
    0.21884218842188422,
    0.2188521885218852,
    0.21886218862188622,
    0.2188721887218872,
    0.2188821888218882,
    0.21889218892188922,
    0.2189021890218902,
    0.21891218912189123,
    0.21892218922189222,
    0.2189321893218932,
    0.21894218942189422,
    0.2189521895218952,
    0.2189621896218962,
    0.21897218972189722,
    0.2189821898218982,
    0.21899218992189923,
    0.21900219002190022,
    0.2190121901219012,
    0.21902219022190222,
    0.21903219032190321,
    0.2190421904219042,
    0.21905219052190522,
    0.2190621906219062,
    0.21907219072190723,
    0.21908219082190822,
    0.2190921909219092,
    0.21910219102191023,
    0.21911219112191122,
    0.2191221912219122,
    0.21913219132191322,
    0.2191421914219142,
    0.2191521915219152,
    0.21916219162191622,
    0.2191721917219172,
    0.21918219182191823,
    0.21919219192191922,
    0.2192021920219202,
    0.21921219212192122,
    0.2192221922219222,
    0.2192321923219232,
    0.21924219242192422,
    0.2192521925219252,
    0.21926219262192623,
    0.21927219272192722,
    0.2192821928219282,
    0.21929219292192922,
    0.21930219302193021,
    0.2193121931219312,
    0.21932219322193222,
    0.2193321933219332,
    0.21934219342193423,
    0.21935219352193522,
    0.2193621936219362,
    0.21937219372193723,
    0.21938219382193822,
    0.2193921939219392,
    0.21940219402194022,
    0.2194121941219412,
    0.21942219422194223,
    0.21943219432194322,
    0.2194421944219442,
    0.21945219452194523,
    0.21946219462194622,
    0.2194721947219472,
    0.21948219482194822,
    0.2194921949219492,
    0.2195021950219502,
    0.21951219512195122,
    0.2195221952219522,
    0.21953219532195323,
    0.21954219542195422,
    0.2195521955219552,
    0.21956219562195622,
    0.21957219572195721,
    0.2195821958219582,
    0.21959219592195922,
    0.2196021960219602,
    0.21961219612196123,
    0.21962219622196222,
    0.2196321963219632,
    0.21964219642196423,
    0.21965219652196522,
    0.2196621966219662,
    0.21967219672196722,
    0.2196821968219682,
    0.21969219692196923,
    0.21970219702197022,
    0.2197121971219712,
    0.21972219722197223,
    0.21973219732197322,
    0.2197421974219742,
    0.21975219752197522,
    0.2197621976219762,
    0.2197721977219772,
    0.21978219782197822,
    0.2197921979219792,
    0.21980219802198023,
    0.21981219812198122,
    0.2198221982219822,
    0.21983219832198322,
    0.21984219842198421,
    0.2198521985219852,
    0.21986219862198622,
    0.2198721987219872,
    0.21988219882198823,
    0.21989219892198922,
    0.2199021990219902,
    0.21991219912199123,
    0.21992219922199222,
    0.2199321993219932,
    0.21994219942199422,
    0.2199521995219952,
    0.21996219962199623,
    0.21997219972199722,
    0.2199821998219982,
    0.21999219992199923,
    0.22000220002200022,
    0.2200122001220012,
    0.22002220022200222,
    0.2200322003220032,
    0.2200422004220042,
    0.22005220052200522,
    0.2200622006220062,
    0.22007220072200723,
    0.22008220082200822,
    0.2200922009220092,
    0.22010220102201022,
    0.22011220112201121,
    0.2201222012220122,
    0.22013220132201322,
    0.2201422014220142,
    0.22015220152201523,
    0.22016220162201622,
    0.2201722017220172,
    0.22018220182201823,
    0.22019220192201922,
    0.2202022020220202,
    0.22021220212202122,
    0.2202222022220222,
    0.22023220232202323,
    0.22024220242202422,
    0.2202522025220252,
    0.22026220262202623,
    0.22027220272202722,
    0.2202822028220282,
    0.22029220292202922,
    0.2203022030220302,
    0.2203122031220312,
    0.22032220322203222,
    0.2203322033220332,
    0.22034220342203423,
    0.22035220352203522,
    0.2203622036220362,
    0.22037220372203722,
    0.22038220382203821,
    0.2203922039220392,
    0.22040220402204022,
    0.2204122041220412,
    0.22042220422204223,
    0.22043220432204322,
    0.2204422044220442,
    0.22045220452204523,
    0.22046220462204622,
    0.2204722047220472,
    0.22048220482204822,
    0.2204922049220492,
    0.22050220502205023,
    0.22051220512205122,
    0.2205222052220522,
    0.22053220532205323,
    0.22054220542205422,
    0.2205522055220552,
    0.22056220562205622,
    0.2205722057220572,
    0.22058220582205823,
    0.22059220592205922,
    0.2206022060220602,
    0.22061220612206123,
    0.22062220622206222,
    0.2206322063220632,
    0.22064220642206422,
    0.22065220652206521,
    0.2206622066220662,
    0.22067220672206722,
    0.2206822068220682,
    0.22069220692206923,
    0.22070220702207022,
    0.2207122071220712,
    0.22072220722207223,
    0.22073220732207322,
    0.2207422074220742,
    0.22075220752207522,
    0.2207622076220762,
    0.22077220772207723,
    0.22078220782207822,
    0.2207922079220792,
    0.22080220802208023,
    0.22081220812208122,
    0.2208222082220822,
    0.22083220832208322,
    0.2208422084220842,
    0.22085220852208523,
    0.22086220862208622,
    0.2208722087220872,
    0.22088220882208823,
    0.22089220892208922,
    0.2209022090220902,
    0.22091220912209122,
    0.22092220922209221,
    0.2209322093220932,
    0.22094220942209422,
    0.2209522095220952,
    0.22096220962209623,
    0.22097220972209722,
    0.2209822098220982,
    0.22099220992209923,
    0.22100221002210022,
    0.2210122101221012,
    0.22102221022210222,
    0.2210322103221032,
    0.22104221042210423,
    0.22105221052210522,
    0.2210622106221062,
    0.22107221072210723,
    0.22108221082210822,
    0.2210922109221092,
    0.22110221102211022,
    0.2211122111221112,
    0.22112221122211223,
    0.22113221132211322,
    0.2211422114221142,
    0.22115221152211523,
    0.22116221162211622,
    0.2211722117221172,
    0.22118221182211822,
    0.22119221192211921,
    0.2212022120221202,
    0.22121221212212122,
    0.2212222122221222,
    0.22123221232212323,
    0.22124221242212422,
    0.2212522125221252,
    0.22126221262212623,
    0.22127221272212722,
    0.2212822128221282,
    0.22129221292212922,
    0.2213022130221302,
    0.22131221312213123,
    0.22132221322213222,
    0.2213322133221332,
    0.22134221342213423,
    0.22135221352213522,
    0.2213622136221362,
    0.22137221372213722,
    0.2213822138221382,
    0.22139221392213923,
    0.22140221402214022,
    0.2214122141221412,
    0.22142221422214223,
    0.22143221432214322,
    0.2214422144221442,
    0.22145221452214522,
    0.22146221462214621,
    0.22147221472214723,
    0.22148221482214822,
    0.2214922149221492,
    0.22150221502215023,
    0.22151221512215122,
    0.2215222152221522,
    0.22153221532215323,
    0.22154221542215422,
    0.2215522155221552,
    0.22156221562215622,
    0.2215722157221572,
    0.22158221582215823,
    0.22159221592215922,
    0.2216022160221602,
    0.22161221612216123,
    0.22162221622216222,
    0.2216322163221632,
    0.22164221642216422,
    0.2216522165221652,
    0.22166221662216623,
    0.22167221672216722,
    0.2216822168221682,
    0.22169221692216923,
    0.22170221702217022,
    0.2217122171221712,
    0.22172221722217222,
    0.22173221732217321,
    0.22174221742217423,
    0.22175221752217522,
    0.2217622176221762,
    0.22177221772217723,
    0.22178221782217822,
    0.2217922179221792,
    0.22180221802218023,
    0.22181221812218122,
    0.2218222182221822,
    0.22183221832218322,
    0.2218422184221842,
    0.22185221852218523,
    0.22186221862218622,
    0.2218722187221872,
    0.22188221882218823,
    0.22189221892218922,
    0.2219022190221902,
    0.22191221912219122,
    0.2219222192221922,
    0.22193221932219323,
    0.22194221942219422,
    0.2219522195221952,
    0.22196221962219623,
    0.22197221972219722,
    0.2219822198221982,
    0.22199221992219922,
    0.22200222002220021,
    0.22201222012220123,
    0.22202222022220222,
    0.2220322203222032,
    0.22204222042220423,
    0.22205222052220522,
    0.2220622206222062,
    0.22207222072220723,
    0.22208222082220822,
    0.2220922209222092,
    0.22210222102221022,
    0.2221122211222112,
    0.22212222122221223,
    0.22213222132221322,
    0.2221422214222142,
    0.22215222152221523,
    0.22216222162221622,
    0.2221722217222172,
    0.22218222182221822,
    0.2221922219222192,
    0.22220222202222023,
    0.22221222212222122,
    0.2222222222222222,
    0.22223222232222323,
    0.22224222242222422,
    0.2222522225222252,
    0.22226222262222622,
    0.22227222272222721,
    0.22228222282222823,
    0.22229222292222922,
    0.2223022230222302,
    0.22231222312223123,
    0.22232222322223222,
    0.2223322233222332,
    0.22234222342223423,
    0.22235222352223522,
    0.2223622236222362,
    0.22237222372223722,
    0.2223822238222382,
    0.22239222392223923,
    0.22240222402224022,
    0.2224122241222412,
    0.22242222422224223,
    0.22243222432224322,
    0.2224422244222442,
    0.22245222452224522,
    0.2224622246222462,
    0.22247222472224723,
    0.22248222482224822,
    0.2224922249222492,
    0.22250222502225023,
    0.22251222512225122,
    0.2225222252222522,
    0.22253222532225322,
    0.22254222542225421,
    0.22255222552225523,
    0.22256222562225622,
    0.2225722257222572,
    0.22258222582225823,
    0.22259222592225922,
    0.2226022260222602,
    0.22261222612226123,
    0.22262222622226222,
    0.22263222632226323,
    0.22264222642226422,
    0.2226522265222652,
    0.22266222662226623,
    0.22267222672226722,
    0.2226822268222682,
    0.22269222692226923,
    0.22270222702227022,
    0.2227122271222712,
    0.22272222722227222,
    0.2227322273222732,
    0.22274222742227423,
    0.22275222752227522,
    0.2227622276222762,
    0.22277222772227723,
    0.22278222782227822,
    0.2227922279222792,
    0.22280222802228022,
    0.22281222812228121,
    0.22282222822228223,
    0.22283222832228322,
    0.2228422284222842,
    0.22285222852228523,
    0.22286222862228622,
    0.2228722287222872,
    0.22288222882228823,
    0.22289222892228922,
    0.22290222902229023,
    0.22291222912229122,
    0.2229222292222922,
    0.22293222932229323,
    0.22294222942229422,
    0.2229522295222952,
    0.22296222962229623,
    0.22297222972229722,
    0.2229822298222982,
    0.22299222992229922,
    0.2230022300223002,
    0.22301223012230123,
    0.22302223022230222,
    0.2230322303223032,
    0.22304223042230423,
    0.22305223052230522,
    0.2230622306223062,
    0.22307223072230722,
    0.22308223082230821,
    0.22309223092230923,
    0.22310223102231022,
    0.2231122311223112,
    0.22312223122231223,
    0.22313223132231322,
    0.2231422314223142,
    0.22315223152231523,
    0.22316223162231622,
    0.22317223172231723,
    0.22318223182231822,
    0.2231922319223192,
    0.22320223202232023,
    0.22321223212232122,
    0.2232222322223222,
    0.22323223232232323,
    0.22324223242232422,
    0.2232522325223252,
    0.22326223262232622,
    0.2232722327223272,
    0.22328223282232823,
    0.22329223292232922,
    0.2233022330223302,
    0.22331223312233123,
    0.22332223322233222,
    0.2233322333223332,
    0.22334223342233422,
    0.22335223352233521,
    0.22336223362233623,
    0.22337223372233722,
    0.2233822338223382,
    0.22339223392233923,
    0.22340223402234022,
    0.2234122341223412,
    0.22342223422234223,
    0.22343223432234322,
    0.22344223442234423,
    0.22345223452234522,
    0.2234622346223462,
    0.22347223472234723,
    0.22348223482234822,
    0.2234922349223492,
    0.22350223502235023,
    0.22351223512235122,
    0.22352223522235223,
    0.22353223532235322,
    0.2235422354223542,
    0.22355223552235523,
    0.22356223562235622,
    0.2235722357223572,
    0.22358223582235823,
    0.22359223592235922,
    0.2236022360223602,
    0.22361223612236122,
    0.22362223622236221,
    0.22363223632236323,
    0.22364223642236422,
    0.2236522365223652,
    0.22366223662236623,
    0.22367223672236722,
    0.2236822368223682,
    0.22369223692236923,
    0.22370223702237022,
    0.22371223712237123,
    0.22372223722237222,
    0.2237322373223732,
    0.22374223742237423,
    0.22375223752237522,
    0.2237622376223762,
    0.22377223772237723,
    0.22378223782237822,
    0.22379223792237923,
    0.22380223802238022,
    0.2238122381223812,
    0.22382223822238223,
    0.22383223832238322,
    0.2238422384223842,
    0.22385223852238523,
    0.22386223862238622,
    0.2238722387223872,
    0.22388223882238822,
    0.22389223892238921,
    0.22390223902239023,
    0.22391223912239122,
    0.2239222392223922,
    0.22393223932239323,
    0.22394223942239422,
    0.2239522395223952,
    0.22396223962239623,
    0.22397223972239722,
    0.22398223982239823,
    0.22399223992239922,
    0.2240022400224002,
    0.22401224012240123,
    0.22402224022240222,
    0.2240322403224032,
    0.22404224042240423,
    0.22405224052240522,
    0.22406224062240623,
    0.22407224072240722,
    0.2240822408224082,
    0.22409224092240923,
    0.22410224102241022,
    0.2241122411224112,
    0.22412224122241223,
    0.22413224132241322,
    0.2241422414224142,
    0.22415224152241522,
    0.22416224162241621,
    0.22417224172241723,
    0.22418224182241822,
    0.2241922419224192,
    0.22420224202242023,
    0.22421224212242122,
    0.2242222422224222,
    0.22423224232242323,
    0.22424224242242422,
    0.22425224252242523,
    0.22426224262242622,
    0.2242722427224272,
    0.22428224282242823,
    0.22429224292242922,
    0.2243022430224302,
    0.22431224312243123,
    0.22432224322243222,
    0.22433224332243323,
    0.22434224342243422,
    0.2243522435224352,
    0.22436224362243623,
    0.22437224372243722,
    0.2243822438224382,
    0.22439224392243923,
    0.22440224402244022,
    0.2244122441224412,
    0.22442224422244222,
    0.22443224432244321,
    0.22444224442244423,
    0.22445224452244522,
    0.2244622446224462,
    0.22447224472244723,
    0.22448224482244822,
    0.2244922449224492,
    0.22450224502245023,
    0.22451224512245122,
    0.22452224522245223,
    0.22453224532245322,
    0.2245422454224542,
    0.22455224552245523,
    0.22456224562245622,
    0.2245722457224572,
    0.22458224582245823,
    0.22459224592245922,
    0.22460224602246023,
    0.22461224612246122,
    0.2246222462224622,
    0.22463224632246323,
    0.22464224642246422,
    0.2246522465224652,
    0.22466224662246623,
    0.22467224672246722,
    0.22468224682246823,
    0.22469224692246922,
    0.22470224702247021,
    0.22471224712247123,
    0.22472224722247222,
    0.2247322473224732,
    0.22474224742247423,
    0.22475224752247522,
    0.2247622476224762,
    0.22477224772247723,
    0.22478224782247822,
    0.22479224792247923,
    0.22480224802248022,
    0.2248122481224812,
    0.22482224822248223,
    0.22483224832248322,
    0.2248422484224842,
    0.22485224852248523,
    0.22486224862248622,
    0.22487224872248723,
    0.22488224882248822,
    0.2248922489224892,
    0.22490224902249023,
    0.22491224912249122,
    0.2249222492224922,
    0.22493224932249323,
    0.22494224942249422,
    0.22495224952249523,
    0.22496224962249622,
    0.22497224972249721,
    0.22498224982249823,
    0.22499224992249922,
    0.2250022500225002,
    0.22501225012250123,
    0.22502225022250222,
    0.2250322503225032,
    0.22504225042250423,
    0.22505225052250522,
    0.22506225062250623,
    0.22507225072250722,
    0.2250822508225082,
    0.22509225092250923,
    0.22510225102251022,
    0.2251122511225112,
    0.22512225122251223,
    0.22513225132251322,
    0.22514225142251423,
    0.22515225152251522,
    0.2251622516225162,
    0.22517225172251723,
    0.22518225182251822,
    0.2251922519225192,
    0.22520225202252023,
    0.22521225212252122,
    0.22522225222252223,
    0.22523225232252322,
    0.22524225242252421,
    0.22525225252252523,
    0.22526225262252622,
    0.2252722527225272,
    0.22528225282252823,
    0.22529225292252922,
    0.2253022530225302,
    0.22531225312253123,
    0.22532225322253222,
    0.22533225332253323,
    0.22534225342253422,
    0.2253522535225352,
    0.22536225362253623,
    0.22537225372253722,
    0.2253822538225382,
    0.22539225392253923,
    0.22540225402254022,
    0.22541225412254123,
    0.22542225422254222,
    0.2254322543225432,
    0.22544225442254423,
    0.22545225452254522,
    0.2254622546225462,
    0.22547225472254723,
    0.22548225482254822,
    0.22549225492254923,
    0.22550225502255022,
    0.22551225512255121,
    0.22552225522255223,
    0.22553225532255322,
    0.2255422554225542,
    0.22555225552255523,
    0.22556225562255622,
    0.22557225572255724,
    0.22558225582255823,
    0.22559225592255921,
    0.22560225602256023,
    0.22561225612256122,
    0.2256222562225622,
    0.22563225632256323,
    0.22564225642256422,
    0.2256522565225652,
    0.22566225662256623,
    0.22567225672256722,
    0.22568225682256823,
    0.22569225692256922,
    0.2257022570225702,
    0.22571225712257123,
    0.22572225722257222,
    0.2257322573225732,
    0.22574225742257423,
    0.22575225752257522,
    0.22576225762257623,
    0.22577225772257722,
    0.22578225782257821,
    0.22579225792257923,
    0.22580225802258022,
    0.2258122581225812,
    0.22582225822258223,
    0.22583225832258322,
    0.22584225842258424,
    0.22585225852258523,
    0.22586225862258621,
    0.22587225872258723,
    0.22588225882258822,
    0.2258922589225892,
    0.22590225902259023,
    0.22591225912259122,
    0.2259222592225922,
    0.22593225932259323,
    0.22594225942259422,
    0.22595225952259523,
    0.22596225962259622,
    0.2259722597225972,
    0.22598225982259823,
    0.22599225992259922,
    0.2260022600226002,
    0.22601226012260123,
    0.22602226022260222,
    0.22603226032260323,
    0.22604226042260422,
    0.22605226052260521,
    0.22606226062260623,
    0.22607226072260722,
    0.2260822608226082,
    0.22609226092260923,
    0.22610226102261022,
    0.22611226112261124,
    0.22612226122261223,
    0.22613226132261321,
    0.22614226142261423,
    0.22615226152261522,
    0.2261622616226162,
    0.22617226172261723,
    0.22618226182261822,
    0.2261922619226192,
    0.22620226202262023,
    0.22621226212262122,
    0.22622226222262223,
    0.22623226232262322,
    0.2262422624226242,
    0.22625226252262523,
    0.22626226262262622,
    0.2262722627226272,
    0.22628226282262823,
    0.22629226292262922,
    0.22630226302263023,
    0.22631226312263122,
    0.22632226322263221,
    0.22633226332263323,
    0.22634226342263422,
    0.2263522635226352,
    0.22636226362263623,
    0.22637226372263722,
    0.22638226382263824,
    0.22639226392263923,
    0.22640226402264021,
    0.22641226412264123,
    0.22642226422264222,
    0.2264322643226432,
    0.22644226442264423,
    0.22645226452264522,
    0.2264622646226462,
    0.22647226472264723,
    0.22648226482264822,
    0.22649226492264923,
    0.22650226502265022,
    0.2265122651226512,
    0.22652226522265223,
    0.22653226532265322,
    0.2265422654226542,
    0.22655226552265523,
    0.22656226562265622,
    0.22657226572265723,
    0.22658226582265822,
    0.22659226592265921,
    0.22660226602266023,
    0.22661226612266122,
    0.2266222662226622,
    0.22663226632266323,
    0.22664226642266422,
    0.22665226652266524,
    0.22666226662266623,
    0.22667226672266721,
    0.22668226682266823,
    0.22669226692266922,
    0.2267022670226702,
    0.22671226712267123,
    0.22672226722267222,
    0.22673226732267324,
    0.22674226742267423,
    0.22675226752267522,
    0.22676226762267623,
    0.22677226772267722,
    0.2267822678226782,
    0.22679226792267923,
    0.22680226802268022,
    0.2268122681226812,
    0.22682226822268223,
    0.22683226832268322,
    0.22684226842268423,
    0.22685226852268522,
    0.22686226862268621,
    0.22687226872268723,
    0.22688226882268822,
    0.2268922689226892,
    0.22690226902269023,
    0.22691226912269122,
    0.22692226922269224,
    0.22693226932269323,
    0.22694226942269421,
    0.22695226952269523,
    0.22696226962269622,
    0.2269722697226972,
    0.22698226982269823,
    0.22699226992269922,
    0.22700227002270024,
    0.22701227012270123,
    0.22702227022270222,
    0.22703227032270323,
    0.22704227042270422,
    0.2270522705227052,
    0.22706227062270623,
    0.22707227072270722,
    0.2270822708227082,
    0.22709227092270923,
    0.22710227102271022,
    0.22711227112271123,
    0.22712227122271222,
    0.22713227132271321,
    0.22714227142271423,
    0.22715227152271522,
    0.2271622716227162,
    0.22717227172271723,
    0.22718227182271822,
    0.22719227192271924,
    0.22720227202272023,
    0.22721227212272121,
    0.22722227222272223,
    0.22723227232272322,
    0.2272422724227242,
    0.22725227252272523,
    0.22726227262272622,
    0.22727227272272724,
    0.22728227282272823,
    0.22729227292272922,
    0.22730227302273023,
    0.22731227312273122,
    0.2273222732227322,
    0.22733227332273323,
    0.22734227342273422,
    0.2273522735227352,
    0.22736227362273623,
    0.22737227372273722,
    0.22738227382273823,
    0.22739227392273922,
    0.22740227402274021,
    0.22741227412274123,
    0.22742227422274222,
    0.2274322743227432,
    0.22744227442274423,
    0.22745227452274522,
    0.22746227462274624,
    0.22747227472274723,
    0.22748227482274821,
    0.22749227492274923,
    0.22750227502275022,
    0.2275122751227512,
    0.22752227522275223,
    0.22753227532275322,
    0.22754227542275424,
    0.22755227552275523,
    0.22756227562275622,
    0.22757227572275723,
    0.22758227582275822,
    0.2275922759227592,
    0.22760227602276023,
    0.22761227612276122,
    0.22762227622276224,
    0.22763227632276323,
    0.22764227642276422,
    0.22765227652276523,
    0.22766227662276622,
    0.22767227672276721,
    0.22768227682276823,
    0.22769227692276922,
    0.2277022770227702,
    0.22771227712277123,
    0.22772227722277222,
    0.22773227732277324,
    0.22774227742277423,
    0.22775227752277521,
    0.22776227762277623,
    0.22777227772277722,
    0.2277822778227782,
    0.22779227792277923,
    0.22780227802278022,
    0.22781227812278124,
    0.22782227822278223,
    0.22783227832278322,
    0.22784227842278423,
    0.22785227852278522,
    0.2278622786227862,
    0.22787227872278723,
    0.22788227882278822,
    0.22789227892278924,
    0.22790227902279023,
    0.22791227912279122,
    0.22792227922279223,
    0.22793227932279322,
    0.22794227942279421,
    0.22795227952279523,
    0.22796227962279622,
    0.2279722797227972,
    0.22798227982279823,
    0.22799227992279922,
    0.22800228002280024,
    0.22801228012280123,
    0.22802228022280221,
    0.22803228032280323,
    0.22804228042280422,
    0.2280522805228052,
    0.22806228062280623,
    0.22807228072280722,
    0.22808228082280824,
    0.22809228092280923,
    0.22810228102281022,
    0.22811228112281123,
    0.22812228122281222,
    0.2281322813228132,
    0.22814228142281423,
    0.22815228152281522,
    0.22816228162281624,
    0.22817228172281723,
    0.22818228182281822,
    0.22819228192281923,
    0.22820228202282022,
    0.22821228212282121,
    0.22822228222282223,
    0.22823228232282322,
    0.2282422824228242,
    0.22825228252282523,
    0.22826228262282622,
    0.22827228272282724,
    0.22828228282282823,
    0.22829228292282921,
    0.22830228302283023,
    0.22831228312283122,
    0.2283222832228322,
    0.22833228332283323,
    0.22834228342283422,
    0.22835228352283524,
    0.22836228362283623,
    0.22837228372283722,
    0.22838228382283823,
    0.22839228392283922,
    0.2284022840228402,
    0.22841228412284123,
    0.22842228422284222,
    0.22843228432284324,
    0.22844228442284423,
    0.22845228452284522,
    0.22846228462284623,
    0.22847228472284722,
    0.22848228482284821,
    0.22849228492284923,
    0.22850228502285022,
    0.2285122851228512,
    0.22852228522285223,
    0.22853228532285322,
    0.22854228542285424,
    0.22855228552285523,
    0.22856228562285621,
    0.22857228572285723,
    0.22858228582285822,
    0.2285922859228592,
    0.22860228602286023,
    0.22861228612286122,
    0.22862228622286224,
    0.22863228632286323,
    0.22864228642286422,
    0.22865228652286523,
    0.22866228662286622,
    0.2286722867228672,
    0.22868228682286823,
    0.22869228692286922,
    0.22870228702287024,
    0.22871228712287123,
    0.22872228722287222,
    0.22873228732287323,
    0.22874228742287422,
    0.22875228752287521,
    0.22876228762287623,
    0.22877228772287722,
    0.22878228782287824,
    0.22879228792287923,
    0.22880228802288022,
    0.22881228812288124,
    0.22882228822288223,
    0.22883228832288321,
    0.22884228842288423,
    0.22885228852288522,
    0.2288622886228862,
    0.22887228872288723,
    0.22888228882288822,
    0.22889228892288924,
    0.22890228902289023,
    0.22891228912289122,
    0.22892228922289223,
    0.22893228932289322,
    0.2289422894228942,
    0.22895228952289523,
    0.22896228962289622,
    0.22897228972289724,
    0.22898228982289823,
    0.22899228992289922,
    0.22900229002290023,
    0.22901229012290122,
    0.22902229022290221,
    0.22903229032290323,
    0.22904229042290422,
    0.22905229052290524,
    0.22906229062290623,
    0.22907229072290722,
    0.22908229082290824,
    0.22909229092290923,
    0.22910229102291021,
    0.22911229112291123,
    0.22912229122291222,
    0.2291322913229132,
    0.22914229142291423,
    0.22915229152291522,
    0.22916229162291624,
    0.22917229172291723,
    0.22918229182291822,
    0.22919229192291923,
    0.22920229202292022,
    0.2292122921229212,
    0.22922229222292223,
    0.22923229232292322,
    0.22924229242292424,
    0.22925229252292523,
    0.22926229262292622,
    0.22927229272292723,
    0.22928229282292822,
    0.22929229292292921,
    0.22930229302293023,
    0.22931229312293122,
    0.22932229322293224,
    0.22933229332293323,
    0.22934229342293422,
    0.22935229352293524,
    0.22936229362293623,
    0.22937229372293721,
    0.22938229382293823,
    0.22939229392293922,
    0.2294022940229402,
    0.22941229412294123,
    0.22942229422294222,
    0.22943229432294324,
    0.22944229442294423,
    0.22945229452294522,
    0.22946229462294623,
    0.22947229472294722,
    0.2294822948229482,
    0.22949229492294923,
    0.22950229502295022,
    0.22951229512295124,
    0.22952229522295223,
    0.22953229532295322,
    0.22954229542295423,
    0.22955229552295522,
    0.22956229562295621,
    0.22957229572295723,
    0.22958229582295822,
    0.22959229592295924,
    0.22960229602296023,
    0.22961229612296122,
    0.22962229622296224,
    0.22963229632296323,
    0.22964229642296421,
    0.22965229652296523,
    0.22966229662296622,
    0.2296722967229672,
    0.22968229682296823,
    0.22969229692296922,
    0.22970229702297024,
    0.22971229712297123,
    0.22972229722297222,
    0.22973229732297323,
    0.22974229742297422,
    0.2297522975229752,
    0.22976229762297623,
    0.22977229772297722,
    0.22978229782297824,
    0.22979229792297923,
    0.22980229802298022,
    0.22981229812298123,
    0.22982229822298222,
    0.22983229832298321,
    0.22984229842298423,
    0.22985229852298522,
    0.22986229862298624,
    0.22987229872298723,
    0.22988229882298822,
    0.22989229892298924,
    0.22990229902299023,
    0.22991229912299121,
    0.22992229922299223,
    0.22993229932299322,
    0.22994229942299424,
    0.22995229952299523,
    0.22996229962299622,
    0.22997229972299724,
    0.22998229982299823,
    0.22999229992299922,
    0.23000230002300023,
    0.23001230012300122,
    0.2300223002230022,
    0.23003230032300323,
    0.23004230042300422,
    0.23005230052300524,
    0.23006230062300623,
    0.23007230072300722,
    0.23008230082300823,
    0.23009230092300922,
    0.23010230102301021,
    0.23011230112301123,
    0.23012230122301222,
    0.23013230132301324,
    0.23014230142301423,
    0.23015230152301522,
    0.23016230162301624,
    0.23017230172301723,
    0.23018230182301821,
    0.23019230192301923,
    0.23020230202302022,
    0.23021230212302124,
    0.23022230222302223,
    0.23023230232302322,
    0.23024230242302424,
    0.23025230252302523,
    0.23026230262302622,
    0.23027230272302723,
    0.23028230282302822,
    0.2302923029230292,
    0.23030230302303023,
    0.23031230312303122,
    0.23032230322303224,
    0.23033230332303323,
    0.23034230342303422,
    0.23035230352303523,
    0.23036230362303622,
    0.23037230372303721,
    0.23038230382303823,
    0.23039230392303922,
    0.23040230402304024,
    0.23041230412304123,
    0.23042230422304222,
    0.23043230432304324,
    0.23044230442304423,
    0.23045230452304521,
    0.23046230462304623,
    0.23047230472304722,
    0.23048230482304824,
    0.23049230492304923,
    0.23050230502305022,
    0.23051230512305124,
    0.23052230522305223,
    0.23053230532305322,
    0.23054230542305423,
    0.23055230552305522,
    0.2305623056230562,
    0.23057230572305723,
    0.23058230582305822,
    0.23059230592305924,
    0.23060230602306023,
    0.23061230612306122,
    0.23062230622306223,
    0.23063230632306322,
    0.23064230642306421,
    0.23065230652306523,
    0.23066230662306622,
    0.23067230672306724,
    0.23068230682306823,
    0.23069230692306922,
    0.23070230702307024,
    0.23071230712307123,
    0.23072230722307221,
    0.23073230732307323,
    0.23074230742307422,
    0.23075230752307524,
    0.23076230762307623,
    0.23077230772307722,
    0.23078230782307824,
    0.23079230792307923,
    0.23080230802308022,
    0.23081230812308123,
    0.23082230822308222,
    0.23083230832308324,
    0.23084230842308423,
    0.23085230852308522,
    0.23086230862308624,
    0.23087230872308723,
    0.23088230882308822,
    0.23089230892308923,
    0.23090230902309022,
    0.23091230912309121,
    0.23092230922309223,
    0.23093230932309322,
    0.23094230942309424,
    0.23095230952309523,
    0.23096230962309622,
    0.23097230972309724,
    0.23098230982309823,
    0.23099230992309921,
    0.23100231002310023,
    0.23101231012310122,
    0.23102231022310224,
    0.23103231032310323,
    0.23104231042310422,
    0.23105231052310524,
    0.23106231062310623,
    0.23107231072310722,
    0.23108231082310823,
    0.23109231092310922,
    0.23110231102311024,
    0.23111231112311123,
    0.23112231122311222,
    0.23113231132311324,
    0.23114231142311423,
    0.23115231152311522,
    0.23116231162311623,
    0.23117231172311722,
    0.23118231182311821,
    0.23119231192311923,
    0.23120231202312022,
    0.23121231212312124,
    0.23122231222312223,
    0.23123231232312322,
    0.23124231242312424,
    0.23125231252312523,
    0.23126231262312621,
    0.23127231272312723,
    0.23128231282312822,
    0.23129231292312924,
    0.23130231302313023,
    0.23131231312313122,
    0.23132231322313224,
    0.23133231332313323,
    0.23134231342313422,
    0.23135231352313523,
    0.23136231362313622,
    0.23137231372313724,
    0.23138231382313823,
    0.23139231392313922,
    0.23140231402314024,
    0.23141231412314123,
    0.23142231422314222,
    0.23143231432314323,
    0.23144231442314422,
    0.23145231452314521,
    0.23146231462314623,
    0.23147231472314722,
    0.23148231482314824,
    0.23149231492314923,
    0.23150231502315022,
    0.23151231512315124,
    0.23152231522315223,
    0.23153231532315321,
    0.23154231542315423,
    0.23155231552315522,
    0.23156231562315624,
    0.23157231572315723,
    0.23158231582315822,
    0.23159231592315924,
    0.23160231602316023,
    0.23161231612316122,
    0.23162231622316223,
    0.23163231632316322,
    0.23164231642316424,
    0.23165231652316523,
    0.23166231662316622,
    0.23167231672316724,
    0.23168231682316823,
    0.23169231692316922,
    0.23170231702317023,
    0.23171231712317122,
    0.23172231722317221,
    0.23173231732317323,
    0.23174231742317422,
    0.23175231752317524,
    0.23176231762317623,
    0.23177231772317722,
    0.23178231782317824,
    0.23179231792317923,
    0.23180231802318021,
    0.23181231812318123,
    0.23182231822318222,
    0.23183231832318324,
    0.23184231842318423,
    0.23185231852318522,
    0.23186231862318624,
    0.23187231872318723,
    0.23188231882318822,
    0.23189231892318923,
    0.23190231902319022,
    0.23191231912319124,
    0.23192231922319223,
    0.23193231932319322,
    0.23194231942319424,
    0.23195231952319523,
    0.23196231962319622,
    0.23197231972319723,
    0.23198231982319822,
    0.23199231992319924,
    0.23200232002320023,
    0.23201232012320122,
    0.23202232022320224,
    0.23203232032320323,
    0.23204232042320422,
    0.23205232052320524,
    0.23206232062320623,
    0.23207232072320721,
    0.23208232082320823,
    0.23209232092320922,
    0.23210232102321024,
    0.23211232112321123,
    0.23212232122321222,
    0.23213232132321324,
    0.23214232142321423,
    0.23215232152321522,
    0.23216232162321623,
    0.23217232172321722,
    0.23218232182321824,
    0.23219232192321923,
    0.23220232202322022,
    0.23221232212322124,
    0.23222232222322223,
    0.23223232232322322,
    0.23224232242322423,
    0.23225232252322522,
    0.23226232262322624,
    0.23227232272322723,
    0.23228232282322822,
    0.23229232292322924,
    0.23230232302323023,
    0.23231232312323122,
    0.23232232322323224,
    0.23233232332323323,
    0.23234232342323421,
    0.23235232352323523,
    0.23236232362323622,
    0.23237232372323724,
    0.23238232382323823,
    0.23239232392323922,
    0.23240232402324024,
    0.23241232412324123,
    0.23242232422324222,
    0.23243232432324323,
    0.23244232442324422,
    0.23245232452324524,
    0.23246232462324623,
    0.23247232472324722,
    0.23248232482324824,
    0.23249232492324923,
    0.23250232502325022,
    0.23251232512325123,
    0.23252232522325222,
    0.23253232532325324,
    0.23254232542325423,
    0.23255232552325522,
    0.23256232562325624,
    0.23257232572325723,
    0.23258232582325822,
    0.23259232592325924,
    0.23260232602326023,
    0.23261232612326121,
    0.23262232622326223,
    0.23263232632326322,
    0.23264232642326424,
    0.23265232652326523,
    0.23266232662326622,
    0.23267232672326724,
    0.23268232682326823,
    0.23269232692326922,
    0.23270232702327023,
    0.23271232712327122,
    0.23272232722327224,
    0.23273232732327323,
    0.23274232742327422,
    0.23275232752327524,
    0.23276232762327623,
    0.23277232772327722,
    0.23278232782327823,
    0.23279232792327922,
    0.23280232802328024,
    0.23281232812328123,
    0.23282232822328222,
    0.23283232832328324,
    0.23284232842328423,
    0.23285232852328522,
    0.23286232862328624,
    0.23287232872328723,
    0.23288232882328824,
    0.23289232892328923,
    0.23290232902329022,
    0.23291232912329124,
    0.23292232922329223,
    0.23293232932329322,
    0.23294232942329424,
    0.23295232952329523,
    0.23296232962329622,
    0.23297232972329723,
    0.23298232982329822,
    0.23299232992329924,
    0.23300233002330023,
    0.23301233012330122,
    0.23302233022330224,
    0.23303233032330323,
    0.23304233042330422,
    0.23305233052330523,
    0.23306233062330622,
    0.23307233072330724,
    0.23308233082330823,
    0.23309233092330922,
    0.23310233102331024,
    0.23311233112331123,
    0.23312233122331222,
    0.23313233132331324,
    0.23314233142331423,
    0.23315233152331524,
    0.23316233162331623,
    0.23317233172331722,
    0.23318233182331824,
    0.23319233192331923,
    0.23320233202332022,
    0.23321233212332124,
    0.23322233222332223,
    0.23323233232332322,
    0.23324233242332423,
    0.23325233252332522,
    0.23326233262332624,
    0.23327233272332723,
    0.23328233282332822,
    0.23329233292332924,
    0.23330233302333023,
    0.23331233312333122,
    0.23332233322333223,
    0.23333233332333322,
    0.23334233342333424,
    0.23335233352333523,
    0.23336233362333622,
    0.23337233372333724,
    0.23338233382333823,
    0.23339233392333922,
    0.23340233402334024,
    0.23341233412334123,
    0.23342233422334224,
    0.23343233432334323,
    0.23344233442334422,
    0.23345233452334524,
    0.23346233462334623,
    0.23347233472334722,
    0.23348233482334824,
    0.23349233492334923,
    0.23350233502335022,
    0.23351233512335123,
    0.23352233522335222,
    0.23353233532335324,
    0.23354233542335423,
    0.23355233552335522,
    0.23356233562335624,
    0.23357233572335723,
    0.23358233582335822,
    0.23359233592335923,
    0.23360233602336022,
    0.23361233612336124,
    0.23362233622336223,
    0.23363233632336322,
    0.23364233642336424,
    0.23365233652336523,
    0.23366233662336622,
    0.23367233672336724,
    0.23368233682336823,
    0.23369233692336924,
    0.23370233702337023,
    0.23371233712337122,
    0.23372233722337224,
    0.23373233732337323,
    0.23374233742337422,
    0.23375233752337524,
    0.23376233762337623,
    0.23377233772337722,
    0.23378233782337823,
    0.23379233792337922,
    0.23380233802338024,
    0.23381233812338123,
    0.23382233822338222,
    0.23383233832338324,
    0.23384233842338423,
    0.23385233852338522,
    0.23386233862338623,
    0.23387233872338722,
    0.23388233882338824,
    0.23389233892338923,
    0.23390233902339022,
    0.23391233912339124,
    0.23392233922339223,
    0.23393233932339322,
    0.23394233942339424,
    0.23395233952339523,
    0.23396233962339624,
    0.23397233972339723,
    0.23398233982339822,
    0.23399233992339924,
    0.23400234002340023,
    0.23401234012340122,
    0.23402234022340224,
    0.23403234032340323,
    0.23404234042340424,
    0.23405234052340523,
    0.23406234062340622,
    0.23407234072340724,
    0.23408234082340823,
    0.23409234092340922,
    0.23410234102341024,
    0.23411234112341123,
    0.23412234122341222,
    0.23413234132341323,
    0.23414234142341422,
    0.23415234152341524,
    0.23416234162341623,
    0.23417234172341722,
    0.23418234182341824,
    0.23419234192341923,
    0.23420234202342022,
    0.23421234212342124,
    0.23422234222342223,
    0.23423234232342324,
    0.23424234242342423,
    0.23425234252342522,
    0.23426234262342624,
    0.23427234272342723,
    0.23428234282342822,
    0.23429234292342924,
    0.23430234302343023,
    0.23431234312343124,
    0.23432234322343223,
    0.23433234332343322,
    0.23434234342343424,
    0.23435234352343523,
    0.23436234362343622,
    0.23437234372343724,
    0.23438234382343823,
    0.23439234392343922,
    0.23440234402344023,
    0.23441234412344122,
    0.23442234422344224,
    0.23443234432344323,
    0.23444234442344422,
    0.23445234452344524,
    0.23446234462344623,
    0.23447234472344722,
    0.23448234482344824,
    0.23449234492344923,
    0.23450234502345024,
    0.23451234512345123,
    0.23452234522345222,
    0.23453234532345324,
    0.23454234542345423,
    0.23455234552345522,
    0.23456234562345624,
    0.23457234572345723,
    0.23458234582345824,
    0.23459234592345923,
    0.23460234602346022,
    0.23461234612346124,
    0.23462234622346223,
    0.23463234632346322,
    0.23464234642346424,
    0.23465234652346523,
    0.23466234662346622,
    0.23467234672346723,
    0.23468234682346822,
    0.23469234692346924,
    0.23470234702347023,
    0.23471234712347122,
    0.23472234722347224,
    0.23473234732347323,
    0.23474234742347422,
    0.23475234752347524,
    0.23476234762347623,
    0.23477234772347724,
    0.23478234782347823,
    0.23479234792347922,
    0.23480234802348024,
    0.23481234812348123,
    0.23482234822348222,
    0.23483234832348324,
    0.23484234842348423,
    0.23485234852348524,
    0.23486234862348623,
    0.23487234872348722,
    0.23488234882348824,
    0.23489234892348923,
    0.23490234902349022,
    0.23491234912349124,
    0.23492234922349223,
    0.23493234932349324,
    0.23494234942349423,
    0.23495234952349522,
    0.23496234962349624,
    0.23497234972349723,
    0.23498234982349822,
    0.23499234992349924,
    0.23500235002350023,
    0.23501235012350122,
    0.23502235022350224,
    0.23503235032350323,
    0.23504235042350424,
    0.23505235052350523,
    0.23506235062350622,
    0.23507235072350724,
    0.23508235082350823,
    0.23509235092350922,
    0.23510235102351024,
    0.23511235112351123,
    0.23512235122351224,
    0.23513235132351323,
    0.23514235142351422,
    0.23515235152351524,
    0.23516235162351623,
    0.23517235172351722,
    0.23518235182351824,
    0.23519235192351923,
    0.23520235202352024,
    0.23521235212352123,
    0.23522235222352222,
    0.23523235232352324,
    0.23524235242352423,
    0.23525235252352522,
    0.23526235262352624,
    0.23527235272352723,
    0.23528235282352822,
    0.23529235292352924,
    0.23530235302353023,
    0.23531235312353124,
    0.23532235322353223,
    0.23533235332353322,
    0.23534235342353424,
    0.23535235352353523,
    0.23536235362353622,
    0.23537235372353724,
    0.23538235382353823,
    0.23539235392353924,
    0.23540235402354023,
    0.23541235412354122,
    0.23542235422354224,
    0.23543235432354323,
    0.23544235442354422,
    0.23545235452354524,
    0.23546235462354623,
    0.23547235472354724,
    0.23548235482354823,
    0.23549235492354922,
    0.23550235502355024,
    0.23551235512355123,
    0.23552235522355222,
    0.23553235532355324,
    0.23554235542355423,
    0.23555235552355522,
    0.23556235562355624,
    0.23557235572355723,
    0.23558235582355824,
    0.23559235592355923,
    0.23560235602356022,
    0.23561235612356124,
    0.23562235622356223,
    0.23563235632356322,
    0.23564235642356424,
    0.23565235652356523,
    0.23566235662356624,
    0.23567235672356723,
    0.23568235682356822,
    0.23569235692356924,
    0.23570235702357023,
    0.23571235712357122,
    0.23572235722357224,
    0.23573235732357323,
    0.23574235742357424,
    0.23575235752357523,
    0.23576235762357622,
    0.23577235772357724,
    0.23578235782357823,
    0.23579235792357922,
    0.23580235802358024,
    0.23581235812358123,
    0.23582235822358222,
    0.23583235832358324,
    0.23584235842358423,
    0.23585235852358524,
    0.23586235862358623,
    0.23587235872358722,
    0.23588235882358824,
    0.23589235892358923,
    0.23590235902359022,
    0.23591235912359124,
    0.23592235922359223,
    0.23593235932359324,
    0.23594235942359423,
    0.23595235952359522,
    0.23596235962359624,
    0.23597235972359723,
    0.23598235982359822,
    0.23599235992359924,
    0.23600236002360023,
    0.23601236012360124,
    0.23602236022360223,
    0.23603236032360322,
    0.23604236042360424,
    0.23605236052360523,
    0.23606236062360622,
    0.23607236072360724,
    0.23608236082360823,
    0.23609236092360925,
    0.23610236102361024,
    0.23611236112361123,
    0.23612236122361224,
    0.23613236132361323,
    0.23614236142361422,
    0.23615236152361524,
    0.23616236162361623,
    0.23617236172361722,
    0.23618236182361824,
    0.23619236192361923,
    0.23620236202362024,
    0.23621236212362123,
    0.23622236222362222,
    0.23623236232362324,
    0.23624236242362423,
    0.23625236252362522,
    0.23626236262362624,
    0.23627236272362723,
    0.23628236282362824,
    0.23629236292362923,
    0.23630236302363022,
    0.23631236312363124,
    0.23632236322363223,
    0.23633236332363322,
    0.23634236342363424,
    0.23635236352363523,
    0.23636236362363625,
    0.23637236372363724,
    0.23638236382363823,
    0.23639236392363924,
    0.23640236402364023,
    0.23641236412364122,
    0.23642236422364224,
    0.23643236432364323,
    0.23644236442364422,
    0.23645236452364524,
    0.23646236462364623,
    0.23647236472364724,
    0.23648236482364823,
    0.23649236492364922,
    0.23650236502365024,
    0.23651236512365123,
    0.23652236522365222,
    0.23653236532365324,
    0.23654236542365423,
    0.23655236552365524,
    0.23656236562365623,
    0.23657236572365722,
    0.23658236582365824,
    0.23659236592365923,
    0.23660236602366022,
    0.23661236612366124,
    0.23662236622366223,
    0.23663236632366325,
    0.23664236642366424,
    0.23665236652366523,
    0.23666236662366624,
    0.23667236672366723,
    0.23668236682366822,
    0.23669236692366924,
    0.23670236702367023,
    0.23671236712367122,
    0.23672236722367224,
    0.23673236732367323,
    0.23674236742367424,
    0.23675236752367523,
    0.23676236762367622,
    0.23677236772367724,
    0.23678236782367823,
    0.23679236792367922,
    0.23680236802368024,
    0.23681236812368123,
    0.23682236822368224,
    0.23683236832368323,
    0.23684236842368422,
    0.23685236852368524,
    0.23686236862368623,
    0.23687236872368722,
    0.23688236882368824,
    0.23689236892368923,
    0.23690236902369025,
    0.23691236912369124,
    0.23692236922369223,
    0.23693236932369324,
    0.23694236942369423,
    0.23695236952369522,
    0.23696236962369624,
    0.23697236972369723,
    0.23698236982369825,
    0.23699236992369924,
    0.23700237002370023,
    0.23701237012370124,
    0.23702237022370223,
    0.23703237032370322,
    0.23704237042370424,
    0.23705237052370523,
    0.23706237062370622,
    0.23707237072370724,
    0.23708237082370823,
    0.23709237092370924,
    0.23710237102371023,
    0.23711237112371122,
    0.23712237122371224,
    0.23713237132371323,
    0.23714237142371422,
    0.23715237152371524,
    0.23716237162371623,
    0.23717237172371725,
    0.23718237182371824,
    0.23719237192371923,
    0.23720237202372024,
    0.23721237212372123,
    0.23722237222372222,
    0.23723237232372324,
    0.23724237242372423,
    0.23725237252372525,
    0.23726237262372624,
    0.23727237272372723,
    0.23728237282372824,
    0.23729237292372923,
    0.23730237302373022,
    0.23731237312373124,
    0.23732237322373223,
    0.23733237332373322,
    0.23734237342373424,
    0.23735237352373523,
    0.23736237362373624,
    0.23737237372373723,
    0.23738237382373822,
    0.23739237392373924,
    0.23740237402374023,
    0.23741237412374122,
    0.23742237422374224,
    0.23743237432374323,
    0.23744237442374425,
    0.23745237452374524,
    0.23746237462374623,
    0.23747237472374724,
    0.23748237482374823,
    0.23749237492374922,
    0.23750237502375024,
    0.23751237512375123,
    0.23752237522375225,
    0.23753237532375324,
    0.23754237542375423,
    0.23755237552375524,
    0.23756237562375623,
    0.23757237572375722,
    0.23758237582375824,
    0.23759237592375923,
    0.23760237602376022,
    0.23761237612376124,
    0.23762237622376223,
    0.23763237632376324,
    0.23764237642376423,
    0.23765237652376522,
    0.23766237662376624,
    0.23767237672376723,
    0.23768237682376822,
    0.23769237692376924,
    0.23770237702377023,
    0.23771237712377125,
    0.23772237722377224,
    0.23773237732377323,
    0.23774237742377424,
    0.23775237752377523,
    0.23776237762377622,
    0.23777237772377724,
    0.23778237782377823,
    0.23779237792377925,
    0.23780237802378024,
    0.23781237812378123,
    0.23782237822378224,
    0.23783237832378323,
    0.23784237842378422,
    0.23785237852378524,
    0.23786237862378623,
    0.23787237872378722,
    0.23788237882378824,
    0.23789237892378923,
    0.23790237902379024,
    0.23791237912379123,
    0.23792237922379222,
    0.23793237932379324,
    0.23794237942379423,
    0.23795237952379522,
    0.23796237962379624,
    0.23797237972379723,
    0.23798237982379825,
    0.23799237992379924,
    0.23800238002380023,
    0.23801238012380124,
    0.23802238022380223,
    0.23803238032380322,
    0.23804238042380424,
    0.23805238052380523,
    0.23806238062380625,
    0.23807238072380724,
    0.23808238082380823,
    0.23809238092380924,
    0.23810238102381023,
    0.23811238112381122,
    0.23812238122381224,
    0.23813238132381323,
    0.23814238142381425,
    0.23815238152381524,
    0.23816238162381623,
    0.23817238172381724,
    0.23818238182381823,
    0.23819238192381922,
    0.23820238202382024,
    0.23821238212382123,
    0.23822238222382222,
    0.23823238232382324,
    0.23824238242382423,
    0.23825238252382525,
    0.23826238262382624,
    0.23827238272382723,
    0.23828238282382824,
    0.23829238292382923,
    0.23830238302383022,
    0.23831238312383124,
    0.23832238322383223,
    0.23833238332383325,
    0.23834238342383424,
    0.23835238352383523,
    0.23836238362383624,
    0.23837238372383723,
    0.23838238382383822,
    0.23839238392383924,
    0.23840238402384023,
    0.23841238412384125,
    0.23842238422384224,
    0.23843238432384323,
    0.23844238442384424,
    0.23845238452384523,
    0.23846238462384622,
    0.23847238472384724,
    0.23848238482384823,
    0.23849238492384922,
    0.23850238502385024,
    0.23851238512385123,
    0.23852238522385225,
    0.23853238532385324,
    0.23854238542385423,
    0.23855238552385524,
    0.23856238562385623,
    0.23857238572385722,
    0.23858238582385824,
    0.23859238592385923,
    0.23860238602386025,
    0.23861238612386124,
    0.23862238622386223,
    0.23863238632386324,
    0.23864238642386423,
    0.23865238652386522,
    0.23866238662386624,
    0.23867238672386723,
    0.23868238682386825,
    0.23869238692386924,
    0.23870238702387023,
    0.23871238712387124,
    0.23872238722387223,
    0.23873238732387322,
    0.23874238742387424,
    0.23875238752387523,
    0.23876238762387622,
    0.23877238772387724,
    0.23878238782387823,
    0.23879238792387925,
    0.23880238802388024,
    0.23881238812388123,
    0.23882238822388224,
    0.23883238832388323,
    0.23884238842388422,
    0.23885238852388524,
    0.23886238862388623,
    0.23887238872388725,
    0.23888238882388824,
    0.23889238892388923,
    0.23890238902389024,
    0.23891238912389123,
    0.23892238922389222,
    0.23893238932389324,
    0.23894238942389423,
    0.23895238952389525,
    0.23896238962389624,
    0.23897238972389723,
    0.23898238982389824,
    0.23899238992389923,
    0.23900239002390022,
    0.23901239012390124,
    0.23902239022390223,
    0.23903239032390325,
    0.23904239042390424,
    0.23905239052390523,
    0.23906239062390625,
    0.23907239072390724,
    0.23908239082390823,
    0.23909239092390924,
    0.23910239102391023,
    0.23911239112391122,
    0.23912239122391224,
    0.23913239132391323,
    0.23914239142391425,
    0.23915239152391524,
    0.23916239162391623,
    0.23917239172391724,
    0.23918239182391823,
    0.23919239192391922,
    0.23920239202392024,
    0.23921239212392123,
    0.23922239222392225,
    0.23923239232392324,
    0.23924239242392423,
    0.23925239252392524,
    0.23926239262392623,
    0.23927239272392722,
    0.23928239282392824,
    0.23929239292392923,
    0.23930239302393025,
    0.23931239312393124,
    0.23932239322393223,
    0.23933239332393325,
    0.23934239342393424,
    0.23935239352393523,
    0.23936239362393624,
    0.23937239372393723,
    0.23938239382393822,
    0.23939239392393924,
    0.23940239402394023,
    0.23941239412394125,
    0.23942239422394224,
    0.23943239432394323,
    0.23944239442394424,
    0.23945239452394523,
    0.23946239462394622,
    0.23947239472394724,
    0.23948239482394823,
    0.23949239492394925,
    0.23950239502395024,
    0.23951239512395123,
    0.23952239522395224,
    0.23953239532395323,
    0.23954239542395422,
    0.23955239552395524,
    0.23956239562395623,
    0.23957239572395725,
    0.23958239582395824,
    0.23959239592395923,
    0.23960239602396025,
    0.23961239612396124,
    0.23962239622396223,
    0.23963239632396324,
    0.23964239642396423,
    0.23965239652396522,
    0.23966239662396624,
    0.23967239672396723,
    0.23968239682396825,
    0.23969239692396924,
    0.23970239702397023,
    0.23971239712397124,
    0.23972239722397223,
    0.23973239732397322,
    0.23974239742397424,
    0.23975239752397523,
    0.23976239762397625,
    0.23977239772397724,
    0.23978239782397823,
    0.23979239792397924,
    0.23980239802398023,
    0.23981239812398122,
    0.23982239822398224,
    0.23983239832398323,
    0.23984239842398425,
    0.23985239852398524,
    0.23986239862398623,
    0.23987239872398725,
    0.23988239882398824,
    0.23989239892398923,
    0.23990239902399024,
    0.23991239912399123,
    0.23992239922399222,
    0.23993239932399324,
    0.23994239942399423,
    0.23995239952399525,
    0.23996239962399624,
    0.23997239972399723,
    0.23998239982399824,
    0.23999239992399923,
    0.24000240002400022,
    0.24001240012400124,
    0.24002240022400223,
    0.24003240032400325,
    0.24004240042400424,
    0.24005240052400523,
    0.24006240062400624,
    0.24007240072400723,
    0.24008240082400822,
    0.24009240092400924,
    0.24010240102401023,
    0.24011240112401125,
    0.24012240122401224,
    0.24013240132401323,
    0.24014240142401425,
    0.24015240152401524,
    0.24016240162401623,
    0.24017240172401724,
    0.24018240182401823,
    0.24019240192401925,
    0.24020240202402024,
    0.24021240212402123,
    0.24022240222402225,
    0.24023240232402324,
    0.24024240242402423,
    0.24025240252402524,
    0.24026240262402623,
    0.24027240272402722,
    0.24028240282402824,
    0.24029240292402923,
    0.24030240302403025,
    0.24031240312403124,
    0.24032240322403223,
    0.24033240332403324,
    0.24034240342403423,
    0.24035240352403522,
    0.24036240362403624,
    0.24037240372403723,
    0.24038240382403825,
    0.24039240392403924,
    0.24040240402404023,
    0.24041240412404125,
    0.24042240422404224,
    0.24043240432404323,
    0.24044240442404424,
    0.24045240452404523,
    0.24046240462404625,
    0.24047240472404724,
    0.24048240482404823,
    0.24049240492404925,
    0.24050240502405024,
    0.24051240512405123,
    0.24052240522405224,
    0.24053240532405323,
    0.24054240542405422,
    0.24055240552405524,
    0.24056240562405623,
    0.24057240572405725,
    0.24058240582405824,
    0.24059240592405923,
    0.24060240602406024,
    0.24061240612406123,
    0.24062240622406222,
    0.24063240632406324,
    0.24064240642406423,
    0.24065240652406525,
    0.24066240662406624,
    0.24067240672406723,
    0.24068240682406825,
    0.24069240692406924,
    0.24070240702407023,
    0.24071240712407124,
    0.24072240722407223,
    0.24073240732407325,
    0.24074240742407424,
    0.24075240752407523,
    0.24076240762407625,
    0.24077240772407724,
    0.24078240782407823,
    0.24079240792407924,
    0.24080240802408023,
    0.24081240812408122,
    0.24082240822408224,
    0.24083240832408323,
    0.24084240842408425,
    0.24085240852408524,
    0.24086240862408623,
    0.24087240872408724,
    0.24088240882408823,
    0.24089240892408922,
    0.24090240902409024,
    0.24091240912409123,
    0.24092240922409225,
    0.24093240932409324,
    0.24094240942409423,
    0.24095240952409525,
    0.24096240962409624,
    0.24097240972409723,
    0.24098240982409824,
    0.24099240992409923,
    0.24100241002410025,
    0.24101241012410124,
    0.24102241022410223,
    0.24103241032410325,
    0.24104241042410424,
    0.24105241052410523,
    0.24106241062410624,
    0.24107241072410723,
    0.24108241082410822,
    0.24109241092410924,
    0.24110241102411023,
    0.24111241112411125,
    0.24112241122411224,
    0.24113241132411323,
    0.24114241142411424,
    0.24115241152411523,
    0.24116241162411622,
    0.24117241172411724,
    0.24118241182411823,
    0.24119241192411925,
    0.24120241202412024,
    0.24121241212412123,
    0.24122241222412225,
    0.24123241232412324,
    0.24124241242412423,
    0.24125241252412524,
    0.24126241262412623,
    0.24127241272412725,
    0.24128241282412824,
    0.24129241292412923,
    0.24130241302413025,
    0.24131241312413124,
    0.24132241322413223,
    0.24133241332413324,
    0.24134241342413423,
    0.24135241352413525,
    0.24136241362413624,
    0.24137241372413723,
    0.24138241382413825,
    0.24139241392413924,
    0.24140241402414023,
    0.24141241412414124,
    0.24142241422414223,
    0.24143241432414322,
    0.24144241442414424,
    0.24145241452414523,
    0.24146241462414625,
    0.24147241472414724,
    0.24148241482414823,
    0.24149241492414925,
    0.24150241502415024,
    0.24151241512415123,
    0.24152241522415224,
    0.24153241532415323,
    0.24154241542415425,
    0.24155241552415524,
    0.24156241562415623,
    0.24157241572415725,
    0.24158241582415824,
    0.24159241592415923,
    0.24160241602416024,
    0.24161241612416123,
    0.24162241622416225,
    0.24163241632416324,
    0.24164241642416423,
    0.24165241652416525,
    0.24166241662416624,
    0.24167241672416723,
    0.24168241682416824,
    0.24169241692416923,
    0.24170241702417022,
    0.24171241712417124,
    0.24172241722417223,
    0.24173241732417325,
    0.24174241742417424,
    0.24175241752417523,
    0.24176241762417625,
    0.24177241772417724,
    0.24178241782417823,
    0.24179241792417924,
    0.24180241802418023,
    0.24181241812418125,
    0.24182241822418224,
    0.24183241832418323,
    0.24184241842418425,
    0.24185241852418524,
    0.24186241862418623,
    0.24187241872418724,
    0.24188241882418823,
    0.24189241892418925,
    0.24190241902419024,
    0.24191241912419123,
    0.24192241922419225,
    0.24193241932419324,
    0.24194241942419423,
    0.24195241952419524,
    0.24196241962419623,
    0.24197241972419722,
    0.24198241982419824,
    0.24199241992419923,
    0.24200242002420025,
    0.24201242012420124,
    0.24202242022420223,
    0.24203242032420325,
    0.24204242042420424,
    0.24205242052420523,
    0.24206242062420624,
    0.24207242072420723,
    0.24208242082420825,
    0.24209242092420924,
    0.24210242102421023,
    0.24211242112421125,
    0.24212242122421224,
    0.24213242132421323,
    0.24214242142421424,
    0.24215242152421523,
    0.24216242162421625,
    0.24217242172421724,
    0.24218242182421823,
    0.24219242192421925,
    0.24220242202422024,
    0.24221242212422123,
    0.24222242222422224,
    0.24223242232422323,
    0.24224242242422425,
    0.24225242252422524,
    0.24226242262422623,
    0.24227242272422725,
    0.24228242282422824,
    0.24229242292422923,
    0.24230242302423025,
    0.24231242312423124,
    0.24232242322423223,
    0.24233242332423324,
    0.24234242342423423,
    0.24235242352423525,
    0.24236242362423624,
    0.24237242372423723,
    0.24238242382423825,
    0.24239242392423924,
    0.24240242402424023,
    0.24241242412424124,
    0.24242242422424223,
    0.24243242432424325,
    0.24244242442424424,
    0.24245242452424523,
    0.24246242462424625,
    0.24247242472424724,
    0.24248242482424823,
    0.24249242492424924,
    0.24250242502425023,
    0.24251242512425125,
    0.24252242522425224,
    0.24253242532425323,
    0.24254242542425425,
    0.24255242552425524,
    0.24256242562425623,
    0.24257242572425725,
    0.24258242582425824,
    0.24259242592425923,
    0.24260242602426024,
    0.24261242612426123,
    0.24262242622426225,
    0.24263242632426324,
    0.24264242642426423,
    0.24265242652426525,
    0.24266242662426624,
    0.24267242672426723,
    0.24268242682426824,
    0.24269242692426923,
    0.24270242702427025,
    0.24271242712427124,
    0.24272242722427223,
    0.24273242732427325,
    0.24274242742427424,
    0.24275242752427523,
    0.24276242762427624,
    0.24277242772427723,
    0.24278242782427825,
    0.24279242792427924,
    0.24280242802428023,
    0.24281242812428125,
    0.24282242822428224,
    0.24283242832428323,
    0.24284242842428425,
    0.24285242852428524,
    0.24286242862428623,
    0.24287242872428724,
    0.24288242882428823,
    0.24289242892428925,
    0.24290242902429024,
    0.24291242912429123,
    0.24292242922429225,
    0.24293242932429324,
    0.24294242942429423,
    0.24295242952429524,
    0.24296242962429623,
    0.24297242972429725,
    0.24298242982429824,
    0.24299242992429923,
    0.24300243002430025,
    0.24301243012430124,
    0.24302243022430223,
    0.24303243032430324,
    0.24304243042430423,
    0.24305243052430525,
    0.24306243062430624,
    0.24307243072430723,
    0.24308243082430825,
    0.24309243092430924,
    0.24310243102431023,
    0.24311243112431125,
    0.24312243122431224,
    0.24313243132431323,
    0.24314243142431424,
    0.24315243152431523,
    0.24316243162431625,
    0.24317243172431724,
    0.24318243182431823,
    0.24319243192431925,
    0.24320243202432024,
    0.24321243212432123,
    0.24322243222432224,
    0.24323243232432323,
    0.24324243242432425,
    0.24325243252432524,
    0.24326243262432623,
    0.24327243272432725,
    0.24328243282432824,
    0.24329243292432923,
    0.24330243302433024,
    0.24331243312433123,
    0.24332243322433225,
    0.24333243332433324,
    0.24334243342433423,
    0.24335243352433525,
    0.24336243362433624,
    0.24337243372433723,
    0.24338243382433825,
    0.24339243392433924,
    0.24340243402434025,
    0.24341243412434124,
    0.24342243422434223,
    0.24343243432434325,
    0.24344243442434424,
    0.24345243452434523,
    0.24346243462434625,
    0.24347243472434724,
    0.24348243482434823,
    0.24349243492434924,
    0.24350243502435023,
    0.24351243512435125,
    0.24352243522435224,
    0.24353243532435323,
    0.24354243542435425,
    0.24355243552435524,
    0.24356243562435623,
    0.24357243572435724,
    0.24358243582435823,
    0.24359243592435925,
    0.24360243602436024,
    0.24361243612436123,
    0.24362243622436225,
    0.24363243632436324,
    0.24364243642436423,
    0.24365243652436525,
    0.24366243662436624,
    0.24367243672436725,
    0.24368243682436824,
    0.24369243692436923,
    0.24370243702437025,
    0.24371243712437124,
    0.24372243722437223,
    0.24373243732437325,
    0.24374243742437424,
    0.24375243752437523,
    0.24376243762437624,
    0.24377243772437723,
    0.24378243782437825,
    0.24379243792437924,
    0.24380243802438023,
    0.24381243812438125,
    0.24382243822438224,
    0.24383243832438323,
    0.24384243842438424,
    0.24385243852438523,
    0.24386243862438625,
    0.24387243872438724,
    0.24388243882438823,
    0.24389243892438925,
    0.24390243902439024,
    0.24391243912439123,
    0.24392243922439225,
    0.24393243932439324,
    0.24394243942439425,
    0.24395243952439524,
    0.24396243962439623,
    0.24397243972439725,
    0.24398243982439824,
    0.24399243992439923,
    0.24400244002440025,
    0.24401244012440124,
    0.24402244022440223,
    0.24403244032440324,
    0.24404244042440423,
    0.24405244052440525,
    0.24406244062440624,
    0.24407244072440723,
    0.24408244082440825,
    0.24409244092440924,
    0.24410244102441023,
    0.24411244112441124,
    0.24412244122441223,
    0.24413244132441325,
    0.24414244142441424,
    0.24415244152441523,
    0.24416244162441625,
    0.24417244172441724,
    0.24418244182441823,
    0.24419244192441925,
    0.24420244202442024,
    0.24421244212442125,
    0.24422244222442224,
    0.24423244232442323,
    0.24424244242442425,
    0.24425244252442524,
    0.24426244262442623,
    0.24427244272442725,
    0.24428244282442824,
    0.24429244292442925,
    0.24430244302443024,
    0.24431244312443123,
    0.24432244322443225,
    0.24433244332443324,
    0.24434244342443423,
    0.24435244352443525,
    0.24436244362443624,
    0.24437244372443723,
    0.24438244382443824,
    0.24439244392443923,
    0.24440244402444025,
    0.24441244412444124,
    0.24442244422444223,
    0.24443244432444325,
    0.24444244442444424,
    0.24445244452444523,
    0.24446244462444625,
    0.24447244472444724,
    0.24448244482444825,
    0.24449244492444924,
    0.24450244502445023,
    0.24451244512445125,
    0.24452244522445224,
    0.24453244532445323,
    0.24454244542445425,
    0.24455244552445524,
    0.24456244562445625,
    0.24457244572445724,
    0.24458244582445823,
    0.24459244592445925,
    0.24460244602446024,
    0.24461244612446123,
    0.24462244622446225,
    0.24463244632446324,
    0.24464244642446423,
    0.24465244652446524,
    0.24466244662446623,
    0.24467244672446725,
    0.24468244682446824,
    0.24469244692446923,
    0.24470244702447025,
    0.24471244712447124,
    0.24472244722447223,
    0.24473244732447325,
    0.24474244742447424,
    0.24475244752447525,
    0.24476244762447624,
    0.24477244772447723,
    0.24478244782447825,
    0.24479244792447924,
    0.24480244802448023,
    0.24481244812448125,
    0.24482244822448224,
    0.24483244832448325,
    0.24484244842448424,
    0.24485244852448523,
    0.24486244862448625,
    0.24487244872448724,
    0.24488244882448823,
    0.24489244892448925,
    0.24490244902449024,
    0.24491244912449123,
    0.24492244922449224,
    0.24493244932449323,
    0.24494244942449425,
    0.24495244952449524,
    0.24496244962449623,
    0.24497244972449725,
    0.24498244982449824,
    0.24499244992449923,
    0.24500245002450025,
    0.24501245012450124,
    0.24502245022450225,
    0.24503245032450324,
    0.24504245042450423,
    0.24505245052450525,
    0.24506245062450624,
    0.24507245072450723,
    0.24508245082450825,
    0.24509245092450924,
    0.24510245102451025,
    0.24511245112451124,
    0.24512245122451223,
    0.24513245132451325,
    0.24514245142451424,
    0.24515245152451523,
    0.24516245162451625,
    0.24517245172451724,
    0.24518245182451823,
    0.24519245192451924,
    0.24520245202452023,
    0.24521245212452125,
    0.24522245222452224,
    0.24523245232452323,
    0.24524245242452425,
    0.24525245252452524,
    0.24526245262452623,
    0.24527245272452725,
    0.24528245282452824,
    0.24529245292452925,
    0.24530245302453024,
    0.24531245312453123,
    0.24532245322453225,
    0.24533245332453324,
    0.24534245342453423,
    0.24535245352453525,
    0.24536245362453624,
    0.24537245372453725,
    0.24538245382453824,
    0.24539245392453923,
    0.24540245402454025,
    0.24541245412454124,
    0.24542245422454223,
    0.24543245432454325,
    0.24544245442454424,
    0.24545245452454525,
    0.24546245462454624,
    0.24547245472454723,
    0.24548245482454825,
    0.24549245492454924,
    0.24550245502455023,
    0.24551245512455125,
    0.24552245522455224,
    0.24553245532455323,
    0.24554245542455425,
    0.24555245552455524,
    0.24556245562455625,
    0.24557245572455724,
    0.24558245582455823,
    0.24559245592455925,
    0.24560245602456024,
    0.24561245612456123,
    0.24562245622456225,
    0.24563245632456324,
    0.24564245642456425,
    0.24565245652456524,
    0.24566245662456623,
    0.24567245672456725,
    0.24568245682456824,
    0.24569245692456923,
    0.24570245702457025,
    0.24571245712457124,
    0.24572245722457225,
    0.24573245732457324,
    0.24574245742457423,
    0.24575245752457525,
    0.24576245762457624,
    0.24577245772457723,
    0.24578245782457825,
    0.24579245792457924,
    0.24580245802458023,
    0.24581245812458125,
    0.24582245822458224,
    0.24583245832458325,
    0.24584245842458424,
    0.24585245852458523,
    0.24586245862458625,
    0.24587245872458724,
    0.24588245882458823,
    0.24589245892458925,
    0.24590245902459024,
    0.24591245912459125,
    0.24592245922459224,
    0.24593245932459323,
    0.24594245942459425,
    0.24595245952459524,
    0.24596245962459623,
    0.24597245972459725,
    0.24598245982459824,
    0.24599245992459925,
    0.24600246002460024,
    0.24601246012460123,
    0.24602246022460225,
    0.24603246032460324,
    0.24604246042460423,
    0.24605246052460525,
    0.24606246062460624,
    0.24607246072460723,
    0.24608246082460825,
    0.24609246092460924,
    0.24610246102461025,
    0.24611246112461124,
    0.24612246122461223,
    0.24613246132461325,
    0.24614246142461424,
    0.24615246152461523,
    0.24616246162461625,
    0.24617246172461724,
    0.24618246182461825,
    0.24619246192461924,
    0.24620246202462023,
    0.24621246212462125,
    0.24622246222462224,
    0.24623246232462323,
    0.24624246242462425,
    0.24625246252462524,
    0.24626246262462625,
    0.24627246272462724,
    0.24628246282462823,
    0.24629246292462925,
    0.24630246302463024,
    0.24631246312463123,
    0.24632246322463225,
    0.24633246332463324,
    0.24634246342463426,
    0.24635246352463525,
    0.24636246362463624,
    0.24637246372463725,
    0.24638246382463824,
    0.24639246392463923,
    0.24640246402464025,
    0.24641246412464124,
    0.24642246422464223,
    0.24643246432464325,
    0.24644246442464424,
    0.24645246452464525,
    0.24646246462464624,
    0.24647246472464723,
    0.24648246482464825,
    0.24649246492464924,
    0.24650246502465023,
    0.24651246512465125,
    0.24652246522465224,
    0.24653246532465325,
    0.24654246542465424,
    0.24655246552465523,
    0.24656246562465625,
    0.24657246572465724,
    0.24658246582465823,
    0.24659246592465925,
    0.24660246602466024,
    0.24661246612466126,
    0.24662246622466225,
    0.24663246632466324,
    0.24664246642466425,
    0.24665246652466524,
    0.24666246662466623,
    0.24667246672466725,
    0.24668246682466824,
    0.24669246692466923,
    0.24670246702467025,
    0.24671246712467124,
    0.24672246722467225,
    0.24673246732467324,
    0.24674246742467423,
    0.24675246752467525,
    0.24676246762467624,
    0.24677246772467723,
    0.24678246782467825,
    0.24679246792467924,
    0.24680246802468025,
    0.24681246812468124,
    0.24682246822468223,
    0.24683246832468325,
    0.24684246842468424,
    0.24685246852468523,
    0.24686246862468625,
    0.24687246872468724,
    0.24688246882468826,
    0.24689246892468925,
    0.24690246902469024,
    0.24691246912469125,
    0.24692246922469224,
    0.24693246932469323,
    0.24694246942469425,
    0.24695246952469524,
    0.24696246962469623,
    0.24697246972469725,
    0.24698246982469824,
    0.24699246992469925,
    0.24700247002470024,
    0.24701247012470123,
    0.24702247022470225,
    0.24703247032470324,
    0.24704247042470423,
    0.24705247052470525,
    0.24706247062470624,
    0.24707247072470725,
    0.24708247082470824,
    0.24709247092470923,
    0.24710247102471025,
    0.24711247112471124,
    0.24712247122471223,
    0.24713247132471325,
    0.24714247142471424,
    0.24715247152471526,
    0.24716247162471625,
    0.24717247172471724,
    0.24718247182471825,
    0.24719247192471924,
    0.24720247202472023,
    0.24721247212472125,
    0.24722247222472224,
    0.24723247232472323,
    0.24724247242472425,
    0.24725247252472524,
    0.24726247262472625,
    0.24727247272472724,
    0.24728247282472823,
    0.24729247292472925,
    0.24730247302473024,
    0.24731247312473123,
    0.24732247322473225,
    0.24733247332473324,
    0.24734247342473425,
    0.24735247352473524,
    0.24736247362473623,
    0.24737247372473725,
    0.24738247382473824,
    0.24739247392473923,
    0.24740247402474025,
    0.24741247412474124,
    0.24742247422474226,
    0.24743247432474325,
    0.24744247442474424,
    0.24745247452474525,
    0.24746247462474624,
    0.24747247472474723,
    0.24748247482474825,
    0.24749247492474924,
    0.24750247502475026,
    0.24751247512475125,
    0.24752247522475224,
    0.24753247532475325,
    0.24754247542475424,
    0.24755247552475523,
    0.24756247562475625,
    0.24757247572475724,
    0.24758247582475823,
    0.24759247592475925,
    0.24760247602476024,
    0.24761247612476125,
    0.24762247622476224,
    0.24763247632476323,
    0.24764247642476425,
    0.24765247652476524,
    0.24766247662476623,
    0.24767247672476725,
    0.24768247682476824,
    0.24769247692476926,
    0.24770247702477025,
    0.24771247712477124,
    0.24772247722477225,
    0.24773247732477324,
    0.24774247742477423,
    0.24775247752477525,
    0.24776247762477624,
    0.24777247772477726,
    0.24778247782477825,
    0.24779247792477924,
    0.24780247802478025,
    0.24781247812478124,
    0.24782247822478223,
    0.24783247832478325,
    0.24784247842478424,
    0.24785247852478523,
    0.24786247862478625,
    0.24787247872478724,
    0.24788247882478825,
    0.24789247892478924,
    0.24790247902479023,
    0.24791247912479125,
    0.24792247922479224,
    0.24793247932479323,
    0.24794247942479425,
    0.24795247952479524,
    0.24796247962479626,
    0.24797247972479725,
    0.24798247982479824,
    0.24799247992479925,
    0.24800248002480024,
    0.24801248012480123,
    0.24802248022480225,
    0.24803248032480324,
    0.24804248042480426,
    0.24805248052480525,
    0.24806248062480624,
    0.24807248072480725,
    0.24808248082480824,
    0.24809248092480923,
    0.24810248102481025,
    0.24811248112481124,
    0.24812248122481223,
    0.24813248132481325,
    0.24814248142481424,
    0.24815248152481525,
    0.24816248162481624,
    0.24817248172481723,
    0.24818248182481825,
    0.24819248192481924,
    0.24820248202482023,
    0.24821248212482125,
    0.24822248222482224,
    0.24823248232482326,
    0.24824248242482425,
    0.24825248252482524,
    0.24826248262482625,
    0.24827248272482724,
    0.24828248282482823,
    0.24829248292482925,
    0.24830248302483024,
    0.24831248312483126,
    0.24832248322483225,
    0.24833248332483324,
    0.24834248342483425,
    0.24835248352483524,
    0.24836248362483623,
    0.24837248372483725,
    0.24838248382483824,
    0.24839248392483926,
    0.24840248402484025,
    0.24841248412484124,
    0.24842248422484225,
    0.24843248432484324,
    0.24844248442484423,
    0.24845248452484525,
    0.24846248462484624,
    0.24847248472484723,
    0.24848248482484825,
    0.24849248492484924,
    0.24850248502485026,
    0.24851248512485125,
    0.24852248522485224,
    0.24853248532485325,
    0.24854248542485424,
    0.24855248552485523,
    0.24856248562485625,
    0.24857248572485724,
    0.24858248582485826,
    0.24859248592485925,
    0.24860248602486024,
    0.24861248612486125,
    0.24862248622486224,
    0.24863248632486323,
    0.24864248642486425,
    0.24865248652486524,
    0.24866248662486626,
    0.24867248672486725,
    0.24868248682486824,
    0.24869248692486925,
    0.24870248702487024,
    0.24871248712487123,
    0.24872248722487225,
    0.24873248732487324,
    0.24874248742487423,
    0.24875248752487525,
    0.24876248762487624,
    0.24877248772487726,
    0.24878248782487825,
    0.24879248792487924,
    0.24880248802488025,
    0.24881248812488124,
    0.24882248822488223,
    0.24883248832488325,
    0.24884248842488424,
    0.24885248852488526,
    0.24886248862488625,
    0.24887248872488724,
    0.24888248882488825,
    0.24889248892488924,
    0.24890248902489023,
    0.24891248912489125,
    0.24892248922489224,
    0.24893248932489326,
    0.24894248942489425,
    0.24895248952489524,
    0.24896248962489625,
    0.24897248972489724,
    0.24898248982489823,
    0.24899248992489925,
    0.24900249002490024,
    0.24901249012490123,
    0.24902249022490225,
    0.24903249032490324,
    0.24904249042490426,
    0.24905249052490525,
    0.24906249062490624,
    0.24907249072490725,
    0.24908249082490824,
    0.24909249092490923,
    0.24910249102491025,
    0.24911249112491124,
    0.24912249122491226,
    0.24913249132491325,
    0.24914249142491424,
    0.24915249152491525,
    0.24916249162491624,
    0.24917249172491723,
    0.24918249182491825,
    0.24919249192491924,
    0.24920249202492026,
    0.24921249212492125,
    0.24922249222492224,
    0.24923249232492325,
    0.24924249242492424,
    0.24925249252492523,
    0.24926249262492625,
    0.24927249272492724,
    0.24928249282492823,
    0.24929249292492925,
    0.24930249302493024,
    0.24931249312493126,
    0.24932249322493225,
    0.24933249332493324,
    0.24934249342493425,
    0.24935249352493524,
    0.24936249362493623,
    0.24937249372493725,
    0.24938249382493824,
    0.24939249392493926,
    0.24940249402494025,
    0.24941249412494124,
    0.24942249422494225,
    0.24943249432494324,
    0.24944249442494423,
    0.24945249452494525,
    0.24946249462494624,
    0.24947249472494726,
    0.24948249482494825,
    0.24949249492494924,
    0.24950249502495025,
    0.24951249512495124,
    0.24952249522495223,
    0.24953249532495325,
    0.24954249542495424,
    0.24955249552495526,
    0.24956249562495625,
    0.24957249572495724,
    0.24958249582495826,
    0.24959249592495925,
    0.24960249602496024,
    0.24961249612496125,
    0.24962249622496224,
    0.24963249632496323,
    0.24964249642496425,
    0.24965249652496524,
    0.24966249662496626,
    0.24967249672496725,
    0.24968249682496824,
    0.24969249692496925,
    0.24970249702497024,
    0.24971249712497123,
    0.24972249722497225,
    0.24973249732497324,
    0.24974249742497426,
    0.24975249752497525,
    0.24976249762497624,
    0.24977249772497725,
    0.24978249782497824,
    0.24979249792497923,
    0.24980249802498025,
    0.24981249812498124,
    0.24982249822498226,
    0.24983249832498325,
    0.24984249842498424,
    0.24985249852498526,
    0.24986249862498625,
    0.24987249872498724,
    0.24988249882498825,
    0.24989249892498924,
    0.24990249902499023,
    0.24991249912499125,
    0.24992249922499224,
    0.24993249932499326,
    0.24994249942499425,
    0.24995249952499524,
    0.24996249962499625,
    0.24997249972499724,
    0.24998249982499823,
    0.24999249992499925,
    0.25000250002500024,
    0.25001250012500126,
    0.2500225002250022,
    0.25003250032500324,
    0.25004250042500425,
    0.2500525005250053,
    0.25006250062500623,
    0.25007250072500725,
    0.25008250082500827,
    0.25009250092500923,
    0.25010250102501025,
    0.25011250112501127,
    0.25012250122501223,
    0.25013250132501325,
    0.25014250142501426,
    0.2501525015250152,
    0.25016250162501624,
    0.25017250172501726,
    0.2501825018250182,
    0.25019250192501924,
    0.25020250202502026,
    0.2502125021250212,
    0.25022250222502224,
    0.25023250232502325,
    0.25024250242502427,
    0.25025250252502523,
    0.25026250262502625,
    0.25027250272502727,
    0.25028250282502823,
    0.25029250292502925,
    0.25030250302503027,
    0.2503125031250312,
    0.25032250322503224,
    0.25033250332503326,
    0.2503425034250342,
    0.25035250352503524,
    0.25036250362503626,
    0.2503725037250372,
    0.25038250382503824,
    0.25039250392503926,
    0.2504025040250403,
    0.25041250412504124,
    0.25042250422504225,
    0.25043250432504327,
    0.25044250442504423,
    0.25045250452504525,
    0.25046250462504627,
    0.25047250472504723,
    0.25048250482504825,
    0.25049250492504926,
    0.2505025050250502,
    0.25051250512505124,
    0.25052250522505226,
    0.2505325053250532,
    0.25054250542505424,
    0.25055250552505526,
    0.2505625056250562,
    0.25057250572505724,
    0.25058250582505825,
    0.2505925059250593,
    0.25060250602506023,
    0.25061250612506125,
    0.25062250622506227,
    0.25063250632506323,
    0.25064250642506425,
    0.25065250652506527,
    0.25066250662506623,
    0.25067250672506725,
    0.25068250682506826,
    0.2506925069250692,
    0.25070250702507024,
    0.25071250712507126,
    0.2507225072250722,
    0.25073250732507324,
    0.25074250742507426,
    0.2507525075250752,
    0.25076250762507624,
    0.25077250772507725,
    0.25078250782507827,
    0.25079250792507923,
    0.25080250802508025,
    0.25081250812508127,
    0.25082250822508223,
    0.25083250832508325,
    0.25084250842508427,
    0.2508525085250852,
    0.25086250862508624,
    0.25087250872508726,
    0.2508825088250882,
    0.25089250892508924,
    0.25090250902509026,
    0.2509125091250912,
    0.25092250922509224,
    0.25093250932509326,
    0.2509425094250943,
    0.25095250952509524,
    0.25096250962509625,
    0.25097250972509727,
    0.25098250982509823,
    0.25099250992509925,
    0.25100251002510027,
    0.25101251012510123,
    0.25102251022510225,
    0.25103251032510326,
    0.2510425104251042,
    0.25105251052510524,
    0.25106251062510626,
    0.2510725107251072,
    0.25108251082510824,
    0.25109251092510926,
    0.2511025110251102,
    0.25111251112511124,
    0.25112251122511225,
    0.2511325113251133,
    0.25114251142511423,
    0.25115251152511525,
    0.25116251162511627,
    0.25117251172511723,
    0.25118251182511825,
    0.25119251192511927,
    0.25120251202512023,
    0.25121251212512125,
    0.25122251222512226,
    0.2512325123251232,
    0.25124251242512424,
    0.25125251252512526,
    0.2512625126251262,
    0.25127251272512724,
    0.25128251282512826,
    0.2512925129251293,
    0.25130251302513024,
    0.25131251312513125,
    0.25132251322513227,
    0.25133251332513323,
    0.25134251342513425,
    0.25135251352513527,
    0.25136251362513623,
    0.25137251372513725,
    0.25138251382513827,
    0.2513925139251392,
    0.25140251402514024,
    0.25141251412514126,
    0.2514225142251422,
    0.25143251432514324,
    0.25144251442514426,
    0.2514525145251452,
    0.25146251462514624,
    0.25147251472514726,
    0.2514825148251483,
    0.25149251492514924,
    0.25150251502515025,
    0.25151251512515127,
    0.25152251522515223,
    0.25153251532515325,
    0.25154251542515427,
    0.25155251552515523,
    0.25156251562515625,
    0.25157251572515726,
    0.2515825158251582,
    0.25159251592515924,
    0.25160251602516026,
    0.2516125161251612,
    0.25162251622516224,
    0.25163251632516326,
    0.2516425164251642,
    0.25165251652516524,
    0.25166251662516625,
    0.2516725167251673,
    0.25168251682516823,
    0.25169251692516925,
    0.25170251702517027,
    0.25171251712517123,
    0.25172251722517225,
    0.25173251732517327,
    0.25174251742517423,
    0.25175251752517525,
    0.25176251762517626,
    0.2517725177251772,
    0.25178251782517824,
    0.25179251792517926,
    0.2518025180251802,
    0.25181251812518124,
    0.25182251822518226,
    0.2518325183251833,
    0.25184251842518424,
    0.25185251852518525,
    0.25186251862518627,
    0.25187251872518723,
    0.25188251882518825,
    0.25189251892518927,
    0.25190251902519023,
    0.25191251912519125,
    0.25192251922519227,
    0.2519325193251932,
    0.25194251942519424,
    0.25195251952519526,
    0.2519625196251962,
    0.25197251972519724,
    0.25198251982519826,
    0.2519925199251992,
    0.25200252002520024,
    0.25201252012520126,
    0.2520225202252023,
    0.25203252032520324,
    0.25204252042520425,
    0.25205252052520527,
    0.25206252062520623,
    0.25207252072520725,
    0.25208252082520827,
    0.25209252092520923,
    0.25210252102521025,
    0.25211252112521126,
    0.2521225212252122,
    0.25213252132521324,
    0.25214252142521426,
    0.2521525215252152,
    0.25216252162521624,
    0.25217252172521726,
    0.2521825218252183,
    0.25219252192521924,
    0.25220252202522025,
    0.2522125221252213,
    0.25222252222522223,
    0.25223252232522325,
    0.25224252242522427,
    0.25225252252522523,
    0.25226252262522625,
    0.25227252272522727,
    0.25228252282522823,
    0.25229252292522925,
    0.25230252302523026,
    0.2523125231252312,
    0.25232252322523224,
    0.25233252332523326,
    0.2523425234252342,
    0.25235252352523524,
    0.25236252362523626,
    0.2523725237252373,
    0.25238252382523824,
    0.25239252392523925,
    0.25240252402524027,
    0.25241252412524123,
    0.25242252422524225,
    0.25243252432524327,
    0.25244252442524423,
    0.25245252452524525,
    0.25246252462524627,
    0.2524725247252472,
    0.25248252482524824,
    0.25249252492524926,
    0.2525025250252502,
    0.25251252512525124,
    0.25252252522525226,
    0.2525325253252532,
    0.25254252542525424,
    0.25255252552525526,
    0.2525625256252563,
    0.25257252572525724,
    0.25258252582525825,
    0.25259252592525927,
    0.25260252602526023,
    0.25261252612526125,
    0.25262252622526227,
    0.25263252632526323,
    0.25264252642526425,
    0.25265252652526526,
    0.2526625266252662,
    0.25267252672526724,
    0.25268252682526826,
    0.2526925269252692,
    0.25270252702527024,
    0.25271252712527126,
    0.2527225272252723,
    0.25273252732527324,
    0.25274252742527425,
    0.2527525275252753,
    0.25276252762527623,
    0.25277252772527725,
    0.25278252782527827,
    0.25279252792527923,
    0.25280252802528025,
    0.25281252812528127,
    0.25282252822528223,
    0.25283252832528325,
    0.25284252842528426,
    0.2528525285252852,
    0.25286252862528624,
    0.25287252872528726,
    0.2528825288252882,
    0.25289252892528924,
    0.25290252902529026,
    0.2529125291252913,
    0.25292252922529224,
    0.25293252932529325,
    0.25294252942529427,
    0.25295252952529523,
    0.25296252962529625,
    0.25297252972529727,
    0.25298252982529823,
    0.25299252992529925,
    0.25300253002530027,
    0.2530125301253012,
    0.25302253022530224,
    0.25303253032530326,
    0.2530425304253042,
    0.25305253052530524,
    0.25306253062530626,
    0.2530725307253073,
    0.25308253082530824,
    0.25309253092530926,
    0.2531025310253103,
    0.25311253112531124,
    0.25312253122531225,
    0.25313253132531327,
    0.25314253142531423,
    0.25315253152531525,
    0.25316253162531627,
    0.25317253172531723,
    0.25318253182531825,
    0.25319253192531926,
    0.2532025320253202,
    0.25321253212532124,
    0.25322253222532226,
    0.2532325323253232,
    0.25324253242532424,
    0.25325253252532526,
    0.2532625326253263,
    0.25327253272532724,
    0.25328253282532825,
    0.2532925329253293,
    0.25330253302533023,
    0.25331253312533125,
    0.25332253322533227,
    0.25333253332533323,
    0.25334253342533425,
    0.25335253352533527,
    0.25336253362533623,
    0.25337253372533725,
    0.25338253382533826,
    0.2533925339253392,
    0.25340253402534024,
    0.25341253412534126,
    0.2534225342253422,
    0.25343253432534324,
    0.25344253442534426,
    0.2534525345253453,
    0.25346253462534624,
    0.25347253472534725,
    0.25348253482534827,
    0.25349253492534923,
    0.25350253502535025,
    0.25351253512535127,
    0.25352253522535223,
    0.25353253532535325,
    0.25354253542535427,
    0.2535525355253552,
    0.25356253562535624,
    0.25357253572535726,
    0.2535825358253582,
    0.25359253592535924,
    0.25360253602536026,
    0.2536125361253613,
    0.25362253622536224,
    0.25363253632536326,
    0.2536425364253643,
    0.25365253652536524,
    0.25366253662536625,
    0.25367253672536727,
    0.25368253682536823,
    0.25369253692536925,
    0.25370253702537027,
    0.25371253712537123,
    0.25372253722537225,
    0.25373253732537326,
    0.2537425374253742,
    0.25375253752537524,
    0.25376253762537626,
    0.2537725377253772,
    0.25378253782537824,
    0.25379253792537926,
    0.2538025380253803,
    0.25381253812538124,
    0.25382253822538225,
    0.2538325383253833,
    0.25384253842538423,
    0.25385253852538525,
    0.25386253862538627,
    0.25387253872538723,
    0.25388253882538825,
    0.25389253892538927,
    0.25390253902539023,
    0.25391253912539125,
    0.25392253922539226,
    0.2539325393253932,
    0.25394253942539424,
    0.25395253952539526,
    0.2539625396253962,
    0.25397253972539724,
    0.25398253982539826,
    0.2539925399253993,
    0.25400254002540024,
    0.25401254012540125,
    0.25402254022540227,
    0.25403254032540323,
    0.25404254042540425,
    0.25405254052540527,
    0.25406254062540623,
    0.25407254072540725,
    0.25408254082540827,
    0.2540925409254092,
    0.25410254102541024,
    0.25411254112541126,
    0.2541225412254122,
    0.25413254132541324,
    0.25414254142541426,
    0.2541525415254153,
    0.25416254162541624,
    0.25417254172541726,
    0.2541825418254183,
    0.25419254192541924,
    0.25420254202542025,
    0.25421254212542127,
    0.25422254222542223,
    0.25423254232542325,
    0.25424254242542427,
    0.25425254252542523,
    0.25426254262542625,
    0.25427254272542726,
    0.2542825428254282,
    0.25429254292542924,
    0.25430254302543026,
    0.2543125431254312,
    0.25432254322543224,
    0.25433254332543326,
    0.2543425434254343,
    0.25435254352543524,
    0.25436254362543625,
    0.2543725437254373,
    0.25438254382543823,
    0.25439254392543925,
    0.25440254402544027,
    0.25441254412544123,
    0.25442254422544225,
    0.25443254432544327,
    0.25444254442544423,
    0.25445254452544525,
    0.25446254462544626,
    0.2544725447254472,
    0.25448254482544824,
    0.25449254492544926,
    0.2545025450254503,
    0.25451254512545124,
    0.25452254522545226,
    0.2545325453254533,
    0.25454254542545424,
    0.25455254552545525,
    0.25456254562545627,
    0.25457254572545723,
    0.25458254582545825,
    0.25459254592545927,
    0.25460254602546023,
    0.25461254612546125,
    0.25462254622546227,
    0.2546325463254632,
    0.25464254642546424,
    0.25465254652546526,
    0.2546625466254662,
    0.25467254672546724,
    0.25468254682546826,
    0.2546925469254693,
    0.25470254702547024,
    0.25471254712547126,
    0.2547225472254723,
    0.25473254732547324,
    0.25474254742547425,
    0.25475254752547527,
    0.25476254762547623,
    0.25477254772547725,
    0.25478254782547827,
    0.25479254792547923,
    0.25480254802548025,
    0.25481254812548126,
    0.2548225482254822,
    0.25483254832548324,
    0.25484254842548426,
    0.2548525485254852,
    0.25486254862548624,
    0.25487254872548726,
    0.2548825488254883,
    0.25489254892548924,
    0.25490254902549025,
    0.2549125491254913,
    0.25492254922549223,
    0.25493254932549325,
    0.25494254942549427,
    0.25495254952549523,
    0.25496254962549625,
    0.25497254972549727,
    0.25498254982549823,
    0.25499254992549925,
    0.25500255002550026,
    0.2550125501255012,
    0.25502255022550224,
    0.25503255032550326,
    0.2550425504255043,
    0.25505255052550524,
    0.25506255062550626,
    0.2550725507255073,
    0.25508255082550824,
    0.25509255092550925,
    0.25510255102551027,
    0.25511255112551123,
    0.25512255122551225,
    0.25513255132551327,
    0.25514255142551423,
    0.25515255152551525,
    0.25516255162551627,
    0.2551725517255172,
    0.25518255182551824,
    0.25519255192551926,
    0.2552025520255202,
    0.25521255212552124,
    0.25522255222552226,
    0.2552325523255233,
    0.25524255242552424,
    0.25525255252552526,
    0.2552625526255263,
    0.25527255272552724,
    0.25528255282552825,
    0.25529255292552927,
    0.25530255302553023,
    0.25531255312553125,
    0.25532255322553227,
    0.25533255332553323,
    0.25534255342553425,
    0.25535255352553526,
    0.2553625536255362,
    0.25537255372553724,
    0.25538255382553826,
    0.2553925539255393,
    0.25540255402554024,
    0.25541255412554126,
    0.2554225542255423,
    0.25543255432554324,
    0.25544255442554425,
    0.2554525545255453,
    0.25546255462554623,
    0.25547255472554725,
    0.25548255482554827,
    0.25549255492554923,
    0.25550255502555025,
    0.25551255512555127,
    0.25552255522555223,
    0.25553255532555325,
    0.25554255542555426,
    0.2555525555255552,
    0.25556255562555624,
    0.25557255572555726,
    0.2555825558255583,
    0.25559255592555924,
    0.25560255602556026,
    0.2556125561255613,
    0.25562255622556224,
    0.25563255632556325,
    0.25564255642556427,
    0.25565255652556523,
    0.25566255662556625,
    0.25567255672556727,
    0.25568255682556823,
    0.25569255692556925,
    0.25570255702557027,
    0.2557125571255712,
    0.25572255722557224,
    0.25573255732557326,
    0.2557425574255742,
    0.25575255752557524,
    0.25576255762557626,
    0.2557725577255773,
    0.25578255782557824,
    0.25579255792557926,
    0.2558025580255803,
    0.25581255812558124,
    0.25582255822558225,
    0.25583255832558327,
    0.25584255842558423,
    0.25585255852558525,
    0.25586255862558627,
    0.25587255872558723,
    0.25588255882558825,
    0.25589255892558926,
    0.2559025590255902,
    0.25591255912559124,
    0.25592255922559226,
    0.2559325593255933,
    0.25594255942559424,
    0.25595255952559526,
    0.2559625596255963,
    0.25597255972559724,
    0.25598255982559825,
    0.2559925599255993,
    0.25600256002560023,
    0.25601256012560125,
    0.25602256022560227,
    0.25603256032560323,
    0.25604256042560425,
    0.25605256052560527,
    0.25606256062560623,
    0.25607256072560725,
    0.25608256082560826,
    0.2560925609256092,
    0.25610256102561024,
    0.25611256112561126,
    0.2561225612256123,
    0.25613256132561324,
    0.25614256142561426,
    0.2561525615256153,
    0.25616256162561624,
    0.25617256172561725,
    0.25618256182561827,
    0.25619256192561923,
    0.25620256202562025,
    0.25621256212562127,
    0.25622256222562223,
    0.25623256232562325,
    0.25624256242562427,
    0.2562525625256252,
    0.25626256262562624,
    0.25627256272562726,
    0.2562825628256283,
    0.25629256292562924,
    0.25630256302563026,
    0.2563125631256313,
    0.25632256322563224,
    0.25633256332563326,
    0.2563425634256343,
    0.25635256352563524,
    0.25636256362563625,
    0.25637256372563727,
    0.25638256382563823,
    0.25639256392563925,
    0.25640256402564027,
    0.25641256412564123,
    0.25642256422564225,
    0.25643256432564326,
    0.2564425644256442,
    0.25645256452564524,
    0.25646256462564626,
    0.2564725647256473,
    0.25648256482564824,
    0.25649256492564926,
    0.2565025650256503,
    0.25651256512565124,
    0.25652256522565225,
    0.2565325653256533,
    0.25654256542565423,
    0.25655256552565525,
    0.25656256562565627,
    0.25657256572565723,
    0.25658256582565825,
    0.25659256592565927,
    0.25660256602566023,
    0.25661256612566125,
    0.25662256622566226,
    0.2566325663256632,
    0.25664256642566424,
    0.25665256652566526,
    0.2566625666256663,
    0.25667256672566724,
    0.25668256682566826,
    0.2566925669256693,
    0.25670256702567024,
    0.25671256712567125,
    0.25672256722567227,
    0.25673256732567323,
    0.25674256742567425,
    0.25675256752567527,
    0.25676256762567623,
    0.25677256772567725,
    0.25678256782567827,
    0.2567925679256792,
    0.25680256802568024,
    0.25681256812568126,
    0.2568225682256823,
    0.25683256832568324,
    0.25684256842568426,
    0.2568525685256853,
    0.25686256862568624,
    0.25687256872568726,
    0.2568825688256883,
    0.25689256892568924,
    0.25690256902569025,
    0.25691256912569127,
    0.25692256922569223,
    0.25693256932569325,
    0.25694256942569427,
    0.25695256952569523,
    0.25696256962569625,
    0.25697256972569726,
    0.2569825698256982,
    0.25699256992569924,
    0.25700257002570026,
    0.2570125701257013,
    0.25702257022570224,
    0.25703257032570326,
    0.2570425704257043,
    0.25705257052570524,
    0.25706257062570625,
    0.2570725707257073,
    0.25708257082570823,
    0.25709257092570925,
    0.25710257102571027,
    0.25711257112571123,
    0.25712257122571225,
    0.25713257132571327,
    0.25714257142571423,
    0.25715257152571525,
    0.25716257162571626,
    0.2571725717257173,
    0.25718257182571824,
    0.25719257192571926,
    0.2572025720257203,
    0.25721257212572124,
    0.25722257222572226,
    0.2572325723257233,
    0.25724257242572424,
    0.25725257252572525,
    0.25726257262572627,
    0.25727257272572723,
    0.25728257282572825,
    0.25729257292572927,
    0.25730257302573023,
    0.25731257312573125,
    0.25732257322573227,
    0.2573325733257332,
    0.25734257342573424,
    0.25735257352573526,
    0.2573625736257363,
    0.25737257372573724,
    0.25738257382573826,
    0.2573925739257393,
    0.25740257402574024,
    0.25741257412574126,
    0.2574225742257423,
    0.25743257432574324,
    0.25744257442574425,
    0.25745257452574527,
    0.25746257462574623,
    0.25747257472574725,
    0.25748257482574827,
    0.25749257492574923,
    0.25750257502575025,
    0.25751257512575126,
    0.2575225752257522,
    0.25753257532575324,
    0.25754257542575426,
    0.2575525755257553,
    0.25756257562575624,
    0.25757257572575726,
    0.2575825758257583,
    0.25759257592575924,
    0.25760257602576025,
    0.2576125761257613,
    0.25762257622576223,
    0.25763257632576325,
    0.25764257642576427,
    0.25765257652576523,
    0.25766257662576625,
    0.25767257672576727,
    0.25768257682576823,
    0.25769257692576925,
    0.25770257702577026,
    0.2577125771257713,
    0.25772257722577224,
    0.25773257732577326,
    0.2577425774257743,
    0.25775257752577524,
    0.25776257762577626,
    0.2577725777257773,
    0.25778257782577824,
    0.25779257792577925,
    0.25780257802578027,
    0.25781257812578123,
    0.25782257822578225,
    0.25783257832578327,
    0.25784257842578423,
    0.25785257852578525,
    0.25786257862578627,
    0.2578725787257872,
    0.25788257882578824,
    0.25789257892578926,
    0.2579025790257903,
    0.25791257912579124,
    0.25792257922579226,
    0.2579325793257933,
    0.25794257942579424,
    0.25795257952579526,
    0.2579625796257963,
    0.25797257972579724,
    0.25798257982579825,
    0.25799257992579927,
    0.25800258002580023,
    0.25801258012580125,
    0.25802258022580227,
    0.25803258032580323,
    0.25804258042580425,
    0.25805258052580526,
    0.2580625806258062,
    0.25807258072580724,
    0.25808258082580826,
    0.2580925809258093,
    0.25810258102581024,
    0.25811258112581126,
    0.2581225812258123,
    0.25813258132581324,
    0.25814258142581425,
    0.2581525815258153,
    0.25816258162581623,
    0.25817258172581725,
    0.25818258182581827,
    0.25819258192581923,
    0.25820258202582025,
    0.25821258212582127,
    0.25822258222582223,
    0.25823258232582325,
    0.25824258242582426,
    0.2582525825258253,
    0.25826258262582624,
    0.25827258272582726,
    0.2582825828258283,
    0.25829258292582924,
    0.25830258302583026,
    0.2583125831258313,
    0.25832258322583224,
    0.25833258332583325,
    0.25834258342583427,
    0.25835258352583523,
    0.25836258362583625,
    0.25837258372583727,
    0.25838258382583823,
    0.25839258392583925,
    0.25840258402584027,
    0.2584125841258412,
    0.25842258422584224,
    0.25843258432584326,
    0.2584425844258443,
    0.25845258452584524,
    0.25846258462584626,
    0.2584725847258473,
    0.25848258482584824,
    0.25849258492584926,
    0.2585025850258503,
    0.25851258512585124,
    0.25852258522585225,
    0.25853258532585327,
    0.25854258542585423,
    0.25855258552585525,
    0.25856258562585627,
    0.25857258572585723,
    0.25858258582585825,
    0.25859258592585926,
    0.2586025860258603,
    0.25861258612586124,
    0.25862258622586226,
    0.2586325863258633,
    0.25864258642586424,
    0.25865258652586526,
    0.2586625866258663,
    0.25867258672586724,
    0.25868258682586825,
    0.2586925869258693,
    0.25870258702587023,
    0.25871258712587125,
    0.25872258722587227,
    0.25873258732587323,
    0.25874258742587425,
    0.25875258752587527,
    0.25876258762587623,
    0.25877258772587725,
    0.25878258782587826,
    0.2587925879258793,
    0.25880258802588024,
    0.25881258812588126,
    0.2588225882258823,
    0.25883258832588324,
    0.25884258842588426,
    0.2588525885258853,
    0.25886258862588624,
    0.25887258872588725,
    0.25888258882588827,
    0.25889258892588923,
    0.25890258902589025,
    0.25891258912589127,
    0.25892258922589223,
    0.25893258932589325,
    0.25894258942589427,
    0.2589525895258952,
    0.25896258962589624,
    0.25897258972589726,
    0.2589825898258983,
    0.25899258992589924,
    0.25900259002590026,
    0.2590125901259013,
    0.25902259022590224,
    0.25903259032590326,
    0.2590425904259043,
    0.25905259052590524,
    0.25906259062590625,
    0.25907259072590727,
    0.25908259082590823,
    0.25909259092590925,
    0.25910259102591027,
    0.25911259112591123,
    0.25912259122591225,
    0.25913259132591326,
    0.2591425914259143,
    0.25915259152591524,
    0.25916259162591626,
    0.2591725917259173,
    0.25918259182591824,
    0.25919259192591926,
    0.2592025920259203,
    0.25921259212592124,
    0.25922259222592225,
    0.2592325923259233,
    0.25924259242592423,
    0.25925259252592525,
    0.25926259262592627,
    0.25927259272592723,
    0.25928259282592825,
    0.25929259292592927,
    0.25930259302593023,
    0.25931259312593125,
    0.25932259322593226,
    0.2593325933259333,
    0.25934259342593424,
    0.25935259352593526,
    0.2593625936259363,
    0.25937259372593724,
    0.25938259382593826,
    0.2593925939259393,
    0.25940259402594024,
    0.25941259412594125,
    0.25942259422594227,
    0.25943259432594323,
    0.25944259442594425,
    0.25945259452594527,
    0.25946259462594623,
    0.25947259472594725,
    0.25948259482594827,
    0.2594925949259493,
    0.25950259502595024,
    0.25951259512595126,
    0.2595225952259523,
    0.25953259532595324,
    0.25954259542595426,
    0.2595525955259553,
    0.25956259562595624,
    0.25957259572595726,
    0.2595825958259583,
    0.25959259592595924,
    0.25960259602596025,
    0.25961259612596127,
    0.25962259622596223,
    0.25963259632596325,
    0.25964259642596427,
    0.25965259652596523,
    0.25966259662596625,
    0.25967259672596726,
    0.2596825968259683,
    0.25969259692596924,
    0.25970259702597026,
    0.2597125971259713,
    0.25972259722597224,
    0.25973259732597326,
    0.2597425974259743,
    0.25975259752597524,
    0.25976259762597625,
    0.2597725977259773,
    0.25978259782597823,
    0.25979259792597925,
    0.25980259802598027,
    0.25981259812598123,
    0.25982259822598225,
    0.25983259832598327,
    0.25984259842598423,
    0.25985259852598525,
    0.25986259862598626,
    0.2598725987259873,
    0.25988259882598824,
    0.25989259892598926,
    0.2599025990259903,
    0.25991259912599124,
    0.25992259922599226,
    0.2599325993259933,
    0.25994259942599424,
    0.25995259952599525,
    0.25996259962599627,
    0.25997259972599723,
    0.25998259982599825,
    0.25999259992599927,
    0.26000260002600023,
    0.26001260012600125,
    0.26002260022600227,
    0.2600326003260033,
    0.26004260042600424,
    0.26005260052600526,
    0.2600626006260063,
    0.26007260072600724,
    0.26008260082600826,
    0.2600926009260093,
    0.26010260102601024,
    0.26011260112601126,
    0.2601226012260123,
    0.26013260132601324,
    0.26014260142601425,
    0.26015260152601527,
    0.26016260162601623,
    0.26017260172601725,
    0.26018260182601827,
    0.26019260192601923,
    0.26020260202602025,
    0.26021260212602126,
    0.2602226022260223,
    0.26023260232602324,
    0.26024260242602426,
    0.2602526025260253,
    0.26026260262602624,
    0.26027260272602726,
    0.2602826028260283,
    0.26029260292602924,
    0.26030260302603025,
    0.2603126031260313,
    0.26032260322603223,
    0.26033260332603325,
    0.26034260342603427,
    0.26035260352603523,
    0.26036260362603625,
    0.26037260372603727,
    0.2603826038260383,
    0.26039260392603925,
    0.26040260402604026,
    0.2604126041260413,
    0.26042260422604224,
    0.26043260432604326,
    0.2604426044260443,
    0.26045260452604524,
    0.26046260462604626,
    0.2604726047260473,
    0.26048260482604824,
    0.26049260492604925,
    0.26050260502605027,
    0.26051260512605123,
    0.26052260522605225,
    0.26053260532605327,
    0.26054260542605423,
    0.26055260552605525,
    0.26056260562605627,
    0.2605726057260573,
    0.26058260582605824,
    0.26059260592605926,
    0.2606026060260603,
    0.26061260612606124,
    0.26062260622606226,
    0.2606326063260633,
    0.26064260642606424,
    0.26065260652606526,
    0.2606626066260663,
    0.26067260672606724,
    0.26068260682606825,
    0.26069260692606927,
    0.26070260702607023,
    0.26071260712607125,
    0.26072260722607227,
    0.26073260732607323,
    0.26074260742607425,
    0.26075260752607526,
    0.2607626076260763,
    0.26077260772607724,
    0.26078260782607826,
    0.2607926079260793,
    0.26080260802608024,
    0.26081260812608126,
    0.2608226082260823,
    0.26083260832608324,
    0.26084260842608425,
    0.2608526085260853,
    0.26086260862608623,
    0.26087260872608725,
    0.26088260882608827,
    0.26089260892608923,
    0.26090260902609025,
    0.26091260912609127,
    0.2609226092260923,
    0.26093260932609325,
    0.26094260942609426,
    0.2609526095260953,
    0.26096260962609624,
    0.26097260972609726,
    0.2609826098260983,
    0.26099260992609924,
    0.26100261002610026,
    0.2610126101261013,
    0.26102261022610224,
    0.26103261032610325,
    0.26104261042610427,
    0.26105261052610523,
    0.26106261062610625,
    0.26107261072610727,
    0.26108261082610823,
    0.26109261092610925,
    0.26110261102611027,
    0.2611126111261113,
    0.26112261122611224,
    0.26113261132611326,
    0.2611426114261143,
    0.26115261152611524,
    0.26116261162611626,
    0.2611726117261173,
    0.26118261182611824,
    0.26119261192611926,
    0.2612026120261203,
    0.26121261212612124,
    0.26122261222612225,
    0.26123261232612327,
    0.26124261242612423,
    0.26125261252612525,
    0.26126261262612627,
    0.26127261272612723,
    0.26128261282612825,
    0.26129261292612926,
    0.2613026130261303,
    0.26131261312613124,
    0.26132261322613226,
    0.2613326133261333,
    0.26134261342613424,
    0.26135261352613526,
    0.2613626136261363,
    0.26137261372613724,
    0.26138261382613825,
    0.2613926139261393,
    0.26140261402614023,
    0.26141261412614125,
    0.26142261422614227,
    0.26143261432614323,
    0.26144261442614425,
    0.26145261452614527,
    0.2614626146261463,
    0.26147261472614725,
    0.26148261482614826,
    0.2614926149261493,
    0.26150261502615024,
    0.26151261512615126,
    0.2615226152261523,
    0.26153261532615324,
    0.26154261542615426,
    0.2615526155261553,
    0.26156261562615624,
    0.26157261572615725,
    0.26158261582615827,
    0.26159261592615923,
    0.26160261602616025,
    0.26161261612616127,
    0.26162261622616223,
    0.26163261632616325,
    0.26164261642616427,
    0.2616526165261653,
    0.26166261662616624,
    0.26167261672616726,
    0.2616826168261683,
    0.26169261692616924,
    0.26170261702617026,
    0.2617126171261713,
    0.26172261722617224,
    0.26173261732617326,
    0.2617426174261743,
    0.26175261752617524,
    0.26176261762617625,
    0.26177261772617727,
    0.26178261782617823,
    0.26179261792617925,
    0.26180261802618027,
    0.2618126181261813,
    0.26182261822618225,
    0.26183261832618326,
    0.2618426184261843,
    0.26185261852618524,
    0.26186261862618626,
    0.2618726187261873,
    0.26188261882618824,
    0.26189261892618926,
    0.2619026190261903,
    0.26191261912619124,
    0.26192261922619225,
    0.2619326193261933,
    0.26194261942619423,
    0.26195261952619525,
    0.26196261962619627,
    0.26197261972619723,
    0.26198261982619825,
    0.26199261992619927,
    0.2620026200262003,
    0.26201262012620125,
    0.26202262022620226,
    0.2620326203262033,
    0.26204262042620424,
    0.26205262052620526,
    0.2620626206262063,
    0.26207262072620724,
    0.26208262082620826,
    0.2620926209262093,
    0.26210262102621024,
    0.26211262112621125,
    0.26212262122621227,
    0.26213262132621323,
    0.26214262142621425,
    0.26215262152621527,
    0.26216262162621623,
    0.26217262172621725,
    0.26218262182621827,
    0.2621926219262193,
    0.26220262202622024,
    0.26221262212622126,
    0.2622226222262223,
    0.26223262232622324,
    0.26224262242622426,
    0.2622526225262253,
    0.26226262262622624,
    0.26227262272622726,
    0.2622826228262283,
    0.26229262292622924,
    0.26230262302623025,
    0.26231262312623127,
    0.26232262322623223,
    0.26233262332623325,
    0.26234262342623427,
    0.2623526235262353,
    0.26236262362623625,
    0.26237262372623726,
    0.2623826238262383,
    0.26239262392623924,
    0.26240262402624026,
    0.2624126241262413,
    0.26242262422624224,
    0.26243262432624326,
    0.2624426244262443,
    0.26245262452624524,
    0.26246262462624625,
    0.2624726247262473,
    0.26248262482624823,
    0.26249262492624925,
    0.26250262502625027,
    0.26251262512625123,
    0.26252262522625225,
    0.26253262532625327,
    0.2625426254262543,
    0.26255262552625525,
    0.26256262562625626,
    0.2625726257262573,
    0.26258262582625824,
    0.26259262592625926,
    0.2626026260262603,
    0.26261262612626124,
    0.26262262622626226,
    0.2626326263262633,
    0.26264262642626424,
    0.26265262652626525,
    0.26266262662626627,
    0.26267262672626723,
    0.26268262682626825,
    0.26269262692626927,
    0.2627026270262703,
    0.26271262712627125,
    0.26272262722627227,
    0.2627326273262733,
    0.26274262742627424,
    0.26275262752627526,
    0.2627626276262763,
    0.26277262772627724,
    0.26278262782627826,
    0.2627926279262793,
    0.26280262802628024,
    0.26281262812628126,
    0.2628226282262823,
    0.26283262832628324,
    0.26284262842628425,
    0.26285262852628527,
    0.26286262862628623,
    0.26287262872628725,
    0.26288262882628827,
    0.2628926289262893,
    0.26290262902629025,
    0.26291262912629126,
    0.2629226292262923,
    0.26293262932629324,
    0.26294262942629426,
    0.2629526295262953,
    0.26296262962629624,
    0.26297262972629726,
    0.2629826298262983,
    0.26299262992629924,
    0.26300263002630025,
    0.2630126301263013,
    0.26302263022630223,
    0.26303263032630325,
    0.26304263042630427,
    0.26305263052630523,
    0.26306263062630625,
    0.26307263072630727,
    0.2630826308263083,
    0.26309263092630925,
    0.26310263102631026,
    0.2631126311263113,
    0.26312263122631224,
    0.26313263132631326,
    0.2631426314263143,
    0.26315263152631524,
    0.26316263162631626,
    0.2631726317263173,
    0.26318263182631824,
    0.26319263192631925,
    0.26320263202632027,
    0.26321263212632123,
    0.26322263222632225,
    0.26323263232632327,
    0.2632426324263243,
    0.26325263252632525,
    0.26326263262632627,
    0.2632726327263273,
    0.26328263282632824,
    0.26329263292632926,
    0.2633026330263303,
    0.26331263312633124,
    0.26332263322633226,
    0.2633326333263333,
    0.26334263342633424,
    0.26335263352633526,
    0.2633626336263363,
    0.26337263372633724,
    0.26338263382633825,
    0.26339263392633927,
    0.26340263402634023,
    0.26341263412634125,
    0.26342263422634227,
    0.2634326343263433,
    0.26344263442634425,
    0.26345263452634526,
    0.2634626346263463,
    0.26347263472634724,
    0.26348263482634826,
    0.2634926349263493,
    0.26350263502635024,
    0.26351263512635126,
    0.2635226352263523,
    0.26353263532635324,
    0.26354263542635425,
    0.26355263552635527,
    0.26356263562635623,
    0.26357263572635725,
    0.26358263582635827,
    0.2635926359263593,
    0.26360263602636025,
    0.26361263612636127,
    0.2636226362263623,
    0.26363263632636325,
    0.26364263642636426,
    0.2636526365263653,
    0.26366263662636624,
    0.26367263672636726,
    0.2636826368263683,
    0.26369263692636924,
    0.26370263702637026,
    0.2637126371263713,
    0.26372263722637224,
    0.26373263732637325,
    0.26374263742637427,
    0.26375263752637523,
    0.26376263762637625,
    0.26377263772637727,
    0.2637826378263783,
    0.26379263792637925,
    0.26380263802638027,
    0.2638126381263813,
    0.26382263822638224,
    0.26383263832638326,
    0.2638426384263843,
    0.26385263852638524,
    0.26386263862638626,
    0.2638726387263873,
    0.26388263882638824,
    0.26389263892638926,
    0.2639026390263903,
    0.26391263912639124,
    0.26392263922639225,
    0.26393263932639327,
    0.26394263942639423,
    0.26395263952639525,
    0.26396263962639627,
    0.2639726397263973,
    0.26398263982639825,
    0.26399263992639926,
    0.2640026400264003,
    0.26401264012640124,
    0.26402264022640226,
    0.2640326403264033,
    0.26404264042640424,
    0.26405264052640526,
    0.2640626406264063,
    0.26407264072640724,
    0.26408264082640825,
    0.26409264092640927,
    0.26410264102641023,
    0.26411264112641125,
    0.26412264122641227,
    0.2641326413264133,
    0.26414264142641425,
    0.26415264152641527,
    0.2641626416264163,
    0.26417264172641725,
    0.26418264182641826,
    0.2641926419264193,
    0.26420264202642024,
    0.26421264212642126,
    0.2642226422264223,
    0.26423264232642324,
    0.26424264242642426,
    0.2642526425264253,
    0.26426264262642624,
    0.26427264272642725,
    0.26428264282642827,
    0.26429264292642923,
    0.26430264302643025,
    0.26431264312643127,
    0.2643226432264323,
    0.26433264332643325,
    0.26434264342643427,
    0.2643526435264353,
    0.26436264362643624,
    0.26437264372643726,
    0.2643826438264383,
    0.26439264392643924,
    0.26440264402644026,
    0.2644126441264413,
    0.26442264422644224,
    0.26443264432644326,
    0.2644426444264443,
    0.26445264452644524,
    0.26446264462644625,
    0.26447264472644727,
    0.2644826448264483,
    0.26449264492644925,
    0.26450264502645027,
    0.2645126451264513,
    0.26452264522645225,
    0.26453264532645326,
    0.2645426454264543,
    0.26455264552645524,
    0.26456264562645626,
    0.2645726457264573,
    0.26458264582645824,
    0.26459264592645926,
    0.2646026460264603,
    0.26461264612646124,
    0.26462264622646225,
    0.26463264632646327,
    0.26464264642646423,
    0.26465264652646525,
    0.26466264662646627,
    0.2646726467264673,
    0.26468264682646825,
    0.26469264692646927,
    0.2647026470264703,
    0.26471264712647125,
    0.26472264722647226,
    0.2647326473264733,
    0.26474264742647424,
    0.26475264752647526,
    0.2647626476264763,
    0.26477264772647724,
    0.26478264782647826,
    0.2647926479264793,
    0.26480264802648024,
    0.26481264812648125,
    0.26482264822648227,
    0.26483264832648323,
    0.26484264842648425,
    0.26485264852648527,
    0.2648626486264863,
    0.26487264872648725,
    0.26488264882648827,
    0.2648926489264893,
    0.26490264902649024,
    0.26491264912649126,
    0.2649226492264923,
    0.26493264932649324,
    0.26494264942649426,
    0.2649526495264953,
    0.26496264962649624,
    0.26497264972649726,
    0.2649826498264983,
    0.26499264992649924,
    0.26500265002650025,
    0.26501265012650127,
    0.2650226502265023,
    0.26503265032650325,
    0.26504265042650427,
    0.2650526505265053,
    0.26506265062650625,
    0.26507265072650726,
    0.2650826508265083,
    0.26509265092650924,
    0.26510265102651026,
    0.2651126511265113,
    0.26512265122651224,
    0.26513265132651326,
    0.2651426514265143,
    0.26515265152651524,
    0.26516265162651625,
    0.26517265172651727,
    0.26518265182651823,
    0.26519265192651925,
    0.26520265202652027,
    0.2652126521265213,
    0.26522265222652225,
    0.26523265232652327,
    0.2652426524265243,
    0.26525265252652525,
    0.26526265262652626,
    0.2652726527265273,
    0.26528265282652824,
    0.26529265292652926,
    0.2653026530265303,
    0.26531265312653124,
    0.26532265322653226,
    0.2653326533265333,
    0.26534265342653424,
    0.26535265352653525,
    0.26536265362653627,
    0.26537265372653723,
    0.26538265382653825,
    0.26539265392653927,
    0.2654026540265403,
    0.26541265412654125,
    0.26542265422654227,
    0.2654326543265433,
    0.26544265442654424,
    0.26545265452654526,
    0.2654626546265463,
    0.26547265472654724,
    0.26548265482654826,
    0.2654926549265493,
    0.26550265502655024,
    0.26551265512655126,
    0.2655226552265523,
    0.26553265532655324,
    0.26554265542655425,
    0.26555265552655527,
    0.2655626556265563,
    0.26557265572655725,
    0.26558265582655827,
    0.2655926559265593,
    0.26560265602656025,
    0.26561265612656126,
    0.2656226562265623,
    0.26563265632656324,
    0.26564265642656426,
    0.2656526565265653,
    0.26566265662656624,
    0.26567265672656726,
    0.2656826568265683,
    0.26569265692656924,
    0.26570265702657025,
    0.26571265712657127,
    0.26572265722657223,
    0.26573265732657325,
    0.26574265742657427,
    0.2657526575265753,
    0.26576265762657625,
    0.26577265772657727,
    0.2657826578265783,
    0.26579265792657925,
    0.26580265802658026,
    0.2658126581265813,
    0.26582265822658224,
    0.26583265832658326,
    0.2658426584265843,
    0.26585265852658524,
    0.26586265862658626,
    0.2658726587265873,
    0.26588265882658824,
    0.26589265892658925,
    0.26590265902659027,
    0.2659126591265913,
    0.26592265922659225,
    0.26593265932659327,
    0.2659426594265943,
    0.26595265952659525,
    0.26596265962659627,
    0.2659726597265973,
    0.26598265982659824,
    0.26599265992659926,
    0.2660026600266003,
    0.26601266012660124,
    0.26602266022660226,
    0.2660326603266033,
    0.26604266042660424,
    0.26605266052660526,
    0.2660626606266063,
    0.26607266072660724,
    0.26608266082660825,
    0.26609266092660927,
    0.2661026610266103,
    0.26611266112661125,
    0.26612266122661227,
    0.2661326613266133,
    0.26614266142661425,
    0.26615266152661526,
    0.2661626616266163,
    0.26617266172661724,
    0.26618266182661826,
    0.2661926619266193,
    0.26620266202662024,
    0.26621266212662126,
    0.2662226622266223,
    0.26623266232662324,
    0.26624266242662425,
    0.26625266252662527,
    0.26626266262662623,
    0.26627266272662725,
    0.26628266282662827,
    0.2662926629266293,
    0.26630266302663025,
    0.26631266312663127,
    0.2663226632266323,
    0.26633266332663325,
    0.26634266342663426,
    0.2663526635266353,
    0.26636266362663624,
    0.26637266372663726,
    0.2663826638266383,
    0.26639266392663924,
    0.26640266402664026,
    0.2664126641266413,
    0.26642266422664224,
    0.26643266432664325,
    0.26644266442664427,
    0.2664526645266453,
    0.26646266462664625,
    0.26647266472664727,
    0.2664826648266483,
    0.26649266492664925,
    0.26650266502665027,
    0.2665126651266513,
    0.26652266522665224,
    0.26653266532665326,
    0.2665426654266543,
    0.26655266552665524,
    0.26656266562665626,
    0.2665726657266573,
    0.26658266582665824,
    0.26659266592665926,
    0.2666026660266603,
    0.26661266612666124,
    0.26662266622666225,
    0.26663266632666327,
    0.2666426664266643,
    0.26665266652666525,
    0.26666266662666627,
    0.2666726667266673,
    0.26668266682666825,
    0.26669266692666926,
    0.2667026670266703,
    0.26671266712667124,
    0.26672266722667226,
    0.2667326673266733,
    0.26674266742667424,
    0.26675266752667526,
    0.2667626676266763,
    0.26677266772667724,
    0.26678266782667825,
    0.26679266792667927,
    0.2668026680266803,
    0.26681266812668125,
    0.26682266822668227,
    0.2668326683266833,
    0.26684266842668425,
    0.26685266852668527,
    0.2668626686266863,
    0.26687266872668725,
    0.26688266882668826,
    0.2668926689266893,
    0.26690266902669024,
    0.26691266912669126,
    0.2669226692266923,
    0.26693266932669324,
    0.26694266942669426,
    0.2669526695266953,
    0.26696266962669624,
    0.26697266972669725,
    0.26698266982669827,
    0.2669926699266993,
    0.26700267002670025,
    0.26701267012670127,
    0.2670226702267023,
    0.26703267032670325,
    0.26704267042670427,
    0.2670526705267053,
    0.26706267062670624,
    0.26707267072670726,
    0.2670826708267083,
    0.26709267092670924,
    0.26710267102671026,
    0.2671126711267113,
    0.26712267122671224,
    0.26713267132671326,
    0.2671426714267143,
    0.26715267152671524,
    0.26716267162671625,
    0.26717267172671727,
    0.2671826718267183,
    0.26719267192671925,
    0.26720267202672027,
    0.2672126721267213,
    0.26722267222672225,
    0.26723267232672326,
    0.2672426724267243,
    0.26725267252672524,
    0.26726267262672626,
    0.2672726727267273,
    0.26728267282672824,
    0.26729267292672926,
    0.2673026730267303,
    0.26731267312673124,
    0.26732267322673225,
    0.26733267332673327,
    0.2673426734267343,
    0.26735267352673525,
    0.26736267362673627,
    0.2673726737267373,
    0.26738267382673825,
    0.26739267392673927,
    0.2674026740267403,
    0.26741267412674125,
    0.26742267422674226,
    0.2674326743267433,
    0.26744267442674424,
    0.26745267452674526,
    0.2674626746267463,
    0.26747267472674724,
    0.26748267482674826,
    0.2674926749267493,
    0.26750267502675024,
    0.26751267512675125,
    0.26752267522675227,
    0.2675326753267533,
    0.26754267542675425,
    0.26755267552675527,
    0.2675626756267563,
    0.26757267572675725,
    0.26758267582675827,
    0.2675926759267593,
    0.26760267602676024,
    0.26761267612676126,
    0.2676226762267623,
    0.26763267632676324,
    0.26764267642676426,
    0.2676526765267653,
    0.26766267662676624,
    0.26767267672676726,
    0.2676826768267683,
    0.2676926769267693,
    0.26770267702677025,
    0.26771267712677127,
    0.2677226772267723,
    0.26773267732677325,
    0.26774267742677427,
    0.2677526775267753,
    0.26776267762677625,
    0.26777267772677726,
    0.2677826778267783,
    0.26779267792677924,
    0.26780267802678026,
    0.2678126781267813,
    0.26782267822678224,
    0.26783267832678326,
    0.2678426784267843,
    0.26785267852678524,
    0.26786267862678625,
    0.26787267872678727,
    0.2678826788267883,
    0.26789267892678925,
    0.26790267902679027,
    0.2679126791267913,
    0.26792267922679225,
    0.26793267932679327,
    0.2679426794267943,
    0.26795267952679525,
    0.26796267962679626,
    0.2679726797267973,
    0.26798267982679824,
    0.26799267992679926,
    0.2680026800268003,
    0.26801268012680124,
    0.26802268022680226,
    0.2680326803268033,
    0.26804268042680424,
    0.26805268052680525,
    0.26806268062680627,
    0.2680726807268073,
    0.26808268082680825,
    0.26809268092680927,
    0.2681026810268103,
    0.26811268112681125,
    0.26812268122681227,
    0.2681326813268133,
    0.26814268142681424,
    0.26815268152681526,
    0.2681626816268163,
    0.26817268172681724,
    0.26818268182681826,
    0.2681926819268193,
    0.26820268202682024,
    0.26821268212682126,
    0.2682226822268223,
    0.2682326823268233,
    0.26824268242682425,
    0.26825268252682527,
    0.2682626826268263,
    0.26827268272682725,
    0.26828268282682827,
    0.2682926829268293,
    0.26830268302683025,
    0.26831268312683126,
    0.2683226832268323,
    0.26833268332683324,
    0.26834268342683426,
    0.2683526835268353,
    0.26836268362683624,
    0.26837268372683726,
    0.2683826838268383,
    0.26839268392683924,
    0.26840268402684025,
    0.26841268412684127,
    0.2684226842268423,
    0.26843268432684325,
    0.26844268442684427,
    0.2684526845268453,
    0.26846268462684625,
    0.26847268472684727,
    0.2684826848268483,
    0.26849268492684925,
    0.26850268502685026,
    0.2685126851268513,
    0.26852268522685224,
    0.26853268532685326,
    0.2685426854268543,
    0.26855268552685524,
    0.26856268562685626,
    0.2685726857268573,
    0.2685826858268583,
    0.26859268592685925,
    0.26860268602686027,
    0.2686126861268613,
    0.26862268622686225,
    0.26863268632686327,
    0.2686426864268643,
    0.26865268652686525,
    0.26866268662686627,
    0.2686726867268673,
    0.26868268682686824,
    0.26869268692686926,
    0.2687026870268703,
    0.26871268712687124,
    0.26872268722687226,
    0.2687326873268733,
    0.26874268742687424,
    0.26875268752687526,
    0.2687626876268763,
    0.2687726877268773,
    0.26878268782687825,
    0.26879268792687927,
    0.2688026880268803,
    0.26881268812688125,
    0.26882268822688227,
    0.2688326883268833,
    0.26884268842688425,
    0.26885268852688526,
    0.2688626886268863,
    0.26887268872688724,
    0.26888268882688826,
    0.2688926889268893,
    0.26890268902689024,
    0.26891268912689126,
    0.2689226892268923,
    0.26893268932689324,
    0.26894268942689425,
    0.26895268952689527,
    0.2689626896268963,
    0.26897268972689725,
    0.26898268982689827,
    0.2689926899268993,
    0.26900269002690025,
    0.26901269012690127,
    0.2690226902269023,
    0.26903269032690325,
    0.26904269042690426,
    0.2690526905269053,
    0.26906269062690624,
    0.26907269072690726,
    0.2690826908269083,
    0.26909269092690924,
    0.26910269102691026,
    0.2691126911269113,
    0.2691226912269123,
    0.26913269132691325,
    0.26914269142691427,
    0.2691526915269153,
    0.26916269162691625,
    0.26917269172691727,
    0.2691826918269183,
    0.26919269192691925,
    0.26920269202692027,
    0.2692126921269213,
    0.26922269222692224,
    0.26923269232692326,
    0.2692426924269243,
    0.26925269252692524,
    0.26926269262692626,
    0.2692726927269273,
    0.26928269282692824,
    0.26929269292692926,
    0.2693026930269303,
    0.2693126931269313,
    0.26932269322693225,
    0.26933269332693327,
    0.2693426934269343,
    0.26935269352693525,
    0.26936269362693627,
    0.2693726937269373,
    0.26938269382693825,
    0.26939269392693926,
    0.2694026940269403,
    0.26941269412694124,
    0.26942269422694226,
    0.2694326943269433,
    0.26944269442694424,
    0.26945269452694526,
    0.2694626946269463,
    0.26947269472694724,
    0.26948269482694825,
    0.26949269492694927,
    0.2695026950269503,
    0.26951269512695125,
    0.26952269522695227,
    0.2695326953269533,
    0.26954269542695425,
    0.26955269552695527,
    0.2695626956269563,
    0.26957269572695725,
    0.26958269582695826,
    0.2695926959269593,
    0.26960269602696024,
    0.26961269612696126,
    0.2696226962269623,
    0.26963269632696324,
    0.26964269642696426,
    0.2696526965269653,
    0.2696626966269663,
    0.26967269672696725,
    0.26968269682696827,
    0.2696926969269693,
    0.26970269702697025,
    0.26971269712697127,
    0.2697226972269723,
    0.26973269732697325,
    0.26974269742697426,
    0.2697526975269753,
    0.26976269762697624,
    0.26977269772697726,
    0.2697826978269783,
    0.26979269792697924,
    0.26980269802698026,
    0.2698126981269813,
    0.26982269822698224,
    0.26983269832698326,
    0.2698426984269843,
    0.2698526985269853,
    0.26986269862698625,
    0.26987269872698727,
    0.2698826988269883,
    0.26989269892698925,
    0.26990269902699027,
    0.2699126991269913,
    0.26992269922699225,
    0.26993269932699326,
    0.2699426994269943,
    0.26995269952699524,
    0.26996269962699626,
    0.2699726997269973,
    0.26998269982699824,
    0.26999269992699926,
    0.2700027000270003,
    0.2700127001270013,
    0.27002270022700225,
    0.27003270032700327,
    0.2700427004270043,
    0.27005270052700525,
    0.27006270062700627,
    0.2700727007270073,
    0.27008270082700825,
    0.27009270092700927,
    0.2701027010270103,
    0.27011270112701125,
    0.27012270122701226,
    0.2701327013270133,
    0.27014270142701424,
    0.27015270152701526,
    0.2701627016270163,
    0.27017270172701724,
    0.27018270182701826,
    0.2701927019270193,
    0.2702027020270203,
    0.27021270212702125,
    0.27022270222702227,
    0.2702327023270233,
    0.27024270242702425,
    0.27025270252702527,
    0.2702627026270263,
    0.27027270272702725,
    0.27028270282702826,
    0.2702927029270293,
    0.27030270302703024,
    0.27031270312703126,
    0.2703227032270323,
    0.27033270332703324,
    0.27034270342703426,
    0.2703527035270353,
    0.27036270362703624,
    0.27037270372703726,
    0.2703827038270383,
    0.2703927039270393,
    0.27040270402704025,
    0.27041270412704127,
    0.2704227042270423,
    0.27043270432704325,
    0.27044270442704427,
    0.2704527045270453,
    0.27046270462704625,
    0.27047270472704726,
    0.2704827048270483,
    0.27049270492704924,
    0.27050270502705026,
    0.2705127051270513,
    0.27052270522705224,
    0.27053270532705326,
    0.2705427054270543,
    0.2705527055270553,
    0.27056270562705625,
    0.27057270572705727,
    0.2705827058270583,
    0.27059270592705925,
    0.27060270602706027,
    0.2706127061270613,
    0.27062270622706225,
    0.27063270632706327,
    0.2706427064270643,
    0.27065270652706525,
    0.27066270662706626,
    0.2706727067270673,
    0.27068270682706824,
    0.27069270692706926,
    0.2707027070270703,
    0.27071270712707124,
    0.27072270722707226,
    0.2707327073270733,
    0.2707427074270743,
    0.27075270752707525,
    0.27076270762707627,
    0.2707727077270773,
    0.27078270782707825,
    0.27079270792707927,
    0.2708027080270803,
    0.27081270812708125,
    0.27082270822708226,
    0.2708327083270833,
    0.27084270842708424,
    0.27085270852708526,
    0.2708627086270863,
    0.27087270872708724,
    0.27088270882708826,
    0.2708927089270893,
    0.2709027090270903,
    0.27091270912709126,
    0.2709227092270923,
    0.2709327093270933,
    0.27094270942709425,
    0.27095270952709527,
    0.2709627096270963,
    0.27097270972709725,
    0.27098270982709827,
    0.2709927099270993,
    0.27100271002710025,
    0.27101271012710126,
    0.2710227102271023,
    0.27103271032710324,
    0.27104271042710426,
    0.2710527105271053,
    0.27106271062710624,
    0.27107271072710726,
    0.2710827108271083,
    0.2710927109271093,
    0.27110271102711025,
    0.27111271112711127,
    0.2711227112271123,
    0.27113271132711325,
    0.27114271142711427,
    0.2711527115271153,
    0.27116271162711625,
    0.27117271172711727,
    0.2711827118271183,
    0.27119271192711925,
    0.27120271202712026,
    0.2712127121271213,
    0.27122271222712224,
    0.27123271232712326,
    0.2712427124271243,
    0.27125271252712524,
    0.27126271262712626,
    0.2712727127271273,
    0.2712827128271283,
    0.27129271292712925,
    0.27130271302713027,
    0.2713127131271313,
    0.27132271322713225,
    0.27133271332713327,
    0.2713427134271343,
    0.27135271352713525,
    0.27136271362713626,
    0.2713727137271373,
    0.27138271382713824,
    0.27139271392713926,
    0.2714027140271403,
    0.27141271412714124,
    0.27142271422714226,
    0.2714327143271433,
    0.2714427144271443,
    0.27145271452714526,
    0.2714627146271463,
    0.2714727147271473,
    0.27148271482714825,
    0.27149271492714927,
    0.2715027150271503,
    0.27151271512715125,
    0.27152271522715227,
    0.2715327153271533,
    0.27154271542715425,
    0.27155271552715526,
    0.2715627156271563,
    0.27157271572715724,
    0.27158271582715826,
    0.2715927159271593,
    0.27160271602716024,
    0.27161271612716126,
    0.2716227162271623,
    0.2716327163271633,
    0.27164271642716425,
    0.27165271652716527,
    0.2716627166271663,
    0.27167271672716725,
    0.27168271682716827,
    0.2716927169271693,
    0.27170271702717025,
    0.27171271712717127,
    0.2717227172271723,
    0.27173271732717325,
    0.27174271742717426,
    0.2717527175271753,
    0.27176271762717624,
    0.27177271772717726,
    0.2717827178271783,
    0.2717927179271793,
    0.27180271802718026,
    0.2718127181271813,
    0.2718227182271823,
    0.27183271832718325,
    0.27184271842718427,
    0.2718527185271853,
    0.27186271862718625,
    0.27187271872718727,
    0.2718827188271883,
    0.27189271892718925,
    0.27190271902719026,
    0.2719127191271913,
    0.27192271922719224,
    0.27193271932719326,
    0.2719427194271943,
    0.27195271952719524,
    0.27196271962719626,
    0.2719727197271973,
    0.2719827198271983,
    0.27199271992719926,
    0.2720027200272003,
    0.2720127201272013,
    0.27202272022720225,
    0.27203272032720327,
    0.2720427204272043,
    0.27205272052720525,
    0.27206272062720627,
    0.2720727207272073,
    0.27208272082720825,
    0.27209272092720926,
    0.2721027210272103,
    0.27211272112721124,
    0.27212272122721226,
    0.2721327213272133,
    0.27214272142721424,
    0.27215272152721526,
    0.2721627216272163,
    0.2721727217272173,
    0.27218272182721825,
    0.27219272192721927,
    0.2722027220272203,
    0.27221272212722125,
    0.27222272222722227,
    0.2722327223272233,
    0.27224272242722425,
    0.27225272252722527,
    0.2722627226272263,
    0.27227272272722725,
    0.27228272282722826,
    0.2722927229272293,
    0.27230272302723024,
    0.27231272312723126,
    0.2723227232272323,
    0.2723327233272333,
    0.27234272342723426,
    0.2723527235272353,
    0.2723627236272363,
    0.27237272372723725,
    0.27238272382723827,
    0.2723927239272393,
    0.27240272402724025,
    0.27241272412724127,
    0.2724227242272423,
    0.27243272432724325,
    0.27244272442724426,
    0.2724527245272453,
    0.27246272462724624,
    0.27247272472724726,
    0.2724827248272483,
    0.27249272492724924,
    0.27250272502725026,
    0.2725127251272513,
    0.2725227252272523,
    0.27253272532725326,
    0.2725427254272543,
    0.2725527255272553,
    0.27256272562725625,
    0.27257272572725727,
    0.2725827258272583,
    0.27259272592725925,
    0.27260272602726027,
    0.2726127261272613,
    0.27262272622726225,
    0.27263272632726326,
    0.2726427264272643,
    0.27265272652726524,
    0.27266272662726626,
    0.2726727267272673,
    0.27268272682726824,
    0.27269272692726926,
    0.2727027270272703,
    0.2727127271272713,
    0.27272272722727225,
    0.27273272732727327,
    0.2727427274272743,
    0.27275272752727525,
    0.27276272762727627,
    0.2727727277272773,
    0.27278272782727825,
    0.27279272792727927,
    0.2728027280272803,
    0.27281272812728125,
    0.27282272822728226,
    0.2728327283272833,
    0.27284272842728424,
    0.27285272852728526,
    0.2728627286272863,
    0.2728727287272873,
    0.27288272882728826,
    0.2728927289272893,
    0.2729027290272903,
    0.27291272912729125,
    0.27292272922729227,
    0.2729327293272933,
    0.27294272942729425,
    0.27295272952729527,
    0.2729627296272963,
    0.27297272972729725,
    0.27298272982729826,
    0.2729927299272993,
    0.27300273002730024,
    0.27301273012730126,
    0.2730227302273023,
    0.27303273032730324,
    0.27304273042730426,
    0.2730527305273053,
    0.2730627306273063,
    0.27307273072730726,
    0.2730827308273083,
    0.2730927309273093,
    0.27310273102731025,
    0.27311273112731127,
    0.2731227312273123,
    0.27313273132731325,
    0.27314273142731427,
    0.2731527315273153,
    0.27316273162731625,
    0.27317273172731726,
    0.2731827318273183,
    0.27319273192731924,
    0.27320273202732026,
    0.2732127321273213,
    0.2732227322273223,
    0.27323273232732326,
    0.2732427324273243,
    0.2732527325273253,
    0.27326273262732625,
    0.27327273272732727,
    0.2732827328273283,
    0.27329273292732925,
    0.27330273302733027,
    0.2733127331273313,
    0.27332273322733225,
    0.27333273332733327,
    0.2733427334273343,
    0.27335273352733525,
    0.27336273362733626,
    0.2733727337273373,
    0.27338273382733824,
    0.27339273392733926,
    0.2734027340273403,
    0.2734127341273413,
    0.27342273422734226,
    0.2734327343273433,
    0.2734427344273443,
    0.27345273452734525,
    0.27346273462734627,
    0.2734727347273473,
    0.27348273482734825,
    0.27349273492734927,
    0.2735027350273503,
    0.27351273512735125,
    0.27352273522735226,
    0.2735327353273533,
    0.27354273542735424,
    0.27355273552735526,
    0.2735627356273563,
    0.27357273572735724,
    0.27358273582735826,
    0.2735927359273593,
    0.2736027360273603,
    0.27361273612736126,
    0.2736227362273623,
    0.2736327363273633,
    0.27364273642736425,
    0.27365273652736527,
    0.2736627366273663,
    0.27367273672736725,
    0.27368273682736827,
    0.2736927369273693,
    0.27370273702737025,
    0.27371273712737126,
    0.2737227372273723,
    0.27373273732737324,
    0.27374273742737426,
    0.2737527375273753,
    0.2737627376273763,
    0.27377273772737726,
    0.2737827378273783,
    0.2737927379273793,
    0.27380273802738025,
    0.27381273812738127,
    0.2738227382273823,
    0.27383273832738325,
    0.27384273842738427,
    0.2738527385273853,
    0.27386273862738625,
    0.27387273872738727,
    0.2738827388273883,
    0.27389273892738925,
    0.27390273902739026,
    0.2739127391273913,
    0.27392273922739224,
    0.27393273932739326,
    0.2739427394273943,
    0.2739527395273953,
    0.27396273962739626,
    0.2739727397273973,
    0.2739827398273983,
    0.27399273992739925,
    0.27400274002740027,
    0.2740127401274013,
    0.27402274022740225,
    0.27403274032740327,
    0.2740427404274043,
    0.27405274052740525,
    0.27406274062740626,
    0.2740727407274073,
    0.27408274082740824,
    0.27409274092740926,
    0.2741027410274103,
    0.2741127411274113,
    0.27412274122741226,
    0.2741327413274133,
    0.2741427414274143,
    0.27415274152741526,
    0.2741627416274163,
    0.2741727417274173,
    0.27418274182741825,
    0.27419274192741927,
    0.2742027420274203,
    0.27421274212742125,
    0.27422274222742227,
    0.2742327423274233,
    0.27424274242742425,
    0.27425274252742526,
    0.2742627426274263,
    0.27427274272742724,
    0.27428274282742826,
    0.2742927429274293,
    0.2743027430274303,
    0.27431274312743126,
    0.2743227432274323,
    0.2743327433274333,
    0.27434274342743425,
    0.27435274352743527,
    0.2743627436274363,
    0.27437274372743725,
    0.27438274382743827,
    0.2743927439274393,
    0.27440274402744025,
    0.27441274412744127,
    0.2744227442274423,
    0.27443274432744325,
    0.27444274442744426,
    0.2744527445274453,
    0.27446274462744624,
    0.27447274472744726,
    0.2744827448274483,
    0.2744927449274493,
    0.27450274502745026,
    0.2745127451274513,
    0.2745227452274523,
    0.27453274532745325,
    0.27454274542745427,
    0.2745527455274553,
    0.27456274562745625,
    0.27457274572745727,
    0.2745827458274583,
    0.27459274592745925,
    0.27460274602746026,
    0.2746127461274613,
    0.27462274622746224,
    0.27463274632746326,
    0.2746427464274643,
    0.2746527465274653,
    0.27466274662746626,
    0.2746727467274673,
    0.2746827468274683,
    0.27469274692746926,
    0.2747027470274703,
    0.2747127471274713,
    0.27472274722747225,
    0.27473274732747327,
    0.2747427474274743,
    0.27475274752747525,
    0.27476274762747627,
    0.2747727477274773,
    0.27478274782747825,
    0.27479274792747926,
    0.2748027480274803,
    0.27481274812748124,
    0.27482274822748226,
    0.2748327483274833,
    0.2748427484274843,
    0.27485274852748526,
    0.2748627486274863,
    0.2748727487274873,
    0.27488274882748825,
    0.27489274892748927,
    0.2749027490274903,
    0.27491274912749125,
    0.27492274922749227,
    0.2749327493274933,
    0.27494274942749425,
    0.27495274952749527,
    0.2749627496274963,
    0.27497274972749725,
    0.27498274982749826,
    0.2749927499274993,
    0.2750027500275003,
    0.27501275012750126,
    0.2750227502275023,
    0.2750327503275033,
    0.27504275042750426,
    0.2750527505275053,
    0.2750627506275063,
    0.27507275072750725,
    0.27508275082750827,
    0.2750927509275093,
    0.27510275102751025,
    0.27511275112751127,
    0.2751227512275123,
    0.27513275132751325,
    0.27514275142751426,
    0.2751527515275153,
    0.27516275162751624,
    0.27517275172751726,
    0.2751827518275183,
    0.2751927519275193,
    0.27520275202752026,
    0.2752127521275213,
    0.2752227522275223,
    0.27523275232752326,
    0.2752427524275243,
    0.2752527525275253,
    0.27526275262752625,
    0.27527275272752727,
    0.2752827528275283,
    0.27529275292752925,
    0.27530275302753027,
    0.2753127531275313,
    0.27532275322753225,
    0.27533275332753326,
    0.2753427534275343,
    0.27535275352753524,
    0.27536275362753626,
    0.2753727537275373,
    0.2753827538275383,
    0.27539275392753926,
    0.2754027540275403,
    0.2754127541275413,
    0.27542275422754225,
    0.27543275432754327,
    0.2754427544275443,
    0.27545275452754525,
    0.27546275462754627,
    0.2754727547275473,
    0.27548275482754825,
    0.27549275492754927,
    0.2755027550275503,
    0.27551275512755125,
    0.27552275522755226,
    0.2755327553275533,
    0.2755427554275543,
    0.27555275552755526,
    0.2755627556275563,
    0.2755727557275573,
    0.27558275582755826,
    0.2755927559275593,
    0.2756027560275603,
    0.27561275612756125,
    0.27562275622756227,
    0.2756327563275633,
    0.27564275642756425,
    0.27565275652756527,
    0.2756627566275663,
    0.27567275672756725,
    0.27568275682756826,
    0.2756927569275693,
    0.27570275702757024,
    0.27571275712757126,
    0.2757227572275723,
    0.2757327573275733,
    0.27574275742757426,
    0.2757527575275753,
    0.2757627576275763,
    0.27577275772757726,
    0.2757827578275783,
    0.2757927579275793,
    0.27580275802758025,
    0.27581275812758127,
    0.2758227582275823,
    0.27583275832758325,
    0.27584275842758427,
    0.2758527585275853,
    0.27586275862758625,
    0.27587275872758726,
    0.2758827588275883,
    0.2758927589275893,
    0.27590275902759026,
    0.2759127591275913,
    0.2759227592275923,
    0.27593275932759326,
    0.2759427594275943,
    0.2759527595275953,
    0.27596275962759625,
    0.27597275972759727,
    0.2759827598275983,
    0.27599275992759925,
    0.27600276002760027,
    0.2760127601276013,
    0.27602276022760225,
    0.27603276032760327,
    0.2760427604276043,
    0.27605276052760525,
    0.27606276062760626,
    0.2760727607276073,
    0.2760827608276083,
    0.27609276092760926,
    0.2761027610276103,
    0.2761127611276113,
    0.27612276122761226,
    0.2761327613276133,
    0.2761427614276143,
    0.27615276152761525,
    0.27616276162761627,
    0.2761727617276173,
    0.27618276182761825,
    0.27619276192761927,
    0.2762027620276203,
    0.27621276212762125,
    0.27622276222762226,
    0.2762327623276233,
    0.27624276242762424,
    0.27625276252762526,
    0.2762627626276263,
    0.2762727627276273,
    0.27628276282762826,
    0.2762927629276293,
    0.2763027630276303,
    0.27631276312763126,
    0.2763227632276323,
    0.2763327633276333,
    0.27634276342763425,
    0.27635276352763527,
    0.2763627636276363,
    0.27637276372763725,
    0.27638276382763827,
    0.2763927639276393,
    0.27640276402764025,
    0.27641276412764126,
    0.2764227642276423,
    0.2764327643276433,
    0.27644276442764426,
    0.2764527645276453,
    0.2764627646276463,
    0.27647276472764726,
    0.2764827648276483,
    0.2764927649276493,
    0.27650276502765025,
    0.27651276512765127,
    0.2765227652276523,
    0.27653276532765325,
    0.27654276542765427,
    0.2765527655276553,
    0.27656276562765625,
    0.27657276572765727,
    0.2765827658276583,
    0.27659276592765925,
    0.27660276602766026,
    0.2766127661276613,
    0.2766227662276623,
    0.27663276632766326,
    0.2766427664276643,
    0.2766527665276653,
    0.27666276662766626,
    0.2766727667276673,
    0.2766827668276683,
    0.27669276692766925,
    0.27670276702767027,
    0.2767127671276713,
    0.27672276722767225,
    0.27673276732767327,
    0.2767427674276743,
    0.27675276752767525,
    0.27676276762767626,
    0.2767727677276773,
    0.27678276782767824,
    0.27679276792767926,
    0.2768027680276803,
    0.2768127681276813,
    0.27682276822768226,
    0.2768327683276833,
    0.2768427684276843,
    0.27685276852768526,
    0.2768627686276863,
    0.2768727687276873,
    0.27688276882768825,
    0.27689276892768927,
    0.2769027690276903,
    0.27691276912769125,
    0.27692276922769227,
    0.2769327693276933,
    0.27694276942769425,
    0.27695276952769526,
    0.2769627696276963,
    0.2769727697276973,
    0.27698276982769826,
    0.2769927699276993,
    0.2770027700277003,
    0.27701277012770126,
    0.2770227702277023,
    0.2770327703277033,
    0.27704277042770425,
    0.27705277052770527,
    0.2770627706277063,
    0.27707277072770725,
    0.27708277082770827,
    0.2770927709277093,
    0.27710277102771025,
    0.27711277112771127,
    0.2771227712277123,
    0.27713277132771325,
    0.27714277142771426,
    0.2771527715277153,
    0.2771627716277163,
    0.27717277172771726,
    0.2771827718277183,
    0.2771927719277193,
    0.27720277202772026,
    0.2772127721277213,
    0.2772227722277223,
    0.27723277232772325,
    0.27724277242772427,
    0.2772527725277253,
    0.27726277262772625,
    0.27727277272772727,
    0.2772827728277283,
    0.27729277292772925,
    0.27730277302773026,
    0.2773127731277313,
    0.2773227732277323,
    0.27733277332773326,
    0.2773427734277343,
    0.2773527735277353,
    0.27736277362773626,
    0.2773727737277373,
    0.2773827738277383,
    0.27739277392773926,
    0.2774027740277403,
    0.2774127741277413,
    0.27742277422774225,
    0.27743277432774327,
    0.2774427744277443,
    0.27745277452774525,
    0.27746277462774627,
    0.2774727747277473,
    0.27748277482774825,
    0.27749277492774926,
    0.2775027750277503,
    0.2775127751277513,
    0.27752277522775226,
    0.2775327753277533,
    0.2775427754277543,
    0.27755277552775526,
    0.2775627756277563,
    0.2775727757277573,
    0.27758277582775825,
    0.27759277592775927,
    0.2776027760277603,
    0.27761277612776125,
    0.27762277622776227,
    0.2776327763277633,
    0.27764277642776425,
    0.27765277652776527,
    0.2776627766277663,
    0.27767277672776725,
    0.27768277682776826,
    0.2776927769277693,
    0.2777027770277703,
    0.27771277712777126,
    0.2777227772277723,
    0.2777327773277733,
    0.27774277742777426,
    0.2777527775277753,
    0.2777627776277763,
    0.27777277772777725,
    0.27778277782777827,
    0.2777927779277793,
    0.27780277802778025,
    0.27781277812778127,
    0.2778227782277823,
    0.27783277832778325,
    0.27784277842778426,
    0.2778527785277853,
    0.2778627786277863,
    0.27787277872778726,
    0.2778827788277883,
    0.2778927789277893,
    0.27790277902779026,
    0.2779127791277913,
    0.2779227792277923,
    0.27793277932779326,
    0.2779427794277943,
    0.2779527795277953,
    0.27796277962779625,
    0.27797277972779727,
    0.2779827798277983,
    0.27799277992779925,
    0.27800278002780027,
    0.2780127801278013,
    0.27802278022780225,
    0.27803278032780326,
    0.2780427804278043,
    0.2780527805278053,
    0.27806278062780626,
    0.2780727807278073,
    0.2780827808278083,
    0.27809278092780926,
    0.2781027810278103,
    0.2781127811278113,
    0.27812278122781225,
    0.27813278132781327,
    0.2781427814278143,
    0.27815278152781525,
    0.27816278162781627,
    0.2781727817278173,
    0.27818278182781825,
    0.27819278192781927,
    0.2782027820278203,
    0.2782127821278213,
    0.27822278222782226,
    0.2782327823278233,
    0.2782427824278243,
    0.27825278252782526,
    0.2782627826278263,
    0.2782727827278273,
    0.27828278282782826,
    0.2782927829278293,
    0.2783027830278303,
    0.27831278312783125,
    0.27832278322783227,
    0.2783327833278333,
    0.27834278342783425,
    0.27835278352783527,
    0.2783627836278363,
    0.27837278372783725,
    0.27838278382783826,
    0.2783927839278393,
    0.2784027840278403,
    0.27841278412784126,
    0.2784227842278423,
    0.2784327843278433,
    0.27844278442784426,
    0.2784527845278453,
    0.2784627846278463,
    0.27847278472784726,
    0.2784827848278483,
    0.2784927849278493,
    0.27850278502785025,
    0.27851278512785127,
    0.2785227852278523,
    0.27853278532785325,
    0.27854278542785427,
    0.2785527855278553,
    0.27856278562785625,
    0.27857278572785726,
    0.2785827858278583,
    0.2785927859278593,
    0.27860278602786026,
    0.2786127861278613,
    0.2786227862278623,
    0.27863278632786326,
    0.2786427864278643,
    0.2786527865278653,
    0.27866278662786625,
    0.27867278672786727,
    0.2786827868278683,
    0.27869278692786925,
    0.27870278702787027,
    0.2787127871278713,
    0.27872278722787225,
    0.27873278732787327,
    0.2787427874278743,
    0.2787527875278753,
    0.27876278762787626,
    0.2787727877278773,
    0.2787827878278783,
    0.27879278792787926,
    0.2788027880278803,
    0.2788127881278813,
    0.27882278822788226,
    0.2788327883278833,
    0.2788427884278843,
    0.27885278852788525,
    0.27886278862788627,
    0.2788727887278873,
    0.27888278882788825,
    0.27889278892788927,
    0.2789027890278903,
    0.27891278912789125,
    0.27892278922789226,
    0.2789327893278933,
    0.2789427894278943,
    0.27895278952789526,
    0.2789627896278963,
    0.2789727897278973,
    0.27898278982789826,
    0.2789927899278993,
    0.2790027900279003,
    0.27901279012790126,
    0.2790227902279023,
    0.2790327903279033,
    0.27904279042790425,
    0.27905279052790527,
    0.2790627906279063,
    0.27907279072790725,
    0.27908279082790827,
    0.2790927909279093,
    0.2791027910279103,
    0.27911279112791126,
    0.2791227912279123,
    0.2791327913279133,
    0.27914279142791426,
    0.2791527915279153,
    0.2791627916279163,
    0.27917279172791726,
    0.2791827918279183,
    0.2791927919279193,
    0.27920279202792025,
    0.27921279212792127,
    0.2792227922279223,
    0.27923279232792325,
    0.27924279242792427,
    0.2792527925279253,
    0.27926279262792625,
    0.27927279272792727,
    0.2792827928279283,
    0.2792927929279293,
    0.27930279302793026,
    0.2793127931279313,
    0.2793227932279323,
    0.27933279332793326,
    0.2793427934279343,
    0.2793527935279353,
    0.27936279362793626,
    0.2793727937279373,
    0.2793827938279383,
    0.27939279392793925,
    0.27940279402794027,
    0.2794127941279413,
    0.27942279422794225,
    0.27943279432794327,
    0.2794427944279443,
    0.27945279452794525,
    0.27946279462794626,
    0.2794727947279473,
    0.2794827948279483,
    0.27949279492794926,
    0.2795027950279503,
    0.2795127951279513,
    0.27952279522795226,
    0.2795327953279533,
    0.2795427954279543,
    0.27955279552795526,
    0.2795627956279563,
    0.2795727957279573,
    0.27958279582795825,
    0.27959279592795927,
    0.2796027960279603,
    0.27961279612796125,
    0.27962279622796227,
    0.2796327963279633,
    0.2796427964279643,
    0.27965279652796526,
    0.2796627966279663,
    0.2796727967279673,
    0.27968279682796826,
    0.2796927969279693,
    0.2797027970279703,
    0.27971279712797126,
    0.2797227972279723,
    0.2797327973279733,
    0.27974279742797425,
    0.27975279752797527,
    0.2797627976279763,
    0.27977279772797725,
    0.27978279782797827,
    0.2797927979279793,
    0.27980279802798025,
    0.27981279812798127,
    0.2798227982279823,
    0.2798327983279833,
    0.27984279842798426,
    0.2798527985279853,
    0.2798627986279863,
    0.27987279872798726,
    0.2798827988279883,
    0.2798927989279893,
    0.27990279902799026,
    0.2799127991279913,
    0.2799227992279923,
    0.27993279932799325,
    0.27994279942799427,
    0.2799527995279953,
    0.27996279962799625,
    0.27997279972799727,
    0.2799827998279983,
    0.2799927999279993,
    0.28000280002800026,
    0.2800128001280013,
    0.2800228002280023,
    0.28003280032800326,
    0.2800428004280043,
    0.2800528005280053,
    0.28006280062800626,
    0.2800728007280073,
    0.2800828008280083,
    0.28009280092800926,
    0.2801028010280103,
    0.2801128011280113,
    0.28012280122801225,
    0.28013280132801327,
    0.2801428014280143,
    0.28015280152801525,
    0.28016280162801627,
    0.2801728017280173,
    0.2801828018280183,
    0.28019280192801926,
    0.2802028020280203,
    0.2802128021280213,
    0.28022280222802226,
    0.2802328023280233,
    0.2802428024280243,
    0.28025280252802526,
    0.2802628026280263,
    0.2802728027280273,
    0.28028280282802825,
    0.28029280292802927,
    0.2803028030280303,
    0.28031280312803125,
    0.28032280322803227,
    0.2803328033280333,
    0.28034280342803425,
    0.28035280352803527,
    0.2803628036280363,
    0.2803728037280373,
    0.28038280382803826,
    0.2803928039280393,
    0.2804028040280403,
    0.28041280412804126,
    0.2804228042280423,
    0.2804328043280433,
    0.28044280442804426,
    0.2804528045280453,
    0.2804628046280463,
    0.28047280472804725,
    0.28048280482804827,
    0.2804928049280493,
    0.28050280502805025,
    0.28051280512805127,
    0.2805228052280523,
    0.2805328053280533,
    0.28054280542805426,
    0.2805528055280553,
    0.2805628056280563,
    0.28057280572805726,
    0.2805828058280583,
    0.2805928059280593,
    0.28060280602806026,
    0.2806128061280613,
    0.2806228062280623,
    0.28063280632806326,
    0.2806428064280643,
    0.2806528065280653,
    0.28066280662806625,
    0.28067280672806727,
    0.2806828068280683,
    0.28069280692806925,
    0.28070280702807027,
    0.2807128071280713,
    0.2807228072280723,
    0.28073280732807326,
    0.2807428074280743,
    0.2807528075280753,
    0.28076280762807626,
    0.2807728077280773,
    0.2807828078280783,
    0.28079280792807926,
    0.2808028080280803,
    0.2808128081280813,
    0.28082280822808225,
    0.28083280832808327,
    0.2808428084280843,
    0.28085280852808525,
    0.28086280862808627,
    0.2808728087280873,
    0.28088280882808825,
    0.28089280892808927,
    0.2809028090280903,
    0.2809128091280913,
    0.28092280922809226,
    0.2809328093280933,
    0.2809428094280943,
    0.28095280952809526,
    0.2809628096280963,
    0.2809728097280973,
    0.28098280982809826,
    0.2809928099280993,
    0.2810028100281003,
    0.28101281012810125,
    0.28102281022810227,
    0.2810328103281033,
    0.28104281042810425,
    0.28105281052810527,
    0.2810628106281063,
    0.2810728107281073,
    0.28108281082810826,
    0.2810928109281093,
    0.2811028110281103,
    0.28111281112811126,
    0.2811228112281123,
    0.2811328113281133,
    0.28114281142811426,
    0.2811528115281153,
    0.2811628116281163,
    0.28117281172811726,
    0.2811828118281183,
    0.2811928119281193,
    0.28120281202812025,
    0.28121281212812127,
    0.2812228122281223,
    0.28123281232812325,
    0.28124281242812427,
    0.2812528125281253,
    0.2812628126281263,
    0.28127281272812726,
    0.2812828128281283,
    0.2812928129281293,
    0.28130281302813026,
    0.2813128131281313,
    0.2813228132281323,
    0.28133281332813326,
    0.2813428134281343,
    0.2813528135281353,
    0.28136281362813625,
    0.28137281372813727,
    0.2813828138281383,
    0.28139281392813925,
    0.28140281402814027,
    0.2814128141281413,
    0.2814228142281423,
    0.28143281432814327,
    0.2814428144281443,
    0.2814528145281453,
    0.28146281462814626,
    0.2814728147281473,
    0.2814828148281483,
    0.28149281492814926,
    0.2815028150281503,
    0.2815128151281513,
    0.28152281522815226,
    0.2815328153281533,
    0.2815428154281543,
    0.28155281552815525,
    0.28156281562815627,
    0.2815728157281573,
    0.28158281582815825,
    0.28159281592815927,
    0.2816028160281603,
    0.2816128161281613,
    0.28162281622816226,
    0.2816328163281633,
    0.2816428164281643,
    0.28165281652816526,
    0.2816628166281663,
    0.2816728167281673,
    0.28168281682816826,
    0.2816928169281693,
    0.2817028170281703,
    0.28171281712817126,
    0.2817228172281723,
    0.2817328173281733,
    0.28174281742817425,
    0.28175281752817527,
    0.2817628176281763,
    0.28177281772817725,
    0.28178281782817827,
    0.2817928179281793,
    0.2818028180281803,
    0.28181281812818126,
    0.2818228182281823,
    0.2818328183281833,
    0.28184281842818426,
    0.2818528185281853,
    0.2818628186281863,
    0.28187281872818726,
    0.2818828188281883,
    0.2818928189281893,
    0.28190281902819025,
    0.28191281912819127,
    0.2819228192281923,
    0.28193281932819325,
    0.28194281942819427,
    0.2819528195281953,
    0.2819628196281963,
    0.28197281972819727,
    0.2819828198281983,
    0.2819928199281993,
    0.28200282002820026,
    0.2820128201282013,
    0.2820228202282023,
    0.28203282032820326,
    0.2820428204282043,
    0.2820528205282053,
    0.28206282062820626,
    0.2820728207282073,
    0.2820828208282083,
    0.28209282092820925,
    0.28210282102821027,
    0.2821128211282113,
    0.28212282122821225,
    0.28213282132821327,
    0.2821428214282143,
    0.2821528215282153,
    0.28216282162821626,
    0.2821728217282173,
    0.2821828218282183,
    0.28219282192821926,
    0.2822028220282203,
    0.2822128221282213,
    0.28222282222822226,
    0.2822328223282233,
    0.2822428224282243,
    0.28225282252822526,
    0.2822628226282263,
    0.2822728227282273,
    0.28228282282822825,
    0.28229282292822927,
    0.2823028230282303,
    0.2823128231282313,
    0.28232282322823227,
    0.2823328233282333,
    0.2823428234282343,
    0.28235282352823526,
    0.2823628236282363,
    0.2823728237282373,
    0.28238282382823826,
    0.2823928239282393,
    0.2824028240282403,
    0.28241282412824126,
    0.2824228242282423,
    0.2824328243282433,
    0.28244282442824425,
    0.28245282452824527,
    0.2824628246282463,
    0.28247282472824725,
    0.28248282482824827,
    0.2824928249282493,
    0.2825028250282503,
    0.28251282512825127,
    0.2825228252282523,
    0.2825328253282533,
    0.28254282542825426,
    0.2825528255282553,
    0.2825628256282563,
    0.28257282572825726,
    0.2825828258282583,
    0.2825928259282593,
    0.28260282602826026,
    0.2826128261282613,
    0.2826228262282623,
    0.28263282632826325,
    0.28264282642826427,
    0.2826528265282653,
    0.28266282662826625,
    0.28267282672826727,
    0.2826828268282683,
    0.2826928269282693,
    0.28270282702827026,
    0.2827128271282713,
    0.2827228272282723,
    0.28273282732827326,
    0.2827428274282743,
    0.2827528275282753,
    0.28276282762827626,
    0.2827728277282773,
    0.2827828278282783,
    0.28279282792827926,
    0.2828028280282803,
    0.2828128281282813,
    0.28282282822828225,
    0.28283282832828327,
    0.2828428284282843,
    0.2828528285282853,
    0.28286282862828627,
    0.2828728287282873,
    0.2828828288282883,
    0.28289282892828926,
    0.2829028290282903,
    0.2829128291282913,
    0.28292282922829226,
    0.2829328293282933,
    0.2829428294282943,
    0.28295282952829526,
    0.2829628296282963,
    0.2829728297282973,
    0.28298282982829825,
    0.28299282992829927,
    0.2830028300283003,
    0.28301283012830125,
    0.28302283022830227,
    0.2830328303283033,
    0.2830428304283043,
    0.28305283052830527,
    0.2830628306283063,
    0.2830728307283073,
    0.28308283082830826,
    0.2830928309283093,
    0.2831028310283103,
    0.28311283112831126,
    0.2831228312283123,
    0.2831328313283133,
    0.28314283142831426,
    0.2831528315283153,
    0.2831628316283163,
    0.28317283172831725,
    0.28318283182831827,
    0.2831928319283193,
    0.2832028320283203,
    0.28321283212832127,
    0.2832228322283223,
    0.2832328323283233,
    0.28324283242832426,
    0.2832528325283253,
    0.2832628326283263,
    0.28327283272832726,
    0.2832828328283283,
    0.2832928329283293,
    0.28330283302833026,
    0.2833128331283313,
    0.2833228332283323,
    0.28333283332833326,
    0.2833428334283343,
    0.2833528335283353,
    0.28336283362833625,
    0.28337283372833727,
    0.2833828338283383,
    0.2833928339283393,
    0.28340283402834027,
    0.2834128341283413,
    0.2834228342283423,
    0.28343283432834326,
    0.2834428344283443,
    0.2834528345283453,
    0.28346283462834626,
    0.2834728347283473,
    0.2834828348283483,
    0.28349283492834926,
    0.2835028350283503,
    0.2835128351283513,
    0.28352283522835225,
    0.28353283532835327,
    0.2835428354283543,
    0.28355283552835525,
    0.28356283562835627,
    0.2835728357283573,
    0.2835828358283583,
    0.28359283592835927,
    0.2836028360283603,
    0.2836128361283613,
    0.28362283622836226,
    0.2836328363283633,
    0.2836428364283643,
    0.28365283652836526,
    0.2836628366283663,
    0.2836728367283673,
    0.28368283682836826,
    0.2836928369283693,
    0.2837028370283703,
    0.28371283712837125,
    0.28372283722837227,
    0.2837328373283733,
    0.2837428374283743,
    0.28375283752837527,
    0.2837628376283763,
    0.2837728377283773,
    0.28378283782837826,
    0.2837928379283793,
    0.2838028380283803,
    0.28381283812838126,
    0.2838228382283823,
    0.2838328383283833,
    0.28384283842838426,
    0.2838528385283853,
    0.2838628386283863,
    0.28387283872838726,
    0.2838828388283883,
    0.2838928389283893,
    0.28390283902839025,
    0.28391283912839127,
    0.2839228392283923,
    0.2839328393283933,
    0.28394283942839427,
    0.2839528395283953,
    0.2839628396283963,
    0.28397283972839726,
    0.2839828398283983,
    0.2839928399283993,
    0.28400284002840026,
    0.2840128401284013,
    0.2840228402284023,
    0.28403284032840326,
    0.2840428404284043,
    0.2840528405284053,
    0.28406284062840625,
    0.28407284072840727,
    0.2840828408284083,
    0.28409284092840925,
    0.28410284102841027,
    0.2841128411284113,
    0.2841228412284123,
    0.28413284132841327,
    0.2841428414284143,
    0.2841528415284153,
    0.28416284162841626,
    0.2841728417284173,
    0.2841828418284183,
    0.28419284192841926,
    0.2842028420284203,
    0.2842128421284213,
    0.28422284222842226,
    0.2842328423284233,
    0.2842428424284243,
    0.28425284252842525,
    0.28426284262842627,
    0.2842728427284273,
    0.2842828428284283,
    0.28429284292842927,
    0.2843028430284303,
    0.2843128431284313,
    0.28432284322843226,
    0.2843328433284333,
    0.2843428434284343,
    0.28435284352843526,
    0.2843628436284363,
    0.2843728437284373,
    0.28438284382843826,
    0.2843928439284393,
    0.2844028440284403,
    0.28441284412844126,
    0.2844228442284423,
    0.2844328443284433,
    0.28444284442844425,
    0.28445284452844527,
    0.2844628446284463,
    0.2844728447284473,
    0.28448284482844827,
    0.2844928449284493,
    0.2845028450284503,
    0.28451284512845126,
    0.2845228452284523,
    0.2845328453284533,
    0.28454284542845426,
    0.2845528455284553,
    0.2845628456284563,
    0.28457284572845726,
    0.2845828458284583,
    0.2845928459284593,
    0.28460284602846025,
    0.28461284612846127,
    0.2846228462284623,
    0.2846328463284633,
    0.28464284642846427,
    0.2846528465284653,
    0.2846628466284663,
    0.28467284672846727,
    0.2846828468284683,
    0.2846928469284693,
    0.28470284702847026,
    0.2847128471284713,
    0.2847228472284723,
    0.28473284732847326,
    0.2847428474284743,
    0.2847528475284753,
    0.28476284762847626,
    0.2847728477284773,
    0.2847828478284783,
    0.28479284792847925,
    0.28480284802848027,
    0.2848128481284813,
    0.2848228482284823,
    0.28483284832848327,
    0.2848428484284843,
    0.2848528485284853,
    0.28486284862848626,
    0.2848728487284873,
    0.2848828488284883,
    0.28489284892848926,
    0.2849028490284903,
    0.2849128491284913,
    0.28492284922849226,
    0.2849328493284933,
    0.2849428494284943,
    0.28495284952849526,
    0.2849628496284963,
    0.2849728497284973,
    0.28498284982849825,
    0.28499284992849927,
    0.2850028500285003,
    0.2850128501285013,
    0.28502285022850227,
    0.2850328503285033,
    0.2850428504285043,
    0.28505285052850526,
    0.2850628506285063,
    0.2850728507285073,
    0.28508285082850826,
    0.2850928509285093,
    0.2851028510285103,
    0.28511285112851126,
    0.2851228512285123,
    0.2851328513285133,
    0.28514285142851425,
    0.28515285152851527,
    0.2851628516285163,
    0.2851728517285173,
    0.28518285182851827,
    0.2851928519285193,
    0.2852028520285203,
    0.28521285212852127,
    0.2852228522285223,
    0.2852328523285233,
    0.28524285242852426,
    0.2852528525285253,
    0.2852628526285263,
    0.28527285272852726,
    0.2852828528285283,
    0.2852928529285293,
    0.28530285302853026,
    0.2853128531285313,
    0.2853228532285323,
    0.28533285332853325,
    0.28534285342853427,
    0.2853528535285353,
    0.2853628536285363,
    0.28537285372853727,
    0.2853828538285383,
    0.2853928539285393,
    0.28540285402854026,
    0.2854128541285413,
    0.2854228542285423,
    0.28543285432854326,
    0.2854428544285443,
    0.2854528545285453,
    0.28546285462854626,
    0.2854728547285473,
    0.2854828548285483,
    0.28549285492854926,
    0.2855028550285503,
    0.2855128551285513,
    0.2855228552285523,
    0.28553285532855327,
    0.2855428554285543,
    0.2855528555285553,
    0.28556285562855627,
    0.2855728557285573,
    0.2855828558285583,
    0.28559285592855926,
    0.2856028560285603,
    0.2856128561285613,
    0.28562285622856226,
    0.2856328563285633,
    0.2856428564285643,
    0.28565285652856526,
    0.2856628566285663,
    0.2856728567285673,
    0.28568285682856825,
    0.28569285692856927,
    0.2857028570285703,
    0.2857128571285713,
    0.28572285722857227,
    0.2857328573285733,
    0.2857428574285743,
    0.28575285752857527,
    0.2857628576285763,
    0.2857728577285773,
    0.28578285782857826,
    0.2857928579285793,
    0.2858028580285803,
    0.28581285812858126,
    0.2858228582285823,
    0.2858328583285833,
    0.28584285842858426,
    0.2858528585285853,
    0.2858628586285863,
    0.28587285872858725,
    0.28588285882858827,
    0.2858928589285893,
    0.2859028590285903,
    0.28591285912859127,
    0.2859228592285923,
    0.2859328593285933,
    0.28594285942859426,
    0.2859528595285953,
    0.2859628596285963,
    0.28597285972859726,
    0.2859828598285983,
    0.2859928599285993,
    0.28600286002860026,
    0.2860128601286013,
    0.2860228602286023,
    0.28603286032860326,
    0.2860428604286043,
    0.2860528605286053,
    0.2860628606286063,
    0.28607286072860727,
    0.2860828608286083,
    0.2860928609286093,
    0.28610286102861027,
    0.2861128611286113,
    0.2861228612286123,
    0.28613286132861326,
    0.2861428614286143,
    0.2861528615286153,
    0.28616286162861626,
    0.2861728617286173,
    0.2861828618286183,
    0.28619286192861926,
    0.2862028620286203,
    0.2862128621286213,
    0.28622286222862225,
    0.28623286232862327,
    0.2862428624286243,
    0.2862528625286253,
    0.28626286262862627,
    0.2862728627286273,
    0.2862828628286283,
    0.28629286292862927,
    0.2863028630286303,
    0.2863128631286313,
    0.28632286322863226,
    0.2863328633286333,
    0.2863428634286343,
    0.28635286352863526,
    0.2863628636286363,
    0.2863728637286373,
    0.28638286382863826,
    0.2863928639286393,
    0.2864028640286403,
    0.2864128641286413,
    0.28642286422864227,
    0.2864328643286433,
    0.2864428644286443,
    0.28645286452864527,
    0.2864628646286463,
    0.2864728647286473,
    0.28648286482864826,
    0.2864928649286493,
    0.2865028650286503,
    0.28651286512865126,
    0.2865228652286523,
    0.2865328653286533,
    0.28654286542865426,
    0.2865528655286553,
    0.2865628656286563,
    0.28657286572865726,
    0.2865828658286583,
    0.2865928659286593,
    0.2866028660286603,
    0.28661286612866127,
    0.2866228662286623,
    0.2866328663286633,
    0.28664286642866427,
    0.2866528665286653,
    0.2866628666286663,
    0.28667286672866726,
    0.2866828668286683,
    0.2866928669286693,
    0.28670286702867026,
    0.2867128671286713,
    0.2867228672286723,
    0.28673286732867326,
    0.2867428674286743,
    0.2867528675286753,
    0.28676286762867625,
    0.28677286772867727,
    0.2867828678286783,
    0.2867928679286793,
    0.28680286802868027,
    0.2868128681286813,
    0.2868228682286823,
    0.28683286832868327,
    0.2868428684286843,
    0.2868528685286853,
    0.28686286862868626,
    0.2868728687286873,
    0.2868828688286883,
    0.28689286892868926,
    0.2869028690286903,
    0.2869128691286913,
    0.28692286922869226,
    0.2869328693286933,
    0.2869428694286943,
    0.2869528695286953,
    0.28696286962869627,
    0.2869728697286973,
    0.2869828698286983,
    0.28699286992869927,
    0.2870028700287003,
    0.2870128701287013,
    0.28702287022870226,
    0.2870328703287033,
    0.2870428704287043,
    0.28705287052870526,
    0.2870628706287063,
    0.2870728707287073,
    0.28708287082870826,
    0.2870928709287093,
    0.2871028710287103,
    0.28711287112871126,
    0.2871228712287123,
    0.2871328713287133,
    0.2871428714287143,
    0.28715287152871527,
    0.2871628716287163,
    0.2871728717287173,
    0.28718287182871827,
    0.2871928719287193,
    0.2872028720287203,
    0.28721287212872126,
    0.2872228722287223,
    0.2872328723287233,
    0.28724287242872426,
    0.2872528725287253,
    0.2872628726287263,
    0.28727287272872726,
    0.2872828728287283,
    0.2872928729287293,
    0.2873028730287303,
    0.28731287312873127,
    0.2873228732287323,
    0.2873328733287333,
    0.28734287342873427,
    0.2873528735287353,
    0.2873628736287363,
    0.28737287372873727,
    0.2873828738287383,
    0.2873928739287393,
    0.28740287402874026,
    0.2874128741287413,
    0.2874228742287423,
    0.28743287432874326,
    0.2874428744287443,
    0.2874528745287453,
    0.28746287462874626,
    0.2874728747287473,
    0.2874828748287483,
    0.2874928749287493,
    0.28750287502875027,
    0.2875128751287513,
    0.2875228752287523,
    0.28753287532875327,
    0.2875428754287543,
    0.2875528755287553,
    0.28756287562875626,
    0.2875728757287573,
    0.2875828758287583,
    0.28759287592875926,
    0.2876028760287603,
    0.2876128761287613,
    0.28762287622876226,
    0.2876328763287633,
    0.2876428764287643,
    0.28765287652876526,
    0.2876628766287663,
    0.2876728767287673,
    0.2876828768287683,
    0.28769287692876927,
    0.2877028770287703,
    0.2877128771287713,
    0.28772287722877227,
    0.2877328773287733,
    0.2877428774287743,
    0.28775287752877526,
    0.2877628776287763,
    0.2877728777287773,
    0.28778287782877826,
    0.2877928779287793,
    0.2878028780287803,
    0.28781287812878126,
    0.2878228782287823,
    0.2878328783287833,
    0.2878428784287843,
    0.28785287852878527,
    0.2878628786287863,
    0.2878728787287873,
    0.28788287882878827,
    0.2878928789287893,
    0.2879028790287903,
    0.28791287912879127,
    0.2879228792287923,
    0.2879328793287933,
    0.28794287942879426,
    0.2879528795287953,
    0.2879628796287963,
    0.28797287972879726,
    0.2879828798287983,
    0.2879928799287993,
    0.28800288002880026,
    0.2880128801288013,
    0.2880228802288023,
    0.2880328803288033,
    0.28804288042880427,
    0.2880528805288053,
    0.2880628806288063,
    0.28807288072880727,
    0.2880828808288083,
    0.2880928809288093,
    0.28810288102881026,
    0.2881128811288113,
    0.2881228812288123,
    0.28813288132881326,
    0.2881428814288143,
    0.2881528815288153,
    0.28816288162881626,
    0.2881728817288173,
    0.2881828818288183,
    0.28819288192881926,
    0.2882028820288203,
    0.2882128821288213,
    0.2882228822288223,
    0.28823288232882327,
    0.2882428824288243,
    0.2882528825288253,
    0.28826288262882627,
    0.2882728827288273,
    0.2882828828288283,
    0.28829288292882926,
    0.2883028830288303,
    0.2883128831288313,
    0.28832288322883226,
    0.2883328833288333,
    0.2883428834288343,
    0.28835288352883526,
    0.2883628836288363,
    0.2883728837288373,
    0.2883828838288383,
    0.28839288392883927,
    0.2884028840288403,
    0.2884128841288413,
    0.28842288422884227,
    0.2884328843288433,
    0.2884428844288443,
    0.28845288452884527,
    0.2884628846288463,
    0.2884728847288473,
    0.28848288482884826,
    0.2884928849288493,
    0.2885028850288503,
    0.28851288512885126,
    0.2885228852288523,
    0.2885328853288533,
    0.28854288542885426,
    0.2885528855288553,
    0.2885628856288563,
    0.2885728857288573,
    0.28858288582885827,
    0.2885928859288593,
    0.2886028860288603,
    0.28861288612886127,
    0.2886228862288623,
    0.2886328863288633,
    0.28864288642886426,
    0.2886528865288653,
    0.2886628866288663,
    0.28867288672886726,
    0.2886828868288683,
    0.2886928869288693,
    0.28870288702887026,
    0.2887128871288713,
    0.2887228872288723,
    0.2887328873288733,
    0.2887428874288743,
    0.2887528875288753,
    0.2887628876288763,
    0.28877288772887727,
    0.2887828878288783,
    0.2887928879288793,
    0.28880288802888027,
    0.2888128881288813,
    0.2888228882288823,
    0.28883288832888326,
    0.2888428884288843,
    0.2888528885288853,
    0.28886288862888626,
    0.2888728887288873,
    0.2888828888288883,
    0.28889288892888926,
    0.2889028890288903,
    0.2889128891288913,
    0.2889228892288923,
    0.28893288932889327,
    0.2889428894288943,
    0.2889528895288953,
    0.28896288962889627,
    0.2889728897288973,
    0.2889828898288983,
    0.28899288992889927,
    0.2890028900289003,
    0.2890128901289013,
    0.28902289022890226,
    0.2890328903289033,
    0.2890428904289043,
    0.28905289052890526,
    0.2890628906289063,
    0.2890728907289073,
    0.28908289082890826,
    0.2890928909289093,
    0.2891028910289103,
    0.2891128911289113,
    0.28912289122891227,
    0.2891328913289133,
    0.2891428914289143,
    0.28915289152891527,
    0.2891628916289163,
    0.2891728917289173,
    0.28918289182891826,
    0.2891928919289193,
    0.2892028920289203,
    0.28921289212892126,
    0.2892228922289223,
    0.2892328923289233,
    0.28924289242892426,
    0.2892528925289253,
    0.2892628926289263,
    0.2892728927289273,
    0.2892828928289283,
    0.2892928929289293,
    0.2893028930289303,
    0.28931289312893127,
    0.2893228932289323,
    0.2893328933289333,
    0.28934289342893427,
    0.2893528935289353,
    0.2893628936289363,
    0.28937289372893726,
    0.2893828938289383,
    0.2893928939289393,
    0.28940289402894026,
    0.2894128941289413,
    0.2894228942289423,
    0.28943289432894326,
    0.2894428944289443,
    0.2894528945289453,
    0.2894628946289463,
    0.28947289472894727,
    0.2894828948289483,
    0.2894928949289493,
    0.28950289502895027,
    0.2895128951289513,
    0.2895228952289523,
    0.28953289532895327,
    0.2895428954289543,
    0.2895528955289553,
    0.28956289562895626,
    0.2895728957289573,
    0.2895828958289583,
    0.28959289592895926,
    0.2896028960289603,
    0.2896128961289613,
    0.2896228962289623,
    0.2896328963289633,
    0.2896428964289643,
    0.2896528965289653,
    0.28966289662896627,
    0.2896728967289673,
    0.2896828968289683,
    0.28969289692896927,
    0.2897028970289703,
    0.2897128971289713,
    0.28972289722897226,
    0.2897328973289733,
    0.2897428974289743,
    0.28975289752897526,
    0.2897628976289763,
    0.2897728977289773,
    0.28978289782897826,
    0.2897928979289793,
    0.2898028980289803,
    0.2898128981289813,
    0.2898228982289823,
    0.2898328983289833,
    0.2898428984289843,
    0.28985289852898527,
    0.2898628986289863,
    0.2898728987289873,
    0.28988289882898827,
    0.2898928989289893,
    0.2899028990289903,
    0.28991289912899126,
    0.2899228992289923,
    0.2899328993289933,
    0.28994289942899426,
    0.2899528995289953,
    0.2899628996289963,
    0.28997289972899726,
    0.2899828998289983,
    0.2899928999289993,
    0.2900029000290003,
    0.29001290012900127,
    0.2900229002290023,
    0.2900329003290033,
    0.29004290042900427,
    0.2900529005290053,
    0.2900629006290063,
    0.29007290072900727,
    0.2900829008290083,
    0.2900929009290093,
    0.29010290102901026,
    0.2901129011290113,
    0.2901229012290123,
    0.29013290132901326,
    0.2901429014290143,
    0.2901529015290153,
    0.2901629016290163,
    0.2901729017290173,
    0.2901829018290183,
    0.2901929019290193,
    0.29020290202902027,
    0.2902129021290213,
    0.2902229022290223,
    0.29023290232902327,
    0.2902429024290243,
    0.2902529025290253,
    0.29026290262902626,
    0.2902729027290273,
    0.2902829028290283,
    0.29029290292902926,
    0.2903029030290303,
    0.2903129031290313,
    0.29032290322903226,
    0.2903329033290333,
    0.2903429034290343,
    0.2903529035290353,
    0.2903629036290363,
    0.2903729037290373,
    0.2903829038290383,
    0.29039290392903927,
    0.2904029040290403,
    0.2904129041290413,
    0.29042290422904227,
    0.2904329043290433,
    0.2904429044290443,
    0.29045290452904526,
    0.2904629046290463,
    0.2904729047290473,
    0.29048290482904826,
    0.2904929049290493,
    0.2905029050290503,
    0.2905129051290513,
    0.2905229052290523,
    0.2905329053290533,
    0.2905429054290543,
    0.29055290552905527,
    0.2905629056290563,
    0.2905729057290573,
    0.29058290582905827,
    0.2905929059290593,
    0.2906029060290603,
    0.29061290612906127,
    0.2906229062290623,
    0.2906329063290633,
    0.29064290642906426,
    0.2906529065290653,
    0.2906629066290663,
    0.29067290672906726,
    0.2906829068290683,
    0.2906929069290693,
    0.2907029070290703,
    0.2907129071290713,
    0.2907229072290723,
    0.2907329073290733,
    0.29074290742907427,
    0.2907529075290753,
    0.2907629076290763,
    0.29077290772907727,
    0.2907829078290783,
    0.2907929079290793,
    0.29080290802908026,
    0.2908129081290813,
    0.2908229082290823,
    0.29083290832908326,
    0.2908429084290843,
    0.2908529085290853,
    0.29086290862908626,
    0.2908729087290873,
    0.2908829088290883,
    0.2908929089290893,
    0.2909029090290903,
    0.2909129091290913,
    0.2909229092290923,
    0.29093290932909327,
    0.2909429094290943,
    0.2909529095290953,
    0.29096290962909627,
    0.2909729097290973,
    0.2909829098290983,
    0.29099290992909926,
    0.2910029100291003,
    0.2910129101291013,
    0.29102291022910226,
    0.2910329103291033,
    0.2910429104291043,
    0.2910529105291053,
    0.2910629106291063,
    0.2910729107291073,
    0.2910829108291083,
    0.29109291092910927,
    0.2911029110291103,
    0.2911129111291113,
    0.29112291122911227,
    0.2911329113291133,
    0.2911429114291143,
    0.29115291152911527,
    0.2911629116291163,
    0.2911729117291173,
    0.29118291182911826,
    0.2911929119291193,
    0.2912029120291203,
    0.29121291212912126,
    0.2912229122291223,
    0.2912329123291233,
    0.2912429124291243,
    0.2912529125291253,
    0.2912629126291263,
    0.2912729127291273,
    0.29128291282912827,
    0.2912929129291293,
    0.2913029130291303,
    0.29131291312913127,
    0.2913229132291323,
    0.2913329133291333,
    0.29134291342913426,
    0.2913529135291353,
    0.2913629136291363,
    0.29137291372913726,
    0.2913829138291383,
    0.2913929139291393,
    0.2914029140291403,
    0.2914129141291413,
    0.2914229142291423,
    0.2914329143291433,
    0.2914429144291443,
    0.2914529145291453,
    0.2914629146291463,
    0.29147291472914727,
    0.2914829148291483,
    0.2914929149291493,
    0.29150291502915027,
    0.2915129151291513,
    0.2915229152291523,
    0.29153291532915326,
    0.2915429154291543,
    0.2915529155291553,
    0.29156291562915626,
    0.2915729157291573,
    0.2915829158291583,
    0.2915929159291593,
    0.2916029160291603,
    0.2916129161291613,
    0.2916229162291623,
    0.29163291632916327,
    0.2916429164291643,
    0.2916529165291653,
    0.29166291662916627,
    0.2916729167291673,
    0.2916829168291683,
    0.29169291692916927,
    0.2917029170291703,
    0.2917129171291713,
    0.29172291722917226,
    0.2917329173291733,
    0.2917429174291743,
    0.29175291752917526,
    0.2917629176291763,
    0.2917729177291773,
    0.2917829178291783,
    0.2917929179291793,
    0.2918029180291803,
    0.2918129181291813,
    0.29182291822918227,
    0.2918329183291833,
    0.2918429184291843,
    0.29185291852918527,
    0.2918629186291863,
    0.2918729187291873,
    0.29188291882918826,
    0.2918929189291893,
    0.2919029190291903,
    0.29191291912919126,
    0.2919229192291923,
    0.2919329193291933,
    0.2919429194291943,
    0.2919529195291953,
    0.2919629196291963,
    0.2919729197291973,
    0.2919829198291983,
    0.2919929199291993,
    0.2920029200292003,
    0.29201292012920127,
    0.2920229202292023,
    0.2920329203292033,
    0.29204292042920427,
    0.2920529205292053,
    0.2920629206292063,
    0.29207292072920726,
    0.2920829208292083,
    0.2920929209292093,
    0.29210292102921026,
    0.2921129211292113,
    0.2921229212292123,
    0.2921329213292133,
    0.2921429214292143,
    0.2921529215292153,
    0.2921629216292163,
    0.29217292172921727,
    0.2921829218292183,
    0.2921929219292193,
    0.29220292202922027,
    0.2922129221292213,
    0.2922229222292223,
    0.29223292232922327,
    0.2922429224292243,
    0.2922529225292253,
    0.29226292262922626,
    0.2922729227292273,
    0.2922829228292283,
    0.29229292292922926,
    0.2923029230292303,
    0.2923129231292313,
    0.2923229232292323,
    0.2923329233292333,
    0.2923429234292343,
    0.2923529235292353,
    0.29236292362923627,
    0.2923729237292373,
    0.2923829238292383,
    0.29239292392923927,
    0.2924029240292403,
    0.2924129241292413,
    0.29242292422924226,
    0.2924329243292433,
    0.2924429244292443,
    0.29245292452924526,
    0.2924629246292463,
    0.2924729247292473,
    0.2924829248292483,
    0.2924929249292493,
    0.2925029250292503,
    0.2925129251292513,
    0.2925229252292523,
    0.2925329253292533,
    0.2925429254292543,
    0.29255292552925527,
    0.2925629256292563,
    0.2925729257292573,
    0.29258292582925827,
    0.2925929259292593,
    0.2926029260292603,
    0.29261292612926126,
    0.2926229262292623,
    0.2926329263292633,
    0.29264292642926426,
    0.2926529265292653,
    0.2926629266292663,
    0.2926729267292673,
    0.2926829268292683,
    0.2926929269292693,
    0.2927029270292703,
    0.29271292712927127,
    0.2927229272292723,
    0.2927329273292733,
    0.29274292742927427,
    0.2927529275292753,
    0.2927629276292763,
    0.29277292772927727,
    0.2927829278292783,
    0.2927929279292793,
    0.29280292802928026,
    0.2928129281292813,
    0.2928229282292823,
    0.2928329283292833,
    0.2928429284292843,
    0.2928529285292853,
    0.2928629286292863,
    0.2928729287292873,
    0.2928829288292883,
    0.2928929289292893,
    0.29290292902929027,
    0.2929129291292913,
    0.2929229292292923,
    0.29293292932929327,
    0.2929429294292943,
    0.2929529295292953,
    0.29296292962929626,
    0.2929729297292973,
    0.2929829298292983,
    0.29299292992929926,
    0.2930029300293003,
    0.2930129301293013,
    0.2930229302293023,
    0.2930329303293033,
    0.2930429304293043,
    0.2930529305293053,
    0.2930629306293063,
    0.2930729307293073,
    0.2930829308293083,
    0.29309293092930927,
    0.2931029310293103,
    0.2931129311293113,
    0.29312293122931227,
    0.2931329313293133,
    0.2931429314293143,
    0.29315293152931526,
    0.2931629316293163,
    0.2931729317293173,
    0.29318293182931826,
    0.2931929319293193,
    0.2932029320293203,
    0.2932129321293213,
    0.2932229322293223,
    0.2932329323293233,
    0.2932429324293243,
    0.29325293252932527,
    0.2932629326293263,
    0.2932729327293273,
    0.29328293282932827,
    0.2932929329293293,
    0.2933029330293303,
    0.29331293312933127,
    0.2933229332293323,
    0.2933329333293333,
    0.29334293342933426,
    0.2933529335293353,
    0.2933629336293363,
    0.2933729337293373,
    0.2933829338293383,
    0.2933929339293393,
    0.2934029340293403,
    0.2934129341293413,
    0.2934229342293423,
    0.2934329343293433,
    0.29344293442934427,
    0.2934529345293453,
    0.2934629346293463,
    0.29347293472934727,
    0.2934829348293483,
    0.2934929349293493,
    0.29350293502935026,
    0.2935129351293513,
    0.2935229352293523,
    0.29353293532935326,
    0.2935429354293543,
    0.2935529355293553,
    0.2935629356293563,
    0.2935729357293573,
    0.2935829358293583,
    0.2935929359293593,
    0.2936029360293603,
    0.2936129361293613,
    0.2936229362293623,
    0.29363293632936327,
    0.2936429364293643,
    0.2936529365293653,
    0.29366293662936627,
    0.2936729367293673,
    0.2936829368293683,
    0.29369293692936926,
    0.2937029370293703,
    0.2937129371293713,
    0.2937229372293723,
    0.2937329373293733,
    0.2937429374293743,
    0.2937529375293753,
    0.2937629376293763,
    0.2937729377293773,
    0.2937829378293783,
    0.29379293792937927,
    0.2938029380293803,
    0.2938129381293813,
    0.29382293822938227,
    0.2938329383293833,
    0.2938429384293843,
    0.29385293852938527,
    0.2938629386293863,
    0.2938729387293873,
    0.29388293882938826,
    0.2938929389293893,
    0.2939029390293903,
    0.2939129391293913,
    0.2939229392293923,
    0.2939329393293933,
    0.2939429394293943,
    0.2939529395293953,
    0.2939629396293963,
    0.2939729397293973,
    0.29398293982939827,
    0.2939929399293993,
    0.2940029400294003,
    0.29401294012940127,
    0.2940229402294023,
    0.2940329403294033,
    0.29404294042940426,
    0.2940529405294053,
    0.2940629406294063,
    0.29407294072940726,
    0.2940829408294083,
    0.2940929409294093,
    0.2941029410294103,
    0.2941129411294113,
    0.2941229412294123,
    0.2941329413294133,
    0.2941429414294143,
    0.2941529415294153,
    0.2941629416294163,
    0.29417294172941727,
    0.2941829418294183,
    0.2941929419294193,
    0.29420294202942027,
    0.2942129421294213,
    0.2942229422294223,
    0.29423294232942326,
    0.2942429424294243,
    0.2942529425294253,
    0.2942629426294263,
    0.2942729427294273,
    0.2942829428294283,
    0.2942929429294293,
    0.2943029430294303,
    0.2943129431294313,
    0.2943229432294323,
    0.29433294332943327,
    0.2943429434294343,
    0.2943529435294353,
    0.29436294362943627,
    0.2943729437294373,
    0.2943829438294383,
    0.29439294392943927,
    0.2944029440294403,
    0.2944129441294413,
    0.29442294422944226,
    0.2944329443294433,
    0.2944429444294443,
    0.2944529445294453,
    0.2944629446294463,
    0.2944729447294473,
    0.2944829448294483,
    0.2944929449294493,
    0.2945029450294503,
    0.2945129451294513,
    0.29452294522945227,
    0.2945329453294533,
    0.2945429454294543,
    0.29455294552945527,
    0.2945629456294563,
    0.2945729457294573,
    0.29458294582945826,
    0.2945929459294593,
    0.2946029460294603,
    0.2946129461294613,
    0.2946229462294623,
    0.2946329463294633,
    0.2946429464294643,
    0.2946529465294653,
    0.2946629466294663,
    0.2946729467294673,
    0.2946829468294683,
    0.2946929469294693,
    0.2947029470294703,
    0.29471294712947127,
    0.2947229472294723,
    0.2947329473294733,
    0.29474294742947427,
    0.2947529475294753,
    0.2947629476294763,
    0.29477294772947726,
    0.2947829478294783,
    0.2947929479294793,
    0.2948029480294803,
    0.2948129481294813,
    0.2948229482294823,
    0.2948329483294833,
    0.2948429484294843,
    0.2948529485294853,
    0.2948629486294863,
    0.29487294872948727,
    0.2948829488294883,
    0.2948929489294893,
    0.29490294902949027,
    0.2949129491294913,
    0.2949229492294923,
    0.29493294932949327,
    0.2949429494294943,
    0.2949529495294953,
    0.29496294962949626,
    0.2949729497294973,
    0.2949829498294983,
    0.2949929499294993,
    0.2950029500295003,
    0.2950129501295013,
    0.2950229502295023,
    0.2950329503295033,
    0.2950429504295043,
    0.2950529505295053,
    0.29506295062950627,
    0.2950729507295073,
    0.2950829508295083,
    0.29509295092950927,
    0.2951029510295103,
    0.2951129511295113,
    0.29512295122951226,
    0.2951329513295133,
    0.2951429514295143,
    0.2951529515295153,
    0.2951629516295163,
    0.2951729517295173,
    0.2951829518295183,
    0.2951929519295193,
    0.2952029520295203,
    0.2952129521295213,
    0.2952229522295223,
    0.2952329523295233,
    0.2952429524295243,
    0.29525295252952527,
    0.2952629526295263,
    0.2952729527295273,
    0.29528295282952827,
    0.2952929529295293,
    0.2953029530295303,
    0.29531295312953126,
    0.2953229532295323,
    0.2953329533295333,
    0.2953429534295343,
    0.2953529535295353,
    0.2953629536295363,
    0.2953729537295373,
    0.2953829538295383,
    0.2953929539295393,
    0.2954029540295403,
    0.29541295412954127,
    0.2954229542295423,
    0.2954329543295433,
    0.29544295442954427,
    0.2954529545295453,
    0.2954629546295463,
    0.29547295472954727,
    0.2954829548295483,
    0.2954929549295493,
    0.29550295502955026,
    0.2955129551295513,
    0.2955229552295523,
    0.2955329553295533,
    0.2955429554295543,
    0.2955529555295553,
    0.2955629556295563,
    0.2955729557295573,
    0.2955829558295583,
    0.2955929559295593,
    0.29560295602956027,
    0.2956129561295613,
    0.2956229562295623,
    0.29563295632956327,
    0.2956429564295643,
    0.2956529565295653,
    0.29566295662956626,
    0.2956729567295673,
    0.2956829568295683,
    0.2956929569295693,
    0.2957029570295703,
    0.2957129571295713,
    0.2957229572295723,
    0.2957329573295733,
    0.2957429574295743,
    0.2957529575295753,
    0.2957629576295763,
    0.2957729577295773,
    0.2957829578295783,
    0.29579295792957927,
    0.2958029580295803,
    0.2958129581295813,
    0.29582295822958227,
    0.2958329583295833,
    0.2958429584295843,
    0.29585295852958526,
    0.2958629586295863,
    0.2958729587295873,
    0.2958829588295883,
    0.2958929589295893,
    0.2959029590295903,
    0.2959129591295913,
    0.2959229592295923,
    0.2959329593295933,
    0.2959429594295943,
    0.29595295952959527,
    0.2959629596295963,
    0.2959729597295973,
    0.29598295982959827,
    0.2959929599295993,
    0.2960029600296003,
    0.29601296012960127,
    0.2960229602296023,
    0.2960329603296033,
    0.2960429604296043,
    0.2960529605296053,
    0.2960629606296063,
    0.2960729607296073,
    0.2960829608296083,
    0.2960929609296093,
    0.2961029610296103,
    0.2961129611296113,
    0.2961229612296123,
    0.2961329613296133,
    0.29614296142961427,
    0.2961529615296153,
    0.2961629616296163,
    0.29617296172961727,
    0.2961829618296183,
    0.2961929619296193,
    0.29620296202962026,
    0.2962129621296213,
    0.2962229622296223,
    0.2962329623296233,
    0.2962429624296243,
    0.2962529625296253,
    0.2962629626296263,
    0.2962729627296273,
    0.2962829628296283,
    0.2962929629296293,
    0.2963029630296303,
    0.2963129631296313,
    0.2963229632296323,
    0.29633296332963327,
    0.2963429634296343,
    0.2963529635296353,
    0.29636296362963627,
    0.2963729637296373,
    0.2963829638296383,
    0.29639296392963926,
    0.2964029640296403,
    0.2964129641296413,
    0.2964229642296423,
    0.2964329643296433,
    0.2964429644296443,
    0.2964529645296453,
    0.2964629646296463,
    0.2964729647296473,
    0.2964829648296483,
    0.29649296492964927,
    0.2965029650296503,
    0.2965129651296513,
    0.29652296522965227,
    0.2965329653296533,
    0.2965429654296543,
    0.29655296552965527,
    0.2965629656296563,
    0.2965729657296573,
    0.2965829658296583,
    0.2965929659296593,
    0.2966029660296603,
    0.2966129661296613,
    0.2966229662296623,
    0.2966329663296633,
    0.2966429664296643,
    0.2966529665296653,
    0.2966629666296663,
    0.2966729667296673,
    0.29668296682966827,
    0.2966929669296693,
    0.2967029670296703,
    0.29671296712967127,
    0.2967229672296723,
    0.2967329673296733,
    0.29674296742967426,
    0.2967529675296753,
    0.2967629676296763,
    0.2967729677296773,
    0.2967829678296783,
    0.2967929679296793,
    0.2968029680296803,
    0.2968129681296813,
    0.2968229682296823,
    0.2968329683296833,
    0.2968429684296843,
    0.2968529685296853,
    0.2968629686296863,
    0.29687296872968727,
    0.2968829688296883,
    0.2968929689296893,
    0.29690296902969027,
    0.2969129691296913,
    0.2969229692296923,
    0.2969329693296933,
    0.2969429694296943,
    0.2969529695296953,
    0.2969629696296963,
    0.2969729697296973,
    0.2969829698296983,
    0.2969929699296993,
    0.2970029700297003,
    0.2970129701297013,
    0.2970229702297023,
    0.29703297032970327,
    0.2970429704297043,
    0.2970529705297053,
    0.29706297062970627,
    0.2970729707297073,
    0.2970829708297083,
    0.29709297092970927,
    0.2971029710297103,
    0.2971129711297113,
    0.2971229712297123,
    0.2971329713297133,
    0.2971429714297143,
    0.2971529715297153,
    0.2971629716297163,
    0.2971729717297173,
    0.2971829718297183,
    0.2971929719297193,
    0.2972029720297203,
    0.2972129721297213,
    0.29722297222972227,
    0.2972329723297233,
    0.2972429724297243,
    0.29725297252972527,
    0.2972629726297263,
    0.2972729727297273,
    0.29728297282972826,
    0.2972929729297293,
    0.2973029730297303,
    0.2973129731297313,
    0.2973229732297323,
    0.2973329733297333,
    0.2973429734297343,
    0.2973529735297353,
    0.2973629736297363,
    0.2973729737297373,
    0.2973829738297383,
    0.2973929739297393,
    0.2974029740297403,
    0.29741297412974127,
    0.2974229742297423,
    0.2974329743297433,
    0.29744297442974427,
    0.2974529745297453,
    0.2974629746297463,
    0.2974729747297473,
    0.2974829748297483,
    0.2974929749297493,
    0.2975029750297503,
    0.2975129751297513,
    0.2975229752297523,
    0.2975329753297533,
    0.2975429754297543,
    0.2975529755297553,
    0.2975629756297563,
    0.29757297572975727,
    0.2975829758297583,
    0.2975929759297593,
    0.29760297602976027,
    0.2976129761297613,
    0.2976229762297623,
    0.29763297632976327,
    0.2976429764297643,
    0.2976529765297653,
    0.2976629766297663,
    0.2976729767297673,
    0.2976829768297683,
    0.2976929769297693,
    0.2977029770297703,
    0.2977129771297713,
    0.2977229772297723,
    0.2977329773297733,
    0.2977429774297743,
    0.2977529775297753,
    0.29776297762977627,
    0.2977729777297773,
    0.2977829778297783,
    0.29779297792977927,
    0.2978029780297803,
    0.2978129781297813,
    0.2978229782297823,
    0.2978329783297833,
    0.2978429784297843,
    0.2978529785297853,
    0.2978629786297863,
    0.2978729787297873,
    0.2978829788297883,
    0.2978929789297893,
    0.2979029790297903,
    0.2979129791297913,
    0.2979229792297923,
    0.2979329793297933,
    0.2979429794297943,
    0.29795297952979527,
    0.2979629796297963,
    0.2979729797297973,
    0.29798297982979827,
    0.2979929799297993,
    0.2980029800298003,
    0.2980129801298013,
    0.2980229802298023,
    0.2980329803298033,
    0.2980429804298043,
    0.2980529805298053,
    0.2980629806298063,
    0.2980729807298073,
    0.2980829808298083,
    0.2980929809298093,
    0.2981029810298103,
    0.29811298112981127,
    0.2981229812298123,
    0.2981329813298133,
    0.29814298142981427,
    0.2981529815298153,
    0.2981629816298163,
    0.29817298172981727,
    0.2981829818298183,
    0.2981929819298193,
    0.2982029820298203,
    0.2982129821298213,
    0.2982229822298223,
    0.2982329823298233,
    0.2982429824298243,
    0.2982529825298253,
    0.2982629826298263,
    0.2982729827298273,
    0.2982829828298283,
    0.2982929829298293,
    0.29830298302983027,
    0.2983129831298313,
    0.2983229832298323,
    0.29833298332983327,
    0.2983429834298343,
    0.2983529835298353,
    0.2983629836298363,
    0.2983729837298373,
    0.2983829838298383,
    0.2983929839298393,
    0.2984029840298403,
    0.2984129841298413,
    0.2984229842298423,
    0.2984329843298433,
    0.2984429844298443,
    0.2984529845298453,
    0.2984629846298463,
    0.2984729847298473,
    0.2984829848298483,
    0.29849298492984927,
    0.2985029850298503,
    0.2985129851298513,
    0.29852298522985227,
    0.2985329853298533,
    0.2985429854298543,
    0.2985529855298553,
    0.2985629856298563,
    0.2985729857298573,
    0.2985829858298583,
    0.2985929859298593,
    0.2986029860298603,
    0.2986129861298613,
    0.2986229862298623,
    0.2986329863298633,
    0.2986429864298643,
    0.29865298652986527,
    0.2986629866298663,
    0.2986729867298673,
    0.29868298682986827,
    0.2986929869298693,
    0.2987029870298703,
    0.2987129871298713,
    0.2987229872298723,
    0.2987329873298733,
    0.2987429874298743,
    0.2987529875298753,
    0.2987629876298763,
    0.2987729877298773,
    0.2987829878298783,
    0.2987929879298793,
    0.2988029880298803,
    0.2988129881298813,
    0.2988229882298823,
    0.2988329883298833,
    0.29884298842988427,
    0.2988529885298853,
    0.2988629886298863,
    0.29887298872988727,
    0.2988829888298883,
    0.2988929889298893,
    0.2989029890298903,
    0.2989129891298913,
    0.2989229892298923,
    0.2989329893298933,
    0.2989429894298943,
    0.2989529895298953,
    0.2989629896298963,
    0.2989729897298973,
    0.2989829898298983,
    0.2989929899298993,
    0.2990029900299003,
    0.2990129901299013,
    0.2990229902299023,
    0.29903299032990327,
    0.2990429904299043,
    0.2990529905299053,
    0.29906299062990627,
    0.2990729907299073,
    0.2990829908299083,
    0.2990929909299093,
    0.2991029910299103,
    0.2991129911299113,
    0.2991229912299123,
    0.2991329913299133,
    0.2991429914299143,
    0.2991529915299153,
    0.2991629916299163,
    0.2991729917299173,
    0.2991829918299183,
    0.29919299192991927,
    0.2992029920299203,
    0.2992129921299213,
    0.29922299222992227,
    0.2992329923299233,
    0.2992429924299243,
    0.2992529925299253,
    0.2992629926299263,
    0.2992729927299273,
    0.2992829928299283,
    0.2992929929299293,
    0.2993029930299303,
    0.2993129931299313,
    0.2993229932299323,
    0.2993329933299333,
    0.2993429934299343,
    0.2993529935299353,
    0.2993629936299363,
    0.2993729937299373,
    0.29938299382993827,
    0.2993929939299393,
    0.2994029940299403,
    0.29941299412994127,
    0.2994229942299423,
    0.2994329943299433,
    0.2994429944299443,
    0.2994529945299453,
    0.2994629946299463,
    0.2994729947299473,
    0.2994829948299483,
    0.2994929949299493,
    0.2995029950299503,
    0.2995129951299513,
    0.2995229952299523,
    0.2995329953299533,
    0.2995429954299543,
    0.2995529955299553,
    0.2995629956299563,
    0.29957299572995727,
    0.2995829958299583,
    0.2995929959299593,
    0.29960299602996027,
    0.2996129961299613,
    0.2996229962299623,
    0.2996329963299633,
    0.2996429964299643,
    0.2996529965299653,
    0.2996629966299663,
    0.2996729967299673,
    0.2996829968299683,
    0.2996929969299693,
    0.2997029970299703,
    0.2997129971299713,
    0.2997229972299723,
    0.29973299732997327,
    0.2997429974299743,
    0.2997529975299753,
    0.29976299762997627,
    0.2997729977299773,
    0.2997829978299783,
    0.2997929979299793,
    0.2998029980299803,
    0.2998129981299813,
    0.2998229982299823,
    0.2998329983299833,
    0.2998429984299843,
    0.2998529985299853,
    0.2998629986299863,
    0.2998729987299873,
    0.2998829988299883,
    0.2998929989299893,
    0.2999029990299903,
    0.2999129991299913,
    0.29992299922999227,
    0.2999329993299933,
    0.2999429994299943,
    0.29995299952999527,
    0.2999629996299963,
    0.2999729997299973,
    0.2999829998299983,
    0.2999929999299993,
    0.3000030000300003,
    0.3000130001300013,
    0.3000230002300023,
    0.3000330003300033,
    0.3000430004300043,
    0.3000530005300053,
    0.3000630006300063,
    0.3000730007300073,
    0.3000830008300083,
    0.3000930009300093,
    0.3001030010300103,
    0.30011300113001127,
    0.3001230012300123,
    0.3001330013300133,
    0.3001430014300143,
    0.3001530015300153,
    0.3001630016300163,
    0.3001730017300173,
    0.3001830018300183,
    0.3001930019300193,
    0.3002030020300203,
    0.3002130021300213,
    0.3002230022300223,
    0.3002330023300233,
    0.3002430024300243,
    0.3002530025300253,
    0.3002630026300263,
    0.30027300273002727,
    0.3002830028300283,
    0.3002930029300293,
    0.30030300303003027,
    0.3003130031300313,
    0.3003230032300323,
    0.3003330033300333,
    0.3003430034300343,
    0.3003530035300353,
    0.3003630036300363,
    0.3003730037300373,
    0.3003830038300383,
    0.3003930039300393,
    0.3004030040300403,
    0.3004130041300413,
    0.3004230042300423,
    0.3004330043300433,
    0.3004430044300443,
    0.3004530045300453,
    0.30046300463004627,
    0.3004730047300473,
    0.3004830048300483,
    0.30049300493004927,
    0.3005030050300503,
    0.3005130051300513,
    0.3005230052300523,
    0.3005330053300533,
    0.3005430054300543,
    0.3005530055300553,
    0.3005630056300563,
    0.3005730057300573,
    0.3005830058300583,
    0.3005930059300593,
    0.3006030060300603,
    0.3006130061300613,
    0.3006230062300623,
    0.3006330063300633,
    0.3006430064300643,
    0.30065300653006527,
    0.3006630066300663,
    0.3006730067300673,
    0.3006830068300683,
    0.3006930069300693,
    0.3007030070300703,
    0.3007130071300713,
    0.3007230072300723,
    0.3007330073300733,
    0.3007430074300743,
    0.3007530075300753,
    0.3007630076300763,
    0.3007730077300773,
    0.3007830078300783,
    0.3007930079300793,
    0.3008030080300803,
    0.30081300813008127,
    0.3008230082300823,
    0.3008330083300833,
    0.30084300843008427,
    0.3008530085300853,
    0.3008630086300863,
    0.3008730087300873,
    0.3008830088300883,
    0.3008930089300893,
    0.3009030090300903,
    0.3009130091300913,
    0.3009230092300923,
    0.3009330093300933,
    0.3009430094300943,
    0.3009530095300953,
    0.3009630096300963,
    0.3009730097300973,
    0.3009830098300983,
    0.3009930099300993,
    0.30100301003010027,
    0.3010130101301013,
    0.3010230102301023,
    0.3010330103301033,
    0.3010430104301043,
    0.3010530105301053,
    0.3010630106301063,
    0.3010730107301073,
    0.3010830108301083,
    0.3010930109301093,
    0.3011030110301103,
    0.3011130111301113,
    0.3011230112301123,
    0.3011330113301133,
    0.3011430114301143,
    0.3011530115301153,
    0.3011630116301163,
    0.3011730117301173,
    0.3011830118301183,
    0.30119301193011927,
    0.3012030120301203,
    0.3012130121301213,
    0.3012230122301223,
    0.3012330123301233,
    0.3012430124301243,
    0.3012530125301253,
    0.3012630126301263,
    0.3012730127301273,
    0.3012830128301283,
    0.3012930129301293,
    0.3013030130301303,
    0.3013130131301313,
    0.3013230132301323,
    0.3013330133301333,
    0.3013430134301343,
    0.30135301353013527,
    0.3013630136301363,
    0.3013730137301373,
    0.30138301383013827,
    0.3013930139301393,
    0.3014030140301403,
    0.3014130141301413,
    0.3014230142301423,
    0.3014330143301433,
    0.3014430144301443,
    0.3014530145301453,
    0.3014630146301463,
    0.3014730147301473,
    0.3014830148301483,
    0.3014930149301493,
    0.3015030150301503,
    0.3015130151301513,
    0.3015230152301523,
    0.3015330153301533,
    0.30154301543015427,
    0.3015530155301553,
    0.3015630156301563,
    0.3015730157301573,
    0.3015830158301583,
    0.3015930159301593,
    0.3016030160301603,
    0.3016130161301613,
    0.3016230162301623,
    0.3016330163301633,
    0.3016430164301643,
    0.3016530165301653,
    0.3016630166301663,
    0.3016730167301673,
    0.3016830168301683,
    0.3016930169301693,
    0.3017030170301703,
    0.3017130171301713,
    0.3017230172301723,
    0.30173301733017327,
    0.3017430174301743,
    0.3017530175301753,
    0.3017630176301763,
    0.3017730177301773,
    0.3017830178301783,
    0.3017930179301793,
    0.3018030180301803,
    0.3018130181301813,
    0.3018230182301823,
    0.3018330183301833,
    0.3018430184301843,
    0.3018530185301853,
    0.3018630186301863,
    0.3018730187301873,
    0.3018830188301883,
    0.30189301893018927,
    0.3019030190301903,
    0.3019130191301913,
    0.3019230192301923,
    0.3019330193301933,
    0.3019430194301943,
    0.3019530195301953,
    0.3019630196301963,
    0.3019730197301973,
    0.3019830198301983,
    0.3019930199301993,
    0.3020030200302003,
    0.3020130201302013,
    0.3020230202302023,
    0.3020330203302033,
    0.3020430204302043,
    0.3020530205302053,
    0.3020630206302063,
    0.3020730207302073,
    0.30208302083020827,
    0.3020930209302093,
    0.3021030210302103,
    0.3021130211302113,
    0.3021230212302123,
    0.3021330213302133,
    0.3021430214302143,
    0.3021530215302153,
    0.3021630216302163,
    0.3021730217302173,
    0.3021830218302183,
    0.3021930219302193,
    0.3022030220302203,
    0.3022130221302213,
    0.3022230222302223,
    0.3022330223302233,
    0.3022430224302243,
    0.3022530225302253,
    0.3022630226302263,
    0.30227302273022727,
    0.3022830228302283,
    0.3022930229302293,
    0.3023030230302303,
    0.3023130231302313,
    0.3023230232302323,
    0.3023330233302333,
    0.3023430234302343,
    0.3023530235302353,
    0.3023630236302363,
    0.3023730237302373,
    0.3023830238302383,
    0.3023930239302393,
    0.3024030240302403,
    0.3024130241302413,
    0.3024230242302423,
    0.30243302433024327,
    0.3024430244302443,
    0.3024530245302453,
    0.3024630246302463,
    0.3024730247302473,
    0.3024830248302483,
    0.3024930249302493,
    0.3025030250302503,
    0.3025130251302513,
    0.3025230252302523,
    0.3025330253302533,
    0.3025430254302543,
    0.3025530255302553,
    0.3025630256302563,
    0.3025730257302573,
    0.3025830258302583,
    0.3025930259302593,
    0.3026030260302603,
    0.3026130261302613,
    0.30262302623026227,
    0.3026330263302633,
    0.3026430264302643,
    0.3026530265302653,
    0.3026630266302663,
    0.3026730267302673,
    0.3026830268302683,
    0.3026930269302693,
    0.3027030270302703,
    0.3027130271302713,
    0.3027230272302723,
    0.3027330273302733,
    0.3027430274302743,
    0.3027530275302753,
    0.3027630276302763,
    0.3027730277302773,
    0.3027830278302783,
    0.3027930279302793,
    0.3028030280302803,
    0.3028130281302813,
    0.3028230282302823,
    0.3028330283302833,
    0.3028430284302843,
    0.3028530285302853,
    0.3028630286302863,
    0.3028730287302873,
    0.3028830288302883,
    0.3028930289302893,
    0.3029030290302903,
    0.3029130291302913,
    0.3029230292302923,
    0.3029330293302933,
    0.3029430294302943,
    0.3029530295302953,
    0.3029630296302963,
    0.30297302973029727,
    0.3029830298302983,
    0.3029930299302993,
    0.3030030300303003,
    0.3030130301303013,
    0.3030230302303023,
    0.3030330303303033,
    0.3030430304303043,
    0.3030530305303053,
    0.3030630306303063,
    0.3030730307303073,
    0.3030830308303083,
    0.3030930309303093,
    0.3031030310303103,
    0.3031130311303113,
    0.3031230312303123,
    0.3031330313303133,
    0.3031430314303143,
    0.3031530315303153,
    0.30316303163031627,
    0.3031730317303173,
    0.3031830318303183,
    0.3031930319303193,
    0.3032030320303203,
    0.3032130321303213,
    0.3032230322303223,
    0.3032330323303233,
    0.3032430324303243,
    0.3032530325303253,
    0.3032630326303263,
    0.3032730327303273,
    0.3032830328303283,
    0.3032930329303293,
    0.3033030330303303,
    0.3033130331303313,
    0.3033230332303323,
    0.3033330333303333,
    0.3033430334303343,
    0.3033530335303353,
    0.3033630336303363,
    0.3033730337303373,
    0.3033830338303383,
    0.3033930339303393,
    0.3034030340303403,
    0.3034130341303413,
    0.3034230342303423,
    0.3034330343303433,
    0.3034430344303443,
    0.3034530345303453,
    0.3034630346303463,
    0.3034730347303473,
    0.3034830348303483,
    0.3034930349303493,
    0.3035030350303503,
    0.30351303513035127,
    0.3035230352303523,
    0.3035330353303533,
    0.3035430354303543,
    0.3035530355303553,
    0.3035630356303563,
    0.3035730357303573,
    0.3035830358303583,
    0.3035930359303593,
    0.3036030360303603,
    0.3036130361303613,
    0.3036230362303623,
    0.3036330363303633,
    0.3036430364303643,
    0.3036530365303653,
    0.3036630366303663,
    0.3036730367303673,
    0.3036830368303683,
    0.3036930369303693,
    0.30370303703037027,
    0.3037130371303713,
    0.3037230372303723,
    0.3037330373303733,
    0.3037430374303743,
    0.3037530375303753,
    0.3037630376303763,
    0.3037730377303773,
    0.3037830378303783,
    0.3037930379303793,
    0.3038030380303803,
    0.3038130381303813,
    0.3038230382303823,
    0.3038330383303833,
    0.3038430384303843,
    0.3038530385303853,
    0.3038630386303863,
    0.3038730387303873,
    0.3038830388303883,
    0.3038930389303893,
    0.3039030390303903,
    0.3039130391303913,
    0.3039230392303923,
    0.3039330393303933,
    0.3039430394303943,
    0.3039530395303953,
    0.3039630396303963,
    0.3039730397303973,
    0.3039830398303983,
    0.3039930399303993,
    0.3040030400304003,
    0.3040130401304013,
    0.3040230402304023,
    0.3040330403304033,
    0.3040430404304043,
    0.30405304053040527,
    0.3040630406304063,
    0.3040730407304073,
    0.3040830408304083,
    0.3040930409304093,
    0.3041030410304103,
    0.3041130411304113,
    0.3041230412304123,
    0.3041330413304133,
    0.3041430414304143,
    0.3041530415304153,
    0.3041630416304163,
    0.3041730417304173,
    0.3041830418304183,
    0.3041930419304193,
    0.3042030420304203,
    0.3042130421304213,
    0.3042230422304223,
    0.3042330423304233,
    0.3042430424304243,
    0.3042530425304253,
    0.3042630426304263,
    0.3042730427304273,
    0.3042830428304283,
    0.3042930429304293,
    0.3043030430304303,
    0.3043130431304313,
    0.3043230432304323,
    0.3043330433304333,
    0.3043430434304343,
    0.3043530435304353,
    0.3043630436304363,
    0.3043730437304373,
    0.3043830438304383,
    0.3043930439304393,
    0.3044030440304403,
    0.3044130441304413,
    0.3044230442304423,
    0.3044330443304433,
    0.3044430444304443,
    0.3044530445304453,
    0.3044630446304463,
    0.3044730447304473,
    0.3044830448304483,
    0.3044930449304493,
    0.3045030450304503,
    0.3045130451304513,
    0.3045230452304523,
    0.3045330453304533,
    0.3045430454304543,
    0.3045530455304553,
    0.3045630456304563,
    0.3045730457304573,
    0.3045830458304583,
    0.30459304593045927,
    0.3046030460304603,
    0.3046130461304613,
    0.3046230462304623,
    0.3046330463304633,
    0.3046430464304643,
    0.3046530465304653,
    0.3046630466304663,
    0.3046730467304673,
    0.3046830468304683,
    0.3046930469304693,
    0.3047030470304703,
    0.3047130471304713,
    0.3047230472304723,
    0.3047330473304733,
    0.3047430474304743,
    0.3047530475304753,
    0.3047630476304763,
    0.3047730477304773,
    0.3047830478304783,
    0.3047930479304793,
    0.3048030480304803,
    0.3048130481304813,
    0.3048230482304823,
    0.3048330483304833,
    0.3048430484304843,
    0.3048530485304853,
    0.3048630486304863,
    0.3048730487304873,
    0.3048830488304883,
    0.3048930489304893,
    0.3049030490304903,
    0.3049130491304913,
    0.3049230492304923,
    0.3049330493304933,
    0.3049430494304943,
    0.3049530495304953,
    0.3049630496304963,
    0.3049730497304973,
    0.3049830498304983,
    0.3049930499304993,
    0.3050030500305003,
    0.3050130501305013,
    0.3050230502305023,
    0.3050330503305033,
    0.3050430504305043,
    0.3050530505305053,
    0.3050630506305063,
    0.3050730507305073,
    0.3050830508305083,
    0.3050930509305093,
    0.3051030510305103,
    0.3051130511305113,
    0.3051230512305123,
    0.3051330513305133,
    0.3051430514305143,
    0.3051530515305153,
    0.3051630516305163,
    0.3051730517305173,
    0.3051830518305183,
    0.3051930519305193,
    0.3052030520305203,
    0.3052130521305213,
    0.3052230522305223,
    0.3052330523305233,
    0.3052430524305243,
    0.3052530525305253,
    0.3052630526305263,
    0.3052730527305273,
    0.3052830528305283,
    0.3052930529305293,
    0.3053030530305303,
    0.3053130531305313,
    0.3053230532305323,
    0.3053330533305333,
    0.3053430534305343,
    0.3053530535305353,
    0.3053630536305363,
    0.3053730537305373,
    0.3053830538305383,
    0.3053930539305393,
    0.3054030540305403,
    0.3054130541305413,
    0.3054230542305423,
    0.3054330543305433,
    0.3054430544305443,
    0.3054530545305453,
    0.3054630546305463,
    0.3054730547305473,
    0.3054830548305483,
    0.3054930549305493,
    0.3055030550305503,
    0.3055130551305513,
    0.3055230552305523,
    0.3055330553305533,
    0.3055430554305543,
    0.3055530555305553,
    0.3055630556305563,
    0.3055730557305573,
    0.3055830558305583,
    0.3055930559305593,
    0.3056030560305603,
    0.3056130561305613,
    0.3056230562305623,
    0.3056330563305633,
    0.3056430564305643,
    0.3056530565305653,
    0.3056630566305663,
    0.3056730567305673,
    0.3056830568305683,
    0.3056930569305693,
    0.3057030570305703,
    0.3057130571305713,
    0.3057230572305723,
    0.3057330573305733,
    0.3057430574305743,
    0.3057530575305753,
    0.3057630576305763,
    0.3057730577305773,
    0.3057830578305783,
    0.3057930579305793,
    0.3058030580305803,
    0.3058130581305813,
    0.3058230582305823,
    0.3058330583305833,
    0.3058430584305843,
    0.3058530585305853,
    0.3058630586305863,
    0.3058730587305873,
    0.3058830588305883,
    0.3058930589305893,
    0.3059030590305903,
    0.3059130591305913,
    0.3059230592305923,
    0.3059330593305933,
    0.3059430594305943,
    0.3059530595305953,
    0.3059630596305963,
    0.3059730597305973,
    0.3059830598305983,
    0.3059930599305993,
    0.3060030600306003,
    0.3060130601306013,
    0.30602306023060233,
    0.3060330603306033,
    0.3060430604306043,
    0.3060530605306053,
    0.3060630606306063,
    0.3060730607306073,
    0.3060830608306083,
    0.3060930609306093,
    0.3061030610306103,
    0.3061130611306113,
    0.3061230612306123,
    0.3061330613306133,
    0.3061430614306143,
    0.3061530615306153,
    0.3061630616306163,
    0.3061730617306173,
    0.3061830618306183,
    0.3061930619306193,
    0.3062030620306203,
    0.3062130621306213,
    0.3062230622306223,
    0.3062330623306233,
    0.3062430624306243,
    0.3062530625306253,
    0.3062630626306263,
    0.3062730627306273,
    0.3062830628306283,
    0.3062930629306293,
    0.3063030630306303,
    0.3063130631306313,
    0.3063230632306323,
    0.3063330633306333,
    0.3063430634306343,
    0.3063530635306353,
    0.3063630636306363,
    0.3063730637306373,
    0.3063830638306383,
    0.3063930639306393,
    0.3064030640306403,
    0.3064130641306413,
    0.3064230642306423,
    0.3064330643306433,
    0.3064430644306443,
    0.3064530645306453,
    0.3064630646306463,
    0.3064730647306473,
    0.3064830648306483,
    0.3064930649306493,
    0.3065030650306503,
    0.3065130651306513,
    0.3065230652306523,
    0.3065330653306533,
    0.3065430654306543,
    0.3065530655306553,
    0.30656306563065633,
    0.3065730657306573,
    0.3065830658306583,
    0.3065930659306593,
    0.3066030660306603,
    0.3066130661306613,
    0.3066230662306623,
    0.3066330663306633,
    0.3066430664306643,
    0.3066530665306653,
    0.3066630666306663,
    0.3066730667306673,
    0.3066830668306683,
    0.3066930669306693,
    0.3067030670306703,
    0.3067130671306713,
    0.3067230672306723,
    0.3067330673306733,
    0.3067430674306743,
    0.3067530675306753,
    0.3067630676306763,
    0.3067730677306773,
    0.3067830678306783,
    0.3067930679306793,
    0.3068030680306803,
    0.3068130681306813,
    0.3068230682306823,
    0.3068330683306833,
    0.3068430684306843,
    0.3068530685306853,
    0.3068630686306863,
    0.3068730687306873,
    0.3068830688306883,
    0.3068930689306893,
    0.3069030690306903,
    0.3069130691306913,
    0.3069230692306923,
    0.3069330693306933,
    0.3069430694306943,
    0.3069530695306953,
    0.3069630696306963,
    0.3069730697306973,
    0.3069830698306983,
    0.3069930699306993,
    0.3070030700307003,
    0.3070130701307013,
    0.3070230702307023,
    0.3070330703307033,
    0.3070430704307043,
    0.3070530705307053,
    0.3070630706307063,
    0.3070730707307073,
    0.3070830708307083,
    0.3070930709307093,
    0.30710307103071033,
    0.3071130711307113,
    0.3071230712307123,
    0.3071330713307133,
    0.3071430714307143,
    0.3071530715307153,
    0.3071630716307163,
    0.3071730717307173,
    0.3071830718307183,
    0.3071930719307193,
    0.3072030720307203,
    0.3072130721307213,
    0.3072230722307223,
    0.3072330723307233,
    0.3072430724307243,
    0.3072530725307253,
    0.3072630726307263,
    0.3072730727307273,
    0.3072830728307283,
    0.3072930729307293,
    0.3073030730307303,
    0.3073130731307313,
    0.3073230732307323,
    0.3073330733307333,
    0.3073430734307343,
    0.3073530735307353,
    0.3073630736307363,
    0.3073730737307373,
    0.3073830738307383,
    0.3073930739307393,
    0.3074030740307403,
    0.3074130741307413,
    0.3074230742307423,
    0.3074330743307433,
    0.3074430744307443,
    0.30745307453074533,
    0.3074630746307463,
    0.3074730747307473,
    0.3074830748307483,
    0.3074930749307493,
    0.3075030750307503,
    0.3075130751307513,
    0.3075230752307523,
    0.3075330753307533,
    0.3075430754307543,
    0.3075530755307553,
    0.3075630756307563,
    0.3075730757307573,
    0.3075830758307583,
    0.3075930759307593,
    0.3076030760307603,
    0.3076130761307613,
    0.3076230762307623,
    0.3076330763307633,
    0.30764307643076433,
    0.3076530765307653,
    0.3076630766307663,
    0.3076730767307673,
    0.3076830768307683,
    0.3076930769307693,
    0.3077030770307703,
    0.3077130771307713,
    0.3077230772307723,
    0.3077330773307733,
    0.3077430774307743,
    0.3077530775307753,
    0.3077630776307763,
    0.3077730777307773,
    0.3077830778307783,
    0.3077930779307793,
    0.3078030780307803,
    0.3078130781307813,
    0.3078230782307823,
    0.3078330783307833,
    0.3078430784307843,
    0.3078530785307853,
    0.3078630786307863,
    0.3078730787307873,
    0.3078830788307883,
    0.3078930789307893,
    0.3079030790307903,
    0.3079130791307913,
    0.3079230792307923,
    0.3079330793307933,
    0.3079430794307943,
    0.3079530795307953,
    0.3079630796307963,
    0.3079730797307973,
    0.3079830798307983,
    0.30799307993079933,
    0.3080030800308003,
    0.3080130801308013,
    0.3080230802308023,
    0.3080330803308033,
    0.3080430804308043,
    0.3080530805308053,
    0.3080630806308063,
    0.3080730807308073,
    0.3080830808308083,
    0.3080930809308093,
    0.3081030810308103,
    0.3081130811308113,
    0.3081230812308123,
    0.3081330813308133,
    0.3081430814308143,
    0.3081530815308153,
    0.3081630816308163,
    0.3081730817308173,
    0.30818308183081833,
    0.3081930819308193,
    0.3082030820308203,
    0.3082130821308213,
    0.3082230822308223,
    0.3082330823308233,
    0.3082430824308243,
    0.3082530825308253,
    0.3082630826308263,
    0.3082730827308273,
    0.3082830828308283,
    0.3082930829308293,
    0.3083030830308303,
    0.3083130831308313,
    0.3083230832308323,
    0.3083330833308333,
    0.30834308343083433,
    0.3083530835308353,
    0.3083630836308363,
    0.3083730837308373,
    0.3083830838308383,
    0.3083930839308393,
    0.3084030840308403,
    0.3084130841308413,
    0.3084230842308423,
    0.3084330843308433,
    0.3084430844308443,
    0.3084530845308453,
    0.3084630846308463,
    0.3084730847308473,
    0.3084830848308483,
    0.3084930849308493,
    0.3085030850308503,
    0.3085130851308513,
    0.3085230852308523,
    0.30853308533085333,
    0.3085430854308543,
    0.3085530855308553,
    0.3085630856308563,
    0.3085730857308573,
    0.3085830858308583,
    0.3085930859308593,
    0.3086030860308603,
    0.3086130861308613,
    0.3086230862308623,
    0.3086330863308633,
    0.3086430864308643,
    0.3086530865308653,
    0.3086630866308663,
    0.3086730867308673,
    0.3086830868308683,
    0.3086930869308693,
    0.3087030870308703,
    0.3087130871308713,
    0.30872308723087233,
    0.3087330873308733,
    0.3087430874308743,
    0.3087530875308753,
    0.3087630876308763,
    0.3087730877308773,
    0.3087830878308783,
    0.3087930879308793,
    0.3088030880308803,
    0.3088130881308813,
    0.3088230882308823,
    0.3088330883308833,
    0.3088430884308843,
    0.3088530885308853,
    0.3088630886308863,
    0.3088730887308873,
    0.30888308883088833,
    0.3088930889308893,
    0.3089030890308903,
    0.3089130891308913,
    0.3089230892308923,
    0.3089330893308933,
    0.3089430894308943,
    0.3089530895308953,
    0.3089630896308963,
    0.3089730897308973,
    0.3089830898308983,
    0.3089930899308993,
    0.3090030900309003,
    0.3090130901309013,
    0.3090230902309023,
    0.3090330903309033,
    0.3090430904309043,
    0.3090530905309053,
    0.3090630906309063,
    0.30907309073090733,
    0.3090830908309083,
    0.3090930909309093,
    0.3091030910309103,
    0.3091130911309113,
    0.3091230912309123,
    0.3091330913309133,
    0.3091430914309143,
    0.3091530915309153,
    0.3091630916309163,
    0.3091730917309173,
    0.3091830918309183,
    0.3091930919309193,
    0.3092030920309203,
    0.3092130921309213,
    0.3092230922309223,
    0.30923309233092333,
    0.3092430924309243,
    0.3092530925309253,
    0.30926309263092633,
    0.3092730927309273,
    0.3092830928309283,
    0.3092930929309293,
    0.3093030930309303,
    0.3093130931309313,
    0.3093230932309323,
    0.3093330933309333,
    0.3093430934309343,
    0.3093530935309353,
    0.3093630936309363,
    0.3093730937309373,
    0.3093830938309383,
    0.3093930939309393,
    0.3094030940309403,
    0.3094130941309413,
    0.30942309423094233,
    0.3094330943309433,
    0.3094430944309443,
    0.3094530945309453,
    0.3094630946309463,
    0.3094730947309473,
    0.3094830948309483,
    0.3094930949309493,
    0.3095030950309503,
    0.3095130951309513,
    0.3095230952309523,
    0.3095330953309533,
    0.3095430954309543,
    0.3095530955309553,
    0.3095630956309563,
    0.3095730957309573,
    0.3095830958309583,
    0.3095930959309593,
    0.3096030960309603,
    0.30961309613096133,
    0.3096230962309623,
    0.3096330963309633,
    0.3096430964309643,
    0.3096530965309653,
    0.3096630966309663,
    0.3096730967309673,
    0.3096830968309683,
    0.3096930969309693,
    0.3097030970309703,
    0.3097130971309713,
    0.3097230972309723,
    0.3097330973309733,
    0.3097430974309743,
    0.3097530975309753,
    0.3097630976309763,
    0.30977309773097733,
    0.3097830978309783,
    0.3097930979309793,
    0.30980309803098033,
    0.3098130981309813,
    0.3098230982309823,
    0.3098330983309833,
    0.3098430984309843,
    0.3098530985309853,
    0.3098630986309863,
    0.3098730987309873,
    0.3098830988309883,
    0.3098930989309893,
    0.3099030990309903,
    0.3099130991309913,
    0.3099230992309923,
    0.3099330993309933,
    0.3099430994309943,
    0.3099530995309953,
    0.30996309963099633,
    0.3099730997309973,
    0.3099830998309983,
    0.3099930999309993,
    0.3100031000310003,
    0.3100131001310013,
    0.3100231002310023,
    0.3100331003310033,
    0.3100431004310043,
    0.3100531005310053,
    0.3100631006310063,
    0.3100731007310073,
    0.3100831008310083,
    0.3100931009310093,
    0.3101031010310103,
    0.3101131011310113,
    0.31012310123101233,
    0.3101331013310133,
    0.3101431014310143,
    0.31015310153101533,
    0.3101631016310163,
    0.3101731017310173,
    0.3101831018310183,
    0.3101931019310193,
    0.3102031020310203,
    0.3102131021310213,
    0.3102231022310223,
    0.3102331023310233,
    0.3102431024310243,
    0.3102531025310253,
    0.3102631026310263,
    0.3102731027310273,
    0.3102831028310283,
    0.3102931029310293,
    0.3103031030310303,
    0.31031310313103133,
    0.3103231032310323,
    0.3103331033310333,
    0.31034310343103433,
    0.3103531035310353,
    0.3103631036310363,
    0.3103731037310373,
    0.3103831038310383,
    0.3103931039310393,
    0.3104031040310403,
    0.3104131041310413,
    0.3104231042310423,
    0.3104331043310433,
    0.3104431044310443,
    0.3104531045310453,
    0.3104631046310463,
    0.3104731047310473,
    0.3104831048310483,
    0.3104931049310493,
    0.31050310503105033,
    0.3105131051310513,
    0.3105231052310523,
    0.3105331053310533,
    0.3105431054310543,
    0.3105531055310553,
    0.3105631056310563,
    0.3105731057310573,
    0.3105831058310583,
    0.3105931059310593,
    0.3106031060310603,
    0.3106131061310613,
    0.3106231062310623,
    0.3106331063310633,
    0.3106431064310643,
    0.3106531065310653,
    0.31066310663106633,
    0.3106731067310673,
    0.3106831068310683,
    0.31069310693106933,
    0.3107031070310703,
    0.3107131071310713,
    0.3107231072310723,
    0.3107331073310733,
    0.3107431074310743,
    0.3107531075310753,
    0.3107631076310763,
    0.3107731077310773,
    0.3107831078310783,
    0.3107931079310793,
    0.3108031080310803,
    0.3108131081310813,
    0.3108231082310823,
    0.3108331083310833,
    0.3108431084310843,
    0.31085310853108533,
    0.3108631086310863,
    0.3108731087310873,
    0.31088310883108833,
    0.3108931089310893,
    0.3109031090310903,
    0.3109131091310913,
    0.3109231092310923,
    0.3109331093310933,
    0.3109431094310943,
    0.3109531095310953,
    0.3109631096310963,
    0.3109731097310973,
    0.3109831098310983,
    0.3109931099310993,
    0.3110031100311003,
    0.3110131101311013,
    0.3110231102311023,
    0.3110331103311033,
    0.31104311043110433,
    0.3110531105311053,
    0.3110631106311063,
    0.3110731107311073,
    0.3110831108311083,
    0.3110931109311093,
    0.3111031110311103,
    0.3111131111311113,
    0.3111231112311123,
    0.3111331113311133,
    0.3111431114311143,
    0.3111531115311153,
    0.3111631116311163,
    0.3111731117311173,
    0.3111831118311183,
    0.3111931119311193,
    0.31120311203112033,
    0.3112131121311213,
    0.3112231122311223,
    0.31123311233112333,
    0.3112431124311243,
    0.3112531125311253,
    0.3112631126311263,
    0.3112731127311273,
    0.3112831128311283,
    0.3112931129311293,
    0.3113031130311303,
    0.3113131131311313,
    0.3113231132311323,
    0.3113331133311333,
    0.3113431134311343,
    0.3113531135311353,
    0.3113631136311363,
    0.3113731137311373,
    0.3113831138311383,
    0.31139311393113933,
    0.3114031140311403,
    0.3114131141311413,
    0.31142311423114233,
    0.3114331143311433,
    0.3114431144311443,
    0.3114531145311453,
    0.3114631146311463,
    0.3114731147311473,
    0.3114831148311483,
    0.3114931149311493,
    0.3115031150311503,
    0.3115131151311513,
    0.3115231152311523,
    0.3115331153311533,
    0.3115431154311543,
    0.31155311553115533,
    0.3115631156311563,
    0.3115731157311573,
    0.31158311583115833,
    0.3115931159311593,
    0.3116031160311603,
    0.3116131161311613,
    0.3116231162311623,
    0.3116331163311633,
    0.3116431164311643,
    0.3116531165311653,
    0.3116631166311663,
    0.3116731167311673,
    0.3116831168311683,
    0.3116931169311693,
    0.3117031170311703,
    0.3117131171311713,
    0.3117231172311723,
    0.3117331173311733,
    0.31174311743117433,
    0.3117531175311753,
    0.3117631176311763,
    0.31177311773117733,
    0.3117831178311783,
    0.3117931179311793,
    0.3118031180311803,
    0.3118131181311813,
    0.3118231182311823,
    0.3118331183311833,
    0.3118431184311843,
    0.3118531185311853,
    0.3118631186311863,
    0.3118731187311873,
    0.3118831188311883,
    0.3118931189311893,
    0.3119031190311903,
    0.3119131191311913,
    0.3119231192311923,
    0.31193311933119333,
    0.3119431194311943,
    0.3119531195311953,
    0.31196311963119633,
    0.3119731197311973,
    0.3119831198311983,
    0.3119931199311993,
    0.3120031200312003,
    0.3120131201312013,
    0.3120231202312023,
    0.3120331203312033,
    0.3120431204312043,
    0.3120531205312053,
    0.3120631206312063,
    0.3120731207312073,
    0.3120831208312083,
    0.31209312093120933,
    0.3121031210312103,
    0.3121131211312113,
    0.31212312123121233,
    0.3121331213312133,
    0.3121431214312143,
    0.3121531215312153,
    0.3121631216312163,
    0.3121731217312173,
    0.3121831218312183,
    0.3121931219312193,
    0.3122031220312203,
    0.3122131221312213,
    0.3122231222312223,
    0.3122331223312233,
    0.3122431224312243,
    0.3122531225312253,
    0.3122631226312263,
    0.3122731227312273,
    0.31228312283122833,
    0.3122931229312293,
    0.3123031230312303,
    0.31231312313123133,
    0.3123231232312323,
    0.3123331233312333,
    0.3123431234312343,
    0.3123531235312353,
    0.3123631236312363,
    0.3123731237312373,
    0.3123831238312383,
    0.3123931239312393,
    0.3124031240312403,
    0.3124131241312413,
    0.3124231242312423,
    0.3124331243312433,
    0.31244312443124433,
    0.3124531245312453,
    0.3124631246312463,
    0.31247312473124733,
    0.3124831248312483,
    0.3124931249312493,
    0.31250312503125033,
    0.3125131251312513,
    0.3125231252312523,
    0.3125331253312533,
    0.3125431254312543,
    0.3125531255312553,
    0.3125631256312563,
    0.3125731257312573,
    0.3125831258312583,
    0.3125931259312593,
    0.3126031260312603,
    0.3126131261312613,
    0.3126231262312623,
    0.31263312633126333,
    0.3126431264312643,
    0.3126531265312653,
    0.31266312663126633,
    0.3126731267312673,
    0.3126831268312683,
    0.3126931269312693,
    0.3127031270312703,
    0.3127131271312713,
    0.3127231272312723,
    0.3127331273312733,
    0.3127431274312743,
    0.3127531275312753,
    0.3127631276312763,
    0.3127731277312773,
    0.3127831278312783,
    0.3127931279312793,
    0.3128031280312803,
    0.3128131281312813,
    0.31282312823128233,
    0.3128331283312833,
    0.3128431284312843,
    0.31285312853128533,
    0.3128631286312863,
    0.3128731287312873,
    0.3128831288312883,
    0.3128931289312893,
    0.3129031290312903,
    0.3129131291312913,
    0.3129231292312923,
    0.3129331293312933,
    0.3129431294312943,
    0.3129531295312953,
    0.3129631296312963,
    0.3129731297312973,
    0.31298312983129833,
    0.3129931299312993,
    0.3130031300313003,
    0.31301313013130133,
    0.3130231302313023,
    0.3130331303313033,
    0.31304313043130433,
    0.3130531305313053,
    0.3130631306313063,
    0.3130731307313073,
    0.3130831308313083,
    0.3130931309313093,
    0.3131031310313103,
    0.3131131311313113,
    0.3131231312313123,
    0.3131331313313133,
    0.3131431314313143,
    0.3131531315313153,
    0.3131631316313163,
    0.31317313173131733,
    0.3131831318313183,
    0.3131931319313193,
    0.31320313203132033,
    0.3132131321313213,
    0.3132231322313223,
    0.3132331323313233,
    0.3132431324313243,
    0.3132531325313253,
    0.3132631326313263,
    0.3132731327313273,
    0.3132831328313283,
    0.3132931329313293,
    0.3133031330313303,
    0.3133131331313313,
    0.3133231332313323,
    0.31333313333133334,
    0.3133431334313343,
    0.3133531335313353,
    0.31336313363133633,
    0.3133731337313373,
    0.3133831338313383,
    0.31339313393133933,
    0.3134031340313403,
    0.3134131341313413,
    0.3134231342313423,
    0.3134331343313433,
    0.3134431344313443,
    0.3134531345313453,
    0.3134631346313463,
    0.3134731347313473,
    0.3134831348313483,
    0.3134931349313493,
    0.3135031350313503,
    0.3135131351313513,
    0.31352313523135233,
    0.3135331353313533,
    0.3135431354313543,
    0.31355313553135533,
    0.3135631356313563,
    0.3135731357313573,
    0.31358313583135833,
    0.3135931359313593,
    0.3136031360313603,
    0.3136131361313613,
    0.3136231362313623,
    0.3136331363313633,
    0.3136431364313643,
    0.3136531365313653,
    0.3136631366313663,
    0.3136731367313673,
    0.3136831368313683,
    0.3136931369313693,
    0.3137031370313703,
    0.31371313713137133,
    0.3137231372313723,
    0.3137331373313733,
    0.31374313743137433,
    0.3137531375313753,
    0.3137631376313763,
    0.3137731377313773,
    0.3137831378313783,
    0.3137931379313793,
    0.3138031380313803,
    0.3138131381313813,
    0.3138231382313823,
    0.3138331383313833,
    0.3138431384313843,
    0.3138531385313853,
    0.3138631386313863,
    0.31387313873138734,
    0.3138831388313883,
    0.3138931389313893,
    0.31390313903139033,
    0.3139131391313913,
    0.3139231392313923,
    0.31393313933139333,
    0.3139431394313943,
    0.3139531395313953,
    0.3139631396313963,
    0.3139731397313973,
    0.3139831398313983,
    0.3139931399313993,
    0.3140031400314003,
    0.3140131401314013,
    0.3140231402314023,
    0.3140331403314033,
    0.3140431404314043,
    0.3140531405314053,
    0.31406314063140633,
    0.3140731407314073,
    0.3140831408314083,
    0.31409314093140933,
    0.3141031410314103,
    0.3141131411314113,
    0.31412314123141233,
    0.3141331413314133,
    0.3141431414314143,
    0.3141531415314153,
    0.3141631416314163,
    0.3141731417314173,
    0.3141831418314183,
    0.3141931419314193,
    0.3142031420314203,
    0.3142131421314213,
    0.31422314223142234,
    0.3142331423314233,
    0.3142431424314243,
    0.31425314253142533,
    0.3142631426314263,
    0.3142731427314273,
    0.31428314283142833,
    0.3142931429314293,
    0.3143031430314303,
    0.3143131431314313,
    0.3143231432314323,
    0.3143331433314333,
    0.3143431434314343,
    0.3143531435314353,
    0.3143631436314363,
    0.3143731437314373,
    0.3143831438314383,
    0.3143931439314393,
    0.3144031440314403,
    0.31441314413144134,
    0.3144231442314423,
    0.3144331443314433,
    0.31444314443144433,
    0.3144531445314453,
    0.3144631446314463,
    0.31447314473144733,
    0.3144831448314483,
    0.3144931449314493,
    0.3145031450314503,
    0.3145131451314513,
    0.3145231452314523,
    0.3145331453314533,
    0.3145431454314543,
    0.3145531455314553,
    0.3145631456314563,
    0.3145731457314573,
    0.3145831458314583,
    0.3145931459314593,
    0.31460314603146033,
    0.3146131461314613,
    0.3146231462314623,
    0.31463314633146333,
    0.3146431464314643,
    0.3146531465314653,
    0.31466314663146633,
    0.3146731467314673,
    0.3146831468314683,
    0.3146931469314693,
    0.3147031470314703,
    0.3147131471314713,
    0.3147231472314723,
    0.3147331473314733,
    0.3147431474314743,
    0.3147531475314753,
    0.31476314763147634,
    0.3147731477314773,
    0.3147831478314783,
    0.31479314793147933,
    0.3148031480314803,
    0.3148131481314813,
    0.31482314823148233,
    0.3148331483314833,
    0.3148431484314843,
    0.3148531485314853,
    0.3148631486314863,
    0.3148731487314873,
    0.3148831488314883,
    0.3148931489314893,
    0.3149031490314903,
    0.3149131491314913,
    0.3149231492314923,
    0.3149331493314933,
    0.3149431494314943,
    0.31495314953149534,
    0.3149631496314963,
    0.3149731497314973,
    0.31498314983149833,
    0.3149931499314993,
    0.3150031500315003,
    0.31501315013150133,
    0.3150231502315023,
    0.3150331503315033,
    0.3150431504315043,
    0.3150531505315053,
    0.3150631506315063,
    0.3150731507315073,
    0.3150831508315083,
    0.3150931509315093,
    0.3151031510315103,
    0.3151131511315113,
    0.3151231512315123,
    0.3151331513315133,
    0.31514315143151433,
    0.3151531515315153,
    0.3151631516315163,
    0.31517315173151733,
    0.3151831518315183,
    0.3151931519315193,
    0.31520315203152033,
    0.3152131521315213,
    0.3152231522315223,
    0.3152331523315233,
    0.3152431524315243,
    0.3152531525315253,
    0.3152631526315263,
    0.3152731527315273,
    0.3152831528315283,
    0.3152931529315293,
    0.31530315303153034,
    0.3153131531315313,
    0.3153231532315323,
    0.31533315333153333,
    0.3153431534315343,
    0.3153531535315353,
    0.31536315363153633,
    0.3153731537315373,
    0.3153831538315383,
    0.3153931539315393,
    0.3154031540315403,
    0.3154131541315413,
    0.3154231542315423,
    0.3154331543315433,
    0.3154431544315443,
    0.3154531545315453,
    0.3154631546315463,
    0.3154731547315473,
    0.3154831548315483,
    0.31549315493154934,
    0.3155031550315503,
    0.3155131551315513,
    0.31552315523155233,
    0.3155331553315533,
    0.3155431554315543,
    0.31555315553155533,
    0.3155631556315563,
    0.3155731557315573,
    0.3155831558315583,
    0.3155931559315593,
    0.3156031560315603,
    0.3156131561315613,
    0.3156231562315623,
    0.3156331563315633,
    0.3156431564315643,
    0.31565315653156534,
    0.3156631566315663,
    0.3156731567315673,
    0.31568315683156833,
    0.3156931569315693,
    0.3157031570315703,
    0.31571315713157133,
    0.3157231572315723,
    0.3157331573315733,
    0.31574315743157433,
    0.3157531575315753,
    0.3157631576315763,
    0.3157731577315773,
    0.3157831578315783,
    0.3157931579315793,
    0.3158031580315803,
    0.3158131581315813,
    0.3158231582315823,
    0.3158331583315833,
    0.31584315843158434,
    0.3158531585315853,
    0.3158631586315863,
    0.31587315873158733,
    0.3158831588315883,
    0.3158931589315893,
    0.31590315903159033,
    0.3159131591315913,
    0.3159231592315923,
    0.3159331593315933,
    0.3159431594315943,
    0.3159531595315953,
    0.3159631596315963,
    0.3159731597315973,
    0.3159831598315983,
    0.3159931599315993,
    0.3160031600316003,
    0.3160131601316013,
    0.3160231602316023,
    0.31603316033160334,
    0.3160431604316043,
    0.3160531605316053,
    0.31606316063160633,
    0.3160731607316073,
    0.3160831608316083,
    0.31609316093160933,
    0.3161031610316103,
    0.3161131611316113,
    0.3161231612316123,
    0.3161331613316133,
    0.3161431614316143,
    0.3161531615316153,
    0.3161631616316163,
    0.3161731617316173,
    0.3161831618316183,
    0.31619316193161934,
    0.3162031620316203,
    0.3162131621316213,
    0.31622316223162233,
    0.3162331623316233,
    0.3162431624316243,
    0.31625316253162533,
    0.3162631626316263,
    0.3162731627316273,
    0.31628316283162833,
    0.3162931629316293,
    0.3163031630316303,
    0.3163131631316313,
    0.3163231632316323,
    0.3163331633316333,
    0.3163431634316343,
    0.3163531635316353,
    0.3163631636316363,
    0.3163731637316373,
    0.31638316383163834,
    0.3163931639316393,
    0.3164031640316403,
    0.31641316413164133,
    0.3164231642316423,
    0.3164331643316433,
    0.31644316443164433,
    0.3164531645316453,
    0.3164631646316463,
    0.3164731647316473,
    0.3164831648316483,
    0.3164931649316493,
    0.3165031650316503,
    0.3165131651316513,
    0.3165231652316523,
    0.3165331653316533,
    0.31654316543165434,
    0.3165531655316553,
    0.3165631656316563,
    0.31657316573165734,
    0.3165831658316583,
    0.3165931659316593,
    0.31660316603166033,
    0.3166131661316613,
    0.3166231662316623,
    0.31663316633166333,
    0.3166431664316643,
    0.3166531665316653,
    0.3166631666316663,
    0.3166731667316673,
    0.3166831668316683,
    0.3166931669316693,
    0.3167031670316703,
    0.3167131671316713,
    0.3167231672316723,
    0.31673316733167334,
    0.3167431674316743,
    0.3167531675316753,
    0.31676316763167633,
    0.3167731677316773,
    0.3167831678316783,
    0.31679316793167933,
    0.3168031680316803,
    0.3168131681316813,
    0.31682316823168233,
    0.3168331683316833,
    0.3168431684316843,
    0.3168531685316853,
    0.3168631686316863,
    0.3168731687316873,
    0.3168831688316883,
    0.3168931689316893,
    0.3169031690316903,
    0.3169131691316913,
    0.31692316923169234,
    0.3169331693316933,
    0.3169431694316943,
    0.31695316953169533,
    0.3169631696316963,
    0.3169731697316973,
    0.31698316983169833,
    0.3169931699316993,
    0.3170031700317003,
    0.3170131701317013,
    0.3170231702317023,
    0.3170331703317033,
    0.3170431704317043,
    0.3170531705317053,
    0.3170631706317063,
    0.3170731707317073,
    0.31708317083170834,
    0.3170931709317093,
    0.3171031710317103,
    0.31711317113171134,
    0.3171231712317123,
    0.3171331713317133,
    0.31714317143171433,
    0.3171531715317153,
    0.3171631716317163,
    0.31717317173171733,
    0.3171831718317183,
    0.3171931719317193,
    0.3172031720317203,
    0.3172131721317213,
    0.3172231722317223,
    0.3172331723317233,
    0.3172431724317243,
    0.3172531725317253,
    0.3172631726317263,
    0.31727317273172734,
    0.3172831728317283,
    0.3172931729317293,
    0.31730317303173033,
    0.3173131731317313,
    0.3173231732317323,
    0.31733317333173333,
    0.3173431734317343,
    0.3173531735317353,
    0.31736317363173633,
    0.3173731737317373,
    0.3173831738317383,
    0.3173931739317393,
    0.3174031740317403,
    0.3174131741317413,
    0.3174231742317423,
    0.31743317433174334,
    0.3174431744317443,
    0.3174531745317453,
    0.31746317463174634,
    0.3174731747317473,
    0.3174831748317483,
    0.31749317493174933,
    0.3175031750317503,
    0.3175131751317513,
    0.31752317523175233,
    0.3175331753317533,
    0.3175431754317543,
    0.3175531755317553,
    0.3175631756317563,
    0.3175731757317573,
    0.3175831758317583,
    0.3175931759317593,
    0.3176031760317603,
    0.3176131761317613,
    0.31762317623176234,
    0.3176331763317633,
    0.3176431764317643,
    0.31765317653176534,
    0.3176631766317663,
    0.3176731767317673,
    0.31768317683176833,
    0.3176931769317693,
    0.3177031770317703,
    0.31771317713177133,
    0.3177231772317723,
    0.3177331773317733,
    0.3177431774317743,
    0.3177531775317753,
    0.3177631776317763,
    0.3177731777317773,
    0.3177831778317783,
    0.3177931779317793,
    0.3178031780317803,
    0.31781317813178134,
    0.3178231782317823,
    0.3178331783317833,
    0.31784317843178433,
    0.3178531785317853,
    0.3178631786317863,
    0.31787317873178733,
    0.3178831788317883,
    0.3178931789317893,
    0.31790317903179033,
    0.3179131791317913,
    0.3179231792317923,
    0.3179331793317933,
    0.3179431794317943,
    0.3179531795317953,
    0.3179631796317963,
    0.31797317973179734,
    0.3179831798317983,
    0.3179931799317993,
    0.31800318003180034,
    0.3180131801318013,
    0.3180231802318023,
    0.31803318033180333,
    0.3180431804318043,
    0.3180531805318053,
    0.31806318063180633,
    0.3180731807318073,
    0.3180831808318083,
    0.3180931809318093,
    0.3181031810318103,
    0.3181131811318113,
    0.3181231812318123,
    0.3181331813318133,
    0.3181431814318143,
    0.3181531815318153,
    0.31816318163181634,
    0.3181731817318173,
    0.3181831818318183,
    0.31819318193181934,
    0.3182031820318203,
    0.3182131821318213,
    0.31822318223182233,
    0.3182331823318233,
    0.3182431824318243,
    0.31825318253182533,
    0.3182631826318263,
    0.3182731827318273,
    0.3182831828318283,
    0.3182931829318293,
    0.3183031830318303,
    0.3183131831318313,
    0.3183231832318323,
    0.3183331833318333,
    0.3183431834318343,
    0.31835318353183534,
    0.3183631836318363,
    0.3183731837318373,
    0.31838318383183833,
    0.3183931839318393,
    0.3184031840318403,
    0.31841318413184133,
    0.3184231842318423,
    0.3184331843318433,
    0.31844318443184433,
    0.3184531845318453,
    0.3184631846318463,
    0.3184731847318473,
    0.3184831848318483,
    0.3184931849318493,
    0.3185031850318503,
    0.31851318513185134,
    0.3185231852318523,
    0.3185331853318533,
    0.31854318543185434,
    0.3185531855318553,
    0.3185631856318563,
    0.31857318573185733,
    0.3185831858318583,
    0.3185931859318593,
    0.31860318603186033,
    0.3186131861318613,
    0.3186231862318623,
    0.3186331863318633,
    0.3186431864318643,
    0.3186531865318653,
    0.3186631866318663,
    0.3186731867318673,
    0.3186831868318683,
    0.3186931869318693,
    0.31870318703187034,
    0.3187131871318713,
    0.3187231872318723,
    0.31873318733187334,
    0.3187431874318743,
    0.3187531875318753,
    0.31876318763187633,
    0.3187731877318773,
    0.3187831878318783,
    0.31879318793187933,
    0.3188031880318803,
    0.3188131881318813,
    0.3188231882318823,
    0.3188331883318833,
    0.3188431884318843,
    0.3188531885318853,
    0.31886318863188634,
    0.3188731887318873,
    0.3188831888318883,
    0.31889318893188934,
    0.3189031890318903,
    0.3189131891318913,
    0.31892318923189233,
    0.3189331893318933,
    0.3189431894318943,
    0.31895318953189533,
    0.3189631896318963,
    0.3189731897318973,
    0.31898318983189833,
    0.3189931899318993,
    0.3190031900319003,
    0.3190131901319013,
    0.3190231902319023,
    0.3190331903319033,
    0.3190431904319043,
    0.31905319053190534,
    0.3190631906319063,
    0.3190731907319073,
    0.31908319083190834,
    0.3190931909319093,
    0.3191031910319103,
    0.31911319113191133,
    0.3191231912319123,
    0.3191331913319133,
    0.31914319143191433,
    0.3191531915319153,
    0.3191631916319163,
    0.3191731917319173,
    0.3191831918319183,
    0.3191931919319193,
    0.3192031920319203,
    0.3192131921319213,
    0.3192231922319223,
    0.3192331923319233,
    0.31924319243192434,
    0.3192531925319253,
    0.3192631926319263,
    0.31927319273192734,
    0.3192831928319283,
    0.3192931929319293,
    0.31930319303193033,
    0.3193131931319313,
    0.3193231932319323,
    0.31933319333193333,
    0.3193431934319343,
    0.3193531935319353,
    0.3193631936319363,
    0.3193731937319373,
    0.3193831938319383,
    0.3193931939319393,
    0.31940319403194034,
    0.3194131941319413,
    0.3194231942319423,
    0.31943319433194334,
    0.3194431944319443,
    0.3194531945319453,
    0.31946319463194633,
    0.3194731947319473,
    0.3194831948319483,
    0.31949319493194933,
    0.3195031950319503,
    0.3195131951319513,
    0.31952319523195233,
    0.3195331953319533,
    0.3195431954319543,
    0.3195531955319553,
    0.3195631956319563,
    0.3195731957319573,
    0.3195831958319583,
    0.31959319593195934,
    0.3196031960319603,
    0.3196131961319613,
    0.31962319623196234,
    0.3196331963319633,
    0.3196431964319643,
    0.31965319653196533,
    0.3196631966319663,
    0.3196731967319673,
    0.31968319683196833,
    0.3196931969319693,
    0.3197031970319703,
    0.3197131971319713,
    0.3197231972319723,
    0.3197331973319733,
    0.3197431974319743,
    0.31975319753197534,
    0.3197631976319763,
    0.3197731977319773,
    0.31978319783197834,
    0.3197931979319793,
    0.3198031980319803,
    0.31981319813198134,
    0.3198231982319823,
    0.3198331983319833,
    0.31984319843198433,
    0.3198531985319853,
    0.3198631986319863,
    0.31987319873198733,
    0.3198831988319883,
    0.3198931989319893,
    0.3199031990319903,
    0.3199131991319913,
    0.3199231992319923,
    0.3199331993319933,
    0.31994319943199434,
    0.3199531995319953,
    0.3199631996319963,
    0.31997319973199734,
    0.3199831998319983,
    0.3199931999319993,
    0.32000320003200033,
    0.3200132001320013,
    0.3200232002320023,
    0.32003320033200333,
    0.3200432004320043,
    0.3200532005320053,
    0.32006320063200633,
    0.3200732007320073,
    0.3200832008320083,
    0.3200932009320093,
    0.3201032010320103,
    0.3201132011320113,
    0.3201232012320123,
    0.32013320133201334,
    0.3201432014320143,
    0.3201532015320153,
    0.32016320163201634,
    0.3201732017320173,
    0.3201832018320183,
    0.32019320193201933,
    0.3202032020320203,
    0.3202132021320213,
    0.32022320223202233,
    0.3202332023320233,
    0.3202432024320243,
    0.3202532025320253,
    0.3202632026320263,
    0.3202732027320273,
    0.3202832028320283,
    0.32029320293202934,
    0.3203032030320303,
    0.3203132031320313,
    0.32032320323203234,
    0.3203332033320333,
    0.3203432034320343,
    0.32035320353203534,
    0.3203632036320363,
    0.3203732037320373,
    0.32038320383203833,
    0.3203932039320393,
    0.3204032040320403,
    0.32041320413204133,
    0.3204232042320423,
    0.3204332043320433,
    0.3204432044320443,
    0.3204532045320453,
    0.3204632046320463,
    0.3204732047320473,
    0.32048320483204834,
    0.3204932049320493,
    0.3205032050320503,
    0.32051320513205134,
    0.3205232052320523,
    0.3205332053320533,
    0.32054320543205433,
    0.3205532055320553,
    0.3205632056320563,
    0.32057320573205733,
    0.3205832058320583,
    0.3205932059320593,
    0.32060320603206033,
    0.3206132061320613,
    0.3206232062320623,
    0.3206332063320633,
    0.32064320643206434,
    0.3206532065320653,
    0.3206632066320663,
    0.32067320673206734,
    0.3206832068320683,
    0.3206932069320693,
    0.32070320703207034,
    0.3207132071320713,
    0.3207232072320723,
    0.32073320733207333,
    0.3207432074320743,
    0.3207532075320753,
    0.32076320763207633,
    0.3207732077320773,
    0.3207832078320783,
    0.3207932079320793,
    0.3208032080320803,
    0.3208132081320813,
    0.3208232082320823,
    0.32083320833208334,
    0.3208432084320843,
    0.3208532085320853,
    0.32086320863208634,
    0.3208732087320873,
    0.3208832088320883,
    0.32089320893208934,
    0.3209032090320903,
    0.3209132091320913,
    0.32092320923209233,
    0.3209332093320933,
    0.3209432094320943,
    0.32095320953209533,
    0.3209632096320963,
    0.3209732097320973,
    0.3209832098320983,
    0.3209932099320993,
    0.3210032100321003,
    0.3210132101321013,
    0.32102321023210234,
    0.3210332103321033,
    0.3210432104321043,
    0.32105321053210534,
    0.3210632106321063,
    0.3210732107321073,
    0.32108321083210833,
    0.3210932109321093,
    0.3211032110321103,
    0.32111321113211133,
    0.3211232112321123,
    0.3211332113321133,
    0.32114321143211433,
    0.3211532115321153,
    0.3211632116321163,
    0.3211732117321173,
    0.32118321183211834,
    0.3211932119321193,
    0.3212032120321203,
    0.32121321213212134,
    0.3212232122321223,
    0.3212332123321233,
    0.32124321243212434,
    0.3212532125321253,
    0.3212632126321263,
    0.32127321273212733,
    0.3212832128321283,
    0.3212932129321293,
    0.32130321303213033,
    0.3213132131321313,
    0.3213232132321323,
    0.3213332133321333,
    0.3213432134321343,
    0.3213532135321353,
    0.3213632136321363,
    0.32137321373213734,
    0.3213832138321383,
    0.3213932139321393,
    0.32140321403214034,
    0.3214132141321413,
    0.3214232142321423,
    0.32143321433214334,
    0.3214432144321443,
    0.3214532145321453,
    0.32146321463214633,
    0.3214732147321473,
    0.3214832148321483,
    0.32149321493214933,
    0.3215032150321503,
    0.3215132151321513,
    0.3215232152321523,
    0.32153321533215334,
    0.3215432154321543,
    0.3215532155321553,
    0.32156321563215634,
    0.3215732157321573,
    0.3215832158321583,
    0.32159321593215934,
    0.3216032160321603,
    0.3216132161321613,
    0.32162321623216233,
    0.3216332163321633,
    0.3216432164321643,
    0.32165321653216533,
    0.3216632166321663,
    0.3216732167321673,
    0.32168321683216833,
    0.3216932169321693,
    0.3217032170321703,
    0.3217132171321713,
    0.32172321723217234,
    0.3217332173321733,
    0.3217432174321743,
    0.32175321753217534,
    0.3217632176321763,
    0.3217732177321773,
    0.32178321783217834,
    0.3217932179321793,
    0.3218032180321803,
    0.32181321813218133,
    0.3218232182321823,
    0.3218332183321833,
    0.32184321843218433,
    0.3218532185321853,
    0.3218632186321863,
    0.3218732187321873,
    0.3218832188321883,
    0.3218932189321893,
    0.3219032190321903,
    0.32191321913219134,
    0.3219232192321923,
    0.3219332193321933,
    0.32194321943219434,
    0.3219532195321953,
    0.3219632196321963,
    0.32197321973219734,
    0.3219832198321983,
    0.3219932199321993,
    0.32200322003220033,
    0.3220132201322013,
    0.3220232202322023,
    0.32203322033220333,
    0.3220432204322043,
    0.3220532205322053,
    0.3220632206322063,
    0.32207322073220734,
    0.3220832208322083,
    0.3220932209322093,
    0.32210322103221034,
    0.3221132211322113,
    0.3221232212322123,
    0.32213322133221334,
    0.3221432214322143,
    0.3221532215322153,
    0.32216322163221633,
    0.3221732217322173,
    0.3221832218322183,
    0.32219322193221933,
    0.3222032220322203,
    0.3222132221322213,
    0.32222322223222233,
    0.3222332223322233,
    0.3222432224322243,
    0.3222532225322253,
    0.32226322263222634,
    0.3222732227322273,
    0.3222832228322283,
    0.32229322293222934,
    0.3223032230322303,
    0.3223132231322313,
    0.32232322323223234,
    0.3223332233322333,
    0.3223432234322343,
    0.32235322353223533,
    0.3223632236322363,
    0.3223732237322373,
    0.32238322383223833,
    0.3223932239322393,
    0.3224032240322403,
    0.3224132241322413,
    0.3224232242322423,
    0.3224332243322433,
    0.3224432244322443,
    0.32245322453224534,
    0.3224632246322463,
    0.3224732247322473,
    0.32248322483224834,
    0.3224932249322493,
    0.3225032250322503,
    0.32251322513225134,
    0.3225232252322523,
    0.3225332253322533,
    0.32254322543225433,
    0.3225532255322553,
    0.3225632256322563,
    0.32257322573225733,
    0.3225832258322583,
    0.3225932259322593,
    0.3226032260322603,
    0.32261322613226134,
    0.3226232262322623,
    0.3226332263322633,
    0.32264322643226434,
    0.3226532265322653,
    0.3226632266322663,
    0.32267322673226734,
    0.3226832268322683,
    0.3226932269322693,
    0.32270322703227033,
    0.3227132271322713,
    0.3227232272322723,
    0.32273322733227333,
    0.3227432274322743,
    0.3227532275322753,
    0.32276322763227633,
    0.3227732277322773,
    0.3227832278322783,
    0.3227932279322793,
    0.32280322803228034,
    0.3228132281322813,
    0.3228232282322823,
    0.32283322833228334,
    0.3228432284322843,
    0.3228532285322853,
    0.32286322863228634,
    0.3228732287322873,
    0.3228832288322883,
    0.32289322893228933,
    0.3229032290322903,
    0.3229132291322913,
    0.32292322923229233,
    0.3229332293322933,
    0.3229432294322943,
    0.3229532295322953,
    0.32296322963229634,
    0.3229732297322973,
    0.3229832298322983,
    0.32299322993229934,
    0.3230032300323003,
    0.3230132301323013,
    0.32302323023230234,
    0.3230332303323033,
    0.3230432304323043,
    0.32305323053230534,
    0.3230632306323063,
    0.3230732307323073,
    0.32308323083230833,
    0.3230932309323093,
    0.3231032310323103,
    0.32311323113231133,
    0.3231232312323123,
    0.3231332313323133,
    0.3231432314323143,
    0.32315323153231534,
    0.3231632316323163,
    0.3231732317323173,
    0.32318323183231834,
    0.3231932319323193,
    0.3232032320323203,
    0.32321323213232134,
    0.3232232322323223,
    0.3232332323323233,
    0.32324323243232433,
    0.3232532325323253,
    0.3232632326323263,
    0.32327323273232733,
    0.3232832328323283,
    0.3232932329323293,
    0.32330323303233033,
    0.3233132331323313,
    0.3233232332323323,
    0.3233332333323333,
    0.32334323343233434,
    0.3233532335323353,
    0.3233632336323363,
    0.32337323373233734,
    0.3233832338323383,
    0.3233932339323393,
    0.32340323403234034,
    0.3234132341323413,
    0.3234232342323423,
    0.32343323433234333,
    0.3234432344323443,
    0.3234532345323453,
    0.32346323463234633,
    0.3234732347323473,
    0.3234832348323483,
    0.3234932349323493,
    0.32350323503235034,
    0.3235132351323513,
    0.3235232352323523,
    0.32353323533235334,
    0.3235432354323543,
    0.3235532355323553,
    0.32356323563235634,
    0.3235732357323573,
    0.3235832358323583,
    0.32359323593235934,
    0.3236032360323603,
    0.3236132361323613,
    0.32362323623236233,
    0.3236332363323633,
    0.3236432364323643,
    0.32365323653236533,
    0.3236632366323663,
    0.3236732367323673,
    0.3236832368323683,
    0.32369323693236934,
    0.3237032370323703,
    0.3237132371323713,
    0.32372323723237234,
    0.3237332373323733,
    0.3237432374323743,
    0.32375323753237534,
    0.3237632376323763,
    0.3237732377323773,
    0.32378323783237833,
    0.3237932379323793,
    0.3238032380323803,
    0.32381323813238133,
    0.3238232382323823,
    0.3238332383323833,
    0.32384323843238433,
    0.32385323853238535,
    0.3238632386323863,
    0.3238732387323873,
    0.32388323883238834,
    0.3238932389323893,
    0.3239032390323903,
    0.32391323913239134,
    0.3239232392323923,
    0.3239332393323933,
    0.32394323943239434,
    0.3239532395323953,
    0.3239632396323963,
    0.32397323973239733,
    0.3239832398323983,
    0.3239932399323993,
    0.32400324003240033,
    0.3240132401324013,
    0.3240232402324023,
    0.3240332403324033,
    0.32404324043240434,
    0.3240532405324053,
    0.3240632406324063,
    0.32407324073240734,
    0.3240832408324083,
    0.3240932409324093,
    0.32410324103241034,
    0.3241132411324113,
    0.3241232412324123,
    0.32413324133241334,
    0.3241432414324143,
    0.3241532415324153,
    0.32416324163241633,
    0.3241732417324173,
    0.3241832418324183,
    0.32419324193241933,
    0.3242032420324203,
    0.3242132421324213,
    0.3242232422324223,
    0.32423324233242334,
    0.3242432424324243,
    0.3242532425324253,
    0.32426324263242634,
    0.3242732427324273,
    0.3242832428324283,
    0.32429324293242934,
    0.3243032430324303,
    0.3243132431324313,
    0.32432324323243233,
    0.3243332433324333,
    0.3243432434324343,
    0.32435324353243533,
    0.3243632436324363,
    0.3243732437324373,
    0.32438324383243833,
    0.32439324393243935,
    0.3244032440324403,
    0.3244132441324413,
    0.32442324423244234,
    0.3244332443324433,
    0.3244432444324443,
    0.32445324453244534,
    0.3244632446324463,
    0.3244732447324473,
    0.32448324483244834,
    0.3244932449324493,
    0.3245032450324503,
    0.32451324513245133,
    0.3245232452324523,
    0.3245332453324533,
    0.32454324543245433,
    0.3245532455324553,
    0.3245632456324563,
    0.3245732457324573,
    0.32458324583245834,
    0.3245932459324593,
    0.3246032460324603,
    0.32461324613246134,
    0.3246232462324623,
    0.3246332463324633,
    0.32464324643246434,
    0.3246532465324653,
    0.3246632466324663,
    0.32467324673246734,
    0.3246832468324683,
    0.3246932469324693,
    0.32470324703247033,
    0.3247132471324713,
    0.3247232472324723,
    0.32473324733247333,
    0.32474324743247435,
    0.3247532475324753,
    0.3247632476324763,
    0.32477324773247734,
    0.3247832478324783,
    0.3247932479324793,
    0.32480324803248034,
    0.3248132481324813,
    0.3248232482324823,
    0.32483324833248334,
    0.3248432484324843,
    0.3248532485324853,
    0.32486324863248633,
    0.3248732487324873,
    0.3248832488324883,
    0.32489324893248933,
    0.3249032490324903,
    0.3249132491324913,
    0.32492324923249233,
    0.32493324933249335,
    0.3249432494324943,
    0.3249532495324953,
    0.32496324963249634,
    0.3249732497324973,
    0.3249832498324983,
    0.32499324993249934,
    0.3250032500325003,
    0.3250132501325013,
    0.32502325023250234,
    0.3250332503325033,
    0.3250432504325043,
    0.32505325053250533,
    0.3250632506325063,
    0.3250732507325073,
    0.32508325083250833,
    0.3250932509325093,
    0.3251032510325103,
    0.3251132511325113,
    0.32512325123251234,
    0.3251332513325133,
    0.3251432514325143,
    0.32515325153251534,
    0.3251632516325163,
    0.3251732517325173,
    0.32518325183251834,
    0.3251932519325193,
    0.3252032520325203,
    0.32521325213252134,
    0.3252232522325223,
    0.3252332523325233,
    0.32524325243252433,
    0.3252532525325253,
    0.3252632526325263,
    0.32527325273252733,
    0.32528325283252835,
    0.3252932529325293,
    0.3253032530325303,
    0.32531325313253134,
    0.3253232532325323,
    0.3253332533325333,
    0.32534325343253434,
    0.3253532535325353,
    0.3253632536325363,
    0.32537325373253734,
    0.3253832538325383,
    0.3253932539325393,
    0.32540325403254033,
    0.3254132541325413,
    0.3254232542325423,
    0.32543325433254333,
    0.3254432544325443,
    0.3254532545325453,
    0.32546325463254633,
    0.32547325473254735,
    0.3254832548325483,
    0.3254932549325493,
    0.32550325503255034,
    0.3255132551325513,
    0.3255232552325523,
    0.32553325533255334,
    0.3255432554325543,
    0.3255532555325553,
    0.32556325563255634,
    0.3255732557325573,
    0.3255832558325583,
    0.32559325593255933,
    0.3256032560325603,
    0.3256132561325613,
    0.32562325623256233,
    0.32563325633256335,
    0.3256432564325643,
    0.3256532565325653,
    0.32566325663256634,
    0.3256732567325673,
    0.3256832568325683,
    0.32569325693256934,
    0.3257032570325703,
    0.3257132571325713,
    0.32572325723257234,
    0.3257332573325733,
    0.3257432574325743,
    0.32575325753257534,
    0.3257632576325763,
    0.3257732577325773,
    0.32578325783257833,
    0.3257932579325793,
    0.3258032580325803,
    0.32581325813258133,
    0.32582325823258235,
    0.3258332583325833,
    0.3258432584325843,
    0.32585325853258534,
    0.3258632586325863,
    0.3258732587325873,
    0.32588325883258834,
    0.3258932589325893,
    0.3259032590325903,
    0.32591325913259134,
    0.3259232592325923,
    0.3259332593325933,
    0.32594325943259433,
    0.3259532595325953,
    0.3259632596325963,
    0.32597325973259733,
    0.3259832598325983,
    0.3259932599325993,
    0.32600326003260033,
    0.32601326013260135,
    0.3260232602326023,
    0.3260332603326033,
    0.32604326043260434,
    0.3260532605326053,
    0.3260632606326063,
    0.32607326073260734,
    0.3260832608326083,
    0.3260932609326093,
    0.32610326103261034,
    0.3261132611326113,
    0.3261232612326123,
    0.32613326133261333,
    0.3261432614326143,
    0.3261532615326153,
    0.32616326163261633,
    0.32617326173261735,
    0.3261832618326183,
    0.3261932619326193,
    0.32620326203262034,
    0.3262132621326213,
    0.3262232622326223,
    0.32623326233262334,
    0.3262432624326243,
    0.3262532625326253,
    0.32626326263262634,
    0.3262732627326273,
    0.3262832628326283,
    0.32629326293262934,
    0.3263032630326303,
    0.3263132631326313,
    0.32632326323263233,
    0.3263332633326333,
    0.3263432634326343,
    0.32635326353263533,
    0.32636326363263635,
    0.3263732637326373,
    0.3263832638326383,
    0.32639326393263934,
    0.3264032640326403,
    0.3264132641326413,
    0.32642326423264234,
    0.3264332643326433,
    0.3264432644326443,
    0.32645326453264534,
    0.3264632646326463,
    0.3264732647326473,
    0.32648326483264833,
    0.3264932649326493,
    0.3265032650326503,
    0.32651326513265133,
    0.3265232652326523,
    0.3265332653326533,
    0.32654326543265433,
    0.32655326553265535,
    0.3265632656326563,
    0.3265732657326573,
    0.32658326583265834,
    0.3265932659326593,
    0.3266032660326603,
    0.32661326613266134,
    0.3266232662326623,
    0.3266332663326633,
    0.32664326643266434,
    0.3266532665326653,
    0.3266632666326663,
    0.32667326673266733,
    0.3266832668326683,
    0.3266932669326693,
    0.32670326703267033,
    0.32671326713267135,
    0.3267232672326723,
    0.3267332673326733,
    0.32674326743267434,
    0.3267532675326753,
    0.3267632676326763,
    0.32677326773267734,
    0.3267832678326783,
    0.3267932679326793,
    0.32680326803268034,
    0.3268132681326813,
    0.3268232682326823,
    0.32683326833268334,
    0.3268432684326843,
    0.3268532685326853,
    0.32686326863268633,
    0.3268732687326873,
    0.3268832688326883,
    0.32689326893268933,
    0.32690326903269035,
    0.3269132691326913,
    0.3269232692326923,
    0.32693326933269334,
    0.3269432694326943,
    0.3269532695326953,
    0.32696326963269634,
    0.3269732697326973,
    0.3269832698326983,
    0.32699326993269934,
    0.3270032700327003,
    0.3270132701327013,
    0.32702327023270233,
    0.3270332703327033,
    0.3270432704327043,
    0.32705327053270533,
    0.32706327063270635,
    0.3270732707327073,
    0.32708327083270833,
    0.32709327093270935,
    0.3271032710327103,
    0.3271132711327113,
    0.32712327123271234,
    0.3271332713327133,
    0.3271432714327143,
    0.32715327153271534,
    0.3271632716327163,
    0.3271732717327173,
    0.32718327183271834,
    0.3271932719327193,
    0.3272032720327203,
    0.32721327213272133,
    0.3272232722327223,
    0.3272332723327233,
    0.32724327243272433,
    0.32725327253272535,
    0.3272632726327263,
    0.3272732727327273,
    0.32728327283272834,
    0.3272932729327293,
    0.3273032730327303,
    0.32731327313273134,
    0.3273232732327323,
    0.3273332733327333,
    0.32734327343273434,
    0.3273532735327353,
    0.3273632736327363,
    0.32737327373273734,
    0.3273832738327383,
    0.3273932739327393,
    0.32740327403274033,
    0.3274132741327413,
    0.3274232742327423,
    0.32743327433274333,
    0.32744327443274435,
    0.3274532745327453,
    0.3274632746327463,
    0.32747327473274734,
    0.3274832748327483,
    0.3274932749327493,
    0.32750327503275034,
    0.3275132751327513,
    0.3275232752327523,
    0.32753327533275334,
    0.3275432754327543,
    0.3275532755327553,
    0.32756327563275633,
    0.3275732757327573,
    0.3275832758327583,
    0.32759327593275933,
    0.32760327603276035,
    0.3276132761327613,
    0.32762327623276233,
    0.32763327633276335,
    0.3276432764327643,
    0.3276532765327653,
    0.32766327663276634,
    0.3276732767327673,
    0.3276832768327683,
    0.32769327693276934,
    0.3277032770327703,
    0.3277132771327713,
    0.32772327723277234,
    0.3277332773327733,
    0.3277432774327743,
    0.32775327753277533,
    0.3277632776327763,
    0.3277732777327773,
    0.32778327783277833,
    0.32779327793277935,
    0.3278032780327803,
    0.3278132781327813,
    0.32782327823278234,
    0.3278332783327833,
    0.3278432784327843,
    0.32785327853278534,
    0.3278632786327863,
    0.3278732787327873,
    0.32788327883278834,
    0.3278932789327893,
    0.3279032790327903,
    0.32791327913279134,
    0.3279232792327923,
    0.3279332793327933,
    0.32794327943279433,
    0.32795327953279535,
    0.3279632796327963,
    0.32797327973279733,
    0.32798327983279835,
    0.3279932799327993,
    0.3280032800328003,
    0.32801328013280134,
    0.3280232802328023,
    0.3280332803328033,
    0.32804328043280434,
    0.3280532805328053,
    0.3280632806328063,
    0.32807328073280734,
    0.3280832808328083,
    0.3280932809328093,
    0.32810328103281033,
    0.3281132811328113,
    0.3281232812328123,
    0.32813328133281333,
    0.32814328143281435,
    0.3281532815328153,
    0.32816328163281633,
    0.32817328173281735,
    0.3281832818328183,
    0.3281932819328193,
    0.32820328203282034,
    0.3282132821328213,
    0.3282232822328223,
    0.32823328233282334,
    0.3282432824328243,
    0.3282532825328253,
    0.32826328263282634,
    0.3282732827328273,
    0.3282832828328283,
    0.32829328293282933,
    0.3283032830328303,
    0.3283132831328313,
    0.32832328323283233,
    0.32833328333283335,
    0.3283432834328343,
    0.3283532835328353,
    0.32836328363283634,
    0.3283732837328373,
    0.3283832838328383,
    0.32839328393283934,
    0.3284032840328403,
    0.3284132841328413,
    0.32842328423284234,
    0.3284332843328433,
    0.3284432844328443,
    0.32845328453284534,
    0.3284632846328463,
    0.3284732847328473,
    0.32848328483284833,
    0.32849328493284935,
    0.3285032850328503,
    0.32851328513285133,
    0.32852328523285235,
    0.3285332853328533,
    0.3285432854328543,
    0.32855328553285534,
    0.3285632856328563,
    0.3285732857328573,
    0.32858328583285834,
    0.3285932859328593,
    0.3286032860328603,
    0.32861328613286134,
    0.3286232862328623,
    0.3286332863328633,
    0.32864328643286433,
    0.3286532865328653,
    0.3286632866328663,
    0.32867328673286733,
    0.32868328683286835,
    0.3286932869328693,
    0.32870328703287033,
    0.32871328713287135,
    0.3287232872328723,
    0.3287332873328733,
    0.32874328743287434,
    0.3287532875328753,
    0.3287632876328763,
    0.32877328773287734,
    0.3287832878328783,
    0.3287932879328793,
    0.32880328803288034,
    0.3288132881328813,
    0.3288232882328823,
    0.32883328833288333,
    0.32884328843288435,
    0.3288532885328853,
    0.32886328863288633,
    0.32887328873288735,
    0.3288832888328883,
    0.3288932889328893,
    0.32890328903289034,
    0.3289132891328913,
    0.3289232892328923,
    0.32893328933289334,
    0.3289432894328943,
    0.3289532895328953,
    0.32896328963289634,
    0.3289732897328973,
    0.3289832898328983,
    0.32899328993289934,
    0.3290032900329003,
    0.3290132901329013,
    0.32902329023290233,
    0.32903329033290335,
    0.3290432904329043,
    0.32905329053290533,
    0.32906329063290635,
    0.3290732907329073,
    0.3290832908329083,
    0.32909329093290934,
    0.3291032910329103,
    0.3291132911329113,
    0.32912329123291234,
    0.3291332913329133,
    0.3291432914329143,
    0.32915329153291534,
    0.3291632916329163,
    0.3291732917329173,
    0.32918329183291833,
    0.3291932919329193,
    0.3292032920329203,
    0.32921329213292133,
    0.32922329223292235,
    0.3292332923329233,
    0.32924329243292433,
    0.32925329253292535,
    0.3292632926329263,
    0.3292732927329273,
    0.32928329283292834,
    0.3292932929329293,
    0.3293032930329303,
    0.32931329313293134,
    0.3293232932329323,
    0.3293332933329333,
    0.32934329343293434,
    0.3293532935329353,
    0.3293632936329363,
    0.32937329373293733,
    0.32938329383293835,
    0.3293932939329393,
    0.32940329403294033,
    0.32941329413294135,
    0.3294232942329423,
    0.3294332943329433,
    0.32944329443294434,
    0.3294532945329453,
    0.3294632946329463,
    0.32947329473294734,
    0.3294832948329483,
    0.3294932949329493,
    0.32950329503295034,
    0.3295132951329513,
    0.3295232952329523,
    0.32953329533295334,
    0.3295432954329543,
    0.3295532955329553,
    0.32956329563295633,
    0.32957329573295735,
    0.3295832958329583,
    0.32959329593295933,
    0.32960329603296035,
    0.3296132961329613,
    0.3296232962329623,
    0.32963329633296334,
    0.3296432964329643,
    0.3296532965329653,
    0.32966329663296634,
    0.3296732967329673,
    0.3296832968329683,
    0.32969329693296934,
    0.3297032970329703,
    0.3297132971329713,
    0.32972329723297233,
    0.3297332973329733,
    0.3297432974329743,
    0.32975329753297533,
    0.32976329763297635,
    0.3297732977329773,
    0.32978329783297833,
    0.32979329793297935,
    0.3298032980329803,
    0.3298132981329813,
    0.32982329823298234,
    0.3298332983329833,
    0.3298432984329843,
    0.32985329853298534,
    0.3298632986329863,
    0.3298732987329873,
    0.32988329883298834,
    0.3298932989329893,
    0.3299032990329903,
    0.32991329913299133,
    0.32992329923299235,
    0.3299332993329933,
    0.32994329943299433,
    0.32995329953299535,
    0.3299632996329963,
    0.3299732997329973,
    0.32998329983299834,
    0.3299932999329993,
    0.3300033000330003,
    0.33001330013300134,
    0.3300233002330023,
    0.3300333003330033,
    0.33004330043300434,
    0.3300533005330053,
    0.3300633006330063,
    0.33007330073300734,
    0.3300833008330083,
    0.3300933009330093,
    0.33010330103301033,
    0.33011330113301135,
    0.3301233012330123,
    0.33013330133301333,
    0.33014330143301435,
    0.3301533015330153,
    0.3301633016330163,
    0.33017330173301734,
    0.3301833018330183,
    0.3301933019330193,
    0.33020330203302034,
    0.3302133021330213,
    0.3302233022330223,
    0.33023330233302334,
    0.3302433024330243,
    0.3302533025330253,
    0.33026330263302633,
    0.33027330273302735,
    0.3302833028330283,
    0.33029330293302933,
    0.33030330303303035,
    0.3303133031330313,
    0.33032330323303233,
    0.33033330333303335,
    0.3303433034330343,
    0.3303533035330353,
    0.33036330363303634,
    0.3303733037330373,
    0.3303833038330383,
    0.33039330393303934,
    0.3304033040330403,
    0.3304133041330413,
    0.33042330423304234,
    0.3304333043330433,
    0.3304433044330443,
    0.33045330453304533,
    0.33046330463304635,
    0.3304733047330473,
    0.33048330483304833,
    0.33049330493304935,
    0.3305033050330503,
    0.3305133051330513,
    0.33052330523305234,
    0.3305333053330533,
    0.3305433054330543,
    0.33055330553305534,
    0.3305633056330563,
    0.3305733057330573,
    0.33058330583305834,
    0.3305933059330593,
    0.3306033060330603,
    0.33061330613306134,
    0.3306233062330623,
    0.3306333063330633,
    0.33064330643306433,
    0.33065330653306535,
    0.3306633066330663,
    0.33067330673306733,
    0.33068330683306835,
    0.3306933069330693,
    0.3307033070330703,
    0.33071330713307134,
    0.3307233072330723,
    0.3307333073330733,
    0.33074330743307434,
    0.3307533075330753,
    0.3307633076330763,
    0.33077330773307734,
    0.3307833078330783,
    0.3307933079330793,
    0.33080330803308033,
    0.33081330813308135,
    0.3308233082330823,
    0.33083330833308333,
    0.33084330843308435,
    0.3308533085330853,
    0.33086330863308633,
    0.33087330873308735,
    0.3308833088330883,
    0.3308933089330893,
    0.33090330903309034,
    0.3309133091330913,
    0.3309233092330923,
    0.33093330933309334,
    0.3309433094330943,
    0.3309533095330953,
    0.33096330963309634,
    0.3309733097330973,
    0.3309833098330983,
    0.33099330993309933,
    0.33100331003310035,
    0.3310133101331013,
    0.33102331023310233,
    0.33103331033310335,
    0.3310433104331043,
    0.3310533105331053,
    0.33106331063310634,
    0.3310733107331073,
    0.3310833108331083,
    0.33109331093310934,
    0.3311033110331103,
    0.3311133111331113,
    0.33112331123311234,
    0.3311333113331133,
    0.3311433114331143,
    0.33115331153311534,
    0.33116331163311635,
    0.3311733117331173,
    0.33118331183311833,
    0.33119331193311935,
    0.3312033120331203,
    0.33121331213312133,
    0.33122331223312235,
    0.3312333123331233,
    0.3312433124331243,
    0.33125331253312534,
    0.3312633126331263,
    0.3312733127331273,
    0.33128331283312834,
    0.3312933129331293,
    0.3313033130331303,
    0.33131331313313134,
    0.3313233132331323,
    0.3313333133331333,
    0.33134331343313433,
    0.33135331353313535,
    0.3313633136331363,
    0.33137331373313733,
    0.33138331383313835,
    0.3313933139331393,
    0.33140331403314033,
    0.33141331413314135,
    0.3314233142331423,
    0.3314333143331433,
    0.33144331443314434,
    0.3314533145331453,
    0.3314633146331463,
    0.33147331473314734,
    0.3314833148331483,
    0.3314933149331493,
    0.33150331503315034,
    0.3315133151331513,
    0.3315233152331523,
    0.33153331533315333,
    0.33154331543315435,
    0.3315533155331553,
    0.33156331563315633,
    0.33157331573315735,
    0.3315833158331583,
    0.3315933159331593,
    0.33160331603316034,
    0.3316133161331613,
    0.3316233162331623,
    0.33163331633316334,
    0.3316433164331643,
    0.3316533165331653,
    0.33166331663316634,
    0.3316733167331673,
    0.3316833168331683,
    0.33169331693316934,
    0.33170331703317035,
    0.3317133171331713,
    0.33172331723317233,
    0.33173331733317335,
    0.3317433174331743,
    0.33175331753317533,
    0.33176331763317635,
    0.3317733177331773,
    0.3317833178331783,
    0.33179331793317934,
    0.3318033180331803,
    0.3318133181331813,
    0.33182331823318234,
    0.3318333183331833,
    0.3318433184331843,
    0.33185331853318534,
    0.3318633186331863,
    0.3318733187331873,
    0.33188331883318833,
    0.33189331893318935,
    0.3319033190331903,
    0.33191331913319133,
    0.33192331923319235,
    0.3319333193331933,
    0.33194331943319433,
    0.33195331953319535,
    0.3319633196331963,
    0.3319733197331973,
    0.33198331983319834,
    0.3319933199331993,
    0.3320033200332003,
    0.33201332013320134,
    0.3320233202332023,
    0.3320333203332033,
    0.33204332043320434,
    0.33205332053320535,
    0.3320633206332063,
    0.33207332073320733,
    0.33208332083320835,
    0.3320933209332093,
    0.33210332103321033,
    0.33211332113321135,
    0.3321233212332123,
    0.3321333213332133,
    0.33214332143321434,
    0.3321533215332153,
    0.3321633216332163,
    0.33217332173321734,
    0.3321833218332183,
    0.3321933219332193,
    0.33220332203322034,
    0.3322133221332213,
    0.3322233222332223,
    0.33223332233322334,
    0.33224332243322435,
    0.3322533225332253,
    0.33226332263322633,
    0.33227332273322735,
    0.3322833228332283,
    0.33229332293322933,
    0.33230332303323035,
    0.3323133231332313,
    0.3323233232332323,
    0.33233332333323334,
    0.3323433234332343,
    0.3323533235332353,
    0.33236332363323634,
    0.3323733237332373,
    0.3323833238332383,
    0.33239332393323934,
    0.3324033240332403,
    0.3324133241332413,
    0.33242332423324233,
    0.33243332433324335,
    0.3324433244332443,
    0.33245332453324533,
    0.33246332463324635,
    0.3324733247332473,
    0.33248332483324833,
    0.33249332493324935,
    0.3325033250332503,
    0.3325133251332513,
    0.33252332523325234,
    0.3325333253332533,
    0.3325433254332543,
    0.33255332553325534,
    0.3325633256332563,
    0.3325733257332573,
    0.33258332583325834,
    0.33259332593325935,
    0.3326033260332603,
    0.33261332613326133,
    0.33262332623326235,
    0.3326333263332633,
    0.33264332643326433,
    0.33265332653326535,
    0.3326633266332663,
    0.3326733267332673,
    0.33268332683326834,
    0.3326933269332693,
    0.3327033270332703,
    0.33271332713327134,
    0.3327233272332723,
    0.3327333273332733,
    0.33274332743327434,
    0.3327533275332753,
    0.3327633276332763,
    0.33277332773327734,
    0.33278332783327835,
    0.3327933279332793,
    0.33280332803328033,
    0.33281332813328135,
    0.3328233282332823,
    0.33283332833328333,
    0.33284332843328435,
    0.3328533285332853,
    0.3328633286332863,
    0.33287332873328734,
    0.3328833288332883,
    0.3328933289332893,
    0.33290332903329034,
    0.3329133291332913,
    0.3329233292332923,
    0.33293332933329334,
    0.33294332943329435,
    0.3329533295332953,
    0.33296332963329633,
    0.33297332973329735,
    0.3329833298332983,
    0.33299332993329933,
    0.33300333003330035,
    0.3330133301333013,
    0.33302333023330233,
    0.33303333033330335,
    0.3330433304333043,
    0.3330533305333053,
    0.33306333063330634,
    0.3330733307333073,
    0.3330833308333083,
    0.33309333093330934,
    0.3331033310333103,
    0.3331133311333113,
    0.33312333123331234,
    0.33313333133331335,
    0.3331433314333143,
    0.33315333153331533,
    0.33316333163331635,
    0.3331733317333173,
    0.33318333183331833,
    0.33319333193331935,
    0.3332033320333203,
    0.3332133321333213,
    0.33322333223332234,
    0.3332333323333233,
    0.3332433324333243,
    0.33325333253332534,
    0.3332633326333263,
    0.3332733327333273,
    0.33328333283332834,
    0.3332933329333293,
    0.3333033330333303,
    0.33331333313333134,
    0.33332333323333235,
    0.3333333333333333,
    0.33334333343333433,
    0.33335333353333535,
    0.3333633336333363,
    0.33337333373333733,
    0.33338333383333835,
    0.3333933339333393,
    0.3334033340333403,
    0.33341333413334134,
    0.3334233342333423,
    0.3334333343333433,
    0.33344333443334434,
    0.3334533345333453,
    0.3334633346333463,
    0.33347333473334734,
    0.33348333483334835,
    0.3334933349333493,
    0.33350333503335033,
    0.33351333513335135,
    0.3335233352333523,
    0.33353333533335333,
    0.33354333543335435,
    0.3335533355333553,
    0.33356333563335633,
    0.33357333573335735,
    0.3335833358333583,
    0.3335933359333593,
    0.33360333603336034,
    0.3336133361333613,
    0.3336233362333623,
    0.33363333633336334,
    0.3336433364333643,
    0.3336533365333653,
    0.33366333663336634,
    0.33367333673336735,
    0.3336833368333683,
    0.33369333693336933,
    0.33370333703337035,
    0.3337133371333713,
    0.33372333723337233,
    0.33373333733337335,
    0.3337433374333743,
    0.3337533375333753,
    0.33376333763337634,
    0.3337733377333773,
    0.3337833378333783,
    0.33379333793337934,
    0.3338033380333803,
    0.3338133381333813,
    0.33382333823338234,
    0.3338333383333833,
    0.3338433384333843,
    0.33385333853338534,
    0.33386333863338635,
    0.3338733387333873,
    0.33388333883338833,
    0.33389333893338935,
    0.3339033390333903,
    0.33391333913339133,
    0.33392333923339235,
    0.3339333393333933,
    0.3339433394333943,
    0.33395333953339534,
    0.3339633396333963,
    0.3339733397333973,
    0.33398333983339834,
    0.3339933399333993,
    0.3340033400334003,
    0.33401334013340134,
    0.33402334023340235,
    0.3340333403334033,
    0.33404334043340433,
    0.33405334053340535,
    0.3340633406334063,
    0.33407334073340733,
    0.33408334083340835,
    0.3340933409334093,
    0.33410334103341033,
    0.33411334113341135,
    0.3341233412334123,
    0.3341333413334133,
    0.33414334143341434,
    0.3341533415334153,
    0.3341633416334163,
    0.33417334173341734,
    0.3341833418334183,
    0.3341933419334193,
    0.33420334203342034,
    0.33421334213342135,
    0.3342233422334223,
    0.33423334233342333,
    0.33424334243342435,
    0.3342533425334253,
    0.33426334263342633,
    0.33427334273342735,
    0.3342833428334283,
    0.3342933429334293,
    0.33430334303343034,
    0.3343133431334313,
    0.3343233432334323,
    0.33433334333343334,
    0.3343433434334343,
    0.3343533435334353,
    0.33436334363343634,
    0.33437334373343736,
    0.3343833438334383,
    0.33439334393343934,
    0.33440334403344035,
    0.3344133441334413,
    0.33442334423344233,
    0.33443334433344335,
    0.3344433444334443,
    0.33445334453344533,
    0.33446334463344635,
    0.3344733447334473,
    0.3344833448334483,
    0.33449334493344934,
    0.3345033450334503,
    0.3345133451334513,
    0.33452334523345234,
    0.3345333453334533,
    0.3345433454334543,
    0.33455334553345534,
    0.33456334563345635,
    0.3345733457334573,
    0.33458334583345833,
    0.33459334593345935,
    0.3346033460334603,
    0.33461334613346133,
    0.33462334623346235,
    0.3346333463334633,
    0.33464334643346433,
    0.33465334653346535,
    0.3346633466334663,
    0.3346733467334673,
    0.33468334683346834,
    0.3346933469334693,
    0.3347033470334703,
    0.33471334713347134,
    0.3347233472334723,
    0.3347333473334733,
    0.33474334743347434,
    0.33475334753347535,
    0.3347633476334763,
    0.33477334773347733,
    0.33478334783347835,
    0.3347933479334793,
    0.33480334803348033,
    0.33481334813348135,
    0.3348233482334823,
    0.3348333483334833,
    0.33484334843348434,
    0.3348533485334853,
    0.3348633486334863,
    0.33487334873348734,
    0.3348833488334883,
    0.3348933489334893,
    0.33490334903349034,
    0.33491334913349136,
    0.3349233492334923,
    0.33493334933349334,
    0.33494334943349435,
    0.3349533495334953,
    0.33496334963349633,
    0.33497334973349735,
    0.3349833498334983,
    0.33499334993349933,
    0.33500335003350035,
    0.3350133501335013,
    0.3350233502335023,
    0.33503335033350334,
    0.3350433504335043,
    0.3350533505335053,
    0.33506335063350634,
    0.3350733507335073,
    0.3350833508335083,
    0.33509335093350934,
    0.33510335103351035,
    0.3351133511335113,
    0.33512335123351233,
    0.33513335133351335,
    0.3351433514335143,
    0.33515335153351533,
    0.33516335163351635,
    0.3351733517335173,
    0.33518335183351833,
    0.33519335193351935,
    0.3352033520335203,
    0.3352133521335213,
    0.33522335223352234,
    0.3352333523335233,
    0.3352433524335243,
    0.33525335253352534,
    0.33526335263352636,
    0.3352733527335273,
    0.33528335283352834,
    0.33529335293352935,
    0.3353033530335303,
    0.33531335313353133,
    0.33532335323353235,
    0.3353333533335333,
    0.33534335343353433,
    0.33535335353353535,
    0.3353633536335363,
    0.3353733537335373,
    0.33538335383353834,
    0.3353933539335393,
    0.3354033540335403,
    0.33541335413354134,
    0.3354233542335423,
    0.3354333543335433,
    0.33544335443354434,
    0.33545335453354536,
    0.3354633546335463,
    0.33547335473354734,
    0.33548335483354835,
    0.3354933549335493,
    0.33550335503355033,
    0.33551335513355135,
    0.3355233552335523,
    0.33553335533355333,
    0.33554335543355435,
    0.3355533555335553,
    0.3355633556335563,
    0.33557335573355734,
    0.3355833558335583,
    0.3355933559335593,
    0.33560335603356034,
    0.3356133561335613,
    0.3356233562335623,
    0.33563335633356334,
    0.33564335643356435,
    0.3356533565335653,
    0.33566335663356633,
    0.33567335673356735,
    0.3356833568335683,
    0.33569335693356933,
    0.33570335703357035,
    0.3357133571335713,
    0.33572335723357233,
    0.33573335733357335,
    0.3357433574335743,
    0.3357533575335753,
    0.33576335763357634,
    0.3357733577335773,
    0.3357833578335783,
    0.33579335793357934,
    0.33580335803358036,
    0.3358133581335813,
    0.33582335823358234,
    0.33583335833358335,
    0.3358433584335843,
    0.33585335853358533,
    0.33586335863358635,
    0.3358733587335873,
    0.33588335883358833,
    0.33589335893358935,
    0.3359033590335903,
    0.3359133591335913,
    0.33592335923359234,
    0.3359333593335933,
    0.3359433594335943,
    0.33595335953359534,
    0.3359633596335963,
    0.3359733597335973,
    0.33598335983359834,
    0.33599335993359936,
    0.3360033600336003,
    0.33601336013360134,
    0.33602336023360235,
    0.3360333603336033,
    0.33604336043360433,
    0.33605336053360535,
    0.3360633606336063,
    0.33607336073360733,
    0.33608336083360835,
    0.3360933609336093,
    0.3361033610336103,
    0.33611336113361134,
    0.3361233612336123,
    0.3361333613336133,
    0.33614336143361434,
    0.33615336153361536,
    0.3361633616336163,
    0.33617336173361734,
    0.33618336183361835,
    0.3361933619336193,
    0.33620336203362033,
    0.33621336213362135,
    0.3362233622336223,
    0.33623336233362333,
    0.33624336243362435,
    0.3362533625336253,
    0.33626336263362633,
    0.33627336273362735,
    0.3362833628336283,
    0.3362933629336293,
    0.33630336303363034,
    0.3363133631336313,
    0.3363233632336323,
    0.33633336333363334,
    0.33634336343363436,
    0.3363533635336353,
    0.33636336363363634,
    0.33637336373363735,
    0.3363833638336383,
    0.33639336393363933,
    0.33640336403364035,
    0.3364133641336413,
    0.33642336423364233,
    0.33643336433364335,
    0.3364433644336443,
    0.3364533645336453,
    0.33646336463364634,
    0.3364733647336473,
    0.3364833648336483,
    0.33649336493364934,
    0.3365033650336503,
    0.3365133651336513,
    0.33652336523365234,
    0.33653336533365336,
    0.3365433654336543,
    0.33655336553365534,
    0.33656336563365635,
    0.3365733657336573,
    0.33658336583365833,
    0.33659336593365935,
    0.3366033660336603,
    0.33661336613366133,
    0.33662336623366235,
    0.3366333663336633,
    0.3366433664336643,
    0.33665336653366534,
    0.3366633666336663,
    0.3366733667336673,
    0.33668336683366834,
    0.33669336693366936,
    0.3367033670336703,
    0.33671336713367134,
    0.33672336723367235,
    0.3367333673336733,
    0.33674336743367433,
    0.33675336753367535,
    0.3367633676336763,
    0.33677336773367733,
    0.33678336783367835,
    0.3367933679336793,
    0.33680336803368033,
    0.33681336813368135,
    0.3368233682336823,
    0.3368333683336833,
    0.33684336843368434,
    0.3368533685336853,
    0.3368633686336863,
    0.33687336873368734,
    0.33688336883368836,
    0.3368933689336893,
    0.33690336903369034,
    0.33691336913369135,
    0.3369233692336923,
    0.33693336933369333,
    0.33694336943369435,
    0.3369533695336953,
    0.33696336963369633,
    0.33697336973369735,
    0.3369833698336983,
    0.3369933699336993,
    0.33700337003370034,
    0.3370133701337013,
    0.3370233702337023,
    0.33703337033370334,
    0.33704337043370436,
    0.3370533705337053,
    0.33706337063370634,
    0.33707337073370736,
    0.3370833708337083,
    0.33709337093370934,
    0.33710337103371035,
    0.3371133711337113,
    0.33712337123371233,
    0.33713337133371335,
    0.3371433714337143,
    0.33715337153371533,
    0.33716337163371635,
    0.3371733717337173,
    0.3371833718337183,
    0.33719337193371934,
    0.3372033720337203,
    0.3372133721337213,
    0.33722337223372234,
    0.33723337233372336,
    0.3372433724337243,
    0.33725337253372534,
    0.33726337263372635,
    0.3372733727337273,
    0.33728337283372833,
    0.33729337293372935,
    0.3373033730337303,
    0.33731337313373133,
    0.33732337323373235,
    0.3373333733337333,
    0.33734337343373433,
    0.33735337353373535,
    0.3373633736337363,
    0.3373733737337373,
    0.33738337383373834,
    0.3373933739337393,
    0.3374033740337403,
    0.33741337413374134,
    0.33742337423374236,
    0.3374333743337433,
    0.33744337443374434,
    0.33745337453374535,
    0.3374633746337463,
    0.33747337473374733,
    0.33748337483374835,
    0.3374933749337493,
    0.33750337503375033,
    0.33751337513375135,
    0.3375233752337523,
    0.3375333753337533,
    0.33754337543375434,
    0.3375533755337553,
    0.3375633756337563,
    0.33757337573375734,
    0.33758337583375836,
    0.3375933759337593,
    0.33760337603376034,
    0.33761337613376136,
    0.3376233762337623,
    0.33763337633376334,
    0.33764337643376435,
    0.3376533765337653,
    0.33766337663376633,
    0.33767337673376735,
    0.3376833768337683,
    0.33769337693376933,
    0.33770337703377035,
    0.3377133771337713,
    0.3377233772337723,
    0.33773337733377334,
    0.3377433774337743,
    0.3377533775337753,
    0.33776337763377634,
    0.33777337773377736,
    0.3377833778337783,
    0.33779337793377934,
    0.33780337803378035,
    0.3378133781337813,
    0.33782337823378233,
    0.33783337833378335,
    0.3378433784337843,
    0.33785337853378533,
    0.33786337863378635,
    0.3378733787337873,
    0.33788337883378833,
    0.33789337893378935,
    0.3379033790337903,
    0.3379133791337913,
    0.33792337923379234,
    0.3379333793337933,
    0.3379433794337943,
    0.33795337953379534,
    0.33796337963379636,
    0.3379733797337973,
    0.33798337983379834,
    0.33799337993379935,
    0.3380033800338003,
    0.33801338013380133,
    0.33802338023380235,
    0.3380333803338033,
    0.33804338043380433,
    0.33805338053380535,
    0.3380633806338063,
    0.3380733807338073,
    0.33808338083380834,
    0.3380933809338093,
    0.3381033810338103,
    0.33811338113381134,
    0.33812338123381236,
    0.3381333813338133,
    0.33814338143381434,
    0.33815338153381536,
    0.3381633816338163,
    0.33817338173381734,
    0.33818338183381835,
    0.3381933819338193,
    0.33820338203382033,
    0.33821338213382135,
    0.3382233822338223,
    0.33823338233382333,
    0.33824338243382435,
    0.3382533825338253,
    0.3382633826338263,
    0.33827338273382734,
    0.3382833828338283,
    0.3382933829338293,
    0.33830338303383034,
    0.33831338313383136,
    0.3383233832338323,
    0.33833338333383334,
    0.33834338343383435,
    0.3383533835338353,
    0.33836338363383633,
    0.33837338373383735,
    0.3383833838338383,
    0.33839338393383933,
    0.33840338403384035,
    0.3384133841338413,
    0.33842338423384233,
    0.33843338433384335,
    0.3384433844338443,
    0.3384533845338453,
    0.33846338463384634,
    0.33847338473384736,
    0.3384833848338483,
    0.33849338493384934,
    0.33850338503385036,
    0.3385133851338513,
    0.33852338523385234,
    0.33853338533385335,
    0.3385433854338543,
    0.33855338553385533,
    0.33856338563385635,
    0.3385733857338573,
    0.33858338583385833,
    0.33859338593385935,
    0.3386033860338603,
    0.3386133861338613,
    0.33862338623386234,
    0.3386333863338633,
    0.3386433864338643,
    0.33865338653386534,
    0.33866338663386636,
    0.3386733867338673,
    0.33868338683386834,
    0.33869338693386936,
    0.3387033870338703,
    0.33871338713387134,
    0.33872338723387235,
    0.3387333873338733,
    0.33874338743387433,
    0.33875338753387535,
    0.3387633876338763,
    0.33877338773387733,
    0.33878338783387835,
    0.3387933879338793,
    0.3388033880338803,
    0.33881338813388134,
    0.3388233882338823,
    0.3388333883338833,
    0.33884338843388434,
    0.33885338853388536,
    0.3388633886338863,
    0.33887338873388734,
    0.33888338883388835,
    0.3388933889338893,
    0.33890338903389033,
    0.33891338913389135,
    0.3389233892338923,
    0.33893338933389333,
    0.33894338943389435,
    0.3389533895338953,
    0.33896338963389633,
    0.33897338973389735,
    0.3389833898338983,
    0.3389933899338993,
    0.33900339003390034,
    0.33901339013390136,
    0.3390233902339023,
    0.33903339033390334,
    0.33904339043390436,
    0.3390533905339053,
    0.33906339063390634,
    0.33907339073390735,
    0.3390833908339083,
    0.33909339093390933,
    0.33910339103391035,
    0.3391133911339113,
    0.33912339123391233,
    0.33913339133391335,
    0.3391433914339143,
    0.3391533915339153,
    0.33916339163391634,
    0.3391733917339173,
    0.3391833918339183,
    0.33919339193391934,
    0.33920339203392036,
    0.3392133921339213,
    0.33922339223392234,
    0.33923339233392336,
    0.3392433924339243,
    0.33925339253392534,
    0.33926339263392635,
    0.3392733927339273,
    0.33928339283392833,
    0.33929339293392935,
    0.3393033930339303,
    0.33931339313393133,
    0.33932339323393235,
    0.3393333933339333,
    0.3393433934339343,
    0.33935339353393534,
    0.33936339363393636,
    0.3393733937339373,
    0.33938339383393834,
    0.33939339393393936,
    0.3394033940339403,
    0.33941339413394134,
    0.33942339423394235,
    0.3394333943339433,
    0.33944339443394433,
    0.33945339453394535,
    0.3394633946339463,
    0.33947339473394733,
    0.33948339483394835,
    0.3394933949339493,
    0.33950339503395033,
    0.33951339513395135,
    0.3395233952339523,
    0.3395333953339533,
    0.33954339543395434,
    0.33955339553395536,
    0.3395633956339563,
    0.33957339573395734,
    0.33958339583395836,
    0.3395933959339593,
    0.33960339603396034,
    0.33961339613396135,
    0.3396233962339623,
    0.33963339633396333,
    0.33964339643396435,
    0.3396533965339653,
    0.33966339663396633,
    0.33967339673396735,
    0.3396833968339683,
    0.3396933969339693,
    0.33970339703397034,
    0.3397133971339713,
    0.3397233972339723,
    0.33973339733397334,
    0.33974339743397436,
    0.3397533975339753,
    0.33976339763397634,
    0.33977339773397736,
    0.3397833978339783,
    0.33979339793397934,
    0.33980339803398035,
    0.3398133981339813,
    0.33982339823398233,
    0.33983339833398335,
    0.3398433984339843,
    0.33985339853398533,
    0.33986339863398635,
    0.3398733987339873,
    0.3398833988339883,
    0.33989339893398934,
    0.33990339903399036,
    0.3399133991339913,
    0.33992339923399234,
    0.33993339933399336,
    0.3399433994339943,
    0.33995339953399534,
    0.33996339963399635,
    0.3399733997339973,
    0.33998339983399833,
    0.33999339993399935,
    0.3400034000340003,
    0.34001340013400133,
    0.34002340023400235,
    0.3400334003340033,
    0.34004340043400433,
    0.34005340053400535,
    0.3400634006340063,
    0.3400734007340073,
    0.34008340083400834,
    0.34009340093400936,
    0.3401034010340103,
    0.34011340113401134,
    0.34012340123401236,
    0.3401334013340133,
    0.34014340143401434,
    0.34015340153401535,
    0.3401634016340163,
    0.34017340173401733,
    0.34018340183401835,
    0.3401934019340193,
    0.34020340203402033,
    0.34021340213402135,
    0.3402234022340223,
    0.3402334023340233,
    0.34024340243402434,
    0.34025340253402536,
    0.3402634026340263,
    0.34027340273402734,
    0.34028340283402836,
    0.3402934029340293,
    0.34030340303403034,
    0.34031340313403136,
    0.3403234032340323,
    0.34033340333403334,
    0.34034340343403435,
    0.3403534035340353,
    0.34036340363403633,
    0.34037340373403735,
    0.3403834038340383,
    0.34039340393403933,
    0.34040340403404035,
    0.3404134041340413,
    0.3404234042340423,
    0.34043340433404334,
    0.34044340443404436,
    0.3404534045340453,
    0.34046340463404634,
    0.34047340473404736,
    0.3404834048340483,
    0.34049340493404934,
    0.34050340503405035,
    0.3405134051340513,
    0.34052340523405233,
    0.34053340533405335,
    0.3405434054340543,
    0.34055340553405533,
    0.34056340563405635,
    0.3405734057340573,
    0.34058340583405833,
    0.34059340593405935,
    0.3406034060340603,
    0.3406134061340613,
    0.34062340623406234,
    0.34063340633406336,
    0.3406434064340643,
    0.34065340653406534,
    0.34066340663406636,
    0.3406734067340673,
    0.34068340683406834,
    0.34069340693406935,
    0.3407034070340703,
    0.34071340713407133,
    0.34072340723407235,
    0.3407334073340733,
    0.34074340743407433,
    0.34075340753407535,
    0.3407634076340763,
    0.3407734077340773,
    0.34078340783407834,
    0.34079340793407936,
    0.3408034080340803,
    0.34081340813408134,
    0.34082340823408236,
    0.3408334083340833,
    0.34084340843408434,
    0.34085340853408536,
    0.3408634086340863,
    0.34087340873408734,
    0.34088340883408835,
    0.3408934089340893,
    0.34090340903409033,
    0.34091340913409135,
    0.3409234092340923,
    0.34093340933409333,
    0.34094340943409435,
    0.3409534095340953,
    0.3409634096340963,
    0.34097340973409734,
    0.34098340983409836,
    0.3409934099340993,
    0.34100341003410034,
    0.34101341013410136,
    0.3410234102341023,
    0.34103341033410334,
    0.34104341043410435,
    0.3410534105341053,
    0.34106341063410633,
    0.34107341073410735,
    0.3410834108341083,
    0.34109341093410933,
    0.34110341103411035,
    0.3411134111341113,
    0.34112341123411233,
    0.34113341133411335,
    0.3411434114341143,
    0.3411534115341153,
    0.34116341163411634,
    0.34117341173411736,
    0.3411834118341183,
    0.34119341193411934,
    0.34120341203412036,
    0.3412134121341213,
    0.34122341223412234,
    0.34123341233412335,
    0.3412434124341243,
    0.34125341253412533,
    0.34126341263412635,
    0.3412734127341273,
    0.34128341283412833,
    0.34129341293412935,
    0.3413034130341303,
    0.3413134131341313,
    0.34132341323413234,
    0.34133341333413336,
    0.3413434134341343,
    0.34135341353413534,
    0.34136341363413636,
    0.3413734137341373,
    0.34138341383413834,
    0.34139341393413936,
    0.3414034140341403,
    0.34141341413414134,
    0.34142341423414235,
    0.3414334143341433,
    0.34144341443414433,
    0.34145341453414535,
    0.3414634146341463,
    0.34147341473414733,
    0.34148341483414835,
    0.3414934149341493,
    0.3415034150341503,
    0.34151341513415134,
    0.34152341523415236,
    0.3415334153341533,
    0.34154341543415434,
    0.34155341553415536,
    0.3415634156341563,
    0.34157341573415734,
    0.34158341583415835,
    0.3415934159341593,
    0.34160341603416033,
    0.34161341613416135,
    0.3416234162341623,
    0.34163341633416333,
    0.34164341643416435,
    0.3416534165341653,
    0.34166341663416633,
    0.34167341673416735,
    0.34168341683416836,
    0.3416934169341693,
    0.34170341703417034,
    0.34171341713417136,
    0.3417234172341723,
    0.34173341733417334,
    0.34174341743417436,
    0.3417534175341753,
    0.34176341763417634,
    0.34177341773417735,
    0.3417834178341783,
    0.34179341793417933,
    0.34180341803418035,
    0.3418134181341813,
    0.34182341823418233,
    0.34183341833418335,
    0.3418434184341843,
    0.3418534185341853,
    0.34186341863418634,
    0.34187341873418736,
    0.3418834188341883,
    0.34189341893418934,
    0.34190341903419036,
    0.3419134191341913,
    0.34192341923419234,
    0.34193341933419336,
    0.3419434194341943,
    0.34195341953419534,
    0.34196341963419635,
    0.3419734197341973,
    0.34198341983419833,
    0.34199341993419935,
    0.3420034200342003,
    0.34201342013420133,
    0.34202342023420235,
    0.3420334203342033,
    0.3420434204342043,
    0.34205342053420534,
    0.34206342063420636,
    0.3420734207342073,
    0.34208342083420834,
    0.34209342093420936,
    0.3421034210342103,
    0.34211342113421134,
    0.34212342123421235,
    0.3421334213342133,
    0.34214342143421433,
    0.34215342153421535,
    0.3421634216342163,
    0.34217342173421733,
    0.34218342183421835,
    0.3421934219342193,
    0.34220342203422033,
    0.34221342213422135,
    0.34222342223422236,
    0.3422334223342233,
    0.34224342243422434,
    0.34225342253422536,
    0.3422634226342263,
    0.34227342273422734,
    0.34228342283422836,
    0.3422934229342293,
    0.34230342303423034,
    0.34231342313423135,
    0.3423234232342323,
    0.34233342333423333,
    0.34234342343423435,
    0.3423534235342353,
    0.34236342363423633,
    0.34237342373423735,
    0.3423834238342383,
    0.3423934239342393,
    0.34240342403424034,
    0.34241342413424136,
    0.3424234242342423,
    0.34243342433424334,
    0.34244342443424436,
    0.3424534245342453,
    0.34246342463424634,
    0.34247342473424736,
    0.3424834248342483,
    0.34249342493424934,
    0.34250342503425035,
    0.3425134251342513,
    0.34252342523425233,
    0.34253342533425335,
    0.3425434254342543,
    0.34255342553425533,
    0.34256342563425635,
    0.34257342573425736,
    0.3425834258342583,
    0.34259342593425934,
    0.34260342603426036,
    0.3426134261342613,
    0.34262342623426234,
    0.34263342633426336,
    0.3426434264342643,
    0.34265342653426534,
    0.34266342663426635,
    0.3426734267342673,
    0.34268342683426833,
    0.34269342693426935,
    0.3427034270342703,
    0.34271342713427133,
    0.34272342723427235,
    0.3427334273342733,
    0.34274342743427433,
    0.34275342753427535,
    0.34276342763427636,
    0.3427734277342773,
    0.34278342783427834,
    0.34279342793427936,
    0.3428034280342803,
    0.34281342813428134,
    0.34282342823428236,
    0.3428334283342833,
    0.34284342843428434,
    0.34285342853428535,
    0.3428634286342863,
    0.34287342873428733,
    0.34288342883428835,
    0.3428934289342893,
    0.34290342903429033,
    0.34291342913429135,
    0.3429234292342923,
    0.3429334293342933,
    0.34294342943429434,
    0.34295342953429536,
    0.3429634296342963,
    0.34297342973429734,
    0.34298342983429836,
    0.3429934299342993,
    0.34300343003430034,
    0.34301343013430136,
    0.3430234302343023,
    0.34303343033430334,
    0.34304343043430435,
    0.3430534305343053,
    0.34306343063430633,
    0.34307343073430735,
    0.3430834308343083,
    0.34309343093430933,
    0.34310343103431035,
    0.34311343113431136,
    0.3431234312343123,
    0.34313343133431334,
    0.34314343143431436,
    0.3431534315343153,
    0.34316343163431634,
    0.34317343173431736,
    0.3431834318343183,
    0.34319343193431934,
    0.34320343203432035,
    0.3432134321343213,
    0.34322343223432233,
    0.34323343233432335,
    0.3432434324343243,
    0.34325343253432533,
    0.34326343263432635,
    0.3432734327343273,
    0.34328343283432833,
    0.34329343293432935,
    0.34330343303433036,
    0.3433134331343313,
    0.34332343323433234,
    0.34333343333433336,
    0.3433434334343343,
    0.34335343353433534,
    0.34336343363433636,
    0.3433734337343373,
    0.34338343383433834,
    0.34339343393433935,
    0.3434034340343403,
    0.34341343413434133,
    0.34342343423434235,
    0.3434334343343433,
    0.34344343443434433,
    0.34345343453434535,
    0.34346343463434637,
    0.3434734347343473,
    0.34348343483434834,
    0.34349343493434936,
    0.3435034350343503,
    0.34351343513435134,
    0.34352343523435236,
    0.3435334353343533,
    0.34354343543435434,
    0.34355343553435536,
    0.3435634356343563,
    0.34357343573435734,
    0.34358343583435835,
    0.3435934359343593,
    0.34360343603436033,
    0.34361343613436135,
    0.3436234362343623,
    0.34363343633436333,
    0.34364343643436435,
    0.34365343653436536,
    0.3436634366343663,
    0.34367343673436734,
    0.34368343683436836,
    0.3436934369343693,
    0.34370343703437034,
    0.34371343713437136,
    0.3437234372343723,
    0.34373343733437334,
    0.34374343743437435,
    0.3437534375343753,
    0.34376343763437633,
    0.34377343773437735,
    0.3437834378343783,
    0.34379343793437933,
    0.34380343803438035,
    0.3438134381343813,
    0.34382343823438233,
    0.34383343833438335,
    0.34384343843438436,
    0.3438534385343853,
    0.34386343863438634,
    0.34387343873438736,
    0.3438834388343883,
    0.34389343893438934,
    0.34390343903439036,
    0.3439134391343913,
    0.34392343923439234,
    0.34393343933439335,
    0.3439434394343943,
    0.34395343953439533,
    0.34396343963439635,
    0.3439734397343973,
    0.34398343983439833,
    0.34399343993439935,
    0.34400344003440037,
    0.3440134401344013,
    0.34402344023440234,
    0.34403344033440336,
    0.3440434404344043,
    0.34405344053440534,
    0.34406344063440636,
    0.3440734407344073,
    0.34408344083440834,
    0.34409344093440936,
    0.3441034410344103,
    0.34411344113441134,
    0.34412344123441235,
    0.3441334413344133,
    0.34414344143441433,
    0.34415344153441535,
    0.3441634416344163,
    0.34417344173441733,
    0.34418344183441835,
    0.34419344193441936,
    0.3442034420344203,
    0.34421344213442134,
    0.34422344223442236,
    0.3442334423344233,
    0.34424344243442434,
    0.34425344253442536,
    0.3442634426344263,
    0.34427344273442734,
    0.34428344283442835,
    0.3442934429344293,
    0.34430344303443033,
    0.34431344313443135,
    0.3443234432344323,
    0.34433344333443333,
    0.34434344343443435,
    0.34435344353443537,
    0.34436344363443633,
    0.34437344373443735,
    0.34438344383443836,
    0.3443934439344393,
    0.34440344403444034,
    0.34441344413444136,
    0.3444234442344423,
    0.34443344433444334,
    0.34444344443444436,
    0.3444534445344453,
    0.34446344463444634,
    0.34447344473444735,
    0.3444834448344483,
    0.34449344493444933,
    0.34450344503445035,
    0.3445134451344513,
    0.34452344523445233,
    0.34453344533445335,
    0.34454344543445437,
    0.3445534455344553,
    0.34456344563445634,
    0.34457344573445736,
    0.3445834458344583,
    0.34459344593445934,
    0.34460344603446036,
    0.3446134461344613,
    0.34462344623446234,
    0.34463344633446336,
    0.3446434464344643,
    0.34465344653446534,
    0.34466344663446635,
    0.3446734467344673,
    0.34468344683446833,
    0.34469344693446935,
    0.3447034470344703,
    0.34471344713447133,
    0.34472344723447235,
    0.34473344733447336,
    0.3447434474344743,
    0.34475344753447534,
    0.34476344763447636,
    0.3447734477344773,
    0.34478344783447834,
    0.34479344793447936,
    0.3448034480344803,
    0.34481344813448134,
    0.34482344823448235,
    0.3448334483344833,
    0.34484344843448433,
    0.34485344853448535,
    0.3448634486344863,
    0.34487344873448733,
    0.34488344883448835,
    0.34489344893448937,
    0.34490344903449033,
    0.34491344913449135,
    0.34492344923449236,
    0.3449334493344933,
    0.34494344943449434,
    0.34495344953449536,
    0.3449634496344963,
    0.34497344973449734,
    0.34498344983449836,
    0.3449934499344993,
    0.34500345003450034,
    0.34501345013450135,
    0.3450234502345023,
    0.34503345033450333,
    0.34504345043450435,
    0.3450534505345053,
    0.34506345063450633,
    0.34507345073450735,
    0.34508345083450837,
    0.3450934509345093,
    0.34510345103451034,
    0.34511345113451136,
    0.3451234512345123,
    0.34513345133451334,
    0.34514345143451436,
    0.3451534515345153,
    0.34516345163451634,
    0.34517345173451736,
    0.3451834518345183,
    0.34519345193451934,
    0.34520345203452035,
    0.3452134521345213,
    0.34522345223452233,
    0.34523345233452335,
    0.3452434524345243,
    0.34525345253452533,
    0.34526345263452635,
    0.34527345273452736,
    0.3452834528345283,
    0.34529345293452934,
    0.34530345303453036,
    0.3453134531345313,
    0.34532345323453234,
    0.34533345333453336,
    0.3453434534345343,
    0.34535345353453534,
    0.34536345363453635,
    0.3453734537345373,
    0.34538345383453833,
    0.34539345393453935,
    0.3454034540345403,
    0.34541345413454133,
    0.34542345423454235,
    0.34543345433454337,
    0.34544345443454433,
    0.34545345453454535,
    0.34546345463454636,
    0.3454734547345473,
    0.34548345483454834,
    0.34549345493454936,
    0.3455034550345503,
    0.34551345513455134,
    0.34552345523455236,
    0.3455334553345533,
    0.34554345543455434,
    0.34555345553455535,
    0.3455634556345563,
    0.34557345573455733,
    0.34558345583455835,
    0.3455934559345593,
    0.34560345603456033,
    0.34561345613456135,
    0.34562345623456237,
    0.3456334563345633,
    0.34564345643456434,
    0.34565345653456536,
    0.3456634566345663,
    0.34567345673456734,
    0.34568345683456836,
    0.3456934569345693,
    0.34570345703457034,
    0.34571345713457136,
    0.3457234572345723,
    0.34573345733457334,
    0.34574345743457435,
    0.3457534575345753,
    0.34576345763457633,
    0.34577345773457735,
    0.34578345783457837,
    0.34579345793457933,
    0.34580345803458035,
    0.34581345813458136,
    0.3458234582345823,
    0.34583345833458334,
    0.34584345843458436,
    0.3458534585345853,
    0.34586345863458634,
    0.34587345873458736,
    0.3458834588345883,
    0.34589345893458934,
    0.34590345903459035,
    0.3459134591345913,
    0.34592345923459233,
    0.34593345933459335,
    0.3459434594345943,
    0.34595345953459533,
    0.34596345963459635,
    0.34597345973459737,
    0.34598345983459833,
    0.34599345993459935,
    0.34600346003460036,
    0.3460134601346013,
    0.34602346023460234,
    0.34603346033460336,
    0.3460434604346043,
    0.34605346053460534,
    0.34606346063460636,
    0.3460734607346073,
    0.34608346083460834,
    0.34609346093460935,
    0.3461034610346103,
    0.34611346113461133,
    0.34612346123461235,
    0.3461334613346133,
    0.34614346143461433,
    0.34615346153461535,
    0.34616346163461637,
    0.3461734617346173,
    0.34618346183461834,
    0.34619346193461936,
    0.3462034620346203,
    0.34621346213462134,
    0.34622346223462236,
    0.3462334623346233,
    0.34624346243462434,
    0.34625346253462536,
    0.3462634626346263,
    0.34627346273462734,
    0.34628346283462835,
    0.3462934629346293,
    0.34630346303463033,
    0.34631346313463135,
    0.34632346323463237,
    0.34633346333463333,
    0.34634346343463435,
    0.34635346353463536,
    0.3463634636346363,
    0.34637346373463734,
    0.34638346383463836,
    0.3463934639346393,
    0.34640346403464034,
    0.34641346413464136,
    0.3464234642346423,
    0.34643346433464334,
    0.34644346443464435,
    0.3464534645346453,
    0.34646346463464633,
    0.34647346473464735,
    0.3464834648346483,
    0.34649346493464933,
    0.34650346503465035,
    0.34651346513465137,
    0.34652346523465233,
    0.34653346533465335,
    0.34654346543465436,
    0.3465534655346553,
    0.34656346563465634,
    0.34657346573465736,
    0.3465834658346583,
    0.34659346593465934,
    0.34660346603466036,
    0.3466134661346613,
    0.34662346623466234,
    0.34663346633466335,
    0.3466434664346643,
    0.34665346653466533,
    0.34666346663466635,
    0.34667346673466737,
    0.34668346683466833,
    0.34669346693466935,
    0.34670346703467037,
    0.3467134671346713,
    0.34672346723467234,
    0.34673346733467336,
    0.3467434674346743,
    0.34675346753467534,
    0.34676346763467636,
    0.3467734677346773,
    0.34678346783467834,
    0.34679346793467936,
    0.3468034680346803,
    0.34681346813468134,
    0.34682346823468235,
    0.3468334683346833,
    0.34684346843468433,
    0.34685346853468535,
    0.34686346863468637,
    0.34687346873468733,
    0.34688346883468835,
    0.34689346893468936,
    0.3469034690346903,
    0.34691346913469134,
    0.34692346923469236,
    0.3469334693346933,
    0.34694346943469434,
    0.34695346953469536,
    0.3469634696346963,
    0.34697346973469734,
    0.34698346983469835,
    0.3469934699346993,
    0.34700347003470033,
    0.34701347013470135,
    0.3470234702347023,
    0.34703347033470333,
    0.34704347043470435,
    0.34705347053470537,
    0.34706347063470633,
    0.34707347073470735,
    0.34708347083470836,
    0.3470934709347093,
    0.34710347103471034,
    0.34711347113471136,
    0.3471234712347123,
    0.34713347133471334,
    0.34714347143471436,
    0.3471534715347153,
    0.34716347163471634,
    0.34717347173471735,
    0.3471834718347183,
    0.34719347193471933,
    0.34720347203472035,
    0.34721347213472137,
    0.34722347223472233,
    0.34723347233472335,
    0.34724347243472437,
    0.3472534725347253,
    0.34726347263472634,
    0.34727347273472736,
    0.3472834728347283,
    0.34729347293472934,
    0.34730347303473036,
    0.3473134731347313,
    0.34732347323473234,
    0.34733347333473336,
    0.3473434734347343,
    0.34735347353473534,
    0.34736347363473635,
    0.3473734737347373,
    0.34738347383473833,
    0.34739347393473935,
    0.34740347403474037,
    0.34741347413474133,
    0.34742347423474235,
    0.34743347433474336,
    0.3474434744347443,
    0.34745347453474534,
    0.34746347463474636,
    0.3474734747347473,
    0.34748347483474834,
    0.34749347493474936,
    0.3475034750347503,
    0.34751347513475134,
    0.34752347523475235,
    0.3475334753347533,
    0.34754347543475433,
    0.34755347553475535,
    0.34756347563475637,
    0.34757347573475733,
    0.34758347583475835,
    0.34759347593475937,
    0.34760347603476033,
    0.34761347613476135,
    0.34762347623476236,
    0.3476334763347633,
    0.34764347643476434,
    0.34765347653476536,
    0.3476634766347663,
    0.34767347673476734,
    0.34768347683476836,
    0.3476934769347693,
    0.34770347703477034,
    0.34771347713477135,
    0.3477234772347723,
    0.34773347733477333,
    0.34774347743477435,
    0.34775347753477537,
    0.34776347763477633,
    0.34777347773477735,
    0.34778347783477837,
    0.3477934779347793,
    0.34780347803478034,
    0.34781347813478136,
    0.3478234782347823,
    0.34783347833478334,
    0.34784347843478436,
    0.3478534785347853,
    0.34786347863478634,
    0.34787347873478736,
    0.3478834788347883,
    0.34789347893478934,
    0.34790347903479035,
    0.3479134791347913,
    0.34792347923479233,
    0.34793347933479335,
    0.34794347943479437,
    0.34795347953479533,
    0.34796347963479635,
    0.34797347973479736,
    0.3479834798347983,
    0.34799347993479934,
    0.34800348003480036,
    0.3480134801348013,
    0.34802348023480234,
    0.34803348033480336,
    0.3480434804348043,
    0.34805348053480534,
    0.34806348063480635,
    0.3480734807348073,
    0.34808348083480833,
    0.34809348093480935,
    0.34810348103481037,
    0.34811348113481133,
    0.34812348123481235,
    0.34813348133481337,
    0.34814348143481433,
    0.34815348153481535,
    0.34816348163481636,
    0.3481734817348173,
    0.34818348183481834,
    0.34819348193481936,
    0.3482034820348203,
    0.34821348213482134,
    0.34822348223482236,
    0.3482334823348233,
    0.34824348243482434,
    0.34825348253482535,
    0.3482634826348263,
    0.34827348273482733,
    0.34828348283482835,
    0.34829348293482937,
    0.34830348303483033,
    0.34831348313483135,
    0.34832348323483237,
    0.3483334833348333,
    0.34834348343483434,
    0.34835348353483536,
    0.3483634836348363,
    0.34837348373483734,
    0.34838348383483836,
    0.3483934839348393,
    0.34840348403484034,
    0.34841348413484136,
    0.3484234842348423,
    0.34843348433484334,
    0.34844348443484435,
    0.34845348453484537,
    0.34846348463484633,
    0.34847348473484735,
    0.34848348483484837,
    0.34849348493484933,
    0.34850348503485035,
    0.34851348513485136,
    0.3485234852348523,
    0.34853348533485334,
    0.34854348543485436,
    0.3485534855348553,
    0.34856348563485634,
    0.34857348573485736,
    0.3485834858348583,
    0.34859348593485934,
    0.34860348603486035,
    0.3486134861348613,
    0.34862348623486233,
    0.34863348633486335,
    0.34864348643486437,
    0.34865348653486533,
    0.34866348663486635,
    0.34867348673486737,
    0.34868348683486833,
    0.34869348693486935,
    0.34870348703487036,
    0.3487134871348713,
    0.34872348723487234,
    0.34873348733487336,
    0.3487434874348743,
    0.34875348753487534,
    0.34876348763487636,
    0.3487734877348773,
    0.34878348783487834,
    0.34879348793487935,
    0.3488034880348803,
    0.34881348813488133,
    0.34882348823488235,
    0.34883348833488337,
    0.34884348843488433,
    0.34885348853488535,
    0.34886348863488637,
    0.3488734887348873,
    0.34888348883488834,
    0.34889348893488936,
    0.3489034890348903,
    0.34891348913489134,
    0.34892348923489236,
    0.3489334893348933,
    0.34894348943489434,
    0.34895348953489536,
    0.3489634896348963,
    0.34897348973489734,
    0.34898348983489835,
    0.34899348993489937,
    0.34900349003490033,
    0.34901349013490135,
    0.34902349023490237,
    0.34903349033490333,
    0.34904349043490435,
    0.34905349053490536,
    0.3490634906349063,
    0.34907349073490734,
    0.34908349083490836,
    0.3490934909349093,
    0.34910349103491034,
    0.34911349113491136,
    0.3491234912349123,
    0.34913349133491334,
    0.34914349143491435,
    0.3491534915349153,
    0.34916349163491633,
    0.34917349173491735,
    0.34918349183491837,
    0.34919349193491933,
    0.34920349203492035,
    0.34921349213492137,
    0.34922349223492233,
    0.34923349233492335,
    0.34924349243492436,
    0.3492534925349253,
    0.34926349263492634,
    0.34927349273492736,
    0.3492834928349283,
    0.34929349293492934,
    0.34930349303493036,
    0.3493134931349313,
    0.34932349323493234,
    0.34933349333493335,
    0.3493434934349343,
    0.34935349353493533,
    0.34936349363493635,
    0.34937349373493737,
    0.34938349383493833,
    0.34939349393493935,
    0.34940349403494037,
    0.3494134941349413,
    0.34942349423494234,
    0.34943349433494336,
    0.3494434944349443,
    0.34945349453494534,
    0.34946349463494636,
    0.3494734947349473,
    0.34948349483494834,
    0.34949349493494936,
    0.3495034950349503,
    0.34951349513495134,
    0.34952349523495235,
    0.34953349533495337,
    0.34954349543495433,
    0.34955349553495535,
    0.34956349563495637,
    0.34957349573495733,
    0.34958349583495835,
    0.34959349593495936,
    0.3496034960349603,
    0.34961349613496134,
    0.34962349623496236,
    0.3496334963349633,
    0.34964349643496434,
    0.34965349653496536,
    0.3496634966349663,
    0.34967349673496734,
    0.34968349683496835,
    0.3496934969349693,
    0.34970349703497033,
    0.34971349713497135,
    0.34972349723497237,
    0.34973349733497333,
    0.34974349743497435,
    0.34975349753497537,
    0.34976349763497633,
    0.34977349773497735,
    0.34978349783497836,
    0.3497934979349793,
    0.34980349803498034,
    0.34981349813498136,
    0.3498234982349823,
    0.34983349833498334,
    0.34984349843498436,
    0.3498534985349853,
    0.34986349863498634,
    0.34987349873498735,
    0.34988349883498837,
    0.34989349893498933,
    0.34990349903499035,
    0.34991349913499137,
    0.34992349923499233,
    0.34993349933499335,
    0.34994349943499437,
    0.3499534995349953,
    0.34996349963499634,
    0.34997349973499736,
    0.3499834998349983,
    0.34999349993499934,
    0.35000350003500036,
    0.3500135001350013,
    0.35002350023500234,
    0.35003350033500336,
    0.3500435004350043,
    0.35005350053500534,
    0.35006350063500635,
    0.35007350073500737,
    0.35008350083500833,
    0.35009350093500935,
    0.35010350103501037,
    0.35011350113501133,
    0.35012350123501235,
    0.35013350133501336,
    0.3501435014350143,
    0.35015350153501534,
    0.35016350163501636,
    0.3501735017350173,
    0.35018350183501834,
    0.35019350193501936,
    0.3502035020350203,
    0.35021350213502134,
    0.35022350223502235,
    0.3502335023350233,
    0.35024350243502433,
    0.35025350253502535,
    0.35026350263502637,
    0.35027350273502733,
    0.35028350283502835,
    0.35029350293502937,
    0.35030350303503033,
    0.35031350313503135,
    0.35032350323503236,
    0.3503335033350333,
    0.35034350343503434,
    0.35035350353503536,
    0.3503635036350363,
    0.35037350373503734,
    0.35038350383503836,
    0.3503935039350393,
    0.35040350403504034,
    0.35041350413504135,
    0.35042350423504237,
    0.35043350433504333,
    0.35044350443504435,
    0.35045350453504537,
    0.35046350463504633,
    0.35047350473504735,
    0.35048350483504837,
    0.3504935049350493,
    0.35050350503505034,
    0.35051350513505136,
    0.3505235052350523,
    0.35053350533505334,
    0.35054350543505436,
    0.3505535055350553,
    0.35056350563505634,
    0.35057350573505736,
    0.3505835058350583,
    0.35059350593505934,
    0.35060350603506035,
    0.35061350613506137,
    0.35062350623506233,
    0.35063350633506335,
    0.35064350643506437,
    0.35065350653506533,
    0.35066350663506635,
    0.35067350673506736,
    0.3506835068350683,
    0.35069350693506934,
    0.35070350703507036,
    0.3507135071350713,
    0.35072350723507234,
    0.35073350733507336,
    0.3507435074350743,
    0.35075350753507534,
    0.35076350763507635,
    0.3507735077350774,
    0.35078350783507833,
    0.35079350793507935,
    0.35080350803508037,
    0.35081350813508133,
    0.35082350823508235,
    0.35083350833508337,
    0.35084350843508433,
    0.35085350853508535,
    0.35086350863508636,
    0.3508735087350873,
    0.35088350883508834,
    0.35089350893508936,
    0.3509035090350903,
    0.35091350913509134,
    0.35092350923509236,
    0.3509335093350933,
    0.35094350943509434,
    0.35095350953509535,
    0.35096350963509637,
    0.35097350973509733,
    0.35098350983509835,
    0.35099350993509937,
    0.35100351003510033,
    0.35101351013510135,
    0.35102351023510237,
    0.3510335103351033,
    0.35104351043510434,
    0.35105351053510536,
    0.3510635106351063,
    0.35107351073510734,
    0.35108351083510836,
    0.3510935109351093,
    0.35110351103511034,
    0.35111351113511136,
    0.3511235112351123,
    0.35113351133511334,
    0.35114351143511435,
    0.35115351153511537,
    0.35116351163511633,
    0.35117351173511735,
    0.35118351183511837,
    0.35119351193511933,
    0.35120351203512035,
    0.35121351213512136,
    0.3512235122351223,
    0.35123351233512334,
    0.35124351243512436,
    0.3512535125351253,
    0.35126351263512634,
    0.35127351273512736,
    0.3512835128351283,
    0.35129351293512934,
    0.35130351303513035,
    0.3513135131351314,
    0.35132351323513233,
    0.35133351333513335,
    0.35134351343513437,
    0.35135351353513533,
    0.35136351363513635,
    0.35137351373513737,
    0.35138351383513833,
    0.35139351393513935,
    0.35140351403514036,
    0.3514135141351413,
    0.35142351423514234,
    0.35143351433514336,
    0.3514435144351443,
    0.35145351453514534,
    0.35146351463514636,
    0.3514735147351473,
    0.35148351483514834,
    0.35149351493514935,
    0.35150351503515037,
    0.35151351513515133,
    0.35152351523515235,
    0.35153351533515337,
    0.35154351543515433,
    0.35155351553515535,
    0.35156351563515637,
    0.3515735157351573,
    0.35158351583515834,
    0.35159351593515936,
    0.3516035160351603,
    0.35161351613516134,
    0.35162351623516236,
    0.3516335163351633,
    0.35164351643516434,
    0.35165351653516536,
    0.3516635166351664,
    0.35167351673516734,
    0.35168351683516835,
    0.35169351693516937,
    0.35170351703517033,
    0.35171351713517135,
    0.35172351723517237,
    0.35173351733517333,
    0.35174351743517435,
    0.35175351753517536,
    0.3517635176351763,
    0.35177351773517734,
    0.35178351783517836,
    0.3517935179351793,
    0.35180351803518034,
    0.35181351813518136,
    0.3518235182351823,
    0.35183351833518334,
    0.35184351843518435,
    0.35185351853518537,
    0.35186351863518633,
    0.35187351873518735,
    0.35188351883518837,
    0.35189351893518933,
    0.35190351903519035,
    0.35191351913519137,
    0.35192351923519233,
    0.35193351933519335,
    0.35194351943519436,
    0.3519535195351953,
    0.35196351963519634,
    0.35197351973519736,
    0.3519835198351983,
    0.35199351993519934,
    0.35200352003520036,
    0.3520135201352013,
    0.35202352023520234,
    0.35203352033520335,
    0.35204352043520437,
    0.35205352053520533,
    0.35206352063520635,
    0.35207352073520737,
    0.35208352083520833,
    0.35209352093520935,
    0.35210352103521037,
    0.3521135211352113,
    0.35212352123521234,
    0.35213352133521336,
    0.3521435214352143,
    0.35215352153521534,
    0.35216352163521636,
    0.3521735217352173,
    0.35218352183521834,
    0.35219352193521936,
    0.3522035220352204,
    0.35221352213522134,
    0.35222352223522235,
    0.35223352233522337,
    0.35224352243522433,
    0.35225352253522535,
    0.35226352263522637,
    0.35227352273522733,
    0.35228352283522835,
    0.35229352293522936,
    0.3523035230352303,
    0.35231352313523134,
    0.35232352323523236,
    0.3523335233352333,
    0.35234352343523434,
    0.35235352353523536,
    0.3523635236352363,
    0.35237352373523734,
    0.35238352383523835,
    0.35239352393523937,
    0.35240352403524033,
    0.35241352413524135,
    0.35242352423524237,
    0.35243352433524333,
    0.35244352443524435,
    0.35245352453524537,
    0.35246352463524633,
    0.35247352473524735,
    0.35248352483524836,
    0.3524935249352493,
    0.35250352503525034,
    0.35251352513525136,
    0.3525235252352523,
    0.35253352533525334,
    0.35254352543525436,
    0.3525535255352553,
    0.35256352563525634,
    0.35257352573525735,
    0.35258352583525837,
    0.35259352593525933,
    0.35260352603526035,
    0.35261352613526137,
    0.35262352623526233,
    0.35263352633526335,
    0.35264352643526437,
    0.3526535265352653,
    0.35266352663526634,
    0.35267352673526736,
    0.3526835268352683,
    0.35269352693526934,
    0.35270352703527036,
    0.3527135271352713,
    0.35272352723527234,
    0.35273352733527336,
    0.3527435274352744,
    0.35275352753527534,
    0.35276352763527635,
    0.35277352773527737,
    0.35278352783527833,
    0.35279352793527935,
    0.35280352803528037,
    0.35281352813528133,
    0.35282352823528235,
    0.35283352833528336,
    0.3528435284352843,
    0.35285352853528534,
    0.35286352863528636,
    0.3528735287352873,
    0.35288352883528834,
    0.35289352893528936,
    0.3529035290352903,
    0.35291352913529134,
    0.35292352923529235,
    0.35293352933529337,
    0.35294352943529433,
    0.35295352953529535,
    0.35296352963529637,
    0.35297352973529733,
    0.35298352983529835,
    0.35299352993529937,
    0.35300353003530033,
    0.35301353013530135,
    0.35302353023530236,
    0.3530335303353033,
    0.35304353043530434,
    0.35305353053530536,
    0.3530635306353063,
    0.35307353073530734,
    0.35308353083530836,
    0.3530935309353094,
    0.35310353103531034,
    0.35311353113531135,
    0.35312353123531237,
    0.35313353133531333,
    0.35314353143531435,
    0.35315353153531537,
    0.35316353163531633,
    0.35317353173531735,
    0.35318353183531837,
    0.3531935319353193,
    0.35320353203532034,
    0.35321353213532136,
    0.3532235322353223,
    0.35323353233532334,
    0.35324353243532436,
    0.3532535325353253,
    0.35326353263532634,
    0.35327353273532736,
    0.3532835328353284,
    0.35329353293532934,
    0.35330353303533035,
    0.35331353313533137,
    0.35332353323533233,
    0.35333353333533335,
    0.35334353343533437,
    0.35335353353533533,
    0.35336353363533635,
    0.35337353373533736,
    0.3533835338353383,
    0.35339353393533934,
    0.35340353403534036,
    0.3534135341353413,
    0.35342353423534234,
    0.35343353433534336,
    0.3534435344353443,
    0.35345353453534534,
    0.35346353463534635,
    0.35347353473534737,
    0.35348353483534833,
    0.35349353493534935,
    0.35350353503535037,
    0.35351353513535133,
    0.35352353523535235,
    0.35353353533535337,
    0.35354353543535433,
    0.35355353553535535,
    0.35356353563535636,
    0.3535735357353573,
    0.35358353583535834,
    0.35359353593535936,
    0.3536035360353603,
    0.35361353613536134,
    0.35362353623536236,
    0.3536335363353634,
    0.35364353643536434,
    0.35365353653536535,
    0.35366353663536637,
    0.35367353673536733,
    0.35368353683536835,
    0.35369353693536937,
    0.35370353703537033,
    0.35371353713537135,
    0.35372353723537237,
    0.3537335373353733,
    0.35374353743537434,
    0.35375353753537536,
    0.3537635376353763,
    0.35377353773537734,
    0.35378353783537836,
    0.3537935379353793,
    0.35380353803538034,
    0.35381353813538136,
    0.3538235382353824,
    0.35383353833538334,
    0.35384353843538435,
    0.35385353853538537,
    0.35386353863538633,
    0.35387353873538735,
    0.35388353883538837,
    0.35389353893538933,
    0.35390353903539035,
    0.35391353913539136,
    0.3539235392353923,
    0.35393353933539334,
    0.35394353943539436,
    0.3539535395353953,
    0.35396353963539634,
    0.35397353973539736,
    0.3539835398353984,
    0.35399353993539934,
    0.35400354003540035,
    0.35401354013540137,
    0.35402354023540233,
    0.35403354033540335,
    0.35404354043540437,
    0.35405354053540533,
    0.35406354063540635,
    0.35407354073540737,
    0.35408354083540833,
    0.35409354093540935,
    0.35410354103541036,
    0.3541135411354113,
    0.35412354123541234,
    0.35413354133541336,
    0.3541435414354143,
    0.35415354153541534,
    0.35416354163541636,
    0.3541735417354174,
    0.35418354183541834,
    0.35419354193541935,
    0.35420354203542037,
    0.35421354213542133,
    0.35422354223542235,
    0.35423354233542337,
    0.35424354243542433,
    0.35425354253542535,
    0.35426354263542637,
    0.3542735427354273,
    0.35428354283542834,
    0.35429354293542936,
    0.3543035430354303,
    0.35431354313543134,
    0.35432354323543236,
    0.3543335433354333,
    0.35434354343543434,
    0.35435354353543536,
    0.3543635436354364,
    0.35437354373543734,
    0.35438354383543835,
    0.35439354393543937,
    0.35440354403544033,
    0.35441354413544135,
    0.35442354423544237,
    0.35443354433544333,
    0.35444354443544435,
    0.35445354453544536,
    0.3544635446354463,
    0.35447354473544734,
    0.35448354483544836,
    0.3544935449354493,
    0.35450354503545034,
    0.35451354513545136,
    0.3545235452354524,
    0.35453354533545334,
    0.35454354543545435,
    0.35455354553545537,
    0.35456354563545633,
    0.35457354573545735,
    0.35458354583545837,
    0.35459354593545933,
    0.35460354603546035,
    0.35461354613546137,
    0.35462354623546233,
    0.35463354633546335,
    0.35464354643546436,
    0.3546535465354653,
    0.35466354663546634,
    0.35467354673546736,
    0.3546835468354683,
    0.35469354693546934,
    0.35470354703547036,
    0.3547135471354714,
    0.35472354723547234,
    0.35473354733547335,
    0.35474354743547437,
    0.35475354753547533,
    0.35476354763547635,
    0.35477354773547737,
    0.35478354783547833,
    0.35479354793547935,
    0.35480354803548037,
    0.3548135481354813,
    0.35482354823548234,
    0.35483354833548336,
    0.3548435484354843,
    0.35485354853548534,
    0.35486354863548636,
    0.3548735487354874,
    0.35488354883548834,
    0.35489354893548936,
    0.3549035490354904,
    0.35491354913549134,
    0.35492354923549235,
    0.35493354933549337,
    0.35494354943549433,
    0.35495354953549535,
    0.35496354963549637,
    0.35497354973549733,
    0.35498354983549835,
    0.35499354993549936,
    0.3550035500355003,
    0.35501355013550134,
    0.35502355023550236,
    0.3550335503355033,
    0.35504355043550434,
    0.35505355053550536,
    0.3550635506355064,
    0.35507355073550734,
    0.35508355083550835,
    0.35509355093550937,
    0.35510355103551033,
    0.35511355113551135,
    0.35512355123551237,
    0.35513355133551333,
    0.35514355143551435,
    0.35515355153551537,
    0.35516355163551633,
    0.35517355173551735,
    0.35518355183551836,
    0.3551935519355193,
    0.35520355203552034,
    0.35521355213552136,
    0.3552235522355223,
    0.35523355233552334,
    0.35524355243552436,
    0.3552535525355254,
    0.35526355263552634,
    0.35527355273552735,
    0.35528355283552837,
    0.35529355293552933,
    0.35530355303553035,
    0.35531355313553137,
    0.35532355323553233,
    0.35533355333553335,
    0.35534355343553437,
    0.3553535535355353,
    0.35536355363553634,
    0.35537355373553736,
    0.3553835538355383,
    0.35539355393553934,
    0.35540355403554036,
    0.3554135541355414,
    0.35542355423554234,
    0.35543355433554336,
    0.3554435544355444,
    0.35545355453554534,
    0.35546355463554635,
    0.35547355473554737,
    0.35548355483554833,
    0.35549355493554935,
    0.35550355503555037,
    0.35551355513555133,
    0.35552355523555235,
    0.35553355533555336,
    0.3555435554355543,
    0.35555355553555534,
    0.35556355563555636,
    0.3555735557355573,
    0.35558355583555834,
    0.35559355593555936,
    0.3556035560355604,
    0.35561355613556134,
    0.35562355623556235,
    0.35563355633556337,
    0.35564355643556433,
    0.35565355653556535,
    0.35566355663556637,
    0.35567355673556733,
    0.35568355683556835,
    0.35569355693556937,
    0.35570355703557033,
    0.35571355713557135,
    0.35572355723557236,
    0.3557335573355733,
    0.35574355743557434,
    0.35575355753557536,
    0.3557635576355764,
    0.35577355773557734,
    0.35578355783557836,
    0.3557935579355794,
    0.35580355803558034,
    0.35581355813558135,
    0.35582355823558237,
    0.35583355833558333,
    0.35584355843558435,
    0.35585355853558537,
    0.35586355863558633,
    0.35587355873558735,
    0.35588355883558837,
    0.3558935589355893,
    0.35590355903559034,
    0.35591355913559136,
    0.3559235592355923,
    0.35593355933559334,
    0.35594355943559436,
    0.3559535595355954,
    0.35596355963559634,
    0.35597355973559736,
    0.3559835598355984,
    0.35599355993559934,
    0.35600356003560035,
    0.35601356013560137,
    0.35602356023560233,
    0.35603356033560335,
    0.35604356043560437,
    0.35605356053560533,
    0.35606356063560635,
    0.35607356073560736,
    0.3560835608356083,
    0.35609356093560934,
    0.35610356103561036,
    0.3561135611356113,
    0.35612356123561234,
    0.35613356133561336,
    0.3561435614356144,
    0.35615356153561534,
    0.35616356163561635,
    0.35617356173561737,
    0.35618356183561833,
    0.35619356193561935,
    0.35620356203562037,
    0.35621356213562133,
    0.35622356223562235,
    0.35623356233562337,
    0.35624356243562433,
    0.35625356253562535,
    0.35626356263562636,
    0.3562735627356273,
    0.35628356283562834,
    0.35629356293562936,
    0.3563035630356304,
    0.35631356313563134,
    0.35632356323563236,
    0.3563335633356334,
    0.35634356343563434,
    0.35635356353563535,
    0.35636356363563637,
    0.35637356373563733,
    0.35638356383563835,
    0.35639356393563937,
    0.35640356403564033,
    0.35641356413564135,
    0.35642356423564237,
    0.3564335643356433,
    0.35644356443564434,
    0.35645356453564536,
    0.3564635646356463,
    0.35647356473564734,
    0.35648356483564836,
    0.3564935649356494,
    0.35650356503565034,
    0.35651356513565136,
    0.3565235652356524,
    0.35653356533565334,
    0.35654356543565435,
    0.35655356553565537,
    0.35656356563565633,
    0.35657356573565735,
    0.35658356583565837,
    0.35659356593565933,
    0.35660356603566035,
    0.35661356613566136,
    0.3566235662356623,
    0.35663356633566334,
    0.35664356643566436,
    0.3566535665356653,
    0.35666356663566634,
    0.35667356673566736,
    0.3566835668356684,
    0.35669356693566934,
    0.35670356703567035,
    0.35671356713567137,
    0.35672356723567233,
    0.35673356733567335,
    0.35674356743567437,
    0.35675356753567533,
    0.35676356763567635,
    0.35677356773567737,
    0.35678356783567833,
    0.35679356793567935,
    0.35680356803568036,
    0.3568135681356813,
    0.35682356823568234,
    0.35683356833568336,
    0.3568435684356844,
    0.35685356853568534,
    0.35686356863568636,
    0.3568735687356874,
    0.35688356883568834,
    0.35689356893568935,
    0.35690356903569037,
    0.35691356913569133,
    0.35692356923569235,
    0.35693356933569337,
    0.35694356943569433,
    0.35695356953569535,
    0.35696356963569637,
    0.3569735697356973,
    0.35698356983569834,
    0.35699356993569936,
    0.3570035700357003,
    0.35701357013570134,
    0.35702357023570236,
    0.3570335703357034,
    0.35704357043570434,
    0.35705357053570536,
    0.3570635706357064,
    0.35707357073570734,
    0.35708357083570835,
    0.35709357093570937,
    0.35710357103571033,
    0.35711357113571135,
    0.35712357123571237,
    0.35713357133571333,
    0.35714357143571435,
    0.35715357153571536,
    0.3571635716357163,
    0.35717357173571734,
    0.35718357183571836,
    0.3571935719357194,
    0.35720357203572034,
    0.35721357213572136,
    0.3572235722357224,
    0.35723357233572334,
    0.35724357243572435,
    0.35725357253572537,
    0.35726357263572633,
    0.35727357273572735,
    0.35728357283572837,
    0.35729357293572933,
    0.35730357303573035,
    0.35731357313573137,
    0.35732357323573233,
    0.35733357333573335,
    0.35734357343573436,
    0.3573535735357353,
    0.35736357363573634,
    0.35737357373573736,
    0.3573835738357384,
    0.35739357393573934,
    0.35740357403574036,
    0.3574135741357414,
    0.35742357423574234,
    0.35743357433574335,
    0.35744357443574437,
    0.35745357453574533,
    0.35746357463574635,
    0.35747357473574737,
    0.35748357483574833,
    0.35749357493574935,
    0.35750357503575036,
    0.3575135751357513,
    0.35752357523575234,
    0.35753357533575336,
    0.3575435754357543,
    0.35755357553575534,
    0.35756357563575636,
    0.3575735757357574,
    0.35758357583575834,
    0.35759357593575936,
    0.3576035760357604,
    0.35761357613576134,
    0.35762357623576235,
    0.35763357633576337,
    0.35764357643576433,
    0.35765357653576535,
    0.35766357663576637,
    0.35767357673576733,
    0.35768357683576835,
    0.35769357693576936,
    0.3577035770357703,
    0.35771357713577134,
    0.35772357723577236,
    0.3577335773357734,
    0.35774357743577434,
    0.35775357753577536,
    0.3577635776357764,
    0.35777357773577734,
    0.35778357783577835,
    0.35779357793577937,
    0.35780357803578033,
    0.35781357813578135,
    0.35782357823578237,
    0.35783357833578333,
    0.35784357843578435,
    0.35785357853578537,
    0.35786357863578633,
    0.35787357873578735,
    0.35788357883578836,
    0.3578935789357893,
    0.35790357903579034,
    0.35791357913579136,
    0.3579235792357924,
    0.35793357933579334,
    0.35794357943579436,
    0.3579535795357954,
    0.35796357963579634,
    0.35797357973579735,
    0.35798357983579837,
    0.35799357993579933,
    0.35800358003580035,
    0.35801358013580137,
    0.35802358023580233,
    0.35803358033580335,
    0.35804358043580436,
    0.3580535805358053,
    0.35806358063580634,
    0.35807358073580736,
    0.3580835808358084,
    0.35809358093580934,
    0.35810358103581036,
    0.3581135811358114,
    0.35812358123581234,
    0.35813358133581336,
    0.3581435814358144,
    0.35815358153581534,
    0.35816358163581635,
    0.35817358173581737,
    0.35818358183581833,
    0.35819358193581935,
    0.35820358203582037,
    0.35821358213582133,
    0.35822358223582235,
    0.35823358233582336,
    0.3582435824358243,
    0.35825358253582534,
    0.35826358263582636,
    0.3582735827358274,
    0.35828358283582834,
    0.35829358293582936,
    0.3583035830358304,
    0.35831358313583134,
    0.35832358323583235,
    0.35833358333583337,
    0.35834358343583433,
    0.35835358353583535,
    0.35836358363583637,
    0.35837358373583733,
    0.35838358383583835,
    0.35839358393583937,
    0.35840358403584033,
    0.35841358413584135,
    0.35842358423584236,
    0.3584335843358433,
    0.35844358443584434,
    0.35845358453584536,
    0.3584635846358464,
    0.35847358473584734,
    0.35848358483584836,
    0.3584935849358494,
    0.35850358503585034,
    0.35851358513585135,
    0.35852358523585237,
    0.35853358533585333,
    0.35854358543585435,
    0.35855358553585537,
    0.35856358563585633,
    0.35857358573585735,
    0.35858358583585836,
    0.3585935859358593,
    0.35860358603586034,
    0.35861358613586136,
    0.3586235862358624,
    0.35863358633586334,
    0.35864358643586436,
    0.3586535865358654,
    0.35866358663586634,
    0.35867358673586736,
    0.3586835868358684,
    0.35869358693586934,
    0.35870358703587035,
    0.35871358713587137,
    0.35872358723587233,
    0.35873358733587335,
    0.35874358743587437,
    0.35875358753587533,
    0.35876358763587635,
    0.35877358773587736,
    0.3587835878358783,
    0.35879358793587934,
    0.35880358803588036,
    0.3588135881358814,
    0.35882358823588234,
    0.35883358833588336,
    0.3588435884358844,
    0.35885358853588534,
    0.35886358863588635,
    0.35887358873588737,
    0.35888358883588833,
    0.35889358893588935,
    0.35890358903589037,
    0.35891358913589133,
    0.35892358923589235,
    0.35893358933589337,
    0.35894358943589433,
    0.35895358953589535,
    0.35896358963589636,
    0.3589735897358974,
    0.35898358983589834,
    0.35899358993589936,
    0.3590035900359004,
    0.35901359013590134,
    0.35902359023590236,
    0.3590335903359034,
    0.35904359043590434,
    0.35905359053590535,
    0.35906359063590637,
    0.35907359073590733,
    0.35908359083590835,
    0.35909359093590937,
    0.35910359103591033,
    0.35911359113591135,
    0.35912359123591236,
    0.3591335913359133,
    0.35914359143591434,
    0.35915359153591536,
    0.3591635916359164,
    0.35917359173591734,
    0.35918359183591836,
    0.3591935919359194,
    0.35920359203592034,
    0.35921359213592136,
    0.3592235922359224,
    0.35923359233592334,
    0.35924359243592435,
    0.35925359253592537,
    0.35926359263592633,
    0.35927359273592735,
    0.35928359283592837,
    0.35929359293592933,
    0.35930359303593035,
    0.35931359313593136,
    0.3593235932359323,
    0.35933359333593334,
    0.35934359343593436,
    0.3593535935359354,
    0.35936359363593634,
    0.35937359373593736,
    0.3593835938359384,
    0.35939359393593934,
    0.35940359403594035,
    0.35941359413594137,
    0.35942359423594233,
    0.35943359433594335,
    0.35944359443594437,
    0.35945359453594533,
    0.35946359463594635,
    0.35947359473594737,
    0.35948359483594833,
    0.35949359493594935,
    0.35950359503595036,
    0.3595135951359514,
    0.35952359523595234,
    0.35953359533595336,
    0.3595435954359544,
    0.35955359553595534,
    0.35956359563595636,
    0.3595735957359574,
    0.35958359583595834,
    0.35959359593595935,
    0.35960359603596037,
    0.35961359613596133,
    0.35962359623596235,
    0.35963359633596337,
    0.35964359643596433,
    0.35965359653596535,
    0.35966359663596636,
    0.3596735967359673,
    0.35968359683596834,
    0.35969359693596936,
    0.3597035970359704,
    0.35971359713597134,
    0.35972359723597236,
    0.3597335973359734,
    0.35974359743597434,
    0.35975359753597536,
    0.3597635976359764,
    0.35977359773597734,
    0.35978359783597835,
    0.35979359793597937,
    0.35980359803598033,
    0.35981359813598135,
    0.35982359823598237,
    0.35983359833598333,
    0.35984359843598435,
    0.35985359853598536,
    0.3598635986359864,
    0.35987359873598734,
    0.35988359883598836,
    0.3598935989359894,
    0.35990359903599034,
    0.35991359913599136,
    0.3599235992359924,
    0.35993359933599334,
    0.35994359943599435,
    0.35995359953599537,
    0.35996359963599633,
    0.35997359973599735,
    0.35998359983599837,
    0.35999359993599933,
    0.36000360003600035,
    0.36001360013600137,
    0.36002360023600233,
    0.36003360033600335,
    0.36004360043600436,
    0.3600536005360054,
    0.36006360063600634,
    0.36007360073600736,
    0.3600836008360084,
    0.36009360093600934,
    0.36010360103601036,
    0.3601136011360114,
    0.36012360123601234,
    0.36013360133601335,
    0.36014360143601437,
    0.36015360153601533,
    0.36016360163601635,
    0.36017360173601737,
    0.36018360183601833,
    0.36019360193601935,
    0.36020360203602036,
    0.3602136021360213,
    0.36022360223602234,
    0.36023360233602336,
    0.3602436024360244,
    0.36025360253602534,
    0.36026360263602636,
    0.3602736027360274,
    0.36028360283602834,
    0.36029360293602936,
    0.3603036030360304,
    0.36031360313603134,
    0.36032360323603235,
    0.36033360333603337,
    0.36034360343603433,
    0.36035360353603535,
    0.36036360363603637,
    0.36037360373603733,
    0.36038360383603835,
    0.36039360393603936,
    0.3604036040360404,
    0.36041360413604134,
    0.36042360423604236,
    0.3604336043360434,
    0.36044360443604434,
    0.36045360453604536,
    0.3604636046360464,
    0.36047360473604734,
    0.36048360483604835,
    0.36049360493604937,
    0.36050360503605033,
    0.36051360513605135,
    0.36052360523605237,
    0.36053360533605333,
    0.36054360543605435,
    0.36055360553605537,
    0.36056360563605633,
    0.36057360573605735,
    0.36058360583605836,
    0.3605936059360594,
    0.36060360603606034,
    0.36061360613606136,
    0.3606236062360624,
    0.36063360633606334,
    0.36064360643606436,
    0.3606536065360654,
    0.36066360663606634,
    0.36067360673606735,
    0.36068360683606837,
    0.36069360693606933,
    0.36070360703607035,
    0.36071360713607137,
    0.36072360723607233,
    0.36073360733607335,
    0.36074360743607436,
    0.3607536075360753,
    0.36076360763607634,
    0.36077360773607736,
    0.3607836078360784,
    0.36079360793607934,
    0.36080360803608036,
    0.3608136081360814,
    0.36082360823608234,
    0.36083360833608336,
    0.3608436084360844,
    0.36085360853608534,
    0.36086360863608635,
    0.36087360873608737,
    0.36088360883608833,
    0.36089360893608935,
    0.36090360903609037,
    0.36091360913609133,
    0.36092360923609235,
    0.36093360933609336,
    0.3609436094360944,
    0.36095360953609534,
    0.36096360963609636,
    0.3609736097360974,
    0.36098360983609834,
    0.36099360993609936,
    0.3610036100361004,
    0.36101361013610134,
    0.36102361023610235,
    0.36103361033610337,
    0.36104361043610433,
    0.36105361053610535,
    0.36106361063610637,
    0.36107361073610733,
    0.36108361083610835,
    0.36109361093610937,
    0.36110361103611033,
    0.36111361113611135,
    0.36112361123611236,
    0.3611336113361134,
    0.36114361143611434,
    0.36115361153611536,
    0.3611636116361164,
    0.36117361173611734,
    0.36118361183611836,
    0.3611936119361194,
    0.36120361203612034,
    0.36121361213612135,
    0.36122361223612237,
    0.36123361233612333,
    0.36124361243612435,
    0.36125361253612537,
    0.36126361263612633,
    0.36127361273612735,
    0.36128361283612836,
    0.3612936129361294,
    0.36130361303613034,
    0.36131361313613136,
    0.3613236132361324,
    0.36133361333613334,
    0.36134361343613436,
    0.3613536135361354,
    0.36136361363613634,
    0.36137361373613736,
    0.3613836138361384,
    0.36139361393613934,
    0.36140361403614035,
    0.36141361413614137,
    0.36142361423614233,
    0.36143361433614335,
    0.36144361443614437,
    0.36145361453614533,
    0.36146361463614635,
    0.36147361473614736,
    0.3614836148361484,
    0.36149361493614934,
    0.36150361503615036,
    0.3615136151361514,
    0.36152361523615234,
    0.36153361533615336,
    0.3615436154361544,
    0.36155361553615534,
    0.36156361563615635,
    0.36157361573615737,
    0.36158361583615833,
    0.36159361593615935,
    0.36160361603616037,
    0.36161361613616133,
    0.36162361623616235,
    0.36163361633616337,
    0.36164361643616433,
    0.36165361653616535,
    0.36166361663616636,
    0.3616736167361674,
    0.36168361683616834,
    0.36169361693616936,
    0.3617036170361704,
    0.36171361713617134,
    0.36172361723617236,
    0.3617336173361734,
    0.36174361743617434,
    0.36175361753617535,
    0.36176361763617637,
    0.36177361773617733,
    0.36178361783617835,
    0.36179361793617937,
    0.36180361803618033,
    0.36181361813618135,
    0.36182361823618236,
    0.3618336183361834,
    0.36184361843618434,
    0.36185361853618536,
    0.3618636186361864,
    0.36187361873618734,
    0.36188361883618836,
    0.3618936189361894,
    0.36190361903619034,
    0.36191361913619136,
    0.3619236192361924,
    0.36193361933619334,
    0.36194361943619435,
    0.36195361953619537,
    0.36196361963619633,
    0.36197361973619735,
    0.36198361983619837,
    0.36199361993619933,
    0.36200362003620035,
    0.36201362013620136,
    0.3620236202362024,
    0.36203362033620334,
    0.36204362043620436,
    0.3620536205362054,
    0.36206362063620634,
    0.36207362073620736,
    0.3620836208362084,
    0.36209362093620934,
    0.36210362103621035,
    0.36211362113621137,
    0.36212362123621233,
    0.36213362133621335,
    0.36214362143621437,
    0.36215362153621533,
    0.36216362163621635,
    0.36217362173621737,
    0.3621836218362184,
    0.36219362193621935,
    0.36220362203622036,
    0.3622136221362214,
    0.36222362223622234,
    0.36223362233622336,
    0.3622436224362244,
    0.36225362253622534,
    0.36226362263622636,
    0.3622736227362274,
    0.36228362283622834,
    0.36229362293622935,
    0.36230362303623037,
    0.36231362313623133,
    0.36232362323623235,
    0.36233362333623337,
    0.36234362343623433,
    0.36235362353623535,
    0.36236362363623636,
    0.3623736237362374,
    0.36238362383623834,
    0.36239362393623936,
    0.3624036240362404,
    0.36241362413624134,
    0.36242362423624236,
    0.3624336243362434,
    0.36244362443624434,
    0.36245362453624536,
    0.3624636246362464,
    0.36247362473624734,
    0.36248362483624835,
    0.36249362493624937,
    0.36250362503625033,
    0.36251362513625135,
    0.36252362523625237,
    0.36253362533625333,
    0.36254362543625435,
    0.36255362553625536,
    0.3625636256362564,
    0.36257362573625734,
    0.36258362583625836,
    0.3625936259362594,
    0.36260362603626034,
    0.36261362613626136,
    0.3626236262362624,
    0.36263362633626334,
    0.36264362643626435,
    0.36265362653626537,
    0.36266362663626633,
    0.36267362673626735,
    0.36268362683626837,
    0.36269362693626933,
    0.36270362703627035,
    0.36271362713627137,
    0.3627236272362724,
    0.36273362733627335,
    0.36274362743627436,
    0.3627536275362754,
    0.36276362763627634,
    0.36277362773627736,
    0.3627836278362784,
    0.36279362793627934,
    0.36280362803628036,
    0.3628136281362814,
    0.36282362823628234,
    0.36283362833628335,
    0.36284362843628437,
    0.36285362853628533,
    0.36286362863628635,
    0.36287362873628737,
    0.36288362883628833,
    0.36289362893628935,
    0.36290362903629036,
    0.3629136291362914,
    0.36292362923629234,
    0.36293362933629336,
    0.3629436294362944,
    0.36295362953629534,
    0.36296362963629636,
    0.3629736297362974,
    0.36298362983629834,
    0.36299362993629936,
    0.3630036300363004,
    0.36301363013630134,
    0.36302363023630235,
    0.36303363033630337,
    0.36304363043630433,
    0.36305363053630535,
    0.36306363063630637,
    0.3630736307363074,
    0.36308363083630835,
    0.36309363093630936,
    0.3631036310363104,
    0.36311363113631134,
    0.36312363123631236,
    0.3631336313363134,
    0.36314363143631434,
    0.36315363153631536,
    0.3631636316363164,
    0.36317363173631734,
    0.36318363183631835,
    0.36319363193631937,
    0.36320363203632033,
    0.36321363213632135,
    0.36322363223632237,
    0.36323363233632333,
    0.36324363243632435,
    0.36325363253632537,
    0.3632636326363264,
    0.36327363273632735,
    0.36328363283632836,
    0.3632936329363294,
    0.36330363303633034,
    0.36331363313633136,
    0.3633236332363324,
    0.36333363333633334,
    0.36334363343633436,
    0.3633536335363354,
    0.36336363363633634,
    0.36337363373633735,
    0.36338363383633837,
    0.36339363393633933,
    0.36340363403634035,
    0.36341363413634137,
    0.36342363423634233,
    0.36343363433634335,
    0.36344363443634436,
    0.3634536345363454,
    0.36346363463634634,
    0.36347363473634736,
    0.3634836348363484,
    0.36349363493634934,
    0.36350363503635036,
    0.3635136351363514,
    0.36352363523635234,
    0.36353363533635336,
    0.3635436354363544,
    0.36355363553635534,
    0.36356363563635635,
    0.36357363573635737,
    0.36358363583635833,
    0.36359363593635935,
    0.36360363603636037,
    0.3636136361363614,
    0.36362363623636235,
    0.36363363633636336,
    0.3636436364363644,
    0.36365363653636534,
    0.36366363663636636,
    0.3636736367363674,
    0.36368363683636834,
    0.36369363693636936,
    0.3637036370363704,
    0.36371363713637134,
    0.36372363723637235,
    0.36373363733637337,
    0.36374363743637433,
    0.36375363753637535,
    0.36376363763637637,
    0.36377363773637733,
    0.36378363783637835,
    0.36379363793637937,
    0.3638036380363804,
    0.36381363813638135,
    0.36382363823638236,
    0.3638336383363834,
    0.36384363843638434,
    0.36385363853638536,
    0.3638636386363864,
    0.36387363873638734,
    0.36388363883638836,
    0.3638936389363894,
    0.36390363903639034,
    0.36391363913639135,
    0.36392363923639237,
    0.36393363933639333,
    0.36394363943639435,
    0.36395363953639537,
    0.36396363963639633,
    0.36397363973639735,
    0.36398363983639836,
    0.3639936399363994,
    0.36400364003640034,
    0.36401364013640136,
    0.3640236402364024,
    0.36403364033640334,
    0.36404364043640436,
    0.3640536405364054,
    0.36406364063640634,
    0.36407364073640736,
    0.3640836408364084,
    0.36409364093640934,
    0.36410364103641035,
    0.36411364113641137,
    0.36412364123641233,
    0.36413364133641335,
    0.36414364143641437,
    0.3641536415364154,
    0.36416364163641635,
    0.36417364173641736,
    0.3641836418364184,
    0.36419364193641934,
    0.36420364203642036,
    0.3642136421364214,
    0.36422364223642234,
    0.36423364233642336,
    0.3642436424364244,
    0.36425364253642534,
    0.36426364263642635,
    0.36427364273642737,
    0.36428364283642833,
    0.36429364293642935,
    0.36430364303643037,
    0.36431364313643133,
    0.36432364323643235,
    0.36433364333643337,
    0.3643436434364344,
    0.36435364353643535,
    0.36436364363643636,
    0.3643736437364374,
    0.36438364383643834,
    0.36439364393643936,
    0.3644036440364404,
    0.36441364413644134,
    0.36442364423644236,
    0.3644336443364434,
    0.36444364443644434,
    0.36445364453644535,
    0.36446364463644637,
    0.36447364473644733,
    0.36448364483644835,
    0.36449364493644937,
    0.3645036450364504,
    0.36451364513645135,
    0.36452364523645236,
    0.3645336453364534,
    0.36454364543645434,
    0.36455364553645536,
    0.3645636456364564,
    0.36457364573645734,
    0.36458364583645836,
    0.3645936459364594,
    0.36460364603646034,
    0.36461364613646136,
    0.3646236462364624,
    0.36463364633646334,
    0.36464364643646435,
    0.36465364653646537,
    0.36466364663646633,
    0.36467364673646735,
    0.36468364683646837,
    0.3646936469364694,
    0.36470364703647035,
    0.36471364713647136,
    0.3647236472364724,
    0.36473364733647334,
    0.36474364743647436,
    0.3647536475364754,
    0.36476364763647634,
    0.36477364773647736,
    0.3647836478364784,
    0.36479364793647934,
    0.36480364803648035,
    0.36481364813648137,
    0.36482364823648233,
    0.36483364833648335,
    0.36484364843648437,
    0.36485364853648533,
    0.36486364863648635,
    0.36487364873648737,
    0.3648836488364884,
    0.36489364893648935,
    0.36490364903649036,
    0.3649136491364914,
    0.36492364923649234,
    0.36493364933649336,
    0.3649436494364944,
    0.36495364953649534,
    0.36496364963649636,
    0.3649736497364974,
    0.36498364983649834,
    0.36499364993649935,
    0.36500365003650037,
    0.36501365013650133,
    0.36502365023650235,
    0.36503365033650337,
    0.3650436504365044,
    0.36505365053650535,
    0.36506365063650636,
    0.3650736507365074,
    0.36508365083650834,
    0.36509365093650936,
    0.3651036510365104,
    0.36511365113651134,
    0.36512365123651236,
    0.3651336513365134,
    0.36514365143651434,
    0.36515365153651536,
    0.3651636516365164,
    0.36517365173651734,
    0.36518365183651835,
    0.36519365193651937,
    0.36520365203652033,
    0.36521365213652135,
    0.36522365223652237,
    0.3652336523365234,
    0.36524365243652435,
    0.36525365253652536,
    0.3652636526365264,
    0.36527365273652734,
    0.36528365283652836,
    0.3652936529365294,
    0.36530365303653034,
    0.36531365313653136,
    0.3653236532365324,
    0.36533365333653334,
    0.36534365343653435,
    0.36535365353653537,
    0.36536365363653633,
    0.36537365373653735,
    0.36538365383653837,
    0.3653936539365394,
    0.36540365403654035,
    0.36541365413654137,
    0.3654236542365424,
    0.36543365433654335,
    0.36544365443654436,
    0.3654536545365454,
    0.36546365463654634,
    0.36547365473654736,
    0.3654836548365484,
    0.36549365493654934,
    0.36550365503655036,
    0.3655136551365514,
    0.36552365523655234,
    0.36553365533655335,
    0.36554365543655437,
    0.36555365553655533,
    0.36556365563655635,
    0.36557365573655737,
    0.3655836558365584,
    0.36559365593655935,
    0.36560365603656036,
    0.3656136561365614,
    0.36562365623656234,
    0.36563365633656336,
    0.3656436564365644,
    0.36565365653656534,
    0.36566365663656636,
    0.3656736567365674,
    0.36568365683656834,
    0.36569365693656936,
    0.3657036570365704,
    0.36571365713657134,
    0.36572365723657235,
    0.36573365733657337,
    0.36574365743657433,
    0.36575365753657535,
    0.36576365763657637,
    0.3657736577365774,
    0.36578365783657835,
    0.36579365793657936,
    0.3658036580365804,
    0.36581365813658134,
    0.36582365823658236,
    0.3658336583365834,
    0.36584365843658434,
    0.36585365853658536,
    0.3658636586365864,
    0.36587365873658734,
    0.36588365883658835,
    0.36589365893658937,
    0.36590365903659033,
    0.36591365913659135,
    0.36592365923659237,
    0.3659336593365934,
    0.36594365943659435,
    0.36595365953659537,
    0.3659636596365964,
    0.36597365973659735,
    0.36598365983659836,
    0.3659936599365994,
    0.36600366003660034,
    0.36601366013660136,
    0.3660236602366024,
    0.36603366033660334,
    0.36604366043660436,
    0.3660536605366054,
    0.36606366063660634,
    0.36607366073660735,
    0.36608366083660837,
    0.36609366093660933,
    0.36610366103661035,
    0.36611366113661137,
    0.3661236612366124,
    0.36613366133661335,
    0.36614366143661436,
    0.3661536615366154,
    0.36616366163661634,
    0.36617366173661736,
    0.3661836618366184,
    0.36619366193661934,
    0.36620366203662036,
    0.3662136621366214,
    0.36622366223662234,
    0.36623366233662336,
    0.3662436624366244,
    0.36625366253662534,
    0.36626366263662635,
    0.36627366273662737,
    0.3662836628366284,
    0.36629366293662935,
    0.36630366303663037,
    0.3663136631366314,
    0.36632366323663235,
    0.36633366333663336,
    0.3663436634366344,
    0.36635366353663534,
    0.36636366363663636,
    0.3663736637366374,
    0.36638366383663834,
    0.36639366393663936,
    0.3664036640366404,
    0.36641366413664134,
    0.36642366423664235,
    0.36643366433664337,
    0.36644366443664433,
    0.36645366453664535,
    0.36646366463664637,
    0.3664736647366474,
    0.36648366483664835,
    0.36649366493664937,
    0.3665036650366504,
    0.36651366513665135,
    0.36652366523665236,
    0.3665336653366534,
    0.36654366543665434,
    0.36655366553665536,
    0.3665636656366564,
    0.36657366573665734,
    0.36658366583665836,
    0.3665936659366594,
    0.36660366603666034,
    0.36661366613666135,
    0.36662366623666237,
    0.36663366633666333,
    0.36664366643666435,
    0.36665366653666537,
    0.3666636666366664,
    0.36667366673666735,
    0.36668366683666836,
    0.3666936669366694,
    0.36670366703667034,
    0.36671366713667136,
    0.3667236672366724,
    0.36673366733667334,
    0.36674366743667436,
    0.3667536675366754,
    0.36676366763667634,
    0.36677366773667736,
    0.3667836678366784,
    0.36679366793667934,
    0.36680366803668035,
    0.36681366813668137,
    0.3668236682366824,
    0.36683366833668335,
    0.36684366843668437,
    0.3668536685366854,
    0.36686366863668635,
    0.36687366873668736,
    0.3668836688366884,
    0.36689366893668934,
    0.36690366903669036,
    0.3669136691366914,
    0.36692366923669234,
    0.36693366933669336,
    0.3669436694366944,
    0.36695366953669534,
    0.36696366963669635,
    0.36697366973669737,
    0.36698366983669833,
    0.36699366993669935,
    0.36700367003670037,
    0.3670136701367014,
    0.36702367023670235,
    0.36703367033670337,
    0.3670436704367044,
    0.36705367053670535,
    0.36706367063670636,
    0.3670736707367074,
    0.36708367083670834,
    0.36709367093670936,
    0.3671036710367104,
    0.36711367113671134,
    0.36712367123671236,
    0.3671336713367134,
    0.36714367143671434,
    0.36715367153671535,
    0.36716367163671637,
    0.3671736717367174,
    0.36718367183671835,
    0.36719367193671937,
    0.3672036720367204,
    0.36721367213672135,
    0.36722367223672236,
    0.3672336723367234,
    0.36724367243672434,
    0.36725367253672536,
    0.3672636726367264,
    0.36727367273672734,
    0.36728367283672836,
    0.3672936729367294,
    0.36730367303673034,
    0.36731367313673136,
    0.3673236732367324,
    0.36733367333673334,
    0.36734367343673435,
    0.36735367353673537,
    0.3673636736367364,
    0.36737367373673735,
    0.36738367383673837,
    0.3673936739367394,
    0.36740367403674035,
    0.36741367413674136,
    0.3674236742367424,
    0.36743367433674334,
    0.36744367443674436,
    0.3674536745367454,
    0.36746367463674634,
    0.36747367473674736,
    0.3674836748367484,
    0.36749367493674934,
    0.36750367503675035,
    0.36751367513675137,
    0.36752367523675233,
    0.36753367533675335,
    0.36754367543675437,
    0.3675536755367554,
    0.36756367563675635,
    0.36757367573675737,
    0.3675836758367584,
    0.36759367593675935,
    0.36760367603676036,
    0.3676136761367614,
    0.36762367623676234,
    0.36763367633676336,
    0.3676436764367644,
    0.36765367653676534,
    0.36766367663676636,
    0.3676736767367674,
    0.36768367683676834,
    0.36769367693676935,
    0.36770367703677037,
    0.3677136771367714,
    0.36772367723677235,
    0.36773367733677337,
    0.3677436774367744,
    0.36775367753677535,
    0.36776367763677636,
    0.3677736777367774,
    0.36778367783677834,
    0.36779367793677936,
    0.3678036780367804,
    0.36781367813678134,
    0.36782367823678236,
    0.3678336783367834,
    0.36784367843678434,
    0.36785367853678536,
    0.3678636786367864,
    0.36787367873678734,
    0.36788367883678835,
    0.36789367893678937,
    0.3679036790367904,
    0.36791367913679135,
    0.36792367923679237,
    0.3679336793367934,
    0.36794367943679435,
    0.36795367953679536,
    0.3679636796367964,
    0.36797367973679734,
    0.36798367983679836,
    0.3679936799367994,
    0.36800368003680034,
    0.36801368013680136,
    0.3680236802368024,
    0.36803368033680334,
    0.36804368043680435,
    0.36805368053680537,
    0.36806368063680633,
    0.36807368073680735,
    0.36808368083680837,
    0.3680936809368094,
    0.36810368103681035,
    0.36811368113681137,
    0.3681236812368124,
    0.36813368133681335,
    0.36814368143681436,
    0.3681536815368154,
    0.36816368163681634,
    0.36817368173681736,
    0.3681836818368184,
    0.36819368193681934,
    0.36820368203682036,
    0.3682136821368214,
    0.36822368223682234,
    0.36823368233682335,
    0.36824368243682437,
    0.3682536825368254,
    0.36826368263682635,
    0.36827368273682737,
    0.3682836828368284,
    0.36829368293682935,
    0.36830368303683036,
    0.3683136831368314,
    0.36832368323683234,
    0.36833368333683336,
    0.3683436834368344,
    0.36835368353683534,
    0.36836368363683636,
    0.3683736837368374,
    0.36838368383683834,
    0.36839368393683936,
    0.3684036840368404,
    0.36841368413684134,
    0.36842368423684235,
    0.36843368433684337,
    0.3684436844368444,
    0.36845368453684535,
    0.36846368463684637,
    0.3684736847368474,
    0.36848368483684835,
    0.36849368493684936,
    0.3685036850368504,
    0.36851368513685134,
    0.36852368523685236,
    0.3685336853368534,
    0.36854368543685434,
    0.36855368553685536,
    0.3685636856368564,
    0.36857368573685734,
    0.36858368583685835,
    0.36859368593685937,
    0.3686036860368604,
    0.36861368613686135,
    0.36862368623686237,
    0.3686336863368634,
    0.36864368643686435,
    0.36865368653686537,
    0.3686636866368664,
    0.36867368673686735,
    0.36868368683686836,
    0.3686936869368694,
    0.36870368703687034,
    0.36871368713687136,
    0.3687236872368724,
    0.36873368733687334,
    0.36874368743687436,
    0.3687536875368754,
    0.36876368763687634,
    0.36877368773687735,
    0.36878368783687837,
    0.3687936879368794,
    0.36880368803688035,
    0.36881368813688137,
    0.3688236882368824,
    0.36883368833688335,
    0.36884368843688436,
    0.3688536885368854,
    0.36886368863688634,
    0.36887368873688736,
    0.3688836888368884,
    0.36889368893688934,
    0.36890368903689036,
    0.3689136891368914,
    0.36892368923689234,
    0.36893368933689336,
    0.3689436894368944,
    0.36895368953689534,
    0.36896368963689635,
    0.36897368973689737,
    0.3689836898368984,
    0.36899368993689935,
    0.36900369003690037,
    0.3690136901369014,
    0.36902369023690235,
    0.36903369033690336,
    0.3690436904369044,
    0.36905369053690534,
    0.36906369063690636,
    0.3690736907369074,
    0.36908369083690834,
    0.36909369093690936,
    0.3691036910369104,
    0.36911369113691134,
    0.36912369123691235,
    0.36913369133691337,
    0.3691436914369144,
    0.36915369153691535,
    0.36916369163691637,
    0.3691736917369174,
    0.36918369183691835,
    0.36919369193691937,
    0.3692036920369204,
    0.36921369213692135,
    0.36922369223692236,
    0.3692336923369234,
    0.36924369243692434,
    0.36925369253692536,
    0.3692636926369264,
    0.36927369273692734,
    0.36928369283692836,
    0.3692936929369294,
    0.36930369303693034,
    0.36931369313693135,
    0.36932369323693237,
    0.3693336933369334,
    0.36934369343693435,
    0.36935369353693537,
    0.3693636936369364,
    0.36937369373693735,
    0.36938369383693836,
    0.3693936939369394,
    0.36940369403694034,
    0.36941369413694136,
    0.3694236942369424,
    0.36943369433694334,
    0.36944369443694436,
    0.3694536945369454,
    0.36946369463694634,
    0.36947369473694736,
    0.3694836948369484,
    0.3694936949369494,
    0.36950369503695035,
    0.36951369513695137,
    0.3695236952369524,
    0.36953369533695335,
    0.36954369543695437,
    0.3695536955369554,
    0.36956369563695635,
    0.36957369573695736,
    0.3695836958369584,
    0.36959369593695934,
    0.36960369603696036,
    0.3696136961369614,
    0.36962369623696234,
    0.36963369633696336,
    0.3696436964369644,
    0.36965369653696534,
    0.36966369663696635,
    0.36967369673696737,
    0.3696836968369684,
    0.36969369693696935,
    0.36970369703697037,
    0.3697136971369714,
    0.36972369723697235,
    0.36973369733697337,
    0.3697436974369744,
    0.36975369753697535,
    0.36976369763697636,
    0.3697736977369774,
    0.36978369783697834,
    0.36979369793697936,
    0.3698036980369804,
    0.36981369813698134,
    0.36982369823698236,
    0.3698336983369834,
    0.36984369843698434,
    0.36985369853698535,
    0.36986369863698637,
    0.3698736987369874,
    0.36988369883698835,
    0.36989369893698937,
    0.3699036990369904,
    0.36991369913699135,
    0.36992369923699236,
    0.3699336993369934,
    0.36994369943699434,
    0.36995369953699536,
    0.3699636996369964,
    0.36997369973699734,
    0.36998369983699836,
    0.3699936999369994,
    0.37000370003700034,
    0.37001370013700136,
    0.3700237002370024,
    0.3700337003370034,
    0.37004370043700435,
    0.37005370053700537,
    0.3700637006370064,
    0.37007370073700735,
    0.37008370083700837,
    0.3700937009370094,
    0.37010370103701035,
    0.37011370113701136,
    0.3701237012370124,
    0.37013370133701334,
    0.37014370143701436,
    0.3701537015370154,
    0.37016370163701634,
    0.37017370173701736,
    0.3701837018370184,
    0.37019370193701934,
    0.37020370203702035,
    0.37021370213702137,
    0.3702237022370224,
    0.37023370233702335,
    0.37024370243702437,
    0.3702537025370254,
    0.37026370263702635,
    0.37027370273702737,
    0.3702837028370284,
    0.37029370293702935,
    0.37030370303703036,
    0.3703137031370314,
    0.37032370323703234,
    0.37033370333703336,
    0.3703437034370344,
    0.37035370353703534,
    0.37036370363703636,
    0.3703737037370374,
    0.3703837038370384,
    0.37039370393703935,
    0.37040370403704037,
    0.3704137041370414,
    0.37042370423704235,
    0.37043370433704337,
    0.3704437044370444,
    0.37045370453704535,
    0.37046370463704636,
    0.3704737047370474,
    0.37048370483704834,
    0.37049370493704936,
    0.3705037050370504,
    0.37051370513705134,
    0.37052370523705236,
    0.3705337053370534,
    0.37054370543705434,
    0.37055370553705536,
    0.3705637056370564,
    0.3705737057370574,
    0.37058370583705835,
    0.37059370593705937,
    0.3706037060370604,
    0.37061370613706135,
    0.37062370623706237,
    0.3706337063370634,
    0.37064370643706435,
    0.37065370653706536,
    0.3706637066370664,
    0.37067370673706734,
    0.37068370683706836,
    0.3706937069370694,
    0.37070370703707034,
    0.37071370713707136,
    0.3707237072370724,
    0.37073370733707334,
    0.37074370743707435,
    0.37075370753707537,
    0.3707637076370764,
    0.37077370773707735,
    0.37078370783707837,
    0.3707937079370794,
    0.37080370803708035,
    0.37081370813708137,
    0.3708237082370824,
    0.37083370833708335,
    0.37084370843708436,
    0.3708537085370854,
    0.37086370863708634,
    0.37087370873708736,
    0.3708837088370884,
    0.37089370893708934,
    0.37090370903709036,
    0.3709137091370914,
    0.3709237092370924,
    0.37093370933709335,
    0.37094370943709437,
    0.3709537095370954,
    0.37096370963709635,
    0.37097370973709737,
    0.3709837098370984,
    0.37099370993709935,
    0.37100371003710036,
    0.3710137101371014,
    0.37102371023710234,
    0.37103371033710336,
    0.3710437104371044,
    0.37105371053710534,
    0.37106371063710636,
    0.3710737107371074,
    0.37108371083710834,
    0.37109371093710936,
    0.3711037110371104,
    0.3711137111371114,
    0.37112371123711235,
    0.37113371133711337,
    0.3711437114371144,
    0.37115371153711535,
    0.37116371163711637,
    0.3711737117371174,
    0.37118371183711835,
    0.37119371193711936,
    0.3712037120371204,
    0.37121371213712134,
    0.37122371223712236,
    0.3712337123371234,
    0.37124371243712434,
    0.37125371253712536,
    0.3712637126371264,
    0.3712737127371274,
    0.37128371283712835,
    0.37129371293712937,
    0.3713037130371304,
    0.37131371313713135,
    0.37132371323713237,
    0.3713337133371334,
    0.37134371343713435,
    0.37135371353713537,
    0.3713637136371364,
    0.37137371373713735,
    0.37138371383713836,
    0.3713937139371394,
    0.37140371403714034,
    0.37141371413714136,
    0.3714237142371424,
    0.37143371433714334,
    0.37144371443714436,
    0.3714537145371454,
    0.3714637146371464,
    0.37147371473714735,
    0.37148371483714837,
    0.3714937149371494,
    0.37150371503715035,
    0.37151371513715137,
    0.3715237152371524,
    0.37153371533715335,
    0.37154371543715436,
    0.3715537155371554,
    0.37156371563715634,
    0.37157371573715736,
    0.3715837158371584,
    0.37159371593715934,
    0.37160371603716036,
    0.3716137161371614,
    0.37162371623716234,
    0.37163371633716336,
    0.3716437164371644,
    0.3716537165371654,
    0.37166371663716635,
    0.37167371673716737,
    0.3716837168371684,
    0.37169371693716935,
    0.37170371703717037,
    0.3717137171371714,
    0.37172371723717235,
    0.37173371733717336,
    0.3717437174371744,
    0.37175371753717534,
    0.37176371763717636,
    0.3717737177371774,
    0.37178371783717834,
    0.37179371793717936,
    0.3718037180371804,
    0.3718137181371814,
    0.37182371823718235,
    0.37183371833718337,
    0.3718437184371844,
    0.37185371853718535,
    0.37186371863718637,
    0.3718737187371874,
    0.37188371883718835,
    0.37189371893718937,
    0.3719037190371904,
    0.37191371913719135,
    0.37192371923719236,
    0.3719337193371934,
    0.37194371943719434,
    0.37195371953719536,
    0.3719637196371964,
    0.37197371973719734,
    0.37198371983719836,
    0.3719937199371994,
    0.3720037200372004,
    0.37201372013720135,
    0.37202372023720237,
    0.3720337203372034,
    0.37204372043720435,
    0.37205372053720537,
    0.3720637206372064,
    0.37207372073720735,
    0.37208372083720836,
    0.3720937209372094,
    0.37210372103721034,
    0.37211372113721136,
    0.3721237212372124,
    0.37213372133721334,
    0.37214372143721436,
    0.3721537215372154,
    0.37216372163721634,
    0.37217372173721736,
    0.3721837218372184,
    0.3721937219372194,
    0.37220372203722035,
    0.37221372213722137,
    0.3722237222372224,
    0.37223372233722335,
    0.37224372243722437,
    0.3722537225372254,
    0.37226372263722635,
    0.37227372273722736,
    0.3722837228372284,
    0.37229372293722934,
    0.37230372303723036,
    0.3723137231372314,
    0.37232372323723234,
    0.37233372333723336,
    0.3723437234372344,
    0.3723537235372354,
    0.37236372363723635,
    0.37237372373723737,
    0.3723837238372384,
    0.37239372393723935,
    0.37240372403724037,
    0.3724137241372414,
    0.37242372423724235,
    0.37243372433724337,
    0.3724437244372444,
    0.37245372453724535,
    0.37246372463724636,
    0.3724737247372474,
    0.37248372483724834,
    0.37249372493724936,
    0.3725037250372504,
    0.37251372513725134,
    0.37252372523725236,
    0.3725337253372534,
    0.3725437254372544,
    0.37255372553725535,
    0.37256372563725637,
    0.3725737257372574,
    0.37258372583725835,
    0.37259372593725937,
    0.3726037260372604,
    0.37261372613726135,
    0.37262372623726236,
    0.3726337263372634,
    0.37264372643726434,
    0.37265372653726536,
    0.3726637266372664,
    0.37267372673726734,
    0.37268372683726836,
    0.3726937269372694,
    0.3727037270372704,
    0.37271372713727136,
    0.3727237272372724,
    0.3727337273372734,
    0.37274372743727435,
    0.37275372753727537,
    0.3727637276372764,
    0.37277372773727735,
    0.37278372783727837,
    0.3727937279372794,
    0.37280372803728035,
    0.37281372813728136,
    0.3728237282372824,
    0.37283372833728334,
    0.37284372843728436,
    0.3728537285372854,
    0.37286372863728634,
    0.37287372873728736,
    0.3728837288372884,
    0.3728937289372894,
    0.37290372903729035,
    0.37291372913729137,
    0.3729237292372924,
    0.37293372933729335,
    0.37294372943729437,
    0.3729537295372954,
    0.37296372963729635,
    0.37297372973729737,
    0.3729837298372984,
    0.37299372993729935,
    0.37300373003730036,
    0.3730137301373014,
    0.37302373023730234,
    0.37303373033730336,
    0.3730437304373044,
    0.37305373053730534,
    0.37306373063730636,
    0.3730737307373074,
    0.3730837308373084,
    0.37309373093730935,
    0.37310373103731037,
    0.3731137311373114,
    0.37312373123731235,
    0.37313373133731337,
    0.3731437314373144,
    0.37315373153731535,
    0.37316373163731636,
    0.3731737317373174,
    0.37318373183731834,
    0.37319373193731936,
    0.3732037320373204,
    0.37321373213732134,
    0.37322373223732236,
    0.3732337323373234,
    0.3732437324373244,
    0.37325373253732536,
    0.3732637326373264,
    0.3732737327373274,
    0.37328373283732835,
    0.37329373293732937,
    0.3733037330373304,
    0.37331373313733135,
    0.37332373323733237,
    0.3733337333373334,
    0.37334373343733435,
    0.37335373353733536,
    0.3733637336373364,
    0.37337373373733734,
    0.37338373383733836,
    0.3733937339373394,
    0.37340373403734034,
    0.37341373413734136,
    0.3734237342373424,
    0.3734337343373434,
    0.37344373443734435,
    0.37345373453734537,
    0.3734637346373464,
    0.37347373473734735,
    0.37348373483734837,
    0.3734937349373494,
    0.37350373503735035,
    0.37351373513735137,
    0.3735237352373524,
    0.37353373533735335,
    0.37354373543735436,
    0.3735537355373554,
    0.37356373563735634,
    0.37357373573735736,
    0.3735837358373584,
    0.3735937359373594,
    0.37360373603736036,
    0.3736137361373614,
    0.3736237362373624,
    0.37363373633736335,
    0.37364373643736437,
    0.3736537365373654,
    0.37366373663736635,
    0.37367373673736737,
    0.3736837368373684,
    0.37369373693736935,
    0.37370373703737036,
    0.3737137371373714,
    0.37372373723737234,
    0.37373373733737336,
    0.3737437374373744,
    0.37375373753737534,
    0.37376373763737636,
    0.3737737377373774,
    0.3737837378373784,
    0.37379373793737936,
    0.3738037380373804,
    0.3738137381373814,
    0.37382373823738235,
    0.37383373833738337,
    0.3738437384373844,
    0.37385373853738535,
    0.37386373863738637,
    0.3738737387373874,
    0.37388373883738835,
    0.37389373893738936,
    0.3739037390373904,
    0.37391373913739134,
    0.37392373923739236,
    0.3739337393373934,
    0.37394373943739434,
    0.37395373953739536,
    0.3739637396373964,
    0.3739737397373974,
    0.37398373983739835,
    0.37399373993739937,
    0.3740037400374004,
    0.37401374013740135,
    0.37402374023740237,
    0.3740337403374034,
    0.37404374043740435,
    0.37405374053740537,
    0.3740637406374064,
    0.37407374073740735,
    0.37408374083740836,
    0.3740937409374094,
    0.37410374103741034,
    0.37411374113741136,
    0.3741237412374124,
    0.3741337413374134,
    0.37414374143741436,
    0.3741537415374154,
    0.3741637416374164,
    0.37417374173741735,
    0.37418374183741837,
    0.3741937419374194,
    0.37420374203742035,
    0.37421374213742137,
    0.3742237422374224,
    0.37423374233742335,
    0.37424374243742436,
    0.3742537425374254,
    0.37426374263742634,
    0.37427374273742736,
    0.3742837428374284,
    0.37429374293742934,
    0.37430374303743036,
    0.3743137431374314,
    0.3743237432374324,
    0.37433374333743336,
    0.3743437434374344,
    0.3743537435374354,
    0.37436374363743635,
    0.37437374373743737,
    0.3743837438374384,
    0.37439374393743935,
    0.37440374403744037,
    0.3744137441374414,
    0.37442374423744235,
    0.37443374433744336,
    0.3744437444374444,
    0.37445374453744534,
    0.37446374463744636,
    0.3744737447374474,
    0.3744837448374484,
    0.37449374493744936,
    0.3745037450374504,
    0.3745137451374514,
    0.37452374523745235,
    0.37453374533745337,
    0.3745437454374544,
    0.37455374553745535,
    0.37456374563745637,
    0.3745737457374574,
    0.37458374583745835,
    0.37459374593745937,
    0.3746037460374604,
    0.37461374613746135,
    0.37462374623746236,
    0.3746337463374634,
    0.37464374643746434,
    0.37465374653746536,
    0.3746637466374664,
    0.3746737467374674,
    0.37468374683746836,
    0.3746937469374694,
    0.3747037470374704,
    0.37471374713747135,
    0.37472374723747237,
    0.3747337473374734,
    0.37474374743747435,
    0.37475374753747537,
    0.3747637476374764,
    0.37477374773747735,
    0.37478374783747836,
    0.3747937479374794,
    0.37480374803748034,
    0.37481374813748136,
    0.3748237482374824,
    0.37483374833748334,
    0.37484374843748436,
    0.3748537485374854,
    0.3748637486374864,
    0.37487374873748736,
    0.3748837488374884,
    0.3748937489374894,
    0.37490374903749035,
    0.37491374913749137,
    0.3749237492374924,
    0.37493374933749335,
    0.37494374943749437,
    0.3749537495374954,
    0.37496374963749635,
    0.37497374973749736,
    0.3749837498374984,
    0.37499374993749934,
    0.37500375003750036,
    0.3750137501375014,
    0.3750237502375024,
    0.37503375033750336,
    0.3750437504375044,
    0.3750537505375054,
    0.37506375063750635,
    0.37507375073750737,
    0.3750837508375084,
    0.37509375093750935,
    0.37510375103751037,
    0.3751137511375114,
    0.37512375123751235,
    0.37513375133751337,
    0.3751437514375144,
    0.37515375153751535,
    0.37516375163751636,
    0.3751737517375174,
    0.37518375183751834,
    0.37519375193751936,
    0.3752037520375204,
    0.3752137521375214,
    0.37522375223752236,
    0.3752337523375234,
    0.3752437524375244,
    0.37525375253752535,
    0.37526375263752637,
    0.3752737527375274,
    0.37528375283752835,
    0.37529375293752937,
    0.3753037530375304,
    0.37531375313753135,
    0.37532375323753236,
    0.3753337533375334,
    0.37534375343753434,
    0.37535375353753536,
    0.3753637536375364,
    0.37537375373753734,
    0.37538375383753836,
    0.3753937539375394,
    0.3754037540375404,
    0.37541375413754136,
    0.3754237542375424,
    0.3754337543375434,
    0.37544375443754435,
    0.37545375453754537,
    0.3754637546375464,
    0.37547375473754735,
    0.37548375483754837,
    0.3754937549375494,
    0.37550375503755035,
    0.37551375513755136,
    0.3755237552375524,
    0.37553375533755334,
    0.37554375543755436,
    0.3755537555375554,
    0.3755637556375564,
    0.37557375573755736,
    0.3755837558375584,
    0.3755937559375594,
    0.37560375603756035,
    0.37561375613756137,
    0.3756237562375624,
    0.37563375633756335,
    0.37564375643756437,
    0.3756537565375654,
    0.37566375663756635,
    0.37567375673756737,
    0.3756837568375684,
    0.37569375693756935,
    0.37570375703757036,
    0.3757137571375714,
    0.37572375723757234,
    0.37573375733757336,
    0.3757437574375744,
    0.3757537575375754,
    0.37576375763757636,
    0.3757737577375774,
    0.3757837578375784,
    0.37579375793757935,
    0.37580375803758037,
    0.3758137581375814,
    0.37582375823758235,
    0.37583375833758337,
    0.3758437584375844,
    0.37585375853758535,
    0.37586375863758636,
    0.3758737587375874,
    0.37588375883758834,
    0.37589375893758936,
    0.3759037590375904,
    0.3759137591375914,
    0.37592375923759236,
    0.3759337593375934,
    0.3759437594375944,
    0.37595375953759536,
    0.3759637596375964,
    0.3759737597375974,
    0.37598375983759835,
    0.37599375993759937,
    0.3760037600376004,
    0.37601376013760135,
    0.37602376023760237,
    0.3760337603376034,
    0.37604376043760435,
    0.37605376053760536,
    0.3760637606376064,
    0.37607376073760734,
    0.37608376083760836,
    0.3760937609376094,
    0.3761037610376104,
    0.37611376113761136,
    0.3761237612376124,
    0.3761337613376134,
    0.37614376143761435,
    0.37615376153761537,
    0.3761637616376164,
    0.37617376173761735,
    0.37618376183761837,
    0.3761937619376194,
    0.37620376203762035,
    0.37621376213762137,
    0.3762237622376224,
    0.37623376233762335,
    0.37624376243762436,
    0.3762537625376254,
    0.37626376263762634,
    0.37627376273762736,
    0.3762837628376284,
    0.3762937629376294,
    0.37630376303763036,
    0.3763137631376314,
    0.3763237632376324,
    0.37633376333763335,
    0.37634376343763437,
    0.3763537635376354,
    0.37636376363763635,
    0.37637376373763737,
    0.3763837638376384,
    0.37639376393763935,
    0.37640376403764036,
    0.3764137641376414,
    0.37642376423764234,
    0.37643376433764336,
    0.3764437644376444,
    0.3764537645376454,
    0.37646376463764636,
    0.3764737647376474,
    0.3764837648376484,
    0.37649376493764936,
    0.3765037650376504,
    0.3765137651376514,
    0.37652376523765235,
    0.37653376533765337,
    0.3765437654376544,
    0.37655376553765535,
    0.37656376563765637,
    0.3765737657376574,
    0.37658376583765835,
    0.37659376593765936,
    0.3766037660376604,
    0.37661376613766134,
    0.37662376623766236,
    0.3766337663376634,
    0.3766437664376644,
    0.37665376653766536,
    0.3766637666376664,
    0.3766737667376674,
    0.37668376683766835,
    0.37669376693766937,
    0.3767037670376704,
    0.37671376713767135,
    0.37672376723767237,
    0.3767337673376734,
    0.37674376743767435,
    0.37675376753767537,
    0.3767637676376764,
    0.37677376773767735,
    0.37678376783767836,
    0.3767937679376794,
    0.3768037680376804,
    0.37681376813768136,
    0.3768237682376824,
    0.3768337683376834,
    0.37684376843768436,
    0.3768537685376854,
    0.3768637686376864,
    0.37687376873768735,
    0.37688376883768837,
    0.3768937689376894,
    0.37690376903769035,
    0.37691376913769137,
    0.3769237692376924,
    0.37693376933769335,
    0.37694376943769436,
    0.3769537695376954,
    0.37696376963769634,
    0.37697376973769736,
    0.3769837698376984,
    0.3769937699376994,
    0.37700377003770036,
    0.3770137701377014,
    0.3770237702377024,
    0.37703377033770336,
    0.3770437704377044,
    0.3770537705377054,
    0.37706377063770635,
    0.37707377073770737,
    0.3770837708377084,
    0.37709377093770935,
    0.37710377103771037,
    0.3771137711377114,
    0.37712377123771235,
    0.37713377133771336,
    0.3771437714377144,
    0.37715377153771534,
    0.37716377163771636,
    0.3771737717377174,
    0.3771837718377184,
    0.37719377193771936,
    0.3772037720377204,
    0.3772137721377214,
    0.37722377223772235,
    0.37723377233772337,
    0.3772437724377244,
    0.37725377253772535,
    0.37726377263772637,
    0.3772737727377274,
    0.37728377283772835,
    0.37729377293772937,
    0.3773037730377304,
    0.37731377313773135,
    0.37732377323773236,
    0.3773337733377334,
    0.3773437734377344,
    0.37735377353773536,
    0.3773637736377364,
    0.3773737737377374,
    0.37738377383773836,
    0.3773937739377394,
    0.3774037740377404,
    0.37741377413774135,
    0.37742377423774237,
    0.3774337743377434,
    0.37744377443774435,
    0.37745377453774537,
    0.3774637746377464,
    0.37747377473774735,
    0.37748377483774836,
    0.3774937749377494,
    0.37750377503775034,
    0.37751377513775136,
    0.3775237752377524,
    0.3775337753377534,
    0.37754377543775436,
    0.3775537755377554,
    0.3775637756377564,
    0.37757377573775736,
    0.3775837758377584,
    0.3775937759377594,
    0.37760377603776035,
    0.37761377613776137,
    0.3776237762377624,
    0.37763377633776335,
    0.37764377643776437,
    0.3776537765377654,
    0.37766377663776635,
    0.37767377673776736,
    0.3776837768377684,
    0.3776937769377694,
    0.37770377703777036,
    0.3777137771377714,
    0.3777237772377724,
    0.37773377733777336,
    0.3777437774377744,
    0.3777537775377754,
    0.37776377763777635,
    0.37777377773777737,
    0.3777837778377784,
    0.37779377793777935,
    0.37780377803778037,
    0.3778137781377814,
    0.37782377823778235,
    0.37783377833778337,
    0.3778437784377844,
    0.37785377853778535,
    0.37786377863778636,
    0.3778737787377874,
    0.3778837788377884,
    0.37789377893778936,
    0.3779037790377904,
    0.3779137791377914,
    0.37792377923779236,
    0.3779337793377934,
    0.3779437794377944,
    0.37795377953779535,
    0.37796377963779637,
    0.3779737797377974,
    0.37798377983779835,
    0.37799377993779937,
    0.3780037800378004,
    0.37801378013780135,
    0.37802378023780236,
    0.3780337803378034,
    0.37804378043780434,
    0.37805378053780536,
    0.3780637806378064,
    0.3780737807378074,
    0.37808378083780836,
    0.3780937809378094,
    0.3781037810378104,
    0.37811378113781136,
    0.3781237812378124,
    0.3781337813378134,
    0.37814378143781435,
    0.37815378153781537,
    0.3781637816378164,
    0.37817378173781735,
    0.37818378183781837,
    0.3781937819378194,
    0.37820378203782035,
    0.37821378213782136,
    0.3782237822378224,
    0.3782337823378234,
    0.37824378243782436,
    0.3782537825378254,
    0.3782637826378264,
    0.37827378273782736,
    0.3782837828378284,
    0.3782937829378294,
    0.37830378303783035,
    0.37831378313783137,
    0.3783237832378324,
    0.37833378333783335,
    0.37834378343783437,
    0.3783537835378354,
    0.37836378363783635,
    0.37837378373783737,
    0.3783837838378384,
    0.37839378393783935,
    0.37840378403784036,
    0.3784137841378414,
    0.3784237842378424,
    0.37843378433784336,
    0.3784437844378444,
    0.3784537845378454,
    0.37846378463784636,
    0.3784737847378474,
    0.3784837848378484,
    0.37849378493784935,
    0.37850378503785037,
    0.3785137851378514,
    0.37852378523785235,
    0.37853378533785337,
    0.3785437854378544,
    0.37855378553785535,
    0.37856378563785636,
    0.3785737857378574,
    0.3785837858378584,
    0.37859378593785936,
    0.3786037860378604,
    0.3786137861378614,
    0.37862378623786236,
    0.3786337863378634,
    0.3786437864378644,
    0.37865378653786536,
    0.3786637866378664,
    0.3786737867378674,
    0.37868378683786835,
    0.37869378693786937,
    0.3787037870378704,
    0.37871378713787135,
    0.37872378723787237,
    0.3787337873378734,
    0.37874378743787435,
    0.37875378753787536,
    0.3787637876378764,
    0.3787737877378774,
    0.37878378783787836,
    0.3787937879378794,
    0.3788037880378804,
    0.37881378813788136,
    0.3788237882378824,
    0.3788337883378834,
    0.37884378843788435,
    0.37885378853788537,
    0.3788637886378864,
    0.37887378873788735,
    0.37888378883788837,
    0.3788937889378894,
    0.37890378903789035,
    0.37891378913789137,
    0.3789237892378924,
    0.37893378933789335,
    0.37894378943789436,
    0.3789537895378954,
    0.3789637896378964,
    0.37897378973789736,
    0.3789837898378984,
    0.3789937899378994,
    0.37900379003790036,
    0.3790137901379014,
    0.3790237902379024,
    0.37903379033790335,
    0.37904379043790437,
    0.3790537905379054,
    0.37906379063790635,
    0.37907379073790737,
    0.3790837908379084,
    0.37909379093790935,
    0.37910379103791036,
    0.3791137911379114,
    0.3791237912379124,
    0.37913379133791336,
    0.3791437914379144,
    0.3791537915379154,
    0.37916379163791636,
    0.3791737917379174,
    0.3791837918379184,
    0.37919379193791936,
    0.3792037920379204,
    0.3792137921379214,
    0.37922379223792235,
    0.37923379233792337,
    0.3792437924379244,
    0.37925379253792535,
    0.37926379263792637,
    0.3792737927379274,
    0.37928379283792835,
    0.37929379293792936,
    0.3793037930379304,
    0.3793137931379314,
    0.37932379323793236,
    0.3793337933379334,
    0.3793437934379344,
    0.37935379353793536,
    0.3793637936379364,
    0.3793737937379374,
    0.37938379383793835,
    0.37939379393793937,
    0.3794037940379404,
    0.37941379413794135,
    0.37942379423794237,
    0.3794337943379434,
    0.37944379443794435,
    0.37945379453794537,
    0.3794637946379464,
    0.37947379473794735,
    0.37948379483794836,
    0.3794937949379494,
    0.3795037950379504,
    0.37951379513795136,
    0.3795237952379524,
    0.3795337953379534,
    0.37954379543795436,
    0.3795537955379554,
    0.3795637956379564,
    0.37957379573795735,
    0.37958379583795837,
    0.3795937959379594,
    0.37960379603796035,
    0.37961379613796137,
    0.3796237962379624,
    0.37963379633796335,
    0.37964379643796436,
    0.3796537965379654,
    0.3796637966379664,
    0.37967379673796736,
    0.3796837968379684,
    0.3796937969379694,
    0.37970379703797036,
    0.3797137971379714,
    0.3797237972379724,
    0.37973379733797336,
    0.3797437974379744,
    0.3797537975379754,
    0.37976379763797635,
    0.37977379773797737,
    0.3797837978379784,
    0.37979379793797935,
    0.37980379803798037,
    0.3798137981379814,
    0.37982379823798235,
    0.37983379833798336,
    0.3798437984379844,
    0.3798537985379854,
    0.37986379863798636,
    0.3798737987379874,
    0.3798837988379884,
    0.37989379893798936,
    0.3799037990379904,
    0.3799137991379914,
    0.37992379923799235,
    0.37993379933799337,
    0.3799437994379944,
    0.37995379953799535,
    0.37996379963799637,
    0.3799737997379974,
    0.37998379983799835,
    0.37999379993799937,
    0.3800038000380004,
    0.3800138001380014,
    0.38002380023800236,
    0.3800338003380034,
    0.3800438004380044,
    0.38005380053800536,
    0.3800638006380064,
    0.3800738007380074,
    0.38008380083800836,
    0.3800938009380094,
    0.3801038010380104,
    0.38011380113801135,
    0.38012380123801237,
    0.3801338013380134,
    0.38014380143801435,
    0.38015380153801537,
    0.3801638016380164,
    0.38017380173801735,
    0.38018380183801836,
    0.3801938019380194,
    0.3802038020380204,
    0.38021380213802136,
    0.3802238022380224,
    0.3802338023380234,
    0.38024380243802436,
    0.3802538025380254,
    0.3802638026380264,
    0.38027380273802736,
    0.3802838028380284,
    0.3802938029380294,
    0.38030380303803035,
    0.38031380313803137,
    0.3803238032380324,
    0.38033380333803335,
    0.38034380343803437,
    0.3803538035380354,
    0.38036380363803635,
    0.38037380373803736,
    0.3803838038380384,
    0.3803938039380394,
    0.38040380403804036,
    0.3804138041380414,
    0.3804238042380424,
    0.38043380433804336,
    0.3804438044380444,
    0.3804538045380454,
    0.38046380463804635,
    0.38047380473804737,
    0.3804838048380484,
    0.38049380493804935,
    0.38050380503805037,
    0.3805138051380514,
    0.38052380523805235,
    0.38053380533805337,
    0.3805438054380544,
    0.3805538055380554,
    0.38056380563805636,
    0.3805738057380574,
    0.3805838058380584,
    0.38059380593805936,
    0.3806038060380604,
    0.3806138061380614,
    0.38062380623806236,
    0.3806338063380634,
    0.3806438064380644,
    0.38065380653806535,
    0.38066380663806637,
    0.3806738067380674,
    0.38068380683806835,
    0.38069380693806937,
    0.3807038070380704,
    0.38071380713807135,
    0.38072380723807236,
    0.3807338073380734,
    0.3807438074380744,
    0.38075380753807536,
    0.3807638076380764,
    0.3807738077380774,
    0.38078380783807836,
    0.3807938079380794,
    0.3808038080380804,
    0.38081380813808136,
    0.3808238082380824,
    0.3808338083380834,
    0.38084380843808435,
    0.38085380853808537,
    0.3808638086380864,
    0.38087380873808735,
    0.38088380883808837,
    0.3808938089380894,
    0.3809038090380904,
    0.38091380913809136,
    0.3809238092380924,
    0.3809338093380934,
    0.38094380943809436,
    0.3809538095380954,
    0.3809638096380964,
    0.38097380973809736,
    0.3809838098380984,
    0.3809938099380994,
    0.38100381003810035,
    0.38101381013810137,
    0.3810238102381024,
    0.38103381033810335,
    0.38104381043810437,
    0.3810538105381054,
    0.38106381063810635,
    0.38107381073810737,
    0.3810838108381084,
    0.3810938109381094,
    0.38110381103811036,
    0.3811138111381114,
    0.3811238112381124,
    0.38113381133811336,
    0.3811438114381144,
    0.3811538115381154,
    0.38116381163811636,
    0.3811738117381174,
    0.3811838118381184,
    0.38119381193811935,
    0.38120381203812037,
    0.3812138121381214,
    0.38122381223812235,
    0.38123381233812337,
    0.3812438124381244,
    0.38125381253812535,
    0.38126381263812636,
    0.3812738127381274,
    0.3812838128381284,
    0.38129381293812936,
    0.3813038130381304,
    0.3813138131381314,
    0.38132381323813236,
    0.3813338133381334,
    0.3813438134381344,
    0.38135381353813536,
    0.3813638136381364,
    0.3813738137381374,
    0.38138381383813835,
    0.38139381393813937,
    0.3814038140381404,
    0.38141381413814135,
    0.38142381423814237,
    0.3814338143381434,
    0.3814438144381444,
    0.38145381453814536,
    0.3814638146381464,
    0.3814738147381474,
    0.38148381483814836,
    0.3814938149381494,
    0.3815038150381504,
    0.38151381513815136,
    0.3815238152381524,
    0.3815338153381534,
    0.38154381543815435,
    0.38155381553815537,
    0.3815638156381564,
    0.38157381573815735,
    0.38158381583815837,
    0.3815938159381594,
    0.38160381603816035,
    0.38161381613816137,
    0.3816238162381624,
    0.3816338163381634,
    0.38164381643816436,
    0.3816538165381654,
    0.3816638166381664,
    0.38167381673816736,
    0.3816838168381684,
    0.3816938169381694,
    0.38170381703817036,
    0.3817138171381714,
    0.3817238172381724,
    0.38173381733817335,
    0.38174381743817437,
    0.3817538175381754,
    0.38176381763817635,
    0.38177381773817737,
    0.3817838178381784,
    0.3817938179381794,
    0.38180381803818036,
    0.3818138181381814,
    0.3818238182381824,
    0.38183381833818336,
    0.3818438184381844,
    0.3818538185381854,
    0.38186381863818636,
    0.3818738187381874,
    0.3818838188381884,
    0.38189381893818936,
    0.3819038190381904,
    0.3819138191381914,
    0.38192381923819235,
    0.38193381933819337,
    0.3819438194381944,
    0.38195381953819535,
    0.38196381963819637,
    0.3819738197381974,
    0.3819838198381984,
    0.38199381993819936,
    0.3820038200382004,
    0.3820138201382014,
    0.38202382023820236,
    0.3820338203382034,
    0.3820438204382044,
    0.38205382053820536,
    0.3820638206382064,
    0.3820738207382074,
    0.38208382083820835,
    0.38209382093820937,
    0.3821038210382104,
    0.38211382113821135,
    0.38212382123821237,
    0.3821338213382134,
    0.38214382143821435,
    0.38215382153821537,
    0.3821638216382164,
    0.3821738217382174,
    0.38218382183821836,
    0.3821938219382194,
    0.3822038220382204,
    0.38221382213822136,
    0.3822238222382224,
    0.3822338223382234,
    0.38224382243822436,
    0.3822538225382254,
    0.3822638226382264,
    0.38227382273822735,
    0.38228382283822837,
    0.3822938229382294,
    0.38230382303823035,
    0.38231382313823137,
    0.3823238232382324,
    0.3823338233382334,
    0.38234382343823436,
    0.3823538235382354,
    0.3823638236382364,
    0.38237382373823736,
    0.3823838238382384,
    0.3823938239382394,
    0.38240382403824036,
    0.3824138241382414,
    0.3824238242382424,
    0.38243382433824336,
    0.3824438244382444,
    0.3824538245382454,
    0.38246382463824635,
    0.38247382473824737,
    0.3824838248382484,
    0.38249382493824935,
    0.38250382503825037,
    0.3825138251382514,
    0.3825238252382524,
    0.38253382533825336,
    0.3825438254382544,
    0.3825538255382554,
    0.38256382563825636,
    0.3825738257382574,
    0.3825838258382584,
    0.38259382593825936,
    0.3826038260382604,
    0.3826138261382614,
    0.38262382623826235,
    0.38263382633826337,
    0.3826438264382644,
    0.38265382653826535,
    0.38266382663826637,
    0.3826738267382674,
    0.3826838268382684,
    0.38269382693826937,
    0.3827038270382704,
    0.3827138271382714,
    0.38272382723827236,
    0.3827338273382734,
    0.3827438274382744,
    0.38275382753827536,
    0.3827638276382764,
    0.3827738277382774,
    0.38278382783827836,
    0.3827938279382794,
    0.3828038280382804,
    0.38281382813828135,
    0.38282382823828237,
    0.3828338283382834,
    0.38284382843828435,
    0.38285382853828537,
    0.3828638286382864,
    0.3828738287382874,
    0.38288382883828836,
    0.3828938289382894,
    0.3829038290382904,
    0.38291382913829136,
    0.3829238292382924,
    0.3829338293382934,
    0.38294382943829436,
    0.3829538295382954,
    0.3829638296382964,
    0.38297382973829736,
    0.3829838298382984,
    0.3829938299382994,
    0.38300383003830035,
    0.38301383013830137,
    0.3830238302383024,
    0.38303383033830335,
    0.38304383043830437,
    0.3830538305383054,
    0.3830638306383064,
    0.38307383073830736,
    0.3830838308383084,
    0.3830938309383094,
    0.38310383103831036,
    0.3831138311383114,
    0.3831238312383124,
    0.38313383133831336,
    0.3831438314383144,
    0.3831538315383154,
    0.38316383163831635,
    0.38317383173831737,
    0.3831838318383184,
    0.38319383193831935,
    0.38320383203832037,
    0.3832138321383214,
    0.3832238322383224,
    0.38323383233832337,
    0.3832438324383244,
    0.3832538325383254,
    0.38326383263832636,
    0.3832738327383274,
    0.3832838328383284,
    0.38329383293832936,
    0.3833038330383304,
    0.3833138331383314,
    0.38332383323833236,
    0.3833338333383334,
    0.3833438334383344,
    0.38335383353833535,
    0.38336383363833637,
    0.3833738337383374,
    0.38338383383833835,
    0.38339383393833937,
    0.3834038340383404,
    0.3834138341383414,
    0.38342383423834236,
    0.3834338343383434,
    0.3834438344383444,
    0.38345383453834536,
    0.3834638346383464,
    0.3834738347383474,
    0.38348383483834836,
    0.3834938349383494,
    0.3835038350383504,
    0.38351383513835136,
    0.3835238352383524,
    0.3835338353383534,
    0.38354383543835435,
    0.38355383553835537,
    0.3835638356383564,
    0.38357383573835735,
    0.38358383583835837,
    0.3835938359383594,
    0.3836038360383604,
    0.38361383613836136,
    0.3836238362383624,
    0.3836338363383634,
    0.38364383643836436,
    0.3836538365383654,
    0.3836638366383664,
    0.38367383673836736,
    0.3836838368383684,
    0.3836938369383694,
    0.38370383703837035,
    0.38371383713837137,
    0.3837238372383724,
    0.38373383733837335,
    0.38374383743837437,
    0.3837538375383754,
    0.3837638376383764,
    0.38377383773837737,
    0.3837838378383784,
    0.3837938379383794,
    0.38380383803838036,
    0.3838138381383814,
    0.3838238382383824,
    0.38383383833838336,
    0.3838438384383844,
    0.3838538385383854,
    0.38386383863838636,
    0.3838738387383874,
    0.3838838388383884,
    0.38389383893838935,
    0.38390383903839037,
    0.3839138391383914,
    0.38392383923839235,
    0.38393383933839337,
    0.3839438394383944,
    0.3839538395383954,
    0.38396383963839636,
    0.3839738397383974,
    0.3839838398383984,
    0.38399383993839936,
    0.3840038400384004,
    0.3840138401384014,
    0.38402384023840236,
    0.3840338403384034,
    0.3840438404384044,
    0.38405384053840536,
    0.3840638406384064,
    0.3840738407384074,
    0.38408384083840835,
    0.38409384093840937,
    0.3841038410384104,
    0.3841138411384114,
    0.38412384123841237,
    0.3841338413384134,
    0.3841438414384144,
    0.38415384153841536,
    0.3841638416384164,
    0.3841738417384174,
    0.38418384183841836,
    0.3841938419384194,
    0.3842038420384204,
    0.38421384213842136,
    0.3842238422384224,
    0.3842338423384234,
    0.38424384243842435,
    0.38425384253842537,
    0.3842638426384264,
    0.38427384273842735,
    0.38428384283842837,
    0.3842938429384294,
    0.3843038430384304,
    0.38431384313843137,
    0.3843238432384324,
    0.3843338433384334,
    0.38434384343843436,
    0.3843538435384354,
    0.3843638436384364,
    0.38437384373843736,
    0.3843838438384384,
    0.3843938439384394,
    0.38440384403844036,
    0.3844138441384414,
    0.3844238442384424,
    0.38443384433844335,
    0.38444384443844437,
    0.3844538445384454,
    0.38446384463844635,
    0.38447384473844737,
    0.3844838448384484,
    0.3844938449384494,
    0.38450384503845036,
    0.3845138451384514,
    0.3845238452384524,
    0.38453384533845336,
    0.3845438454384544,
    0.3845538455384554,
    0.38456384563845636,
    0.3845738457384574,
    0.3845838458384584,
    0.38459384593845936,
    0.3846038460384604,
    0.3846138461384614,
    0.38462384623846235,
    0.38463384633846337,
    0.3846438464384644,
    0.3846538465384654,
    0.38466384663846637,
    0.3846738467384674,
    0.3846838468384684,
    0.38469384693846936,
    0.3847038470384704,
    0.3847138471384714,
    0.38472384723847236,
    0.3847338473384734,
    0.3847438474384744,
    0.38475384753847536,
    0.3847638476384764,
    0.3847738477384774,
    0.38478384783847835,
    0.38479384793847937,
    0.3848038480384804,
    0.38481384813848135,
    0.38482384823848237,
    0.3848338483384834,
    0.3848438484384844,
    0.38485384853848537,
    0.3848638486384864,
    0.3848738487384874,
    0.38488384883848836,
    0.3848938489384894,
    0.3849038490384904,
    0.38491384913849136,
    0.3849238492384924,
    0.3849338493384934,
    0.38494384943849436,
    0.3849538495384954,
    0.3849638496384964,
    0.38497384973849735,
    0.38498384983849837,
    0.3849938499384994,
    0.3850038500385004,
    0.38501385013850137,
    0.3850238502385024,
    0.3850338503385034,
    0.38504385043850436,
    0.3850538505385054,
    0.3850638506385064,
    0.38507385073850736,
    0.3850838508385084,
    0.3850938509385094,
    0.38510385103851036,
    0.3851138511385114,
    0.3851238512385124,
    0.38513385133851336,
    0.3851438514385144,
    0.3851538515385154,
    0.38516385163851635,
    0.38517385173851737,
    0.3851838518385184,
    0.3851938519385194,
    0.38520385203852037,
    0.3852138521385214,
    0.3852238522385224,
    0.38523385233852336,
    0.3852438524385244,
    0.3852538525385254,
    0.38526385263852636,
    0.3852738527385274,
    0.3852838528385284,
    0.38529385293852936,
    0.3853038530385304,
    0.3853138531385314,
    0.38532385323853235,
    0.38533385333853337,
    0.3853438534385344,
    0.38535385353853535,
    0.38536385363853637,
    0.3853738537385374,
    0.3853838538385384,
    0.38539385393853937,
    0.3854038540385404,
    0.3854138541385414,
    0.38542385423854236,
    0.3854338543385434,
    0.3854438544385444,
    0.38545385453854536,
    0.3854638546385464,
    0.3854738547385474,
    0.38548385483854836,
    0.3854938549385494,
    0.3855038550385504,
    0.38551385513855135,
    0.38552385523855237,
    0.3855338553385534,
    0.3855438554385544,
    0.38555385553855537,
    0.3855638556385564,
    0.3855738557385574,
    0.38558385583855836,
    0.3855938559385594,
    0.3856038560385604,
    0.38561385613856136,
    0.3856238562385624,
    0.3856338563385634,
    0.38564385643856436,
    0.3856538565385654,
    0.3856638566385664,
    0.38567385673856736,
    0.3856838568385684,
    0.3856938569385694,
    0.38570385703857035,
    0.38571385713857137,
    0.3857238572385724,
    0.3857338573385734,
    0.38574385743857437,
    0.3857538575385754,
    0.3857638576385764,
    0.38577385773857736,
    0.3857838578385784,
    0.3857938579385794,
    0.38580385803858036,
    0.3858138581385814,
    0.3858238582385824,
    0.38583385833858336,
    0.3858438584385844,
    0.3858538585385854,
    0.38586385863858635,
    0.38587385873858737,
    0.3858838588385884,
    0.3858938589385894,
    0.38590385903859037,
    0.3859138591385914,
    0.3859238592385924,
    0.38593385933859337,
    0.3859438594385944,
    0.3859538595385954,
    0.38596385963859636,
    0.3859738597385974,
    0.3859838598385984,
    0.38599385993859936,
    0.3860038600386004,
    0.3860138601386014,
    0.38602386023860236,
    0.3860338603386034,
    0.3860438604386044,
    0.38605386053860535,
    0.38606386063860637,
    0.3860738607386074,
    0.3860838608386084,
    0.38609386093860937,
    0.3861038610386104,
    0.3861138611386114,
    0.38612386123861236,
    0.3861338613386134,
    0.3861438614386144,
    0.38615386153861536,
    0.3861638616386164,
    0.3861738617386174,
    0.38618386183861836,
    0.3861938619386194,
    0.3862038620386204,
    0.38621386213862136,
    0.3862238622386224,
    0.3862338623386234,
    0.38624386243862435,
    0.38625386253862537,
    0.3862638626386264,
    0.3862738627386274,
    0.38628386283862837,
    0.3862938629386294,
    0.3863038630386304,
    0.38631386313863136,
    0.3863238632386324,
    0.3863338633386334,
    0.38634386343863436,
    0.3863538635386354,
    0.3863638636386364,
    0.38637386373863736,
    0.3863838638386384,
    0.3863938639386394,
    0.38640386403864035,
    0.38641386413864137,
    0.3864238642386424,
    0.3864338643386434,
    0.38644386443864437,
    0.3864538645386454,
    0.3864638646386464,
    0.38647386473864737,
    0.3864838648386484,
    0.3864938649386494,
    0.38650386503865036,
    0.3865138651386514,
    0.3865238652386524,
    0.38653386533865336,
    0.3865438654386544,
    0.3865538655386554,
    0.38656386563865636,
    0.3865738657386574,
    0.3865838658386584,
    0.38659386593865935,
    0.38660386603866037,
    0.3866138661386614,
    0.3866238662386624,
    0.38663386633866337,
    0.3866438664386644,
    0.3866538665386654,
    0.38666386663866636,
    0.3866738667386674,
    0.3866838668386684,
    0.38669386693866936,
    0.3867038670386704,
    0.3867138671386714,
    0.38672386723867236,
    0.3867338673386734,
    0.3867438674386744,
    0.38675386753867536,
    0.3867638676386764,
    0.3867738677386774,
    0.38678386783867835,
    0.38679386793867937,
    0.3868038680386804,
    0.3868138681386814,
    0.38682386823868237,
    0.3868338683386834,
    0.3868438684386844,
    0.38685386853868536,
    0.3868638686386864,
    0.3868738687386874,
    0.38688386883868836,
    0.3868938689386894,
    0.3869038690386904,
    0.38691386913869136,
    0.3869238692386924,
    0.3869338693386934,
    0.38694386943869435,
    0.38695386953869537,
    0.3869638696386964,
    0.3869738697386974,
    0.38698386983869837,
    0.3869938699386994,
    0.3870038700387004,
    0.38701387013870137,
    0.3870238702387024,
    0.3870338703387034,
    0.38704387043870436,
    0.3870538705387054,
    0.3870638706387064,
    0.38707387073870736,
    0.3870838708387084,
    0.3870938709387094,
    0.38710387103871036,
    0.3871138711387114,
    0.3871238712387124,
    0.38713387133871335,
    0.38714387143871437,
    0.3871538715387154,
    0.3871638716387164,
    0.38717387173871737,
    0.3871838718387184,
    0.3871938719387194,
    0.38720387203872036,
    0.3872138721387214,
    0.3872238722387224,
    0.38723387233872336,
    0.3872438724387244,
    0.3872538725387254,
    0.38726387263872636,
    0.3872738727387274,
    0.3872838728387284,
    0.38729387293872936,
    0.3873038730387304,
    0.3873138731387314,
    0.3873238732387324,
    0.38733387333873337,
    0.3873438734387344,
    0.3873538735387354,
    0.38736387363873637,
    0.3873738737387374,
    0.3873838738387384,
    0.38739387393873936,
    0.3874038740387404,
    0.3874138741387414,
    0.38742387423874236,
    0.3874338743387434,
    0.3874438744387444,
    0.38745387453874536,
    0.3874638746387464,
    0.3874738747387474,
    0.38748387483874835,
    0.38749387493874937,
    0.3875038750387504,
    0.3875138751387514,
    0.38752387523875237,
    0.3875338753387534,
    0.3875438754387544,
    0.38755387553875537,
    0.3875638756387564,
    0.3875738757387574,
    0.38758387583875836,
    0.3875938759387594,
    0.3876038760387604,
    0.38761387613876136,
    0.3876238762387624,
    0.3876338763387634,
    0.38764387643876436,
    0.3876538765387654,
    0.3876638766387664,
    0.38767387673876735,
    0.38768387683876837,
    0.3876938769387694,
    0.3877038770387704,
    0.38771387713877137,
    0.3877238772387724,
    0.3877338773387734,
    0.38774387743877436,
    0.3877538775387754,
    0.3877638776387764,
    0.38777387773877736,
    0.3877838778387784,
    0.3877938779387794,
    0.38780387803878036,
    0.3878138781387814,
    0.3878238782387824,
    0.38783387833878336,
    0.3878438784387844,
    0.3878538785387854,
    0.3878638786387864,
    0.38787387873878737,
    0.3878838788387884,
    0.3878938789387894,
    0.38790387903879037,
    0.3879138791387914,
    0.3879238792387924,
    0.38793387933879336,
    0.3879438794387944,
    0.3879538795387954,
    0.38796387963879636,
    0.3879738797387974,
    0.3879838798387984,
    0.38799387993879936,
    0.3880038800388004,
    0.3880138801388014,
    0.38802388023880235,
    0.38803388033880337,
    0.3880438804388044,
    0.3880538805388054,
    0.38806388063880637,
    0.3880738807388074,
    0.3880838808388084,
    0.38809388093880937,
    0.3881038810388104,
    0.3881138811388114,
    0.38812388123881236,
    0.3881338813388134,
    0.3881438814388144,
    0.38815388153881536,
    0.3881638816388164,
    0.3881738817388174,
    0.38818388183881836,
    0.3881938819388194,
    0.3882038820388204,
    0.3882138821388214,
    0.38822388223882237,
    0.3882338823388234,
    0.3882438824388244,
    0.38825388253882537,
    0.3882638826388264,
    0.3882738827388274,
    0.38828388283882836,
    0.3882938829388294,
    0.3883038830388304,
    0.38831388313883136,
    0.3883238832388324,
    0.3883338833388334,
    0.38834388343883436,
    0.3883538835388354,
    0.3883638836388364,
    0.38837388373883736,
    0.3883838838388384,
    0.3883938839388394,
    0.3884038840388404,
    0.38841388413884137,
    0.3884238842388424,
    0.3884338843388434,
    0.38844388443884437,
    0.3884538845388454,
    0.3884638846388464,
    0.38847388473884736,
    0.3884838848388484,
    0.3884938849388494,
    0.38850388503885036,
    0.3885138851388514,
    0.3885238852388524,
    0.38853388533885336,
    0.3885438854388544,
    0.3885538855388554,
    0.38856388563885635,
    0.38857388573885737,
    0.3885838858388584,
    0.3885938859388594,
    0.38860388603886037,
    0.3886138861388614,
    0.3886238862388624,
    0.38863388633886337,
    0.3886438864388644,
    0.3886538865388654,
    0.38866388663886636,
    0.3886738867388674,
    0.3886838868388684,
    0.38869388693886936,
    0.3887038870388704,
    0.3887138871388714,
    0.38872388723887236,
    0.3887338873388734,
    0.3887438874388744,
    0.3887538875388754,
    0.38876388763887637,
    0.3887738877388774,
    0.3887838878388784,
    0.38879388793887937,
    0.3888038880388804,
    0.3888138881388814,
    0.38882388823888236,
    0.3888338883388834,
    0.3888438884388844,
    0.38885388853888536,
    0.3888638886388864,
    0.3888738887388874,
    0.38888388883888836,
    0.3888938889388894,
    0.3889038890388904,
    0.38891388913889136,
    0.3889238892388924,
    0.3889338893388934,
    0.3889438894388944,
    0.38895388953889537,
    0.3889638896388964,
    0.3889738897388974,
    0.38898388983889837,
    0.3889938899388994,
    0.3890038900389004,
    0.38901389013890136,
    0.3890238902389024,
    0.3890338903389034,
    0.38904389043890436,
    0.3890538905389054,
    0.3890638906389064,
    0.38907389073890736,
    0.3890838908389084,
    0.3890938909389094,
    0.3891038910389104,
    0.38911389113891137,
    0.3891238912389124,
    0.3891338913389134,
    0.38914389143891437,
    0.3891538915389154,
    0.3891638916389164,
    0.38917389173891737,
    0.3891838918389184,
    0.3891938919389194,
    0.38920389203892036,
    0.3892138921389214,
    0.3892238922389224,
    0.38923389233892336,
    0.3892438924389244,
    0.3892538925389254,
    0.38926389263892636,
    0.3892738927389274,
    0.3892838928389284,
    0.3892938929389294,
    0.38930389303893037,
    0.3893138931389314,
    0.3893238932389324,
    0.38933389333893337,
    0.3893438934389344,
    0.3893538935389354,
    0.38936389363893636,
    0.3893738937389374,
    0.3893838938389384,
    0.38939389393893936,
    0.3894038940389404,
    0.3894138941389414,
    0.38942389423894236,
    0.3894338943389434,
    0.3894438944389444,
    0.38945389453894536,
    0.3894638946389464,
    0.3894738947389474,
    0.3894838948389484,
    0.38949389493894937,
    0.3895038950389504,
    0.3895138951389514,
    0.38952389523895237,
    0.3895338953389534,
    0.3895438954389544,
    0.38955389553895536,
    0.3895638956389564,
    0.3895738957389574,
    0.38958389583895836,
    0.3895938959389594,
    0.3896038960389604,
    0.38961389613896136,
    0.3896238962389624,
    0.3896338963389634,
    0.3896438964389644,
    0.38965389653896537,
    0.3896638966389664,
    0.3896738967389674,
    0.38968389683896837,
    0.3896938969389694,
    0.3897038970389704,
    0.38971389713897137,
    0.3897238972389724,
    0.3897338973389734,
    0.38974389743897436,
    0.3897538975389754,
    0.3897638976389764,
    0.38977389773897736,
    0.3897838978389784,
    0.3897938979389794,
    0.38980389803898036,
    0.3898138981389814,
    0.3898238982389824,
    0.3898338983389834,
    0.38984389843898437,
    0.3898538985389854,
    0.3898638986389864,
    0.38987389873898737,
    0.3898838988389884,
    0.3898938989389894,
    0.38990389903899036,
    0.3899138991389914,
    0.3899238992389924,
    0.38993389933899336,
    0.3899438994389944,
    0.3899538995389954,
    0.38996389963899636,
    0.3899738997389974,
    0.3899838998389984,
    0.3899938999389994,
    0.3900039000390004,
    0.3900139001390014,
    0.3900239002390024,
    0.39003390033900337,
    0.3900439004390044,
    0.3900539005390054,
    0.39006390063900637,
    0.3900739007390074,
    0.3900839008390084,
    0.39009390093900936,
    0.3901039010390104,
    0.3901139011390114,
    0.39012390123901236,
    0.3901339013390134,
    0.3901439014390144,
    0.39015390153901536,
    0.3901639016390164,
    0.3901739017390174,
    0.3901839018390184,
    0.39019390193901937,
    0.3902039020390204,
    0.3902139021390214,
    0.39022390223902237,
    0.3902339023390234,
    0.3902439024390244,
    0.39025390253902537,
    0.3902639026390264,
    0.3902739027390274,
    0.39028390283902836,
    0.3902939029390294,
    0.3903039030390304,
    0.39031390313903136,
    0.3903239032390324,
    0.3903339033390334,
    0.39034390343903436,
    0.3903539035390354,
    0.3903639036390364,
    0.3903739037390374,
    0.39038390383903837,
    0.3903939039390394,
    0.3904039040390404,
    0.39041390413904137,
    0.3904239042390424,
    0.3904339043390434,
    0.39044390443904436,
    0.3904539045390454,
    0.3904639046390464,
    0.39047390473904736,
    0.3904839048390484,
    0.3904939049390494,
    0.39050390503905036,
    0.3905139051390514,
    0.3905239052390524,
    0.3905339053390534,
    0.3905439054390544,
    0.3905539055390554,
    0.3905639056390564,
    0.39057390573905737,
    0.3905839058390584,
    0.3905939059390594,
    0.39060390603906037,
    0.3906139061390614,
    0.3906239062390624,
    0.39063390633906336,
    0.3906439064390644,
    0.3906539065390654,
    0.39066390663906636,
    0.3906739067390674,
    0.3906839068390684,
    0.39069390693906936,
    0.3907039070390704,
    0.3907139071390714,
    0.3907239072390724,
    0.39073390733907337,
    0.3907439074390744,
    0.3907539075390754,
    0.39076390763907637,
    0.3907739077390774,
    0.3907839078390784,
    0.39079390793907937,
    0.3908039080390804,
    0.3908139081390814,
    0.39082390823908236,
    0.3908339083390834,
    0.3908439084390844,
    0.39085390853908536,
    0.3908639086390864,
    0.3908739087390874,
    0.39088390883908836,
    0.3908939089390894,
    0.3909039090390904,
    0.3909139091390914,
    0.39092390923909237,
    0.3909339093390934,
    0.3909439094390944,
    0.39095390953909537,
    0.3909639096390964,
    0.3909739097390974,
    0.39098390983909836,
    0.3909939099390994,
    0.3910039100391004,
    0.39101391013910136,
    0.3910239102391024,
    0.3910339103391034,
    0.39104391043910436,
    0.3910539105391054,
    0.3910639106391064,
    0.3910739107391074,
    0.3910839108391084,
    0.3910939109391094,
    0.3911039110391104,
    0.39111391113911137,
    0.3911239112391124,
    0.3911339113391134,
    0.39114391143911437,
    0.3911539115391154,
    0.3911639116391164,
    0.39117391173911736,
    0.3911839118391184,
    0.3911939119391194,
    0.39120391203912036,
    0.3912139121391214,
    0.3912239122391224,
    0.39123391233912336,
    0.3912439124391244,
    0.3912539125391254,
    0.3912639126391264,
    0.39127391273912737,
    0.3912839128391284,
    0.3912939129391294,
    0.39130391303913037,
    0.3913139131391314,
    0.3913239132391324,
    0.39133391333913337,
    0.3913439134391344,
    0.3913539135391354,
    0.39136391363913636,
    0.3913739137391374,
    0.3913839138391384,
    0.39139391393913936,
    0.3914039140391404,
    0.3914139141391414,
    0.3914239142391424,
    0.3914339143391434,
    0.3914439144391444,
    0.3914539145391454,
    0.39146391463914637,
    0.3914739147391474,
    0.3914839148391484,
    0.39149391493914937,
    0.3915039150391504,
    0.3915139151391514,
    0.39152391523915236,
    0.3915339153391534,
    0.3915439154391544,
    0.39155391553915536,
    0.3915639156391564,
    0.3915739157391574,
    0.39158391583915836,
    0.3915939159391594,
    0.3916039160391604,
    0.3916139161391614,
    0.3916239162391624,
    0.3916339163391634,
    0.3916439164391644,
    0.39165391653916537,
    0.3916639166391664,
    0.3916739167391674,
    0.39168391683916837,
    0.3916939169391694,
    0.3917039170391704,
    0.39171391713917136,
    0.3917239172391724,
    0.3917339173391734,
    0.39174391743917436,
    0.3917539175391754,
    0.3917639176391764,
    0.39177391773917736,
    0.3917839178391784,
    0.3917939179391794,
    0.3918039180391804,
    0.39181391813918137,
    0.3918239182391824,
    0.3918339183391834,
    0.39184391843918437,
    0.3918539185391854,
    0.3918639186391864,
    0.39187391873918737,
    0.3918839188391884,
    0.3918939189391894,
    0.39190391903919036,
    0.3919139191391914,
    0.3919239192391924,
    0.39193391933919336,
    0.3919439194391944,
    0.3919539195391954,
    0.3919639196391964,
    0.3919739197391974,
    0.3919839198391984,
    0.3919939199391994,
    0.39200392003920037,
    0.3920139201392014,
    0.3920239202392024,
    0.39203392033920337,
    0.3920439204392044,
    0.3920539205392054,
    0.39206392063920636,
    0.3920739207392074,
    0.3920839208392084,
    0.39209392093920936,
    0.3921039210392104,
    0.3921139211392114,
    0.39212392123921236,
    0.3921339213392134,
    0.3921439214392144,
    0.3921539215392154,
    0.3921639216392164,
    0.3921739217392174,
    0.3921839218392184,
    0.39219392193921937,
    0.3922039220392204,
    0.3922139221392214,
    0.39222392223922237,
    0.3922339223392234,
    0.3922439224392244,
    0.39225392253922536,
    0.3922639226392264,
    0.3922739227392274,
    0.39228392283922836,
    0.3922939229392294,
    0.3923039230392304,
    0.3923139231392314,
    0.3923239232392324,
    0.3923339233392334,
    0.3923439234392344,
    0.39235392353923537,
    0.3923639236392364,
    0.3923739237392374,
    0.39238392383923837,
    0.3923939239392394,
    0.3924039240392404,
    0.39241392413924137,
    0.3924239242392424,
    0.3924339243392434,
    0.39244392443924436,
    0.3924539245392454,
    0.3924639246392464,
    0.39247392473924736,
    0.3924839248392484,
    0.3924939249392494,
    0.3925039250392504,
    0.3925139251392514,
    0.3925239252392524,
    0.3925339253392534,
    0.39254392543925437,
    0.3925539255392554,
    0.3925639256392564,
    0.39257392573925737,
    0.3925839258392584,
    0.3925939259392594,
    0.39260392603926036,
    0.3926139261392614,
    0.3926239262392624,
    0.39263392633926336,
    0.3926439264392644,
    0.3926539265392654,
    0.39266392663926636,
    0.3926739267392674,
    0.3926839268392684,
    0.3926939269392694,
    0.3927039270392704,
    0.3927139271392714,
    0.3927239272392724,
    0.39273392733927337,
    0.3927439274392744,
    0.3927539275392754,
    0.39276392763927637,
    0.3927739277392774,
    0.3927839278392784,
    0.39279392793927936,
    0.3928039280392804,
    0.3928139281392814,
    0.39282392823928236,
    0.3928339283392834,
    0.3928439284392844,
    0.3928539285392854,
    0.3928639286392864,
    0.3928739287392874,
    0.3928839288392884,
    0.39289392893928937,
    0.3929039290392904,
    0.3929139291392914,
    0.39292392923929237,
    0.3929339293392934,
    0.3929439294392944,
    0.39295392953929537,
    0.3929639296392964,
    0.3929739297392974,
    0.39298392983929836,
    0.3929939299392994,
    0.3930039300393004,
    0.39301393013930136,
    0.3930239302393024,
    0.3930339303393034,
    0.3930439304393044,
    0.3930539305393054,
    0.3930639306393064,
    0.3930739307393074,
    0.39308393083930837,
    0.3930939309393094,
    0.3931039310393104,
    0.39311393113931137,
    0.3931239312393124,
    0.3931339313393134,
    0.39314393143931436,
    0.3931539315393154,
    0.3931639316393164,
    0.39317393173931736,
    0.3931839318393184,
    0.3931939319393194,
    0.3932039320393204,
    0.3932139321393214,
    0.3932239322393224,
    0.3932339323393234,
    0.3932439324393244,
    0.3932539325393254,
    0.3932639326393264,
    0.39327393273932737,
    0.3932839328393284,
    0.3932939329393294,
    0.39330393303933037,
    0.3933139331393314,
    0.3933239332393324,
    0.39333393333933336,
    0.3933439334393344,
    0.3933539335393354,
    0.39336393363933636,
    0.3933739337393374,
    0.3933839338393384,
    0.3933939339393394,
    0.3934039340393404,
    0.3934139341393414,
    0.3934239342393424,
    0.39343393433934337,
    0.3934439344393444,
    0.3934539345393454,
    0.39346393463934637,
    0.3934739347393474,
    0.3934839348393484,
    0.39349393493934937,
    0.3935039350393504,
    0.3935139351393514,
    0.39352393523935236,
    0.3935339353393534,
    0.3935439354393544,
    0.39355393553935536,
    0.3935639356393564,
    0.3935739357393574,
    0.3935839358393584,
    0.3935939359393594,
    0.3936039360393604,
    0.3936139361393614,
    0.39362393623936237,
    0.3936339363393634,
    0.3936439364393644,
    0.39365393653936537,
    0.3936639366393664,
    0.3936739367393674,
    0.39368393683936836,
    0.3936939369393694,
    0.3937039370393704,
    0.39371393713937136,
    0.3937239372393724,
    0.3937339373393734,
    0.3937439374393744,
    0.3937539375393754,
    0.3937639376393764,
    0.3937739377393774,
    0.3937839378393784,
    0.3937939379393794,
    0.3938039380393804,
    0.39381393813938137,
    0.3938239382393824,
    0.3938339383393834,
    0.39384393843938437,
    0.3938539385393854,
    0.3938639386393864,
    0.39387393873938736,
    0.3938839388393884,
    0.3938939389393894,
    0.39390393903939036,
    0.3939139391393914,
    0.3939239392393924,
    0.3939339393393934,
    0.3939439394393944,
    0.3939539395393954,
    0.3939639396393964,
    0.39397393973939737,
    0.3939839398393984,
    0.3939939399393994,
    0.39400394003940037,
    0.3940139401394014,
    0.3940239402394024,
    0.39403394033940337,
    0.3940439404394044,
    0.3940539405394054,
    0.39406394063940636,
    0.3940739407394074,
    0.3940839408394084,
    0.3940939409394094,
    0.3941039410394104,
    0.3941139411394114,
    0.3941239412394124,
    0.3941339413394134,
    0.3941439414394144,
    0.3941539415394154,
    0.39416394163941637,
    0.3941739417394174,
    0.3941839418394184,
    0.39419394193941937,
    0.3942039420394204,
    0.3942139421394214,
    0.39422394223942236,
    0.3942339423394234,
    0.3942439424394244,
    0.39425394253942536,
    0.3942639426394264,
    0.3942739427394274,
    0.3942839428394284,
    0.3942939429394294,
    0.3943039430394304,
    0.3943139431394314,
    0.3943239432394324,
    0.3943339433394334,
    0.3943439434394344,
    0.39435394353943537,
    0.3943639436394364,
    0.3943739437394374,
    0.39438394383943837,
    0.3943939439394394,
    0.3944039440394404,
    0.39441394413944136,
    0.3944239442394424,
    0.3944339443394434,
    0.39444394443944436,
    0.3944539445394454,
    0.3944639446394464,
    0.3944739447394474,
    0.3944839448394484,
    0.3944939449394494,
    0.3945039450394504,
    0.39451394513945137,
    0.3945239452394524,
    0.3945339453394534,
    0.39454394543945437,
    0.3945539455394554,
    0.3945639456394564,
    0.39457394573945737,
    0.3945839458394584,
    0.3945939459394594,
    0.39460394603946036,
    0.3946139461394614,
    0.3946239462394624,
    0.3946339463394634,
    0.3946439464394644,
    0.3946539465394654,
    0.3946639466394664,
    0.3946739467394674,
    0.3946839468394684,
    0.3946939469394694,
    0.39470394703947037,
    0.3947139471394714,
    0.3947239472394724,
    0.39473394733947337,
    0.3947439474394744,
    0.3947539475394754,
    0.39476394763947636,
    0.3947739477394774,
    0.3947839478394784,
    0.39479394793947936,
    0.3948039480394804,
    0.3948139481394814,
    0.3948239482394824,
    0.3948339483394834,
    0.3948439484394844,
    0.3948539485394854,
    0.3948639486394864,
    0.3948739487394874,
    0.3948839488394884,
    0.39489394893948937,
    0.3949039490394904,
    0.3949139491394914,
    0.39492394923949237,
    0.3949339493394934,
    0.3949439494394944,
    0.39495394953949536,
    0.3949639496394964,
    0.3949739497394974,
    0.39498394983949836,
    0.3949939499394994,
    0.3950039500395004,
    0.3950139501395014,
    0.3950239502395024,
    0.3950339503395034,
    0.3950439504395044,
    0.39505395053950537,
    0.3950639506395064,
    0.3950739507395074,
    0.39508395083950837,
    0.3950939509395094,
    0.3951039510395104,
    0.39511395113951137,
    0.3951239512395124,
    0.3951339513395134,
    0.39514395143951436,
    0.3951539515395154,
    0.3951639516395164,
    0.3951739517395174,
    0.3951839518395184,
    0.3951939519395194,
    0.3952039520395204,
    0.3952139521395214,
    0.3952239522395224,
    0.3952339523395234,
    0.39524395243952437,
    0.3952539525395254,
    0.3952639526395264,
    0.39527395273952737,
    0.3952839528395284,
    0.3952939529395294,
    0.39530395303953036,
    0.3953139531395314,
    0.3953239532395324,
    0.39533395333953336,
    0.3953439534395344,
    0.3953539535395354,
    0.3953639536395364,
    0.3953739537395374,
    0.3953839538395384,
    0.3953939539395394,
    0.3954039540395404,
    0.3954139541395414,
    0.3954239542395424,
    0.39543395433954337,
    0.3954439544395444,
    0.3954539545395454,
    0.39546395463954637,
    0.3954739547395474,
    0.3954839548395484,
    0.39549395493954936,
    0.3955039550395504,
    0.3955139551395514,
    0.3955239552395524,
    0.3955339553395534,
    0.3955439554395544,
    0.3955539555395554,
    0.3955639556395564,
    0.3955739557395574,
    0.3955839558395584,
    0.39559395593955937,
    0.3956039560395604,
    0.3956139561395614,
    0.39562395623956237,
    0.3956339563395634,
    0.3956439564395644,
    0.39565395653956537,
    0.3956639566395664,
    0.3956739567395674,
    0.39568395683956836,
    0.3956939569395694,
    0.3957039570395704,
    0.3957139571395714,
    0.3957239572395724,
    0.3957339573395734,
    0.3957439574395744,
    0.3957539575395754,
    0.3957639576395764,
    0.3957739577395774,
    0.39578395783957837,
    0.3957939579395794,
    0.3958039580395804,
    0.39581395813958137,
    0.3958239582395824,
    0.3958339583395834,
    0.39584395843958436,
    0.3958539585395854,
    0.3958639586395864,
    0.39587395873958736,
    0.3958839588395884,
    0.3958939589395894,
    0.3959039590395904,
    0.3959139591395914,
    0.3959239592395924,
    0.3959339593395934,
    0.3959439594395944,
    0.3959539595395954,
    0.3959639596395964,
    0.39597395973959737,
    0.3959839598395984,
    0.3959939599395994,
    0.39600396003960037,
    0.3960139601396014,
    0.3960239602396024,
    0.39603396033960336,
    0.3960439604396044,
    0.3960539605396054,
    0.3960639606396064,
    0.3960739607396074,
    0.3960839608396084,
    0.3960939609396094,
    0.3961039610396104,
    0.3961139611396114,
    0.3961239612396124,
    0.39613396133961337,
    0.3961439614396144,
    0.3961539615396154,
    0.39616396163961637,
    0.3961739617396174,
    0.3961839618396184,
    0.39619396193961937,
    0.3962039620396204,
    0.3962139621396214,
    0.39622396223962236,
    0.3962339623396234,
    0.3962439624396244,
    0.3962539625396254,
    0.3962639626396264,
    0.3962739627396274,
    0.3962839628396284,
    0.3962939629396294,
    0.3963039630396304,
    0.3963139631396314,
    0.39632396323963237,
    0.3963339633396334,
    0.3963439634396344,
    0.39635396353963537,
    0.3963639636396364,
    0.3963739637396374,
    0.39638396383963836,
    0.3963939639396394,
    0.3964039640396404,
    0.3964139641396414,
    0.3964239642396424,
    0.3964339643396434,
    0.3964439644396444,
    0.3964539645396454,
    0.3964639646396464,
    0.3964739647396474,
    0.3964839648396484,
    0.3964939649396494,
    0.3965039650396504,
    0.39651396513965137,
    0.3965239652396524,
    0.3965339653396534,
    0.39654396543965437,
    0.3965539655396554,
    0.3965639656396564,
    0.39657396573965736,
    0.3965839658396584,
    0.3965939659396594,
    0.3966039660396604,
    0.3966139661396614,
    0.3966239662396624,
    0.3966339663396634,
    0.3966439664396644,
    0.3966539665396654,
    0.3966639666396664,
    0.39667396673966737,
    0.3966839668396684,
    0.3966939669396694,
    0.39670396703967037,
    0.3967139671396714,
    0.3967239672396724,
    0.39673396733967337,
    0.3967439674396744,
    0.3967539675396754,
    0.39676396763967636,
    0.3967739677396774,
    0.3967839678396784,
    0.3967939679396794,
    0.3968039680396804,
    0.3968139681396814,
    0.3968239682396824,
    0.3968339683396834,
    0.3968439684396844,
    0.3968539685396854,
    0.39686396863968637,
    0.3968739687396874,
    0.3968839688396884,
    0.39689396893968937,
    0.3969039690396904,
    0.3969139691396914,
    0.39692396923969236,
    0.3969339693396934,
    0.3969439694396944,
    0.3969539695396954,
    0.3969639696396964,
    0.3969739697396974,
    0.3969839698396984,
    0.3969939699396994,
    0.3970039700397004,
    0.3970139701397014,
    0.3970239702397024,
    0.3970339703397034,
    0.3970439704397044,
    0.39705397053970537,
    0.3970639706397064,
    0.3970739707397074,
    0.39708397083970837,
    0.3970939709397094,
    0.3971039710397104,
    0.39711397113971136,
    0.3971239712397124,
    0.3971339713397134,
    0.3971439714397144,
    0.3971539715397154,
    0.3971639716397164,
    0.3971739717397174,
    0.3971839718397184,
    0.3971939719397194,
    0.3972039720397204,
    0.39721397213972137,
    0.3972239722397224,
    0.3972339723397234,
    0.39724397243972437,
    0.3972539725397254,
    0.3972639726397264,
    0.39727397273972737,
    0.3972839728397284,
    0.3972939729397294,
    0.3973039730397304,
    0.3973139731397314,
    0.3973239732397324,
    0.3973339733397334,
    0.3973439734397344,
    0.3973539735397354,
    0.3973639736397364,
    0.3973739737397374,
    0.3973839738397384,
    0.3973939739397394,
    0.39740397403974037,
    0.3974139741397414,
    0.3974239742397424,
    0.39743397433974337,
    0.3974439744397444,
    0.3974539745397454,
    0.39746397463974636,
    0.3974739747397474,
    0.3974839748397484,
    0.3974939749397494,
    0.3975039750397504,
    0.3975139751397514,
    0.3975239752397524,
    0.3975339753397534,
    0.3975439754397544,
    0.3975539755397554,
    0.3975639756397564,
    0.3975739757397574,
    0.3975839758397584,
    0.39759397593975937,
    0.3976039760397604,
    0.3976139761397614,
    0.39762397623976237,
    0.3976339763397634,
    0.3976439764397644,
    0.39765397653976536,
    0.3976639766397664,
    0.3976739767397674,
    0.3976839768397684,
    0.3976939769397694,
    0.3977039770397704,
    0.3977139771397714,
    0.3977239772397724,
    0.3977339773397734,
    0.3977439774397744,
    0.39775397753977537,
    0.3977639776397764,
    0.3977739777397774,
    0.39778397783977837,
    0.3977939779397794,
    0.3978039780397804,
    0.39781397813978137,
    0.3978239782397824,
    0.3978339783397834,
    0.3978439784397844,
    0.3978539785397854,
    0.3978639786397864,
    0.3978739787397874,
    0.3978839788397884,
    0.3978939789397894,
    0.3979039790397904,
    0.3979139791397914,
    0.3979239792397924,
    0.3979339793397934,
    0.39794397943979437,
    0.3979539795397954,
    0.3979639796397964,
    0.39797397973979737,
    0.3979839798397984,
    0.3979939799397994,
    0.39800398003980036,
    0.3980139801398014,
    0.3980239802398024,
    0.3980339803398034,
    0.3980439804398044,
    0.3980539805398054,
    0.3980639806398064,
    0.3980739807398074,
    0.3980839808398084,
    0.3980939809398094,
    0.3981039810398104,
    0.3981139811398114,
    0.3981239812398124,
    0.39813398133981337,
    0.3981439814398144,
    0.3981539815398154,
    0.39816398163981637,
    0.3981739817398174,
    0.3981839818398184,
    0.39819398193981936,
    0.3982039820398204,
    0.3982139821398214,
    0.3982239822398224,
    0.3982339823398234,
    0.3982439824398244,
    0.3982539825398254,
    0.3982639826398264,
    0.3982739827398274,
    0.3982839828398284,
    0.39829398293982937,
    0.3983039830398304,
    0.3983139831398314,
    0.39832398323983237,
    0.3983339833398334,
    0.3983439834398344,
    0.39835398353983537,
    0.3983639836398364,
    0.3983739837398374,
    0.3983839838398384,
    0.3983939839398394,
    0.3984039840398404,
    0.3984139841398414,
    0.3984239842398424,
    0.3984339843398434,
    0.3984439844398444,
    0.3984539845398454,
    0.3984639846398464,
    0.3984739847398474,
    0.39848398483984837,
    0.3984939849398494,
    0.3985039850398504,
    0.39851398513985137,
    0.3985239852398524,
    0.3985339853398534,
    0.39854398543985436,
    0.3985539855398554,
    0.3985639856398564,
    0.3985739857398574,
    0.3985839858398584,
    0.3985939859398594,
    0.3986039860398604,
    0.3986139861398614,
    0.3986239862398624,
    0.3986339863398634,
    0.3986439864398644,
    0.3986539865398654,
    0.3986639866398664,
    0.39867398673986737,
    0.3986839868398684,
    0.3986939869398694,
    0.39870398703987037,
    0.3987139871398714,
    0.3987239872398724,
    0.3987339873398734,
    0.3987439874398744,
    0.3987539875398754,
    0.3987639876398764,
    0.3987739877398774,
    0.3987839878398784,
    0.3987939879398794,
    0.3988039880398804,
    0.3988139881398814,
    0.3988239882398824,
    0.39883398833988337,
    0.3988439884398844,
    0.3988539885398854,
    0.39886398863988637,
    0.3988739887398874,
    0.3988839888398884,
    0.39889398893988937,
    0.3989039890398904,
    0.3989139891398914,
    0.3989239892398924,
    0.3989339893398934,
    0.3989439894398944,
    0.3989539895398954,
    0.3989639896398964,
    0.3989739897398974,
    0.3989839898398984,
    0.3989939899398994,
    0.3990039900399004,
    0.3990139901399014,
    0.39902399023990237,
    0.3990339903399034,
    0.3990439904399044,
    0.39905399053990537,
    0.3990639906399064,
    0.3990739907399074,
    0.39908399083990836,
    0.3990939909399094,
    0.3991039910399104,
    0.3991139911399114,
    0.3991239912399124,
    0.3991339913399134,
    0.3991439914399144,
    0.3991539915399154,
    0.3991639916399164,
    0.3991739917399174,
    0.3991839918399184,
    0.3991939919399194,
    0.3992039920399204,
    0.39921399213992137,
    0.3992239922399224,
    0.3992339923399234,
    0.39924399243992437,
    0.3992539925399254,
    0.3992639926399264,
    0.3992739927399274,
    0.3992839928399284,
    0.3992939929399294,
    0.3993039930399304,
    0.3993139931399314,
    0.3993239932399324,
    0.3993339933399334,
    0.3993439934399344,
    0.3993539935399354,
    0.3993639936399364,
    0.39937399373993737,
    0.3993839938399384,
    0.3993939939399394,
    0.39940399403994037,
    0.3994139941399414,
    0.3994239942399424,
    0.39943399433994337,
    0.3994439944399444,
    0.3994539945399454,
    0.3994639946399464,
    0.3994739947399474,
    0.3994839948399484,
    0.3994939949399494,
    0.3995039950399504,
    0.3995139951399514,
    0.3995239952399524,
    0.3995339953399534,
    0.3995439954399544,
    0.3995539955399554,
    0.39956399563995637,
    0.3995739957399574,
    0.3995839958399584,
    0.39959399593995937,
    0.3996039960399604,
    0.3996139961399614,
    0.3996239962399624,
    0.3996339963399634,
    0.3996439964399644,
    0.3996539965399654,
    0.3996639966399664,
    0.3996739967399674,
    0.3996839968399684,
    0.3996939969399694,
    0.3997039970399704,
    0.3997139971399714,
    0.3997239972399724,
    0.3997339973399734,
    0.3997439974399744,
    0.39975399753997537,
    0.3997639976399764,
    0.3997739977399774,
    0.39978399783997837,
    0.3997939979399794,
    0.3998039980399804,
    0.3998139981399814,
    0.3998239982399824,
    0.3998339983399834,
    0.3998439984399844,
    0.3998539985399854,
    0.3998639986399864,
    0.3998739987399874,
    0.3998839988399884,
    0.3998939989399894,
    0.3999039990399904,
    0.39991399913999137,
    0.3999239992399924,
    0.3999339993399934,
    0.39994399943999437,
    0.3999539995399954,
    0.3999639996399964,
    0.39997399973999737,
    0.3999839998399984,
    0.3999939999399994,
    0.4000040000400004,
    0.4000140001400014,
    0.4000240002400024,
    0.4000340003400034,
    0.4000440004400044,
    0.4000540005400054,
    0.4000640006400064,
    0.4000740007400074,
    0.4000840008400084,
    0.4000940009400094,
    0.40010400104001037,
    0.4001140011400114,
    0.4001240012400124,
    0.40013400134001337,
    0.4001440014400144,
    0.4001540015400154,
    0.4001640016400164,
    0.4001740017400174,
    0.4001840018400184,
    0.4001940019400194,
    0.4002040020400204,
    0.4002140021400214,
    0.4002240022400224,
    0.4002340023400234,
    0.4002440024400244,
    0.4002540025400254,
    0.4002640026400264,
    0.4002740027400274,
    0.4002840028400284,
    0.40029400294002937,
    0.4003040030400304,
    0.4003140031400314,
    0.40032400324003237,
    0.4003340033400334,
    0.4003440034400344,
    0.4003540035400354,
    0.4003640036400364,
    0.4003740037400374,
    0.4003840038400384,
    0.4003940039400394,
    0.4004040040400404,
    0.4004140041400414,
    0.4004240042400424,
    0.4004340043400434,
    0.4004440044400444,
    0.40045400454004537,
    0.4004640046400464,
    0.4004740047400474,
    0.40048400484004837,
    0.4004940049400494,
    0.4005040050400504,
    0.4005140051400514,
    0.4005240052400524,
    0.4005340053400534,
    0.4005440054400544,
    0.4005540055400554,
    0.4005640056400564,
    0.4005740057400574,
    0.4005840058400584,
    0.4005940059400594,
    0.4006040060400604,
    0.4006140061400614,
    0.4006240062400624,
    0.4006340063400634,
    0.40064400644006437,
    0.4006540065400654,
    0.4006640066400664,
    0.40067400674006737,
    0.4006840068400684,
    0.4006940069400694,
    0.4007040070400704,
    0.4007140071400714,
    0.4007240072400724,
    0.4007340073400734,
    0.4007440074400744,
    0.4007540075400754,
    0.4007640076400764,
    0.4007740077400774,
    0.4007840078400784,
    0.4007940079400794,
    0.4008040080400804,
    0.4008140081400814,
    0.4008240082400824,
    0.40083400834008337,
    0.4008440084400844,
    0.4008540085400854,
    0.40086400864008637,
    0.4008740087400874,
    0.4008840088400884,
    0.4008940089400894,
    0.4009040090400904,
    0.4009140091400914,
    0.4009240092400924,
    0.4009340093400934,
    0.4009440094400944,
    0.4009540095400954,
    0.4009640096400964,
    0.4009740097400974,
    0.4009840098400984,
    0.40099400994009937,
    0.4010040100401004,
    0.4010140101401014,
    0.40102401024010237,
    0.4010340103401034,
    0.4010440104401044,
    0.4010540105401054,
    0.4010640106401064,
    0.4010740107401074,
    0.4010840108401084,
    0.4010940109401094,
    0.4011040110401104,
    0.4011140111401114,
    0.4011240112401124,
    0.4011340113401134,
    0.4011440114401144,
    0.4011540115401154,
    0.4011640116401164,
    0.4011740117401174,
    0.40118401184011837,
    0.4011940119401194,
    0.4012040120401204,
    0.40121401214012137,
    0.4012240122401224,
    0.4012340123401234,
    0.4012440124401244,
    0.4012540125401254,
    0.4012640126401264,
    0.4012740127401274,
    0.4012840128401284,
    0.4012940129401294,
    0.4013040130401304,
    0.4013140131401314,
    0.4013240132401324,
    0.4013340133401334,
    0.4013440134401344,
    0.4013540135401354,
    0.4013640136401364,
    0.40137401374013737,
    0.4013840138401384,
    0.4013940139401394,
    0.4014040140401404,
    0.4014140141401414,
    0.4014240142401424,
    0.4014340143401434,
    0.4014440144401444,
    0.4014540145401454,
    0.4014640146401464,
    0.4014740147401474,
    0.4014840148401484,
    0.4014940149401494,
    0.4015040150401504,
    0.4015140151401514,
    0.4015240152401524,
    0.40153401534015337,
    0.4015440154401544,
    0.4015540155401554,
    0.40156401564015637,
    0.4015740157401574,
    0.4015840158401584,
    0.4015940159401594,
    0.4016040160401604,
    0.4016140161401614,
    0.4016240162401624,
    0.4016340163401634,
    0.4016440164401644,
    0.4016540165401654,
    0.4016640166401664,
    0.4016740167401674,
    0.4016840168401684,
    0.4016940169401694,
    0.4017040170401704,
    0.4017140171401714,
    0.40172401724017237,
    0.4017340173401734,
    0.4017440174401744,
    0.40175401754017537,
    0.4017640176401764,
    0.4017740177401774,
    0.4017840178401784,
    0.4017940179401794,
    0.4018040180401804,
    0.4018140181401814,
    0.4018240182401824,
    0.4018340183401834,
    0.4018440184401844,
    0.4018540185401854,
    0.4018640186401864,
    0.4018740187401874,
    0.4018840188401884,
    0.4018940189401894,
    0.4019040190401904,
    0.40191401914019137,
    0.4019240192401924,
    0.4019340193401934,
    0.4019440194401944,
    0.4019540195401954,
    0.4019640196401964,
    0.4019740197401974,
    0.4019840198401984,
    0.4019940199401994,
    0.4020040200402004,
    0.4020140201402014,
    0.4020240202402024,
    0.4020340203402034,
    0.4020440204402044,
    0.4020540205402054,
    0.4020640206402064,
    0.40207402074020737,
    0.4020840208402084,
    0.4020940209402094,
    0.40210402104021037,
    0.4021140211402114,
    0.4021240212402124,
    0.4021340213402134,
    0.4021440214402144,
    0.4021540215402154,
    0.4021640216402164,
    0.4021740217402174,
    0.4021840218402184,
    0.4021940219402194,
    0.4022040220402204,
    0.4022140221402214,
    0.4022240222402224,
    0.4022340223402234,
    0.4022440224402244,
    0.4022540225402254,
    0.40226402264022637,
    0.4022740227402274,
    0.4022840228402284,
    0.40229402294022937,
    0.4023040230402304,
    0.4023140231402314,
    0.4023240232402324,
    0.4023340233402334,
    0.4023440234402344,
    0.4023540235402354,
    0.4023640236402364,
    0.4023740237402374,
    0.4023840238402384,
    0.4023940239402394,
    0.4024040240402404,
    0.4024140241402414,
    0.4024240242402424,
    0.4024340243402434,
    0.4024440244402444,
    0.40245402454024537,
    0.4024640246402464,
    0.4024740247402474,
    0.4024840248402484,
    0.4024940249402494,
    0.4025040250402504,
    0.4025140251402514,
    0.4025240252402524,
    0.4025340253402534,
    0.4025440254402544,
    0.4025540255402554,
    0.4025640256402564,
    0.4025740257402574,
    0.4025840258402584,
    0.4025940259402594,
    0.4026040260402604,
    0.40261402614026137,
    0.4026240262402624,
    0.4026340263402634,
    0.40264402644026437,
    0.4026540265402654,
    0.4026640266402664,
    0.4026740267402674,
    0.4026840268402684,
    0.4026940269402694,
    0.4027040270402704,
    0.4027140271402714,
    0.4027240272402724,
    0.4027340273402734,
    0.4027440274402744,
    0.4027540275402754,
    0.4027640276402764,
    0.4027740277402774,
    0.4027840278402784,
    0.4027940279402794,
    0.40280402804028037,
    0.4028140281402814,
    0.4028240282402824,
    0.4028340283402834,
