/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * license agreements; and to You under the Apache License, version 2.0:
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * This file is part of the Apache Pekko project, which was derived from Akka.
 */

/*
 * Copyright (C) 2014 - 2016 Softwaremill <https://softwaremill.com>
 * Copyright (C) 2016 - 2020 Lightbend Inc. <https://www.lightbend.com>
 */

package org.apache.pekko.kafka.internal
import org.apache.pekko
import pekko.annotation.InternalApi
import pekko.stream.stage.{ AsyncCallback, GraphStageLogic }
import org.apache.kafka.clients.consumer.ConsumerRecord
import org.apache.kafka.common.TopicPartition

/**
 * A buffer of messages provided by the [[KafkaConsumerActor]] for a Source Logic. When partitions are rebalanced
 * away from this Source Logic preemptively filter out messages for those partitions.
 *
 * NOTE: Due to the asynchronous nature of Apache Pekko Streams, it's not possible to guarantee that a message has not
 * already been sent downstream for a revoked partition before the rebalance handler invokes
 * `filterRevokedPartitionsCB`. The best we can do is filter as many messages as possible to reduce the amount of
 * duplicate messages sent downstream.
 */
@InternalApi
private[kafka] trait SourceLogicBuffer[K, V, Msg] {
  self: GraphStageLogic with StageIdLogging =>

  protected var buffer: Iterator[ConsumerRecord[K, V]] = Iterator.empty

  protected val filterRevokedPartitionsCB: AsyncCallback[Set[TopicPartition]] =
    getAsyncCallback[Set[TopicPartition]](filterRevokedPartitions)

  private def filterRevokedPartitions(topicPartitions: Set[TopicPartition]): Unit = {
    if (topicPartitions.nonEmpty) {
      log.debug("filtering out messages from revoked partitions {}", topicPartitions)
      // as buffer is an Iterator the filtering will be applied during `pump`
      buffer = buffer.filterNot { record =>
        val tp = new TopicPartition(record.topic, record.partition)
        topicPartitions.contains(tp)
      }
    }
  }
}
