/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.hdds.scm.cli.datanode;

import static org.apache.hadoop.hdds.scm.cli.datanode.DecommissionSubCommand.showErrors;

import java.io.IOException;
import java.util.List;
import org.apache.hadoop.hdds.cli.HddsVersionProvider;
import org.apache.hadoop.hdds.scm.DatanodeAdminError;
import org.apache.hadoop.hdds.scm.cli.ScmSubcommand;
import org.apache.hadoop.hdds.scm.client.ScmClient;
import picocli.CommandLine;
import picocli.CommandLine.Command;

/**
 * Place one or more datanodes into Maintenance Mode.
 */
@Command(
    name = "maintenance",
    description = "Put a datanode into Maintenance Mode",
    mixinStandardHelpOptions = true,
    versionProvider = HddsVersionProvider.class)
public class MaintenanceSubCommand extends ScmSubcommand {

  @CommandLine.Mixin
  private HostNameParameters hostNameParams;

  @CommandLine.Option(names = {"--end"},
      description = "Automatically end maintenance after the given hours. " +
          "By default, maintenance must be ended manually.")
  private int endInHours = 0;

  @CommandLine.Option(names = { "--force" },
      defaultValue = "false",
      description = "Forcefully try to decommission the datanode(s)")
  private boolean force;


  @Override
  public void execute(ScmClient scmClient) throws IOException {
    List<String> hosts = hostNameParams.getHostNames();
    List<DatanodeAdminError> errors =
        scmClient.startMaintenanceNodes(hosts, endInHours, force);
    System.out.println("Entering maintenance mode on datanode(s):\n" +
        String.join("\n", hosts));
    showErrors(errors, "Some nodes could not start the maintenance workflow");
  }
}
