/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.vault.blob;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.InputStream;
import java.time.Clock;
import java.time.ZonedDateTime;
import java.util.Optional;
import javax.inject.Inject;
import org.apache.james.blob.api.BlobId;
import org.apache.james.blob.api.BlobStore;
import org.apache.james.blob.api.BlobStoreDAO;
import org.apache.james.blob.api.BucketName;
import org.apache.james.blob.api.ObjectNotFoundException;
import org.apache.james.core.Username;
import org.apache.james.mailbox.model.MessageId;
import org.apache.james.metrics.api.MetricFactory;
import org.apache.james.task.Task;
import org.apache.james.vault.DeletedMessage;
import org.apache.james.vault.DeletedMessageContentNotFoundException;
import org.apache.james.vault.DeletedMessageVault;
import org.apache.james.vault.VaultConfiguration;
import org.apache.james.vault.blob.BlobStoreVaultGarbageCollectionTask;
import org.apache.james.vault.blob.BucketNameGenerator;
import org.apache.james.vault.metadata.DeletedMessageMetadataVault;
import org.apache.james.vault.metadata.DeletedMessageWithStorageInformation;
import org.apache.james.vault.metadata.StorageInformation;
import org.apache.james.vault.search.Query;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class BlobStoreDeletedMessageVault
implements DeletedMessageVault {
    private static final Logger LOGGER = LoggerFactory.getLogger(BlobStoreDeletedMessageVault.class);
    private static final String BLOBSTORE_DELETED_MESSAGE_VAULT_METRIC = "deletedMessageVault:blobStore:";
    static final String APPEND_METRIC_NAME = "deletedMessageVault:blobStore:append";
    static final String LOAD_MIME_MESSAGE_METRIC_NAME = "deletedMessageVault:blobStore:loadMimeMessage";
    static final String SEARCH_METRIC_NAME = "deletedMessageVault:blobStore:search";
    static final String DELETE_METRIC_NAME = "deletedMessageVault:blobStore:delete";
    static final String DELETE_EXPIRED_MESSAGES_METRIC_NAME = "deletedMessageVault:blobStore:deleteExpiredMessages";
    private final MetricFactory metricFactory;
    private final DeletedMessageMetadataVault messageMetadataVault;
    private final BlobStore blobStore;
    private final BlobStoreDAO blobStoreDAO;
    private final BucketNameGenerator nameGenerator;
    private final Clock clock;
    private final VaultConfiguration vaultConfiguration;
    private final BlobStoreVaultGarbageCollectionTask.Factory taskFactory;

    @Inject
    public BlobStoreDeletedMessageVault(MetricFactory metricFactory, DeletedMessageMetadataVault messageMetadataVault, BlobStore blobStore, BlobStoreDAO blobStoreDAO, BucketNameGenerator nameGenerator, Clock clock, VaultConfiguration vaultConfiguration) {
        this.metricFactory = metricFactory;
        this.messageMetadataVault = messageMetadataVault;
        this.blobStore = blobStore;
        this.blobStoreDAO = blobStoreDAO;
        this.nameGenerator = nameGenerator;
        this.clock = clock;
        this.vaultConfiguration = vaultConfiguration;
        this.taskFactory = new BlobStoreVaultGarbageCollectionTask.Factory(this);
    }

    @Override
    public Publisher<Void> append(DeletedMessage deletedMessage, InputStream mimeMessage) {
        Preconditions.checkNotNull((Object)deletedMessage);
        Preconditions.checkNotNull((Object)mimeMessage);
        BucketName bucketName = this.nameGenerator.currentBucket();
        return this.metricFactory.decoratePublisherWithTimerMetric(APPEND_METRIC_NAME, this.appendMessage(deletedMessage, mimeMessage, bucketName));
    }

    private Mono<Void> appendMessage(DeletedMessage deletedMessage, InputStream mimeMessage, BucketName bucketName) {
        return Mono.from((Publisher)this.blobStore.save(bucketName, mimeMessage, BlobStore.StoragePolicy.LOW_COST)).map(blobId -> StorageInformation.builder().bucketName(bucketName).blobId((BlobId)blobId)).map(storageInformation -> new DeletedMessageWithStorageInformation(deletedMessage, (StorageInformation)storageInformation)).flatMap(message -> Mono.from(this.messageMetadataVault.store((DeletedMessageWithStorageInformation)message))).then();
    }

    @Override
    public Publisher<InputStream> loadMimeMessage(Username username, MessageId messageId) {
        Preconditions.checkNotNull((Object)username);
        Preconditions.checkNotNull((Object)messageId);
        return this.metricFactory.decoratePublisherWithTimerMetric(LOAD_MIME_MESSAGE_METRIC_NAME, (Publisher)Mono.from(this.messageMetadataVault.retrieveStorageInformation(username, messageId)).flatMap(storageInformation -> this.loadMimeMessage((StorageInformation)storageInformation, username, messageId)));
    }

    private Mono<InputStream> loadMimeMessage(StorageInformation storageInformation, Username username, MessageId messageId) {
        return Mono.fromSupplier(() -> this.blobStore.read(storageInformation.getBucketName(), storageInformation.getBlobId(), BlobStore.StoragePolicy.LOW_COST)).onErrorResume(ObjectNotFoundException.class, ex -> Mono.error((Throwable)new DeletedMessageContentNotFoundException(username, messageId)));
    }

    @Override
    public Publisher<DeletedMessage> search(Username username, Query query) {
        Preconditions.checkNotNull((Object)username);
        Preconditions.checkNotNull((Object)query);
        return this.metricFactory.decoratePublisherWithTimerMetric(SEARCH_METRIC_NAME, this.searchOn(username, query));
    }

    private Flux<DeletedMessage> searchOn(Username username, Query query) {
        return Flux.from(this.messageMetadataVault.listRelatedBuckets()).concatMap(bucketName -> this.messageMetadataVault.listMessages((BucketName)bucketName, username)).map(DeletedMessageWithStorageInformation::getDeletedMessage).filter(query.toPredicate());
    }

    @Override
    public Publisher<Void> delete(Username username, MessageId messageId) {
        Preconditions.checkNotNull((Object)username);
        Preconditions.checkNotNull((Object)messageId);
        return this.metricFactory.decoratePublisherWithTimerMetric(DELETE_METRIC_NAME, this.deleteMessage(username, messageId));
    }

    private Mono<Void> deleteMessage(Username username, MessageId messageId) {
        return Mono.from(this.messageMetadataVault.retrieveStorageInformation(username, messageId)).flatMap(storageInformation -> Mono.from(this.messageMetadataVault.remove(storageInformation.getBucketName(), username, messageId)).thenReturn(storageInformation)).flatMap(storageInformation -> Mono.from((Publisher)this.blobStoreDAO.delete(storageInformation.getBucketName(), storageInformation.getBlobId()))).subscribeOn(Schedulers.elastic());
    }

    @Override
    public Task deleteExpiredMessagesTask() {
        return this.taskFactory.create();
    }

    Flux<BucketName> deleteExpiredMessages(ZonedDateTime beginningOfRetentionPeriod) {
        return Flux.from((Publisher)this.metricFactory.decoratePublisherWithTimerMetric(DELETE_EXPIRED_MESSAGES_METRIC_NAME, (Publisher)this.retentionQualifiedBuckets(beginningOfRetentionPeriod).flatMap(bucketName -> this.deleteBucketData((BucketName)bucketName).then(Mono.just((Object)bucketName)), 16)));
    }

    ZonedDateTime getBeginningOfRetentionPeriod() {
        ZonedDateTime now = ZonedDateTime.now(this.clock);
        return now.minus(this.vaultConfiguration.getRetentionPeriod());
    }

    @VisibleForTesting
    Flux<BucketName> retentionQualifiedBuckets(ZonedDateTime beginningOfRetentionPeriod) {
        return Flux.from(this.messageMetadataVault.listRelatedBuckets()).filter(bucketName -> this.isFullyExpired(beginningOfRetentionPeriod, (BucketName)bucketName));
    }

    private boolean isFullyExpired(ZonedDateTime beginningOfRetentionPeriod, BucketName bucketName) {
        Optional<ZonedDateTime> maybeEndDate = this.nameGenerator.bucketEndTime(bucketName);
        if (!maybeEndDate.isPresent()) {
            LOGGER.error("Pattern used for bucketName used in deletedMessageVault is invalid and end date cannot be parsed {}", (Object)bucketName);
        }
        return maybeEndDate.map(endDate -> endDate.isBefore(beginningOfRetentionPeriod)).orElse(false);
    }

    private Mono<Void> deleteBucketData(BucketName bucketName) {
        return Mono.from((Publisher)this.blobStore.deleteBucket(bucketName)).then(Mono.from(this.messageMetadataVault.removeMetadataRelatedToBucket(bucketName)));
    }
}

