/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.fsdataset.impl;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.hdfs.ExtendedBlockId;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsDatasetUtil;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.MappableBlock;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.MappableBlockLoader;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Unstable
public final class PmemVolumeManager {
    private static final Logger LOG = LoggerFactory.getLogger(PmemVolumeManager.class);
    public static final String CACHE_DIR = "hdfs_pmem_cache";
    private static PmemVolumeManager pmemVolumeManager = null;
    private final ArrayList<String> pmemVolumes = new ArrayList();
    private final Map<ExtendedBlockId, Byte> blockKeyToVolume = new ConcurrentHashMap<ExtendedBlockId, Byte>();
    private final List<UsedBytesCount> usedBytesCounts = new ArrayList<UsedBytesCount>();
    private boolean cacheRecoveryEnabled;
    private long cacheCapacity;
    private static long maxBytesPerPmem = -1L;
    private int count = 0;
    private byte nextIndex = 0;

    private PmemVolumeManager(String[] pmemVolumesConfig, boolean cacheRecoveryEnabled) throws IOException {
        if (pmemVolumesConfig == null || pmemVolumesConfig.length == 0) {
            throw new IOException("The persistent memory volume, dfs.datanode.pmem.cache.dirs is not configured!");
        }
        this.cacheRecoveryEnabled = cacheRecoveryEnabled;
        this.loadVolumes(pmemVolumesConfig);
        this.cacheCapacity = 0L;
        for (UsedBytesCount counter : this.usedBytesCounts) {
            this.cacheCapacity += counter.getMaxBytes();
        }
    }

    public static synchronized void init(String[] pmemVolumesConfig, boolean cacheRecoveryEnabled) throws IOException {
        if (pmemVolumeManager == null) {
            pmemVolumeManager = new PmemVolumeManager(pmemVolumesConfig, cacheRecoveryEnabled);
        }
    }

    public static PmemVolumeManager getInstance() {
        if (pmemVolumeManager == null) {
            throw new RuntimeException("The pmemVolumeManager should be instantiated!");
        }
        return pmemVolumeManager;
    }

    @VisibleForTesting
    public static void reset() {
        pmemVolumeManager = null;
    }

    @VisibleForTesting
    public static void setMaxBytes(long maxBytes) {
        maxBytesPerPmem = maxBytes;
    }

    public long getCacheUsed() {
        long usedBytes = 0L;
        for (UsedBytesCount counter : this.usedBytesCounts) {
            usedBytes += counter.getUsedBytes();
        }
        return usedBytes;
    }

    public long getCacheCapacity() {
        return this.cacheCapacity;
    }

    synchronized long reserve(ExtendedBlockId key, long bytesCount) {
        try {
            byte index = this.chooseVolume(bytesCount);
            long usedBytes = this.usedBytesCounts.get(index).reserve(bytesCount);
            if (usedBytes > 0L) {
                this.blockKeyToVolume.put(key, index);
            }
            return usedBytes;
        }
        catch (IOException e) {
            LOG.warn(e.getMessage());
            return -1L;
        }
    }

    long release(ExtendedBlockId key, long bytesCount) {
        Byte index = this.blockKeyToVolume.remove(key);
        return this.usedBytesCounts.get(index.byteValue()).release(bytesCount);
    }

    private void loadVolumes(String[] volumes) throws IOException {
        for (int n = 0; n < volumes.length; n = (int)((byte)(n + 1))) {
            try {
                File pmemDir = new File(volumes[n]);
                File realPmemDir = PmemVolumeManager.verifyIfValidPmemVolume(pmemDir);
                if (!this.cacheRecoveryEnabled) {
                    this.cleanup(realPmemDir);
                }
                this.pmemVolumes.add(realPmemDir.getPath());
                long maxBytes = maxBytesPerPmem == -1L ? realPmemDir.getUsableSpace() : maxBytesPerPmem;
                UsedBytesCount usedBytesCount = new UsedBytesCount(maxBytes);
                this.usedBytesCounts.add(usedBytesCount);
                LOG.info("Added persistent memory - {} with size={}", (Object)volumes[n], (Object)maxBytes);
                continue;
            }
            catch (IllegalArgumentException e) {
                LOG.error("Failed to parse persistent memory volume " + volumes[n], (Throwable)e);
                continue;
            }
            catch (IOException e) {
                LOG.error("Bad persistent memory volume: " + volumes[n], (Throwable)e);
            }
        }
        this.count = this.pmemVolumes.size();
        if (this.count == 0) {
            throw new IOException("At least one valid persistent memory volume is required!");
        }
    }

    void cleanup(File realPmemDir) {
        try {
            FileUtils.cleanDirectory((File)realPmemDir);
        }
        catch (IOException e) {
            LOG.error("Failed to clean up " + realPmemDir.getPath(), (Throwable)e);
        }
    }

    void cleanup() {
        for (String pmemVolume : this.pmemVolumes) {
            this.cleanup(new File(pmemVolume));
        }
    }

    public Map<ExtendedBlockId, MappableBlock> recoverCache(String bpid, MappableBlockLoader cacheLoader) throws IOException {
        ConcurrentHashMap<ExtendedBlockId, MappableBlock> keyToMappableBlock = new ConcurrentHashMap<ExtendedBlockId, MappableBlock>();
        for (byte volumeIndex = 0; volumeIndex < this.pmemVolumes.size(); volumeIndex = (byte)(volumeIndex + 1)) {
            long maxBytes = this.usedBytesCounts.get(volumeIndex).getMaxBytes();
            long usedBytes = 0L;
            File cacheDir = new File(this.pmemVolumes.get(volumeIndex), bpid);
            Collection cachedFileList = FileUtils.listFiles((File)cacheDir, (IOFileFilter)TrueFileFilter.INSTANCE, (IOFileFilter)TrueFileFilter.INSTANCE);
            for (File cachedFile : cachedFileList) {
                MappableBlock mappableBlock = cacheLoader.getRecoveredMappableBlock(cachedFile, bpid, volumeIndex);
                ExtendedBlockId key = mappableBlock.getKey();
                keyToMappableBlock.put(key, mappableBlock);
                usedBytes += cachedFile.length();
            }
            this.usedBytesCounts.get(volumeIndex).setMaxBytes(maxBytes + usedBytes);
            this.cacheCapacity += usedBytes;
            this.usedBytesCounts.get(volumeIndex).reserve(usedBytes);
        }
        return keyToMappableBlock;
    }

    public void recoverBlockKeyToVolume(ExtendedBlockId key, byte volumeIndex) {
        this.blockKeyToVolume.put(key, volumeIndex);
    }

    @VisibleForTesting
    static File verifyIfValidPmemVolume(File pmemDir) throws IOException {
        if (!pmemDir.exists()) {
            String message = pmemDir + " does not exist";
            throw new IOException(message);
        }
        if (!pmemDir.isDirectory()) {
            String message = pmemDir + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        File realPmemDir = new File(PmemVolumeManager.getRealPmemDir(pmemDir.getPath()));
        if (!realPmemDir.exists() && !realPmemDir.mkdir()) {
            throw new IOException("Failed to create " + realPmemDir.getPath());
        }
        String uuidStr = UUID.randomUUID().toString();
        String testFilePath = realPmemDir.getPath() + "/.verify.pmem." + uuidStr;
        byte[] contents = uuidStr.getBytes(StandardCharsets.UTF_8);
        RandomAccessFile testFile = null;
        MappedByteBuffer out = null;
        try {
            testFile = new RandomAccessFile(testFilePath, "rw");
            out = testFile.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, contents.length);
            if (out == null) {
                throw new IOException("Failed to map the test file under " + realPmemDir);
            }
            out.put(contents);
            out.force();
            File file = realPmemDir;
            return file;
        }
        catch (IOException e) {
            throw new IOException("Exception while writing data to persistent storage dir: " + realPmemDir, e);
        }
        finally {
            if (out != null) {
                out.clear();
            }
            if (testFile != null) {
                IOUtils.closeStream((Closeable)testFile);
                NativeIO.POSIX.munmap((MappedByteBuffer)out);
                try {
                    FsDatasetUtil.deleteMappedFile(testFilePath);
                }
                catch (IOException e) {
                    LOG.warn("Failed to delete test file " + testFilePath + " from persistent memory", (Throwable)e);
                }
            }
        }
    }

    public void createBlockPoolDir(String bpid) throws IOException {
        for (String volume : this.pmemVolumes) {
            File cacheDir = new File(volume, bpid);
            if (cacheDir.exists() || cacheDir.mkdir()) continue;
            throw new IOException("Failed to create " + cacheDir.getPath());
        }
    }

    public static String getRealPmemDir(String rawPmemDir) {
        return new File(rawPmemDir, CACHE_DIR).getAbsolutePath();
    }

    synchronized Byte chooseVolume(long bytesCount) throws IOException {
        if (this.count == 0) {
            throw new IOException("No usable persistent memory is found");
        }
        int k = 0;
        long maxAvailableSpace = 0L;
        while (k++ != this.count) {
            byte index;
            if (this.nextIndex == this.count) {
                this.nextIndex = 0;
            }
            this.nextIndex = (byte)(this.nextIndex + 1);
            long availableBytes = this.usedBytesCounts.get(index).getAvailableBytes();
            if (availableBytes >= bytesCount) {
                return index;
            }
            if (availableBytes <= maxAvailableSpace) continue;
            maxAvailableSpace = availableBytes;
        }
        throw new IOException("There is no enough persistent memory space for caching. The current max available space is " + maxAvailableSpace + ", but " + bytesCount + "is required.");
    }

    @VisibleForTesting
    String getVolumeByIndex(Byte index) {
        return this.pmemVolumes.get(index.byteValue());
    }

    ArrayList<String> getVolumes() {
        return this.pmemVolumes;
    }

    public String idToCacheFileName(ExtendedBlockId key) {
        return String.valueOf(key.getBlockId());
    }

    public String idToCacheFilePath(Byte volumeIndex, ExtendedBlockId key) throws IOException {
        String subDir;
        String cacheSubdirPrefix = "subdir";
        long blockId = key.getBlockId();
        String bpid = key.getBlockPoolId();
        int d1 = (int)(blockId >> 16 & 0x1FL);
        int d2 = (int)(blockId >> 8 & 0x1FL);
        String parentDir = this.pmemVolumes.get(volumeIndex.byteValue()) + "/" + bpid;
        File filePath = new File(parentDir, subDir = "subdir" + d1 + "/" + "subdir" + d2);
        if (!filePath.exists() && !filePath.mkdirs()) {
            throw new IOException("Failed to create " + filePath.getPath());
        }
        return filePath.getAbsolutePath() + "/" + this.idToCacheFileName(key);
    }

    public String getCachePath(ExtendedBlockId key) throws IOException {
        Byte volumeIndex = this.blockKeyToVolume.get(key);
        if (volumeIndex == null) {
            return null;
        }
        return this.idToCacheFilePath(volumeIndex, key);
    }

    @VisibleForTesting
    Map<ExtendedBlockId, Byte> getBlockKeyToVolume() {
        return this.blockKeyToVolume;
    }

    private static class UsedBytesCount {
        private long maxBytes;
        private final AtomicLong usedBytes = new AtomicLong(0L);

        UsedBytesCount(long maxBytes) {
            this.maxBytes = maxBytes;
        }

        long reserve(long bytesCount) {
            long next;
            long cur;
            do {
                if ((next = (cur = this.usedBytes.get()) + bytesCount) <= this.maxBytes) continue;
                return -1L;
            } while (!this.usedBytes.compareAndSet(cur, next));
            return next;
        }

        long release(long bytesCount) {
            return this.usedBytes.addAndGet(-bytesCount);
        }

        long getUsedBytes() {
            return this.usedBytes.get();
        }

        long getMaxBytes() {
            return this.maxBytes;
        }

        long getAvailableBytes() {
            return this.maxBytes - this.usedBytes.get();
        }

        void setMaxBytes(long maxBytes) {
            this.maxBytes = maxBytes;
        }
    }
}

