/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.eventmesh.connector.canal;

import org.apache.eventmesh.common.remote.job.SyncConsistency;
import org.apache.eventmesh.common.remote.job.SyncMode;
import org.apache.eventmesh.connector.canal.model.EventColumn;
import org.apache.eventmesh.connector.canal.model.EventType;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import lombok.Data;

@Data
public class CanalConnectRecord implements Serializable {

    private static final long serialVersionUID = 1L;

    private String schemaName;

    private String tableName;

    // mysql instance gtid range
    private String gtid;

    private String currentGtid;

    /**
     * The business type of the changed data (I/U/D/C/A/E), consistent with the EventType defined in EntryProtocol in canal.
     */
    private EventType eventType;

    /**
     * The business time of the changed data.
     */
    private long executeTime;

    /**
     * The primary key value before the change, if it is insert/delete, the primary key value before and after the change is the same.
     */
    private List<EventColumn> oldKeys = new ArrayList<EventColumn>();

    /**
     * The primary key value after the change, if it is insert/delete, the primary key value before and after the change is the same.
     */
    private List<EventColumn> keys = new ArrayList<EventColumn>();

    /**
     * Other fields that are not primary keys
     */
    private List<EventColumn> columns = new ArrayList<EventColumn>();

    // ====================== Additional properties of the data during the running process =============================
    /**
     * The expected size, based on the estimation of the binlog event
     */
    private long size = 1024;

    /**
     * The id of the synchronization mapping relationship
     */
    private long pairId = -1;

    /**
     * When eventType = CREATE/ALTER/ERASE, it is the corresponding SQL statement, other situations are dynamically generated INSERT/UPDATE/DELETE sql
     */
    private String sql;

    /**
     * The schemaName of ddl/query, there will be cross-database ddl, need to keep the current schemaName of executing ddl
     */
    private String ddlSchemaName;

    /**
     * Custom synchronization mode, allows to override the default pipeline parameter, such as for remedial data synchronization
     */
    private SyncMode syncMode;

    /**
     * Custom synchronization consistency, allows to override the default pipeline parameter,
     * such as forcing the database to be queried for field groups
     */
    private SyncConsistency syncConsistency;

    /**
     * Whether it is remedy data, such as data automatically generated by loopback remedy, or manual correction data produced by freedom
     */
    private boolean remedy = false;

    /**
     * Generate the corresponding hint content
     */
    private String hint;

    /**
     * Whether to ignore the schema when generating SQL, such as for tddl/drds, need to ignore the schema
     */
    private boolean withoutSchema = false;

    private String journalName;

    private long binLogOffset;

    public CanalConnectRecord() {
        super();
    }

    // ======================== helper method =================

    /**
     * Return all fields to be changed
     */
    public List<EventColumn> getUpdatedColumns() {
        List<EventColumn> columns = new ArrayList<EventColumn>();
        for (EventColumn column : this.columns) {
            if (column.isUpdate()) {
                columns.add(column);
            }
        }

        return columns;
    }

    /**
     * Return all changed primary key fields
     */
    public List<EventColumn> getUpdatedKeys() {
        List<EventColumn> columns = new ArrayList<EventColumn>();
        for (EventColumn column : this.keys) {
            if (column.isUpdate()) {
                columns.add(column);
            }
        }

        return columns;
    }

    private List<EventColumn> cloneColumn(List<EventColumn> columns) {
        if (columns == null) {
            return null;
        }

        List<EventColumn> cloneColumns = new ArrayList<EventColumn>();
        for (EventColumn column : columns) {
            cloneColumns.add(column.clone());
        }

        return cloneColumns;
    }

    public CanalConnectRecord clone() {
        CanalConnectRecord record = new CanalConnectRecord();
        record.setTableName(tableName);
        record.setSchemaName(schemaName);
        record.setDdlSchemaName(ddlSchemaName);
        record.setEventType(eventType);
        record.setExecuteTime(executeTime);
        record.setKeys(cloneColumn(keys));
        record.setColumns(cloneColumn(columns));
        record.setOldKeys(cloneColumn(oldKeys));
        record.setSize(size);
        record.setPairId(pairId);
        record.setSql(sql);
        record.setSyncMode(syncMode);
        record.setSyncConsistency(syncConsistency);
        record.setRemedy(remedy);
        record.setHint(hint);
        record.setWithoutSchema(withoutSchema);
        return record;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((columns == null) ? 0 : columns.hashCode());
        result = prime * result + ((eventType == null) ? 0 : eventType.hashCode());
        result = prime * result + (int) (executeTime ^ (executeTime >>> 32));
        result = prime * result + ((keys == null) ? 0 : keys.hashCode());
        result = prime * result + ((oldKeys == null) ? 0 : oldKeys.hashCode());
        result = prime * result + (int) (pairId ^ (pairId >>> 32));
        result = prime * result + ((schemaName == null) ? 0 : schemaName.hashCode());
        result = prime * result + ((tableName == null) ? 0 : tableName.hashCode());
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        CanalConnectRecord other = (CanalConnectRecord) obj;
        if (columns == null) {
            if (other.columns != null) {
                return false;
            }
        } else if (!columns.equals(other.columns)) {
            return false;
        }
        if (eventType != other.eventType) {
            return false;
        }
        if (executeTime != other.executeTime) {
            return false;
        }
        if (keys == null) {
            if (other.keys != null) {
                return false;
            }
        } else if (!keys.equals(other.keys)) {
            return false;
        }
        if (oldKeys == null) {
            if (other.oldKeys != null) {
                return false;
            }
        } else if (!oldKeys.equals(other.oldKeys)) {
            return false;
        }
        if (pairId != other.pairId) {
            return false;
        }
        if (schemaName == null) {
            if (other.schemaName != null) {
                return false;
            }
        } else if (!schemaName.equals(other.schemaName)) {
            return false;
        }
        if (tableName == null) {
            if (other.tableName != null) {
                return false;
            }
        } else if (!tableName.equals(other.tableName)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "CanalConnectRecord{"
            + "tableName='" + tableName + '\''
            + ", schemaName='" + schemaName + '\''
            + ", eventType=" + eventType
            + ", executeTime=" + executeTime
            + ", oldKeys=" + oldKeys
            + ", keys=" + keys
            + ", columns=" + columns
            + ", size=" + size
            + ", pairId=" + pairId
            + ", sql='" + sql + '\''
            + ", ddlSchemaName='" + ddlSchemaName + '\''
            + ", syncMode=" + syncMode
            + ", syncConsistency=" + syncConsistency
            + ", remedy=" + remedy
            + ", hint='" + hint + '\''
            + ", withoutSchema=" + withoutSchema
            + '}';
    }
}
