/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.archivers.zip;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.zip.CRC32;
import org.apache.commons.compress.AbstractTest;
import org.apache.commons.compress.archivers.zip.UnicodePathExtraField;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipEncoding;
import org.apache.commons.compress.archivers.zip.ZipEncodingHelper;
import org.apache.commons.compress.archivers.zip.ZipExtraField;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

class UTF8ZipFilesTest
extends AbstractTest {
    private static final String CP437 = "cp437";
    private static final String ASCII_TXT = "ascii.txt";
    private static final String EURO_FOR_DOLLAR_TXT = "\u20ac_for_Dollar.txt";
    private static final String OIL_BARREL_TXT = "\u00d6lf\u00e4sser.txt";

    UTF8ZipFilesTest() {
    }

    private static void assertRawNameOfAcsiiTxt(ZipArchiveEntry ze) {
        byte[] b = ze.getRawName();
        Assertions.assertNotNull((Object)b);
        int len = ASCII_TXT.length();
        Assertions.assertEquals((int)len, (int)b.length);
        for (int i = 0; i < len; ++i) {
            Assertions.assertEquals((byte)((byte)ASCII_TXT.charAt(i)), (byte)b[i], (String)("Byte " + i));
        }
        Assertions.assertNotSame((Object)b, (Object)ze.getRawName());
    }

    private static void assertUnicodeName(ZipArchiveEntry ze, String expectedName, String encoding) throws IOException {
        if (!expectedName.equals(ze.getName())) {
            UnicodePathExtraField ucpf = UTF8ZipFilesTest.findUniCodePath(ze);
            Assertions.assertNotNull((Object)ucpf);
            ZipEncoding enc = ZipEncodingHelper.getZipEncoding((String)encoding);
            ByteBuffer ne = enc.encode(ze.getName());
            CRC32 crc = new CRC32();
            crc.update(ne.array(), ne.arrayOffset(), ne.limit() - ne.position());
            Assertions.assertEquals((long)crc.getValue(), (long)ucpf.getNameCRC32());
            Assertions.assertEquals((Object)expectedName, (Object)new String(ucpf.getUnicodeName(), StandardCharsets.UTF_8));
        }
    }

    private static void createTestFile(File file, String encoding, boolean withEFS, boolean withExplicitUnicodeExtra) throws IOException {
        ZipEncoding zipEncoding = ZipEncodingHelper.getZipEncoding((String)encoding);
        try (ZipArchiveOutputStream zos = new ZipArchiveOutputStream(file);){
            ByteBuffer en;
            zos.setEncoding(encoding);
            zos.setUseLanguageEncodingFlag(withEFS);
            zos.setCreateUnicodeExtraFields(withExplicitUnicodeExtra ? ZipArchiveOutputStream.UnicodeExtraFieldPolicy.NEVER : ZipArchiveOutputStream.UnicodeExtraFieldPolicy.ALWAYS);
            ZipArchiveEntry ze = new ZipArchiveEntry(OIL_BARREL_TXT);
            if (withExplicitUnicodeExtra && !zipEncoding.canEncode(ze.getName())) {
                en = zipEncoding.encode(ze.getName());
                ze.addExtraField((ZipExtraField)new UnicodePathExtraField(ze.getName(), en.array(), en.arrayOffset(), en.limit() - en.position()));
            }
            zos.putArchiveEntry(ze);
            zos.writeUsAscii("Hello, world!");
            zos.closeArchiveEntry();
            ze = new ZipArchiveEntry(EURO_FOR_DOLLAR_TXT);
            if (withExplicitUnicodeExtra && !zipEncoding.canEncode(ze.getName())) {
                en = zipEncoding.encode(ze.getName());
                ze.addExtraField((ZipExtraField)new UnicodePathExtraField(ze.getName(), en.array(), en.arrayOffset(), en.limit() - en.position()));
            }
            zos.putArchiveEntry(ze);
            zos.writeUsAscii("Give me your money!");
            zos.closeArchiveEntry();
            ze = new ZipArchiveEntry(ASCII_TXT);
            if (withExplicitUnicodeExtra && !zipEncoding.canEncode(ze.getName())) {
                en = zipEncoding.encode(ze.getName());
                ze.addExtraField((ZipExtraField)new UnicodePathExtraField(ze.getName(), en.array(), en.arrayOffset(), en.limit() - en.position()));
            }
            zos.putArchiveEntry(ze);
            zos.writeUsAscii("ascii");
            zos.closeArchiveEntry();
            zos.finish();
        }
    }

    private static UnicodePathExtraField findUniCodePath(ZipArchiveEntry ze) {
        return (UnicodePathExtraField)ze.getExtraField(UnicodePathExtraField.UPATH_ID);
    }

    private static void testFile(File file, String encoding) throws IOException {
        try (ZipFile zipFile = ((ZipFile.Builder)((ZipFile.Builder)ZipFile.builder().setFile(file)).setCharset(encoding)).setUseUnicodeExtraFields(false).get();){
            zipFile.stream().forEach(ze -> {
                if (ze.getName().endsWith("sser.txt")) {
                    UTF8ZipFilesTest.assertUnicodeName(ze, OIL_BARREL_TXT, encoding);
                } else if (ze.getName().endsWith("_for_Dollar.txt")) {
                    UTF8ZipFilesTest.assertUnicodeName(ze, EURO_FOR_DOLLAR_TXT, encoding);
                } else if (!ze.getName().equals(ASCII_TXT)) {
                    Assertions.fail((String)("Unrecognized ZIP entry with name [" + ze.getName() + "] found."));
                }
            });
        }
    }

    private void assertCanRead(ZipFile zf, String fileName) throws IOException {
        ZipArchiveEntry entry = zf.getEntry(fileName);
        Assertions.assertNotNull((Object)entry, (String)"Entry doesn't exist");
        try (InputStream is = zf.getInputStream(entry);){
            Assertions.assertNotNull((Object)is, (String)"InputStream is null");
            is.read();
        }
    }

    @Test
    void testASCIIFileRoundtripExplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.US_ASCII.name(), false, true);
    }

    @Test
    void testASCIIFileRoundtripImplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.US_ASCII.name(), false, false);
    }

    @Test
    void testCP437FileRoundtripExplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(CP437, false, true);
    }

    @Test
    void testCP437FileRoundtripImplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(CP437, false, false);
    }

    private void testFileRoundtrip(String encoding, boolean withEFS, boolean withExplicitUnicodeExtra) throws IOException {
        File file = this.createTempFile(encoding + "-test", ".zip");
        UTF8ZipFilesTest.createTestFile(file, encoding, withEFS, withExplicitUnicodeExtra);
        UTF8ZipFilesTest.testFile(file, encoding);
    }

    @Test
    void testRawNameReadFromStream() throws IOException {
        try (ZipArchiveInputStream zi = new ZipArchiveInputStream(UTF8ZipFilesTest.newInputStream("utf8-7zip-test.zip"), CP437, false);){
            UTF8ZipFilesTest.assertRawNameOfAcsiiTxt(zi.getNextEntry());
        }
    }

    @Test
    void testRawNameReadFromZipFile() throws IOException {
        File archive = UTF8ZipFilesTest.getFile("utf8-7zip-test.zip");
        try (ZipFile zf = ((ZipFile.Builder)((ZipFile.Builder)ZipFile.builder().setFile(archive)).setCharset(CP437)).setUseUnicodeExtraFields(false).get();){
            UTF8ZipFilesTest.assertRawNameOfAcsiiTxt(zf.getEntry(ASCII_TXT));
        }
    }

    @Test
    void testRead7ZipArchive() throws IOException {
        File archive = UTF8ZipFilesTest.getFile("utf8-7zip-test.zip");
        try (ZipFile zf = new ZipFile(archive, CP437, false);){
            Assertions.assertNotNull((Object)zf.getEntry(ASCII_TXT));
            Assertions.assertNotNull((Object)zf.getEntry(EURO_FOR_DOLLAR_TXT));
            Assertions.assertNotNull((Object)zf.getEntry(OIL_BARREL_TXT));
        }
    }

    @Test
    void testRead7ZipArchiveForStream() throws IOException {
        try (ZipArchiveInputStream zi = new ZipArchiveInputStream(UTF8ZipFilesTest.newInputStream("utf8-7zip-test.zip"), CP437, false);){
            Assertions.assertEquals((Object)ASCII_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)OIL_BARREL_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)EURO_FOR_DOLLAR_TXT, (Object)zi.getNextEntry().getName());
        }
    }

    @Test
    void testReadWinZipArchive() throws IOException {
        File archive = UTF8ZipFilesTest.getFile("utf8-winzip-test.zip");
        String encoding = null;
        if (Charset.defaultCharset() != StandardCharsets.UTF_8) {
            encoding = StandardCharsets.UTF_8.name();
        }
        try (ZipFile zf = ((ZipFile.Builder)((ZipFile.Builder)ZipFile.builder().setFile(archive)).setCharset(encoding)).setUseUnicodeExtraFields(true).get();){
            this.assertCanRead(zf, ASCII_TXT);
            this.assertCanRead(zf, EURO_FOR_DOLLAR_TXT);
            this.assertCanRead(zf, OIL_BARREL_TXT);
        }
    }

    @Test
    void testReadWinZipArchiveForStream() throws IOException {
        String encoding = null;
        if (Charset.defaultCharset() != StandardCharsets.UTF_8) {
            encoding = StandardCharsets.UTF_8.name();
        }
        try (InputStream archive = UTF8ZipFilesTest.newInputStream("utf8-winzip-test.zip");
             ZipArchiveInputStream zi = new ZipArchiveInputStream(archive, encoding, true);){
            Assertions.assertEquals((Object)EURO_FOR_DOLLAR_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)OIL_BARREL_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)ASCII_TXT, (Object)zi.getNextEntry().getName());
        }
    }

    @Test
    void testStreamSkipsOverUnicodeExtraFieldWithUnsupportedVersion() throws IOException {
        try (InputStream archive = UTF8ZipFilesTest.newInputStream("COMPRESS-479.zip");
             ZipArchiveInputStream zi = new ZipArchiveInputStream(archive);){
            Assertions.assertEquals((Object)OIL_BARREL_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)"%U20AC_for_Dollar.txt", (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)ASCII_TXT, (Object)zi.getNextEntry().getName());
        }
    }

    @Test
    void testUtf8FileRoundtripExplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.UTF_8.name(), true, true);
    }

    @Test
    void testUtf8FileRoundtripImplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.UTF_8.name(), true, false);
    }

    @Test
    void testUtf8FileRoundtripNoEFSExplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.UTF_8.name(), false, true);
    }

    @Test
    void testUtf8FileRoundtripNoEFSImplicitUnicodeExtra() throws IOException {
        this.testFileRoundtrip(StandardCharsets.UTF_8.name(), false, false);
    }

    @Test
    void testUtf8Interoperability() throws IOException {
        File file1 = UTF8ZipFilesTest.getFile("utf8-7zip-test.zip");
        File file2 = UTF8ZipFilesTest.getFile("utf8-winzip-test.zip");
        UTF8ZipFilesTest.testFile(file1, CP437);
        UTF8ZipFilesTest.testFile(file2, CP437);
    }

    @Test
    void testZipArchiveInputStreamReadsUnicodeFields() throws IOException {
        File file = this.createTempFile("unicode-test", ".zip");
        UTF8ZipFilesTest.createTestFile(file, StandardCharsets.US_ASCII.name(), false, true);
        try (ZipFile zf = ((ZipFile.Builder)((ZipFile.Builder)ZipFile.builder().setFile(file)).setCharset(StandardCharsets.US_ASCII)).setUseUnicodeExtraFields(true).get();){
            Assertions.assertNotNull((Object)zf.getEntry(ASCII_TXT));
            Assertions.assertNotNull((Object)zf.getEntry(EURO_FOR_DOLLAR_TXT));
            Assertions.assertNotNull((Object)zf.getEntry(OIL_BARREL_TXT));
        }
    }

    @Test
    void testZipFileReadsUnicodeFields() throws IOException {
        File file = this.createTempFile("unicode-test", ".zip");
        UTF8ZipFilesTest.createTestFile(file, StandardCharsets.US_ASCII.name(), false, true);
        try (ZipArchiveInputStream zi = new ZipArchiveInputStream(Files.newInputStream(file.toPath(), new OpenOption[0]), StandardCharsets.US_ASCII.name(), true);){
            Assertions.assertEquals((Object)OIL_BARREL_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)EURO_FOR_DOLLAR_TXT, (Object)zi.getNextEntry().getName());
            Assertions.assertEquals((Object)ASCII_TXT, (Object)zi.getNextEntry().getName());
        }
    }

    @Test
    void testZipFileSkipsOverUnicodeExtraFieldWithUnsupportedVersion() throws IOException {
        try (ZipFile zf = ((ZipFile.Builder)ZipFile.builder().setFile(UTF8ZipFilesTest.getFile("COMPRESS-479.zip"))).get();){
            Assertions.assertNotNull((Object)zf.getEntry(ASCII_TXT));
            Assertions.assertNotNull((Object)zf.getEntry("%U20AC_for_Dollar.txt"));
            Assertions.assertNotNull((Object)zf.getEntry(OIL_BARREL_TXT));
        }
    }
}

